/**
 * @file debug.c
 * @brief Debugging facilities
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include "iodefine.h"
#include "cpg_iobitmask.h"
#include "scif_iobitmask.h"
#include "debug.h"

//Peripheral clock 1 (66.67 MHz)
#define PCLK1_HZ (400000000 / 6)


/**
 * @brief Debug UART initialization
 * @param[in] baudrate UART baudrate
 **/

void debugInit(uint32_t baudrate)
{
   uint32_t status;

   //Enable SCIF3 peripheral clock
   CPG.STBCR4 &= ~CPG_STBCR4_MSTP44;
   //Disable SCIF3 module
   SCIF3.SCSCR = 0;

   //Reset transmit and receive FIFO
   SCIF3.SCFCR |= SCIF3_SCFCR_TFRST | SCIF3_SCFCR_RFRST;

   //Read serial status register
   status = SCIF3.SCFSR;
   //Clear ER, BRK and DR flags
   SCIF3.SCFSR = status & ~(SCIF3_SCFSR_ER | SCIF3_SCFSR_BRK | SCIF3_SCFSR_DR);

   //Read line status register
   status = SCIF3.SCLSR;
   //Clear ORER flag
   SCIF3.SCLSR = status & ~SCIF3_SCLSR_ORER;

   //Select internal clock
   SCIF3.SCSCR &= ~SCIF3_SCSCR_CKE;

   //Configure UART (8 bits, no parity, 1 stop bit)
   SCIF3.SCSMR = 0;

   //Enable double-speed mode
   SCIF3.SCEMR = SCIF3_SCEMR_BGDM;

   //Set baudrate
   SCIF3.SCBRR = PCLK1_HZ / (16 * baudrate) - 1;

   //Configure transmit and receive FIFO triggers
   SCIF3.SCFCR = 0;

   //Clear TFRST and RFRST bits
   SCIF3.SCFCR &= ~(SCIF3_SCFCR_TFRST | SCIF3_SCFCR_RFRST);

   //Configure TXD3 (P8_8)
   PORT8.PMCn.BIT.PMCn8 = 1;
   PORT8.PFCn.BIT.PFCn8 = 0;
   PORT8.PFCEn.BIT.PFCEn8 = 1;
   PORT8.PFCAEn.BIT.PFCAEn8 = 1;
   PORT8.PIPCn.BIT.PIPCn8 = 1;

   //Configure RXD3 (P8_9)
   PORT8.PMCn.BIT.PMCn9 = 1;
   PORT8.PFCn.BIT.PFCn9 = 0;
   PORT8.PFCEn.BIT.PFCEn9 = 1;
   PORT8.PFCAEn.BIT.PFCAEn9 = 1;
   PORT8.PIPCn.BIT.PIPCn9 = 1;

   //Enable transmission and reception
   SCIF3.SCSCR = SCIF3_SCSCR_TE | SCIF3_SCSCR_RE;
}


/**
 * @brief Display the contents of an array
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @param[in] prepend String to prepend to the left of each line
 * @param[in] data Pointer to the data array
 * @param[in] length Number of bytes to display
 **/

void debugDisplayArray(FILE *stream,
   const char_t *prepend, const void *data, size_t length)
{
   uint_t i;

   for(i = 0; i < length; i++)
   {
      //Beginning of a new line?
      if((i % 16) == 0)
         fprintf(stream, "%s", prepend);
      //Display current data byte
      fprintf(stream, "%02" PRIX8 " ", *((uint8_t *) data + i));
      //End of current line?
      if((i % 16) == 15 || i == (length - 1))
         fprintf(stream, "\r\n");
   }
}


/**
 * @brief Write character to stream
 * @param[in] c The character to be written
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @return On success, the character written is returned. If a writing
 *   error occurs, EOF is returned
 **/

int_t fputc(int_t c, FILE *stream)
{
   //Standard output or error output?
   if(stream == stdout || stream == stderr)
   {
      //Wait for the transmit FIFO to be available for writing
      while(!(SCIF3.SCFSR & SCIF3_SCFSR_TDFE));
      //Send character
      SCIF3.SCFTDR = c;
      //Clear TDFE flag
      SCIF3.SCFSR &= ~SCIF3_SCFSR_TDFE;

      //On success, the character written is returned
      return c;
   }
   //Unknown output?
   else
   {
      //If a writing error occurs, EOF is returned
      return EOF;
   }
}
