/**
 * @file debug.c
 * @brief Debugging facilities
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include "sam4e.h"
#include "sam4e_ek.h"
#include "debug.h"

//Function declaration
void lcdPutChar(char_t c);


/**
 * @brief Debug UART initialization
 * @param[in] baudrate UART baudrate
 **/

void debugInit(uint32_t baudrate)
{
   //Enable PIO peripheral clock
   PMC->PMC_PCER0 = (1 << ID_PIOA);
   //Enable UART0 peripheral clock
   PMC->PMC_PCER0 = (1 << ID_UART0);

   //Disable interrupts on TXD and RXD pins
   PIOA->PIO_IDR = PIO_PA10A_UTXD0 | PIO_PA9A_URXD0;
   //No pull-up resistors
   PIOA->PIO_PUDR = PIO_PA10A_UTXD0 | PIO_PA9A_URXD0;
   //Assign corresponding pins to Peripheral A function
   PIOA->PIO_ABCDSR[0] &= ~(PIO_PA10A_UTXD0 | PIO_PA9A_URXD0);
   PIOA->PIO_ABCDSR[1] &= ~(PIO_PA10A_UTXD0 | PIO_PA9A_URXD0);
   //Disable the PIO from controlling the corresponding pins
   PIOA->PIO_PDR = PIO_PA10A_UTXD0 | PIO_PA9A_URXD0;

   //Reset transmitter and receiver
   UART0->UART_CR = UART_CR_RSTTX | UART_CR_RSTRX;
   //Disable interrupts
   UART0->UART_IDR = 0xFFFFFFFF;

   //Configure baud rate
   UART0->UART_BRGR = SystemCoreClock / (16 * baudrate);

   //Configure mode register
   UART0->UART_MR = UART_MR_CHMODE_NORMAL | UART_MR_PAR_NO;

   //Disable DMA channel
   UART0->UART_PTCR = UART_PTCR_TXTDIS | UART_PTCR_RXTDIS;

   //Enable transmitter and receiver
   UART0->UART_CR = UART_CR_TXEN | UART_CR_RXEN;
}


/**
 * @brief Display the contents of an array
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @param[in] prepend String to prepend to the left of each line
 * @param[in] data Pointer to the data array
 * @param[in] length Number of bytes to display
 **/

void debugDisplayArray(FILE *stream,
   const char_t *prepend, const void *data, size_t length)
{
   uint_t i;

   for(i = 0; i < length; i++)
   {
      //Beginning of a new line?
      if((i % 16) == 0)
         fprintf(stream, "%s", prepend);
      //Display current data byte
      fprintf(stream, "%02" PRIX8 " ", *((uint8_t *) data + i));
      //End of current line?
      if((i % 16) == 15 || i == (length - 1))
         fprintf(stream, "\r\n");
   }
}


/**
 * @brief Write character to stream
 * @param[in] c The character to be written
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @return On success, the character written is returned. If a writing
 *   error occurs, EOF is returned
 **/

int_t fputc(int_t c, FILE *stream)
{
   //Standard output?
   if(stream == stdout)
   {
      //Display current character
      lcdPutChar(c);

      //On success, the character written is returned
      return c;
   }
   //Standard error output?
   else if(stream == stderr)
   {
      //Wait for the transmitter to be ready
      while(!(UART0->UART_SR & UART_SR_TXEMPTY));
      //Send character
      UART0->UART_THR = c;
      //Wait for the transfer to complete
      while(!(UART0->UART_SR & UART_SR_TXEMPTY));

      //On success, the character written is returned
      return c;
   }
   //Unknown output?
   else
   {
      //If a writing error occurs, EOF is returned
      return EOF;
   }
}
