/**
 * \file
 *
 * \brief Low-level initialization functions called upon chip startup.
 *
 * Copyright (c) 2016 Atmel Corporation,
 *                    a wholly owned subsidiary of Microchip Technology Inc.
 *
 * \asf_license_start
 *
 * \page License
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the Licence at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * \asf_license_stop
 *
 */

#include "same54.h"

/**
 * Initial system clock frequency.
 */
#define __SYSTEM_CLOCK    (120000000)

uint32_t SystemCoreClock = __SYSTEM_CLOCK;/*!< System Clock Frequency (Core Clock)*/

/**
 * Initialize the system
 *
 * @brief  Setup the microcontroller system.
 *         Initialize the System and update the SystemCoreClock variable.
 */
void SystemInit(void)
{
#if __FPU_USED
	/* Enable FPU */
	SCB->CPACR |=  (0xFu << 20);
	__DSB();
	__ISB();
#endif

	// Run with a 12MHz external crystal on XOSC1
	OSCCTRL->XOSCCTRL[1].bit.ENALC = 1;
	OSCCTRL->XOSCCTRL[1].bit.IMULT = 4;
	OSCCTRL->XOSCCTRL[1].bit.IPTAT = 3;
	OSCCTRL->XOSCCTRL[1].bit.ONDEMAND = 0;
	OSCCTRL->XOSCCTRL[1].bit.XTALEN = 1;
	OSCCTRL->XOSCCTRL[1].bit.ENABLE = 1;

	// Wait for OSC to be ready
	while(OSCCTRL->STATUS.bit.XOSCRDY1 == 0);

	// Set up DPLL0 to output 120MHz using XOSC1 output divided by 12
	OSCCTRL->Dpll[0].DPLLRATIO.bit.LDRFRAC = 0;
	OSCCTRL->Dpll[0].DPLLRATIO.bit.LDR = 119;
	OSCCTRL->Dpll[0].DPLLCTRLB.bit.DIV = 5;
	OSCCTRL->Dpll[0].DPLLCTRLB.bit.REFCLK = 3;
	OSCCTRL->Dpll[0].DPLLCTRLA.bit.ONDEMAND = 0;
	OSCCTRL->Dpll[0].DPLLCTRLA.bit.ENABLE = 1;

	// Wait for PLL to be locked and ready
	while(OSCCTRL->Dpll[0].DPLLSTATUS.bit.LOCK == 0 ||
		OSCCTRL->Dpll[0].DPLLSTATUS.bit.CLKRDY == 0);

	// Connect DPLL0 to clock generator 0 (120MHz)
	GCLK->GENCTRL[0].reg = GCLK_GENCTRL_SRC_DPLL0 | GCLK_GENCTRL_DIV(1) | GCLK_GENCTRL_GENEN;
	while(GCLK->SYNCBUSY.bit.GENCTRL0 != 0);

	// Keep the default device state after reset
	SystemCoreClock = __SYSTEM_CLOCK;
	return;
}

/**
 * Update SystemCoreClock variable
 *
 * @brief  Updates the SystemCoreClock with current core Clock
 *         retrieved from cpu registers.
 */
void SystemCoreClockUpdate(void)
{
	// Not implemented
	SystemCoreClock = __SYSTEM_CLOCK;
	return;
}
