/**
 * @file spi_driver.c
 * @brief SPI driver
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include "same70.h"
#include "core/net.h"
#include "spi_driver.h"
#include "debug.h"

//SPI bitrate
#define SPI_BITRATE 5000000


/**
 * @brief SPI driver
 **/

const SpiDriver spiDriver =
{
   spiInit,
   spiSetMode,
   spiSetBitrate,
   spiAssertCs,
   spiDeassertCs,
   spiTransfer
};


/**
 * @brief SPI initialization
 * @return Error code
 **/

error_t spiInit(void)
{
   uint32_t div;

   //Enable PIO peripheral clocks
   PMC->PMC_PCER0 = (1 << ID_PIOC) | (1 << ID_PIOD);
   //Enable SPI0 peripheral clock
   PMC->PMC_PCER0 = (1 << ID_SPI0);

   //Disable MDIO and MDC pins in SPI slave mode
   PIOD->PIO_PER = PIO_PD9A_GMDIO | PIO_PD8A_GMDC;
   PIOD->PIO_ODR = PIO_PD9A_GMDIO | PIO_PD8A_GMDC;

   //Configure CS pin as an output
   PIOC->PIO_PER = PIO_PC19;
   PIOC->PIO_OER = PIO_PC19;
   PIOC->PIO_SODR = PIO_PC19;

   //Disable interrupts on SPI pins
   PIOD->PIO_IDR = PIO_PD22B_SPI0_SPCK | PIO_PD21B_SPI0_MOSI | PIO_PD20B_SPI0_MISO;
   //Internal pull-up resistors
   PIOD->PIO_PUER = PIO_PD22B_SPI0_SPCK | PIO_PD21B_SPI0_MOSI | PIO_PD20B_SPI0_MISO;
   //Assign corresponding pins to Peripheral B function
   PIOD->PIO_ABCDSR[0] |= PIO_PD22B_SPI0_SPCK | PIO_PD21B_SPI0_MOSI | PIO_PD20B_SPI0_MISO;
   PIOD->PIO_ABCDSR[1] &= ~(PIO_PD22B_SPI0_SPCK | PIO_PD21B_SPI0_MOSI | PIO_PD20B_SPI0_MISO);
   //Disable the PIO from controlling the corresponding pins
   PIOD->PIO_PDR = PIO_PD22B_SPI0_SPCK | PIO_PD21B_SPI0_MOSI | PIO_PD20B_SPI0_MISO;

   //Disable SPI module
   SPI0->SPI_CR = SPI_CR_SPIDIS;
   //Reset SPI module
   SPI0->SPI_CR = SPI_CR_SWRST;
   SPI0->SPI_CR = SPI_CR_SWRST;

   //Master mode operation
   SPI0->SPI_MR = SPI_MR_MODFDIS | SPI_MR_MSTR;

   //Calculate clock divider
   div = SystemCoreClock / (2 * SPI_BITRATE);

   //SPI configuration (8-bit words, clock phase = 1, clock polarity = 0)
   SPI0->SPI_CSR[0] = SPI_CSR_SCBR(div) | SPI_CSR_BITS_8_BIT | SPI_CSR_NCPHA;

   //Enable SPI module
   SPI0->SPI_CR = SPI_CR_SPIEN;

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Set SPI mode
 * @param mode SPI mode (0, 1, 2 or 3)
 **/

error_t spiSetMode(uint_t mode)
{
   //Not implemented
   return ERROR_NOT_IMPLEMENTED;
}


/**
 * @brief Set SPI bitrate
 * @param bitrate Bitrate value
 **/

error_t spiSetBitrate(uint_t bitrate)
{
   //Not implemented
   return ERROR_NOT_IMPLEMENTED;
}


/**
 * @brief Assert CS
 **/

void spiAssertCs(void)
{
   //Assert CS signal
   PIOC->PIO_CODR = PIO_PC19;
   //CS setup time
   usleep(1);
}


/**
 * @brief Deassert CS
 **/

void spiDeassertCs(void)
{
   //CS hold time
   usleep(1);
   //Deassert CS signal
   PIOC->PIO_SODR = PIO_PC19;
   //CS disable time
   usleep(1);
}


/**
 * @brief Transfer a single byte
 * @param[in] data The data to be written
 * @return The data received from the slave device
 **/

uint8_t spiTransfer(uint8_t data)
{
   //Ensure the TX buffer is empty
   while(!(SPI0->SPI_SR & SPI_SR_TDRE));
   //Start to transfer data
   SPI0->SPI_TDR = data;
   //Wait for the operation to complete
   while(!(SPI0->SPI_SR & SPI_SR_RDRF));

   //Return the received character
   return SPI0->SPI_RDR;
}
