/**
 * @file wwd_buffer.c
 * @brief WICED network buffer management
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include "core/net.h"
#include "wwd_buffer.h"
#include "wwd_buffer_interface.h"
#include "debug.h"

#define WWD_BUFFER_COUNT 6

OsMutex wwdBufferMutex;
OsSemaphore wwdBufferSemaphore;
WwdBuffer wwdBuffer[WWD_BUFFER_COUNT];

wwd_result_t wwd_buffer_init(void *native_arg)
{
   osCreateMutex(&wwdBufferMutex);
   osCreateSemaphore(&wwdBufferSemaphore, WWD_BUFFER_COUNT);

   memset(wwdBuffer, 0, sizeof(wwdBuffer));

   return WWD_SUCCESS;
}

wwd_result_t host_buffer_check_leaked(void)
{
   return WWD_SUCCESS;
}

wwd_result_t internal_host_buffer_get(wiced_buffer_t *buffer,
   wwd_buffer_dir_t direction, unsigned short size, unsigned long timeout)
{
   uint_t i;
   bool_t status;
   wwd_result_t ret;

   if(size > (1518 + WICED_LINK_OVERHEAD_BELOW_ETHERNET_FRAME_MAX))
   {
      *buffer = NULL;
      return WWD_BUFFER_UNAVAILABLE_PERMANENT;
   }

   status = osWaitForSemaphore(&wwdBufferSemaphore, timeout);

   if(!status)
   {
      TRACE_ERROR("#### host_buffer_get TX failed\r\n");
      return WWD_BUFFER_UNAVAILABLE_TEMPORARY;
   }

   osAcquireMutex(&wwdBufferMutex);

   for(i = 0; i < WWD_BUFFER_COUNT; i++)
   {
      if(!wwdBuffer[i].used)
      {
         wwdBuffer[i].used = TRUE;
         wwdBuffer[i].size = size;
         wwdBuffer[i].offset = 0;
         break;
      }
   }

   if(i < WWD_BUFFER_COUNT)
   {
      *buffer = &wwdBuffer[i];
      ret = WWD_SUCCESS;
   }
   else
   {
      *buffer = NULL;
      ret = WWD_BUFFER_UNAVAILABLE_TEMPORARY;
   }

   osReleaseMutex(&wwdBufferMutex);
   return ret;
}

wwd_result_t host_buffer_get(wiced_buffer_t *buffer, wwd_buffer_dir_t direction, unsigned short size, wiced_bool_t wait)
{
   systime_t timeout;

   if(wait)
      timeout = INFINITE_DELAY;
   else
      timeout = 0;

   return internal_host_buffer_get(buffer, direction, size, timeout);
}

void host_buffer_release(wiced_buffer_t buffer, wwd_buffer_dir_t direction)
{
   osAcquireMutex(&wwdBufferMutex);

   buffer->used = FALSE;
   buffer->size = 0;
   buffer->offset = 0;

   osReleaseMutex(&wwdBufferMutex);

   osReleaseSemaphore(&wwdBufferSemaphore);
}

uint8_t *host_buffer_get_current_piece_data_pointer(wiced_buffer_t buffer)
{
   return buffer->data + buffer->offset;
}

uint16_t host_buffer_get_current_piece_size( wiced_buffer_t buffer )
{
   return buffer->size;
}

wiced_buffer_t host_buffer_get_next_piece( wiced_buffer_t buffer )
{
   return NULL;
}

wwd_result_t host_buffer_add_remove_at_front(wiced_buffer_t *buffer, int32_t add_remove_amount)
{
   (*buffer)->offset += add_remove_amount;
   (*buffer)->size -= add_remove_amount;

   return WWD_SUCCESS;
}

wwd_result_t host_buffer_set_size(wiced_buffer_t buffer, uint16_t size)
{
   buffer->size = size;
   return WWD_SUCCESS;
}
