/**
 * @file debug.c
 * @brief Debugging facilities
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include <console_io.h>
#include "mcf52259.h"
#include "twr_mcf5225x.h"
#include "debug.h"


/**
 * @brief Debug UART initialization
 * @param[in] baudrate UART baudrate
 **/

void debugInit(uint32_t baudrate)
{
   uint8_t temp;
   uint16_t div;

   //Configure UTXD0 (PUA0)
   temp = MCF_GPIO_PUAPAR & ~MCF_GPIO_PUAPAR_PUAPAR0(3);
   MCF_GPIO_PUAPAR = temp | MCF_GPIO_PUAPAR_UTXD0_UTXD0;

   //Configure URXD0 (PUA1)
   temp = MCF_GPIO_PUAPAR & ~MCF_GPIO_PUAPAR_PUAPAR1(3);
   MCF_GPIO_PUAPAR = temp | MCF_GPIO_PUAPAR_URXD0_URXD0;

   //Reset transmitter
   MCF_UART0_UCR = MCF_UART_UCR_RESET_TX;
   //Reset receiver
   MCF_UART0_UCR = MCF_UART_UCR_RESET_RX;
   //Reset mode register
   MCF_UART0_UCR = MCF_UART_UCR_RESET_MR;

   //UART0 configuration
   MCF_UART0_UMR1 = MCF_UART_UMR_PM_NONE | MCF_UART_UMR_BC_8;
   MCF_UART0_UMR2 = MCF_UART_UMR_CM_NORMAL | MCF_UART_UMR_SB_STOP_BITS_1;

   //Select clock source
   MCF_UART0_UCSR = MCF_UART_UCSR_RCS_SYS_CLK | MCF_UART_UCSR_TCS_SYS_CLK;

   //Compute the baudrate divider
   div = SYSTEM_CLOCK_HZ / (32 * baudrate);

   //Select baudrate
   MCF_UART0_UBG1 = MSB(div);
   MCF_UART0_UBG2 = LSB(div);

   //Enable transmitter and receiver
   MCF_UART0_UCR = MCF_UART_UCR_TX_ENABLED | MCF_UART_UCR_RX_ENABLED;
}


/**
 * @brief Display the contents of an array
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @param[in] prepend String to prepend to the left of each line
 * @param[in] data Pointer to the data array
 * @param[in] length Number of bytes to display
 **/

void debugDisplayArray(FILE *stream,
   const char_t *prepend, const void *data, size_t length)
{
   uint_t i;

   for(i = 0; i < length; i++)
   {
      //Beginning of a new line?
      if((i % 16) == 0)
         printf("%s", prepend);
      //Display current data byte
      printf("%02" PRIX8 " ", *((uint8_t *) data + i));
      //End of current line?
      if((i % 16) == 15 || i == (length - 1))
         printf("\r\n");
   }
}


/**
 * @brief Write to output stream
 * @param[in] handle Handle that identifies a stream
 * @param[in] buffer Pointer to the data buffer
 * @param[in] count Number of characters to be written
 * @return Error code
 **/

int_t __write_console(__file_handle handle, uint8_t *buffer, size_t *count)
{
   size_t i;

   //Process all the characters
   for(i = 0; i < *count; i++)
   {
      //Wait for the transmitter to be ready
      while(!(MCF_UART0_USR & MCF_UART_USR_TXRDY));
      //Send character
      MCF_UART0_UTB = buffer[i];
      //Wait for the transfer to complete
      while(!(MCF_UART0_USR & MCF_UART_USR_TXRDY));
   }

   //Successful processing
   return __no_io_error;
}


/**
 * @brief Read from input stream
 * @param[in] handle Handle that identifies a stream
 * @param[in] buffer Pointer to the data buffer
 * @param[in] count Maximum number of characters to be read
 * @return Error code
 **/

int_t __read_console(__file_handle handle, uint8_t *buffer, size_t *count)
{
   //Successful processing
   return __no_io_error;
}


/**
 * @brief Close stream
 * @param[in] handle Handle that identifies a stream
 * @return Error code
 **/

int_t __close_console(__file_handle handle)
{
   //Successful processing
   return __no_io_error;
}
