/**
 * @file ext_int_driver.c
 * @brief External interrupt line driver
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include <p32xxxx.h>
#include "core/net.h"
#include "drivers/eth/enc28j60_driver.h"
#include "ext_int_driver.h"
#include "debug.h"


/**
 * @brief External interrupt line driver
 **/

const ExtIntDriver extIntDriver =
{
   extIntInit,
   extIntEnableIrq,
   extIntDisableIrq
};


/**
 * @brief EXTI configuration
 * @return Error code
 **/

error_t extIntInit(void)
{
   //Enable pull-up on INT1 (AN24/RD1)
   CNPUDSET = _CNPUD_CNPUD1_MASK;
   //Disable analog pad
   ANSELDCLR = _ANSELD_ANSD1_MASK;

   //Remap INT1 to RPD1
   INT1R = 0;

   //Configure edge polarity for INT1 interrupt (falling edge)
   INTCONCLR = _INTCON_INT1EP_MASK;

   //Set interrupt priority
   IPC1CLR = _IPC1_INT1IP_MASK;
   IPC1SET = (2 << _IPC1_INT1IP_POSITION);

   //Set interrupt subpriority
   IPC1CLR = _IPC1_INT1IS_MASK;
   IPC1SET = (0 << _IPC1_INT1IS_POSITION);

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Enable external interrupts
 **/

void extIntEnableIrq(void)
{
   //Enable interrupts
   IEC0SET = _IEC0_INT1IE_MASK;
}


/**
 * @brief Disable external interrupts
 **/

void extIntDisableIrq(void)
{
   //Disable interrupts
   IEC0CLR = _IEC0_INT1IE_MASK;
}


/**
 * @brief External interrupt handler
 **/

void extIntHandler(void)
{
   bool_t flag;
   NetInterface *interface;

   //Enter interrupt service routine
   osEnterIsr();

   //Point to the structure describing the network interface
   interface = &netInterface[0];
   //This flag will be set if a higher priority task must be woken
   flag = FALSE;

   //Check interrupt status
   if(IFS0 & _IFS0_INT1IF_MASK)
   {
      //Clear interrupt flag
      IFS0CLR = _IFS0_INT1IF_MASK;
      //Call interrupt handler
      flag = enc28j60IrqHandler(interface);
   }

   //Leave interrupt service routine
   osExitIsr(flag);
}
