/**
 * @file debug.c
 * @brief Debugging facilities
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include "lpc23xx.h"
#include "lpc2378_stk.h"
#include "debug.h"

//Function declaration
void lcdPutChar(char_t c);


/**
 * @brief Debug UART initialization
 * @param[in] baudrate UART baudrate
 **/

void debugInit(uint32_t baudrate)
{
   uint32_t pclk;

   //Power up UART0
   PCONP |= PCONP_PCUART0;

   //Configure P0.15 (TXD1)
   PINSEL0 &= ~PINSEL0_P0_15_MASK;
   PINSEL0 |= PINSEL0_P0_15_TXD1;

   //Configure P0.16 (RXD1)
   PINSEL1 &= ~PINSEL1_P0_16_MASK;
   PINSEL1 |= PINSEL1_P0_16_RXD1;

   //Check the UART0 peripheral clock
   switch(PCLKSEL0 & PCLKSEL0_PCLK_UART1)
   {
   case PCLKSEL0_PCLK_UART1_DIV1:
      pclk = CPU_CLOCK_FREQ;
      break;
   case PCLKSEL0_PCLK_UART1_DIV2:
      pclk = CPU_CLOCK_FREQ / 2;
      break;
   case PCLKSEL0_PCLK_UART1_DIV4:
      pclk = CPU_CLOCK_FREQ / 4;
      break;
   default:
      pclk = CPU_CLOCK_FREQ / 8;
      break;
   }

   //Configure UART0 (8 bits, no parity, 1 stop bit)
   U1LCR = LCR_DLAB | LCR_WORD_LENGTH_SELECT_8;
   //Set baudrate
   U1DLM = MSB(pclk / 16 / baudrate);
   U1DLL = LSB(pclk / 16 / baudrate);
   //Clear DLAB
   U1LCR &= ~LCR_DLAB;

   //Enable and reset FIFOs
   U1FCR = FCR_TX_FIFO_RESET | FCR_RX_FIFO_RESET | FCR_FIFO_ENABLE;
}


/**
 * @brief Display the contents of an array
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @param[in] prepend String to prepend to the left of each line
 * @param[in] data Pointer to the data array
 * @param[in] length Number of bytes to display
 **/

void debugDisplayArray(FILE *stream,
   const char_t *prepend, const void *data, size_t length)
{
   uint_t i;

   for(i = 0; i < length; i++)
   {
      //Beginning of a new line?
      if((i % 16) == 0)
         fprintf(stream, "%s", prepend);
      //Display current data byte
      fprintf(stream, "%02" PRIX8 " ", *((uint8_t *) data + i));
      //End of current line?
      if((i % 16) == 15 || i == (length - 1))
         fprintf(stream, "\r\n");
   }
}


/**
 * @brief Write character to stream
 * @param[in] c The character to be written
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @return On success, the character written is returned. If a writing
 *   error occurs, EOF is returned
 **/

int_t fputc(int_t c, FILE *stream)
{
   //Standard output?
   if(stream == stdout)
   {
      //Display current character
      lcdPutChar(c);

      //On success, the character written is returned
      return c;
   }
   //Standard error output?
   else if(stream == stderr)
   {
      //Wait for the transmitter to be ready
      while(!(U1LSR & LSR_THRE));
      //Send character
      U1THR = c;
      //Wait for the transfer to complete
      while(!(U1LSR & LSR_TEMT));

      //On success, the character written is returned
      return c;
   }
   //Unknown output?
   else
   {
      //If a writing error occurs, EOF is returned
      return EOF;
   }
}
