/**
 * @file ext_int_driver.c
 * @brief External interrupt line driver
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include "91x_lib.h"
#include "core/net.h"
#include "ext_int_driver.h"
#include "debug.h"


/**
 * @brief External interrupt line driver
 **/

const ExtIntDriver extIntDriver =
{
   extIntInit,
   extIntEnableIrq,
   extIntDisableIrq
};


/**
 * @brief EXTI configuration
 * @return Error code
 **/

error_t extIntInit(void)
{
   GPIO_InitTypeDef GPIO_InitStructure;
   WIU_InitTypeDef WIU_InitStructure;

   //Enable GPIO5 clock
   SCU_APBPeriphClockConfig(__GPIO5, ENABLE);
   //Enable WIU clock
   SCU_APBPeriphClockConfig(__WIU, ENABLE);

   //Configure EXTINT13 (P5.5) as an input
   GPIO_InitStructure.GPIO_Pin = GPIO_Pin_5;
   GPIO_InitStructure.GPIO_Direction = GPIO_PinInput;
   GPIO_InitStructure.GPIO_IPInputConnected = GPIO_IPInputConnected_Disable;
   GPIO_InitStructure.GPIO_Alternate = GPIO_InputAlt1;
   GPIO_Init(GPIO5, &GPIO_InitStructure);

   //Reset WIU peripheral
   WIU_DeInit();

   //Configure external interrupt line
   WIU_InitStructure.WIU_Line = WIU_Line13;
   WIU_InitStructure.WIU_TriggerEdge = WIU_FallingEdge;
   WIU_Init(&WIU_InitStructure);

   //Configure WIU interrupt priority
   VIC_Config(WIU_ITLine, VIC_IRQ, 15);

   //Enable WIU
   WIU_Cmd(ENABLE);

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Enable external interrupts
 **/

void extIntEnableIrq(void)
{
   //Enable WIU interrupt
   VIC_ITCmd(WIU_ITLine, ENABLE);
}


/**
 * @brief Disable external interrupts
 **/

void extIntDisableIrq(void)
{
   //Disable WIU interrupt
   VIC_ITCmd(WIU_ITLine, DISABLE);
}


/**
 * @brief External interrupt handler
 **/

void WIU_IRQHandler(void)
{
   bool_t flag;
   NetInterface *interface;

   //Enter interrupt service routine
   osEnterIsr();

   //Point to the structure describing the network interface
   interface = &netInterface[0];
   //This flag will be set if a higher priority task must be woken
   flag = FALSE;

   //Check interrupt status
   if(WIU_GetITStatus(WIU_Line13) != RESET)
   {
      //Clear interrupt flag
      WIU_ClearITPendingBit(WIU_Line13);

      //Set event flag
      interface->phyEvent = TRUE;
      //Notify the TCP/IP stack of the event
      flag = osSetEventFromIsr(&netEvent);
   }

   //Leave interrupt service routine
   osExitIsr(flag);
}
