/**
 * @file ext_int_driver.c
 * @brief External interrupt line driver
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include <iorx63n.h>
#include <intrinsics.h>
#include "core/net.h"
#include "ext_int_driver.h"
#include "debug.h"


/**
 * @brief External interrupt line driver
 **/

const ExtIntDriver extIntDriver =
{
   extIntInit,
   extIntEnableIrq,
   extIntDisableIrq
};


/**
 * @brief EXTI configuration
 * @return Error code
 **/

error_t extIntInit(void)
{
   //Configure reset pin as an output
   //PORTC.PDR.BIT.B3 = 1;
   //Reset the PHY transceiver by asserting RST pin
   //PORTC.PODR.BIT.B3 = 1;
   //sleep(10);
   //Take the PHY transceiver out of reset
   //PORTC.PODR.BIT.B3 = 0;

   //Make sure IQR15 interrupt is disabled
   IEN(ICU, IRQ15) = 0;
   //Disable digital filter
   ICU.IRQFLTE1.BIT.FLTEN15 = 0;

   //Unlock MPC registers
   MPC.PWPR.BIT.B0WI = 0;
   MPC.PWPR.BIT.PFSWE = 1;

   //Configure IRQ15 (P07) as an input
   PORT0.PDR.BIT.B7 = 0;
   PORT0.PMR.BIT.B7 = 0;
   MPC.P07PFS.BYTE  = 0x40;

   //Lock MPC registers
   MPC.PWPR.BIT.PFSWE = 0;
   MPC.PWPR.BIT.B0WI = 0;

   //Set digital filter sampling clock (PCLK)
   ICU.IRQFLTC1.BIT.FCLKSEL15 = 0;
   //Enable digital filter
   ICU.IRQFLTE1.BIT.FLTEN15 = 1;
   //Configure IRQ15 polarity (falling edge)
   ICU.IRQCR[15].BIT.IRQMD = 1;

   //Configure IRQ15 interrupt priority
   IPR(ICU, IRQ15) = 1;
   //Clear interrupt flag
   IR(ICU, IRQ15) = 0;

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Enable external interrupts
 **/

void extIntEnableIrq(void)
{
   //Enable IRQ15 interrupt
   IEN(ICU, IRQ15) = 1;
}


/**
 * @brief Disable external interrupts
 **/

void extIntDisableIrq(void)
{
   //Disable IRQ15 interrupt
   IEN(ICU, IRQ15) = 0;
}


/**
 * @brief External interrupt handler
 **/

#pragma vector = VECT_ICU_IRQ15
__interrupt void extIntHandler(void)
{
   bool_t flag;
   NetInterface *interface;

   //Allow nested interrupts
   __enable_interrupt();

   //Point to the structure describing the network interface
   interface = &netInterface[0];

   //Set event flag
   interface->phyEvent = TRUE;
   //Notify the TCP/IP stack of the event
   flag = osSetEventFromIsr(&netEvent);

   //Leave interrupt service routine
   osExitIsr(flag);
}
