/**
 * @file ssif0.c
 * @brief SSIF (Serial Sound Interface)
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include <stdio.h>
#include <string.h>
#include "iodefine.h"
#include "cpg_iobitmask.h"
#include "ssif0.h"
#include "debug.h"


/**
 * @brief Initialize SSIF0 interface
 **/

void ssif0Init(void)
{
   uint32_t temp;

   //Debug message
   TRACE_INFO("Initializing SSIF0 interface...\r\n");

   //Enable SSIF0 peripheral clock
   CPG.STBCR11 &= ~CPG_STBCR11_MSTP115;

   //SSIF0 software reset
   CPG.SWRSTCR1 |= CPG_SWRSTCR1_SRST16;
   CPG.SWRSTCR1 &= ~CPG_SWRSTCR1_SRST16;

   //Configure SSISCK0 (P4_4)
   PORT4.PMCn.BIT.PMCn4 = 1;
   PORT4.PFCn.BIT.PFCn4 = 0;
   PORT4.PFCEn.BIT.PFCEn4 = 0;
   PORT4.PFCAEn.BIT.PFCAEn4 = 1;
   PORT4.PIPCn.BIT.PIPCn4 = 1;

   //Configure SSIWS0 (P4_5)
   PORT4.PMCn.BIT.PMCn5 = 1;
   PORT4.PFCn.BIT.PFCn5 = 0;
   PORT4.PFCEn.BIT.PFCEn5 = 0;
   PORT4.PFCAEn.BIT.PFCAEn5 = 1;
   PORT4.PIPCn.BIT.PIPCn5 = 1;

   //Configure SSIRXD0 (P4_6)
   PORT4.PMCn.BIT.PMCn6 = 1;
   PORT4.PFCn.BIT.PFCn6 = 0;
   PORT4.PFCEn.BIT.PFCEn6 = 0;
   PORT4.PFCAEn.BIT.PFCAEn6 = 1;
   PORT4.PIPCn.BIT.PIPCn6 = 1;

   //Configure SSITXD0 (P4_7)
   PORT4.PMCn.BIT.PMCn7 = 1;
   PORT4.PFCn.BIT.PFCn7 = 0;
   PORT4.PFCEn.BIT.PFCEn7 = 0;
   PORT4.PFCAEn.BIT.PFCAEn7 = 1;
   PORT4.PIPCn.BIT.PIPCn7 = 1;

   //Clock source for oversampling (AUDIO_X1)
   SSIF0.SSICR &= ~SSIF0_SSICR_CKS;

   //Number of channels per system word (1 channel)
   temp = SSIF0.SSICR & ~SSIF0_SSICR_CHNL;
   SSIF0.SSICR = temp | (0 << SSIF0_SSICR_CHNL_SHIFT);
   //Number of bits in a data word (16 bits)
   temp = SSIF0.SSICR & ~SSIF0_SSICR_DWL;
   SSIF0.SSICR = temp | (1 << SSIF0_SSICR_DWL_SHIFT);
   //Number of bits in a system word (16 bits)
   temp = SSIF0.SSICR & ~SSIF0_SSICR_SWL;
   SSIF0.SSICR = temp | (1 << SSIF0_SSICR_SWL_SHIFT);

   //Serial bit clock direction (master mode)
   SSIF0.SSICR |= SSIF0_SSICR_SCKD;
   //Serial WS direction (master mode)
   SSIF0.SSICR |= SSIF0_SSICR_SWSD;
   //Serial bit clock polarity
   SSIF0.SSICR &= ~SSIF0_SSICR_SCKP;
   //Serial WS polarity
   SSIF0.SSICR &= ~SSIF0_SSICR_SWSP;
   //Serial padding polarity
   SSIF0.SSICR &= ~SSIF0_SSICR_SPDP;
   //Serial data alignment
   SSIF0.SSICR &= ~SSIF0_SSICR_SDTA;
   //Parallel data alignment
   SSIF0.SSICR &= ~SSIF0_SSICR_PDTA;
   //Serial data delay
   SSIF0.SSICR |= SSIF0_SSICR_DEL;

   //Serial oversampling clock division ratio (16)
   temp = SSIF0.SSICR & ~SSIF0_SSICR_CKDV;
   SSIF0.SSICR = temp | (4 << SSIF0_SSICR_CKDV_SHIFT);

   //Disable mute
   SSIF0.SSICR &= ~SSIF0_SSICR_MUEN;

   //Enable WS continue mode
   SSIF0.SSITDMR |= SSIF0_SSITDMR_CONT;
   //Disable TDM mode
   SSIF0.SSITDMR &= ~SSIF0_SSITDMR_TDM;

   //Reset transmit FIFO
   SSIF0.SSIFCR |= SSIF0_SSIFCR_TFRST;

   //Transmit data trigger number (7)
   temp = SSIF0.SSIFCR & ~SSIF0_SSIFCR_TTRG;
   SSIF0.SSIFCR = temp | (0 << SSIF0_SSIFCR_TTRG_SHIFT);
}


/**
 * @brief Start SSIF0 interface
 **/

void ssif0Start(void)
{
   //Take TX FIFO out of reset
   SSIF0.SSIFCR &= ~SSIF0_SSIFCR_TFRST;

   //Clear error flags
   SSIF0.SSISR &= ~(SSIF0_SSISR_TUIRQ | SSIF0_SSISR_TOIRQ);
   //Enable error interrupts
   SSIF0.SSICR |= SSIF0_SSICR_TUIEN | SSIF0_SSICR_TOIEN;

   //Enable transmit interrupt
   SSIF0.SSIFCR |= SSIF0_SSIFCR_TIE;
   //Enable transmit operation
   SSIF0.SSICR |= SSIF0_SSICR_TEN;
}


/**
 * @brief Stop SSIF0 interface
 **/

void ssif0Stop(void)
{
   //Disable transmit operation
   SSIF0.SSICR &= ~SSIF0_SSICR_TEN;
   //Reset TX FIFO
   SSIF0.SSIFCR |= SSIF0_SSIFCR_TFRST;
}
