/**
 * @file ext_int_driver.c
 * @brief External interrupt line driver
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include "bsp_irq_cfg.h"
#include "s7g2.h"
#include "sk_s7g2.h"
#include "core/net.h"
#include "ext_int_driver.h"
#include "debug.h"


/**
 * @brief External interrupt line driver
 **/

const ExtIntDriver extIntDriver =
{
   extIntInit,
   extIntEnableIrq,
   extIntDisableIrq
};


/**
 * @brief EXTI configuration
 * @return Error code
 **/

error_t extIntInit(void)
{
   //Disable digital filter
   R_ICU->IRQCRn_b[14].FLTEN = 0;

   //Unlock PFS registers
   R_PMISC->PWPR_b.BOWI = 0;
   R_PMISC->PWPR_b.PFSWE = 1;

   //Configure PHY IRQ pin as an input
   R_PFS->PHY_IRQ_PFS_b.PMR = 0;
   R_PFS->PHY_IRQ_PFS_b.PDR = 0;
   R_PFS->PHY_IRQ_PFS_b.ISEL = 1;

   //Lock PFS registers
   R_PMISC->PWPR_b.PFSWE = 0;
   R_PMISC->PWPR_b.BOWI = 1;

   //Set digital filter sampling clock (PCLK)
   R_ICU->IRQCRn_b[14].FCLKSEL = 0;
   //Enable digital filter
   R_ICU->IRQCRn_b[14].FLTEN = 1;
   //Configure IRQ14 polarity (falling edge)
   R_ICU->IRQCRn_b[14].IRQMD = 0;

   //Redirect the event to the NVIC
   R_ICU->IELSRn_b[PORT_IRQ14_IRQn].DTCE = 0;

   //Configure IRQ14 interrupt priority
   NVIC_SetPriority(PORT_IRQ14_IRQn, NVIC_EncodePriority(3, 15, 0));

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Enable external interrupts
 **/

void extIntEnableIrq(void)
{
   //Enable IRQ14 interrupt
   NVIC_EnableIRQ(PORT_IRQ14_IRQn);
}


/**
 * @brief Disable external interrupts
 **/

void extIntDisableIrq(void)
{
   //Disable IRQ14 interrupt
   NVIC_DisableIRQ(PORT_IRQ14_IRQn);
}


/**
 * @brief External interrupt handler
 **/

void PORT_IRQ14_IRQHandler(void)
{
   bool_t flag;
   NetInterface *interface;

   //Enter interrupt service routine
   osEnterIsr();

   //Clear interrupt flag
   R_ICU->IELSRn_b[PORT_IRQ14_IRQn].IR = 0;

   //Point to the structure describing the network interface
   interface = &netInterface[0];

   //Set event flag
   interface->phyEvent = TRUE;
   //Notify the TCP/IP stack of the event
   flag = osSetEventFromIsr(&netEvent);

   //Leave interrupt service routine
   osExitIsr(flag);
}
