/**
 * @file debug.c
 * @brief Debugging facilities
 *
 * @section License
 *
 * Copyright (C) 2010-2019 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval.
 *
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.9.4
 **/

//Dependencies
#include "s6e2cc.h"
#include "debug.h"


/**
 * @brief Debug UART initialization
 * @param[in] baudrate UART baudrate
 **/

void debugInit(uint32_t baudrate)
{
   //Deactivate AN31
   FM4_GPIO->ADE_f.AN31 = 0;

   //Configure SIN0_0 (P21) and SOT0_0 (P22)
   FM4_GPIO->PFR2_f.P1 = 1;
   FM4_GPIO->PFR2_f.P2 = 1;

   //Peripheral assignment
   FM4_GPIO->EPFR07_f.SIN0S = 1;
   FM4_GPIO->EPFR07_f.SOT0B = 1;

   //Default UART configuration
   FM4_MFS0->SCR = 0;
   FM4_MFS0->SMR = 0;
   FM4_MFS0->ESCR = 0;

   //Software reset
   FM4_MFS0->SCR_f.UPCL = 1;
   //Enable serial data output
   FM4_MFS0->SMR_f.SOE = 1;

   //Configure baud rate
   FM4_MFS0->BGR  = SystemCoreClock / (2 * baudrate) - 1;

   //Enable transmitter and receiver
   FM4_MFS0->SCR_f.TXE = 1;
   FM4_MFS0->SCR_f.RXE = 1;
}


/**
 * @brief Display the contents of an array
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @param[in] prepend String to prepend to the left of each line
 * @param[in] data Pointer to the data array
 * @param[in] length Number of bytes to display
 **/

void debugDisplayArray(FILE *stream,
   const char_t *prepend, const void *data, size_t length)
{
   uint_t i;

   for(i = 0; i < length; i++)
   {
      //Beginning of a new line?
      if((i % 16) == 0)
         fprintf(stream, "%s", prepend);
      //Display current data byte
      fprintf(stream, "%02" PRIX8 " ", *((uint8_t *) data + i));
      //End of current line?
      if((i % 16) == 15 || i == (length - 1))
         fprintf(stream, "\r\n");
   }
}


/**
 * @brief Write character to stream
 * @param[in] c The character to be written
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @return On success, the character written is returned. If a writing
 *   error occurs, EOF is returned
 **/

int_t fputc(int_t c, FILE *stream)
{
   //Standard output or error output?
   if(stream == stdout || stream == stderr)
   {
      //Wait for the transmitter to be ready
      while(!FM4_MFS0->SSR_f.TDRE);
      //Send character
      FM4_MFS0->RDR = c;
      //Wait for the transfer to complete
      while(!FM4_MFS0->SSR_f.TDRE);

      //On success, the character written is returned
      return c;
   }
   //Unknown output?
   else
   {
      //If a writing error occurs, EOF is returned
      return EOF;
   }
}
