//*****************************************************************************
//
// pushbutton.h - Prototypes for the push buttons.
//
// Copyright (c) 2008-2015 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 2.1.1.71 of the Tiva Graphics Library.
//
//*****************************************************************************

#ifndef __PUSHBUTTON_H__
#define __PUSHBUTTON_H__

//*****************************************************************************
//
//! \addtogroup pushbutton_api
//! @{
//
//*****************************************************************************

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif

//*****************************************************************************
//
//! The structure that describes a push button widget.
//
//*****************************************************************************
typedef struct
{
    //
    //! The generic widget information.
    //
    tWidget sBase;

    //
    //! The style for this widget.  This is a set of flags defined by
    //! PB_STYLE_xxx.
    //
    uint32_t ui32Style;

    //
    //! The 24-bit RGB color used to fill this push button, if PB_STYLE_FILL is
    //! selected, and to use as the background color if PB_STYLE_TEXT_OPAQUE is
    //! selected.
    //
    uint32_t ui32FillColor;

    //
    //! The 24-bit RGB color used to fill this push button when it is pressed,
    //! if PB_STYLE_FILL is selected, and to use as the background color if
    //! PB_STYLE_TEXT_OPAQUE is selected.
    //
    uint32_t ui32PressFillColor;

    //
    //! The 24-bit RGB color used to outline this push button, if
    //! PB_STYLE_OUTLINE is selected.
    //
    uint32_t ui32OutlineColor;

    //
    //! The 24-bit RGB color used to draw text on this push button, if
    //! PB_STYLE_TEXT is selected.
    //
    uint32_t ui32TextColor;

    //
    //! A pointer to the font used to render the push button text, if
    //! PB_STYLE_TEXT is selected.
    //
    const tFont *psFont;

    //
    //! A pointer to the text to draw on this push button, if PB_STYLE_TEXT is
    //! selected.
    //
    const char *pcText;

    //
    //! A pointer to the image to be drawn onto this push button, if
    //! PB_STYLE_IMG is selected.
    //
    const uint8_t *pui8Image;

    //
    //! A pointer to the image to be drawn onto this push button when it is
    //! pressed, if PB_STYLE_IMG is selected.
    //
    const uint8_t *pui8PressImage;

    //
    //! The number of pointer events to delay before starting to auto-repeat,
    //! if PB_STYLE_AUTO_REPEAT is selected.  The amount of time to which this
    //! corresponds is dependent upon the rate at which pointer events are
    //! generated by the pointer driver.
    //
    uint16_t ui16AutoRepeatDelay;

    //
    //! The number of pointer events between button presses generated by the
    //! auto-repeat function, if PB_STYLE_AUTO_REPEAT is selected.  The amount
    //! of time to which this corresponds is dependent up on the rate at which
    //! pointer events are generated by the pointer driver.
    //
    uint16_t ui16AutoRepeatRate;

    //
    //! The number of pointer events that have occurred.  This is used when
    //! PB_STYLE_AUTO_REPEAT is selected to generate the auto-repeat events.
    //
    uint32_t ui32AutoRepeatCount;

    //
    //! A pointer to the function to be called when the button is pressed.
    //! This is repeatedly called when PB_STYLE_AUTO_REPEAT is selected.
    //
    void (*pfnOnClick)(tWidget *psWidget);
}
tPushButtonWidget;

//*****************************************************************************
//
//! This flag indicates that the push button should be outlined.
//
//*****************************************************************************
#define PB_STYLE_OUTLINE        0x00000001

//*****************************************************************************
//
//! This flag indicates that the push button should be filled.
//
//*****************************************************************************
#define PB_STYLE_FILL           0x00000002

//*****************************************************************************
//
//! This flag indicates that the push button should have text drawn on it.
//
//*****************************************************************************
#define PB_STYLE_TEXT           0x00000004

//*****************************************************************************
//
//! This flag indicates that the push button should have an image drawn on it.
//
//*****************************************************************************
#define PB_STYLE_IMG            0x00000008

//*****************************************************************************
//
//! This flag indicates that the push button text should be drawn opaque (in
//! other words, drawing the background pixels as well as the foreground
//! pixels).
//
//*****************************************************************************
#define PB_STYLE_TEXT_OPAQUE    0x00000010

//*****************************************************************************
//
//! This flag indicates that the push button should auto-repeat, generating
//! repeated click events while it is pressed.
//
//*****************************************************************************
#define PB_STYLE_AUTO_REPEAT    0x00000020

//*****************************************************************************
//
//! This flag indicates that the push button is pressed.
//
//*****************************************************************************
#define PB_STYLE_PRESSED        0x00000040

//*****************************************************************************
//
//! This flag indicates that the push button callback should be made when
//! the button is released rather than when it is pressed.  This does not
//! affect the operation of auto repeat buttons.
//
//*****************************************************************************
#define PB_STYLE_RELEASE_NOTIFY 0x00000080

//*****************************************************************************
//
//! Declares an initialized circular push button widget data structure.
//!
//! \param psParent is a pointer to the parent widget.
//! \param psNext is a pointer to the sibling widget.
//! \param psChild is a pointer to the first child widget.
//! \param psDisplay is a pointer to the display on which to draw the push
//! button.
//! \param i32X is the X coordinate of the center of the push button.
//! \param i32Y is the Y coordinate of the center of the push button.
//! \param i32R is the radius of the push button.
//! \param ui32Style is the style to be applied to the push button.
//! \param ui32FillColor is the color used to fill in the push button.
//! \param ui32PressFillColor is the color used to fill in the push button
//! when it is pressed.
//! \param ui32OutlineColor is the color used to outline the push button.
//! \param ui32TextColor is the color used to draw text on the push button.
//! \param psFont is a pointer to the font to be used to draw text on the push
//! button.
//! \param pcText is a pointer to the text to draw on this push button.
//! \param pui8Image is a pointer to the image to draw on this push button.
//! \param pui8PressImage is a pointer to the image to draw on this push button
//! when it is pressed.
//! \param ui16AutoRepeatDelay is the delay before starting auto-repeat.
//! \param ui16AutoRepeatRate is the rate at which auto-repeat events are
//! generated.
//! \param pfnOnClick is a pointer to the function that is called when the push
//! button is pressed.
//!
//! This macro provides an initialized circular push button widget data
//! structure, which can be used to construct the widget tree at compile time
//! in global variables (as opposed to run-time via function calls).  This must
//! be assigned to a variable, such as:
//!
//! \verbatim
//!     tPushButtonWidget g_sPushButton = CircularButtonStruct(...);
//! \endverbatim
//!
//! Or, in an array of variables:
//!
//! \verbatim
//!     tPushButtonWidget g_psPushButtons[] =
//!     {
//!         CircularButtonStruct(...),
//!         CircularButtonStruct(...)
//!     };
//! \endverbatim
//!
//! \e ui32Style is the logical OR of the following:
//!
//! - \b #PB_STYLE_OUTLINE to indicate that the push button should be outlined.
//! - \b #PB_STYLE_FILL to indicate that the push button should be filled.
//! - \b #PB_STYLE_TEXT to indicate that the push button should have text drawn
//!   on it (using \e psFont and \e pcText).
//! - \b #PB_STYLE_IMG to indicate that the push button should have an image
//!   drawn on it (using \e pui8Image).
//! - \b #PB_STYLE_TEXT_OPAQUE to indicate that the push button text should be
//!   drawn opaque (in other words, drawing the background pixels).
//! - \b #PB_STYLE_AUTO_REPEAT to indicate that auto-repeat should be used.
//! - \b #PB_STYLE_RELEASE_NOTIFY to indicate that the callback should be made
//!   when the button is released.  If absent, the callback is called when the
//!   button is initially pressed.
//!
//! \return Nothing; this is not a function.
//
//*****************************************************************************
#define CircularButtonStruct(psParent, psNext, psChild, psDisplay, i32X, i32Y,\
                             i32R, ui32Style, ui32FillColor,                  \
                             ui32PressFillColor, ui32OutlineColor,            \
                             ui32TextColor, psFont, pcText, pui8Image,        \
                             pui8PressImage, ui16AutoRepeatDelay,             \
                             ui16AutoRepeatRate, pfnOnClick)                  \
        {                                                                     \
            {                                                                 \
                sizeof(tPushButtonWidget),                                    \
                (tWidget *)(psParent),                                        \
                (tWidget *)(psNext),                                          \
                (tWidget *)(psChild),                                         \
                psDisplay,                                                    \
                {                                                             \
                    (i32X) - (i32R),                                          \
                    (i32Y) - (i32R),                                          \
                    (i32X) + (i32R),                                          \
                    (i32Y) + (i32R)                                           \
                },                                                            \
                CircularButtonMsgProc                                         \
            },                                                                \
            ui32Style,                                                        \
            ui32FillColor,                                                    \
            ui32PressFillColor,                                               \
            ui32OutlineColor,                                                 \
            ui32TextColor,                                                    \
            psFont,                                                           \
            pcText,                                                           \
            pui8Image,                                                        \
            pui8PressImage,                                                   \
            ui16AutoRepeatDelay,                                              \
            ui16AutoRepeatRate,                                               \
            0,                                                                \
            pfnOnClick                                                        \
        }

//*****************************************************************************
//
//! Declares an initialized variable containing a circular push button widget
//! data structure.
//!
//! \param sName is the name of the variable to be declared.
//! \param psParent is a pointer to the parent widget.
//! \param psNext is a pointer to the sibling widget.
//! \param psChild is a pointer to the first child widget.
//! \param psDisplay is a pointer to the display on which to draw the push
//! button.
//! \param i32X is the X coordinate of the center of the push button.
//! \param i32Y is the Y coordinate of the center of the push button.
//! \param i32R is the radius of the push button.
//! \param ui32Style is the style to be applied to the push button.
//! \param ui32FillColor is the color used to fill in the push button.
//! \param ui32PressFillColor is the color used to fill in the push button
//! when it is pressed.
//! \param ui32OutlineColor is the color used to outline the push button.
//! \param ui32TextColor is the color used to draw text on the push button.
//! \param psFont is a pointer to the font to be used to draw text on the push
//! button.
//! \param pcText is a pointer to the text to draw on this push button.
//! \param pui8Image is a pointer to the image to draw on this push button.
//! \param pui8PressImage is a pointer to the image to draw on this push button
//! when it is pressed.
//! \param ui16AutoRepeatDelay is the delay before starting auto-repeat.
//! \param ui16AutoRepeatRate is the rate at which auto-repeat events are
//! generated.
//! \param pfnOnClick is a pointer to the function that is called when the push
//! button is pressed.
//!
//! This macro provides an initialized circular push button widget data
//! structure, which can be used to construct the widget tree at compile time
//! in global variables (as opposed to run-time via function calls).
//!
//! \e ui32Style is the logical OR of the following:
//!
//! - \b #PB_STYLE_OUTLINE to indicate that the push button should be outlined.
//! - \b #PB_STYLE_FILL to indicate that the push button should be filled.
//! - \b #PB_STYLE_TEXT to indicate that the push button should have text drawn
//!   on it (using \e psFont and \e pcText).
//! - \b #PB_STYLE_IMG to indicate that the push button should have an image
//!   drawn on it (using \e pui8Image).
//! - \b #PB_STYLE_TEXT_OPAQUE to indicate that the push button text should be
//!   drawn opaque (in other words, drawing the background pixels).
//! - \b #PB_STYLE_AUTO_REPEAT to indicate that auto-repeat should be used.
//! - \b #PB_STYLE_RELEASE_NOTIFY to indicate that the callback should be made
//!   when the button is released.  If absent, the callback is called when the
//!   button is initially pressed.
//!
//! \return Nothing; this is not a function.
//
//*****************************************************************************
#define CircularButton(sName, psParent, psNext, psChild, psDisplay, i32X,     \
                       i32Y, i32R, ui32Style, ui32FillColor,                  \
                       ui32PressFillColor, ui32OutlineColor, ui32TextColor,   \
                       psFont, pcText, pui8Image, pui8PressImage,             \
                       ui16AutoRepeatDelay, ui16AutoRepeatRate, pfnOnClick)   \
        tPushButtonWidget sName =                                             \
            CircularButtonStruct(psParent, psNext, psChild, psDisplay, i32X,  \
                                 i32Y, i32R, ui32Style, ui32FillColor,        \
                                 ui32PressFillColor, ui32OutlineColor,        \
                                 ui32TextColor, psFont, pcText, pui8Image,    \
                                 pui8PressImage, ui16AutoRepeatDelay,         \
                                 ui16AutoRepeatRate, pfnOnClick)

//*****************************************************************************
//
//! Declares an initialized rectangular push button widget data structure.
//!
//! \param psParent is a pointer to the parent widget.
//! \param psNext is a pointer to the sibling widget.
//! \param psChild is a pointer to the first child widget.
//! \param psDisplay is a pointer to the display on which to draw the push
//! button.
//! \param i32X is the X coordinate of the upper left corner of the push
//! button.
//! \param i32Y is the Y coordinate of the upper left corner of the push
//! button.
//! \param i32Width is the width of the push button.
//! \param i32Height is the height of the push button.
//! \param ui32Style is the style to be applied to the push button.
//! \param ui32FillColor is the color used to fill in the push button.
//! \param ui32PressFillColor is the color used to fill in the push button when
//! it is pressed.
//! \param ui32OutlineColor is the color used to outline the push button.
//! \param ui32TextColor is the color used to draw text on the push button.
//! \param psFont is a pointer to the font to be used to draw text on the push
//! button.
//! \param pcText is a pointer to the text to draw on this push button.
//! \param pui8Image is a pointer to the image to draw on this push button.
//! \param pui8PressImage is a pointer to the image to draw on this push button
//! when it is pressed.
//! \param ui16AutoRepeatDelay is the delay before starting auto-repeat.
//! \param ui16AutoRepeatRate is the rate at which auto-repeat events are
//! generated.
//! \param pfnOnClick is a pointer to the function that is called when the push
//! button is pressed.
//!
//! This macro provides an initialized rectangular push button widget data
//! structure, which can be used to construct the widget tree at compile time
//! in global variables (as opposed to run-time via function calls).  This must
//! be assigned to a variable, such as:
//!
//! \verbatim
//!     tPushButtonWidget g_sPushButton = RectangularButtonStruct(...);
//! \endverbatim
//!
//! Or, in an array of variables:
//!
//! \verbatim
//!     tPushButtonWidget g_psPushButtons[] =
//!     {
//!         RectangularButtonStruct(...),
//!         RectangularButtonStruct(...)
//!     };
//! \endverbatim
//!
//! \e ui32Style is the logical OR of the following:
//!
//! - \b #PB_STYLE_OUTLINE to indicate that the push button should be outlined.
//! - \b #PB_STYLE_FILL to indicate that the push button should be filled.
//! - \b #PB_STYLE_TEXT to indicate that the push button should have text drawn
//!   on it (using \e psFont and \e pcText).
//! - \b #PB_STYLE_IMG to indicate that the push button should have an image
//!   drawn on it (using \e pui8Image).
//! - \b #PB_STYLE_TEXT_OPAQUE to indicate that the push button text should be
//!   drawn opaque (in other words, drawing the background pixels).
//! - \b #PB_STYLE_AUTO_REPEAT to indicate that auto-repeat should be used.
//! - \b #PB_STYLE_RELEASE_NOTIFY to indicate that the callback should be made
//!   when the button is released.  If absent, the callback is called when the
//!   button is initially pressed.
//!
//! \return Nothing; this is not a function.
//
//*****************************************************************************
#define RectangularButtonStruct(psParent, psNext, psChild, psDisplay, i32X,   \
                                i32Y, i32Width, i32Height, ui32Style,         \
                                ui32FillColor, ui32PressFillColor,            \
                                ui32OutlineColor, ui32TextColor, psFont,      \
                                pcText, pui8Image, pui8PressImage,            \
                                ui16AutoRepeatDelay, ui16AutoRepeatRate,      \
                                pfnOnClick)                                   \
        {                                                                     \
            {                                                                 \
                sizeof(tPushButtonWidget),                                    \
                (tWidget *)(psParent),                                        \
                (tWidget *)(psNext),                                          \
                (tWidget *)(psChild),                                         \
                psDisplay,                                                    \
                {                                                             \
                    i32X,                                                     \
                    i32Y,                                                     \
                    (i32X) + (i32Width) - 1,                                  \
                    (i32Y) + (i32Height) - 1                                  \
                },                                                            \
                RectangularButtonMsgProc                                      \
            },                                                                \
            ui32Style,                                                        \
            ui32FillColor,                                                    \
            ui32PressFillColor,                                               \
            ui32OutlineColor,                                                 \
            ui32TextColor,                                                    \
            psFont,                                                           \
            pcText,                                                           \
            pui8Image,                                                        \
            pui8PressImage,                                                   \
            ui16AutoRepeatDelay,                                              \
            ui16AutoRepeatRate,                                               \
            0,                                                                \
            pfnOnClick                                                        \
        }

//*****************************************************************************
//
//! Declares an initialized variable containing a rectangular push button
//! widget data structure.
//!
//! \param sName is the name of the variable to be declared.
//! \param psParent is a pointer to the parent widget.
//! \param psNext is a pointer to the sibling widget.
//! \param psChild is a pointer to the first child widget.
//! \param psDisplay is a pointer to the display on which to draw the push
//! button.
//! \param i32X is the X coordinate of the upper left corner of the push
//! button.
//! \param i32Y is the Y coordinate of the upper left corner of the push
//! button.
//! \param i32Width is the width of the push button.
//! \param i32Height is the height of the push button.
//! \param ui32Style is the style to be applied to the push button.
//! \param ui32FillColor is the color used to fill in the push button.
//! \param ui32PressFillColor is the color used to fill in the push button when
//! it is pressed.
//! \param ui32OutlineColor is the color used to outline the push button.
//! \param ui32TextColor is the color used to draw text on the push button.
//! \param psFont is a pointer to the font to be used to draw text on the push
//! button.
//! \param pcText is a pointer to the text to draw on this push button.
//! \param pui8Image is a pointer to the image to draw on this push button.
//! \param pui8PressImage is a pointer to the image to draw on this push button
//! when it is pressed.
//! \param ui16AutoRepeatDelay is the delay before starting auto-repeat.
//! \param ui16AutoRepeatRate is the rate at which auto-repeat events are
//! generated.
//! \param pfnOnClick is a pointer to the function that is called when the push
//! button is pressed.
//!
//! This macro provides an initialized rectangular push button widget data
//! structure, which can be used to construct the widget tree at compile time
//! in global variables (as opposed to run-time via function calls).
//!
//! \e ui32Style is the logical OR of the following:
//!
//! - \b #PB_STYLE_OUTLINE to indicate that the push button should be outlined.
//! - \b #PB_STYLE_FILL to indicate that the push button should be filled.
//! - \b #PB_STYLE_TEXT to indicate that the push button should have text drawn
//!   on it (using \e psFont and \e pcText).
//! - \b #PB_STYLE_IMG to indicate that the push button should have an image
//!   drawn on it (using \e pui8Image).
//! - \b #PB_STYLE_TEXT_OPAQUE to indicate that the push button text should be
//!   drawn opaque (in other words, drawing the background pixels).
//! - \b #PB_STYLE_AUTO_REPEAT to indicate that auto-repeat should be used.
//! - \b #PB_STYLE_RELEASE_NOTIFY to indicate that the callback should be made
//!   when the button is released.  If absent, the callback is called when the
//!   button is initially pressed.
//!
//! \return Nothing; this is not a function.
//
//*****************************************************************************
#define RectangularButton(sName, psParent, psNext, psChild, psDisplay, i32X,  \
                          i32Y, i32Width, i32Height, ui32Style,               \
                          ui32FillColor, ui32PressFillColor,                  \
                          ui32OutlineColor, ui32TextColor,                    \
                          psFont, pcText, pui8Image, pui8PressImage,          \
                          ui16AutoRepeatDelay, ui16AutoRepeatRate, pfnOnClick)\
        tPushButtonWidget sName =                                             \
            RectangularButtonStruct(psParent, psNext, psChild, psDisplay,     \
                                    i32X, i32Y, i32Width, i32Height,          \
                                    ui32Style, ui32FillColor,                 \
                                    ui32PressFillColor, ui32OutlineColor,     \
                                    ui32TextColor, psFont, pcText, pui8Image, \
                                    pui8PressImage, ui16AutoRepeatDelay,      \
                                    ui16AutoRepeatRate, pfnOnClick)

//*****************************************************************************
//
//! Sets the auto-repeat delay for a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//! \param ui16Delay is the number of pointer events before auto-repeat starts.
//!
//! This function sets the delay before auto-repeat begins.  Unpredictable
//! behavior will occur if this is called while the push button is pressed.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonAutoRepeatDelaySet(psWidget, ui16Delay)                     \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui16AutoRepeatDelay = ui16Delay;                             \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Disables auto-repeat for a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function disables the auto-repeat behavior of a push button.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonAutoRepeatOff(psWidget)                                     \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style &= ~(PB_STYLE_AUTO_REPEAT);                        \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Enables auto-repeat for a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function enables the auto-repeat behavior of a push button.
//! Unpredictable behavior will occur if this is called while the push button
//! is pressed.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonAutoRepeatOn(psWidget)                                      \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style |= PB_STYLE_AUTO_REPEAT;                           \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Sets the auto-repeat rate for a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//! \param ui16Rate is the number of pointer events between auto-repeat events.
//!
//! This function sets the rate at which auto-repeat events occur.
//! Unpredictable behavior will occur if this is called while the push button
//! is pressed.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonAutoRepeatRateSet(psWidget, ui16Rate)                       \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui16AutoRepeatRate = ui16Rate;                               \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Sets the function to call when this push button widget is pressed.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//! \param pfnOnClik is a pointer to the function to call.
//!
//! This function sets the function to be called when this push button is
//! pressed.  The supplied function is called when the push button is first
//! pressed, and then repeated while the push button is pressed if auto-repeat
//! is enabled.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonCallbackSet(psWidget, pfnOnClik)                            \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->pfnOnClick = pfnOnClik;                                      \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Sets the fill color of a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to be modified.
//! \param ui32Color is the 24-bit RGB color to use to fill the push button.
//!
//! This function changes the color used to fill the push button on the
//! display.  The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonFillColorSet(psWidget, ui32Color)                           \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32FillColor = ui32Color;                                   \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Sets the fill color of a push button widget when it is pressed.
//!
//! \param psWidget is a pointer to the push button widget to be modified.
//! \param ui32Color is the 24-bit RGB color to use to fill the push button
//! when it is pressed.
//!
//! This function changes the color used to fill the push button on the
//! display when it is pressed.  The display is not updated until the next
//! paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonFillColorPressedSet(psWidget, ui32Color)                    \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32PressFillColor = ui32Color;                              \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Disables filling of a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function disables the filling of a push button widget.  The display is
//! not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonFillOff(psWidget)                                           \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style &= ~(PB_STYLE_FILL);                               \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Enables filling of a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function enables the filling of a push button widget.  The display is
//! not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonFillOn(psWidget)                                            \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style |= PB_STYLE_FILL;                                  \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Sets the font for a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//! \param pFnt is a pointer to the font to use to draw text on the push
//! button.
//!
//! This function changes the font used to draw text on the push button.  The
//! display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonFontSet(psWidget, pFnt)                                     \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            const tFont *pF = pFnt;                                           \
            psW->psFont = pF;                                                 \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Changes the image drawn on a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to be modified.
//! \param pImg is a pointer to the image to draw onto the push button.
//!
//! This function changes the image that is drawn onto the push button.  The
//! display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonImageSet(psWidget, pImg)                                    \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            const uint8_t *pI = pImg;                                         \
            psW->pui8Image = pI;                                              \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Changes the image drawn on a push button widget when it is pressed.
//!
//! \param psWidget is a pointer to the push button widget to be modified.
//! \param pImg is a pointer to the image to draw onto the push button when it
//! is pressed.
//!
//! This function changes the image that is drawn onto the push button when it
//! is pressed.  The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonImagePressedSet(psWidget, pImg)                             \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            const uint8_t *pI = pImg;                                         \
            psW->pui8PressImage = pI;                                         \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Disables the image on a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function disables the drawing of an image on a push button widget.
//! The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonImageOff(psWidget)                                          \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style &= ~(PB_STYLE_IMG);                                \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Enables the image on a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function enables the drawing of an image on a push button widget.  The
//! display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonImageOn(psWidget)                                           \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style |= PB_STYLE_IMG;                                   \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Sets the outline color of a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to be modified.
//! \param ui32Color is the 24-bit RGB color to use to outline the push button.
//!
//! This function changes the color used to outline the push button on the
//! display.  The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonOutlineColorSet(psWidget, ui32Color)                        \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32OutlineColor = ui32Color;                                \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Disables outlining of a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function disables the outlining of a push button widget.  The display
//! is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonOutlineOff(psWidget)                                        \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style &= ~(PB_STYLE_OUTLINE);                            \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Enables outlining of a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function enables the outlining of a push button widget.  The display
//! is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonOutlineOn(psWidget)                                         \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style |= PB_STYLE_OUTLINE;                               \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Sets the text color of a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to be modified.
//! \param ui32Color is the 24-bit RGB color to use to draw text on the push
//! button.
//!
//! This function changes the color used to draw text on the push button on the
//! display.  The display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonTextColorSet(psWidget, ui32Color)                           \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32TextColor = ui32Color;                                   \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Disables the text on a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function disables the drawing of text on a push button widget.  The
//! display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonTextOff(psWidget)                                           \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style &= ~(PB_STYLE_TEXT);                               \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Enables the text on a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function enables the drawing of text on a push button widget.  The
//! display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonTextOn(psWidget)                                            \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style |= PB_STYLE_TEXT;                                  \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Disables opaque text on a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function disables the use of opaque text on this push button.  When
//! not using opaque text, only the foreground pixels of the text are drawn on
//! the screen, allowing the previously drawn pixels (such as the push button
//! image) to show through the text.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonTextOpaqueOff(psWidget)                                     \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style &= ~(PB_STYLE_TEXT_OPAQUE);                        \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Enables opaque text on a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to modify.
//!
//! This function enables the use of opaque text on this push button.  When
//! using opaque text, both the foreground and background pixels of the text
//! are drawn on the screen, blocking out the previously drawn pixels.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonTextOpaqueOn(psWidget)                                      \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            psW->ui32Style |= PB_STYLE_TEXT_OPAQUE;                           \
        }                                                                     \
        while(0)

//*****************************************************************************
//
//! Changes the text drawn on a push button widget.
//!
//! \param psWidget is a pointer to the push button widget to be modified.
//! \param pcTxt is a pointer to the text to draw onto the push button.
//!
//! This function changes the text that is drawn onto the push button.  The
//! display is not updated until the next paint request.
//!
//! \return None.
//
//*****************************************************************************
#define PushButtonTextSet(psWidget, pcTxt)                                    \
        do                                                                    \
        {                                                                     \
            tPushButtonWidget *psW = psWidget;                                \
            const char *pcT = pcTxt;                                          \
            psW->pcText = pcT;                                                \
        }                                                                     \
        while(0)

//*****************************************************************************
//
// Prototypes for the push button widget APIs.
//
//*****************************************************************************
extern int32_t RectangularButtonMsgProc(tWidget *psWidget, uint32_t ui32Msg,
                                        uint32_t ui32Param1,
                                        uint32_t ui32Param2);
extern void RectangularButtonInit(tPushButtonWidget *psWidget,
                                  const tDisplay *psDisplay, int32_t i32X,
                                  int32_t i32Y, int32_t i32Width,
                                  int32_t i32Height);
extern int32_t CircularButtonMsgProc(tWidget *psWidget, uint32_t ui32Msg,
                                     uint32_t ui32Param1,
                                     uint32_t ui32Param2);
extern void CircularButtonInit(tPushButtonWidget *psWidget,
                               const tDisplay *psDisplay, int32_t i32X,
                               int32_t i32Y, int32_t i32R);

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

#endif // __PUSHBUTTON_H__
