/**
 * @file tls_cipher_suites.c
 * @brief TLS cipher suites
 *
 * @section License
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * Copyright (C) 2010-2022 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneSSL Open.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.2.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL TLS_TRACE_LEVEL

//Dependencies
#include "tls.h"
#include "tls_cipher_suites.h"
#include "cipher/cipher_algorithms.h"
#include "hash/hash_algorithms.h"
#include "debug.h"

//Check TLS library configuration
#if (TLS_SUPPORT == ENABLED)

//List of supported cipher suites
const TlsCipherSuiteInfo tlsSupportedCipherSuites[] =
{
//TLS_CHACHA20_POLY1305_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_3 && TLS_MIN_VERSION <= TLS_VERSION_1_3 && TLS_CHACHA20_POLY1305_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_CHACHA20_POLY1305_SHA256, TLS_KEY_EXCH_NONE, NULL, CIPHER_MODE_CHACHA20_POLY1305, NULL, SHA256_HASH_ALGO, 0, 32, 12, 0, 16, 32),
#endif

//TLS_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_3 && TLS_MIN_VERSION <= TLS_VERSION_1_3 && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_AES_128_GCM_SHA256, TLS_KEY_EXCH_NONE, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 12, 0, 16, 32),
#endif

//TLS_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_3 && TLS_MIN_VERSION <= TLS_VERSION_1_3 && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_AES_256_GCM_SHA384, TLS_KEY_EXCH_NONE, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 12, 0, 16, 48),
#endif

//TLS_AES_128_CCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_3 && TLS_MIN_VERSION <= TLS_VERSION_1_3 && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_AES_128_CCM_SHA256, TLS_KEY_EXCH_NONE, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 12, 0, 16, 32),
#endif

//TLS_AES_128_CCM_8_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_3 && TLS_MIN_VERSION <= TLS_VERSION_1_3 && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_AES_128_CCM_8_SHA256, TLS_KEY_EXCH_NONE, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 12, 0, 8, 32),
#endif

//TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CHACHA20_POLY1305_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256, TLS_KEY_EXCH_ECDHE_ECDSA, NULL, CIPHER_MODE_CHACHA20_POLY1305, NULL, SHA256_HASH_ALGO, 0, 32, 12, 0, 16, 12),
#endif

//TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CHACHA20_POLY1305_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256, TLS_KEY_EXCH_ECDHE_RSA, NULL, CIPHER_MODE_CHACHA20_POLY1305, NULL, SHA256_HASH_ALGO, 0, 32, 12, 0, 16, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_ECDHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_ECDHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256, TLS_KEY_EXCH_ECDHE_ECDSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256, TLS_KEY_EXCH_ECDHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384, TLS_KEY_EXCH_ECDHE_ECDSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384, TLS_KEY_EXCH_ECDHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256, TLS_KEY_EXCH_ECDHE_ECDSA, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256, TLS_KEY_EXCH_ECDHE_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384, TLS_KEY_EXCH_ECDHE_ECDSA, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384, TLS_KEY_EXCH_ECDHE_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_128_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_128_CCM, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_256_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_256_CCM, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 8, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 8, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_ECDHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384, TLS_KEY_EXCH_ECDHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_ECDHE_ECDSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_ECDHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384, TLS_KEY_EXCH_ECDHE_ECDSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384, TLS_KEY_EXCH_ECDHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_ECDHE_ECDSA, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_ECDHE_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_ECDHE_ECDSA, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_ECDHE_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_ECDHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_ECDHE_ECDSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_ECDHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_ECDHE_ECDSA, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_ECDHE_RSA, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CHACHA20_POLY1305_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256, TLS_KEY_EXCH_DHE_RSA, NULL, CIPHER_MODE_CHACHA20_POLY1305, NULL, SHA256_HASH_ALGO, 0, 32, 12, 0, 16, 12),
#endif

//TLS_DHE_RSA_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_RSA_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256, TLS_KEY_EXCH_DHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384, TLS_KEY_EXCH_DHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256, TLS_KEY_EXCH_DHE_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384, TLS_KEY_EXCH_DHE_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_RSA_WITH_AES_128_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_128_CCM, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_RSA_WITH_AES_256_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_256_CCM, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_RSA_WITH_AES_128_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_128_CCM_8, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 8, 12),
#endif

//TLS_DHE_RSA_WITH_AES_256_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_256_CCM_8, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 8, 12),
#endif

//TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_256_CBC_SHA256, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_DHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256, TLS_KEY_EXCH_DHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_DHE_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_DHE_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_DHE_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA, TLS_KEY_EXCH_DHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA, TLS_KEY_EXCH_DHE_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_SEED_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_SEED_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_SEED_CBC_SHA, TLS_KEY_EXCH_DHE_RSA, SEED_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_DHE_RSA, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_DHE_DSS_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_DHE_DSS, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_DSS_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_DHE_DSS, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256, TLS_KEY_EXCH_DHE_DSS, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384, TLS_KEY_EXCH_DHE_DSS, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256, TLS_KEY_EXCH_DHE_DSS, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384, TLS_KEY_EXCH_DHE_DSS, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_DSS_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_DHE_DSS, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_AES_256_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_AES_256_CBC_SHA256, TLS_KEY_EXCH_DHE_DSS, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_DHE_DSS, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256, TLS_KEY_EXCH_DHE_DSS, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_DHE_DSS, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_DHE_DSS, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_DHE_DSS, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_DHE_DSS, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA, TLS_KEY_EXCH_DHE_DSS, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA, TLS_KEY_EXCH_DHE_DSS, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_SEED_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_SEED_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_SEED_CBC_SHA, TLS_KEY_EXCH_DHE_DSS, SEED_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_DHE_DSS, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CHACHA20_POLY1305_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256, TLS_KEY_EXCH_ECDHE_PSK, NULL, CIPHER_MODE_CHACHA20_POLY1305, NULL, SHA256_HASH_ALGO, 0, 32, 12, 0, 16, 12),
#endif

//TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_ECDHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_ECDHE_PSK_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_ECDHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_ECDHE_PSK_WITH_AES_128_CCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_AES_128_CCM_SHA256, TLS_KEY_EXCH_ECDHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_ECDHE_PSK_WITH_AES_128_CCM_8_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_AES_128_CCM_8_SHA256, TLS_KEY_EXCH_ECDHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 8, 12),
#endif

//TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_ECDHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384, TLS_KEY_EXCH_ECDHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_ECDHE_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384, TLS_KEY_EXCH_ECDHE_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_ECDHE_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_ECDHE_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_ECDHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_ECDHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_ECDHE_PSK, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CHACHA20_POLY1305_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256, TLS_KEY_EXCH_DHE_PSK, NULL, CIPHER_MODE_CHACHA20_POLY1305, NULL, SHA256_HASH_ALGO, 0, 32, 12, 0, 16, 12),
#endif

//TLS_DHE_PSK_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_PSK_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256, TLS_KEY_EXCH_DHE_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384, TLS_KEY_EXCH_DHE_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256, TLS_KEY_EXCH_DHE_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384, TLS_KEY_EXCH_DHE_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_PSK_WITH_AES_128_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_128_CCM, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DHE_PSK_WITH_AES_256_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_256_CCM, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DHE_PSK_WITH_AES_128_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_128_CCM_8, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 8, 12),
#endif

//TLS_DHE_PSK_WITH_AES_256_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_256_CCM_8, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 8, 12),
#endif

//TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_256_CBC_SHA384, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_DHE_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384, TLS_KEY_EXCH_DHE_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_DHE_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_DHE_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_PSK_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DHE_PSK_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_DHE_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_DHE_PSK, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CHACHA20_POLY1305_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256, TLS_KEY_EXCH_RSA_PSK, NULL, CIPHER_MODE_CHACHA20_POLY1305, NULL, SHA256_HASH_ALGO, 0, 32, 12, 0, 16, 12),
#endif

//TLS_RSA_PSK_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_RSA_PSK, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_RSA_PSK_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_RSA_PSK, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256, TLS_KEY_EXCH_RSA_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384, TLS_KEY_EXCH_RSA_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256, TLS_KEY_EXCH_RSA_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384, TLS_KEY_EXCH_RSA_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_RSA_PSK_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_RSA_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_PSK_WITH_AES_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_AES_256_CBC_SHA384, TLS_KEY_EXCH_RSA_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_RSA_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384, TLS_KEY_EXCH_RSA_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_RSA_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_RSA_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_RSA_PSK_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_RSA_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_PSK_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_RSA_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_RSA_PSK, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_RSA_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_RSA_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256, TLS_KEY_EXCH_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384, TLS_KEY_EXCH_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_RSA_WITH_ARIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_ARIA_128_GCM_SHA256, TLS_KEY_EXCH_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_RSA_WITH_ARIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_ARIA_256_GCM_SHA384, TLS_KEY_EXCH_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_RSA_WITH_AES_128_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_128_CCM, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_RSA_WITH_AES_256_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_256_CCM, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_RSA_WITH_AES_128_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_128_CCM_8, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 8, 12),
#endif

//TLS_RSA_WITH_AES_256_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_256_CCM_8, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 8, 12),
#endif

//TLS_RSA_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_AES_256_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_256_CBC_SHA256, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 32, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256, TLS_KEY_EXCH_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 32, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_RSA, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_RSA, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_CAMELLIA_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_CAMELLIA_128_CBC_SHA, TLS_KEY_EXCH_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_CAMELLIA_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_CAMELLIA_256_CBC_SHA, TLS_KEY_EXCH_RSA, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_SEED_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_SEED_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_SEED_CBC_SHA, TLS_KEY_EXCH_RSA, SEED_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_RSA_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_RSA, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_PSK_WITH_CHACHA20_POLY1305_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CHACHA20_POLY1305_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_CHACHA20_POLY1305_SHA256, TLS_KEY_EXCH_PSK, NULL, CIPHER_MODE_CHACHA20_POLY1305, NULL, SHA256_HASH_ALGO, 0, 32, 12, 0, 16, 12),
#endif

//TLS_PSK_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_PSK_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256, TLS_KEY_EXCH_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384, TLS_KEY_EXCH_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_PSK_WITH_ARIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_ARIA_128_GCM_SHA256, TLS_KEY_EXCH_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_PSK_WITH_ARIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_ARIA_256_GCM_SHA384, TLS_KEY_EXCH_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_PSK_WITH_AES_128_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_128_CCM, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_PSK_WITH_AES_256_CCM cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_256_CCM, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_PSK_WITH_AES_128_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_128_CCM_8, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 8, 12),
#endif

//TLS_PSK_WITH_AES_256_CCM_8 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CCM_8_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_256_CCM_8, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CCM, NULL, SHA256_HASH_ALGO, 0, 32, 4, 8, 8, 12),
#endif

//TLS_PSK_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_PSK_WITH_AES_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_256_CBC_SHA384, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384, TLS_KEY_EXCH_PSK, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_PSK_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_PSK_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_PSK, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_PSK_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_PSK_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_PSK, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_PSK_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_PSK, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_ECDHE_ECDSA_WITH_RC4_128_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_RC4_128_SHA, TLS_KEY_EXCH_ECDHE_ECDSA, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, SHA1_HASH_ALGO, NULL, 20, 16, 0, 0, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_RC4_128_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_RC4_128_SHA, TLS_KEY_EXCH_ECDHE_RSA, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, SHA1_HASH_ALGO, NULL, 20, 16, 0, 0, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_RC4_128_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_RC4_128_SHA, TLS_KEY_EXCH_ECDHE_PSK, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, SHA1_HASH_ALGO, NULL, 20, 16, 0, 0, 0, 12),
#endif

//TLS_DHE_PSK_WITH_RC4_128_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_RC4_128_SHA, TLS_KEY_EXCH_DHE_PSK, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, SHA1_HASH_ALGO, NULL, 20, 16, 0, 0, 0, 12),
#endif

//TLS_RSA_PSK_WITH_RC4_128_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_RC4_128_SHA, TLS_KEY_EXCH_RSA_PSK, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, SHA1_HASH_ALGO, NULL, 20, 16, 0, 0, 0, 12),
#endif

//TLS_RSA_WITH_RC4_128_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_RC4_128_SHA, TLS_KEY_EXCH_RSA, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, SHA1_HASH_ALGO, NULL, 20, 16, 0, 0, 0, 12),
#endif

//TLS_RSA_WITH_RC4_128_MD5 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_MD5_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_RC4_128_MD5, TLS_KEY_EXCH_RSA, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, MD5_HASH_ALGO, NULL, 16, 16, 0, 0, 0, 12),
#endif

//TLS_PSK_WITH_RC4_128_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_RC4_128_SHA, TLS_KEY_EXCH_PSK, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, SHA1_HASH_ALGO, NULL, 20, 16, 0, 0, 0, 12),
#endif

//TLS_DHE_RSA_WITH_DES_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_RSA_WITH_DES_CBC_SHA, TLS_KEY_EXCH_DHE_RSA, DES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 8, 8, 8, 0, 12),
#endif

//TLS_DHE_DSS_WITH_DES_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_DSS_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_DSS_WITH_DES_CBC_SHA, TLS_KEY_EXCH_DHE_DSS, DES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 8, 8, 8, 0, 12),
#endif

//TLS_RSA_WITH_DES_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_DES_CBC_SHA, TLS_KEY_EXCH_RSA, DES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 8, 8, 8, 0, 12),
#endif

//TLS_RSA_WITH_IDEA_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_IDEA_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_IDEA_CBC_SHA, TLS_KEY_EXCH_RSA, IDEA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 8, 8, 0, 12),
#endif

//TLS_ECDH_anon_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDH_ANON_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_ECDH_ANON, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_ECDH_anon_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDH_ANON_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_ECDH_ANON, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDH_ANON_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_ECDH_ANON, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_ECDH_anon_WITH_RC4_128_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDH_ANON_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDH_ANON_WITH_RC4_128_SHA, TLS_KEY_EXCH_ECDH_ANON, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, SHA1_HASH_ALGO, NULL, 20, 16, 0, 0, 0, 12),
#endif

//TLS_DH_anon_WITH_AES_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_AES_128_GCM_SHA256, TLS_KEY_EXCH_DH_ANON, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DH_anon_WITH_AES_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_AES_256_GCM_SHA384, TLS_KEY_EXCH_DH_ANON, AES_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DH_anon_WITH_CAMELLIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_CAMELLIA_128_GCM_SHA256, TLS_KEY_EXCH_DH_ANON, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DH_anon_WITH_CAMELLIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_CAMELLIA_256_GCM_SHA384, TLS_KEY_EXCH_DH_ANON, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DH_anon_WITH_ARIA_128_GCM_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_ARIA_128_GCM_SHA256, TLS_KEY_EXCH_DH_ANON, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA256_HASH_ALGO, 0, 16, 4, 8, 16, 12),
#endif

//TLS_DH_anon_WITH_ARIA_256_GCM_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_GCM_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_ARIA_256_GCM_SHA384, TLS_KEY_EXCH_DH_ANON, ARIA_CIPHER_ALGO, CIPHER_MODE_GCM, NULL, SHA384_HASH_ALGO, 0, 32, 4, 8, 16, 12),
#endif

//TLS_DH_anon_WITH_AES_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_AES_128_CBC_SHA256, TLS_KEY_EXCH_DH_ANON, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_AES_256_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_AES_256_CBC_SHA256, TLS_KEY_EXCH_DH_ANON, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 32, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA256, TLS_KEY_EXCH_DH_ANON, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA256, TLS_KEY_EXCH_DH_ANON, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 32, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_ARIA_128_CBC_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_128_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_ARIA_128_CBC_SHA256, TLS_KEY_EXCH_DH_ANON, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 16, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_ARIA_256_CBC_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_ARIA_256_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_ARIA_256_CBC_SHA384, TLS_KEY_EXCH_DH_ANON, ARIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 32, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_AES_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_AES_128_CBC_SHA, TLS_KEY_EXCH_DH_ANON, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_AES_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_AES_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_AES_256_CBC_SHA, TLS_KEY_EXCH_DH_ANON, AES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_128_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA, TLS_KEY_EXCH_DH_ANON, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_CAMELLIA_256_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA, TLS_KEY_EXCH_DH_ANON, CAMELLIA_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 32, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_SEED_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_SEED_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_SEED_CBC_SHA, TLS_KEY_EXCH_DH_ANON, SEED_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 16, 16, 16, 0, 12),
#endif

//TLS_DH_anon_WITH_3DES_EDE_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_3DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_3DES_EDE_CBC_SHA, TLS_KEY_EXCH_DH_ANON, DES3_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 24, 8, 8, 0, 12),
#endif

//TLS_DH_anon_WITH_DES_CBC_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_CBC_CIPHER_SUPPORT == ENABLED && TLS_DES_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_DES_CBC_SHA, TLS_KEY_EXCH_DH_ANON, DES_CIPHER_ALGO, CIPHER_MODE_CBC, SHA1_HASH_ALGO, NULL, 20, 8, 8, 8, 0, 12),
#endif

//TLS_DH_anon_WITH_RC4_128_MD5 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DH_ANON_KE_SUPPORT == ENABLED && TLS_STREAM_CIPHER_SUPPORT == ENABLED && TLS_RC4_SUPPORT == ENABLED && TLS_MD5_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DH_ANON_WITH_RC4_128_MD5, TLS_KEY_EXCH_DH_ANON, RC4_CIPHER_ALGO, CIPHER_MODE_STREAM, MD5_HASH_ALGO, NULL, 16, 16, 0, 0, 0, 12),
#endif

//TLS_SHA256_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_3 && TLS_MIN_VERSION <= TLS_VERSION_1_3 && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_SHA256_SHA256, TLS_KEY_EXCH_NONE, NULL, CIPHER_MODE_NULL, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 0, 32, 12, 0, 0, 32),
#endif

//TLS_SHA384_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_3 && TLS_MIN_VERSION <= TLS_VERSION_1_3 && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_SHA384_SHA384, TLS_KEY_EXCH_NONE, NULL, CIPHER_MODE_NULL, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 0, 48, 12, 0, 0, 48),
#endif

//TLS_ECDHE_ECDSA_WITH_NULL_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_ECDSA_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_ECDSA_WITH_NULL_SHA, TLS_KEY_EXCH_ECDHE_ECDSA, NULL, CIPHER_MODE_NULL, SHA1_HASH_ALGO, NULL, 20, 0, 0, 0, 0, 12),
#endif

//TLS_ECDHE_RSA_WITH_NULL_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_RSA_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_RSA_WITH_NULL_SHA, TLS_KEY_EXCH_ECDHE_RSA, NULL, CIPHER_MODE_NULL, SHA1_HASH_ALGO, NULL, 20, 0, 0, 0, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_NULL_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_NULL_SHA256, TLS_KEY_EXCH_ECDHE_PSK, NULL, CIPHER_MODE_NULL, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 0, 0, 0, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_NULL_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_NULL_SHA384, TLS_KEY_EXCH_ECDHE_PSK, NULL, CIPHER_MODE_NULL, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 0, 0, 0, 0, 12),
#endif

//TLS_ECDHE_PSK_WITH_NULL_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDHE_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDHE_PSK_WITH_NULL_SHA, TLS_KEY_EXCH_ECDHE_PSK, NULL, CIPHER_MODE_NULL, SHA1_HASH_ALGO, NULL, 20, 0, 0, 0, 0, 12),
#endif

//TLS_DHE_PSK_WITH_NULL_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_NULL_SHA256, TLS_KEY_EXCH_DHE_PSK, NULL, CIPHER_MODE_NULL, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 0, 0, 0, 0, 12),
#endif

//TLS_DHE_PSK_WITH_NULL_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_NULL_SHA384, TLS_KEY_EXCH_DHE_PSK, NULL, CIPHER_MODE_NULL, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 0, 0, 0, 0, 12),
#endif

//TLS_DHE_PSK_WITH_NULL_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_DHE_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_DHE_PSK_WITH_NULL_SHA, TLS_KEY_EXCH_DHE_PSK, NULL, CIPHER_MODE_NULL, SHA1_HASH_ALGO, NULL, 20, 0, 0, 0, 0, 12),
#endif

//TLS_RSA_PSK_WITH_NULL_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_NULL_SHA256, TLS_KEY_EXCH_RSA_PSK, NULL, CIPHER_MODE_NULL, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 0, 0, 0, 0, 12),
#endif

//TLS_RSA_PSK_WITH_NULL_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_NULL_SHA384, TLS_KEY_EXCH_RSA_PSK, NULL, CIPHER_MODE_NULL, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 0, 0, 0, 0, 12),
#endif

//TLS_RSA_PSK_WITH_NULL_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_PSK_WITH_NULL_SHA, TLS_KEY_EXCH_RSA_PSK, NULL, CIPHER_MODE_NULL, SHA1_HASH_ALGO, NULL, 20, 0, 0, 0, 0, 12),
#endif

//TLS_RSA_WITH_NULL_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_NULL_SHA256, TLS_KEY_EXCH_RSA, NULL, CIPHER_MODE_NULL, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 0, 0, 0, 0, 12),
#endif

//TLS_RSA_WITH_NULL_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_NULL_SHA, TLS_KEY_EXCH_RSA, NULL, CIPHER_MODE_NULL, SHA1_HASH_ALGO, NULL, 20, 0, 0, 0, 0, 12),
#endif

//TLS_RSA_WITH_NULL_MD5 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_RSA_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_MD5_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_RSA_WITH_NULL_MD5, TLS_KEY_EXCH_RSA, NULL, CIPHER_MODE_NULL, MD5_HASH_ALGO, NULL, 16, 0, 0, 0, 0, 12),
#endif

//TLS_PSK_WITH_NULL_SHA256 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA256_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_NULL_SHA256, TLS_KEY_EXCH_PSK, NULL, CIPHER_MODE_NULL, SHA256_HASH_ALGO, SHA256_HASH_ALGO, 32, 0, 0, 0, 0, 12),
#endif

//TLS_PSK_WITH_NULL_SHA384 cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_2 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA384_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_NULL_SHA384, TLS_KEY_EXCH_PSK, NULL, CIPHER_MODE_NULL, SHA384_HASH_ALGO, SHA384_HASH_ALGO, 48, 0, 0, 0, 0, 12),
#endif

//TLS_PSK_WITH_NULL_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_PSK_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_PSK_WITH_NULL_SHA, TLS_KEY_EXCH_PSK, NULL, CIPHER_MODE_NULL, SHA1_HASH_ALGO, NULL, 20, 0, 0, 0, 0, 12),
#endif

//TLS_ECDH_anon_WITH_NULL_SHA cipher suite
#if (TLS_MAX_VERSION >= TLS_VERSION_1_0 && TLS_MIN_VERSION <= TLS_VERSION_1_2 && TLS_ECDH_ANON_KE_SUPPORT == ENABLED && TLS_NULL_CIPHER_SUPPORT == ENABLED && TLS_SHA1_SUPPORT == ENABLED)
   TLS_CIPHER_SUITE(TLS_ECDH_ANON_WITH_NULL_SHA, TLS_KEY_EXCH_ECDH_ANON, NULL, CIPHER_MODE_NULL, SHA1_HASH_ALGO, NULL, 20, 0, 0, 0, 0, 12),
#endif
};


/**
 * @brief Determine the number of cipher suites supported
 * @return Number of supported cipher suites
 **/

uint_t tlsGetNumSupportedCipherSuites(void)
{
   //Return the number of supported cipher suites
   return arraysize(tlsSupportedCipherSuites);
}


/**
 * @brief Convert cipher suite identifier to string representation
 * @param[in] identifier Cipher suite identifier
 * @return Cipher suite name
 **/

const char_t *tlsGetCipherSuiteName(uint16_t identifier)
{
   uint_t i;

   //Default name for unknown cipher suites
   static const char_t defaultName[] = "Unknown";

   //Parse the list of supported cipher suites
   for(i = 0; i < arraysize(tlsSupportedCipherSuites); i++)
   {
      //The current cipher suite matches the specified identifier?
      if(tlsSupportedCipherSuites[i].identifier == identifier)
         return tlsSupportedCipherSuites[i].name;
   }

   //Unknown cipher suite
   return defaultName;
}


/**
 * @brief Check whether a cipher suite can be used with a given protocol
 *   version
 * @param[in] cipherSuite Cipher suite
 * @param[in] minVersion Minimum TLS protocol version
 * @param[in] maxVersion Maximum TLS protocol version
 * @param[in] transportProtocol Transport protocol (TLS or DTLS)
 * @return TRUE if the cipher suite is acceptable, else FALSE
 **/

bool_t tlsIsCipherSuiteAcceptable(const TlsCipherSuiteInfo *cipherSuite,
   uint16_t minVersion, uint16_t maxVersion,
   TlsTransportProtocol transportProtocol)
{
   bool_t acceptable;

   //Initialize flag
   acceptable = TRUE;

   //TLS 1.2 cipher suites must not be negotiated in older versions of TLS
   if(maxVersion < TLS_VERSION_1_2)
   {
      //TLS 1.2 cipher suite?
      if(cipherSuite->prfHashAlgo != NULL)
         acceptable = FALSE;
   }

   //Although TLS 1.3 uses the same cipher suite space as previous versions
   //of TLS, TLS 1.3 cipher suites are defined differently, only specifying
   //the symmetric ciphers, and cannot be used for TLS 1.2
   if(maxVersion < TLS_VERSION_1_3)
   {
      //TLS 1.3 cipher suite?
      if(cipherSuite->keyExchMethod == TLS_KEY_EXCH_NONE)
         acceptable = FALSE;
   }

   //Similarly, TLS 1.2 and lower cipher suites cannot be used with TLS 1.3
   if(minVersion > TLS_VERSION_1_2)
   {
      //TLS 1.2 and lower cipher suite?
      if(cipherSuite->keyExchMethod != TLS_KEY_EXCH_NONE)
         acceptable = FALSE;
   }

   //The only stream cipher described in TLS 1.2 is RC4, which cannot be
   //randomly accessed. RC4 must not be used with DTLS
   if(transportProtocol == TLS_TRANSPORT_PROTOCOL_DATAGRAM)
   {
      //RC4 stream cipher?
      if(cipherSuite->cipherMode == CIPHER_MODE_STREAM)
         acceptable = FALSE;
   }

   //Return TRUE is the cipher suite can be used with the specified protocol
   //version
   return acceptable;
}


/**
 * @brief Check whether the specified identifier matches an ECC or FFDHE
 *   cipher suite
 * @param[in] identifier Cipher suite identifier
 * @return Cipher suite type
 **/

TlsCipherSuiteType tlsGetCipherSuiteType(uint16_t identifier)
{
   uint_t i;
   TlsCipherSuiteType type;

   //Initialize type
   type = TLS_CIPHER_SUITE_TYPE_UNKNOWN;

   //Loop through the list of supported cipher suites
   for(i = 0; i < arraysize(tlsSupportedCipherSuites); i++)
   {
      //Compare cipher suite identifier against the specified value
      if(tlsSupportedCipherSuites[i].identifier == identifier)
      {
         //Check key exchange mechanism
         switch(tlsSupportedCipherSuites[i].keyExchMethod)
         {
         case TLS_KEY_EXCH_ECDH_ANON:
         case TLS_KEY_EXCH_ECDHE_RSA:
         case TLS_KEY_EXCH_ECDHE_ECDSA:
         case TLS_KEY_EXCH_ECDHE_PSK:
            //ECC cipher suite
            type = TLS_CIPHER_SUITE_TYPE_ECC;
            break;
         case TLS_KEY_EXCH_DH_ANON:
         case TLS_KEY_EXCH_DHE_RSA:
         case TLS_KEY_EXCH_DHE_DSS:
         case TLS_KEY_EXCH_DHE_PSK:
            //FFDHE cipher suite
            type = TLS_CIPHER_SUITE_TYPE_FFDHE;
            break;
         case TLS_KEY_EXCH_NONE:
            //TLS 1.3 cipher suite
            type = TLS_CIPHER_SUITE_TYPE_TLS13;
            break;
         default:
            //Just for sanity
            break;
         }

         //We are done
         break;
      }
   }

   //Return cipher suite type
   return type;
}

#endif
