/*
** ###################################################################
**     Processor:           MPC5748G
**     Reference manual:    MPC5748G RM Rev. 6, 10/2017
**     Version:             rev. 1.4, 2018-01-15
**     Build:               b180116
**
**     Abstract:
**         Peripheral Access Layer for MPC5748G
**
**     Copyright (c) 1997 - 2016 Freescale Semiconductor, Inc.
**     Copyright 2016-2018 NXP
**     All rights reserved.
**
**     THIS SOFTWARE IS PROVIDED BY NXP "AS IS" AND ANY EXPRESSED OR
**     IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
**     OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**     IN NO EVENT SHALL NXP OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
**     INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
**     (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
**     SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
**     HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
**     STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
**     IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
**     THE POSSIBILITY OF SUCH DAMAGE.
**
**     http:                 www.nxp.com
**     mail:                 support@nxp.com
**
**     Revisions:
**     - rev. 1.0 (2016-11-24) - Iulian Talpiga
**         Initial version.
**     - rev. 1.1 (2017-04-04) - Iulian Talpiga
**         Fixed FXOSC register name.
**     - rev. 1.2 (2017-07-13) - Andrei Bolojan
**         Removed second SMPU instance according to the RM.
**         Compacted PCTL registers into array.
**         Fixed MC_ME, FXOSC, uSDHC mismatches with RM.
**         Fixed PLLFM Bit Fields in PLLDIG and MFDEN width.
**     - rev. 1.3 (2017-09-20) - Andrei Bolojan
**         Added INTC interrupt sources according to MPC5748G RM Rev. 5, 12/2016
**     - rev. 1.4 (2018-01-15) - Mihai Volmer
**         Regenerated the header according to RM MPC5748G RM Rev. 6, 10/2017
**         Updated the following peripherals: MLB, SMPU, STCU2, uSDHC
**         Removed one ESW instance
**
** ###################################################################
*/

/*!
 * @file MPC5748G.h
 * @version 1.4
 * @date 2018-01-15
 * @brief Peripheral Access Layer for MPC5748G
 *
 * This file contains register definitions and macros for easy access to their
 * bit fields.
 *
 * This file assumes BIG endian system.
 */

/**
* @page misra_violations MISRA-C:2012 violations
*
* @section [global]
* Violates MISRA 2012 Advisory Rule 2.3, local typedef not referenced
* The SoC header defines typedef for all modules.
*
* @section [global]
* Violates MISRA 2012 Advisory Rule 2.5, local macro not referenced
* The SoC header defines macros for all modules and registers.
*
* @section [global]
* Violates MISRA 2012 Advisory Directive 4.9, Function-like macro
* These are generated macros used for accessing the bit-fields from registers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 5.1, identifier clash
* The supported compilers use more than 31 significant characters for identifiers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 5.2, identifier clash
* The supported compilers use more than 31 significant characters for identifiers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 5.4, identifier clash
* The supported compilers use more than 31 significant characters for identifiers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 5.5, identifier clash
* The supported compilers use more than 31 significant characters for identifiers.
*
* @section [global]
* Violates MISRA 2012 Required Rule 21.1, defined macro '__I' is reserved to the compiler
* This type qualifier is needed to ensure correct I/O access and addressing.
*/

/* ----------------------------------------------------------------------------
   -- MCU activation
   ---------------------------------------------------------------------------- */

/* Prevention from multiple including the same memory map */
#if !defined(MPC5748G_H_)  /* Check if memory map has not been already included */
#define MPC5748G_H_
#define MCU_MPC5748G

/* Check if another memory map has not been also included */
#if (defined(MCU_ACTIVE))
  #error MPC5748G memory map: There is already included another memory map. Only one memory map can be included.
#endif /* (defined(MCU_ACTIVE)) */
#define MCU_ACTIVE

#include <stdint.h>

/** Memory map major version (memory maps with equal major version number are
 * compatible) */
#define MCU_MEM_MAP_VERSION 0x0100u
/** Memory map minor version */
#define MCU_MEM_MAP_VERSION_MINOR 0x0004u

/* ----------------------------------------------------------------------------
   -- Generic macros
   ---------------------------------------------------------------------------- */

/* IO definitions (access restrictions to peripheral registers) */
/**
*   IO Type Qualifiers are used
*   \li to specify the access to peripheral variables.
*   \li for automatic generation of peripheral register debug information.
*/
#ifndef __IO
#ifdef __cplusplus
  #define   __I     volatile             /*!< Defines 'read only' permissions                 */
#else
  #define   __I     volatile const       /*!< Defines 'read only' permissions                 */
#endif
#define     __O     volatile             /*!< Defines 'write only' permissions                */
#define     __IO    volatile             /*!< Defines 'read / write' permissions              */
#endif


/**
* @brief 32 bits memory read macro.
*/
#if !defined(REG_READ32)
  #define REG_READ32(address)               (*(volatile uint32_t*)(address))
#endif

/**
* @brief 32 bits memory write macro.
*/
#if !defined(REG_WRITE32)
  #define REG_WRITE32(address, value)       ((*(volatile uint32_t*)(address))= (uint32_t)(value))
#endif

/**
* @brief 32 bits bits setting macro.
*/
#if !defined(REG_BIT_SET32)
  #define REG_BIT_SET32(address, mask)      ((*(volatile uint32_t*)(address))|= (uint32_t)(mask))
#endif

/**
* @brief 32 bits bits clearing macro.
*/
#if !defined(REG_BIT_CLEAR32)
  #define REG_BIT_CLEAR32(address, mask)    ((*(volatile uint32_t*)(address))&= ((uint32_t)~((uint32_t)(mask))))
#endif

/**
* @brief 32 bit clear bits and set with new value
* @note It is user's responsability to make sure that value has only "mask" bits set - (value&~mask)==0
*/
#if !defined(REG_RMW32)
  #define REG_RMW32(address, mask, value)   (REG_WRITE32((address), ((REG_READ32(address)& ((uint32_t)~((uint32_t)(mask))))| ((uint32_t)(value)))))
#endif


/* ----------------------------------------------------------------------------
   -- Interrupt vector numbers for MPC5748G
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup Interrupt_vector_numbers_MPC5748G Interrupt vector numbers for MPC5748G
 * @{
 */

/** Interrupt Number Definitions */
#define NUMBER_OF_INT_VECTORS 754u               /**< Number of interrupts in the Vector table */

/**
 * @brief Defines the Interrupt Numbers definitions
 *
 * This enumeration is used to configure the interrupts.
 *
 * Implements : IRQn_Type_Class
 */
typedef enum
{
  /* Auxiliary constants */
  NotAvail_IRQn                = -128,             /**< Not available device specific interrupt */

  /* Core interrupts */

  /* Device specific interrupts */
  SS0_IRQn                     = 0u,               /**< Software setable flag 0 SSCIR0[CLR0] */
  SS1_IRQn                     = 1u,               /**< Software setable flag 1 SSCIR0[CLR1] */
  SS2_IRQn                     = 2u,               /**< Software setable flag 2 SSCIR0[CLR2] */
  SS3_IRQn                     = 3u,               /**< Software setable flag 3 SSCIR0[CLR3] */
  SS4_IRQn                     = 4u,               /**< Software setable flag 4 SSCIR0[CLR4] */
  SS5_IRQn                     = 5u,               /**< Software setable flag 5 SSCIR0[CLR5] */
  SS6_IRQn                     = 6u,               /**< Software setable flag 6 SSCIR0[CLR6] */
  SS7_IRQn                     = 7u,               /**< Software setable flag 7 SSCIR0[CLR7] */
  SS8_IRQn                     = 8u,               /**< Software setable flag 8 SSCIR0[CLR8] */
  SS9_IRQn                     = 9u,               /**< Software setable flag 9 SSCIR0[CLR9] */
  SS10_IRQn                    = 10u,              /**< Software setable flag 10 SSCIR0[CLR10] */
  SS11_IRQn                    = 11u,              /**< Software setable flag 11 SSCIR0[CLR11] */
  SS12_IRQn                    = 12u,              /**< Software setable flag 12 SSCIR0[CLR12] */
  SS13_IRQn                    = 13u,              /**< Software setable flag 13 SSCIR0[CLR13] */
  SS14_IRQn                    = 14u,              /**< Software setable flag 14 SSCIR0[CLR14] */
  SS15_IRQn                    = 15u,              /**< Software setable flag 15 SSCIR0[CLR15] */
  SS16_IRQn                    = 16u,              /**< Software setable flag 16 SSCIR0[CLR16] */
  SS17_IRQn                    = 17u,              /**< Software setable flag 17 SSCIR0[CLR17] */
  SS18_IRQn                    = 18u,              /**< Software setable flag 18 SSCIR0[CLR18] */
  SS19_IRQn                    = 19u,              /**< Software setable flag 19 SSCIR0[CLR19] */
  SS20_IRQn                    = 20u,              /**< Software setable flag 20 SSCIR0[CLR20] */
  SS21_IRQn                    = 21u,              /**< Software setable flag 21 SSCIR0[CLR21] */
  SS22_IRQn                    = 22u,              /**< Software setable flag 22 SSCIR0[CLR22] */
  SS23_IRQn                    = 23u,              /**< Software setable flag 23 SSCIR0[CLR23] */
  SWT_0_IRQn                   = 32u,              /**< Platform watchdog timer0 SWT_0_IR[TIF] */
  SWT_1_IRQn                   = 33u,              /**< Platform watchdog timer1 SWT_1_IR[TIF] */
  SWT_2_IRQn                   = 34u,              /**< Platform watchdog timer2 SWT_2_IR[TIF] */
  STM0_Ch0_IRQn                = 36u,              /**< STM0 Channel 0 interrupt STM_0_CIR0[CIF] */
  STM0_Ch1_IRQn                = 37u,              /**< STM0 Channel 1 interrupt STM_0_CIR1[CIF] */
  STM0_Ch2_IRQn                = 38u,              /**< STM0 Channel 2 interrupt STM_0_CIR2[CIF] */
  STM0_Ch3_IRQn                = 39u,              /**< STM0 Channel 3 interrupt STM_0_CIR3[CIF] */
  STM1_Ch0_IRQn                = 40u,              /**< STM1 Channel 0 interrupt STM_1_CIR0[CIF] */
  STM1_Ch1_IRQn                = 41u,              /**< STM1 Channel 1 interrupt STM_1_CIR1[CIF] */
  STM1_Ch2_IRQn                = 42u,              /**< STM1 Channel 2 interrupt STM_1_CIR2[CIF] */
  STM1_Ch3_IRQn                = 43u,              /**< STM1 Channel 3 interrupt STM_1_CIR3[CIF] */
  STM2_Ch0_IRQn                = 44u,              /**< STM2 Channel 0 interrupt STM_2_CIR0[CIF] */
  STM2_Ch1_IRQn                = 45u,              /**< STM2 Channel 1 interrupt STM_2_CIR1[CIF] */
  STM2_Ch2_IRQn                = 46u,              /**< STM2 Channel 2 interrupt STM_2_CIR2[CIF] */
  STM2_Ch3_IRQn                = 47u,              /**< STM2 Channel 3 interrupt STM_2_CIR3[CIF] */
  DMA_Error_IRQn               = 52u,              /**< eDMA Combined Error eDMA Channel Error Flags 63-0 */
  DMA0_IRQn                    = 53u,              /**< eDMA Channel 0 DMA_INTL[INT0] */
  DMA1_IRQn                    = 54u,              /**< eDMA Channel 1 DMA_INTL[INT1] */
  DMA2_IRQn                    = 55u,              /**< eDMA Channel 2 DMA_INTL[INT2] */
  DMA3_IRQn                    = 56u,              /**< eDMA Channel 3 DMA_INTL[INT3] */
  DMA4_IRQn                    = 57u,              /**< eDMA Channel 4 DMA_INTL[INT4] */
  DMA5_IRQn                    = 58u,              /**< eDMA Channel 5 DMA_INTL[INT5] */
  DMA6_IRQn                    = 59u,              /**< eDMA Channel 6 DMA_INTL[INT6] */
  DMA7_IRQn                    = 60u,              /**< eDMA Channel 7 DMA_INTL[INT7] */
  DMA8_IRQn                    = 61u,              /**< eDMA Channel 8 DMA_INTL[INT8] */
  DMA9_IRQn                    = 62u,              /**< eDMA Channel 9 DMA_INTL[INT9] */
  DMA10_IRQn                   = 63u,              /**< eDMA Channel 10 DMA_INTL[INT10] */
  DMA11_IRQn                   = 64u,              /**< eDMA Channel 11 DMA_INTL[INT11] */
  DMA12_IRQn                   = 65u,              /**< eDMA Channel 12 DMA_INTL[INT12] */
  DMA13_IRQn                   = 66u,              /**< eDMA Channel 13 DMA_INTL[INT13] */
  DMA14_IRQn                   = 67u,              /**< eDMA Channel 14 DMA_INTL[INT14] */
  DMA15_IRQn                   = 68u,              /**< eDMA Channel 15 DMA_INTL[INT15] */
  DMA16_IRQn                   = 69u,              /**< eDMA Channel 16 DMA_INTL[INT16] */
  DMA17_IRQn                   = 70u,              /**< eDMA Channel 17 DMA_INTL[INT17] */
  DMA18_IRQn                   = 71u,              /**< eDMA Channel 18 DMA_INTL[INT18] */
  DMA19_IRQn                   = 72u,              /**< eDMA Channel 19 DMA_INTL[INT19] */
  DMA20_IRQn                   = 73u,              /**< eDMA Channel 20 DMA_INTL[INT20] */
  DMA21_IRQn                   = 74u,              /**< eDMA Channel 21 DMA_INTL[INT21] */
  DMA22_IRQn                   = 75u,              /**< eDMA Channel 22 DMA_INTL[INT22] */
  DMA23_IRQn                   = 76u,              /**< eDMA Channel 23 DMA_INTL[INT23] */
  DMA24_IRQn                   = 77u,              /**< eDMA Channel 24 DMA_INTL[INT24] */
  DMA25_IRQn                   = 78u,              /**< eDMA Channel 25 DMA_INTL[INT25] */
  DMA26_IRQn                   = 79u,              /**< eDMA Channel 26 DMA_INTL[INT26] */
  DMA27_IRQn                   = 80u,              /**< eDMA Channel 27 DMA_INTL[INT27] */
  DMA28_IRQn                   = 81u,              /**< eDMA Channel 28 DMA_INTL[INT28] */
  DMA29_IRQn                   = 82u,              /**< eDMA Channel 29 DMA_INTL[INT29] */
  DMA30_IRQn                   = 83u,              /**< eDMA Channel 30 DMA_INTL[INT30] */
  DMA31_IRQn                   = 84u,              /**< eDMA Channel 31 DMA_INTL[INT31] */
  FMC_Done_IRQn                = 185u,             /**< Flash controller Prog/Erase/Suspend IRQ_0 MCR[DONE] */
  ESW_0_IRQn                   = 190u,             /**< Switch Interrupt Group0 ESW_0 */
  ESW_1_IRQn                   = 191u,             /**< Switch Interrupt Group1 ESW_1 */
  ESW_2_IRQn                   = 192u,             /**< Switch Interrupt Group2 ESW_2 */
  ESW_3_IRQn                   = 193u,             /**< Switch Interrupt Group3 ESW_3 */
  ENET1_GROUP0_IRQn            = 201u,             /**< ENET 1 Interrupt Group0 ENET_0 */
  ENET1_GROUP1_IRQn            = 202u,             /**< ENET 1 Interrupt Group1 ENET_1 */
  ENET1_GROUP2_IRQn            = 203u,             /**< ENET 1 Interrupt Group2 ENET_2 */
  ENET1_GROUP3_IRQn            = 204u,             /**< ENET 1 Interrupt Group3 ENET_3 */
  ENET1_GROUP4_IRQn            = 205u,             /**< ENET 1 Interrupt Group4 ENET_4 */
  ENET1_GROUP5_IRQn            = 206u,             /**< ENET 1 Interrupt Group5 ENET_5 */
  ENET1_GROUP6_IRQn            = 207u,             /**< ENET 1 Interrupt Group6 ENET_6 */
  ENET1_GROUP7_IRQn            = 208u,             /**< ENET 1 Interrupt Group7 ENET_7 */
  ENET1_GROUP8_IRQn            = 209u,             /**< ENET 1 Interrupt Group8 ENET_8 */
  ENET0_GROUP0_IRQn            = 210u,             /**< ENET 0 Interrupt Group0 ENET_0 */
  ENET0_GROUP1_IRQn            = 211u,             /**< ENET 0 Interrupt Group1 ENET_1 */
  ENET0_GROUP2_IRQn            = 212u,             /**< ENET 0 Interrupt Group2 ENET_2 */
  ENET0_GROUP3_IRQn            = 213u,             /**< ENET 0 Interrupt Group3 ENET_3 */
  ENET0_GROUP4_IRQn            = 214u,             /**< ENET 0 Interrupt Group4 ENET_4 */
  ENET0_GROUP5_IRQn            = 215u,             /**< ENET 0 Interrupt Group5 ENET_5 */
  ENET0_GROUP6_IRQn            = 216u,             /**< ENET 0 Interrupt Group6 ENET_6 */
  ENET0_GROUP7_IRQn            = 217u,             /**< ENET 0 Interrupt Group7 ENET_7 */
  ENET0_GROUP8_IRQn            = 218u,             /**< ENET 0 Interrupt Group8 ENET_8 */
  RTC0_IRQn                    = 224u,             /**< Real Time Counter (RTC) RTC */
  API0_IRQn                    = 225u,             /**< Autonomous Periodic Interrupt (API) API */
  PIT_Ch0_IRQn                 = 226u,             /**< Periodic Interrupt Timer (PIT0) PIT_1_TFLG0[TIF] */
  PIT_Ch1_IRQn                 = 227u,             /**< Periodic Interrupt Timer (PIT1) PIT_1_TFLG1[TIF] */
  PIT_Ch2_IRQn                 = 228u,             /**< Periodic Interrupt Timer (PIT2) PIT_1_TFLG2[TIF] */
  PIT_Ch3_IRQn                 = 229u,             /**< Periodic Interrupt Timer (PIT3) PIT_1_TFLG3[TIF] */
  PIT_Ch4_IRQn                 = 230u,             /**< Periodic Interrupt Timer (PIT4) PIT_1_TFLG4[TIF] */
  PIT_Ch5_IRQn                 = 231u,             /**< Periodic Interrupt Timer (PIT5) PIT_1_TFLG5[TIF] */
  PIT_Ch6_IRQn                 = 232u,             /**< Periodic Interrupt Timer (PIT6) PIT_1_TFLG6[TIF] */
  PIT_Ch7_IRQn                 = 233u,             /**< Periodic Interrupt Timer (PIT7) PIT_1_TFLG7[TIF] */
  PIT_Ch8_IRQn                 = 234u,             /**< Periodic Interrupt Timer (PIT8) PIT_1_TFLG8[TIF] */
  PIT_Ch9_IRQn                 = 235u,             /**< Periodic Interrupt Timer (PIT9) PIT_1_TFLG9[TIF] */
  PIT_Ch10_IRQn                = 236u,             /**< Periodic Interrupt Timer (PIT10) PIT_1_TFLG10[TIF] */
  PIT_Ch11_IRQn                = 237u,             /**< Periodic Interrupt Timer (PIT11) PIT_1_TFLG11[TIF] */
  PIT_Ch12_IRQn                = 238u,             /**< Periodic Interrupt Timer (PIT12) PIT_1_TFLG12[TIF] */
  PIT_Ch13_IRQn                = 239u,             /**< Periodic Interrupt Timer (PIT13) PIT_1_TFLG13[TIF] */
  PIT_Ch14_IRQn                = 240u,             /**< Periodic Interrupt Timer (PIT14) PIT_1_TFLG14[TIF] */
  PIT_Ch15_IRQn                = 241u,             /**< Periodic Interrupt Timer (PIT15) PIT_1_TFLG15[TIF] */
  PIT_RTI_IRQn                 = 242u,             /**< PIT_RTI PIT_RTI */
  SIUL_EIRQ_00_07_IRQn         = 243u,             /**< SIUL EIRQ [0-7] - SIUL Combined External Interrupt */
  SIUL_EIRQ_08_15_IRQn         = 244u,             /**< SIUL EIRQ [8-15] - SIUL Combined External Interrupt */
  SIUL_EIRQ_16_23_IRQn         = 245u,             /**< SIUL EIRQ [16-23] - SIUL Combined External Interrupt */
  SIUL_EIRQ_24_31_IRQn         = 246u,             /**< SIUL EIRQ [24-31] - SIUL Combined External Interrupt */
  LPU_IRQn                     = 250u,             /**< LPU_CTL LPU_ICR[NEM] | LPU_ICR[MRIG] */
  MC_ME_SAFE_IRQn              = 251u,             /**< MC _ME ME_IS[I_SAFE] */
  MC_ME_MTC_IRQn               = 252u,             /**< MC _ME ME_IS[I_MTC] */
  MC_ME_IMODE_IRQn             = 253u,             /**< MC _ME ME_IS[I_IMODE] */
  MC_ME_ICONF_IRQn             = 254u,             /**< MC _ME ME_IS[I_ICONF] */
  MC_RGM_IRQn                  = 255u,             /**< MC_RGM MC_RGM Functional and destructive reset alternate event interrupt */
  FXOSC_IRQn                   = 257u,             /**< FXOSC Counter FXOSC Counter */
  SXOSC_IRQn                   = 258u,             /**< SXOSC Counter SXOSC Counter */
  DSPI0_FifoErr_IRQn           = 259u,             /**< DSPI0_0 DSPI_0_SR[TFUF] | DSPI_0_SR[RFOF] | DSPI_0_SR[TFIWF] */
  DSPI0_EndOfQueue_IRQn        = 260u,             /**< DSPI0_1 DSPI_0_SR[EOQF] */
  DSPI0_TxFifoNotfull_IRQn     = 261u,             /**< DSPI0_2 DSPI_0_SR[TFFF] */
  DSPI0_TransferComplete_IRQn  = 262u,             /**< DSPI0_3 DSPI_0_SR[TCF] */
  DSPI0_RxFifoNotempty_IRQn    = 263u,             /**< DSPI0_4 DSPI_0_SR[RFDF] */
  DSPI0_EndOfCmd_IRQn          = 264u,             /**< DSPI0_5 DSPI_0_SR[SPITCF] | DSPI_0_SR[CMD_TCF] */
  DSPI0_CmdFifoNotFull_IRQn    = 265u,             /**< DSPI0_6 DSPI_0_SR[DSITCF] | DSPI_0_SR[CMDFFF] */
  DSPI0_ParityError_IRQn       = 266u,             /**< DSPI0_7 DSPI_0_SR[SPEF] | DSPI_0_SR[DPEF] */
  DSPI0_DSIDataMatch_IRQn      = 267u,             /**< DSPI0_8 DSPI_0_SR[DDIF] */
  DSPI1_FifoErr_IRQn           = 268u,             /**< DSPI1_0 DSPI_1_SR[TFUF] | DSPI_1_SR[RFOF] | DSPI_1_SR[TFIWF] */
  DSPI1_EndOfQueue_IRQn        = 269u,             /**< DSPI1_1 DSPI_1_SR[EOQF] */
  DSPI1_TxFifoNotfull_IRQn     = 270u,             /**< DSPI1_2 DSPI_1_SR[TFFF] */
  DSPI1_TransferComplete_IRQn  = 271u,             /**< DSPI1_3 DSPI_1_SR[TCF] */
  DSPI1_RxFifoNotempty_IRQn    = 272u,             /**< DSPI1_4 DSPI_1_SR[RFDF] */
  DSPI1_EndOfCmd_IRQn          = 273u,             /**< DSPI1_5 DSPI_1_SR[SPITCF] | DSPI_1_SR[CMD_TCF] */
  DSPI1_CmdFifoNotFull_IRQn    = 274u,             /**< DSPI1_6 DSPI_1_SR[DSITCF] | DSPI_1_SR[CMDFFF] */
  DSPI1_ParityError_IRQn       = 275u,             /**< DSPI1_7 DSPI_1_SR[SPEF] | DSPI_1_SR[DPEF] */
  DSPI1_DSIDataMatch_IRQn      = 276u,             /**< DSPI1_8 DSPI_1_SR[DDIF] */
  DSPI2_FifoErr_IRQn           = 277u,             /**< DSPI2_0 DSPI_2_SR[TFUF] | DSPI_2_SR[RFOF] | DSPI_2_SR[TFIWF] */
  DSPI2_EndOfQueue_IRQn        = 278u,             /**< DSPI2_1 DSPI_2_SR[EOQF] */
  DSPI2_TxFifoNotfull_IRQn     = 279u,             /**< DSPI2_2 DSPI_2_SR[TFFF] */
  DSPI2_TransferComplete_IRQn  = 280u,             /**< DSPI2_3 DSPI_2_SR[TCF] */
  DSPI2_RxFifoNotempty_IRQn    = 281u,             /**< DSPI2_4 DSPI_2_SR[RFDF] */
  DSPI2_EndOfCmd_IRQn          = 282u,             /**< DSPI2_5 DSPI_2_SR[SPITCF] | DSPI_2_SR[CMD_TCF] */
  DSPI2_CmdFifoNotFull_IRQn    = 283u,             /**< DSPI2_6 DSPI_2_SR[DSITCF] | DSPI_2_SR[CMDFFF] */
  DSPI2_ParityError_IRQn       = 284u,             /**< DSPI2_7 DSPI_2_SR[SPEF] | DSPI_2_SR[DPEF] */
  DSPI2_DSIDataMatch_IRQn      = 285u,             /**< DSPI2_8 DSPI_2_SR[DDIF] */
  DSPI3_FifoErr_IRQn           = 286u,             /**< DSPI3_0 DSPI_3_SR[TFUF] | DSPI_3_SR[RFOF] | DSPI_3_SR[TFIWF] */
  DSPI3_EndOfQueue_IRQn        = 287u,             /**< DSPI3_1 DSPI_3_SR[EOQF] */
  DSPI3_TxFifoNotfull_IRQn     = 288u,             /**< DSPI3_2 DSPI_3_SR[TFFF] */
  DSPI3_TransferComplete_IRQn  = 289u,             /**< DSPI3_3 DSPI_3_SR[TCF] */
  DSPI3_RxFifoNotempty_IRQn    = 290u,             /**< DSPI3_4 DSPI_3_SR[RFDF] */
  DSPI3_EndOfCmd_IRQn          = 291u,             /**< DSPI3_5 DSPI_3_SR[SPITCF] | DSPI_3_SR[CMD_TCF] */
  DSPI3_CmdFifoNotFull_IRQn    = 292u,             /**< DSPI3_6 DSPI_3_SR[DSITCF] | DSPI_3_SR[CMDFFF] */
  DSPI3_ParityError_IRQn       = 293u,             /**< DSPI3_7 DSPI_3_SR[SPEF] | DSPI_3_SR[DPEF] */
  DSPI3_DSIDataMatch_IRQn      = 294u,             /**< DSPI3_8 DSPI_3_SR[DDIF] */
  SPI0_FifoErr_IRQn            = 295u,             /**< SPI0_0 SPI_0_SR[TFUF] | SPI_0_SR[RFOF] | SPI_0_SR[TFIWF] */
  SPI0_EndOfQueue_IRQn         = 296u,             /**< SPI0_1 SPI_0_SR[EOQF] */
  SPI0_TxFifoNotfull_IRQn      = 297u,             /**< SPI0_2 SPI_0_SR[TFFF] */
  SPI0_TransferComplete_IRQn   = 298u,             /**< SPI0_3 SPI_0_SR[TCF] */
  SPI0_RxFifoNotempty_IRQn     = 299u,             /**< SPI0_4 SPI_0_SR[RFDF] */
  SPI0_EndOfCmd_IRQn           = 300u,             /**< SPI0_5 SPI_0_SR[CMD_TCF] */
  SPI0_CmdFifoNotFull_IRQn     = 301u,             /**< SPI0_6 SPI_0_SR[CMDFFF] */
  SPI0_ParityError_IRQn        = 302u,             /**< SPI0_7 SPI_0_SR[SPEF] */
  SPI1_FifoErr_IRQn            = 304u,             /**< SPI1_0 SPI_1_SR[TFUF] | SPI_1_SR[RFOF] | SPI_1_SR[TFIWF] */
  SPI1_EndOfQueue_IRQn         = 305u,             /**< SPI1_1 SPI_1_SR[EOQF] */
  SPI1_TxFifoNotfull_IRQn      = 306u,             /**< SPI1_2 SPI_1_SR[TFFF] */
  SPI1_TransferComplete_IRQn   = 307u,             /**< SPI1_3 SPI_1_SR[TCF] */
  SPI1_RxFifoNotempty_IRQn     = 308u,             /**< SPI1_4 SPI_1_SR[RFDF] */
  SPI1_EndOfCmd_IRQn           = 309u,             /**< SPI1_5 SPI_1_SR[CMD_TCF] */
  SPI1_CmdFifoNotFull_IRQn     = 310u,             /**< SPI1_6 SPI_1_SR[CMDFFF] */
  SPI1_ParityError_IRQn        = 311u,             /**< SPI1_7 SPI_1_SR[SPEF] */
  SPI2_FifoErr_IRQn            = 313u,             /**< SPI2_0 SPI_2_SR[TFUF] | SPI_2_SR[RFOF] | SPI_2_SR[TFIWF] */
  SPI2_EndOfQueue_IRQn         = 314u,             /**< SPI2_1 SPI_2_SR[EOQF] */
  SPI2_TxFifoNotfull_IRQn      = 315u,             /**< SPI2_2 SPI_2_SR[TFFF] */
  SPI2_TransferComplete_IRQn   = 316u,             /**< SPI2_3 SPI_2_SR[TCF] */
  SPI2_RxFifoNotempty_IRQn     = 317u,             /**< SPI2_4 SPI_2_SR[RFDF] */
  SPI2_EndOfCmd_IRQn           = 318u,             /**< SPI2_5 SPI_2_SR[SPITCF] | SPI_2_SR[CMD_TCF] */
  SPI2_CmdFifoNotFull_IRQn     = 319u,             /**< SPI2_6 SPI_2_SR[DSITCF] | SPI_2_SR[CMDFFF] */
  SPI2_ParityError_IRQn        = 320u,             /**< SPI2_7 SPI_2_SR[SPEF] | SPI_2_SR[DPEF] */
  SPI3_FifoErr_IRQn            = 322u,             /**< SPI3_0 SPI_3_SR[TFUF] | SPI_3_SR[RFOF] | SPI_3_SR[TFIWF] */
  SPI3_EndOfQueue_IRQn         = 323u,             /**< SPI3_1 SPI_3_SR[EOQF] */
  SPI3_TxFifoNotfull_IRQn      = 324u,             /**< SPI3_2 SPI_3_SR[TFFF] */
  SPI3_TransferComplete_IRQn   = 325u,             /**< SPI3_3 SPI_3_SR[TCF] */
  SPI3_RxFifoNotempty_IRQn     = 326u,             /**< SPI3_4 SPI_3_SR[RFDF] */
  SPI3_EndOfCmd_IRQn           = 327u,             /**< SPI3_5 Reserved */
  SPI3_CmdFifoNotFull_IRQn     = 328u,             /**< SPI3_6 Reserved */
  SPI3_ParityError_IRQn        = 329u,             /**< SPI3_7 SPI_3_SR[SPEF] */
  SPI4_FifoErr_IRQn            = 331u,             /**< SPI4_0 SPI_4_SR[TFUF] | SPI_4_SR[RFOF] | SPI_4_SR[TFIWF] */
  SPI4_EndOfQueue_IRQn         = 332u,             /**< SPI4_1 SPI_4_SR[EOQF] */
  SPI4_TxFifoNotfull_IRQn      = 333u,             /**< SPI4_2 SPI_4_SR[TFFF] */
  SPI4_TransferComplete_IRQn   = 334u,             /**< SPI4_3 SPI_4_SR[TCF] */
  SPI4_RxFifoNotempty_IRQn     = 335u,             /**< SPI4_4 SPI_4_SR[RFDF] */
  SPI4_EndOfCmd_IRQn           = 336u,             /**< SPI4_5 Reserved */
  SPI4_CmdFifoNotFull_IRQn     = 337u,             /**< SPI4_6 Reserved */
  SPI4_ParityError_IRQn        = 338u,             /**< SPI4_7 SPI_4_SR[SPEF] */
  SPI5_FifoErr_IRQn            = 340u,             /**< SPI5_0 SPI_5_SR[TFUF] | SPI_5_SR[RFOF] | SPI_5_SR[TFIWF] */
  SPI5_EndOfQueue_IRQn         = 341u,             /**< SPI5_1 SPI_5_SR[EOQF] */
  SPI5_TxFifoNotfull_IRQn      = 342u,             /**< SPI5_2 SPI_5_SR[TFFF] */
  SPI5_TransferComplete_IRQn   = 343u,             /**< SPI5_3 SPI_5_SR[TCF] */
  SPI5_RxFifoNotempty_IRQn     = 344u,             /**< SPI5_4 SPI_5_SR[RFDF] */
  SPI5_EndOfCmd_IRQn           = 345u,             /**< SPI5_5 Reserved */
  SPI5_CmdFifoNotFull_IRQn     = 346u,             /**< SPI5_6 Reserved */
  SPI5_ParityError_IRQn        = 347u,             /**< SPI5_7 SPI_5_SR[SPEF] */
  BCTU_ListLast_IRQn           = 372u,             /**< BCTU LIST0_LAST | LIST1_LAST */
  BCTU_ConvUpdate_IRQn         = 373u,             /**< BCTU NDATA0 | NDATA1 | DATA_OVR0 | DATA_OVR1 | Trigger */
  LINFLEXD0_RX_IRQn            = 376u,             /**< LinFlex0_0 LinFlex_0_RXI */
  LINFLEXD0_TX_IRQn            = 377u,             /**< LinFlex0_1 LinFlex_0_TXI */
  LINFLEXD0_ERR_IRQn           = 378u,             /**< LinFlex0_2 LinFlex_0_ERR */
  LINFLEXD1_RX_IRQn            = 379u,             /**< LinFlex1_0 LinFlex_1_RXI */
  LINFLEXD1_TX_IRQn            = 380u,             /**< LinFlex1_1 LinFlex_1_TXI */
  LINFLEXD1_ERR_IRQn           = 381u,             /**< LinFlex1_2 LinFlex_1_ERR */
  LINFLEXD2_RX_IRQn            = 382u,             /**< LinFlex2_0 LinFlex_2_RXI */
  LINFLEXD2_TX_IRQn            = 383u,             /**< LinFlex2_1 LinFlex_2_TXI */
  LINFLEXD2_ERR_IRQn           = 384u,             /**< LinFlex2_2 LinFlex_2_ERR */
  LINFLEXD3_RX_IRQn            = 385u,             /**< LinFlex3_0 LinFlex_3_RXI */
  LINFLEXD3_TX_IRQn            = 386u,             /**< LinFlex3_1 LinFlex_3_TXI */
  LINFLEXD3_ERR_IRQn           = 387u,             /**< LinFlex3_2 LinFlex_3_ERR */
  LINFLEXD4_RX_IRQn            = 388u,             /**< LinFlex4_0 LinFlex_4_RXI */
  LINFLEXD4_TX_IRQn            = 389u,             /**< LinFlex4_1 LinFlex_4_TXI */
  LINFLEXD4_ERR_IRQn           = 390u,             /**< LinFlex4_2 LinFlex_4_ERR */
  LINFLEXD5_RX_IRQn            = 391u,             /**< LinFlex5_0 LinFlex_5_RXI */
  LINFLEXD5_TX_IRQn            = 392u,             /**< LinFlex5_1 LinFlex_5_TXI */
  LINFLEXD5_ERR_IRQn           = 393u,             /**< LinFlex5_2 LinFlex_5_ERR */
  LINFLEXD6_RX_IRQn            = 394u,             /**< LinFlex6_0 LinFlex_6_RXI */
  LINFLEXD6_TX_IRQn            = 395u,             /**< LinFlex6_1 LinFlex_6_TXI */
  LINFLEXD6_ERR_IRQn           = 396u,             /**< LinFlex6_2 LinFlex_6_ERR */
  LINFLEXD7_RX_IRQn            = 397u,             /**< LinFlex7_0 LinFlex_7_RXI */
  LINFLEXD7_TX_IRQn            = 398u,             /**< LinFlex7_1 LinFlex_7_TXI */
  LINFLEXD7_ERR_IRQn           = 399u,             /**< LinFlex7_2 LinFlex_7_ERR */
  LINFLEXD8_RX_IRQn            = 400u,             /**< LinFlex8_0 LinFlex_8_RXI */
  LINFLEXD8_TX_IRQn            = 401u,             /**< LinFlex8_1 LinFlex_8_TXI */
  LINFLEXD8_ERR_IRQn           = 402u,             /**< LinFlex8_2 LinFlex_8_ERR */
  LINFLEXD9_RX_IRQn            = 403u,             /**< LinFlex9_0 LinFlex_9_RXI */
  LINFLEXD9_TX_IRQn            = 404u,             /**< LinFlex9_1 LinFlex_9_TXI */
  LINFLEXD9_ERR_IRQn           = 405u,             /**< LinFlex9_2 LinFlex_9_ERR */
  LINFLEXD10_RX_IRQn           = 406u,             /**< LinFlex10_0 LinFlex_10_RXI */
  LINFLEXD10_TX_IRQn           = 407u,             /**< LinFlex10_1 LinFlex_10_TXI */
  LINFLEXD10_ERR_IRQn          = 408u,             /**< LinFlex10_2 LinFlex_10_ERR */
  LINFLEXD11_RX_IRQn           = 409u,             /**< LinFlex11_0 LinFlex_11_RXI */
  LINFLEXD11_TX_IRQn           = 410u,             /**< LinFlex11_1 LinFlex_11_TXI */
  LINFLEXD11_ERR_IRQn          = 411u,             /**< LinFlex11_2 LinFlex_11_ERR */
  LINFLEXD12_RX_IRQn           = 412u,             /**< LinFlex12_0 LinFlex_12_RXI */
  LINFLEXD12_TX_IRQn           = 413u,             /**< LinFlex12_1 LinFlex_12_TXI */
  LINFLEXD12_ERR_IRQn          = 414u,             /**< LinFlex12_2 LinFlex_12_ERR */
  LINFLEXD13_RX_IRQn           = 415u,             /**< LinFlex13_0 LinFlex_13_RXI */
  LINFLEXD13_TX_IRQn           = 416u,             /**< LinFlex13_1 LinFlex_13_TXI */
  LINFLEXD13_ERR_IRQn          = 417u,             /**< LinFlex13_2 LinFlex_13_ERR */
  LINFLEXD14_RX_IRQn           = 418u,             /**< LinFlex14_0 LinFlex_14_RXI */
  LINFLEXD14_TX_IRQn           = 419u,             /**< LinFlex14_1 LinFlex_14_TXI */
  LINFLEXD14_ERR_IRQn          = 420u,             /**< LinFlex14_2 LinFlex_14_ERR */
  LINFLEXD15_RX_IRQn           = 421u,             /**< LinFlex15_0 LinFlex_15_RXI */
  LINFLEXD15_TX_IRQn           = 422u,             /**< LinFlex15_1 LinFlex_15_TXI */
  LINFLEXD15_ERR_IRQn          = 423u,             /**< LinFlex15_2 LinFlex_15_ERR */
  LINFLEXD16_RX_IRQn           = 424u,             /**< LinFlex16_0 LinFlex_16_RXI */
  LINFLEXD16_TX_IRQn           = 425u,             /**< LinFlex16_1 LinFlex_16_TXI */
  LINFLEXD16_ERR_IRQn          = 426u,             /**< LinFlex16_2 LinFlex_16_ERR */
  LINFLEXD17_RX_IRQn           = 427u,             /**< LinFlex17_0 LinFlex_17_RXI */
  LINFLEXD17_TX_IRQn           = 428u,             /**< LinFlex17_1 LinFlex_17_TXI */
  LINFLEXD17_ERR_IRQn          = 429u,             /**< LinFlex17_2 LinFlex_17_ERR */
  I2C0_IRQn                    = 440u,             /**< I2C_0_0 I2C0_SR[IBAL] | I2C0_SR[TCF] | I2C0_SR[IAAS] */
  I2C1_IRQn                    = 442u,             /**< I2C_1_0 I2C1_SR[IBAL] | I2C1_SR[TCF] | I2C1_SR[IAAS] */
  I2C2_IRQn                    = 444u,             /**< I2C_2_0 I2C2_SR[IBAL] | I2C2_SR[TCF] | I2C2_SR[IAAS] */
  I2C3_IRQn                    = 446u,             /**< I2C_3_0 I2C3_SR[IBAL] | I2C3_SR[TCF] | I2C3_SR[IAAS] */
  FR_LRNEIF_DRNEIF_IRQn        = 453u,             /**< FlexRay_0_0 LRNEIF | DRNEIF */
  FR_LRCEIF_DRCEIF_IRQn        = 454u,             /**< FlexRay_0_1 LRCEIF | DRCEIF */
  FR_FAFAIF_IRQn               = 455u,             /**< FlexRay_0_2 FAFAIF */
  FR_FAFBIF_IRQn               = 456u,             /**< FlexRay_0_3 FAFBIF */
  FR_WUPIF_IRQn                = 457u,             /**< FlexRay_0_4 WUPIF */
  FR_PRIF_IRQn                 = 458u,             /**< FlexRay_0_5 PRIF */
  FR_CHIF_IRQn                 = 459u,             /**< FlexRay_0_6 CHIF */
  FR_TBIF_IRQn                 = 460u,             /**< FlexRay_0_7 TBIF */
  FR_RBIF_IRQn                 = 461u,             /**< FlexRay_0_8 RBIF */
  FR_MIF_IRQn                  = 462u,             /**< FlexRay_0_9 MIF */
  PLL_IRQn                     = 480u,             /**< PLL Loss-of-lock interrupt request for PLL */
  FCCU_ALARM_IRQn              = 488u,             /**< FCCU Interrupt request (ALARM state) */
  FCCU_MISC_IRQn               = 489u,             /**< FCCU Interrupt request (miscellaneous conditions) */
  FCCU_EOUT_IRQn               = 490u,             /**< FCCU EOUT interrupt */
  HSM_0_IRQn                   = 496u,             /**< Hardware Security Module 0 HSM2HTIE[0] */
  HSM_1_IRQn                   = 497u,             /**< Hardware Security Module 1 HSM2HTIE[1] */
  HSM_2_IRQn                   = 498u,             /**< Hardware Security Module 2 HSM2HTIE[2] */
  HSM_3_IRQn                   = 499u,             /**< Hardware Security Module 3 HSM2HTIE[3] */
  HSM_4_IRQn                   = 500u,             /**< Hardware Security Module 4 HSM2HTIE[4] */
  HSM_5_IRQn                   = 501u,             /**< Hardware Security Module 5 HSM2HTIE[5] */
  HSM_6_IRQn                   = 502u,             /**< Hardware Security Module 6 HSM2HTIE[6] */
  HSM_7_IRQn                   = 503u,             /**< Hardware Security Module 7 HSM2HTIE[7] */
  HSM_8_IRQn                   = 504u,             /**< Hardware Security Module 8 HSM2HTIE[8] */
  HSM_9_IRQn                   = 505u,             /**< Hardware Security Module 9 HSM2HTIE[9] */
  HSM_10_IRQn                  = 506u,             /**< Hardware Security Module 10 HSM2HTIE[10] */
  HSM_11_IRQn                  = 507u,             /**< Hardware Security Module 11 HSM2HTIE[11] */
  HSM_12_IRQn                  = 508u,             /**< Hardware Security Module 12 HSM2HTIE[12] */
  HSM_13_IRQn                  = 509u,             /**< Hardware Security Module 13 HSM2HTIE[13] */
  HSM_14_IRQn                  = 510u,             /**< Hardware Security Module 14 HSM2HTIE[14] */
  HSM_15_IRQn                  = 511u,             /**< Hardware Security Module 15 HSM2HTIE[15] */
  HSM_16_IRQn                  = 512u,             /**< Hardware Security Module 16 HSM2HTIE[16] */
  HSM_17_IRQn                  = 513u,             /**< Hardware Security Module 17 HSM2HTIE[17] */
  HSM_18_IRQn                  = 514u,             /**< Hardware Security Module 18 HSM2HTIE[18] */
  HSM_19_IRQn                  = 515u,             /**< Hardware Security Module 19 HSM2HTIE[19] */
  HSM_20_IRQn                  = 516u,             /**< Hardware Security Module 20 HSM2HTIE[20] */
  HSM_21_IRQn                  = 517u,             /**< Hardware Security Module 21 HSM2HTIE[21] */
  HSM_22_IRQn                  = 518u,             /**< Hardware Security Module 22 HSM2HTIE[22] */
  HSM_23_IRQn                  = 519u,             /**< Hardware Security Module 23 HSM2HTIE[23] */
  HSM_24_IRQn                  = 520u,             /**< Hardware Security Module 24 HSM2HTIE[24] */
  HSM_25_IRQn                  = 521u,             /**< Hardware Security Module 25 HSM2HTIE[25] */
  HSM_26_IRQn                  = 522u,             /**< Hardware Security Module 26 HSM2HTIE[26] */
  HSM_27_IRQn                  = 523u,             /**< Hardware Security Module 27 HSM2HTIE[27] */
  HSM_28_IRQn                  = 524u,             /**< Hardware Security Module 28 HSM2HTIE[28] */
  HSM_29_IRQn                  = 525u,             /**< Hardware Security Module 29 HSM2HTIE[29] */
  HSM_30_IRQn                  = 526u,             /**< Hardware Security Module 30 HSM2HTIE[30] */
  HSM_31_IRQn                  = 527u,             /**< Hardware Security Module 31 HSM2HTIE[31] */
  ADC0_EOC_IRQn                = 548u,             /**< ADC_0_EOC ADC_0_EOC */
  ADC0_ER_IRQn                 = 549u,             /**< ADC_0_ER ADC_0_ER */
  ADC0_WD_IRQn                 = 550u,             /**< ADC_0_WD ADC_0_WD */
  ADC1_EOC_IRQn                = 554u,             /**< ADC_1_EOC ADC_1_EOC */
  ADC1_ER_IRQn                 = 555u,             /**< ADC_1_ER ADC_1_ER */
  ADC1_WD_IRQn                 = 556u,             /**< ADC_1_WD ADC_1_WD */
  CMP0_IRQn                    = 560u,             /**< Analogue_Comparator_Module_0 CMP0 */
  CMP1_IRQn                    = 561u,             /**< Analogue_Comparator_Module_1 CMP1 */
  CMP2_IRQn                    = 562u,             /**< Analogue_Comparator_Module_2 CMP2 */
  CAN0_Wake_Up_IRQn            = 565u,             /**< FlexCAN0_0 FlexCAN0_PN */
  CAN0_Error_IRQn              = 566u,             /**< FlexCAN0_1 FLEXCAN_0_ESR[ERR_INT] | FLEXCAN_0_FD_ERROR */
  CAN0_ORed_IRQn               = 567u,             /**< FlexCAN0_2 FLEXCAN_0_ESR_BOFF | FLEXCAN_0_ESR_BOFF_DONE | FLEXCAN_0_Transmit_Warning | FLEXCAN_0_Receive_Warning */
  CAN0_ORed_00_03_MB_IRQn      = 568u,             /**< FlexCAN0_3 FlexCAN0_BUF_00_03 */
  CAN0_ORed_04_07_MB_IRQn      = 569u,             /**< FlexCAN0_4 FlexCAN0_BUF_04_07 */
  CAN0_ORed_08_11_MB_IRQn      = 570u,             /**< FlexCAN0_5 FlexCAN0_BUF_08_11 */
  CAN0_ORed_12_15_MB_IRQn      = 571u,             /**< FlexCAN0_6 FlexCAN0_BUF_12_15 */
  CAN0_ORed_16_31_MB_IRQn      = 572u,             /**< FlexCAN0_7 FlexCAN0_BUF_16_31 */
  CAN0_ORed_32_63_MB_IRQn      = 573u,             /**< FlexCAN0_8 FlexCAN0_BUF_32_63 */
  CAN0_ORed_64_95_MB_IRQn      = 574u,             /**< FlexCAN0_9 FlexCAN0_BUF_64_95 */
  CAN1_Error_IRQn              = 578u,             /**< FlexCAN1_0 FLEXCAN_1_ESR[ERR_INT] | FLEXCAN_1_FD_ERROR */
  CAN1_ORed_IRQn               = 579u,             /**< FlexCAN1_1 FLEXCAN_1_ESR_BOFF | FLEXCAN_1_ESR_BOFF_DONE | FLEXCAN_1_Transmit_Warning | FLEXCAN_1_Receive_Warning */
  CAN1_ORed_00_03_MB_IRQn      = 580u,             /**< FlexCAN1_2 FlexCAN1_BUF_00_03 */
  CAN1_ORed_04_07_MB_IRQn      = 581u,             /**< FlexCAN1_3 FlexCAN1_BUF_04_07 */
  CAN1_ORed_08_11_MB_IRQn      = 582u,             /**< FlexCAN1_4 FlexCAN1_BUF_08_11 */
  CAN1_ORed_12_15_MB_IRQn      = 583u,             /**< FlexCAN1_5 FlexCAN1_BUF_12_15 */
  CAN1_ORed_16_31_MB_IRQn      = 584u,             /**< FlexCAN1_6 FlexCAN1_BUF_16_31 */
  CAN1_ORed_32_63_MB_IRQn      = 585u,             /**< FlexCAN1_7 FlexCAN1_BUF_32_63 */
  CAN1_ORed_64_95_MB_IRQn      = 586u,             /**< FlexCAN1_8 FlexCAN1_BUF_64_95 */
  CAN2_Error_IRQn              = 590u,             /**< FlexCAN2_0 FLEXCAN_2_ESR[ERR_INT] | FLEXCAN_2_FD_ERROR */
  CAN2_ORed_IRQn               = 591u,             /**< FlexCAN2_1 FLEXCAN_2_ESR_BOFF | FLEXCAN_2_ESR_BOFF_DONE | FLEXCAN_2_Transmit_Warning | FLEXCAN_2_Receive_Warning */
  CAN2_ORed_00_03_MB_IRQn      = 592u,             /**< FlexCAN2_2 FlexCAN2_BUF_00_03 */
  CAN2_ORed_04_07_MB_IRQn      = 593u,             /**< FlexCAN2_3 FlexCAN2_BUF_04_07 */
  CAN2_ORed_08_11_MB_IRQn      = 594u,             /**< FlexCAN2_4 FlexCAN2_BUF_08_11 */
  CAN2_ORed_12_15_MB_IRQn      = 595u,             /**< FlexCAN2_5 FlexCAN2_BUF_12_15 */
  CAN2_ORed_16_31_MB_IRQn      = 596u,             /**< FlexCAN2_6 FlexCAN2_BUF_16_31 */
  CAN2_ORed_32_63_MB_IRQn      = 597u,             /**< FlexCAN2_7 FlexCAN2_BUF_32_63 */
  CAN2_ORed_64_95_MB_IRQn      = 598u,             /**< FlexCAN2_8 FlexCAN2_BUF_64_95 */
  CAN3_Error_IRQn              = 602u,             /**< FlexCAN3_0 FLEXCAN_3_ESR[ERR_INT] | FLEXCAN_3_FD_ERROR */
  CAN3_ORed_IRQn               = 603u,             /**< FlexCAN3_1 FLEXCAN_3_ESR_BOFF | FLEXCAN_3_ESR_BOFF_DONE | FLEXCAN_3_Transmit_Warning | FLEXCAN_3_Receive_Warning */
  CAN3_ORed_00_03_MB_IRQn      = 604u,             /**< FlexCAN3_2 FlexCAN3_BUF_00_03 */
  CAN3_ORed_04_07_MB_IRQn      = 605u,             /**< FlexCAN3_3 FlexCAN3_BUF_04_07 */
  CAN3_ORed_08_11_MB_IRQn      = 606u,             /**< FlexCAN3_4 FlexCAN3_BUF_08_11 */
  CAN3_ORed_12_15_MB_IRQn      = 607u,             /**< FlexCAN3_5 FlexCAN3_BUF_12_15 */
  CAN3_ORed_16_31_MB_IRQn      = 608u,             /**< FlexCAN3_6 FlexCAN3_BUF_16_31 */
  CAN3_ORed_32_63_MB_IRQn      = 609u,             /**< FlexCAN3_7 FlexCAN3_BUF_32_63 */
  CAN3_ORed_64_95_MB_IRQn      = 610u,             /**< FlexCAN3_8 FlexCAN3_BUF_64_95 */
  CAN4_Error_IRQn              = 614u,             /**< FlexCAN4_0 FLEXCAN_4_ESR[ERR_INT] | FLEXCAN_4_FD_ERROR */
  CAN4_ORed_IRQn               = 615u,             /**< FlexCAN4_1 FLEXCAN_4_ESR_BOFF | FLEXCAN_4_ESR_BOFF_DONE | FLEXCAN_4_Transmit_Warning | FLEXCAN_4_Receive_Warning */
  CAN4_ORed_00_03_MB_IRQn      = 616u,             /**< FlexCAN4_2 FlexCAN4_BUF_00_03 */
  CAN4_ORed_04_07_MB_IRQn      = 617u,             /**< FlexCAN4_3 FlexCAN4_BUF_04_07 */
  CAN4_ORed_08_11_MB_IRQn      = 618u,             /**< FlexCAN4_4 FlexCAN4_BUF_08_11 */
  CAN4_ORed_12_15_MB_IRQn      = 619u,             /**< FlexCAN4_5 FlexCAN4_BUF_12_15 */
  CAN4_ORed_16_31_MB_IRQn      = 620u,             /**< FlexCAN4_6 FlexCAN4_BUF_16_31 */
  CAN4_ORed_32_63_MB_IRQn      = 621u,             /**< FlexCAN4_7 FlexCAN4_BUF_32_63 */
  CAN4_ORed_64_95_MB_IRQn      = 622u,             /**< FlexCAN4_8 FlexCAN4_BUF_64_95 */
  CAN5_Error_IRQn              = 626u,             /**< FlexCAN5_0 FLEXCAN_5_ESR[ERR_INT] | FLEXCAN_5_FD_ERROR */
  CAN5_ORed_IRQn               = 627u,             /**< FlexCAN5_1 FLEXCAN_5_ESR_BOFF | FLEXCAN_5_ESR_BOFF_DONE | FLEXCAN_5_Transmit_Warning | FLEXCAN_5_Receive_Warning */
  CAN5_ORed_00_03_MB_IRQn      = 628u,             /**< FlexCAN5_2 FlexCAN5_BUF_00_03 */
  CAN5_ORed_04_07_MB_IRQn      = 629u,             /**< FlexCAN5_3 FlexCAN5_BUF_04_07 */
  CAN5_ORed_08_11_MB_IRQn      = 630u,             /**< FlexCAN5_4 FlexCAN5BUF_08_11 */
  CAN5_ORed_12_15_MB_IRQn      = 631u,             /**< FlexCAN5_5 FlexCAN5_BUF_12_15 */
  CAN5_ORed_16_31_MB_IRQn      = 632u,             /**< FlexCAN5_6 FlexCAN5_BUF_16_31 */
  CAN5_ORed_32_63_MB_IRQn      = 633u,             /**< FlexCAN5_7 FlexCAN5_BUF_32_63 */
  CAN5_ORed_64_95_MB_IRQn      = 634u,             /**< FlexCAN5_8 FlexCAN5_BUF_64_95 */
  CAN6_Error_IRQn              = 638u,             /**< FlexCAN6_0 FLEXCAN_6_ESR[ERR_INT] | FLEXCAN_6_FD_ERROR */
  CAN6_ORed_IRQn               = 639u,             /**< FlexCAN6_1 FLEXCAN_6_ESR_BOFF | FLEXCAN_6_ESR_BOFF_DONE | FLEXCAN_6_Transmit_Warning | FLEXCAN_6_Receive_Warning */
  CAN6_ORed_00_03_MB_IRQn      = 640u,             /**< FlexCAN6_2 FlexCAN6_BUF_00_03 */
  CAN6_ORed_04_07_MB_IRQn      = 641u,             /**< FlexCAN6_3 FlexCAN6_BUF_04_07 */
  CAN6_ORed_08_11_MB_IRQn      = 642u,             /**< FlexCAN6_4 FlexCAN6_BUF_08_11 */
  CAN6_ORed_12_15_MB_IRQn      = 643u,             /**< FlexCAN6_5 FlexCAN6_BUF_12_15 */
  CAN6_ORed_16_31_MB_IRQn      = 644u,             /**< FlexCAN6_6 FlexCAN6_BUF_16_31 */
  CAN6_ORed_32_63_MB_IRQn      = 645u,             /**< FlexCAN6_7 FlexCAN6_BUF_32_63 */
  CAN6_ORed_64_95_MB_IRQn      = 646u,             /**< FlexCAN6_8 FlexCAN6_BUF_64_95 */
  CAN7_Error_IRQn              = 650u,             /**< FlexCAN7_0 FLEXCAN_7_ESR[ERR_INT] | FLEXCAN_7_FD_ERROR */
  CAN7_ORed_IRQn               = 651u,             /**< FlexCAN7_1 FLEXCAN_7_ESR_BOFF | FLEXCAN_7_ESR_BOFF_DONE | FLEXCAN_7_Transmit_Warning | FLEXCAN_7_Receive_Warning */
  CAN7_ORed_00_03_MB_IRQn      = 652u,             /**< FlexCAN7_2 FlexCAN7_BUF_00_03 */
  CAN7_ORed_04_07_MB_IRQn      = 653u,             /**< FlexCAN7_3 FlexCAN7_BUF_04_07 */
  CAN7_ORed_08_11_MB_IRQn      = 654u,             /**< FlexCAN7_4 FlexCAN7_BUF_08_11 */
  CAN7_ORed_12_15_MB_IRQn      = 655u,             /**< FlexCAN7_5 FlexCAN7_BUF_12_15 */
  CAN7_ORed_16_31_MB_IRQn      = 656u,             /**< FlexCAN7_6 FlexCAN7_BUF_16_31 */
  CAN7_ORed_32_63_MB_IRQn      = 657u,             /**< FlexCAN7_7 FlexCAN7_BUF_32_63 */
  CAN7_ORed_64_95_MB_IRQn      = 658u,             /**< FlexCAN7_8 FlexCAN7_BUF_64_95 */
  WKPU_07_00_IRQn              = 668u,             /**< WKPU_IRQ_0 WKPU_IRQ_0 */
  WKPU_15_08_IRQn              = 669u,             /**< WKPU_IRQ_1 WKPU_IRQ_1 */
  WKPU_23_16_IRQn              = 670u,             /**< WKPU_IRQ_2 WKPU_IRQ_2 */
  WKPU_31_24_IRQn              = 671u,             /**< WKPU_IRQ_3 WKPU_IRQ_3 */
  uSDHC0_IRQn                  = 680u,             /**< SDIO (SDHC) 0 SDHC0 */
  USB_OTG_IRQn                 = 682u,             /**< USB_OTG USB OTG */
  USB_SPH_IRQn                 = 683u,             /**< USB_SPH USB SPH */
  MLB_Block_IRQn               = 686u,             /**< MLB MLB block interrupt */
  MLB_Channel_IRQn             = 687u,             /**< MLB MLB Channel Interrupts 0-15 Combined */
  I2S_0_Tx_IRQn                = 690u,             /**< SAI0_TX_FIFO_IRQ SAI0_TX_FIFO_IRQ */
  I2S_0_Rx_IRQn                = 691u,             /**< SAI0_RX_FIFO_IRQ SAI0_RX_FIFO_IRQ */
  I2S_1_Tx_IRQn                = 692u,             /**< SAI1_TX_FIFO_IRQ SAI1_TX_FIFO_IRQ */
  I2S_1_Rx_IRQn                = 693u,             /**< SAI1_RX_FIFO_IRQ SAI1_RX_FIFO_IRQ */
  I2S_2_Tx_IRQn                = 694u,             /**< SAI2_TX_FIFO_IRQ SAI2_TX_FIFO_IRQ */
  I2S_2_Rx_IRQn                = 695u,             /**< SAI2_RX_FIFO_IRQ SAI2_RX_FIFO_IRQ */
  JDC_IRQn                     = 701u,             /**< JDC JDC */
  MEMU_IRQn                    = 702u,             /**< MEMU_1 Correctable errors in LPU mode */
  EMIOS0_00_01_IRQn            = 706u,             /**< eMIOS 0 channel 0,1 EMIOS_GFR[F0,F1] */
  EMIOS0_02_03_IRQn            = 707u,             /**< eMIOS 0 channel 2,3 EMIOS_GFR[F2,F3] */
  EMIOS0_04_05_IRQn            = 708u,             /**< eMIOS 0 channel 4,5 EMIOS_GFR[F4,F5] */
  EMIOS0_06_07_IRQn            = 709u,             /**< eMIOS 0 channel 6,7 EMIOS_GFR[F6,F7] */
  EMIOS0_08_09_IRQn            = 710u,             /**< eMIOS 0 channel 8,9 EMIOS_GFR[F8,F9] */
  EMIOS0_10_11_IRQn            = 711u,             /**< eMIOS 0 channel 10,11 EMIOS_GFR[F10,F11] */
  EMIOS0_12_13_IRQn            = 712u,             /**< eMIOS 0 channel 12,13 EMIOS_GFR[F12,F13] */
  EMIOS0_14_15_IRQn            = 713u,             /**< eMIOS 0 channel 14,15 EMIOS_GFR[F14,F15] */
  EMIOS0_16_17_IRQn            = 714u,             /**< eMIOS 0 channel 16,17 EMIOS_GFR[F16,F17] */
  EMIOS0_18_19_IRQn            = 715u,             /**< eMIOS 0 channel 18,19 EMIOS_GFR[F18,F19] */
  EMIOS0_20_21_IRQn            = 716u,             /**< eMIOS 0 channel 20,21 EMIOS_GFR[F20,F21] */
  EMIOS0_22_23_IRQn            = 717u,             /**< eMIOS 0 channel 22,23 EMIOS_GFR[F22,F23] */
  EMIOS0_24_25_IRQn            = 718u,             /**< eMIOS 0 channel 24,25 EMIOS_GFR[F24,F25] */
  EMIOS0_26_27_IRQn            = 719u,             /**< eMIOS 0 channel 26,27 EMIOS_GFR[F26,F27] */
  EMIOS0_28_29_IRQn            = 720u,             /**< eMIOS 0 channel 28,29 EMIOS_GFR[F28,F29] */
  EMIOS0_30_31_IRQn            = 721u,             /**< eMIOS 0 channel 30,31 EMIOS_GFR[F30,F31] */
  EMIOS1_00_01_IRQn            = 722u,             /**< eMIOS 1 channel 0,1 EMIOS_GFR[F32,F33] */
  EMIOS1_02_03_IRQn            = 723u,             /**< eMIOS 1 channel 2, 3 EMIOS_GFR[F34,F35] */
  EMIOS1_04_05_IRQn            = 724u,             /**< eMIOS 1 channel 4, 5 EMIOS_GFR[F36,F37] */
  EMIOS1_06_07_IRQn            = 725u,             /**< eMIOS 1 channel 6, 7 EMIOS_GFR[F38,F39] */
  EMIOS1_08_09_IRQn            = 726u,             /**< eMIOS 1 channel 8, 9 EMIOS_GFR[F40,F41] */
  EMIOS1_10_11_IRQn            = 727u,             /**< eMIOS 1 channel 10, 11 EMIOS_GFR[F42,F43] */
  EMIOS1_12_13_IRQn            = 728u,             /**< eMIOS 1 channel 12, 13 EMIOS_GFR[F44,F45] */
  EMIOS1_14_15_IRQn            = 729u,             /**< eMIOS 1 channel 14, 15 EMIOS_GFR[F46,F47] */
  EMIOS1_16_17_IRQn            = 730u,             /**< eMIOS 1 channel 16, 17 EMIOS_GFR[F48,F49] */
  EMIOS1_18_19_IRQn            = 731u,             /**< eMIOS 1 channel 18, 19 EMIOS_GFR[F50,F51] */
  EMIOS1_20_21_IRQn            = 732u,             /**< eMIOS 1 channel 20, 21 EMIOS_GFR[F52,F53] */
  EMIOS1_22_23_IRQn            = 733u,             /**< eMIOS 1 channel 22, 23 EMIOS_GFR[F54,F55] */
  EMIOS1_24_25_IRQn            = 734u,             /**< eMIOS 1 channel 24, 25 EMIOS_GFR[F56,F57] */
  EMIOS1_26_27_IRQn            = 735u,             /**< eMIOS 1 channel 26, 27 EMIOS_GFR[F58,F59] */
  EMIOS1_28_29_IRQn            = 736u,             /**< eMIOS 1 channel 28, 29 EMIOS_GFR[F60,F61] */
  EMIOS1_30_31_IRQn            = 737u,             /**< eMIOS 1 channel 30, 31 EMIOS_GFR[F62,F63] */
  EMIOS2_00_01_IRQn            = 738u,             /**< eMIOS 2 channel 0, 1 EMIOS_GFR[F64,F65] */
  EMIOS2_02_03_IRQn            = 739u,             /**< eMIOS 2 channel 2, 3 EMIOS_GFR[F66,F67] */
  EMIOS2_04_05_IRQn            = 740u,             /**< eMIOS 2 channel 4, 5 EMIOS_GFR[F68,F69] */
  EMIOS2_06_07_IRQn            = 741u,             /**< eMIOS 2 channel 6, 7 EMIOS_GFR[F70,F71] */
  EMIOS2_08_09_IRQn            = 742u,             /**< eMIOS 2 channel 8, 9 EMIOS_GFR[F72,F73] */
  EMIOS2_10_11_IRQn            = 743u,             /**< eMIOS 2 channel 10, 11 EMIOS_GFR[F74,F75] */
  EMIOS2_12_13_IRQn            = 744u,             /**< eMIOS 2 channel 12, 13 EMIOS_GFR[F76,F77] */
  EMIOS2_14_15_IRQn            = 745u,             /**< eMIOS 2 channel 14, 15 EMIOS_GFR[F78,F79] */
  EMIOS2_16_17_IRQn            = 746u,             /**< eMIOS 2 channel 16, 17 EMIOS_GFR[F80,F81] */
  EMIOS2_18_19_IRQn            = 747u,             /**< eMIOS 2 channel 18, 19 EMIOS_GFR[F82,F83] */
  EMIOS2_20_21_IRQn            = 748u,             /**< eMIOS 2 channel 20, 21 EMIOS_GFR[F84,F85] */
  EMIOS2_22_23_IRQn            = 749u,             /**< eMIOS 2 channel 22, 23 EMIOS_GFR[F86,F87] */
  EMIOS2_24_25_IRQn            = 750u,             /**< eMIOS 2 channel 24, 25 EMIOS_GFR[F88,F89] */
  EMIOS2_26_27_IRQn            = 751u,             /**< eMIOS 2 channel 26, 27 EMIOS_GFR[F90,F91] */
  EMIOS2_28_29_IRQn            = 752u,             /**< eMIOS 2 channel 28, 29 EMIOS_GFR[F92,F93] */
  EMIOS2_30_31_IRQn            = 753u              /**< eMIOS 2 channel 30, 31 EMIOS_GFR[F94,F95] */
} IRQn_Type;

/*!
 * @}
 */ /* end of group Interrupt_vector_numbers_MPC5748G */


/* ----------------------------------------------------------------------------
   -- Device Peripheral Access Layer for MPC5748G
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup Peripheral_access_layer_MPC5748G Device Peripheral Access Layer for MPC5748G
 * @{
 */

/* @brief This module covers memory mapped registers available on SoC */

/* ----------------------------------------------------------------------------
   -- ADC Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ADC_Peripheral_Access_Layer ADC Peripheral Access Layer
 * @{
 */


/** ADC - Size of Registers Arrays */
#define ADC_CEOCFR_COUNT                         3u
#define ADC_CIMR_COUNT                           3u
#define ADC_DMAR_COUNT                           3u
#define ADC_PSR_COUNT                            3u
#define ADC_CTR_COUNT                            3u
#define ADC_NCMR_COUNT                           3u
#define ADC_JCMR_COUNT                           3u
#define ADC_CDR_COUNT                            96u
#define ADC_CWSELR_COUNT                         12u
#define ADC_CWENR_COUNT                          3u
#define ADC_AWORR_COUNT                          3u

/** ADC - Register Layout Typedef */
typedef struct {
  __IO uint32_t MCR;                               /**< Main Configuration Register, offset: 0x0 */
  __I  uint32_t MSR;                               /**< Main Status register, offset: 0x4 */
       uint8_t RESERVED_0[8];
  __IO uint32_t ISR;                               /**< Interrupt Status Register, offset: 0x10 */
  __IO uint32_t CEOCFR[ADC_CEOCFR_COUNT];          /**< Channel Pending register 0..Channel Pending register 2, array offset: 0x14, array step: 0x4 */
  __IO uint32_t IMR;                               /**< Interrupt Mask Register, offset: 0x20 */
  __IO uint32_t CIMR[ADC_CIMR_COUNT];              /**< Channel Interrupt Mask Register 0..Channel Interrupt Mask Register 2, array offset: 0x24, array step: 0x4 */
  __IO uint32_t WTISR;                             /**< Watchdog Threshold Interrupt Status Register, offset: 0x30 */
  __IO uint32_t WTIMR;                             /**< Watchdog Threshold Interrupt Mask Register, offset: 0x34 */
       uint8_t RESERVED_1[8];
  __IO uint32_t DMAE;                              /**< DMA Enable register, offset: 0x40 */
  __IO uint32_t DMAR[ADC_DMAR_COUNT];              /**< DMA Channel Select Register 0..DMA Channel Select Register 2, array offset: 0x44, array step: 0x4 */
       uint8_t RESERVED_2[16];
  __IO uint32_t THRHLR0;                           /**< Threshold Register, offset: 0x60 */
  __IO uint32_t THRHLR1;                           /**< Threshold Register, offset: 0x64 */
  __IO uint32_t THRHLR2;                           /**< Threshold Register, offset: 0x68 */
  __IO uint32_t THRHLR3;                           /**< Threshold Register, offset: 0x6C */
       uint8_t RESERVED_3[16];
  __IO uint32_t PSCR;                              /**< Presampling Control Register, offset: 0x80 */
  __IO uint32_t PSR[ADC_PSR_COUNT];                /**< Presampling register 0..Presampling register 2, array offset: 0x84, array step: 0x4 */
       uint8_t RESERVED_4[4];
  __IO uint32_t CTR[ADC_CTR_COUNT];                /**< Conversion Timing Register 0..Conversion Timing Register 2, array offset: 0x94, array step: 0x4 */
       uint8_t RESERVED_5[4];
  __IO uint32_t NCMR[ADC_NCMR_COUNT];              /**< Normal Conversion Mask Register 0..Normal Conversion Mask Register 2, array offset: 0xA4, array step: 0x4 */
       uint8_t RESERVED_6[4];
  __IO uint32_t JCMR[ADC_JCMR_COUNT];              /**< Injected Conversion Mask Register 0..Injected Conversion Mask Register 2, array offset: 0xB4, array step: 0x4 */
       uint8_t RESERVED_7[4];
  __IO uint32_t DSDR;                              /**< Decode Signals Delay Register, offset: 0xC4 */
  __IO uint32_t PDEDR;                             /**< Power Down Exit Delay Register, offset: 0xC8 */
       uint8_t RESERVED_8[52];
  __I  uint32_t CDR[ADC_CDR_COUNT];                /**< Channel Data Register n (Precision Channels)..Channel Data Register n (External Channels), array offset: 0x100, array step: 0x4 */
  __IO uint32_t THRHLR4;                           /**< Threshold Register, offset: 0x280 */
  __IO uint32_t THRHLR5;                           /**< Threshold Register, offset: 0x284 */
       uint8_t RESERVED_9[40];
  __IO uint32_t CWSELR[ADC_CWSELR_COUNT];          /**< Channel Watchdog Select Register 0..Channel Watchdog Select Register 11, array offset: 0x2B0, array step: 0x4 */
  __IO uint32_t CWENR[ADC_CWENR_COUNT];            /**< Channel Watchdog Enable Register 0..Channel Watchdog Enable Register 2, array offset: 0x2E0, array step: 0x4 */
       uint8_t RESERVED_10[4];
  __IO uint32_t AWORR[ADC_AWORR_COUNT];            /**< Analog Watchdog Out of Range Register 0..Analog Watchdog Out of Range Register 2, array offset: 0x2F0, array step: 0x4 */
       uint8_t RESERVED_11[164];
  __IO uint32_t CALBISTREG;                        /**< Calibration, BIST Control and status Register, offset: 0x3A0 */
       uint8_t RESERVED_12[4];
  __IO uint32_t OFSGNUSR;                          /**< Offset and Gain User Register, offset: 0x3A8 */
} ADC_Type, *ADC_MemMapPtr;

 /** Number of instances of the ADC module. */
#define ADC_INSTANCE_COUNT                       (2u)


/* ADC - Peripheral instance base addresses */
/** Peripheral ADC_0 base address */
#define ADC_0_BASE                               (0xFFE00000u)
/** Peripheral ADC_0 base pointer */
#define ADC_0                                    ((ADC_Type *)ADC_0_BASE)
/** Peripheral ADC_1 base address */
#define ADC_1_BASE                               (0xFBE00000u)
/** Peripheral ADC_1 base pointer */
#define ADC_1                                    ((ADC_Type *)ADC_1_BASE)
/** Array initializer of ADC peripheral base addresses */
#define ADC_BASE_ADDRS                           { ADC_0_BASE, ADC_1_BASE }
/** Array initializer of ADC peripheral base pointers */
#define ADC_BASE_PTRS                            { ADC_0, ADC_1 }

/* ----------------------------------------------------------------------------
   -- ADC Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ADC_Register_Masks ADC Register Masks
 * @{
 */

/* MCR Bit Fields */
#define ADC_MCR_PWDN_MASK                        0x1u
#define ADC_MCR_PWDN_SHIFT                       0u
#define ADC_MCR_PWDN_WIDTH                       1u
#define ADC_MCR_PWDN(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_MCR_PWDN_SHIFT))&ADC_MCR_PWDN_MASK)
#define ADC_MCR_REFSEL_MASK                      0x6u
#define ADC_MCR_REFSEL_SHIFT                     1u
#define ADC_MCR_REFSEL_WIDTH                     2u
#define ADC_MCR_REFSEL(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_REFSEL_SHIFT))&ADC_MCR_REFSEL_MASK)
#define ADC_MCR_ACKO_MASK                        0x20u
#define ADC_MCR_ACKO_SHIFT                       5u
#define ADC_MCR_ACKO_WIDTH                       1u
#define ADC_MCR_ACKO(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_MCR_ACKO_SHIFT))&ADC_MCR_ACKO_MASK)
#define ADC_MCR_ABORT_MASK                       0x40u
#define ADC_MCR_ABORT_SHIFT                      6u
#define ADC_MCR_ABORT_WIDTH                      1u
#define ADC_MCR_ABORT(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_MCR_ABORT_SHIFT))&ADC_MCR_ABORT_MASK)
#define ADC_MCR_ABORT_CHAIN_MASK                 0x80u
#define ADC_MCR_ABORT_CHAIN_SHIFT                7u
#define ADC_MCR_ABORT_CHAIN_WIDTH                1u
#define ADC_MCR_ABORT_CHAIN(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_MCR_ABORT_CHAIN_SHIFT))&ADC_MCR_ABORT_CHAIN_MASK)
#define ADC_MCR_ADCLKSEL_MASK                    0x100u
#define ADC_MCR_ADCLKSEL_SHIFT                   8u
#define ADC_MCR_ADCLKSEL_WIDTH                   1u
#define ADC_MCR_ADCLKSEL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_MCR_ADCLKSEL_SHIFT))&ADC_MCR_ADCLKSEL_MASK)
#define ADC_MCR_CTU_MODE_MASK                    0x10000u
#define ADC_MCR_CTU_MODE_SHIFT                   16u
#define ADC_MCR_CTU_MODE_WIDTH                   1u
#define ADC_MCR_CTU_MODE(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_MCR_CTU_MODE_SHIFT))&ADC_MCR_CTU_MODE_MASK)
#define ADC_MCR_CTUEN_MASK                       0x20000u
#define ADC_MCR_CTUEN_SHIFT                      17u
#define ADC_MCR_CTUEN_WIDTH                      1u
#define ADC_MCR_CTUEN(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_MCR_CTUEN_SHIFT))&ADC_MCR_CTUEN_MASK)
#define ADC_MCR_JSTART_MASK                      0x100000u
#define ADC_MCR_JSTART_SHIFT                     20u
#define ADC_MCR_JSTART_WIDTH                     1u
#define ADC_MCR_JSTART(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_JSTART_SHIFT))&ADC_MCR_JSTART_MASK)
#define ADC_MCR_JEDGE_MASK                       0x200000u
#define ADC_MCR_JEDGE_SHIFT                      21u
#define ADC_MCR_JEDGE_WIDTH                      1u
#define ADC_MCR_JEDGE(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_MCR_JEDGE_SHIFT))&ADC_MCR_JEDGE_MASK)
#define ADC_MCR_JTRGEN_MASK                      0x400000u
#define ADC_MCR_JTRGEN_SHIFT                     22u
#define ADC_MCR_JTRGEN_WIDTH                     1u
#define ADC_MCR_JTRGEN(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_JTRGEN_SHIFT))&ADC_MCR_JTRGEN_MASK)
#define ADC_MCR_NSTART_MASK                      0x1000000u
#define ADC_MCR_NSTART_SHIFT                     24u
#define ADC_MCR_NSTART_WIDTH                     1u
#define ADC_MCR_NSTART(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_NSTART_SHIFT))&ADC_MCR_NSTART_MASK)
#define ADC_MCR_MODE_MASK                        0x20000000u
#define ADC_MCR_MODE_SHIFT                       29u
#define ADC_MCR_MODE_WIDTH                       1u
#define ADC_MCR_MODE(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_MCR_MODE_SHIFT))&ADC_MCR_MODE_MASK)
#define ADC_MCR_WLSIDE_MASK                      0x40000000u
#define ADC_MCR_WLSIDE_SHIFT                     30u
#define ADC_MCR_WLSIDE_WIDTH                     1u
#define ADC_MCR_WLSIDE(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MCR_WLSIDE_SHIFT))&ADC_MCR_WLSIDE_MASK)
#define ADC_MCR_OWREN_MASK                       0x80000000u
#define ADC_MCR_OWREN_SHIFT                      31u
#define ADC_MCR_OWREN_WIDTH                      1u
#define ADC_MCR_OWREN(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_MCR_OWREN_SHIFT))&ADC_MCR_OWREN_MASK)
/* MSR Bit Fields */
#define ADC_MSR_ADCSTATUS_MASK                   0x7u
#define ADC_MSR_ADCSTATUS_SHIFT                  0u
#define ADC_MSR_ADCSTATUS_WIDTH                  3u
#define ADC_MSR_ADCSTATUS(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_MSR_ADCSTATUS_SHIFT))&ADC_MSR_ADCSTATUS_MASK)
#define ADC_MSR_ACKO_MASK                        0x20u
#define ADC_MSR_ACKO_SHIFT                       5u
#define ADC_MSR_ACKO_WIDTH                       1u
#define ADC_MSR_ACKO(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_MSR_ACKO_SHIFT))&ADC_MSR_ACKO_MASK)
#define ADC_MSR_CHADDR_MASK                      0xFE00u
#define ADC_MSR_CHADDR_SHIFT                     9u
#define ADC_MSR_CHADDR_WIDTH                     7u
#define ADC_MSR_CHADDR(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MSR_CHADDR_SHIFT))&ADC_MSR_CHADDR_MASK)
#define ADC_MSR_CTUSTART_MASK                    0x10000u
#define ADC_MSR_CTUSTART_SHIFT                   16u
#define ADC_MSR_CTUSTART_WIDTH                   1u
#define ADC_MSR_CTUSTART(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_MSR_CTUSTART_SHIFT))&ADC_MSR_CTUSTART_MASK)
#define ADC_MSR_JSTART_MASK                      0x100000u
#define ADC_MSR_JSTART_SHIFT                     20u
#define ADC_MSR_JSTART_WIDTH                     1u
#define ADC_MSR_JSTART(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MSR_JSTART_SHIFT))&ADC_MSR_JSTART_MASK)
#define ADC_MSR_JABORT_MASK                      0x800000u
#define ADC_MSR_JABORT_SHIFT                     23u
#define ADC_MSR_JABORT_WIDTH                     1u
#define ADC_MSR_JABORT(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MSR_JABORT_SHIFT))&ADC_MSR_JABORT_MASK)
#define ADC_MSR_NSTART_MASK                      0x1000000u
#define ADC_MSR_NSTART_SHIFT                     24u
#define ADC_MSR_NSTART_WIDTH                     1u
#define ADC_MSR_NSTART(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_MSR_NSTART_SHIFT))&ADC_MSR_NSTART_MASK)
#define ADC_MSR_CALIBRTD_MASK                    0x80000000u
#define ADC_MSR_CALIBRTD_SHIFT                   31u
#define ADC_MSR_CALIBRTD_WIDTH                   1u
#define ADC_MSR_CALIBRTD(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_MSR_CALIBRTD_SHIFT))&ADC_MSR_CALIBRTD_MASK)
/* ISR Bit Fields */
#define ADC_ISR_ECH_MASK                         0x1u
#define ADC_ISR_ECH_SHIFT                        0u
#define ADC_ISR_ECH_WIDTH                        1u
#define ADC_ISR_ECH(x)                           (((uint32_t)(((uint32_t)(x))<<ADC_ISR_ECH_SHIFT))&ADC_ISR_ECH_MASK)
#define ADC_ISR_EOC_MASK                         0x2u
#define ADC_ISR_EOC_SHIFT                        1u
#define ADC_ISR_EOC_WIDTH                        1u
#define ADC_ISR_EOC(x)                           (((uint32_t)(((uint32_t)(x))<<ADC_ISR_EOC_SHIFT))&ADC_ISR_EOC_MASK)
#define ADC_ISR_JECH_MASK                        0x4u
#define ADC_ISR_JECH_SHIFT                       2u
#define ADC_ISR_JECH_WIDTH                       1u
#define ADC_ISR_JECH(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_ISR_JECH_SHIFT))&ADC_ISR_JECH_MASK)
#define ADC_ISR_JEOC_MASK                        0x8u
#define ADC_ISR_JEOC_SHIFT                       3u
#define ADC_ISR_JEOC_WIDTH                       1u
#define ADC_ISR_JEOC(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_ISR_JEOC_SHIFT))&ADC_ISR_JEOC_MASK)
#define ADC_ISR_EOCTU_MASK                       0x10u
#define ADC_ISR_EOCTU_SHIFT                      4u
#define ADC_ISR_EOCTU_WIDTH                      1u
#define ADC_ISR_EOCTU(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_ISR_EOCTU_SHIFT))&ADC_ISR_EOCTU_MASK)
/* CEOCFR Bit Fields */
#define ADC_CEOCFR_EOCCH0_MASK                   0x1u
#define ADC_CEOCFR_EOCCH0_SHIFT                  0u
#define ADC_CEOCFR_EOCCH0_WIDTH                  1u
#define ADC_CEOCFR_EOCCH0(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH0_SHIFT))&ADC_CEOCFR_EOCCH0_MASK)
#define ADC_CEOCFR_EOCCH64_MASK                  0x1u
#define ADC_CEOCFR_EOCCH64_SHIFT                 0u
#define ADC_CEOCFR_EOCCH64_WIDTH                 1u
#define ADC_CEOCFR_EOCCH64(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH64_SHIFT))&ADC_CEOCFR_EOCCH64_MASK)
#define ADC_CEOCFR_EOCCH32_MASK                  0x1u
#define ADC_CEOCFR_EOCCH32_SHIFT                 0u
#define ADC_CEOCFR_EOCCH32_WIDTH                 1u
#define ADC_CEOCFR_EOCCH32(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH32_SHIFT))&ADC_CEOCFR_EOCCH32_MASK)
#define ADC_CEOCFR_EOCCH65_MASK                  0x2u
#define ADC_CEOCFR_EOCCH65_SHIFT                 1u
#define ADC_CEOCFR_EOCCH65_WIDTH                 1u
#define ADC_CEOCFR_EOCCH65(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH65_SHIFT))&ADC_CEOCFR_EOCCH65_MASK)
#define ADC_CEOCFR_EOCCH33_MASK                  0x2u
#define ADC_CEOCFR_EOCCH33_SHIFT                 1u
#define ADC_CEOCFR_EOCCH33_WIDTH                 1u
#define ADC_CEOCFR_EOCCH33(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH33_SHIFT))&ADC_CEOCFR_EOCCH33_MASK)
#define ADC_CEOCFR_EOCCH1_MASK                   0x2u
#define ADC_CEOCFR_EOCCH1_SHIFT                  1u
#define ADC_CEOCFR_EOCCH1_WIDTH                  1u
#define ADC_CEOCFR_EOCCH1(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH1_SHIFT))&ADC_CEOCFR_EOCCH1_MASK)
#define ADC_CEOCFR_EOCCH34_MASK                  0x4u
#define ADC_CEOCFR_EOCCH34_SHIFT                 2u
#define ADC_CEOCFR_EOCCH34_WIDTH                 1u
#define ADC_CEOCFR_EOCCH34(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH34_SHIFT))&ADC_CEOCFR_EOCCH34_MASK)
#define ADC_CEOCFR_EOCCH2_MASK                   0x4u
#define ADC_CEOCFR_EOCCH2_SHIFT                  2u
#define ADC_CEOCFR_EOCCH2_WIDTH                  1u
#define ADC_CEOCFR_EOCCH2(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH2_SHIFT))&ADC_CEOCFR_EOCCH2_MASK)
#define ADC_CEOCFR_EOCCH66_MASK                  0x4u
#define ADC_CEOCFR_EOCCH66_SHIFT                 2u
#define ADC_CEOCFR_EOCCH66_WIDTH                 1u
#define ADC_CEOCFR_EOCCH66(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH66_SHIFT))&ADC_CEOCFR_EOCCH66_MASK)
#define ADC_CEOCFR_EOCCH67_MASK                  0x8u
#define ADC_CEOCFR_EOCCH67_SHIFT                 3u
#define ADC_CEOCFR_EOCCH67_WIDTH                 1u
#define ADC_CEOCFR_EOCCH67(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH67_SHIFT))&ADC_CEOCFR_EOCCH67_MASK)
#define ADC_CEOCFR_EOCCH3_MASK                   0x8u
#define ADC_CEOCFR_EOCCH3_SHIFT                  3u
#define ADC_CEOCFR_EOCCH3_WIDTH                  1u
#define ADC_CEOCFR_EOCCH3(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH3_SHIFT))&ADC_CEOCFR_EOCCH3_MASK)
#define ADC_CEOCFR_EOCCH35_MASK                  0x8u
#define ADC_CEOCFR_EOCCH35_SHIFT                 3u
#define ADC_CEOCFR_EOCCH35_WIDTH                 1u
#define ADC_CEOCFR_EOCCH35(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH35_SHIFT))&ADC_CEOCFR_EOCCH35_MASK)
#define ADC_CEOCFR_EOCCH4_MASK                   0x10u
#define ADC_CEOCFR_EOCCH4_SHIFT                  4u
#define ADC_CEOCFR_EOCCH4_WIDTH                  1u
#define ADC_CEOCFR_EOCCH4(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH4_SHIFT))&ADC_CEOCFR_EOCCH4_MASK)
#define ADC_CEOCFR_EOCCH36_MASK                  0x10u
#define ADC_CEOCFR_EOCCH36_SHIFT                 4u
#define ADC_CEOCFR_EOCCH36_WIDTH                 1u
#define ADC_CEOCFR_EOCCH36(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH36_SHIFT))&ADC_CEOCFR_EOCCH36_MASK)
#define ADC_CEOCFR_EOCCH68_MASK                  0x10u
#define ADC_CEOCFR_EOCCH68_SHIFT                 4u
#define ADC_CEOCFR_EOCCH68_WIDTH                 1u
#define ADC_CEOCFR_EOCCH68(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH68_SHIFT))&ADC_CEOCFR_EOCCH68_MASK)
#define ADC_CEOCFR_EOCCH69_MASK                  0x20u
#define ADC_CEOCFR_EOCCH69_SHIFT                 5u
#define ADC_CEOCFR_EOCCH69_WIDTH                 1u
#define ADC_CEOCFR_EOCCH69(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH69_SHIFT))&ADC_CEOCFR_EOCCH69_MASK)
#define ADC_CEOCFR_EOCCH37_MASK                  0x20u
#define ADC_CEOCFR_EOCCH37_SHIFT                 5u
#define ADC_CEOCFR_EOCCH37_WIDTH                 1u
#define ADC_CEOCFR_EOCCH37(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH37_SHIFT))&ADC_CEOCFR_EOCCH37_MASK)
#define ADC_CEOCFR_EOCCH5_MASK                   0x20u
#define ADC_CEOCFR_EOCCH5_SHIFT                  5u
#define ADC_CEOCFR_EOCCH5_WIDTH                  1u
#define ADC_CEOCFR_EOCCH5(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH5_SHIFT))&ADC_CEOCFR_EOCCH5_MASK)
#define ADC_CEOCFR_EOCCH70_MASK                  0x40u
#define ADC_CEOCFR_EOCCH70_SHIFT                 6u
#define ADC_CEOCFR_EOCCH70_WIDTH                 1u
#define ADC_CEOCFR_EOCCH70(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH70_SHIFT))&ADC_CEOCFR_EOCCH70_MASK)
#define ADC_CEOCFR_EOCCH38_MASK                  0x40u
#define ADC_CEOCFR_EOCCH38_SHIFT                 6u
#define ADC_CEOCFR_EOCCH38_WIDTH                 1u
#define ADC_CEOCFR_EOCCH38(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH38_SHIFT))&ADC_CEOCFR_EOCCH38_MASK)
#define ADC_CEOCFR_EOCCH6_MASK                   0x40u
#define ADC_CEOCFR_EOCCH6_SHIFT                  6u
#define ADC_CEOCFR_EOCCH6_WIDTH                  1u
#define ADC_CEOCFR_EOCCH6(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH6_SHIFT))&ADC_CEOCFR_EOCCH6_MASK)
#define ADC_CEOCFR_EOCCH7_MASK                   0x80u
#define ADC_CEOCFR_EOCCH7_SHIFT                  7u
#define ADC_CEOCFR_EOCCH7_WIDTH                  1u
#define ADC_CEOCFR_EOCCH7(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH7_SHIFT))&ADC_CEOCFR_EOCCH7_MASK)
#define ADC_CEOCFR_EOCCH39_MASK                  0x80u
#define ADC_CEOCFR_EOCCH39_SHIFT                 7u
#define ADC_CEOCFR_EOCCH39_WIDTH                 1u
#define ADC_CEOCFR_EOCCH39(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH39_SHIFT))&ADC_CEOCFR_EOCCH39_MASK)
#define ADC_CEOCFR_EOCCH71_MASK                  0x80u
#define ADC_CEOCFR_EOCCH71_SHIFT                 7u
#define ADC_CEOCFR_EOCCH71_WIDTH                 1u
#define ADC_CEOCFR_EOCCH71(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH71_SHIFT))&ADC_CEOCFR_EOCCH71_MASK)
#define ADC_CEOCFR_EOCCH8_MASK                   0x100u
#define ADC_CEOCFR_EOCCH8_SHIFT                  8u
#define ADC_CEOCFR_EOCCH8_WIDTH                  1u
#define ADC_CEOCFR_EOCCH8(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH8_SHIFT))&ADC_CEOCFR_EOCCH8_MASK)
#define ADC_CEOCFR_EOCCH40_MASK                  0x100u
#define ADC_CEOCFR_EOCCH40_SHIFT                 8u
#define ADC_CEOCFR_EOCCH40_WIDTH                 1u
#define ADC_CEOCFR_EOCCH40(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH40_SHIFT))&ADC_CEOCFR_EOCCH40_MASK)
#define ADC_CEOCFR_EOCCH72_MASK                  0x100u
#define ADC_CEOCFR_EOCCH72_SHIFT                 8u
#define ADC_CEOCFR_EOCCH72_WIDTH                 1u
#define ADC_CEOCFR_EOCCH72(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH72_SHIFT))&ADC_CEOCFR_EOCCH72_MASK)
#define ADC_CEOCFR_EOCCH73_MASK                  0x200u
#define ADC_CEOCFR_EOCCH73_SHIFT                 9u
#define ADC_CEOCFR_EOCCH73_WIDTH                 1u
#define ADC_CEOCFR_EOCCH73(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH73_SHIFT))&ADC_CEOCFR_EOCCH73_MASK)
#define ADC_CEOCFR_EOCCH41_MASK                  0x200u
#define ADC_CEOCFR_EOCCH41_SHIFT                 9u
#define ADC_CEOCFR_EOCCH41_WIDTH                 1u
#define ADC_CEOCFR_EOCCH41(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH41_SHIFT))&ADC_CEOCFR_EOCCH41_MASK)
#define ADC_CEOCFR_EOCCH9_MASK                   0x200u
#define ADC_CEOCFR_EOCCH9_SHIFT                  9u
#define ADC_CEOCFR_EOCCH9_WIDTH                  1u
#define ADC_CEOCFR_EOCCH9(x)                     (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH9_SHIFT))&ADC_CEOCFR_EOCCH9_MASK)
#define ADC_CEOCFR_EOCCH42_MASK                  0x400u
#define ADC_CEOCFR_EOCCH42_SHIFT                 10u
#define ADC_CEOCFR_EOCCH42_WIDTH                 1u
#define ADC_CEOCFR_EOCCH42(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH42_SHIFT))&ADC_CEOCFR_EOCCH42_MASK)
#define ADC_CEOCFR_EOCCH74_MASK                  0x400u
#define ADC_CEOCFR_EOCCH74_SHIFT                 10u
#define ADC_CEOCFR_EOCCH74_WIDTH                 1u
#define ADC_CEOCFR_EOCCH74(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH74_SHIFT))&ADC_CEOCFR_EOCCH74_MASK)
#define ADC_CEOCFR_EOCCH10_MASK                  0x400u
#define ADC_CEOCFR_EOCCH10_SHIFT                 10u
#define ADC_CEOCFR_EOCCH10_WIDTH                 1u
#define ADC_CEOCFR_EOCCH10(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH10_SHIFT))&ADC_CEOCFR_EOCCH10_MASK)
#define ADC_CEOCFR_EOCCH11_MASK                  0x800u
#define ADC_CEOCFR_EOCCH11_SHIFT                 11u
#define ADC_CEOCFR_EOCCH11_WIDTH                 1u
#define ADC_CEOCFR_EOCCH11(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH11_SHIFT))&ADC_CEOCFR_EOCCH11_MASK)
#define ADC_CEOCFR_EOCCH75_MASK                  0x800u
#define ADC_CEOCFR_EOCCH75_SHIFT                 11u
#define ADC_CEOCFR_EOCCH75_WIDTH                 1u
#define ADC_CEOCFR_EOCCH75(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH75_SHIFT))&ADC_CEOCFR_EOCCH75_MASK)
#define ADC_CEOCFR_EOCCH43_MASK                  0x800u
#define ADC_CEOCFR_EOCCH43_SHIFT                 11u
#define ADC_CEOCFR_EOCCH43_WIDTH                 1u
#define ADC_CEOCFR_EOCCH43(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH43_SHIFT))&ADC_CEOCFR_EOCCH43_MASK)
#define ADC_CEOCFR_EOCCH76_MASK                  0x1000u
#define ADC_CEOCFR_EOCCH76_SHIFT                 12u
#define ADC_CEOCFR_EOCCH76_WIDTH                 1u
#define ADC_CEOCFR_EOCCH76(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH76_SHIFT))&ADC_CEOCFR_EOCCH76_MASK)
#define ADC_CEOCFR_EOCCH12_MASK                  0x1000u
#define ADC_CEOCFR_EOCCH12_SHIFT                 12u
#define ADC_CEOCFR_EOCCH12_WIDTH                 1u
#define ADC_CEOCFR_EOCCH12(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH12_SHIFT))&ADC_CEOCFR_EOCCH12_MASK)
#define ADC_CEOCFR_EOCCH44_MASK                  0x1000u
#define ADC_CEOCFR_EOCCH44_SHIFT                 12u
#define ADC_CEOCFR_EOCCH44_WIDTH                 1u
#define ADC_CEOCFR_EOCCH44(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH44_SHIFT))&ADC_CEOCFR_EOCCH44_MASK)
#define ADC_CEOCFR_EOCCH45_MASK                  0x2000u
#define ADC_CEOCFR_EOCCH45_SHIFT                 13u
#define ADC_CEOCFR_EOCCH45_WIDTH                 1u
#define ADC_CEOCFR_EOCCH45(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH45_SHIFT))&ADC_CEOCFR_EOCCH45_MASK)
#define ADC_CEOCFR_EOCCH77_MASK                  0x2000u
#define ADC_CEOCFR_EOCCH77_SHIFT                 13u
#define ADC_CEOCFR_EOCCH77_WIDTH                 1u
#define ADC_CEOCFR_EOCCH77(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH77_SHIFT))&ADC_CEOCFR_EOCCH77_MASK)
#define ADC_CEOCFR_EOCCH13_MASK                  0x2000u
#define ADC_CEOCFR_EOCCH13_SHIFT                 13u
#define ADC_CEOCFR_EOCCH13_WIDTH                 1u
#define ADC_CEOCFR_EOCCH13(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH13_SHIFT))&ADC_CEOCFR_EOCCH13_MASK)
#define ADC_CEOCFR_EOCCH14_MASK                  0x4000u
#define ADC_CEOCFR_EOCCH14_SHIFT                 14u
#define ADC_CEOCFR_EOCCH14_WIDTH                 1u
#define ADC_CEOCFR_EOCCH14(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH14_SHIFT))&ADC_CEOCFR_EOCCH14_MASK)
#define ADC_CEOCFR_EOCCH46_MASK                  0x4000u
#define ADC_CEOCFR_EOCCH46_SHIFT                 14u
#define ADC_CEOCFR_EOCCH46_WIDTH                 1u
#define ADC_CEOCFR_EOCCH46(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH46_SHIFT))&ADC_CEOCFR_EOCCH46_MASK)
#define ADC_CEOCFR_EOCCH78_MASK                  0x4000u
#define ADC_CEOCFR_EOCCH78_SHIFT                 14u
#define ADC_CEOCFR_EOCCH78_WIDTH                 1u
#define ADC_CEOCFR_EOCCH78(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH78_SHIFT))&ADC_CEOCFR_EOCCH78_MASK)
#define ADC_CEOCFR_EOCCH47_MASK                  0x8000u
#define ADC_CEOCFR_EOCCH47_SHIFT                 15u
#define ADC_CEOCFR_EOCCH47_WIDTH                 1u
#define ADC_CEOCFR_EOCCH47(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH47_SHIFT))&ADC_CEOCFR_EOCCH47_MASK)
#define ADC_CEOCFR_EOCCH15_MASK                  0x8000u
#define ADC_CEOCFR_EOCCH15_SHIFT                 15u
#define ADC_CEOCFR_EOCCH15_WIDTH                 1u
#define ADC_CEOCFR_EOCCH15(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH15_SHIFT))&ADC_CEOCFR_EOCCH15_MASK)
#define ADC_CEOCFR_EOCCH79_MASK                  0x8000u
#define ADC_CEOCFR_EOCCH79_SHIFT                 15u
#define ADC_CEOCFR_EOCCH79_WIDTH                 1u
#define ADC_CEOCFR_EOCCH79(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH79_SHIFT))&ADC_CEOCFR_EOCCH79_MASK)
#define ADC_CEOCFR_EOCCH80_MASK                  0x10000u
#define ADC_CEOCFR_EOCCH80_SHIFT                 16u
#define ADC_CEOCFR_EOCCH80_WIDTH                 1u
#define ADC_CEOCFR_EOCCH80(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH80_SHIFT))&ADC_CEOCFR_EOCCH80_MASK)
#define ADC_CEOCFR_EOCCH48_MASK                  0x10000u
#define ADC_CEOCFR_EOCCH48_SHIFT                 16u
#define ADC_CEOCFR_EOCCH48_WIDTH                 1u
#define ADC_CEOCFR_EOCCH48(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH48_SHIFT))&ADC_CEOCFR_EOCCH48_MASK)
#define ADC_CEOCFR_EOCCH81_MASK                  0x20000u
#define ADC_CEOCFR_EOCCH81_SHIFT                 17u
#define ADC_CEOCFR_EOCCH81_WIDTH                 1u
#define ADC_CEOCFR_EOCCH81(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH81_SHIFT))&ADC_CEOCFR_EOCCH81_MASK)
#define ADC_CEOCFR_EOCCH49_MASK                  0x20000u
#define ADC_CEOCFR_EOCCH49_SHIFT                 17u
#define ADC_CEOCFR_EOCCH49_WIDTH                 1u
#define ADC_CEOCFR_EOCCH49(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH49_SHIFT))&ADC_CEOCFR_EOCCH49_MASK)
#define ADC_CEOCFR_EOCCH50_MASK                  0x40000u
#define ADC_CEOCFR_EOCCH50_SHIFT                 18u
#define ADC_CEOCFR_EOCCH50_WIDTH                 1u
#define ADC_CEOCFR_EOCCH50(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH50_SHIFT))&ADC_CEOCFR_EOCCH50_MASK)
#define ADC_CEOCFR_EOCCH82_MASK                  0x40000u
#define ADC_CEOCFR_EOCCH82_SHIFT                 18u
#define ADC_CEOCFR_EOCCH82_WIDTH                 1u
#define ADC_CEOCFR_EOCCH82(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH82_SHIFT))&ADC_CEOCFR_EOCCH82_MASK)
#define ADC_CEOCFR_EOCCH51_MASK                  0x80000u
#define ADC_CEOCFR_EOCCH51_SHIFT                 19u
#define ADC_CEOCFR_EOCCH51_WIDTH                 1u
#define ADC_CEOCFR_EOCCH51(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH51_SHIFT))&ADC_CEOCFR_EOCCH51_MASK)
#define ADC_CEOCFR_EOCCH83_MASK                  0x80000u
#define ADC_CEOCFR_EOCCH83_SHIFT                 19u
#define ADC_CEOCFR_EOCCH83_WIDTH                 1u
#define ADC_CEOCFR_EOCCH83(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH83_SHIFT))&ADC_CEOCFR_EOCCH83_MASK)
#define ADC_CEOCFR_EOCCH52_MASK                  0x100000u
#define ADC_CEOCFR_EOCCH52_SHIFT                 20u
#define ADC_CEOCFR_EOCCH52_WIDTH                 1u
#define ADC_CEOCFR_EOCCH52(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH52_SHIFT))&ADC_CEOCFR_EOCCH52_MASK)
#define ADC_CEOCFR_EOCCH84_MASK                  0x100000u
#define ADC_CEOCFR_EOCCH84_SHIFT                 20u
#define ADC_CEOCFR_EOCCH84_WIDTH                 1u
#define ADC_CEOCFR_EOCCH84(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH84_SHIFT))&ADC_CEOCFR_EOCCH84_MASK)
#define ADC_CEOCFR_EOCCH53_MASK                  0x200000u
#define ADC_CEOCFR_EOCCH53_SHIFT                 21u
#define ADC_CEOCFR_EOCCH53_WIDTH                 1u
#define ADC_CEOCFR_EOCCH53(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH53_SHIFT))&ADC_CEOCFR_EOCCH53_MASK)
#define ADC_CEOCFR_EOCCH85_MASK                  0x200000u
#define ADC_CEOCFR_EOCCH85_SHIFT                 21u
#define ADC_CEOCFR_EOCCH85_WIDTH                 1u
#define ADC_CEOCFR_EOCCH85(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH85_SHIFT))&ADC_CEOCFR_EOCCH85_MASK)
#define ADC_CEOCFR_EOCCH54_MASK                  0x400000u
#define ADC_CEOCFR_EOCCH54_SHIFT                 22u
#define ADC_CEOCFR_EOCCH54_WIDTH                 1u
#define ADC_CEOCFR_EOCCH54(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH54_SHIFT))&ADC_CEOCFR_EOCCH54_MASK)
#define ADC_CEOCFR_EOCCH86_MASK                  0x400000u
#define ADC_CEOCFR_EOCCH86_SHIFT                 22u
#define ADC_CEOCFR_EOCCH86_WIDTH                 1u
#define ADC_CEOCFR_EOCCH86(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH86_SHIFT))&ADC_CEOCFR_EOCCH86_MASK)
#define ADC_CEOCFR_EOCCH55_MASK                  0x800000u
#define ADC_CEOCFR_EOCCH55_SHIFT                 23u
#define ADC_CEOCFR_EOCCH55_WIDTH                 1u
#define ADC_CEOCFR_EOCCH55(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH55_SHIFT))&ADC_CEOCFR_EOCCH55_MASK)
#define ADC_CEOCFR_EOCCH87_MASK                  0x800000u
#define ADC_CEOCFR_EOCCH87_SHIFT                 23u
#define ADC_CEOCFR_EOCCH87_WIDTH                 1u
#define ADC_CEOCFR_EOCCH87(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH87_SHIFT))&ADC_CEOCFR_EOCCH87_MASK)
#define ADC_CEOCFR_EOCCH88_MASK                  0x1000000u
#define ADC_CEOCFR_EOCCH88_SHIFT                 24u
#define ADC_CEOCFR_EOCCH88_WIDTH                 1u
#define ADC_CEOCFR_EOCCH88(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH88_SHIFT))&ADC_CEOCFR_EOCCH88_MASK)
#define ADC_CEOCFR_EOCCH56_MASK                  0x1000000u
#define ADC_CEOCFR_EOCCH56_SHIFT                 24u
#define ADC_CEOCFR_EOCCH56_WIDTH                 1u
#define ADC_CEOCFR_EOCCH56(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH56_SHIFT))&ADC_CEOCFR_EOCCH56_MASK)
#define ADC_CEOCFR_EOCCH89_MASK                  0x2000000u
#define ADC_CEOCFR_EOCCH89_SHIFT                 25u
#define ADC_CEOCFR_EOCCH89_WIDTH                 1u
#define ADC_CEOCFR_EOCCH89(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH89_SHIFT))&ADC_CEOCFR_EOCCH89_MASK)
#define ADC_CEOCFR_EOCCH57_MASK                  0x2000000u
#define ADC_CEOCFR_EOCCH57_SHIFT                 25u
#define ADC_CEOCFR_EOCCH57_WIDTH                 1u
#define ADC_CEOCFR_EOCCH57(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH57_SHIFT))&ADC_CEOCFR_EOCCH57_MASK)
#define ADC_CEOCFR_EOCCH90_MASK                  0x4000000u
#define ADC_CEOCFR_EOCCH90_SHIFT                 26u
#define ADC_CEOCFR_EOCCH90_WIDTH                 1u
#define ADC_CEOCFR_EOCCH90(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH90_SHIFT))&ADC_CEOCFR_EOCCH90_MASK)
#define ADC_CEOCFR_EOCCH58_MASK                  0x4000000u
#define ADC_CEOCFR_EOCCH58_SHIFT                 26u
#define ADC_CEOCFR_EOCCH58_WIDTH                 1u
#define ADC_CEOCFR_EOCCH58(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH58_SHIFT))&ADC_CEOCFR_EOCCH58_MASK)
#define ADC_CEOCFR_EOCCH59_MASK                  0x8000000u
#define ADC_CEOCFR_EOCCH59_SHIFT                 27u
#define ADC_CEOCFR_EOCCH59_WIDTH                 1u
#define ADC_CEOCFR_EOCCH59(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH59_SHIFT))&ADC_CEOCFR_EOCCH59_MASK)
#define ADC_CEOCFR_EOCCH91_MASK                  0x8000000u
#define ADC_CEOCFR_EOCCH91_SHIFT                 27u
#define ADC_CEOCFR_EOCCH91_WIDTH                 1u
#define ADC_CEOCFR_EOCCH91(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH91_SHIFT))&ADC_CEOCFR_EOCCH91_MASK)
#define ADC_CEOCFR_EOCCH60_MASK                  0x10000000u
#define ADC_CEOCFR_EOCCH60_SHIFT                 28u
#define ADC_CEOCFR_EOCCH60_WIDTH                 1u
#define ADC_CEOCFR_EOCCH60(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH60_SHIFT))&ADC_CEOCFR_EOCCH60_MASK)
#define ADC_CEOCFR_EOCCH92_MASK                  0x10000000u
#define ADC_CEOCFR_EOCCH92_SHIFT                 28u
#define ADC_CEOCFR_EOCCH92_WIDTH                 1u
#define ADC_CEOCFR_EOCCH92(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH92_SHIFT))&ADC_CEOCFR_EOCCH92_MASK)
#define ADC_CEOCFR_EOCCH61_MASK                  0x20000000u
#define ADC_CEOCFR_EOCCH61_SHIFT                 29u
#define ADC_CEOCFR_EOCCH61_WIDTH                 1u
#define ADC_CEOCFR_EOCCH61(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH61_SHIFT))&ADC_CEOCFR_EOCCH61_MASK)
#define ADC_CEOCFR_EOCCH93_MASK                  0x20000000u
#define ADC_CEOCFR_EOCCH93_SHIFT                 29u
#define ADC_CEOCFR_EOCCH93_WIDTH                 1u
#define ADC_CEOCFR_EOCCH93(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH93_SHIFT))&ADC_CEOCFR_EOCCH93_MASK)
#define ADC_CEOCFR_EOCCH94_MASK                  0x40000000u
#define ADC_CEOCFR_EOCCH94_SHIFT                 30u
#define ADC_CEOCFR_EOCCH94_WIDTH                 1u
#define ADC_CEOCFR_EOCCH94(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH94_SHIFT))&ADC_CEOCFR_EOCCH94_MASK)
#define ADC_CEOCFR_EOCCH62_MASK                  0x40000000u
#define ADC_CEOCFR_EOCCH62_SHIFT                 30u
#define ADC_CEOCFR_EOCCH62_WIDTH                 1u
#define ADC_CEOCFR_EOCCH62(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH62_SHIFT))&ADC_CEOCFR_EOCCH62_MASK)
#define ADC_CEOCFR_EOCCH95_MASK                  0x80000000u
#define ADC_CEOCFR_EOCCH95_SHIFT                 31u
#define ADC_CEOCFR_EOCCH95_WIDTH                 1u
#define ADC_CEOCFR_EOCCH95(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH95_SHIFT))&ADC_CEOCFR_EOCCH95_MASK)
#define ADC_CEOCFR_EOCCH63_MASK                  0x80000000u
#define ADC_CEOCFR_EOCCH63_SHIFT                 31u
#define ADC_CEOCFR_EOCCH63_WIDTH                 1u
#define ADC_CEOCFR_EOCCH63(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_CEOCFR_EOCCH63_SHIFT))&ADC_CEOCFR_EOCCH63_MASK)
/* IMR Bit Fields */
#define ADC_IMR_MSKECH_MASK                      0x1u
#define ADC_IMR_MSKECH_SHIFT                     0u
#define ADC_IMR_MSKECH_WIDTH                     1u
#define ADC_IMR_MSKECH(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_IMR_MSKECH_SHIFT))&ADC_IMR_MSKECH_MASK)
#define ADC_IMR_MSKEOC_MASK                      0x2u
#define ADC_IMR_MSKEOC_SHIFT                     1u
#define ADC_IMR_MSKEOC_WIDTH                     1u
#define ADC_IMR_MSKEOC(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_IMR_MSKEOC_SHIFT))&ADC_IMR_MSKEOC_MASK)
#define ADC_IMR_MSKJECH_MASK                     0x4u
#define ADC_IMR_MSKJECH_SHIFT                    2u
#define ADC_IMR_MSKJECH_WIDTH                    1u
#define ADC_IMR_MSKJECH(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_IMR_MSKJECH_SHIFT))&ADC_IMR_MSKJECH_MASK)
#define ADC_IMR_MSKJEOC_MASK                     0x8u
#define ADC_IMR_MSKJEOC_SHIFT                    3u
#define ADC_IMR_MSKJEOC_WIDTH                    1u
#define ADC_IMR_MSKJEOC(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_IMR_MSKJEOC_SHIFT))&ADC_IMR_MSKJEOC_MASK)
#define ADC_IMR_MSKEOCTU_MASK                    0x10u
#define ADC_IMR_MSKEOCTU_SHIFT                   4u
#define ADC_IMR_MSKEOCTU_WIDTH                   1u
#define ADC_IMR_MSKEOCTU(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_IMR_MSKEOCTU_SHIFT))&ADC_IMR_MSKEOCTU_MASK)
/* CIMR Bit Fields */
#define ADC_CIMR_CIM0_MASK                       0x1u
#define ADC_CIMR_CIM0_SHIFT                      0u
#define ADC_CIMR_CIM0_WIDTH                      1u
#define ADC_CIMR_CIM0(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM0_SHIFT))&ADC_CIMR_CIM0_MASK)
#define ADC_CIMR_CIM64_MASK                      0x1u
#define ADC_CIMR_CIM64_SHIFT                     0u
#define ADC_CIMR_CIM64_WIDTH                     1u
#define ADC_CIMR_CIM64(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM64_SHIFT))&ADC_CIMR_CIM64_MASK)
#define ADC_CIMR_CIM32_MASK                      0x1u
#define ADC_CIMR_CIM32_SHIFT                     0u
#define ADC_CIMR_CIM32_WIDTH                     1u
#define ADC_CIMR_CIM32(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM32_SHIFT))&ADC_CIMR_CIM32_MASK)
#define ADC_CIMR_CIM65_MASK                      0x2u
#define ADC_CIMR_CIM65_SHIFT                     1u
#define ADC_CIMR_CIM65_WIDTH                     1u
#define ADC_CIMR_CIM65(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM65_SHIFT))&ADC_CIMR_CIM65_MASK)
#define ADC_CIMR_CIM33_MASK                      0x2u
#define ADC_CIMR_CIM33_SHIFT                     1u
#define ADC_CIMR_CIM33_WIDTH                     1u
#define ADC_CIMR_CIM33(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM33_SHIFT))&ADC_CIMR_CIM33_MASK)
#define ADC_CIMR_CIM1_MASK                       0x2u
#define ADC_CIMR_CIM1_SHIFT                      1u
#define ADC_CIMR_CIM1_WIDTH                      1u
#define ADC_CIMR_CIM1(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM1_SHIFT))&ADC_CIMR_CIM1_MASK)
#define ADC_CIMR_CIM34_MASK                      0x4u
#define ADC_CIMR_CIM34_SHIFT                     2u
#define ADC_CIMR_CIM34_WIDTH                     1u
#define ADC_CIMR_CIM34(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM34_SHIFT))&ADC_CIMR_CIM34_MASK)
#define ADC_CIMR_CIM2_MASK                       0x4u
#define ADC_CIMR_CIM2_SHIFT                      2u
#define ADC_CIMR_CIM2_WIDTH                      1u
#define ADC_CIMR_CIM2(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM2_SHIFT))&ADC_CIMR_CIM2_MASK)
#define ADC_CIMR_CIM66_MASK                      0x4u
#define ADC_CIMR_CIM66_SHIFT                     2u
#define ADC_CIMR_CIM66_WIDTH                     1u
#define ADC_CIMR_CIM66(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM66_SHIFT))&ADC_CIMR_CIM66_MASK)
#define ADC_CIMR_CIM67_MASK                      0x8u
#define ADC_CIMR_CIM67_SHIFT                     3u
#define ADC_CIMR_CIM67_WIDTH                     1u
#define ADC_CIMR_CIM67(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM67_SHIFT))&ADC_CIMR_CIM67_MASK)
#define ADC_CIMR_CIM3_MASK                       0x8u
#define ADC_CIMR_CIM3_SHIFT                      3u
#define ADC_CIMR_CIM3_WIDTH                      1u
#define ADC_CIMR_CIM3(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM3_SHIFT))&ADC_CIMR_CIM3_MASK)
#define ADC_CIMR_CIM35_MASK                      0x8u
#define ADC_CIMR_CIM35_SHIFT                     3u
#define ADC_CIMR_CIM35_WIDTH                     1u
#define ADC_CIMR_CIM35(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM35_SHIFT))&ADC_CIMR_CIM35_MASK)
#define ADC_CIMR_CIM4_MASK                       0x10u
#define ADC_CIMR_CIM4_SHIFT                      4u
#define ADC_CIMR_CIM4_WIDTH                      1u
#define ADC_CIMR_CIM4(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM4_SHIFT))&ADC_CIMR_CIM4_MASK)
#define ADC_CIMR_CIM36_MASK                      0x10u
#define ADC_CIMR_CIM36_SHIFT                     4u
#define ADC_CIMR_CIM36_WIDTH                     1u
#define ADC_CIMR_CIM36(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM36_SHIFT))&ADC_CIMR_CIM36_MASK)
#define ADC_CIMR_CIM68_MASK                      0x10u
#define ADC_CIMR_CIM68_SHIFT                     4u
#define ADC_CIMR_CIM68_WIDTH                     1u
#define ADC_CIMR_CIM68(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM68_SHIFT))&ADC_CIMR_CIM68_MASK)
#define ADC_CIMR_CIM69_MASK                      0x20u
#define ADC_CIMR_CIM69_SHIFT                     5u
#define ADC_CIMR_CIM69_WIDTH                     1u
#define ADC_CIMR_CIM69(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM69_SHIFT))&ADC_CIMR_CIM69_MASK)
#define ADC_CIMR_CIM37_MASK                      0x20u
#define ADC_CIMR_CIM37_SHIFT                     5u
#define ADC_CIMR_CIM37_WIDTH                     1u
#define ADC_CIMR_CIM37(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM37_SHIFT))&ADC_CIMR_CIM37_MASK)
#define ADC_CIMR_CIM5_MASK                       0x20u
#define ADC_CIMR_CIM5_SHIFT                      5u
#define ADC_CIMR_CIM5_WIDTH                      1u
#define ADC_CIMR_CIM5(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM5_SHIFT))&ADC_CIMR_CIM5_MASK)
#define ADC_CIMR_CIM70_MASK                      0x40u
#define ADC_CIMR_CIM70_SHIFT                     6u
#define ADC_CIMR_CIM70_WIDTH                     1u
#define ADC_CIMR_CIM70(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM70_SHIFT))&ADC_CIMR_CIM70_MASK)
#define ADC_CIMR_CIM38_MASK                      0x40u
#define ADC_CIMR_CIM38_SHIFT                     6u
#define ADC_CIMR_CIM38_WIDTH                     1u
#define ADC_CIMR_CIM38(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM38_SHIFT))&ADC_CIMR_CIM38_MASK)
#define ADC_CIMR_CIM6_MASK                       0x40u
#define ADC_CIMR_CIM6_SHIFT                      6u
#define ADC_CIMR_CIM6_WIDTH                      1u
#define ADC_CIMR_CIM6(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM6_SHIFT))&ADC_CIMR_CIM6_MASK)
#define ADC_CIMR_CIM7_MASK                       0x80u
#define ADC_CIMR_CIM7_SHIFT                      7u
#define ADC_CIMR_CIM7_WIDTH                      1u
#define ADC_CIMR_CIM7(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM7_SHIFT))&ADC_CIMR_CIM7_MASK)
#define ADC_CIMR_CIM39_MASK                      0x80u
#define ADC_CIMR_CIM39_SHIFT                     7u
#define ADC_CIMR_CIM39_WIDTH                     1u
#define ADC_CIMR_CIM39(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM39_SHIFT))&ADC_CIMR_CIM39_MASK)
#define ADC_CIMR_CIM71_MASK                      0x80u
#define ADC_CIMR_CIM71_SHIFT                     7u
#define ADC_CIMR_CIM71_WIDTH                     1u
#define ADC_CIMR_CIM71(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM71_SHIFT))&ADC_CIMR_CIM71_MASK)
#define ADC_CIMR_CIM8_MASK                       0x100u
#define ADC_CIMR_CIM8_SHIFT                      8u
#define ADC_CIMR_CIM8_WIDTH                      1u
#define ADC_CIMR_CIM8(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM8_SHIFT))&ADC_CIMR_CIM8_MASK)
#define ADC_CIMR_CIM40_MASK                      0x100u
#define ADC_CIMR_CIM40_SHIFT                     8u
#define ADC_CIMR_CIM40_WIDTH                     1u
#define ADC_CIMR_CIM40(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM40_SHIFT))&ADC_CIMR_CIM40_MASK)
#define ADC_CIMR_CIM72_MASK                      0x100u
#define ADC_CIMR_CIM72_SHIFT                     8u
#define ADC_CIMR_CIM72_WIDTH                     1u
#define ADC_CIMR_CIM72(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM72_SHIFT))&ADC_CIMR_CIM72_MASK)
#define ADC_CIMR_CIM73_MASK                      0x200u
#define ADC_CIMR_CIM73_SHIFT                     9u
#define ADC_CIMR_CIM73_WIDTH                     1u
#define ADC_CIMR_CIM73(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM73_SHIFT))&ADC_CIMR_CIM73_MASK)
#define ADC_CIMR_CIM41_MASK                      0x200u
#define ADC_CIMR_CIM41_SHIFT                     9u
#define ADC_CIMR_CIM41_WIDTH                     1u
#define ADC_CIMR_CIM41(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM41_SHIFT))&ADC_CIMR_CIM41_MASK)
#define ADC_CIMR_CIM9_MASK                       0x200u
#define ADC_CIMR_CIM9_SHIFT                      9u
#define ADC_CIMR_CIM9_WIDTH                      1u
#define ADC_CIMR_CIM9(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM9_SHIFT))&ADC_CIMR_CIM9_MASK)
#define ADC_CIMR_CIM42_MASK                      0x400u
#define ADC_CIMR_CIM42_SHIFT                     10u
#define ADC_CIMR_CIM42_WIDTH                     1u
#define ADC_CIMR_CIM42(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM42_SHIFT))&ADC_CIMR_CIM42_MASK)
#define ADC_CIMR_CIM74_MASK                      0x400u
#define ADC_CIMR_CIM74_SHIFT                     10u
#define ADC_CIMR_CIM74_WIDTH                     1u
#define ADC_CIMR_CIM74(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM74_SHIFT))&ADC_CIMR_CIM74_MASK)
#define ADC_CIMR_CIM10_MASK                      0x400u
#define ADC_CIMR_CIM10_SHIFT                     10u
#define ADC_CIMR_CIM10_WIDTH                     1u
#define ADC_CIMR_CIM10(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM10_SHIFT))&ADC_CIMR_CIM10_MASK)
#define ADC_CIMR_CIM11_MASK                      0x800u
#define ADC_CIMR_CIM11_SHIFT                     11u
#define ADC_CIMR_CIM11_WIDTH                     1u
#define ADC_CIMR_CIM11(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM11_SHIFT))&ADC_CIMR_CIM11_MASK)
#define ADC_CIMR_CIM75_MASK                      0x800u
#define ADC_CIMR_CIM75_SHIFT                     11u
#define ADC_CIMR_CIM75_WIDTH                     1u
#define ADC_CIMR_CIM75(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM75_SHIFT))&ADC_CIMR_CIM75_MASK)
#define ADC_CIMR_CIM43_MASK                      0x800u
#define ADC_CIMR_CIM43_SHIFT                     11u
#define ADC_CIMR_CIM43_WIDTH                     1u
#define ADC_CIMR_CIM43(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM43_SHIFT))&ADC_CIMR_CIM43_MASK)
#define ADC_CIMR_CIM76_MASK                      0x1000u
#define ADC_CIMR_CIM76_SHIFT                     12u
#define ADC_CIMR_CIM76_WIDTH                     1u
#define ADC_CIMR_CIM76(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM76_SHIFT))&ADC_CIMR_CIM76_MASK)
#define ADC_CIMR_CIM12_MASK                      0x1000u
#define ADC_CIMR_CIM12_SHIFT                     12u
#define ADC_CIMR_CIM12_WIDTH                     1u
#define ADC_CIMR_CIM12(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM12_SHIFT))&ADC_CIMR_CIM12_MASK)
#define ADC_CIMR_CIM44_MASK                      0x1000u
#define ADC_CIMR_CIM44_SHIFT                     12u
#define ADC_CIMR_CIM44_WIDTH                     1u
#define ADC_CIMR_CIM44(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM44_SHIFT))&ADC_CIMR_CIM44_MASK)
#define ADC_CIMR_CIM45_MASK                      0x2000u
#define ADC_CIMR_CIM45_SHIFT                     13u
#define ADC_CIMR_CIM45_WIDTH                     1u
#define ADC_CIMR_CIM45(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM45_SHIFT))&ADC_CIMR_CIM45_MASK)
#define ADC_CIMR_CIM77_MASK                      0x2000u
#define ADC_CIMR_CIM77_SHIFT                     13u
#define ADC_CIMR_CIM77_WIDTH                     1u
#define ADC_CIMR_CIM77(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM77_SHIFT))&ADC_CIMR_CIM77_MASK)
#define ADC_CIMR_CIM13_MASK                      0x2000u
#define ADC_CIMR_CIM13_SHIFT                     13u
#define ADC_CIMR_CIM13_WIDTH                     1u
#define ADC_CIMR_CIM13(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM13_SHIFT))&ADC_CIMR_CIM13_MASK)
#define ADC_CIMR_CIM14_MASK                      0x4000u
#define ADC_CIMR_CIM14_SHIFT                     14u
#define ADC_CIMR_CIM14_WIDTH                     1u
#define ADC_CIMR_CIM14(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM14_SHIFT))&ADC_CIMR_CIM14_MASK)
#define ADC_CIMR_CIM46_MASK                      0x4000u
#define ADC_CIMR_CIM46_SHIFT                     14u
#define ADC_CIMR_CIM46_WIDTH                     1u
#define ADC_CIMR_CIM46(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM46_SHIFT))&ADC_CIMR_CIM46_MASK)
#define ADC_CIMR_CIM78_MASK                      0x4000u
#define ADC_CIMR_CIM78_SHIFT                     14u
#define ADC_CIMR_CIM78_WIDTH                     1u
#define ADC_CIMR_CIM78(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM78_SHIFT))&ADC_CIMR_CIM78_MASK)
#define ADC_CIMR_CIM47_MASK                      0x8000u
#define ADC_CIMR_CIM47_SHIFT                     15u
#define ADC_CIMR_CIM47_WIDTH                     1u
#define ADC_CIMR_CIM47(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM47_SHIFT))&ADC_CIMR_CIM47_MASK)
#define ADC_CIMR_CIM15_MASK                      0x8000u
#define ADC_CIMR_CIM15_SHIFT                     15u
#define ADC_CIMR_CIM15_WIDTH                     1u
#define ADC_CIMR_CIM15(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM15_SHIFT))&ADC_CIMR_CIM15_MASK)
#define ADC_CIMR_CIM79_MASK                      0x8000u
#define ADC_CIMR_CIM79_SHIFT                     15u
#define ADC_CIMR_CIM79_WIDTH                     1u
#define ADC_CIMR_CIM79(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM79_SHIFT))&ADC_CIMR_CIM79_MASK)
#define ADC_CIMR_CIM80_MASK                      0x10000u
#define ADC_CIMR_CIM80_SHIFT                     16u
#define ADC_CIMR_CIM80_WIDTH                     1u
#define ADC_CIMR_CIM80(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM80_SHIFT))&ADC_CIMR_CIM80_MASK)
#define ADC_CIMR_CIM48_MASK                      0x10000u
#define ADC_CIMR_CIM48_SHIFT                     16u
#define ADC_CIMR_CIM48_WIDTH                     1u
#define ADC_CIMR_CIM48(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM48_SHIFT))&ADC_CIMR_CIM48_MASK)
#define ADC_CIMR_CIM81_MASK                      0x20000u
#define ADC_CIMR_CIM81_SHIFT                     17u
#define ADC_CIMR_CIM81_WIDTH                     1u
#define ADC_CIMR_CIM81(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM81_SHIFT))&ADC_CIMR_CIM81_MASK)
#define ADC_CIMR_CIM49_MASK                      0x20000u
#define ADC_CIMR_CIM49_SHIFT                     17u
#define ADC_CIMR_CIM49_WIDTH                     1u
#define ADC_CIMR_CIM49(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM49_SHIFT))&ADC_CIMR_CIM49_MASK)
#define ADC_CIMR_CIM50_MASK                      0x40000u
#define ADC_CIMR_CIM50_SHIFT                     18u
#define ADC_CIMR_CIM50_WIDTH                     1u
#define ADC_CIMR_CIM50(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM50_SHIFT))&ADC_CIMR_CIM50_MASK)
#define ADC_CIMR_CIM82_MASK                      0x40000u
#define ADC_CIMR_CIM82_SHIFT                     18u
#define ADC_CIMR_CIM82_WIDTH                     1u
#define ADC_CIMR_CIM82(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM82_SHIFT))&ADC_CIMR_CIM82_MASK)
#define ADC_CIMR_CIM51_MASK                      0x80000u
#define ADC_CIMR_CIM51_SHIFT                     19u
#define ADC_CIMR_CIM51_WIDTH                     1u
#define ADC_CIMR_CIM51(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM51_SHIFT))&ADC_CIMR_CIM51_MASK)
#define ADC_CIMR_CIM83_MASK                      0x80000u
#define ADC_CIMR_CIM83_SHIFT                     19u
#define ADC_CIMR_CIM83_WIDTH                     1u
#define ADC_CIMR_CIM83(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM83_SHIFT))&ADC_CIMR_CIM83_MASK)
#define ADC_CIMR_CIM52_MASK                      0x100000u
#define ADC_CIMR_CIM52_SHIFT                     20u
#define ADC_CIMR_CIM52_WIDTH                     1u
#define ADC_CIMR_CIM52(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM52_SHIFT))&ADC_CIMR_CIM52_MASK)
#define ADC_CIMR_CIM84_MASK                      0x100000u
#define ADC_CIMR_CIM84_SHIFT                     20u
#define ADC_CIMR_CIM84_WIDTH                     1u
#define ADC_CIMR_CIM84(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM84_SHIFT))&ADC_CIMR_CIM84_MASK)
#define ADC_CIMR_CIM53_MASK                      0x200000u
#define ADC_CIMR_CIM53_SHIFT                     21u
#define ADC_CIMR_CIM53_WIDTH                     1u
#define ADC_CIMR_CIM53(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM53_SHIFT))&ADC_CIMR_CIM53_MASK)
#define ADC_CIMR_CIM85_MASK                      0x200000u
#define ADC_CIMR_CIM85_SHIFT                     21u
#define ADC_CIMR_CIM85_WIDTH                     1u
#define ADC_CIMR_CIM85(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM85_SHIFT))&ADC_CIMR_CIM85_MASK)
#define ADC_CIMR_CIM54_MASK                      0x400000u
#define ADC_CIMR_CIM54_SHIFT                     22u
#define ADC_CIMR_CIM54_WIDTH                     1u
#define ADC_CIMR_CIM54(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM54_SHIFT))&ADC_CIMR_CIM54_MASK)
#define ADC_CIMR_CIM86_MASK                      0x400000u
#define ADC_CIMR_CIM86_SHIFT                     22u
#define ADC_CIMR_CIM86_WIDTH                     1u
#define ADC_CIMR_CIM86(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM86_SHIFT))&ADC_CIMR_CIM86_MASK)
#define ADC_CIMR_CIM55_MASK                      0x800000u
#define ADC_CIMR_CIM55_SHIFT                     23u
#define ADC_CIMR_CIM55_WIDTH                     1u
#define ADC_CIMR_CIM55(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM55_SHIFT))&ADC_CIMR_CIM55_MASK)
#define ADC_CIMR_CIM87_MASK                      0x800000u
#define ADC_CIMR_CIM87_SHIFT                     23u
#define ADC_CIMR_CIM87_WIDTH                     1u
#define ADC_CIMR_CIM87(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM87_SHIFT))&ADC_CIMR_CIM87_MASK)
#define ADC_CIMR_CIM88_MASK                      0x1000000u
#define ADC_CIMR_CIM88_SHIFT                     24u
#define ADC_CIMR_CIM88_WIDTH                     1u
#define ADC_CIMR_CIM88(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM88_SHIFT))&ADC_CIMR_CIM88_MASK)
#define ADC_CIMR_CIM56_MASK                      0x1000000u
#define ADC_CIMR_CIM56_SHIFT                     24u
#define ADC_CIMR_CIM56_WIDTH                     1u
#define ADC_CIMR_CIM56(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM56_SHIFT))&ADC_CIMR_CIM56_MASK)
#define ADC_CIMR_CIM89_MASK                      0x2000000u
#define ADC_CIMR_CIM89_SHIFT                     25u
#define ADC_CIMR_CIM89_WIDTH                     1u
#define ADC_CIMR_CIM89(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM89_SHIFT))&ADC_CIMR_CIM89_MASK)
#define ADC_CIMR_CIM57_MASK                      0x2000000u
#define ADC_CIMR_CIM57_SHIFT                     25u
#define ADC_CIMR_CIM57_WIDTH                     1u
#define ADC_CIMR_CIM57(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM57_SHIFT))&ADC_CIMR_CIM57_MASK)
#define ADC_CIMR_CIM90_MASK                      0x4000000u
#define ADC_CIMR_CIM90_SHIFT                     26u
#define ADC_CIMR_CIM90_WIDTH                     1u
#define ADC_CIMR_CIM90(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM90_SHIFT))&ADC_CIMR_CIM90_MASK)
#define ADC_CIMR_CIM58_MASK                      0x4000000u
#define ADC_CIMR_CIM58_SHIFT                     26u
#define ADC_CIMR_CIM58_WIDTH                     1u
#define ADC_CIMR_CIM58(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM58_SHIFT))&ADC_CIMR_CIM58_MASK)
#define ADC_CIMR_CIM59_MASK                      0x8000000u
#define ADC_CIMR_CIM59_SHIFT                     27u
#define ADC_CIMR_CIM59_WIDTH                     1u
#define ADC_CIMR_CIM59(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM59_SHIFT))&ADC_CIMR_CIM59_MASK)
#define ADC_CIMR_CIM91_MASK                      0x8000000u
#define ADC_CIMR_CIM91_SHIFT                     27u
#define ADC_CIMR_CIM91_WIDTH                     1u
#define ADC_CIMR_CIM91(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM91_SHIFT))&ADC_CIMR_CIM91_MASK)
#define ADC_CIMR_CIM60_MASK                      0x10000000u
#define ADC_CIMR_CIM60_SHIFT                     28u
#define ADC_CIMR_CIM60_WIDTH                     1u
#define ADC_CIMR_CIM60(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM60_SHIFT))&ADC_CIMR_CIM60_MASK)
#define ADC_CIMR_CIM92_MASK                      0x10000000u
#define ADC_CIMR_CIM92_SHIFT                     28u
#define ADC_CIMR_CIM92_WIDTH                     1u
#define ADC_CIMR_CIM92(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM92_SHIFT))&ADC_CIMR_CIM92_MASK)
#define ADC_CIMR_CIM61_MASK                      0x20000000u
#define ADC_CIMR_CIM61_SHIFT                     29u
#define ADC_CIMR_CIM61_WIDTH                     1u
#define ADC_CIMR_CIM61(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM61_SHIFT))&ADC_CIMR_CIM61_MASK)
#define ADC_CIMR_CIM93_MASK                      0x20000000u
#define ADC_CIMR_CIM93_SHIFT                     29u
#define ADC_CIMR_CIM93_WIDTH                     1u
#define ADC_CIMR_CIM93(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM93_SHIFT))&ADC_CIMR_CIM93_MASK)
#define ADC_CIMR_CIM94_MASK                      0x40000000u
#define ADC_CIMR_CIM94_SHIFT                     30u
#define ADC_CIMR_CIM94_WIDTH                     1u
#define ADC_CIMR_CIM94(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM94_SHIFT))&ADC_CIMR_CIM94_MASK)
#define ADC_CIMR_CIM62_MASK                      0x40000000u
#define ADC_CIMR_CIM62_SHIFT                     30u
#define ADC_CIMR_CIM62_WIDTH                     1u
#define ADC_CIMR_CIM62(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM62_SHIFT))&ADC_CIMR_CIM62_MASK)
#define ADC_CIMR_CIM95_MASK                      0x80000000u
#define ADC_CIMR_CIM95_SHIFT                     31u
#define ADC_CIMR_CIM95_WIDTH                     1u
#define ADC_CIMR_CIM95(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM95_SHIFT))&ADC_CIMR_CIM95_MASK)
#define ADC_CIMR_CIM63_MASK                      0x80000000u
#define ADC_CIMR_CIM63_SHIFT                     31u
#define ADC_CIMR_CIM63_WIDTH                     1u
#define ADC_CIMR_CIM63(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CIMR_CIM63_SHIFT))&ADC_CIMR_CIM63_MASK)
/* WTISR Bit Fields */
#define ADC_WTISR_WDG0L_MASK                     0x1u
#define ADC_WTISR_WDG0L_SHIFT                    0u
#define ADC_WTISR_WDG0L_WIDTH                    1u
#define ADC_WTISR_WDG0L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG0L_SHIFT))&ADC_WTISR_WDG0L_MASK)
#define ADC_WTISR_WDG0H_MASK                     0x2u
#define ADC_WTISR_WDG0H_SHIFT                    1u
#define ADC_WTISR_WDG0H_WIDTH                    1u
#define ADC_WTISR_WDG0H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG0H_SHIFT))&ADC_WTISR_WDG0H_MASK)
#define ADC_WTISR_WDG1L_MASK                     0x4u
#define ADC_WTISR_WDG1L_SHIFT                    2u
#define ADC_WTISR_WDG1L_WIDTH                    1u
#define ADC_WTISR_WDG1L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG1L_SHIFT))&ADC_WTISR_WDG1L_MASK)
#define ADC_WTISR_WDG1H_MASK                     0x8u
#define ADC_WTISR_WDG1H_SHIFT                    3u
#define ADC_WTISR_WDG1H_WIDTH                    1u
#define ADC_WTISR_WDG1H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG1H_SHIFT))&ADC_WTISR_WDG1H_MASK)
#define ADC_WTISR_WDG2L_MASK                     0x10u
#define ADC_WTISR_WDG2L_SHIFT                    4u
#define ADC_WTISR_WDG2L_WIDTH                    1u
#define ADC_WTISR_WDG2L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG2L_SHIFT))&ADC_WTISR_WDG2L_MASK)
#define ADC_WTISR_WDG2H_MASK                     0x20u
#define ADC_WTISR_WDG2H_SHIFT                    5u
#define ADC_WTISR_WDG2H_WIDTH                    1u
#define ADC_WTISR_WDG2H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG2H_SHIFT))&ADC_WTISR_WDG2H_MASK)
#define ADC_WTISR_WDG3L_MASK                     0x40u
#define ADC_WTISR_WDG3L_SHIFT                    6u
#define ADC_WTISR_WDG3L_WIDTH                    1u
#define ADC_WTISR_WDG3L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG3L_SHIFT))&ADC_WTISR_WDG3L_MASK)
#define ADC_WTISR_WDG3H_MASK                     0x80u
#define ADC_WTISR_WDG3H_SHIFT                    7u
#define ADC_WTISR_WDG3H_WIDTH                    1u
#define ADC_WTISR_WDG3H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG3H_SHIFT))&ADC_WTISR_WDG3H_MASK)
#define ADC_WTISR_WDG4L_MASK                     0x100u
#define ADC_WTISR_WDG4L_SHIFT                    8u
#define ADC_WTISR_WDG4L_WIDTH                    1u
#define ADC_WTISR_WDG4L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG4L_SHIFT))&ADC_WTISR_WDG4L_MASK)
#define ADC_WTISR_WDG4H_MASK                     0x200u
#define ADC_WTISR_WDG4H_SHIFT                    9u
#define ADC_WTISR_WDG4H_WIDTH                    1u
#define ADC_WTISR_WDG4H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG4H_SHIFT))&ADC_WTISR_WDG4H_MASK)
#define ADC_WTISR_WDG5L_MASK                     0x400u
#define ADC_WTISR_WDG5L_SHIFT                    10u
#define ADC_WTISR_WDG5L_WIDTH                    1u
#define ADC_WTISR_WDG5L(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG5L_SHIFT))&ADC_WTISR_WDG5L_MASK)
#define ADC_WTISR_WDG5H_MASK                     0x800u
#define ADC_WTISR_WDG5H_SHIFT                    11u
#define ADC_WTISR_WDG5H_WIDTH                    1u
#define ADC_WTISR_WDG5H(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_WTISR_WDG5H_SHIFT))&ADC_WTISR_WDG5H_MASK)
/* WTIMR Bit Fields */
#define ADC_WTIMR_MSKWDG0L_MASK                  0x1u
#define ADC_WTIMR_MSKWDG0L_SHIFT                 0u
#define ADC_WTIMR_MSKWDG0L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG0L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG0L_SHIFT))&ADC_WTIMR_MSKWDG0L_MASK)
#define ADC_WTIMR_MSKWDG0H_MASK                  0x2u
#define ADC_WTIMR_MSKWDG0H_SHIFT                 1u
#define ADC_WTIMR_MSKWDG0H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG0H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG0H_SHIFT))&ADC_WTIMR_MSKWDG0H_MASK)
#define ADC_WTIMR_MSKWDG1L_MASK                  0x4u
#define ADC_WTIMR_MSKWDG1L_SHIFT                 2u
#define ADC_WTIMR_MSKWDG1L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG1L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG1L_SHIFT))&ADC_WTIMR_MSKWDG1L_MASK)
#define ADC_WTIMR_MSKWDG1H_MASK                  0x8u
#define ADC_WTIMR_MSKWDG1H_SHIFT                 3u
#define ADC_WTIMR_MSKWDG1H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG1H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG1H_SHIFT))&ADC_WTIMR_MSKWDG1H_MASK)
#define ADC_WTIMR_MSKWDG2L_MASK                  0x10u
#define ADC_WTIMR_MSKWDG2L_SHIFT                 4u
#define ADC_WTIMR_MSKWDG2L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG2L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG2L_SHIFT))&ADC_WTIMR_MSKWDG2L_MASK)
#define ADC_WTIMR_MSKWDG2H_MASK                  0x20u
#define ADC_WTIMR_MSKWDG2H_SHIFT                 5u
#define ADC_WTIMR_MSKWDG2H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG2H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG2H_SHIFT))&ADC_WTIMR_MSKWDG2H_MASK)
#define ADC_WTIMR_MSKWDG3L_MASK                  0x40u
#define ADC_WTIMR_MSKWDG3L_SHIFT                 6u
#define ADC_WTIMR_MSKWDG3L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG3L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG3L_SHIFT))&ADC_WTIMR_MSKWDG3L_MASK)
#define ADC_WTIMR_MSKWDG3H_MASK                  0x80u
#define ADC_WTIMR_MSKWDG3H_SHIFT                 7u
#define ADC_WTIMR_MSKWDG3H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG3H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG3H_SHIFT))&ADC_WTIMR_MSKWDG3H_MASK)
#define ADC_WTIMR_MSKWDG4L_MASK                  0x100u
#define ADC_WTIMR_MSKWDG4L_SHIFT                 8u
#define ADC_WTIMR_MSKWDG4L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG4L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG4L_SHIFT))&ADC_WTIMR_MSKWDG4L_MASK)
#define ADC_WTIMR_MSKWDG4H_MASK                  0x200u
#define ADC_WTIMR_MSKWDG4H_SHIFT                 9u
#define ADC_WTIMR_MSKWDG4H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG4H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG4H_SHIFT))&ADC_WTIMR_MSKWDG4H_MASK)
#define ADC_WTIMR_MSKWDG5L_MASK                  0x400u
#define ADC_WTIMR_MSKWDG5L_SHIFT                 10u
#define ADC_WTIMR_MSKWDG5L_WIDTH                 1u
#define ADC_WTIMR_MSKWDG5L(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG5L_SHIFT))&ADC_WTIMR_MSKWDG5L_MASK)
#define ADC_WTIMR_MSKWDG5H_MASK                  0x800u
#define ADC_WTIMR_MSKWDG5H_SHIFT                 11u
#define ADC_WTIMR_MSKWDG5H_WIDTH                 1u
#define ADC_WTIMR_MSKWDG5H(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_WTIMR_MSKWDG5H_SHIFT))&ADC_WTIMR_MSKWDG5H_MASK)
/* DMAE Bit Fields */
#define ADC_DMAE_DMAEN_MASK                      0x1u
#define ADC_DMAE_DMAEN_SHIFT                     0u
#define ADC_DMAE_DMAEN_WIDTH                     1u
#define ADC_DMAE_DMAEN(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAE_DMAEN_SHIFT))&ADC_DMAE_DMAEN_MASK)
#define ADC_DMAE_DCLR_MASK                       0x2u
#define ADC_DMAE_DCLR_SHIFT                      1u
#define ADC_DMAE_DCLR_WIDTH                      1u
#define ADC_DMAE_DCLR(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAE_DCLR_SHIFT))&ADC_DMAE_DCLR_MASK)
/* DMAR Bit Fields */
#define ADC_DMAR_DMA0_MASK                       0x1u
#define ADC_DMAR_DMA0_SHIFT                      0u
#define ADC_DMAR_DMA0_WIDTH                      1u
#define ADC_DMAR_DMA0(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA0_SHIFT))&ADC_DMAR_DMA0_MASK)
#define ADC_DMAR_DMA64_MASK                      0x1u
#define ADC_DMAR_DMA64_SHIFT                     0u
#define ADC_DMAR_DMA64_WIDTH                     1u
#define ADC_DMAR_DMA64(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA64_SHIFT))&ADC_DMAR_DMA64_MASK)
#define ADC_DMAR_DMA32_MASK                      0x1u
#define ADC_DMAR_DMA32_SHIFT                     0u
#define ADC_DMAR_DMA32_WIDTH                     1u
#define ADC_DMAR_DMA32(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA32_SHIFT))&ADC_DMAR_DMA32_MASK)
#define ADC_DMAR_DMA65_MASK                      0x2u
#define ADC_DMAR_DMA65_SHIFT                     1u
#define ADC_DMAR_DMA65_WIDTH                     1u
#define ADC_DMAR_DMA65(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA65_SHIFT))&ADC_DMAR_DMA65_MASK)
#define ADC_DMAR_DMA33_MASK                      0x2u
#define ADC_DMAR_DMA33_SHIFT                     1u
#define ADC_DMAR_DMA33_WIDTH                     1u
#define ADC_DMAR_DMA33(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA33_SHIFT))&ADC_DMAR_DMA33_MASK)
#define ADC_DMAR_DMA1_MASK                       0x2u
#define ADC_DMAR_DMA1_SHIFT                      1u
#define ADC_DMAR_DMA1_WIDTH                      1u
#define ADC_DMAR_DMA1(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA1_SHIFT))&ADC_DMAR_DMA1_MASK)
#define ADC_DMAR_DMA34_MASK                      0x4u
#define ADC_DMAR_DMA34_SHIFT                     2u
#define ADC_DMAR_DMA34_WIDTH                     1u
#define ADC_DMAR_DMA34(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA34_SHIFT))&ADC_DMAR_DMA34_MASK)
#define ADC_DMAR_DMA2_MASK                       0x4u
#define ADC_DMAR_DMA2_SHIFT                      2u
#define ADC_DMAR_DMA2_WIDTH                      1u
#define ADC_DMAR_DMA2(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA2_SHIFT))&ADC_DMAR_DMA2_MASK)
#define ADC_DMAR_DMA66_MASK                      0x4u
#define ADC_DMAR_DMA66_SHIFT                     2u
#define ADC_DMAR_DMA66_WIDTH                     1u
#define ADC_DMAR_DMA66(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA66_SHIFT))&ADC_DMAR_DMA66_MASK)
#define ADC_DMAR_DMA67_MASK                      0x8u
#define ADC_DMAR_DMA67_SHIFT                     3u
#define ADC_DMAR_DMA67_WIDTH                     1u
#define ADC_DMAR_DMA67(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA67_SHIFT))&ADC_DMAR_DMA67_MASK)
#define ADC_DMAR_DMA3_MASK                       0x8u
#define ADC_DMAR_DMA3_SHIFT                      3u
#define ADC_DMAR_DMA3_WIDTH                      1u
#define ADC_DMAR_DMA3(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA3_SHIFT))&ADC_DMAR_DMA3_MASK)
#define ADC_DMAR_DMA35_MASK                      0x8u
#define ADC_DMAR_DMA35_SHIFT                     3u
#define ADC_DMAR_DMA35_WIDTH                     1u
#define ADC_DMAR_DMA35(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA35_SHIFT))&ADC_DMAR_DMA35_MASK)
#define ADC_DMAR_DMA4_MASK                       0x10u
#define ADC_DMAR_DMA4_SHIFT                      4u
#define ADC_DMAR_DMA4_WIDTH                      1u
#define ADC_DMAR_DMA4(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA4_SHIFT))&ADC_DMAR_DMA4_MASK)
#define ADC_DMAR_DMA36_MASK                      0x10u
#define ADC_DMAR_DMA36_SHIFT                     4u
#define ADC_DMAR_DMA36_WIDTH                     1u
#define ADC_DMAR_DMA36(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA36_SHIFT))&ADC_DMAR_DMA36_MASK)
#define ADC_DMAR_DMA68_MASK                      0x10u
#define ADC_DMAR_DMA68_SHIFT                     4u
#define ADC_DMAR_DMA68_WIDTH                     1u
#define ADC_DMAR_DMA68(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA68_SHIFT))&ADC_DMAR_DMA68_MASK)
#define ADC_DMAR_DMA69_MASK                      0x20u
#define ADC_DMAR_DMA69_SHIFT                     5u
#define ADC_DMAR_DMA69_WIDTH                     1u
#define ADC_DMAR_DMA69(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA69_SHIFT))&ADC_DMAR_DMA69_MASK)
#define ADC_DMAR_DMA37_MASK                      0x20u
#define ADC_DMAR_DMA37_SHIFT                     5u
#define ADC_DMAR_DMA37_WIDTH                     1u
#define ADC_DMAR_DMA37(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA37_SHIFT))&ADC_DMAR_DMA37_MASK)
#define ADC_DMAR_DMA5_MASK                       0x20u
#define ADC_DMAR_DMA5_SHIFT                      5u
#define ADC_DMAR_DMA5_WIDTH                      1u
#define ADC_DMAR_DMA5(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA5_SHIFT))&ADC_DMAR_DMA5_MASK)
#define ADC_DMAR_DMA70_MASK                      0x40u
#define ADC_DMAR_DMA70_SHIFT                     6u
#define ADC_DMAR_DMA70_WIDTH                     1u
#define ADC_DMAR_DMA70(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA70_SHIFT))&ADC_DMAR_DMA70_MASK)
#define ADC_DMAR_DMA38_MASK                      0x40u
#define ADC_DMAR_DMA38_SHIFT                     6u
#define ADC_DMAR_DMA38_WIDTH                     1u
#define ADC_DMAR_DMA38(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA38_SHIFT))&ADC_DMAR_DMA38_MASK)
#define ADC_DMAR_DMA6_MASK                       0x40u
#define ADC_DMAR_DMA6_SHIFT                      6u
#define ADC_DMAR_DMA6_WIDTH                      1u
#define ADC_DMAR_DMA6(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA6_SHIFT))&ADC_DMAR_DMA6_MASK)
#define ADC_DMAR_DMA7_MASK                       0x80u
#define ADC_DMAR_DMA7_SHIFT                      7u
#define ADC_DMAR_DMA7_WIDTH                      1u
#define ADC_DMAR_DMA7(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA7_SHIFT))&ADC_DMAR_DMA7_MASK)
#define ADC_DMAR_DMA39_MASK                      0x80u
#define ADC_DMAR_DMA39_SHIFT                     7u
#define ADC_DMAR_DMA39_WIDTH                     1u
#define ADC_DMAR_DMA39(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA39_SHIFT))&ADC_DMAR_DMA39_MASK)
#define ADC_DMAR_DMA71_MASK                      0x80u
#define ADC_DMAR_DMA71_SHIFT                     7u
#define ADC_DMAR_DMA71_WIDTH                     1u
#define ADC_DMAR_DMA71(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA71_SHIFT))&ADC_DMAR_DMA71_MASK)
#define ADC_DMAR_DMA8_MASK                       0x100u
#define ADC_DMAR_DMA8_SHIFT                      8u
#define ADC_DMAR_DMA8_WIDTH                      1u
#define ADC_DMAR_DMA8(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA8_SHIFT))&ADC_DMAR_DMA8_MASK)
#define ADC_DMAR_DMA40_MASK                      0x100u
#define ADC_DMAR_DMA40_SHIFT                     8u
#define ADC_DMAR_DMA40_WIDTH                     1u
#define ADC_DMAR_DMA40(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA40_SHIFT))&ADC_DMAR_DMA40_MASK)
#define ADC_DMAR_DMA72_MASK                      0x100u
#define ADC_DMAR_DMA72_SHIFT                     8u
#define ADC_DMAR_DMA72_WIDTH                     1u
#define ADC_DMAR_DMA72(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA72_SHIFT))&ADC_DMAR_DMA72_MASK)
#define ADC_DMAR_DMA73_MASK                      0x200u
#define ADC_DMAR_DMA73_SHIFT                     9u
#define ADC_DMAR_DMA73_WIDTH                     1u
#define ADC_DMAR_DMA73(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA73_SHIFT))&ADC_DMAR_DMA73_MASK)
#define ADC_DMAR_DMA41_MASK                      0x200u
#define ADC_DMAR_DMA41_SHIFT                     9u
#define ADC_DMAR_DMA41_WIDTH                     1u
#define ADC_DMAR_DMA41(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA41_SHIFT))&ADC_DMAR_DMA41_MASK)
#define ADC_DMAR_DMA9_MASK                       0x200u
#define ADC_DMAR_DMA9_SHIFT                      9u
#define ADC_DMAR_DMA9_WIDTH                      1u
#define ADC_DMAR_DMA9(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA9_SHIFT))&ADC_DMAR_DMA9_MASK)
#define ADC_DMAR_DMA42_MASK                      0x400u
#define ADC_DMAR_DMA42_SHIFT                     10u
#define ADC_DMAR_DMA42_WIDTH                     1u
#define ADC_DMAR_DMA42(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA42_SHIFT))&ADC_DMAR_DMA42_MASK)
#define ADC_DMAR_DMA74_MASK                      0x400u
#define ADC_DMAR_DMA74_SHIFT                     10u
#define ADC_DMAR_DMA74_WIDTH                     1u
#define ADC_DMAR_DMA74(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA74_SHIFT))&ADC_DMAR_DMA74_MASK)
#define ADC_DMAR_DMA10_MASK                      0x400u
#define ADC_DMAR_DMA10_SHIFT                     10u
#define ADC_DMAR_DMA10_WIDTH                     1u
#define ADC_DMAR_DMA10(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA10_SHIFT))&ADC_DMAR_DMA10_MASK)
#define ADC_DMAR_DMA11_MASK                      0x800u
#define ADC_DMAR_DMA11_SHIFT                     11u
#define ADC_DMAR_DMA11_WIDTH                     1u
#define ADC_DMAR_DMA11(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA11_SHIFT))&ADC_DMAR_DMA11_MASK)
#define ADC_DMAR_DMA75_MASK                      0x800u
#define ADC_DMAR_DMA75_SHIFT                     11u
#define ADC_DMAR_DMA75_WIDTH                     1u
#define ADC_DMAR_DMA75(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA75_SHIFT))&ADC_DMAR_DMA75_MASK)
#define ADC_DMAR_DMA43_MASK                      0x800u
#define ADC_DMAR_DMA43_SHIFT                     11u
#define ADC_DMAR_DMA43_WIDTH                     1u
#define ADC_DMAR_DMA43(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA43_SHIFT))&ADC_DMAR_DMA43_MASK)
#define ADC_DMAR_DMA76_MASK                      0x1000u
#define ADC_DMAR_DMA76_SHIFT                     12u
#define ADC_DMAR_DMA76_WIDTH                     1u
#define ADC_DMAR_DMA76(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA76_SHIFT))&ADC_DMAR_DMA76_MASK)
#define ADC_DMAR_DMA12_MASK                      0x1000u
#define ADC_DMAR_DMA12_SHIFT                     12u
#define ADC_DMAR_DMA12_WIDTH                     1u
#define ADC_DMAR_DMA12(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA12_SHIFT))&ADC_DMAR_DMA12_MASK)
#define ADC_DMAR_DMA44_MASK                      0x1000u
#define ADC_DMAR_DMA44_SHIFT                     12u
#define ADC_DMAR_DMA44_WIDTH                     1u
#define ADC_DMAR_DMA44(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA44_SHIFT))&ADC_DMAR_DMA44_MASK)
#define ADC_DMAR_DMA45_MASK                      0x2000u
#define ADC_DMAR_DMA45_SHIFT                     13u
#define ADC_DMAR_DMA45_WIDTH                     1u
#define ADC_DMAR_DMA45(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA45_SHIFT))&ADC_DMAR_DMA45_MASK)
#define ADC_DMAR_DMA77_MASK                      0x2000u
#define ADC_DMAR_DMA77_SHIFT                     13u
#define ADC_DMAR_DMA77_WIDTH                     1u
#define ADC_DMAR_DMA77(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA77_SHIFT))&ADC_DMAR_DMA77_MASK)
#define ADC_DMAR_DMA13_MASK                      0x2000u
#define ADC_DMAR_DMA13_SHIFT                     13u
#define ADC_DMAR_DMA13_WIDTH                     1u
#define ADC_DMAR_DMA13(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA13_SHIFT))&ADC_DMAR_DMA13_MASK)
#define ADC_DMAR_DMA14_MASK                      0x4000u
#define ADC_DMAR_DMA14_SHIFT                     14u
#define ADC_DMAR_DMA14_WIDTH                     1u
#define ADC_DMAR_DMA14(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA14_SHIFT))&ADC_DMAR_DMA14_MASK)
#define ADC_DMAR_DMA46_MASK                      0x4000u
#define ADC_DMAR_DMA46_SHIFT                     14u
#define ADC_DMAR_DMA46_WIDTH                     1u
#define ADC_DMAR_DMA46(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA46_SHIFT))&ADC_DMAR_DMA46_MASK)
#define ADC_DMAR_DMA78_MASK                      0x4000u
#define ADC_DMAR_DMA78_SHIFT                     14u
#define ADC_DMAR_DMA78_WIDTH                     1u
#define ADC_DMAR_DMA78(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA78_SHIFT))&ADC_DMAR_DMA78_MASK)
#define ADC_DMAR_DMA47_MASK                      0x8000u
#define ADC_DMAR_DMA47_SHIFT                     15u
#define ADC_DMAR_DMA47_WIDTH                     1u
#define ADC_DMAR_DMA47(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA47_SHIFT))&ADC_DMAR_DMA47_MASK)
#define ADC_DMAR_DMA15_MASK                      0x8000u
#define ADC_DMAR_DMA15_SHIFT                     15u
#define ADC_DMAR_DMA15_WIDTH                     1u
#define ADC_DMAR_DMA15(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA15_SHIFT))&ADC_DMAR_DMA15_MASK)
#define ADC_DMAR_DMA79_MASK                      0x8000u
#define ADC_DMAR_DMA79_SHIFT                     15u
#define ADC_DMAR_DMA79_WIDTH                     1u
#define ADC_DMAR_DMA79(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA79_SHIFT))&ADC_DMAR_DMA79_MASK)
#define ADC_DMAR_DMA80_MASK                      0x10000u
#define ADC_DMAR_DMA80_SHIFT                     16u
#define ADC_DMAR_DMA80_WIDTH                     1u
#define ADC_DMAR_DMA80(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA80_SHIFT))&ADC_DMAR_DMA80_MASK)
#define ADC_DMAR_DMA48_MASK                      0x10000u
#define ADC_DMAR_DMA48_SHIFT                     16u
#define ADC_DMAR_DMA48_WIDTH                     1u
#define ADC_DMAR_DMA48(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA48_SHIFT))&ADC_DMAR_DMA48_MASK)
#define ADC_DMAR_DMA81_MASK                      0x20000u
#define ADC_DMAR_DMA81_SHIFT                     17u
#define ADC_DMAR_DMA81_WIDTH                     1u
#define ADC_DMAR_DMA81(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA81_SHIFT))&ADC_DMAR_DMA81_MASK)
#define ADC_DMAR_DMA49_MASK                      0x20000u
#define ADC_DMAR_DMA49_SHIFT                     17u
#define ADC_DMAR_DMA49_WIDTH                     1u
#define ADC_DMAR_DMA49(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA49_SHIFT))&ADC_DMAR_DMA49_MASK)
#define ADC_DMAR_DMA50_MASK                      0x40000u
#define ADC_DMAR_DMA50_SHIFT                     18u
#define ADC_DMAR_DMA50_WIDTH                     1u
#define ADC_DMAR_DMA50(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA50_SHIFT))&ADC_DMAR_DMA50_MASK)
#define ADC_DMAR_DMA82_MASK                      0x40000u
#define ADC_DMAR_DMA82_SHIFT                     18u
#define ADC_DMAR_DMA82_WIDTH                     1u
#define ADC_DMAR_DMA82(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA82_SHIFT))&ADC_DMAR_DMA82_MASK)
#define ADC_DMAR_DMA51_MASK                      0x80000u
#define ADC_DMAR_DMA51_SHIFT                     19u
#define ADC_DMAR_DMA51_WIDTH                     1u
#define ADC_DMAR_DMA51(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA51_SHIFT))&ADC_DMAR_DMA51_MASK)
#define ADC_DMAR_DMA83_MASK                      0x80000u
#define ADC_DMAR_DMA83_SHIFT                     19u
#define ADC_DMAR_DMA83_WIDTH                     1u
#define ADC_DMAR_DMA83(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA83_SHIFT))&ADC_DMAR_DMA83_MASK)
#define ADC_DMAR_DMA52_MASK                      0x100000u
#define ADC_DMAR_DMA52_SHIFT                     20u
#define ADC_DMAR_DMA52_WIDTH                     1u
#define ADC_DMAR_DMA52(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA52_SHIFT))&ADC_DMAR_DMA52_MASK)
#define ADC_DMAR_DMA84_MASK                      0x100000u
#define ADC_DMAR_DMA84_SHIFT                     20u
#define ADC_DMAR_DMA84_WIDTH                     1u
#define ADC_DMAR_DMA84(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA84_SHIFT))&ADC_DMAR_DMA84_MASK)
#define ADC_DMAR_DMA53_MASK                      0x200000u
#define ADC_DMAR_DMA53_SHIFT                     21u
#define ADC_DMAR_DMA53_WIDTH                     1u
#define ADC_DMAR_DMA53(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA53_SHIFT))&ADC_DMAR_DMA53_MASK)
#define ADC_DMAR_DMA85_MASK                      0x200000u
#define ADC_DMAR_DMA85_SHIFT                     21u
#define ADC_DMAR_DMA85_WIDTH                     1u
#define ADC_DMAR_DMA85(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA85_SHIFT))&ADC_DMAR_DMA85_MASK)
#define ADC_DMAR_DMA54_MASK                      0x400000u
#define ADC_DMAR_DMA54_SHIFT                     22u
#define ADC_DMAR_DMA54_WIDTH                     1u
#define ADC_DMAR_DMA54(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA54_SHIFT))&ADC_DMAR_DMA54_MASK)
#define ADC_DMAR_DMA86_MASK                      0x400000u
#define ADC_DMAR_DMA86_SHIFT                     22u
#define ADC_DMAR_DMA86_WIDTH                     1u
#define ADC_DMAR_DMA86(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA86_SHIFT))&ADC_DMAR_DMA86_MASK)
#define ADC_DMAR_DMA55_MASK                      0x800000u
#define ADC_DMAR_DMA55_SHIFT                     23u
#define ADC_DMAR_DMA55_WIDTH                     1u
#define ADC_DMAR_DMA55(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA55_SHIFT))&ADC_DMAR_DMA55_MASK)
#define ADC_DMAR_DMA87_MASK                      0x800000u
#define ADC_DMAR_DMA87_SHIFT                     23u
#define ADC_DMAR_DMA87_WIDTH                     1u
#define ADC_DMAR_DMA87(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA87_SHIFT))&ADC_DMAR_DMA87_MASK)
#define ADC_DMAR_DMA88_MASK                      0x1000000u
#define ADC_DMAR_DMA88_SHIFT                     24u
#define ADC_DMAR_DMA88_WIDTH                     1u
#define ADC_DMAR_DMA88(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA88_SHIFT))&ADC_DMAR_DMA88_MASK)
#define ADC_DMAR_DMA56_MASK                      0x1000000u
#define ADC_DMAR_DMA56_SHIFT                     24u
#define ADC_DMAR_DMA56_WIDTH                     1u
#define ADC_DMAR_DMA56(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA56_SHIFT))&ADC_DMAR_DMA56_MASK)
#define ADC_DMAR_DMA89_MASK                      0x2000000u
#define ADC_DMAR_DMA89_SHIFT                     25u
#define ADC_DMAR_DMA89_WIDTH                     1u
#define ADC_DMAR_DMA89(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA89_SHIFT))&ADC_DMAR_DMA89_MASK)
#define ADC_DMAR_DMA57_MASK                      0x2000000u
#define ADC_DMAR_DMA57_SHIFT                     25u
#define ADC_DMAR_DMA57_WIDTH                     1u
#define ADC_DMAR_DMA57(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA57_SHIFT))&ADC_DMAR_DMA57_MASK)
#define ADC_DMAR_DMA90_MASK                      0x4000000u
#define ADC_DMAR_DMA90_SHIFT                     26u
#define ADC_DMAR_DMA90_WIDTH                     1u
#define ADC_DMAR_DMA90(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA90_SHIFT))&ADC_DMAR_DMA90_MASK)
#define ADC_DMAR_DMA58_MASK                      0x4000000u
#define ADC_DMAR_DMA58_SHIFT                     26u
#define ADC_DMAR_DMA58_WIDTH                     1u
#define ADC_DMAR_DMA58(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA58_SHIFT))&ADC_DMAR_DMA58_MASK)
#define ADC_DMAR_DMA59_MASK                      0x8000000u
#define ADC_DMAR_DMA59_SHIFT                     27u
#define ADC_DMAR_DMA59_WIDTH                     1u
#define ADC_DMAR_DMA59(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA59_SHIFT))&ADC_DMAR_DMA59_MASK)
#define ADC_DMAR_DMA91_MASK                      0x8000000u
#define ADC_DMAR_DMA91_SHIFT                     27u
#define ADC_DMAR_DMA91_WIDTH                     1u
#define ADC_DMAR_DMA91(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA91_SHIFT))&ADC_DMAR_DMA91_MASK)
#define ADC_DMAR_DMA60_MASK                      0x10000000u
#define ADC_DMAR_DMA60_SHIFT                     28u
#define ADC_DMAR_DMA60_WIDTH                     1u
#define ADC_DMAR_DMA60(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA60_SHIFT))&ADC_DMAR_DMA60_MASK)
#define ADC_DMAR_DMA92_MASK                      0x10000000u
#define ADC_DMAR_DMA92_SHIFT                     28u
#define ADC_DMAR_DMA92_WIDTH                     1u
#define ADC_DMAR_DMA92(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA92_SHIFT))&ADC_DMAR_DMA92_MASK)
#define ADC_DMAR_DMA61_MASK                      0x20000000u
#define ADC_DMAR_DMA61_SHIFT                     29u
#define ADC_DMAR_DMA61_WIDTH                     1u
#define ADC_DMAR_DMA61(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA61_SHIFT))&ADC_DMAR_DMA61_MASK)
#define ADC_DMAR_DMA93_MASK                      0x20000000u
#define ADC_DMAR_DMA93_SHIFT                     29u
#define ADC_DMAR_DMA93_WIDTH                     1u
#define ADC_DMAR_DMA93(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA93_SHIFT))&ADC_DMAR_DMA93_MASK)
#define ADC_DMAR_DMA94_MASK                      0x40000000u
#define ADC_DMAR_DMA94_SHIFT                     30u
#define ADC_DMAR_DMA94_WIDTH                     1u
#define ADC_DMAR_DMA94(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA94_SHIFT))&ADC_DMAR_DMA94_MASK)
#define ADC_DMAR_DMA62_MASK                      0x40000000u
#define ADC_DMAR_DMA62_SHIFT                     30u
#define ADC_DMAR_DMA62_WIDTH                     1u
#define ADC_DMAR_DMA62(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA62_SHIFT))&ADC_DMAR_DMA62_MASK)
#define ADC_DMAR_DMA95_MASK                      0x80000000u
#define ADC_DMAR_DMA95_SHIFT                     31u
#define ADC_DMAR_DMA95_WIDTH                     1u
#define ADC_DMAR_DMA95(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA95_SHIFT))&ADC_DMAR_DMA95_MASK)
#define ADC_DMAR_DMA63_MASK                      0x80000000u
#define ADC_DMAR_DMA63_SHIFT                     31u
#define ADC_DMAR_DMA63_WIDTH                     1u
#define ADC_DMAR_DMA63(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_DMAR_DMA63_SHIFT))&ADC_DMAR_DMA63_MASK)
/* THRHLR0 Bit Fields */
#define ADC_THRHLR0_THRL_MASK                    0xFFFu
#define ADC_THRHLR0_THRL_SHIFT                   0u
#define ADC_THRHLR0_THRL_WIDTH                   12u
#define ADC_THRHLR0_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR0_THRL_SHIFT))&ADC_THRHLR0_THRL_MASK)
#define ADC_THRHLR0_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR0_THRH_SHIFT                   16u
#define ADC_THRHLR0_THRH_WIDTH                   12u
#define ADC_THRHLR0_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR0_THRH_SHIFT))&ADC_THRHLR0_THRH_MASK)
/* THRHLR1 Bit Fields */
#define ADC_THRHLR1_THRL_MASK                    0xFFFu
#define ADC_THRHLR1_THRL_SHIFT                   0u
#define ADC_THRHLR1_THRL_WIDTH                   12u
#define ADC_THRHLR1_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR1_THRL_SHIFT))&ADC_THRHLR1_THRL_MASK)
#define ADC_THRHLR1_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR1_THRH_SHIFT                   16u
#define ADC_THRHLR1_THRH_WIDTH                   12u
#define ADC_THRHLR1_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR1_THRH_SHIFT))&ADC_THRHLR1_THRH_MASK)
/* THRHLR2 Bit Fields */
#define ADC_THRHLR2_THRL_MASK                    0xFFFu
#define ADC_THRHLR2_THRL_SHIFT                   0u
#define ADC_THRHLR2_THRL_WIDTH                   12u
#define ADC_THRHLR2_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR2_THRL_SHIFT))&ADC_THRHLR2_THRL_MASK)
#define ADC_THRHLR2_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR2_THRH_SHIFT                   16u
#define ADC_THRHLR2_THRH_WIDTH                   12u
#define ADC_THRHLR2_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR2_THRH_SHIFT))&ADC_THRHLR2_THRH_MASK)
/* THRHLR3 Bit Fields */
#define ADC_THRHLR3_THRL_MASK                    0xFFFu
#define ADC_THRHLR3_THRL_SHIFT                   0u
#define ADC_THRHLR3_THRL_WIDTH                   12u
#define ADC_THRHLR3_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR3_THRL_SHIFT))&ADC_THRHLR3_THRL_MASK)
#define ADC_THRHLR3_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR3_THRH_SHIFT                   16u
#define ADC_THRHLR3_THRH_WIDTH                   12u
#define ADC_THRHLR3_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR3_THRH_SHIFT))&ADC_THRHLR3_THRH_MASK)
/* PSCR Bit Fields */
#define ADC_PSCR_PRECONV_MASK                    0x1u
#define ADC_PSCR_PRECONV_SHIFT                   0u
#define ADC_PSCR_PRECONV_WIDTH                   1u
#define ADC_PSCR_PRECONV(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_PSCR_PRECONV_SHIFT))&ADC_PSCR_PRECONV_MASK)
#define ADC_PSCR_PREVAL0_MASK                    0x6u
#define ADC_PSCR_PREVAL0_SHIFT                   1u
#define ADC_PSCR_PREVAL0_WIDTH                   2u
#define ADC_PSCR_PREVAL0(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_PSCR_PREVAL0_SHIFT))&ADC_PSCR_PREVAL0_MASK)
#define ADC_PSCR_PREVAL1_MASK                    0x18u
#define ADC_PSCR_PREVAL1_SHIFT                   3u
#define ADC_PSCR_PREVAL1_WIDTH                   2u
#define ADC_PSCR_PREVAL1(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_PSCR_PREVAL1_SHIFT))&ADC_PSCR_PREVAL1_MASK)
#define ADC_PSCR_PREVAL2_MASK                    0x60u
#define ADC_PSCR_PREVAL2_SHIFT                   5u
#define ADC_PSCR_PREVAL2_WIDTH                   2u
#define ADC_PSCR_PREVAL2(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_PSCR_PREVAL2_SHIFT))&ADC_PSCR_PREVAL2_MASK)
/* PSR Bit Fields */
#define ADC_PSR_PRES0_MASK                       0x1u
#define ADC_PSR_PRES0_SHIFT                      0u
#define ADC_PSR_PRES0_WIDTH                      1u
#define ADC_PSR_PRES0(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES0_SHIFT))&ADC_PSR_PRES0_MASK)
#define ADC_PSR_PRES64_MASK                      0x1u
#define ADC_PSR_PRES64_SHIFT                     0u
#define ADC_PSR_PRES64_WIDTH                     1u
#define ADC_PSR_PRES64(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES64_SHIFT))&ADC_PSR_PRES64_MASK)
#define ADC_PSR_PRES32_MASK                      0x1u
#define ADC_PSR_PRES32_SHIFT                     0u
#define ADC_PSR_PRES32_WIDTH                     1u
#define ADC_PSR_PRES32(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES32_SHIFT))&ADC_PSR_PRES32_MASK)
#define ADC_PSR_PRES65_MASK                      0x2u
#define ADC_PSR_PRES65_SHIFT                     1u
#define ADC_PSR_PRES65_WIDTH                     1u
#define ADC_PSR_PRES65(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES65_SHIFT))&ADC_PSR_PRES65_MASK)
#define ADC_PSR_PRES33_MASK                      0x2u
#define ADC_PSR_PRES33_SHIFT                     1u
#define ADC_PSR_PRES33_WIDTH                     1u
#define ADC_PSR_PRES33(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES33_SHIFT))&ADC_PSR_PRES33_MASK)
#define ADC_PSR_PRES1_MASK                       0x2u
#define ADC_PSR_PRES1_SHIFT                      1u
#define ADC_PSR_PRES1_WIDTH                      1u
#define ADC_PSR_PRES1(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES1_SHIFT))&ADC_PSR_PRES1_MASK)
#define ADC_PSR_PRES34_MASK                      0x4u
#define ADC_PSR_PRES34_SHIFT                     2u
#define ADC_PSR_PRES34_WIDTH                     1u
#define ADC_PSR_PRES34(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES34_SHIFT))&ADC_PSR_PRES34_MASK)
#define ADC_PSR_PRES2_MASK                       0x4u
#define ADC_PSR_PRES2_SHIFT                      2u
#define ADC_PSR_PRES2_WIDTH                      1u
#define ADC_PSR_PRES2(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES2_SHIFT))&ADC_PSR_PRES2_MASK)
#define ADC_PSR_PRES66_MASK                      0x4u
#define ADC_PSR_PRES66_SHIFT                     2u
#define ADC_PSR_PRES66_WIDTH                     1u
#define ADC_PSR_PRES66(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES66_SHIFT))&ADC_PSR_PRES66_MASK)
#define ADC_PSR_PRES67_MASK                      0x8u
#define ADC_PSR_PRES67_SHIFT                     3u
#define ADC_PSR_PRES67_WIDTH                     1u
#define ADC_PSR_PRES67(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES67_SHIFT))&ADC_PSR_PRES67_MASK)
#define ADC_PSR_PRES3_MASK                       0x8u
#define ADC_PSR_PRES3_SHIFT                      3u
#define ADC_PSR_PRES3_WIDTH                      1u
#define ADC_PSR_PRES3(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES3_SHIFT))&ADC_PSR_PRES3_MASK)
#define ADC_PSR_PRES35_MASK                      0x8u
#define ADC_PSR_PRES35_SHIFT                     3u
#define ADC_PSR_PRES35_WIDTH                     1u
#define ADC_PSR_PRES35(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES35_SHIFT))&ADC_PSR_PRES35_MASK)
#define ADC_PSR_PRES4_MASK                       0x10u
#define ADC_PSR_PRES4_SHIFT                      4u
#define ADC_PSR_PRES4_WIDTH                      1u
#define ADC_PSR_PRES4(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES4_SHIFT))&ADC_PSR_PRES4_MASK)
#define ADC_PSR_PRES36_MASK                      0x10u
#define ADC_PSR_PRES36_SHIFT                     4u
#define ADC_PSR_PRES36_WIDTH                     1u
#define ADC_PSR_PRES36(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES36_SHIFT))&ADC_PSR_PRES36_MASK)
#define ADC_PSR_PRES68_MASK                      0x10u
#define ADC_PSR_PRES68_SHIFT                     4u
#define ADC_PSR_PRES68_WIDTH                     1u
#define ADC_PSR_PRES68(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES68_SHIFT))&ADC_PSR_PRES68_MASK)
#define ADC_PSR_PRES69_MASK                      0x20u
#define ADC_PSR_PRES69_SHIFT                     5u
#define ADC_PSR_PRES69_WIDTH                     1u
#define ADC_PSR_PRES69(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES69_SHIFT))&ADC_PSR_PRES69_MASK)
#define ADC_PSR_PRES37_MASK                      0x20u
#define ADC_PSR_PRES37_SHIFT                     5u
#define ADC_PSR_PRES37_WIDTH                     1u
#define ADC_PSR_PRES37(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES37_SHIFT))&ADC_PSR_PRES37_MASK)
#define ADC_PSR_PRES5_MASK                       0x20u
#define ADC_PSR_PRES5_SHIFT                      5u
#define ADC_PSR_PRES5_WIDTH                      1u
#define ADC_PSR_PRES5(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES5_SHIFT))&ADC_PSR_PRES5_MASK)
#define ADC_PSR_PRES70_MASK                      0x40u
#define ADC_PSR_PRES70_SHIFT                     6u
#define ADC_PSR_PRES70_WIDTH                     1u
#define ADC_PSR_PRES70(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES70_SHIFT))&ADC_PSR_PRES70_MASK)
#define ADC_PSR_PRES38_MASK                      0x40u
#define ADC_PSR_PRES38_SHIFT                     6u
#define ADC_PSR_PRES38_WIDTH                     1u
#define ADC_PSR_PRES38(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES38_SHIFT))&ADC_PSR_PRES38_MASK)
#define ADC_PSR_PRES6_MASK                       0x40u
#define ADC_PSR_PRES6_SHIFT                      6u
#define ADC_PSR_PRES6_WIDTH                      1u
#define ADC_PSR_PRES6(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES6_SHIFT))&ADC_PSR_PRES6_MASK)
#define ADC_PSR_PRES7_MASK                       0x80u
#define ADC_PSR_PRES7_SHIFT                      7u
#define ADC_PSR_PRES7_WIDTH                      1u
#define ADC_PSR_PRES7(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES7_SHIFT))&ADC_PSR_PRES7_MASK)
#define ADC_PSR_PRES39_MASK                      0x80u
#define ADC_PSR_PRES39_SHIFT                     7u
#define ADC_PSR_PRES39_WIDTH                     1u
#define ADC_PSR_PRES39(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES39_SHIFT))&ADC_PSR_PRES39_MASK)
#define ADC_PSR_PRES71_MASK                      0x80u
#define ADC_PSR_PRES71_SHIFT                     7u
#define ADC_PSR_PRES71_WIDTH                     1u
#define ADC_PSR_PRES71(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES71_SHIFT))&ADC_PSR_PRES71_MASK)
#define ADC_PSR_PRES8_MASK                       0x100u
#define ADC_PSR_PRES8_SHIFT                      8u
#define ADC_PSR_PRES8_WIDTH                      1u
#define ADC_PSR_PRES8(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES8_SHIFT))&ADC_PSR_PRES8_MASK)
#define ADC_PSR_PRES40_MASK                      0x100u
#define ADC_PSR_PRES40_SHIFT                     8u
#define ADC_PSR_PRES40_WIDTH                     1u
#define ADC_PSR_PRES40(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES40_SHIFT))&ADC_PSR_PRES40_MASK)
#define ADC_PSR_PRES72_MASK                      0x100u
#define ADC_PSR_PRES72_SHIFT                     8u
#define ADC_PSR_PRES72_WIDTH                     1u
#define ADC_PSR_PRES72(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES72_SHIFT))&ADC_PSR_PRES72_MASK)
#define ADC_PSR_PRES73_MASK                      0x200u
#define ADC_PSR_PRES73_SHIFT                     9u
#define ADC_PSR_PRES73_WIDTH                     1u
#define ADC_PSR_PRES73(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES73_SHIFT))&ADC_PSR_PRES73_MASK)
#define ADC_PSR_PRES41_MASK                      0x200u
#define ADC_PSR_PRES41_SHIFT                     9u
#define ADC_PSR_PRES41_WIDTH                     1u
#define ADC_PSR_PRES41(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES41_SHIFT))&ADC_PSR_PRES41_MASK)
#define ADC_PSR_PRES9_MASK                       0x200u
#define ADC_PSR_PRES9_SHIFT                      9u
#define ADC_PSR_PRES9_WIDTH                      1u
#define ADC_PSR_PRES9(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES9_SHIFT))&ADC_PSR_PRES9_MASK)
#define ADC_PSR_PRES42_MASK                      0x400u
#define ADC_PSR_PRES42_SHIFT                     10u
#define ADC_PSR_PRES42_WIDTH                     1u
#define ADC_PSR_PRES42(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES42_SHIFT))&ADC_PSR_PRES42_MASK)
#define ADC_PSR_PRES74_MASK                      0x400u
#define ADC_PSR_PRES74_SHIFT                     10u
#define ADC_PSR_PRES74_WIDTH                     1u
#define ADC_PSR_PRES74(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES74_SHIFT))&ADC_PSR_PRES74_MASK)
#define ADC_PSR_PRES10_MASK                      0x400u
#define ADC_PSR_PRES10_SHIFT                     10u
#define ADC_PSR_PRES10_WIDTH                     1u
#define ADC_PSR_PRES10(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES10_SHIFT))&ADC_PSR_PRES10_MASK)
#define ADC_PSR_PRES11_MASK                      0x800u
#define ADC_PSR_PRES11_SHIFT                     11u
#define ADC_PSR_PRES11_WIDTH                     1u
#define ADC_PSR_PRES11(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES11_SHIFT))&ADC_PSR_PRES11_MASK)
#define ADC_PSR_PRES75_MASK                      0x800u
#define ADC_PSR_PRES75_SHIFT                     11u
#define ADC_PSR_PRES75_WIDTH                     1u
#define ADC_PSR_PRES75(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES75_SHIFT))&ADC_PSR_PRES75_MASK)
#define ADC_PSR_PRES43_MASK                      0x800u
#define ADC_PSR_PRES43_SHIFT                     11u
#define ADC_PSR_PRES43_WIDTH                     1u
#define ADC_PSR_PRES43(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES43_SHIFT))&ADC_PSR_PRES43_MASK)
#define ADC_PSR_PRES76_MASK                      0x1000u
#define ADC_PSR_PRES76_SHIFT                     12u
#define ADC_PSR_PRES76_WIDTH                     1u
#define ADC_PSR_PRES76(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES76_SHIFT))&ADC_PSR_PRES76_MASK)
#define ADC_PSR_PRES12_MASK                      0x1000u
#define ADC_PSR_PRES12_SHIFT                     12u
#define ADC_PSR_PRES12_WIDTH                     1u
#define ADC_PSR_PRES12(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES12_SHIFT))&ADC_PSR_PRES12_MASK)
#define ADC_PSR_PRES44_MASK                      0x1000u
#define ADC_PSR_PRES44_SHIFT                     12u
#define ADC_PSR_PRES44_WIDTH                     1u
#define ADC_PSR_PRES44(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES44_SHIFT))&ADC_PSR_PRES44_MASK)
#define ADC_PSR_PRES45_MASK                      0x2000u
#define ADC_PSR_PRES45_SHIFT                     13u
#define ADC_PSR_PRES45_WIDTH                     1u
#define ADC_PSR_PRES45(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES45_SHIFT))&ADC_PSR_PRES45_MASK)
#define ADC_PSR_PRES77_MASK                      0x2000u
#define ADC_PSR_PRES77_SHIFT                     13u
#define ADC_PSR_PRES77_WIDTH                     1u
#define ADC_PSR_PRES77(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES77_SHIFT))&ADC_PSR_PRES77_MASK)
#define ADC_PSR_PRES13_MASK                      0x2000u
#define ADC_PSR_PRES13_SHIFT                     13u
#define ADC_PSR_PRES13_WIDTH                     1u
#define ADC_PSR_PRES13(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES13_SHIFT))&ADC_PSR_PRES13_MASK)
#define ADC_PSR_PRES14_MASK                      0x4000u
#define ADC_PSR_PRES14_SHIFT                     14u
#define ADC_PSR_PRES14_WIDTH                     1u
#define ADC_PSR_PRES14(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES14_SHIFT))&ADC_PSR_PRES14_MASK)
#define ADC_PSR_PRES46_MASK                      0x4000u
#define ADC_PSR_PRES46_SHIFT                     14u
#define ADC_PSR_PRES46_WIDTH                     1u
#define ADC_PSR_PRES46(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES46_SHIFT))&ADC_PSR_PRES46_MASK)
#define ADC_PSR_PRES78_MASK                      0x4000u
#define ADC_PSR_PRES78_SHIFT                     14u
#define ADC_PSR_PRES78_WIDTH                     1u
#define ADC_PSR_PRES78(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES78_SHIFT))&ADC_PSR_PRES78_MASK)
#define ADC_PSR_PRES47_MASK                      0x8000u
#define ADC_PSR_PRES47_SHIFT                     15u
#define ADC_PSR_PRES47_WIDTH                     1u
#define ADC_PSR_PRES47(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES47_SHIFT))&ADC_PSR_PRES47_MASK)
#define ADC_PSR_PRES15_MASK                      0x8000u
#define ADC_PSR_PRES15_SHIFT                     15u
#define ADC_PSR_PRES15_WIDTH                     1u
#define ADC_PSR_PRES15(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES15_SHIFT))&ADC_PSR_PRES15_MASK)
#define ADC_PSR_PRES79_MASK                      0x8000u
#define ADC_PSR_PRES79_SHIFT                     15u
#define ADC_PSR_PRES79_WIDTH                     1u
#define ADC_PSR_PRES79(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES79_SHIFT))&ADC_PSR_PRES79_MASK)
#define ADC_PSR_PRES80_MASK                      0x10000u
#define ADC_PSR_PRES80_SHIFT                     16u
#define ADC_PSR_PRES80_WIDTH                     1u
#define ADC_PSR_PRES80(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES80_SHIFT))&ADC_PSR_PRES80_MASK)
#define ADC_PSR_PRES48_MASK                      0x10000u
#define ADC_PSR_PRES48_SHIFT                     16u
#define ADC_PSR_PRES48_WIDTH                     1u
#define ADC_PSR_PRES48(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES48_SHIFT))&ADC_PSR_PRES48_MASK)
#define ADC_PSR_PRES81_MASK                      0x20000u
#define ADC_PSR_PRES81_SHIFT                     17u
#define ADC_PSR_PRES81_WIDTH                     1u
#define ADC_PSR_PRES81(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES81_SHIFT))&ADC_PSR_PRES81_MASK)
#define ADC_PSR_PRES49_MASK                      0x20000u
#define ADC_PSR_PRES49_SHIFT                     17u
#define ADC_PSR_PRES49_WIDTH                     1u
#define ADC_PSR_PRES49(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES49_SHIFT))&ADC_PSR_PRES49_MASK)
#define ADC_PSR_PRES50_MASK                      0x40000u
#define ADC_PSR_PRES50_SHIFT                     18u
#define ADC_PSR_PRES50_WIDTH                     1u
#define ADC_PSR_PRES50(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES50_SHIFT))&ADC_PSR_PRES50_MASK)
#define ADC_PSR_PRES82_MASK                      0x40000u
#define ADC_PSR_PRES82_SHIFT                     18u
#define ADC_PSR_PRES82_WIDTH                     1u
#define ADC_PSR_PRES82(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES82_SHIFT))&ADC_PSR_PRES82_MASK)
#define ADC_PSR_PRES51_MASK                      0x80000u
#define ADC_PSR_PRES51_SHIFT                     19u
#define ADC_PSR_PRES51_WIDTH                     1u
#define ADC_PSR_PRES51(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES51_SHIFT))&ADC_PSR_PRES51_MASK)
#define ADC_PSR_PRES83_MASK                      0x80000u
#define ADC_PSR_PRES83_SHIFT                     19u
#define ADC_PSR_PRES83_WIDTH                     1u
#define ADC_PSR_PRES83(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES83_SHIFT))&ADC_PSR_PRES83_MASK)
#define ADC_PSR_PRES52_MASK                      0x100000u
#define ADC_PSR_PRES52_SHIFT                     20u
#define ADC_PSR_PRES52_WIDTH                     1u
#define ADC_PSR_PRES52(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES52_SHIFT))&ADC_PSR_PRES52_MASK)
#define ADC_PSR_PRES84_MASK                      0x100000u
#define ADC_PSR_PRES84_SHIFT                     20u
#define ADC_PSR_PRES84_WIDTH                     1u
#define ADC_PSR_PRES84(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES84_SHIFT))&ADC_PSR_PRES84_MASK)
#define ADC_PSR_PRES53_MASK                      0x200000u
#define ADC_PSR_PRES53_SHIFT                     21u
#define ADC_PSR_PRES53_WIDTH                     1u
#define ADC_PSR_PRES53(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES53_SHIFT))&ADC_PSR_PRES53_MASK)
#define ADC_PSR_PRES85_MASK                      0x200000u
#define ADC_PSR_PRES85_SHIFT                     21u
#define ADC_PSR_PRES85_WIDTH                     1u
#define ADC_PSR_PRES85(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES85_SHIFT))&ADC_PSR_PRES85_MASK)
#define ADC_PSR_PRES54_MASK                      0x400000u
#define ADC_PSR_PRES54_SHIFT                     22u
#define ADC_PSR_PRES54_WIDTH                     1u
#define ADC_PSR_PRES54(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES54_SHIFT))&ADC_PSR_PRES54_MASK)
#define ADC_PSR_PRES86_MASK                      0x400000u
#define ADC_PSR_PRES86_SHIFT                     22u
#define ADC_PSR_PRES86_WIDTH                     1u
#define ADC_PSR_PRES86(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES86_SHIFT))&ADC_PSR_PRES86_MASK)
#define ADC_PSR_PRES55_MASK                      0x800000u
#define ADC_PSR_PRES55_SHIFT                     23u
#define ADC_PSR_PRES55_WIDTH                     1u
#define ADC_PSR_PRES55(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES55_SHIFT))&ADC_PSR_PRES55_MASK)
#define ADC_PSR_PRES87_MASK                      0x800000u
#define ADC_PSR_PRES87_SHIFT                     23u
#define ADC_PSR_PRES87_WIDTH                     1u
#define ADC_PSR_PRES87(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES87_SHIFT))&ADC_PSR_PRES87_MASK)
#define ADC_PSR_PRES88_MASK                      0x1000000u
#define ADC_PSR_PRES88_SHIFT                     24u
#define ADC_PSR_PRES88_WIDTH                     1u
#define ADC_PSR_PRES88(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES88_SHIFT))&ADC_PSR_PRES88_MASK)
#define ADC_PSR_PRES56_MASK                      0x1000000u
#define ADC_PSR_PRES56_SHIFT                     24u
#define ADC_PSR_PRES56_WIDTH                     1u
#define ADC_PSR_PRES56(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES56_SHIFT))&ADC_PSR_PRES56_MASK)
#define ADC_PSR_PRES89_MASK                      0x2000000u
#define ADC_PSR_PRES89_SHIFT                     25u
#define ADC_PSR_PRES89_WIDTH                     1u
#define ADC_PSR_PRES89(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES89_SHIFT))&ADC_PSR_PRES89_MASK)
#define ADC_PSR_PRES57_MASK                      0x2000000u
#define ADC_PSR_PRES57_SHIFT                     25u
#define ADC_PSR_PRES57_WIDTH                     1u
#define ADC_PSR_PRES57(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES57_SHIFT))&ADC_PSR_PRES57_MASK)
#define ADC_PSR_PRES90_MASK                      0x4000000u
#define ADC_PSR_PRES90_SHIFT                     26u
#define ADC_PSR_PRES90_WIDTH                     1u
#define ADC_PSR_PRES90(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES90_SHIFT))&ADC_PSR_PRES90_MASK)
#define ADC_PSR_PRES58_MASK                      0x4000000u
#define ADC_PSR_PRES58_SHIFT                     26u
#define ADC_PSR_PRES58_WIDTH                     1u
#define ADC_PSR_PRES58(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES58_SHIFT))&ADC_PSR_PRES58_MASK)
#define ADC_PSR_PRES59_MASK                      0x8000000u
#define ADC_PSR_PRES59_SHIFT                     27u
#define ADC_PSR_PRES59_WIDTH                     1u
#define ADC_PSR_PRES59(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES59_SHIFT))&ADC_PSR_PRES59_MASK)
#define ADC_PSR_PRES91_MASK                      0x8000000u
#define ADC_PSR_PRES91_SHIFT                     27u
#define ADC_PSR_PRES91_WIDTH                     1u
#define ADC_PSR_PRES91(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES91_SHIFT))&ADC_PSR_PRES91_MASK)
#define ADC_PSR_PRES60_MASK                      0x10000000u
#define ADC_PSR_PRES60_SHIFT                     28u
#define ADC_PSR_PRES60_WIDTH                     1u
#define ADC_PSR_PRES60(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES60_SHIFT))&ADC_PSR_PRES60_MASK)
#define ADC_PSR_PRES92_MASK                      0x10000000u
#define ADC_PSR_PRES92_SHIFT                     28u
#define ADC_PSR_PRES92_WIDTH                     1u
#define ADC_PSR_PRES92(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES92_SHIFT))&ADC_PSR_PRES92_MASK)
#define ADC_PSR_PRES61_MASK                      0x20000000u
#define ADC_PSR_PRES61_SHIFT                     29u
#define ADC_PSR_PRES61_WIDTH                     1u
#define ADC_PSR_PRES61(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES61_SHIFT))&ADC_PSR_PRES61_MASK)
#define ADC_PSR_PRES93_MASK                      0x20000000u
#define ADC_PSR_PRES93_SHIFT                     29u
#define ADC_PSR_PRES93_WIDTH                     1u
#define ADC_PSR_PRES93(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES93_SHIFT))&ADC_PSR_PRES93_MASK)
#define ADC_PSR_PRES94_MASK                      0x40000000u
#define ADC_PSR_PRES94_SHIFT                     30u
#define ADC_PSR_PRES94_WIDTH                     1u
#define ADC_PSR_PRES94(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES94_SHIFT))&ADC_PSR_PRES94_MASK)
#define ADC_PSR_PRES62_MASK                      0x40000000u
#define ADC_PSR_PRES62_SHIFT                     30u
#define ADC_PSR_PRES62_WIDTH                     1u
#define ADC_PSR_PRES62(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES62_SHIFT))&ADC_PSR_PRES62_MASK)
#define ADC_PSR_PRES95_MASK                      0x80000000u
#define ADC_PSR_PRES95_SHIFT                     31u
#define ADC_PSR_PRES95_WIDTH                     1u
#define ADC_PSR_PRES95(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES95_SHIFT))&ADC_PSR_PRES95_MASK)
#define ADC_PSR_PRES63_MASK                      0x80000000u
#define ADC_PSR_PRES63_SHIFT                     31u
#define ADC_PSR_PRES63_WIDTH                     1u
#define ADC_PSR_PRES63(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PSR_PRES63_SHIFT))&ADC_PSR_PRES63_MASK)
/* CTR Bit Fields */
#define ADC_CTR_INPSAMP_MASK                     0xFFu
#define ADC_CTR_INPSAMP_SHIFT                    0u
#define ADC_CTR_INPSAMP_WIDTH                    8u
#define ADC_CTR_INPSAMP(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CTR_INPSAMP_SHIFT))&ADC_CTR_INPSAMP_MASK)
/* NCMR Bit Fields */
#define ADC_NCMR_CH0_MASK                        0x1u
#define ADC_NCMR_CH0_SHIFT                       0u
#define ADC_NCMR_CH0_WIDTH                       1u
#define ADC_NCMR_CH0(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH0_SHIFT))&ADC_NCMR_CH0_MASK)
#define ADC_NCMR_CH64_MASK                       0x1u
#define ADC_NCMR_CH64_SHIFT                      0u
#define ADC_NCMR_CH64_WIDTH                      1u
#define ADC_NCMR_CH64(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH64_SHIFT))&ADC_NCMR_CH64_MASK)
#define ADC_NCMR_CH32_MASK                       0x1u
#define ADC_NCMR_CH32_SHIFT                      0u
#define ADC_NCMR_CH32_WIDTH                      1u
#define ADC_NCMR_CH32(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH32_SHIFT))&ADC_NCMR_CH32_MASK)
#define ADC_NCMR_CH65_MASK                       0x2u
#define ADC_NCMR_CH65_SHIFT                      1u
#define ADC_NCMR_CH65_WIDTH                      1u
#define ADC_NCMR_CH65(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH65_SHIFT))&ADC_NCMR_CH65_MASK)
#define ADC_NCMR_CH33_MASK                       0x2u
#define ADC_NCMR_CH33_SHIFT                      1u
#define ADC_NCMR_CH33_WIDTH                      1u
#define ADC_NCMR_CH33(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH33_SHIFT))&ADC_NCMR_CH33_MASK)
#define ADC_NCMR_CH1_MASK                        0x2u
#define ADC_NCMR_CH1_SHIFT                       1u
#define ADC_NCMR_CH1_WIDTH                       1u
#define ADC_NCMR_CH1(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH1_SHIFT))&ADC_NCMR_CH1_MASK)
#define ADC_NCMR_CH34_MASK                       0x4u
#define ADC_NCMR_CH34_SHIFT                      2u
#define ADC_NCMR_CH34_WIDTH                      1u
#define ADC_NCMR_CH34(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH34_SHIFT))&ADC_NCMR_CH34_MASK)
#define ADC_NCMR_CH2_MASK                        0x4u
#define ADC_NCMR_CH2_SHIFT                       2u
#define ADC_NCMR_CH2_WIDTH                       1u
#define ADC_NCMR_CH2(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH2_SHIFT))&ADC_NCMR_CH2_MASK)
#define ADC_NCMR_CH66_MASK                       0x4u
#define ADC_NCMR_CH66_SHIFT                      2u
#define ADC_NCMR_CH66_WIDTH                      1u
#define ADC_NCMR_CH66(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH66_SHIFT))&ADC_NCMR_CH66_MASK)
#define ADC_NCMR_CH67_MASK                       0x8u
#define ADC_NCMR_CH67_SHIFT                      3u
#define ADC_NCMR_CH67_WIDTH                      1u
#define ADC_NCMR_CH67(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH67_SHIFT))&ADC_NCMR_CH67_MASK)
#define ADC_NCMR_CH3_MASK                        0x8u
#define ADC_NCMR_CH3_SHIFT                       3u
#define ADC_NCMR_CH3_WIDTH                       1u
#define ADC_NCMR_CH3(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH3_SHIFT))&ADC_NCMR_CH3_MASK)
#define ADC_NCMR_CH35_MASK                       0x8u
#define ADC_NCMR_CH35_SHIFT                      3u
#define ADC_NCMR_CH35_WIDTH                      1u
#define ADC_NCMR_CH35(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH35_SHIFT))&ADC_NCMR_CH35_MASK)
#define ADC_NCMR_CH4_MASK                        0x10u
#define ADC_NCMR_CH4_SHIFT                       4u
#define ADC_NCMR_CH4_WIDTH                       1u
#define ADC_NCMR_CH4(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH4_SHIFT))&ADC_NCMR_CH4_MASK)
#define ADC_NCMR_CH36_MASK                       0x10u
#define ADC_NCMR_CH36_SHIFT                      4u
#define ADC_NCMR_CH36_WIDTH                      1u
#define ADC_NCMR_CH36(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH36_SHIFT))&ADC_NCMR_CH36_MASK)
#define ADC_NCMR_CH68_MASK                       0x10u
#define ADC_NCMR_CH68_SHIFT                      4u
#define ADC_NCMR_CH68_WIDTH                      1u
#define ADC_NCMR_CH68(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH68_SHIFT))&ADC_NCMR_CH68_MASK)
#define ADC_NCMR_CH69_MASK                       0x20u
#define ADC_NCMR_CH69_SHIFT                      5u
#define ADC_NCMR_CH69_WIDTH                      1u
#define ADC_NCMR_CH69(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH69_SHIFT))&ADC_NCMR_CH69_MASK)
#define ADC_NCMR_CH37_MASK                       0x20u
#define ADC_NCMR_CH37_SHIFT                      5u
#define ADC_NCMR_CH37_WIDTH                      1u
#define ADC_NCMR_CH37(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH37_SHIFT))&ADC_NCMR_CH37_MASK)
#define ADC_NCMR_CH5_MASK                        0x20u
#define ADC_NCMR_CH5_SHIFT                       5u
#define ADC_NCMR_CH5_WIDTH                       1u
#define ADC_NCMR_CH5(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH5_SHIFT))&ADC_NCMR_CH5_MASK)
#define ADC_NCMR_CH70_MASK                       0x40u
#define ADC_NCMR_CH70_SHIFT                      6u
#define ADC_NCMR_CH70_WIDTH                      1u
#define ADC_NCMR_CH70(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH70_SHIFT))&ADC_NCMR_CH70_MASK)
#define ADC_NCMR_CH38_MASK                       0x40u
#define ADC_NCMR_CH38_SHIFT                      6u
#define ADC_NCMR_CH38_WIDTH                      1u
#define ADC_NCMR_CH38(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH38_SHIFT))&ADC_NCMR_CH38_MASK)
#define ADC_NCMR_CH6_MASK                        0x40u
#define ADC_NCMR_CH6_SHIFT                       6u
#define ADC_NCMR_CH6_WIDTH                       1u
#define ADC_NCMR_CH6(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH6_SHIFT))&ADC_NCMR_CH6_MASK)
#define ADC_NCMR_CH7_MASK                        0x80u
#define ADC_NCMR_CH7_SHIFT                       7u
#define ADC_NCMR_CH7_WIDTH                       1u
#define ADC_NCMR_CH7(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH7_SHIFT))&ADC_NCMR_CH7_MASK)
#define ADC_NCMR_CH39_MASK                       0x80u
#define ADC_NCMR_CH39_SHIFT                      7u
#define ADC_NCMR_CH39_WIDTH                      1u
#define ADC_NCMR_CH39(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH39_SHIFT))&ADC_NCMR_CH39_MASK)
#define ADC_NCMR_CH71_MASK                       0x80u
#define ADC_NCMR_CH71_SHIFT                      7u
#define ADC_NCMR_CH71_WIDTH                      1u
#define ADC_NCMR_CH71(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH71_SHIFT))&ADC_NCMR_CH71_MASK)
#define ADC_NCMR_CH8_MASK                        0x100u
#define ADC_NCMR_CH8_SHIFT                       8u
#define ADC_NCMR_CH8_WIDTH                       1u
#define ADC_NCMR_CH8(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH8_SHIFT))&ADC_NCMR_CH8_MASK)
#define ADC_NCMR_CH40_MASK                       0x100u
#define ADC_NCMR_CH40_SHIFT                      8u
#define ADC_NCMR_CH40_WIDTH                      1u
#define ADC_NCMR_CH40(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH40_SHIFT))&ADC_NCMR_CH40_MASK)
#define ADC_NCMR_CH72_MASK                       0x100u
#define ADC_NCMR_CH72_SHIFT                      8u
#define ADC_NCMR_CH72_WIDTH                      1u
#define ADC_NCMR_CH72(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH72_SHIFT))&ADC_NCMR_CH72_MASK)
#define ADC_NCMR_CH73_MASK                       0x200u
#define ADC_NCMR_CH73_SHIFT                      9u
#define ADC_NCMR_CH73_WIDTH                      1u
#define ADC_NCMR_CH73(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH73_SHIFT))&ADC_NCMR_CH73_MASK)
#define ADC_NCMR_CH41_MASK                       0x200u
#define ADC_NCMR_CH41_SHIFT                      9u
#define ADC_NCMR_CH41_WIDTH                      1u
#define ADC_NCMR_CH41(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH41_SHIFT))&ADC_NCMR_CH41_MASK)
#define ADC_NCMR_CH9_MASK                        0x200u
#define ADC_NCMR_CH9_SHIFT                       9u
#define ADC_NCMR_CH9_WIDTH                       1u
#define ADC_NCMR_CH9(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH9_SHIFT))&ADC_NCMR_CH9_MASK)
#define ADC_NCMR_CH42_MASK                       0x400u
#define ADC_NCMR_CH42_SHIFT                      10u
#define ADC_NCMR_CH42_WIDTH                      1u
#define ADC_NCMR_CH42(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH42_SHIFT))&ADC_NCMR_CH42_MASK)
#define ADC_NCMR_CH74_MASK                       0x400u
#define ADC_NCMR_CH74_SHIFT                      10u
#define ADC_NCMR_CH74_WIDTH                      1u
#define ADC_NCMR_CH74(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH74_SHIFT))&ADC_NCMR_CH74_MASK)
#define ADC_NCMR_CH10_MASK                       0x400u
#define ADC_NCMR_CH10_SHIFT                      10u
#define ADC_NCMR_CH10_WIDTH                      1u
#define ADC_NCMR_CH10(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH10_SHIFT))&ADC_NCMR_CH10_MASK)
#define ADC_NCMR_CH11_MASK                       0x800u
#define ADC_NCMR_CH11_SHIFT                      11u
#define ADC_NCMR_CH11_WIDTH                      1u
#define ADC_NCMR_CH11(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH11_SHIFT))&ADC_NCMR_CH11_MASK)
#define ADC_NCMR_CH75_MASK                       0x800u
#define ADC_NCMR_CH75_SHIFT                      11u
#define ADC_NCMR_CH75_WIDTH                      1u
#define ADC_NCMR_CH75(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH75_SHIFT))&ADC_NCMR_CH75_MASK)
#define ADC_NCMR_CH43_MASK                       0x800u
#define ADC_NCMR_CH43_SHIFT                      11u
#define ADC_NCMR_CH43_WIDTH                      1u
#define ADC_NCMR_CH43(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH43_SHIFT))&ADC_NCMR_CH43_MASK)
#define ADC_NCMR_CH76_MASK                       0x1000u
#define ADC_NCMR_CH76_SHIFT                      12u
#define ADC_NCMR_CH76_WIDTH                      1u
#define ADC_NCMR_CH76(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH76_SHIFT))&ADC_NCMR_CH76_MASK)
#define ADC_NCMR_CH12_MASK                       0x1000u
#define ADC_NCMR_CH12_SHIFT                      12u
#define ADC_NCMR_CH12_WIDTH                      1u
#define ADC_NCMR_CH12(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH12_SHIFT))&ADC_NCMR_CH12_MASK)
#define ADC_NCMR_CH44_MASK                       0x1000u
#define ADC_NCMR_CH44_SHIFT                      12u
#define ADC_NCMR_CH44_WIDTH                      1u
#define ADC_NCMR_CH44(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH44_SHIFT))&ADC_NCMR_CH44_MASK)
#define ADC_NCMR_CH45_MASK                       0x2000u
#define ADC_NCMR_CH45_SHIFT                      13u
#define ADC_NCMR_CH45_WIDTH                      1u
#define ADC_NCMR_CH45(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH45_SHIFT))&ADC_NCMR_CH45_MASK)
#define ADC_NCMR_CH77_MASK                       0x2000u
#define ADC_NCMR_CH77_SHIFT                      13u
#define ADC_NCMR_CH77_WIDTH                      1u
#define ADC_NCMR_CH77(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH77_SHIFT))&ADC_NCMR_CH77_MASK)
#define ADC_NCMR_CH13_MASK                       0x2000u
#define ADC_NCMR_CH13_SHIFT                      13u
#define ADC_NCMR_CH13_WIDTH                      1u
#define ADC_NCMR_CH13(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH13_SHIFT))&ADC_NCMR_CH13_MASK)
#define ADC_NCMR_CH14_MASK                       0x4000u
#define ADC_NCMR_CH14_SHIFT                      14u
#define ADC_NCMR_CH14_WIDTH                      1u
#define ADC_NCMR_CH14(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH14_SHIFT))&ADC_NCMR_CH14_MASK)
#define ADC_NCMR_CH46_MASK                       0x4000u
#define ADC_NCMR_CH46_SHIFT                      14u
#define ADC_NCMR_CH46_WIDTH                      1u
#define ADC_NCMR_CH46(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH46_SHIFT))&ADC_NCMR_CH46_MASK)
#define ADC_NCMR_CH78_MASK                       0x4000u
#define ADC_NCMR_CH78_SHIFT                      14u
#define ADC_NCMR_CH78_WIDTH                      1u
#define ADC_NCMR_CH78(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH78_SHIFT))&ADC_NCMR_CH78_MASK)
#define ADC_NCMR_CH47_MASK                       0x8000u
#define ADC_NCMR_CH47_SHIFT                      15u
#define ADC_NCMR_CH47_WIDTH                      1u
#define ADC_NCMR_CH47(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH47_SHIFT))&ADC_NCMR_CH47_MASK)
#define ADC_NCMR_CH15_MASK                       0x8000u
#define ADC_NCMR_CH15_SHIFT                      15u
#define ADC_NCMR_CH15_WIDTH                      1u
#define ADC_NCMR_CH15(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH15_SHIFT))&ADC_NCMR_CH15_MASK)
#define ADC_NCMR_CH79_MASK                       0x8000u
#define ADC_NCMR_CH79_SHIFT                      15u
#define ADC_NCMR_CH79_WIDTH                      1u
#define ADC_NCMR_CH79(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH79_SHIFT))&ADC_NCMR_CH79_MASK)
#define ADC_NCMR_CH80_MASK                       0x10000u
#define ADC_NCMR_CH80_SHIFT                      16u
#define ADC_NCMR_CH80_WIDTH                      1u
#define ADC_NCMR_CH80(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH80_SHIFT))&ADC_NCMR_CH80_MASK)
#define ADC_NCMR_CH48_MASK                       0x10000u
#define ADC_NCMR_CH48_SHIFT                      16u
#define ADC_NCMR_CH48_WIDTH                      1u
#define ADC_NCMR_CH48(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH48_SHIFT))&ADC_NCMR_CH48_MASK)
#define ADC_NCMR_CH81_MASK                       0x20000u
#define ADC_NCMR_CH81_SHIFT                      17u
#define ADC_NCMR_CH81_WIDTH                      1u
#define ADC_NCMR_CH81(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH81_SHIFT))&ADC_NCMR_CH81_MASK)
#define ADC_NCMR_CH49_MASK                       0x20000u
#define ADC_NCMR_CH49_SHIFT                      17u
#define ADC_NCMR_CH49_WIDTH                      1u
#define ADC_NCMR_CH49(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH49_SHIFT))&ADC_NCMR_CH49_MASK)
#define ADC_NCMR_CH50_MASK                       0x40000u
#define ADC_NCMR_CH50_SHIFT                      18u
#define ADC_NCMR_CH50_WIDTH                      1u
#define ADC_NCMR_CH50(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH50_SHIFT))&ADC_NCMR_CH50_MASK)
#define ADC_NCMR_CH82_MASK                       0x40000u
#define ADC_NCMR_CH82_SHIFT                      18u
#define ADC_NCMR_CH82_WIDTH                      1u
#define ADC_NCMR_CH82(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH82_SHIFT))&ADC_NCMR_CH82_MASK)
#define ADC_NCMR_CH51_MASK                       0x80000u
#define ADC_NCMR_CH51_SHIFT                      19u
#define ADC_NCMR_CH51_WIDTH                      1u
#define ADC_NCMR_CH51(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH51_SHIFT))&ADC_NCMR_CH51_MASK)
#define ADC_NCMR_CH83_MASK                       0x80000u
#define ADC_NCMR_CH83_SHIFT                      19u
#define ADC_NCMR_CH83_WIDTH                      1u
#define ADC_NCMR_CH83(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH83_SHIFT))&ADC_NCMR_CH83_MASK)
#define ADC_NCMR_CH52_MASK                       0x100000u
#define ADC_NCMR_CH52_SHIFT                      20u
#define ADC_NCMR_CH52_WIDTH                      1u
#define ADC_NCMR_CH52(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH52_SHIFT))&ADC_NCMR_CH52_MASK)
#define ADC_NCMR_CH84_MASK                       0x100000u
#define ADC_NCMR_CH84_SHIFT                      20u
#define ADC_NCMR_CH84_WIDTH                      1u
#define ADC_NCMR_CH84(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH84_SHIFT))&ADC_NCMR_CH84_MASK)
#define ADC_NCMR_CH53_MASK                       0x200000u
#define ADC_NCMR_CH53_SHIFT                      21u
#define ADC_NCMR_CH53_WIDTH                      1u
#define ADC_NCMR_CH53(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH53_SHIFT))&ADC_NCMR_CH53_MASK)
#define ADC_NCMR_CH85_MASK                       0x200000u
#define ADC_NCMR_CH85_SHIFT                      21u
#define ADC_NCMR_CH85_WIDTH                      1u
#define ADC_NCMR_CH85(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH85_SHIFT))&ADC_NCMR_CH85_MASK)
#define ADC_NCMR_CH54_MASK                       0x400000u
#define ADC_NCMR_CH54_SHIFT                      22u
#define ADC_NCMR_CH54_WIDTH                      1u
#define ADC_NCMR_CH54(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH54_SHIFT))&ADC_NCMR_CH54_MASK)
#define ADC_NCMR_CH86_MASK                       0x400000u
#define ADC_NCMR_CH86_SHIFT                      22u
#define ADC_NCMR_CH86_WIDTH                      1u
#define ADC_NCMR_CH86(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH86_SHIFT))&ADC_NCMR_CH86_MASK)
#define ADC_NCMR_CH55_MASK                       0x800000u
#define ADC_NCMR_CH55_SHIFT                      23u
#define ADC_NCMR_CH55_WIDTH                      1u
#define ADC_NCMR_CH55(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH55_SHIFT))&ADC_NCMR_CH55_MASK)
#define ADC_NCMR_CH87_MASK                       0x800000u
#define ADC_NCMR_CH87_SHIFT                      23u
#define ADC_NCMR_CH87_WIDTH                      1u
#define ADC_NCMR_CH87(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH87_SHIFT))&ADC_NCMR_CH87_MASK)
#define ADC_NCMR_CH88_MASK                       0x1000000u
#define ADC_NCMR_CH88_SHIFT                      24u
#define ADC_NCMR_CH88_WIDTH                      1u
#define ADC_NCMR_CH88(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH88_SHIFT))&ADC_NCMR_CH88_MASK)
#define ADC_NCMR_CH56_MASK                       0x1000000u
#define ADC_NCMR_CH56_SHIFT                      24u
#define ADC_NCMR_CH56_WIDTH                      1u
#define ADC_NCMR_CH56(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH56_SHIFT))&ADC_NCMR_CH56_MASK)
#define ADC_NCMR_CH89_MASK                       0x2000000u
#define ADC_NCMR_CH89_SHIFT                      25u
#define ADC_NCMR_CH89_WIDTH                      1u
#define ADC_NCMR_CH89(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH89_SHIFT))&ADC_NCMR_CH89_MASK)
#define ADC_NCMR_CH57_MASK                       0x2000000u
#define ADC_NCMR_CH57_SHIFT                      25u
#define ADC_NCMR_CH57_WIDTH                      1u
#define ADC_NCMR_CH57(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH57_SHIFT))&ADC_NCMR_CH57_MASK)
#define ADC_NCMR_CH90_MASK                       0x4000000u
#define ADC_NCMR_CH90_SHIFT                      26u
#define ADC_NCMR_CH90_WIDTH                      1u
#define ADC_NCMR_CH90(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH90_SHIFT))&ADC_NCMR_CH90_MASK)
#define ADC_NCMR_CH58_MASK                       0x4000000u
#define ADC_NCMR_CH58_SHIFT                      26u
#define ADC_NCMR_CH58_WIDTH                      1u
#define ADC_NCMR_CH58(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH58_SHIFT))&ADC_NCMR_CH58_MASK)
#define ADC_NCMR_CH59_MASK                       0x8000000u
#define ADC_NCMR_CH59_SHIFT                      27u
#define ADC_NCMR_CH59_WIDTH                      1u
#define ADC_NCMR_CH59(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH59_SHIFT))&ADC_NCMR_CH59_MASK)
#define ADC_NCMR_CH91_MASK                       0x8000000u
#define ADC_NCMR_CH91_SHIFT                      27u
#define ADC_NCMR_CH91_WIDTH                      1u
#define ADC_NCMR_CH91(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH91_SHIFT))&ADC_NCMR_CH91_MASK)
#define ADC_NCMR_CH60_MASK                       0x10000000u
#define ADC_NCMR_CH60_SHIFT                      28u
#define ADC_NCMR_CH60_WIDTH                      1u
#define ADC_NCMR_CH60(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH60_SHIFT))&ADC_NCMR_CH60_MASK)
#define ADC_NCMR_CH92_MASK                       0x10000000u
#define ADC_NCMR_CH92_SHIFT                      28u
#define ADC_NCMR_CH92_WIDTH                      1u
#define ADC_NCMR_CH92(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH92_SHIFT))&ADC_NCMR_CH92_MASK)
#define ADC_NCMR_CH61_MASK                       0x20000000u
#define ADC_NCMR_CH61_SHIFT                      29u
#define ADC_NCMR_CH61_WIDTH                      1u
#define ADC_NCMR_CH61(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH61_SHIFT))&ADC_NCMR_CH61_MASK)
#define ADC_NCMR_CH93_MASK                       0x20000000u
#define ADC_NCMR_CH93_SHIFT                      29u
#define ADC_NCMR_CH93_WIDTH                      1u
#define ADC_NCMR_CH93(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH93_SHIFT))&ADC_NCMR_CH93_MASK)
#define ADC_NCMR_CH94_MASK                       0x40000000u
#define ADC_NCMR_CH94_SHIFT                      30u
#define ADC_NCMR_CH94_WIDTH                      1u
#define ADC_NCMR_CH94(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH94_SHIFT))&ADC_NCMR_CH94_MASK)
#define ADC_NCMR_CH62_MASK                       0x40000000u
#define ADC_NCMR_CH62_SHIFT                      30u
#define ADC_NCMR_CH62_WIDTH                      1u
#define ADC_NCMR_CH62(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH62_SHIFT))&ADC_NCMR_CH62_MASK)
#define ADC_NCMR_CH95_MASK                       0x80000000u
#define ADC_NCMR_CH95_SHIFT                      31u
#define ADC_NCMR_CH95_WIDTH                      1u
#define ADC_NCMR_CH95(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH95_SHIFT))&ADC_NCMR_CH95_MASK)
#define ADC_NCMR_CH63_MASK                       0x80000000u
#define ADC_NCMR_CH63_SHIFT                      31u
#define ADC_NCMR_CH63_WIDTH                      1u
#define ADC_NCMR_CH63(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_NCMR_CH63_SHIFT))&ADC_NCMR_CH63_MASK)
/* JCMR Bit Fields */
#define ADC_JCMR_CH0_MASK                        0x1u
#define ADC_JCMR_CH0_SHIFT                       0u
#define ADC_JCMR_CH0_WIDTH                       1u
#define ADC_JCMR_CH0(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH0_SHIFT))&ADC_JCMR_CH0_MASK)
#define ADC_JCMR_CH64_MASK                       0x1u
#define ADC_JCMR_CH64_SHIFT                      0u
#define ADC_JCMR_CH64_WIDTH                      1u
#define ADC_JCMR_CH64(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH64_SHIFT))&ADC_JCMR_CH64_MASK)
#define ADC_JCMR_CH32_MASK                       0x1u
#define ADC_JCMR_CH32_SHIFT                      0u
#define ADC_JCMR_CH32_WIDTH                      1u
#define ADC_JCMR_CH32(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH32_SHIFT))&ADC_JCMR_CH32_MASK)
#define ADC_JCMR_CH65_MASK                       0x2u
#define ADC_JCMR_CH65_SHIFT                      1u
#define ADC_JCMR_CH65_WIDTH                      1u
#define ADC_JCMR_CH65(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH65_SHIFT))&ADC_JCMR_CH65_MASK)
#define ADC_JCMR_CH33_MASK                       0x2u
#define ADC_JCMR_CH33_SHIFT                      1u
#define ADC_JCMR_CH33_WIDTH                      1u
#define ADC_JCMR_CH33(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH33_SHIFT))&ADC_JCMR_CH33_MASK)
#define ADC_JCMR_CH1_MASK                        0x2u
#define ADC_JCMR_CH1_SHIFT                       1u
#define ADC_JCMR_CH1_WIDTH                       1u
#define ADC_JCMR_CH1(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH1_SHIFT))&ADC_JCMR_CH1_MASK)
#define ADC_JCMR_CH34_MASK                       0x4u
#define ADC_JCMR_CH34_SHIFT                      2u
#define ADC_JCMR_CH34_WIDTH                      1u
#define ADC_JCMR_CH34(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH34_SHIFT))&ADC_JCMR_CH34_MASK)
#define ADC_JCMR_CH2_MASK                        0x4u
#define ADC_JCMR_CH2_SHIFT                       2u
#define ADC_JCMR_CH2_WIDTH                       1u
#define ADC_JCMR_CH2(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH2_SHIFT))&ADC_JCMR_CH2_MASK)
#define ADC_JCMR_CH66_MASK                       0x4u
#define ADC_JCMR_CH66_SHIFT                      2u
#define ADC_JCMR_CH66_WIDTH                      1u
#define ADC_JCMR_CH66(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH66_SHIFT))&ADC_JCMR_CH66_MASK)
#define ADC_JCMR_CH67_MASK                       0x8u
#define ADC_JCMR_CH67_SHIFT                      3u
#define ADC_JCMR_CH67_WIDTH                      1u
#define ADC_JCMR_CH67(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH67_SHIFT))&ADC_JCMR_CH67_MASK)
#define ADC_JCMR_CH3_MASK                        0x8u
#define ADC_JCMR_CH3_SHIFT                       3u
#define ADC_JCMR_CH3_WIDTH                       1u
#define ADC_JCMR_CH3(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH3_SHIFT))&ADC_JCMR_CH3_MASK)
#define ADC_JCMR_CH35_MASK                       0x8u
#define ADC_JCMR_CH35_SHIFT                      3u
#define ADC_JCMR_CH35_WIDTH                      1u
#define ADC_JCMR_CH35(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH35_SHIFT))&ADC_JCMR_CH35_MASK)
#define ADC_JCMR_CH4_MASK                        0x10u
#define ADC_JCMR_CH4_SHIFT                       4u
#define ADC_JCMR_CH4_WIDTH                       1u
#define ADC_JCMR_CH4(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH4_SHIFT))&ADC_JCMR_CH4_MASK)
#define ADC_JCMR_CH36_MASK                       0x10u
#define ADC_JCMR_CH36_SHIFT                      4u
#define ADC_JCMR_CH36_WIDTH                      1u
#define ADC_JCMR_CH36(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH36_SHIFT))&ADC_JCMR_CH36_MASK)
#define ADC_JCMR_CH68_MASK                       0x10u
#define ADC_JCMR_CH68_SHIFT                      4u
#define ADC_JCMR_CH68_WIDTH                      1u
#define ADC_JCMR_CH68(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH68_SHIFT))&ADC_JCMR_CH68_MASK)
#define ADC_JCMR_CH69_MASK                       0x20u
#define ADC_JCMR_CH69_SHIFT                      5u
#define ADC_JCMR_CH69_WIDTH                      1u
#define ADC_JCMR_CH69(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH69_SHIFT))&ADC_JCMR_CH69_MASK)
#define ADC_JCMR_CH37_MASK                       0x20u
#define ADC_JCMR_CH37_SHIFT                      5u
#define ADC_JCMR_CH37_WIDTH                      1u
#define ADC_JCMR_CH37(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH37_SHIFT))&ADC_JCMR_CH37_MASK)
#define ADC_JCMR_CH5_MASK                        0x20u
#define ADC_JCMR_CH5_SHIFT                       5u
#define ADC_JCMR_CH5_WIDTH                       1u
#define ADC_JCMR_CH5(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH5_SHIFT))&ADC_JCMR_CH5_MASK)
#define ADC_JCMR_CH70_MASK                       0x40u
#define ADC_JCMR_CH70_SHIFT                      6u
#define ADC_JCMR_CH70_WIDTH                      1u
#define ADC_JCMR_CH70(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH70_SHIFT))&ADC_JCMR_CH70_MASK)
#define ADC_JCMR_CH38_MASK                       0x40u
#define ADC_JCMR_CH38_SHIFT                      6u
#define ADC_JCMR_CH38_WIDTH                      1u
#define ADC_JCMR_CH38(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH38_SHIFT))&ADC_JCMR_CH38_MASK)
#define ADC_JCMR_CH6_MASK                        0x40u
#define ADC_JCMR_CH6_SHIFT                       6u
#define ADC_JCMR_CH6_WIDTH                       1u
#define ADC_JCMR_CH6(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH6_SHIFT))&ADC_JCMR_CH6_MASK)
#define ADC_JCMR_CH7_MASK                        0x80u
#define ADC_JCMR_CH7_SHIFT                       7u
#define ADC_JCMR_CH7_WIDTH                       1u
#define ADC_JCMR_CH7(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH7_SHIFT))&ADC_JCMR_CH7_MASK)
#define ADC_JCMR_CH39_MASK                       0x80u
#define ADC_JCMR_CH39_SHIFT                      7u
#define ADC_JCMR_CH39_WIDTH                      1u
#define ADC_JCMR_CH39(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH39_SHIFT))&ADC_JCMR_CH39_MASK)
#define ADC_JCMR_CH71_MASK                       0x80u
#define ADC_JCMR_CH71_SHIFT                      7u
#define ADC_JCMR_CH71_WIDTH                      1u
#define ADC_JCMR_CH71(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH71_SHIFT))&ADC_JCMR_CH71_MASK)
#define ADC_JCMR_CH8_MASK                        0x100u
#define ADC_JCMR_CH8_SHIFT                       8u
#define ADC_JCMR_CH8_WIDTH                       1u
#define ADC_JCMR_CH8(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH8_SHIFT))&ADC_JCMR_CH8_MASK)
#define ADC_JCMR_CH40_MASK                       0x100u
#define ADC_JCMR_CH40_SHIFT                      8u
#define ADC_JCMR_CH40_WIDTH                      1u
#define ADC_JCMR_CH40(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH40_SHIFT))&ADC_JCMR_CH40_MASK)
#define ADC_JCMR_CH72_MASK                       0x100u
#define ADC_JCMR_CH72_SHIFT                      8u
#define ADC_JCMR_CH72_WIDTH                      1u
#define ADC_JCMR_CH72(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH72_SHIFT))&ADC_JCMR_CH72_MASK)
#define ADC_JCMR_CH73_MASK                       0x200u
#define ADC_JCMR_CH73_SHIFT                      9u
#define ADC_JCMR_CH73_WIDTH                      1u
#define ADC_JCMR_CH73(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH73_SHIFT))&ADC_JCMR_CH73_MASK)
#define ADC_JCMR_CH41_MASK                       0x200u
#define ADC_JCMR_CH41_SHIFT                      9u
#define ADC_JCMR_CH41_WIDTH                      1u
#define ADC_JCMR_CH41(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH41_SHIFT))&ADC_JCMR_CH41_MASK)
#define ADC_JCMR_CH9_MASK                        0x200u
#define ADC_JCMR_CH9_SHIFT                       9u
#define ADC_JCMR_CH9_WIDTH                       1u
#define ADC_JCMR_CH9(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH9_SHIFT))&ADC_JCMR_CH9_MASK)
#define ADC_JCMR_CH42_MASK                       0x400u
#define ADC_JCMR_CH42_SHIFT                      10u
#define ADC_JCMR_CH42_WIDTH                      1u
#define ADC_JCMR_CH42(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH42_SHIFT))&ADC_JCMR_CH42_MASK)
#define ADC_JCMR_CH74_MASK                       0x400u
#define ADC_JCMR_CH74_SHIFT                      10u
#define ADC_JCMR_CH74_WIDTH                      1u
#define ADC_JCMR_CH74(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH74_SHIFT))&ADC_JCMR_CH74_MASK)
#define ADC_JCMR_CH10_MASK                       0x400u
#define ADC_JCMR_CH10_SHIFT                      10u
#define ADC_JCMR_CH10_WIDTH                      1u
#define ADC_JCMR_CH10(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH10_SHIFT))&ADC_JCMR_CH10_MASK)
#define ADC_JCMR_CH11_MASK                       0x800u
#define ADC_JCMR_CH11_SHIFT                      11u
#define ADC_JCMR_CH11_WIDTH                      1u
#define ADC_JCMR_CH11(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH11_SHIFT))&ADC_JCMR_CH11_MASK)
#define ADC_JCMR_CH75_MASK                       0x800u
#define ADC_JCMR_CH75_SHIFT                      11u
#define ADC_JCMR_CH75_WIDTH                      1u
#define ADC_JCMR_CH75(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH75_SHIFT))&ADC_JCMR_CH75_MASK)
#define ADC_JCMR_CH43_MASK                       0x800u
#define ADC_JCMR_CH43_SHIFT                      11u
#define ADC_JCMR_CH43_WIDTH                      1u
#define ADC_JCMR_CH43(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH43_SHIFT))&ADC_JCMR_CH43_MASK)
#define ADC_JCMR_CH76_MASK                       0x1000u
#define ADC_JCMR_CH76_SHIFT                      12u
#define ADC_JCMR_CH76_WIDTH                      1u
#define ADC_JCMR_CH76(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH76_SHIFT))&ADC_JCMR_CH76_MASK)
#define ADC_JCMR_CH12_MASK                       0x1000u
#define ADC_JCMR_CH12_SHIFT                      12u
#define ADC_JCMR_CH12_WIDTH                      1u
#define ADC_JCMR_CH12(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH12_SHIFT))&ADC_JCMR_CH12_MASK)
#define ADC_JCMR_CH44_MASK                       0x1000u
#define ADC_JCMR_CH44_SHIFT                      12u
#define ADC_JCMR_CH44_WIDTH                      1u
#define ADC_JCMR_CH44(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH44_SHIFT))&ADC_JCMR_CH44_MASK)
#define ADC_JCMR_CH45_MASK                       0x2000u
#define ADC_JCMR_CH45_SHIFT                      13u
#define ADC_JCMR_CH45_WIDTH                      1u
#define ADC_JCMR_CH45(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH45_SHIFT))&ADC_JCMR_CH45_MASK)
#define ADC_JCMR_CH77_MASK                       0x2000u
#define ADC_JCMR_CH77_SHIFT                      13u
#define ADC_JCMR_CH77_WIDTH                      1u
#define ADC_JCMR_CH77(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH77_SHIFT))&ADC_JCMR_CH77_MASK)
#define ADC_JCMR_CH13_MASK                       0x2000u
#define ADC_JCMR_CH13_SHIFT                      13u
#define ADC_JCMR_CH13_WIDTH                      1u
#define ADC_JCMR_CH13(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH13_SHIFT))&ADC_JCMR_CH13_MASK)
#define ADC_JCMR_CH14_MASK                       0x4000u
#define ADC_JCMR_CH14_SHIFT                      14u
#define ADC_JCMR_CH14_WIDTH                      1u
#define ADC_JCMR_CH14(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH14_SHIFT))&ADC_JCMR_CH14_MASK)
#define ADC_JCMR_CH46_MASK                       0x4000u
#define ADC_JCMR_CH46_SHIFT                      14u
#define ADC_JCMR_CH46_WIDTH                      1u
#define ADC_JCMR_CH46(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH46_SHIFT))&ADC_JCMR_CH46_MASK)
#define ADC_JCMR_CH78_MASK                       0x4000u
#define ADC_JCMR_CH78_SHIFT                      14u
#define ADC_JCMR_CH78_WIDTH                      1u
#define ADC_JCMR_CH78(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH78_SHIFT))&ADC_JCMR_CH78_MASK)
#define ADC_JCMR_CH47_MASK                       0x8000u
#define ADC_JCMR_CH47_SHIFT                      15u
#define ADC_JCMR_CH47_WIDTH                      1u
#define ADC_JCMR_CH47(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH47_SHIFT))&ADC_JCMR_CH47_MASK)
#define ADC_JCMR_CH15_MASK                       0x8000u
#define ADC_JCMR_CH15_SHIFT                      15u
#define ADC_JCMR_CH15_WIDTH                      1u
#define ADC_JCMR_CH15(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH15_SHIFT))&ADC_JCMR_CH15_MASK)
#define ADC_JCMR_CH79_MASK                       0x8000u
#define ADC_JCMR_CH79_SHIFT                      15u
#define ADC_JCMR_CH79_WIDTH                      1u
#define ADC_JCMR_CH79(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH79_SHIFT))&ADC_JCMR_CH79_MASK)
#define ADC_JCMR_CH80_MASK                       0x10000u
#define ADC_JCMR_CH80_SHIFT                      16u
#define ADC_JCMR_CH80_WIDTH                      1u
#define ADC_JCMR_CH80(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH80_SHIFT))&ADC_JCMR_CH80_MASK)
#define ADC_JCMR_CH48_MASK                       0x10000u
#define ADC_JCMR_CH48_SHIFT                      16u
#define ADC_JCMR_CH48_WIDTH                      1u
#define ADC_JCMR_CH48(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH48_SHIFT))&ADC_JCMR_CH48_MASK)
#define ADC_JCMR_CH81_MASK                       0x20000u
#define ADC_JCMR_CH81_SHIFT                      17u
#define ADC_JCMR_CH81_WIDTH                      1u
#define ADC_JCMR_CH81(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH81_SHIFT))&ADC_JCMR_CH81_MASK)
#define ADC_JCMR_CH49_MASK                       0x20000u
#define ADC_JCMR_CH49_SHIFT                      17u
#define ADC_JCMR_CH49_WIDTH                      1u
#define ADC_JCMR_CH49(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH49_SHIFT))&ADC_JCMR_CH49_MASK)
#define ADC_JCMR_CH50_MASK                       0x40000u
#define ADC_JCMR_CH50_SHIFT                      18u
#define ADC_JCMR_CH50_WIDTH                      1u
#define ADC_JCMR_CH50(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH50_SHIFT))&ADC_JCMR_CH50_MASK)
#define ADC_JCMR_CH82_MASK                       0x40000u
#define ADC_JCMR_CH82_SHIFT                      18u
#define ADC_JCMR_CH82_WIDTH                      1u
#define ADC_JCMR_CH82(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH82_SHIFT))&ADC_JCMR_CH82_MASK)
#define ADC_JCMR_CH51_MASK                       0x80000u
#define ADC_JCMR_CH51_SHIFT                      19u
#define ADC_JCMR_CH51_WIDTH                      1u
#define ADC_JCMR_CH51(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH51_SHIFT))&ADC_JCMR_CH51_MASK)
#define ADC_JCMR_CH83_MASK                       0x80000u
#define ADC_JCMR_CH83_SHIFT                      19u
#define ADC_JCMR_CH83_WIDTH                      1u
#define ADC_JCMR_CH83(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH83_SHIFT))&ADC_JCMR_CH83_MASK)
#define ADC_JCMR_CH52_MASK                       0x100000u
#define ADC_JCMR_CH52_SHIFT                      20u
#define ADC_JCMR_CH52_WIDTH                      1u
#define ADC_JCMR_CH52(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH52_SHIFT))&ADC_JCMR_CH52_MASK)
#define ADC_JCMR_CH84_MASK                       0x100000u
#define ADC_JCMR_CH84_SHIFT                      20u
#define ADC_JCMR_CH84_WIDTH                      1u
#define ADC_JCMR_CH84(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH84_SHIFT))&ADC_JCMR_CH84_MASK)
#define ADC_JCMR_CH53_MASK                       0x200000u
#define ADC_JCMR_CH53_SHIFT                      21u
#define ADC_JCMR_CH53_WIDTH                      1u
#define ADC_JCMR_CH53(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH53_SHIFT))&ADC_JCMR_CH53_MASK)
#define ADC_JCMR_CH85_MASK                       0x200000u
#define ADC_JCMR_CH85_SHIFT                      21u
#define ADC_JCMR_CH85_WIDTH                      1u
#define ADC_JCMR_CH85(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH85_SHIFT))&ADC_JCMR_CH85_MASK)
#define ADC_JCMR_CH54_MASK                       0x400000u
#define ADC_JCMR_CH54_SHIFT                      22u
#define ADC_JCMR_CH54_WIDTH                      1u
#define ADC_JCMR_CH54(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH54_SHIFT))&ADC_JCMR_CH54_MASK)
#define ADC_JCMR_CH86_MASK                       0x400000u
#define ADC_JCMR_CH86_SHIFT                      22u
#define ADC_JCMR_CH86_WIDTH                      1u
#define ADC_JCMR_CH86(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH86_SHIFT))&ADC_JCMR_CH86_MASK)
#define ADC_JCMR_CH55_MASK                       0x800000u
#define ADC_JCMR_CH55_SHIFT                      23u
#define ADC_JCMR_CH55_WIDTH                      1u
#define ADC_JCMR_CH55(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH55_SHIFT))&ADC_JCMR_CH55_MASK)
#define ADC_JCMR_CH87_MASK                       0x800000u
#define ADC_JCMR_CH87_SHIFT                      23u
#define ADC_JCMR_CH87_WIDTH                      1u
#define ADC_JCMR_CH87(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH87_SHIFT))&ADC_JCMR_CH87_MASK)
#define ADC_JCMR_CH88_MASK                       0x1000000u
#define ADC_JCMR_CH88_SHIFT                      24u
#define ADC_JCMR_CH88_WIDTH                      1u
#define ADC_JCMR_CH88(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH88_SHIFT))&ADC_JCMR_CH88_MASK)
#define ADC_JCMR_CH56_MASK                       0x1000000u
#define ADC_JCMR_CH56_SHIFT                      24u
#define ADC_JCMR_CH56_WIDTH                      1u
#define ADC_JCMR_CH56(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH56_SHIFT))&ADC_JCMR_CH56_MASK)
#define ADC_JCMR_CH89_MASK                       0x2000000u
#define ADC_JCMR_CH89_SHIFT                      25u
#define ADC_JCMR_CH89_WIDTH                      1u
#define ADC_JCMR_CH89(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH89_SHIFT))&ADC_JCMR_CH89_MASK)
#define ADC_JCMR_CH57_MASK                       0x2000000u
#define ADC_JCMR_CH57_SHIFT                      25u
#define ADC_JCMR_CH57_WIDTH                      1u
#define ADC_JCMR_CH57(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH57_SHIFT))&ADC_JCMR_CH57_MASK)
#define ADC_JCMR_CH90_MASK                       0x4000000u
#define ADC_JCMR_CH90_SHIFT                      26u
#define ADC_JCMR_CH90_WIDTH                      1u
#define ADC_JCMR_CH90(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH90_SHIFT))&ADC_JCMR_CH90_MASK)
#define ADC_JCMR_CH58_MASK                       0x4000000u
#define ADC_JCMR_CH58_SHIFT                      26u
#define ADC_JCMR_CH58_WIDTH                      1u
#define ADC_JCMR_CH58(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH58_SHIFT))&ADC_JCMR_CH58_MASK)
#define ADC_JCMR_CH59_MASK                       0x8000000u
#define ADC_JCMR_CH59_SHIFT                      27u
#define ADC_JCMR_CH59_WIDTH                      1u
#define ADC_JCMR_CH59(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH59_SHIFT))&ADC_JCMR_CH59_MASK)
#define ADC_JCMR_CH91_MASK                       0x8000000u
#define ADC_JCMR_CH91_SHIFT                      27u
#define ADC_JCMR_CH91_WIDTH                      1u
#define ADC_JCMR_CH91(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH91_SHIFT))&ADC_JCMR_CH91_MASK)
#define ADC_JCMR_CH60_MASK                       0x10000000u
#define ADC_JCMR_CH60_SHIFT                      28u
#define ADC_JCMR_CH60_WIDTH                      1u
#define ADC_JCMR_CH60(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH60_SHIFT))&ADC_JCMR_CH60_MASK)
#define ADC_JCMR_CH92_MASK                       0x10000000u
#define ADC_JCMR_CH92_SHIFT                      28u
#define ADC_JCMR_CH92_WIDTH                      1u
#define ADC_JCMR_CH92(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH92_SHIFT))&ADC_JCMR_CH92_MASK)
#define ADC_JCMR_CH61_MASK                       0x20000000u
#define ADC_JCMR_CH61_SHIFT                      29u
#define ADC_JCMR_CH61_WIDTH                      1u
#define ADC_JCMR_CH61(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH61_SHIFT))&ADC_JCMR_CH61_MASK)
#define ADC_JCMR_CH93_MASK                       0x20000000u
#define ADC_JCMR_CH93_SHIFT                      29u
#define ADC_JCMR_CH93_WIDTH                      1u
#define ADC_JCMR_CH93(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH93_SHIFT))&ADC_JCMR_CH93_MASK)
#define ADC_JCMR_CH94_MASK                       0x40000000u
#define ADC_JCMR_CH94_SHIFT                      30u
#define ADC_JCMR_CH94_WIDTH                      1u
#define ADC_JCMR_CH94(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH94_SHIFT))&ADC_JCMR_CH94_MASK)
#define ADC_JCMR_CH62_MASK                       0x40000000u
#define ADC_JCMR_CH62_SHIFT                      30u
#define ADC_JCMR_CH62_WIDTH                      1u
#define ADC_JCMR_CH62(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH62_SHIFT))&ADC_JCMR_CH62_MASK)
#define ADC_JCMR_CH95_MASK                       0x80000000u
#define ADC_JCMR_CH95_SHIFT                      31u
#define ADC_JCMR_CH95_WIDTH                      1u
#define ADC_JCMR_CH95(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH95_SHIFT))&ADC_JCMR_CH95_MASK)
#define ADC_JCMR_CH63_MASK                       0x80000000u
#define ADC_JCMR_CH63_SHIFT                      31u
#define ADC_JCMR_CH63_WIDTH                      1u
#define ADC_JCMR_CH63(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_JCMR_CH63_SHIFT))&ADC_JCMR_CH63_MASK)
/* DSDR Bit Fields */
#define ADC_DSDR_DSD_MASK                        0xFFFFu
#define ADC_DSDR_DSD_SHIFT                       0u
#define ADC_DSDR_DSD_WIDTH                       16u
#define ADC_DSDR_DSD(x)                          (((uint32_t)(((uint32_t)(x))<<ADC_DSDR_DSD_SHIFT))&ADC_DSDR_DSD_MASK)
/* PDEDR Bit Fields */
#define ADC_PDEDR_PDED_MASK                      0xFFu
#define ADC_PDEDR_PDED_SHIFT                     0u
#define ADC_PDEDR_PDED_WIDTH                     8u
#define ADC_PDEDR_PDED(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_PDEDR_PDED_SHIFT))&ADC_PDEDR_PDED_MASK)
/* CDR Bit Fields */
#define ADC_CDR_CDATA_MASK                       0xFFFFu
#define ADC_CDR_CDATA_SHIFT                      0u
#define ADC_CDR_CDATA_WIDTH                      16u
#define ADC_CDR_CDATA(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CDR_CDATA_SHIFT))&ADC_CDR_CDATA_MASK)
#define ADC_CDR_RESULT_MASK                      0x30000u
#define ADC_CDR_RESULT_SHIFT                     16u
#define ADC_CDR_RESULT_WIDTH                     2u
#define ADC_CDR_RESULT(x)                        (((uint32_t)(((uint32_t)(x))<<ADC_CDR_RESULT_SHIFT))&ADC_CDR_RESULT_MASK)
#define ADC_CDR_OVERW_MASK                       0x40000u
#define ADC_CDR_OVERW_SHIFT                      18u
#define ADC_CDR_OVERW_WIDTH                      1u
#define ADC_CDR_OVERW(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CDR_OVERW_SHIFT))&ADC_CDR_OVERW_MASK)
#define ADC_CDR_VALID_MASK                       0x80000u
#define ADC_CDR_VALID_SHIFT                      19u
#define ADC_CDR_VALID_WIDTH                      1u
#define ADC_CDR_VALID(x)                         (((uint32_t)(((uint32_t)(x))<<ADC_CDR_VALID_SHIFT))&ADC_CDR_VALID_MASK)
/* THRHLR4 Bit Fields */
#define ADC_THRHLR4_THRL_MASK                    0xFFFu
#define ADC_THRHLR4_THRL_SHIFT                   0u
#define ADC_THRHLR4_THRL_WIDTH                   12u
#define ADC_THRHLR4_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR4_THRL_SHIFT))&ADC_THRHLR4_THRL_MASK)
#define ADC_THRHLR4_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR4_THRH_SHIFT                   16u
#define ADC_THRHLR4_THRH_WIDTH                   12u
#define ADC_THRHLR4_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR4_THRH_SHIFT))&ADC_THRHLR4_THRH_MASK)
/* THRHLR5 Bit Fields */
#define ADC_THRHLR5_THRL_MASK                    0xFFFu
#define ADC_THRHLR5_THRL_SHIFT                   0u
#define ADC_THRHLR5_THRL_WIDTH                   12u
#define ADC_THRHLR5_THRL(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR5_THRL_SHIFT))&ADC_THRHLR5_THRL_MASK)
#define ADC_THRHLR5_THRH_MASK                    0xFFF0000u
#define ADC_THRHLR5_THRH_SHIFT                   16u
#define ADC_THRHLR5_THRH_WIDTH                   12u
#define ADC_THRHLR5_THRH(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_THRHLR5_THRH_SHIFT))&ADC_THRHLR5_THRH_MASK)
/* CWSELR Bit Fields */
#define ADC_CWSELR_WSEL_CH0_MASK                 0xFu
#define ADC_CWSELR_WSEL_CH0_SHIFT                0u
#define ADC_CWSELR_WSEL_CH0_WIDTH                4u
#define ADC_CWSELR_WSEL_CH0(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH0_SHIFT))&ADC_CWSELR_WSEL_CH0_MASK)
#define ADC_CWSELR_WSEL_CH88_MASK                0xFu
#define ADC_CWSELR_WSEL_CH88_SHIFT               0u
#define ADC_CWSELR_WSEL_CH88_WIDTH               4u
#define ADC_CWSELR_WSEL_CH88(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH88_SHIFT))&ADC_CWSELR_WSEL_CH88_MASK)
#define ADC_CWSELR_WSEL_CH80_MASK                0xFu
#define ADC_CWSELR_WSEL_CH80_SHIFT               0u
#define ADC_CWSELR_WSEL_CH80_WIDTH               4u
#define ADC_CWSELR_WSEL_CH80(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH80_SHIFT))&ADC_CWSELR_WSEL_CH80_MASK)
#define ADC_CWSELR_WSEL_CH72_MASK                0xFu
#define ADC_CWSELR_WSEL_CH72_SHIFT               0u
#define ADC_CWSELR_WSEL_CH72_WIDTH               4u
#define ADC_CWSELR_WSEL_CH72(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH72_SHIFT))&ADC_CWSELR_WSEL_CH72_MASK)
#define ADC_CWSELR_WSEL_CH64_MASK                0xFu
#define ADC_CWSELR_WSEL_CH64_SHIFT               0u
#define ADC_CWSELR_WSEL_CH64_WIDTH               4u
#define ADC_CWSELR_WSEL_CH64(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH64_SHIFT))&ADC_CWSELR_WSEL_CH64_MASK)
#define ADC_CWSELR_WSEL_CH56_MASK                0xFu
#define ADC_CWSELR_WSEL_CH56_SHIFT               0u
#define ADC_CWSELR_WSEL_CH56_WIDTH               4u
#define ADC_CWSELR_WSEL_CH56(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH56_SHIFT))&ADC_CWSELR_WSEL_CH56_MASK)
#define ADC_CWSELR_WSEL_CH48_MASK                0xFu
#define ADC_CWSELR_WSEL_CH48_SHIFT               0u
#define ADC_CWSELR_WSEL_CH48_WIDTH               4u
#define ADC_CWSELR_WSEL_CH48(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH48_SHIFT))&ADC_CWSELR_WSEL_CH48_MASK)
#define ADC_CWSELR_WSEL_CH40_MASK                0xFu
#define ADC_CWSELR_WSEL_CH40_SHIFT               0u
#define ADC_CWSELR_WSEL_CH40_WIDTH               4u
#define ADC_CWSELR_WSEL_CH40(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH40_SHIFT))&ADC_CWSELR_WSEL_CH40_MASK)
#define ADC_CWSELR_WSEL_CH8_MASK                 0xFu
#define ADC_CWSELR_WSEL_CH8_SHIFT                0u
#define ADC_CWSELR_WSEL_CH8_WIDTH                4u
#define ADC_CWSELR_WSEL_CH8(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH8_SHIFT))&ADC_CWSELR_WSEL_CH8_MASK)
#define ADC_CWSELR_WSEL_CH32_MASK                0xFu
#define ADC_CWSELR_WSEL_CH32_SHIFT               0u
#define ADC_CWSELR_WSEL_CH32_WIDTH               4u
#define ADC_CWSELR_WSEL_CH32(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH32_SHIFT))&ADC_CWSELR_WSEL_CH32_MASK)
#define ADC_CWSELR_WSEL_CH65_MASK                0xF0u
#define ADC_CWSELR_WSEL_CH65_SHIFT               4u
#define ADC_CWSELR_WSEL_CH65_WIDTH               4u
#define ADC_CWSELR_WSEL_CH65(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH65_SHIFT))&ADC_CWSELR_WSEL_CH65_MASK)
#define ADC_CWSELR_WSEL_CH81_MASK                0xF0u
#define ADC_CWSELR_WSEL_CH81_SHIFT               4u
#define ADC_CWSELR_WSEL_CH81_WIDTH               4u
#define ADC_CWSELR_WSEL_CH81(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH81_SHIFT))&ADC_CWSELR_WSEL_CH81_MASK)
#define ADC_CWSELR_WSEL_CH73_MASK                0xF0u
#define ADC_CWSELR_WSEL_CH73_SHIFT               4u
#define ADC_CWSELR_WSEL_CH73_WIDTH               4u
#define ADC_CWSELR_WSEL_CH73(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH73_SHIFT))&ADC_CWSELR_WSEL_CH73_MASK)
#define ADC_CWSELR_WSEL_CH57_MASK                0xF0u
#define ADC_CWSELR_WSEL_CH57_SHIFT               4u
#define ADC_CWSELR_WSEL_CH57_WIDTH               4u
#define ADC_CWSELR_WSEL_CH57(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH57_SHIFT))&ADC_CWSELR_WSEL_CH57_MASK)
#define ADC_CWSELR_WSEL_CH33_MASK                0xF0u
#define ADC_CWSELR_WSEL_CH33_SHIFT               4u
#define ADC_CWSELR_WSEL_CH33_WIDTH               4u
#define ADC_CWSELR_WSEL_CH33(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH33_SHIFT))&ADC_CWSELR_WSEL_CH33_MASK)
#define ADC_CWSELR_WSEL_CH9_MASK                 0xF0u
#define ADC_CWSELR_WSEL_CH9_SHIFT                4u
#define ADC_CWSELR_WSEL_CH9_WIDTH                4u
#define ADC_CWSELR_WSEL_CH9(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH9_SHIFT))&ADC_CWSELR_WSEL_CH9_MASK)
#define ADC_CWSELR_WSEL_CH49_MASK                0xF0u
#define ADC_CWSELR_WSEL_CH49_SHIFT               4u
#define ADC_CWSELR_WSEL_CH49_WIDTH               4u
#define ADC_CWSELR_WSEL_CH49(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH49_SHIFT))&ADC_CWSELR_WSEL_CH49_MASK)
#define ADC_CWSELR_WSEL_CH1_MASK                 0xF0u
#define ADC_CWSELR_WSEL_CH1_SHIFT                4u
#define ADC_CWSELR_WSEL_CH1_WIDTH                4u
#define ADC_CWSELR_WSEL_CH1(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH1_SHIFT))&ADC_CWSELR_WSEL_CH1_MASK)
#define ADC_CWSELR_WSEL_CH41_MASK                0xF0u
#define ADC_CWSELR_WSEL_CH41_SHIFT               4u
#define ADC_CWSELR_WSEL_CH41_WIDTH               4u
#define ADC_CWSELR_WSEL_CH41(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH41_SHIFT))&ADC_CWSELR_WSEL_CH41_MASK)
#define ADC_CWSELR_WSEL_CH89_MASK                0xF0u
#define ADC_CWSELR_WSEL_CH89_SHIFT               4u
#define ADC_CWSELR_WSEL_CH89_WIDTH               4u
#define ADC_CWSELR_WSEL_CH89(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH89_SHIFT))&ADC_CWSELR_WSEL_CH89_MASK)
#define ADC_CWSELR_WSEL_CH82_MASK                0xF00u
#define ADC_CWSELR_WSEL_CH82_SHIFT               8u
#define ADC_CWSELR_WSEL_CH82_WIDTH               4u
#define ADC_CWSELR_WSEL_CH82(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH82_SHIFT))&ADC_CWSELR_WSEL_CH82_MASK)
#define ADC_CWSELR_WSEL_CH74_MASK                0xF00u
#define ADC_CWSELR_WSEL_CH74_SHIFT               8u
#define ADC_CWSELR_WSEL_CH74_WIDTH               4u
#define ADC_CWSELR_WSEL_CH74(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH74_SHIFT))&ADC_CWSELR_WSEL_CH74_MASK)
#define ADC_CWSELR_WSEL_CH2_MASK                 0xF00u
#define ADC_CWSELR_WSEL_CH2_SHIFT                8u
#define ADC_CWSELR_WSEL_CH2_WIDTH                4u
#define ADC_CWSELR_WSEL_CH2(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH2_SHIFT))&ADC_CWSELR_WSEL_CH2_MASK)
#define ADC_CWSELR_WSEL_CH66_MASK                0xF00u
#define ADC_CWSELR_WSEL_CH66_SHIFT               8u
#define ADC_CWSELR_WSEL_CH66_WIDTH               4u
#define ADC_CWSELR_WSEL_CH66(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH66_SHIFT))&ADC_CWSELR_WSEL_CH66_MASK)
#define ADC_CWSELR_WSEL_CH42_MASK                0xF00u
#define ADC_CWSELR_WSEL_CH42_SHIFT               8u
#define ADC_CWSELR_WSEL_CH42_WIDTH               4u
#define ADC_CWSELR_WSEL_CH42(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH42_SHIFT))&ADC_CWSELR_WSEL_CH42_MASK)
#define ADC_CWSELR_WSEL_CH50_MASK                0xF00u
#define ADC_CWSELR_WSEL_CH50_SHIFT               8u
#define ADC_CWSELR_WSEL_CH50_WIDTH               4u
#define ADC_CWSELR_WSEL_CH50(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH50_SHIFT))&ADC_CWSELR_WSEL_CH50_MASK)
#define ADC_CWSELR_WSEL_CH34_MASK                0xF00u
#define ADC_CWSELR_WSEL_CH34_SHIFT               8u
#define ADC_CWSELR_WSEL_CH34_WIDTH               4u
#define ADC_CWSELR_WSEL_CH34(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH34_SHIFT))&ADC_CWSELR_WSEL_CH34_MASK)
#define ADC_CWSELR_WSEL_CH58_MASK                0xF00u
#define ADC_CWSELR_WSEL_CH58_SHIFT               8u
#define ADC_CWSELR_WSEL_CH58_WIDTH               4u
#define ADC_CWSELR_WSEL_CH58(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH58_SHIFT))&ADC_CWSELR_WSEL_CH58_MASK)
#define ADC_CWSELR_WSEL_CH90_MASK                0xF00u
#define ADC_CWSELR_WSEL_CH90_SHIFT               8u
#define ADC_CWSELR_WSEL_CH90_WIDTH               4u
#define ADC_CWSELR_WSEL_CH90(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH90_SHIFT))&ADC_CWSELR_WSEL_CH90_MASK)
#define ADC_CWSELR_WSEL_CH10_MASK                0xF00u
#define ADC_CWSELR_WSEL_CH10_SHIFT               8u
#define ADC_CWSELR_WSEL_CH10_WIDTH               4u
#define ADC_CWSELR_WSEL_CH10(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH10_SHIFT))&ADC_CWSELR_WSEL_CH10_MASK)
#define ADC_CWSELR_WSEL_CH83_MASK                0xF000u
#define ADC_CWSELR_WSEL_CH83_SHIFT               12u
#define ADC_CWSELR_WSEL_CH83_WIDTH               4u
#define ADC_CWSELR_WSEL_CH83(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH83_SHIFT))&ADC_CWSELR_WSEL_CH83_MASK)
#define ADC_CWSELR_WSEL_CH35_MASK                0xF000u
#define ADC_CWSELR_WSEL_CH35_SHIFT               12u
#define ADC_CWSELR_WSEL_CH35_WIDTH               4u
#define ADC_CWSELR_WSEL_CH35(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH35_SHIFT))&ADC_CWSELR_WSEL_CH35_MASK)
#define ADC_CWSELR_WSEL_CH3_MASK                 0xF000u
#define ADC_CWSELR_WSEL_CH3_SHIFT                12u
#define ADC_CWSELR_WSEL_CH3_WIDTH                4u
#define ADC_CWSELR_WSEL_CH3(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH3_SHIFT))&ADC_CWSELR_WSEL_CH3_MASK)
#define ADC_CWSELR_WSEL_CH51_MASK                0xF000u
#define ADC_CWSELR_WSEL_CH51_SHIFT               12u
#define ADC_CWSELR_WSEL_CH51_WIDTH               4u
#define ADC_CWSELR_WSEL_CH51(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH51_SHIFT))&ADC_CWSELR_WSEL_CH51_MASK)
#define ADC_CWSELR_WSEL_CH59_MASK                0xF000u
#define ADC_CWSELR_WSEL_CH59_SHIFT               12u
#define ADC_CWSELR_WSEL_CH59_WIDTH               4u
#define ADC_CWSELR_WSEL_CH59(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH59_SHIFT))&ADC_CWSELR_WSEL_CH59_MASK)
#define ADC_CWSELR_WSEL_CH43_MASK                0xF000u
#define ADC_CWSELR_WSEL_CH43_SHIFT               12u
#define ADC_CWSELR_WSEL_CH43_WIDTH               4u
#define ADC_CWSELR_WSEL_CH43(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH43_SHIFT))&ADC_CWSELR_WSEL_CH43_MASK)
#define ADC_CWSELR_WSEL_CH67_MASK                0xF000u
#define ADC_CWSELR_WSEL_CH67_SHIFT               12u
#define ADC_CWSELR_WSEL_CH67_WIDTH               4u
#define ADC_CWSELR_WSEL_CH67(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH67_SHIFT))&ADC_CWSELR_WSEL_CH67_MASK)
#define ADC_CWSELR_WSEL_CH91_MASK                0xF000u
#define ADC_CWSELR_WSEL_CH91_SHIFT               12u
#define ADC_CWSELR_WSEL_CH91_WIDTH               4u
#define ADC_CWSELR_WSEL_CH91(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH91_SHIFT))&ADC_CWSELR_WSEL_CH91_MASK)
#define ADC_CWSELR_WSEL_CH11_MASK                0xF000u
#define ADC_CWSELR_WSEL_CH11_SHIFT               12u
#define ADC_CWSELR_WSEL_CH11_WIDTH               4u
#define ADC_CWSELR_WSEL_CH11(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH11_SHIFT))&ADC_CWSELR_WSEL_CH11_MASK)
#define ADC_CWSELR_WSEL_CH75_MASK                0xF000u
#define ADC_CWSELR_WSEL_CH75_SHIFT               12u
#define ADC_CWSELR_WSEL_CH75_WIDTH               4u
#define ADC_CWSELR_WSEL_CH75(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH75_SHIFT))&ADC_CWSELR_WSEL_CH75_MASK)
#define ADC_CWSELR_WSEL_CH76_MASK                0xF0000u
#define ADC_CWSELR_WSEL_CH76_SHIFT               16u
#define ADC_CWSELR_WSEL_CH76_WIDTH               4u
#define ADC_CWSELR_WSEL_CH76(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH76_SHIFT))&ADC_CWSELR_WSEL_CH76_MASK)
#define ADC_CWSELR_WSEL_CH52_MASK                0xF0000u
#define ADC_CWSELR_WSEL_CH52_SHIFT               16u
#define ADC_CWSELR_WSEL_CH52_WIDTH               4u
#define ADC_CWSELR_WSEL_CH52(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH52_SHIFT))&ADC_CWSELR_WSEL_CH52_MASK)
#define ADC_CWSELR_WSEL_CH84_MASK                0xF0000u
#define ADC_CWSELR_WSEL_CH84_SHIFT               16u
#define ADC_CWSELR_WSEL_CH84_WIDTH               4u
#define ADC_CWSELR_WSEL_CH84(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH84_SHIFT))&ADC_CWSELR_WSEL_CH84_MASK)
#define ADC_CWSELR_WSEL_CH92_MASK                0xF0000u
#define ADC_CWSELR_WSEL_CH92_SHIFT               16u
#define ADC_CWSELR_WSEL_CH92_WIDTH               4u
#define ADC_CWSELR_WSEL_CH92(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH92_SHIFT))&ADC_CWSELR_WSEL_CH92_MASK)
#define ADC_CWSELR_WSEL_CH60_MASK                0xF0000u
#define ADC_CWSELR_WSEL_CH60_SHIFT               16u
#define ADC_CWSELR_WSEL_CH60_WIDTH               4u
#define ADC_CWSELR_WSEL_CH60(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH60_SHIFT))&ADC_CWSELR_WSEL_CH60_MASK)
#define ADC_CWSELR_WSEL_CH68_MASK                0xF0000u
#define ADC_CWSELR_WSEL_CH68_SHIFT               16u
#define ADC_CWSELR_WSEL_CH68_WIDTH               4u
#define ADC_CWSELR_WSEL_CH68(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH68_SHIFT))&ADC_CWSELR_WSEL_CH68_MASK)
#define ADC_CWSELR_WSEL_CH12_MASK                0xF0000u
#define ADC_CWSELR_WSEL_CH12_SHIFT               16u
#define ADC_CWSELR_WSEL_CH12_WIDTH               4u
#define ADC_CWSELR_WSEL_CH12(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH12_SHIFT))&ADC_CWSELR_WSEL_CH12_MASK)
#define ADC_CWSELR_WSEL_CH44_MASK                0xF0000u
#define ADC_CWSELR_WSEL_CH44_SHIFT               16u
#define ADC_CWSELR_WSEL_CH44_WIDTH               4u
#define ADC_CWSELR_WSEL_CH44(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH44_SHIFT))&ADC_CWSELR_WSEL_CH44_MASK)
#define ADC_CWSELR_WSEL_CH4_MASK                 0xF0000u
#define ADC_CWSELR_WSEL_CH4_SHIFT                16u
#define ADC_CWSELR_WSEL_CH4_WIDTH                4u
#define ADC_CWSELR_WSEL_CH4(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH4_SHIFT))&ADC_CWSELR_WSEL_CH4_MASK)
#define ADC_CWSELR_WSEL_CH36_MASK                0xF0000u
#define ADC_CWSELR_WSEL_CH36_SHIFT               16u
#define ADC_CWSELR_WSEL_CH36_WIDTH               4u
#define ADC_CWSELR_WSEL_CH36(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH36_SHIFT))&ADC_CWSELR_WSEL_CH36_MASK)
#define ADC_CWSELR_WSEL_CH77_MASK                0xF00000u
#define ADC_CWSELR_WSEL_CH77_SHIFT               20u
#define ADC_CWSELR_WSEL_CH77_WIDTH               4u
#define ADC_CWSELR_WSEL_CH77(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH77_SHIFT))&ADC_CWSELR_WSEL_CH77_MASK)
#define ADC_CWSELR_WSEL_CH69_MASK                0xF00000u
#define ADC_CWSELR_WSEL_CH69_SHIFT               20u
#define ADC_CWSELR_WSEL_CH69_WIDTH               4u
#define ADC_CWSELR_WSEL_CH69(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH69_SHIFT))&ADC_CWSELR_WSEL_CH69_MASK)
#define ADC_CWSELR_WSEL_CH5_MASK                 0xF00000u
#define ADC_CWSELR_WSEL_CH5_SHIFT                20u
#define ADC_CWSELR_WSEL_CH5_WIDTH                4u
#define ADC_CWSELR_WSEL_CH5(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH5_SHIFT))&ADC_CWSELR_WSEL_CH5_MASK)
#define ADC_CWSELR_WSEL_CH53_MASK                0xF00000u
#define ADC_CWSELR_WSEL_CH53_SHIFT               20u
#define ADC_CWSELR_WSEL_CH53_WIDTH               4u
#define ADC_CWSELR_WSEL_CH53(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH53_SHIFT))&ADC_CWSELR_WSEL_CH53_MASK)
#define ADC_CWSELR_WSEL_CH85_MASK                0xF00000u
#define ADC_CWSELR_WSEL_CH85_SHIFT               20u
#define ADC_CWSELR_WSEL_CH85_WIDTH               4u
#define ADC_CWSELR_WSEL_CH85(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH85_SHIFT))&ADC_CWSELR_WSEL_CH85_MASK)
#define ADC_CWSELR_WSEL_CH45_MASK                0xF00000u
#define ADC_CWSELR_WSEL_CH45_SHIFT               20u
#define ADC_CWSELR_WSEL_CH45_WIDTH               4u
#define ADC_CWSELR_WSEL_CH45(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH45_SHIFT))&ADC_CWSELR_WSEL_CH45_MASK)
#define ADC_CWSELR_WSEL_CH37_MASK                0xF00000u
#define ADC_CWSELR_WSEL_CH37_SHIFT               20u
#define ADC_CWSELR_WSEL_CH37_WIDTH               4u
#define ADC_CWSELR_WSEL_CH37(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH37_SHIFT))&ADC_CWSELR_WSEL_CH37_MASK)
#define ADC_CWSELR_WSEL_CH93_MASK                0xF00000u
#define ADC_CWSELR_WSEL_CH93_SHIFT               20u
#define ADC_CWSELR_WSEL_CH93_WIDTH               4u
#define ADC_CWSELR_WSEL_CH93(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH93_SHIFT))&ADC_CWSELR_WSEL_CH93_MASK)
#define ADC_CWSELR_WSEL_CH13_MASK                0xF00000u
#define ADC_CWSELR_WSEL_CH13_SHIFT               20u
#define ADC_CWSELR_WSEL_CH13_WIDTH               4u
#define ADC_CWSELR_WSEL_CH13(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH13_SHIFT))&ADC_CWSELR_WSEL_CH13_MASK)
#define ADC_CWSELR_WSEL_CH61_MASK                0xF00000u
#define ADC_CWSELR_WSEL_CH61_SHIFT               20u
#define ADC_CWSELR_WSEL_CH61_WIDTH               4u
#define ADC_CWSELR_WSEL_CH61(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH61_SHIFT))&ADC_CWSELR_WSEL_CH61_MASK)
#define ADC_CWSELR_WSEL_CH78_MASK                0xF000000u
#define ADC_CWSELR_WSEL_CH78_SHIFT               24u
#define ADC_CWSELR_WSEL_CH78_WIDTH               4u
#define ADC_CWSELR_WSEL_CH78(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH78_SHIFT))&ADC_CWSELR_WSEL_CH78_MASK)
#define ADC_CWSELR_WSEL_CH94_MASK                0xF000000u
#define ADC_CWSELR_WSEL_CH94_SHIFT               24u
#define ADC_CWSELR_WSEL_CH94_WIDTH               4u
#define ADC_CWSELR_WSEL_CH94(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH94_SHIFT))&ADC_CWSELR_WSEL_CH94_MASK)
#define ADC_CWSELR_WSEL_CH14_MASK                0xF000000u
#define ADC_CWSELR_WSEL_CH14_SHIFT               24u
#define ADC_CWSELR_WSEL_CH14_WIDTH               4u
#define ADC_CWSELR_WSEL_CH14(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH14_SHIFT))&ADC_CWSELR_WSEL_CH14_MASK)
#define ADC_CWSELR_WSEL_CH6_MASK                 0xF000000u
#define ADC_CWSELR_WSEL_CH6_SHIFT                24u
#define ADC_CWSELR_WSEL_CH6_WIDTH                4u
#define ADC_CWSELR_WSEL_CH6(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH6_SHIFT))&ADC_CWSELR_WSEL_CH6_MASK)
#define ADC_CWSELR_WSEL_CH38_MASK                0xF000000u
#define ADC_CWSELR_WSEL_CH38_SHIFT               24u
#define ADC_CWSELR_WSEL_CH38_WIDTH               4u
#define ADC_CWSELR_WSEL_CH38(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH38_SHIFT))&ADC_CWSELR_WSEL_CH38_MASK)
#define ADC_CWSELR_WSEL_CH54_MASK                0xF000000u
#define ADC_CWSELR_WSEL_CH54_SHIFT               24u
#define ADC_CWSELR_WSEL_CH54_WIDTH               4u
#define ADC_CWSELR_WSEL_CH54(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH54_SHIFT))&ADC_CWSELR_WSEL_CH54_MASK)
#define ADC_CWSELR_WSEL_CH70_MASK                0xF000000u
#define ADC_CWSELR_WSEL_CH70_SHIFT               24u
#define ADC_CWSELR_WSEL_CH70_WIDTH               4u
#define ADC_CWSELR_WSEL_CH70(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH70_SHIFT))&ADC_CWSELR_WSEL_CH70_MASK)
#define ADC_CWSELR_WSEL_CH46_MASK                0xF000000u
#define ADC_CWSELR_WSEL_CH46_SHIFT               24u
#define ADC_CWSELR_WSEL_CH46_WIDTH               4u
#define ADC_CWSELR_WSEL_CH46(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH46_SHIFT))&ADC_CWSELR_WSEL_CH46_MASK)
#define ADC_CWSELR_WSEL_CH62_MASK                0xF000000u
#define ADC_CWSELR_WSEL_CH62_SHIFT               24u
#define ADC_CWSELR_WSEL_CH62_WIDTH               4u
#define ADC_CWSELR_WSEL_CH62(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH62_SHIFT))&ADC_CWSELR_WSEL_CH62_MASK)
#define ADC_CWSELR_WSEL_CH86_MASK                0xF000000u
#define ADC_CWSELR_WSEL_CH86_SHIFT               24u
#define ADC_CWSELR_WSEL_CH86_WIDTH               4u
#define ADC_CWSELR_WSEL_CH86(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH86_SHIFT))&ADC_CWSELR_WSEL_CH86_MASK)
#define ADC_CWSELR_WSEL_CH95_MASK                0xF0000000u
#define ADC_CWSELR_WSEL_CH95_SHIFT               28u
#define ADC_CWSELR_WSEL_CH95_WIDTH               4u
#define ADC_CWSELR_WSEL_CH95(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH95_SHIFT))&ADC_CWSELR_WSEL_CH95_MASK)
#define ADC_CWSELR_WSEL_CH15_MASK                0xF0000000u
#define ADC_CWSELR_WSEL_CH15_SHIFT               28u
#define ADC_CWSELR_WSEL_CH15_WIDTH               4u
#define ADC_CWSELR_WSEL_CH15(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH15_SHIFT))&ADC_CWSELR_WSEL_CH15_MASK)
#define ADC_CWSELR_WSEL_CH39_MASK                0xF0000000u
#define ADC_CWSELR_WSEL_CH39_SHIFT               28u
#define ADC_CWSELR_WSEL_CH39_WIDTH               4u
#define ADC_CWSELR_WSEL_CH39(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH39_SHIFT))&ADC_CWSELR_WSEL_CH39_MASK)
#define ADC_CWSELR_WSEL_CH7_MASK                 0xF0000000u
#define ADC_CWSELR_WSEL_CH7_SHIFT                28u
#define ADC_CWSELR_WSEL_CH7_WIDTH                4u
#define ADC_CWSELR_WSEL_CH7(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH7_SHIFT))&ADC_CWSELR_WSEL_CH7_MASK)
#define ADC_CWSELR_WSEL_CH87_MASK                0xF0000000u
#define ADC_CWSELR_WSEL_CH87_SHIFT               28u
#define ADC_CWSELR_WSEL_CH87_WIDTH               4u
#define ADC_CWSELR_WSEL_CH87(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH87_SHIFT))&ADC_CWSELR_WSEL_CH87_MASK)
#define ADC_CWSELR_WSEL_CH47_MASK                0xF0000000u
#define ADC_CWSELR_WSEL_CH47_SHIFT               28u
#define ADC_CWSELR_WSEL_CH47_WIDTH               4u
#define ADC_CWSELR_WSEL_CH47(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH47_SHIFT))&ADC_CWSELR_WSEL_CH47_MASK)
#define ADC_CWSELR_WSEL_CH55_MASK                0xF0000000u
#define ADC_CWSELR_WSEL_CH55_SHIFT               28u
#define ADC_CWSELR_WSEL_CH55_WIDTH               4u
#define ADC_CWSELR_WSEL_CH55(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH55_SHIFT))&ADC_CWSELR_WSEL_CH55_MASK)
#define ADC_CWSELR_WSEL_CH63_MASK                0xF0000000u
#define ADC_CWSELR_WSEL_CH63_SHIFT               28u
#define ADC_CWSELR_WSEL_CH63_WIDTH               4u
#define ADC_CWSELR_WSEL_CH63(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH63_SHIFT))&ADC_CWSELR_WSEL_CH63_MASK)
#define ADC_CWSELR_WSEL_CH79_MASK                0xF0000000u
#define ADC_CWSELR_WSEL_CH79_SHIFT               28u
#define ADC_CWSELR_WSEL_CH79_WIDTH               4u
#define ADC_CWSELR_WSEL_CH79(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH79_SHIFT))&ADC_CWSELR_WSEL_CH79_MASK)
#define ADC_CWSELR_WSEL_CH71_MASK                0xF0000000u
#define ADC_CWSELR_WSEL_CH71_SHIFT               28u
#define ADC_CWSELR_WSEL_CH71_WIDTH               4u
#define ADC_CWSELR_WSEL_CH71(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CWSELR_WSEL_CH71_SHIFT))&ADC_CWSELR_WSEL_CH71_MASK)
/* CWENR Bit Fields */
#define ADC_CWENR_CWEN0_MASK                     0x1u
#define ADC_CWENR_CWEN0_SHIFT                    0u
#define ADC_CWENR_CWEN0_WIDTH                    1u
#define ADC_CWENR_CWEN0(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN0_SHIFT))&ADC_CWENR_CWEN0_MASK)
#define ADC_CWENR_CWEN64_MASK                    0x1u
#define ADC_CWENR_CWEN64_SHIFT                   0u
#define ADC_CWENR_CWEN64_WIDTH                   1u
#define ADC_CWENR_CWEN64(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN64_SHIFT))&ADC_CWENR_CWEN64_MASK)
#define ADC_CWENR_CWEN32_MASK                    0x1u
#define ADC_CWENR_CWEN32_SHIFT                   0u
#define ADC_CWENR_CWEN32_WIDTH                   1u
#define ADC_CWENR_CWEN32(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN32_SHIFT))&ADC_CWENR_CWEN32_MASK)
#define ADC_CWENR_CWEN65_MASK                    0x2u
#define ADC_CWENR_CWEN65_SHIFT                   1u
#define ADC_CWENR_CWEN65_WIDTH                   1u
#define ADC_CWENR_CWEN65(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN65_SHIFT))&ADC_CWENR_CWEN65_MASK)
#define ADC_CWENR_CWEN33_MASK                    0x2u
#define ADC_CWENR_CWEN33_SHIFT                   1u
#define ADC_CWENR_CWEN33_WIDTH                   1u
#define ADC_CWENR_CWEN33(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN33_SHIFT))&ADC_CWENR_CWEN33_MASK)
#define ADC_CWENR_CWEN1_MASK                     0x2u
#define ADC_CWENR_CWEN1_SHIFT                    1u
#define ADC_CWENR_CWEN1_WIDTH                    1u
#define ADC_CWENR_CWEN1(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN1_SHIFT))&ADC_CWENR_CWEN1_MASK)
#define ADC_CWENR_CWEN34_MASK                    0x4u
#define ADC_CWENR_CWEN34_SHIFT                   2u
#define ADC_CWENR_CWEN34_WIDTH                   1u
#define ADC_CWENR_CWEN34(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN34_SHIFT))&ADC_CWENR_CWEN34_MASK)
#define ADC_CWENR_CWEN2_MASK                     0x4u
#define ADC_CWENR_CWEN2_SHIFT                    2u
#define ADC_CWENR_CWEN2_WIDTH                    1u
#define ADC_CWENR_CWEN2(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN2_SHIFT))&ADC_CWENR_CWEN2_MASK)
#define ADC_CWENR_CWEN66_MASK                    0x4u
#define ADC_CWENR_CWEN66_SHIFT                   2u
#define ADC_CWENR_CWEN66_WIDTH                   1u
#define ADC_CWENR_CWEN66(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN66_SHIFT))&ADC_CWENR_CWEN66_MASK)
#define ADC_CWENR_CWEN67_MASK                    0x8u
#define ADC_CWENR_CWEN67_SHIFT                   3u
#define ADC_CWENR_CWEN67_WIDTH                   1u
#define ADC_CWENR_CWEN67(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN67_SHIFT))&ADC_CWENR_CWEN67_MASK)
#define ADC_CWENR_CWEN3_MASK                     0x8u
#define ADC_CWENR_CWEN3_SHIFT                    3u
#define ADC_CWENR_CWEN3_WIDTH                    1u
#define ADC_CWENR_CWEN3(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN3_SHIFT))&ADC_CWENR_CWEN3_MASK)
#define ADC_CWENR_CWEN35_MASK                    0x8u
#define ADC_CWENR_CWEN35_SHIFT                   3u
#define ADC_CWENR_CWEN35_WIDTH                   1u
#define ADC_CWENR_CWEN35(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN35_SHIFT))&ADC_CWENR_CWEN35_MASK)
#define ADC_CWENR_CWEN4_MASK                     0x10u
#define ADC_CWENR_CWEN4_SHIFT                    4u
#define ADC_CWENR_CWEN4_WIDTH                    1u
#define ADC_CWENR_CWEN4(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN4_SHIFT))&ADC_CWENR_CWEN4_MASK)
#define ADC_CWENR_CWEN36_MASK                    0x10u
#define ADC_CWENR_CWEN36_SHIFT                   4u
#define ADC_CWENR_CWEN36_WIDTH                   1u
#define ADC_CWENR_CWEN36(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN36_SHIFT))&ADC_CWENR_CWEN36_MASK)
#define ADC_CWENR_CWEN68_MASK                    0x10u
#define ADC_CWENR_CWEN68_SHIFT                   4u
#define ADC_CWENR_CWEN68_WIDTH                   1u
#define ADC_CWENR_CWEN68(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN68_SHIFT))&ADC_CWENR_CWEN68_MASK)
#define ADC_CWENR_CWEN69_MASK                    0x20u
#define ADC_CWENR_CWEN69_SHIFT                   5u
#define ADC_CWENR_CWEN69_WIDTH                   1u
#define ADC_CWENR_CWEN69(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN69_SHIFT))&ADC_CWENR_CWEN69_MASK)
#define ADC_CWENR_CWEN37_MASK                    0x20u
#define ADC_CWENR_CWEN37_SHIFT                   5u
#define ADC_CWENR_CWEN37_WIDTH                   1u
#define ADC_CWENR_CWEN37(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN37_SHIFT))&ADC_CWENR_CWEN37_MASK)
#define ADC_CWENR_CWEN5_MASK                     0x20u
#define ADC_CWENR_CWEN5_SHIFT                    5u
#define ADC_CWENR_CWEN5_WIDTH                    1u
#define ADC_CWENR_CWEN5(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN5_SHIFT))&ADC_CWENR_CWEN5_MASK)
#define ADC_CWENR_CWEN70_MASK                    0x40u
#define ADC_CWENR_CWEN70_SHIFT                   6u
#define ADC_CWENR_CWEN70_WIDTH                   1u
#define ADC_CWENR_CWEN70(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN70_SHIFT))&ADC_CWENR_CWEN70_MASK)
#define ADC_CWENR_CWEN38_MASK                    0x40u
#define ADC_CWENR_CWEN38_SHIFT                   6u
#define ADC_CWENR_CWEN38_WIDTH                   1u
#define ADC_CWENR_CWEN38(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN38_SHIFT))&ADC_CWENR_CWEN38_MASK)
#define ADC_CWENR_CWEN6_MASK                     0x40u
#define ADC_CWENR_CWEN6_SHIFT                    6u
#define ADC_CWENR_CWEN6_WIDTH                    1u
#define ADC_CWENR_CWEN6(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN6_SHIFT))&ADC_CWENR_CWEN6_MASK)
#define ADC_CWENR_CWEN7_MASK                     0x80u
#define ADC_CWENR_CWEN7_SHIFT                    7u
#define ADC_CWENR_CWEN7_WIDTH                    1u
#define ADC_CWENR_CWEN7(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN7_SHIFT))&ADC_CWENR_CWEN7_MASK)
#define ADC_CWENR_CWEN39_MASK                    0x80u
#define ADC_CWENR_CWEN39_SHIFT                   7u
#define ADC_CWENR_CWEN39_WIDTH                   1u
#define ADC_CWENR_CWEN39(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN39_SHIFT))&ADC_CWENR_CWEN39_MASK)
#define ADC_CWENR_CWEN71_MASK                    0x80u
#define ADC_CWENR_CWEN71_SHIFT                   7u
#define ADC_CWENR_CWEN71_WIDTH                   1u
#define ADC_CWENR_CWEN71(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN71_SHIFT))&ADC_CWENR_CWEN71_MASK)
#define ADC_CWENR_CWEN8_MASK                     0x100u
#define ADC_CWENR_CWEN8_SHIFT                    8u
#define ADC_CWENR_CWEN8_WIDTH                    1u
#define ADC_CWENR_CWEN8(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN8_SHIFT))&ADC_CWENR_CWEN8_MASK)
#define ADC_CWENR_CWEN40_MASK                    0x100u
#define ADC_CWENR_CWEN40_SHIFT                   8u
#define ADC_CWENR_CWEN40_WIDTH                   1u
#define ADC_CWENR_CWEN40(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN40_SHIFT))&ADC_CWENR_CWEN40_MASK)
#define ADC_CWENR_CWEN72_MASK                    0x100u
#define ADC_CWENR_CWEN72_SHIFT                   8u
#define ADC_CWENR_CWEN72_WIDTH                   1u
#define ADC_CWENR_CWEN72(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN72_SHIFT))&ADC_CWENR_CWEN72_MASK)
#define ADC_CWENR_CWEN73_MASK                    0x200u
#define ADC_CWENR_CWEN73_SHIFT                   9u
#define ADC_CWENR_CWEN73_WIDTH                   1u
#define ADC_CWENR_CWEN73(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN73_SHIFT))&ADC_CWENR_CWEN73_MASK)
#define ADC_CWENR_CWEN41_MASK                    0x200u
#define ADC_CWENR_CWEN41_SHIFT                   9u
#define ADC_CWENR_CWEN41_WIDTH                   1u
#define ADC_CWENR_CWEN41(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN41_SHIFT))&ADC_CWENR_CWEN41_MASK)
#define ADC_CWENR_CWEN9_MASK                     0x200u
#define ADC_CWENR_CWEN9_SHIFT                    9u
#define ADC_CWENR_CWEN9_WIDTH                    1u
#define ADC_CWENR_CWEN9(x)                       (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN9_SHIFT))&ADC_CWENR_CWEN9_MASK)
#define ADC_CWENR_CWEN42_MASK                    0x400u
#define ADC_CWENR_CWEN42_SHIFT                   10u
#define ADC_CWENR_CWEN42_WIDTH                   1u
#define ADC_CWENR_CWEN42(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN42_SHIFT))&ADC_CWENR_CWEN42_MASK)
#define ADC_CWENR_CWEN74_MASK                    0x400u
#define ADC_CWENR_CWEN74_SHIFT                   10u
#define ADC_CWENR_CWEN74_WIDTH                   1u
#define ADC_CWENR_CWEN74(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN74_SHIFT))&ADC_CWENR_CWEN74_MASK)
#define ADC_CWENR_CWEN10_MASK                    0x400u
#define ADC_CWENR_CWEN10_SHIFT                   10u
#define ADC_CWENR_CWEN10_WIDTH                   1u
#define ADC_CWENR_CWEN10(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN10_SHIFT))&ADC_CWENR_CWEN10_MASK)
#define ADC_CWENR_CWEN11_MASK                    0x800u
#define ADC_CWENR_CWEN11_SHIFT                   11u
#define ADC_CWENR_CWEN11_WIDTH                   1u
#define ADC_CWENR_CWEN11(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN11_SHIFT))&ADC_CWENR_CWEN11_MASK)
#define ADC_CWENR_CWEN75_MASK                    0x800u
#define ADC_CWENR_CWEN75_SHIFT                   11u
#define ADC_CWENR_CWEN75_WIDTH                   1u
#define ADC_CWENR_CWEN75(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN75_SHIFT))&ADC_CWENR_CWEN75_MASK)
#define ADC_CWENR_CWEN43_MASK                    0x800u
#define ADC_CWENR_CWEN43_SHIFT                   11u
#define ADC_CWENR_CWEN43_WIDTH                   1u
#define ADC_CWENR_CWEN43(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN43_SHIFT))&ADC_CWENR_CWEN43_MASK)
#define ADC_CWENR_CWEN76_MASK                    0x1000u
#define ADC_CWENR_CWEN76_SHIFT                   12u
#define ADC_CWENR_CWEN76_WIDTH                   1u
#define ADC_CWENR_CWEN76(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN76_SHIFT))&ADC_CWENR_CWEN76_MASK)
#define ADC_CWENR_CWEN12_MASK                    0x1000u
#define ADC_CWENR_CWEN12_SHIFT                   12u
#define ADC_CWENR_CWEN12_WIDTH                   1u
#define ADC_CWENR_CWEN12(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN12_SHIFT))&ADC_CWENR_CWEN12_MASK)
#define ADC_CWENR_CWEN44_MASK                    0x1000u
#define ADC_CWENR_CWEN44_SHIFT                   12u
#define ADC_CWENR_CWEN44_WIDTH                   1u
#define ADC_CWENR_CWEN44(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN44_SHIFT))&ADC_CWENR_CWEN44_MASK)
#define ADC_CWENR_CWEN45_MASK                    0x2000u
#define ADC_CWENR_CWEN45_SHIFT                   13u
#define ADC_CWENR_CWEN45_WIDTH                   1u
#define ADC_CWENR_CWEN45(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN45_SHIFT))&ADC_CWENR_CWEN45_MASK)
#define ADC_CWENR_CWEN77_MASK                    0x2000u
#define ADC_CWENR_CWEN77_SHIFT                   13u
#define ADC_CWENR_CWEN77_WIDTH                   1u
#define ADC_CWENR_CWEN77(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN77_SHIFT))&ADC_CWENR_CWEN77_MASK)
#define ADC_CWENR_CWEN13_MASK                    0x2000u
#define ADC_CWENR_CWEN13_SHIFT                   13u
#define ADC_CWENR_CWEN13_WIDTH                   1u
#define ADC_CWENR_CWEN13(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN13_SHIFT))&ADC_CWENR_CWEN13_MASK)
#define ADC_CWENR_CWEN14_MASK                    0x4000u
#define ADC_CWENR_CWEN14_SHIFT                   14u
#define ADC_CWENR_CWEN14_WIDTH                   1u
#define ADC_CWENR_CWEN14(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN14_SHIFT))&ADC_CWENR_CWEN14_MASK)
#define ADC_CWENR_CWEN46_MASK                    0x4000u
#define ADC_CWENR_CWEN46_SHIFT                   14u
#define ADC_CWENR_CWEN46_WIDTH                   1u
#define ADC_CWENR_CWEN46(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN46_SHIFT))&ADC_CWENR_CWEN46_MASK)
#define ADC_CWENR_CWEN78_MASK                    0x4000u
#define ADC_CWENR_CWEN78_SHIFT                   14u
#define ADC_CWENR_CWEN78_WIDTH                   1u
#define ADC_CWENR_CWEN78(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN78_SHIFT))&ADC_CWENR_CWEN78_MASK)
#define ADC_CWENR_CWEN47_MASK                    0x8000u
#define ADC_CWENR_CWEN47_SHIFT                   15u
#define ADC_CWENR_CWEN47_WIDTH                   1u
#define ADC_CWENR_CWEN47(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN47_SHIFT))&ADC_CWENR_CWEN47_MASK)
#define ADC_CWENR_CWEN15_MASK                    0x8000u
#define ADC_CWENR_CWEN15_SHIFT                   15u
#define ADC_CWENR_CWEN15_WIDTH                   1u
#define ADC_CWENR_CWEN15(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN15_SHIFT))&ADC_CWENR_CWEN15_MASK)
#define ADC_CWENR_CWEN79_MASK                    0x8000u
#define ADC_CWENR_CWEN79_SHIFT                   15u
#define ADC_CWENR_CWEN79_WIDTH                   1u
#define ADC_CWENR_CWEN79(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN79_SHIFT))&ADC_CWENR_CWEN79_MASK)
#define ADC_CWENR_CWEN80_MASK                    0x10000u
#define ADC_CWENR_CWEN80_SHIFT                   16u
#define ADC_CWENR_CWEN80_WIDTH                   1u
#define ADC_CWENR_CWEN80(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN80_SHIFT))&ADC_CWENR_CWEN80_MASK)
#define ADC_CWENR_CWEN48_MASK                    0x10000u
#define ADC_CWENR_CWEN48_SHIFT                   16u
#define ADC_CWENR_CWEN48_WIDTH                   1u
#define ADC_CWENR_CWEN48(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN48_SHIFT))&ADC_CWENR_CWEN48_MASK)
#define ADC_CWENR_CWEN81_MASK                    0x20000u
#define ADC_CWENR_CWEN81_SHIFT                   17u
#define ADC_CWENR_CWEN81_WIDTH                   1u
#define ADC_CWENR_CWEN81(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN81_SHIFT))&ADC_CWENR_CWEN81_MASK)
#define ADC_CWENR_CWEN49_MASK                    0x20000u
#define ADC_CWENR_CWEN49_SHIFT                   17u
#define ADC_CWENR_CWEN49_WIDTH                   1u
#define ADC_CWENR_CWEN49(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN49_SHIFT))&ADC_CWENR_CWEN49_MASK)
#define ADC_CWENR_CWEN50_MASK                    0x40000u
#define ADC_CWENR_CWEN50_SHIFT                   18u
#define ADC_CWENR_CWEN50_WIDTH                   1u
#define ADC_CWENR_CWEN50(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN50_SHIFT))&ADC_CWENR_CWEN50_MASK)
#define ADC_CWENR_CWEN82_MASK                    0x40000u
#define ADC_CWENR_CWEN82_SHIFT                   18u
#define ADC_CWENR_CWEN82_WIDTH                   1u
#define ADC_CWENR_CWEN82(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN82_SHIFT))&ADC_CWENR_CWEN82_MASK)
#define ADC_CWENR_CWEN51_MASK                    0x80000u
#define ADC_CWENR_CWEN51_SHIFT                   19u
#define ADC_CWENR_CWEN51_WIDTH                   1u
#define ADC_CWENR_CWEN51(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN51_SHIFT))&ADC_CWENR_CWEN51_MASK)
#define ADC_CWENR_CWEN83_MASK                    0x80000u
#define ADC_CWENR_CWEN83_SHIFT                   19u
#define ADC_CWENR_CWEN83_WIDTH                   1u
#define ADC_CWENR_CWEN83(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN83_SHIFT))&ADC_CWENR_CWEN83_MASK)
#define ADC_CWENR_CWEN52_MASK                    0x100000u
#define ADC_CWENR_CWEN52_SHIFT                   20u
#define ADC_CWENR_CWEN52_WIDTH                   1u
#define ADC_CWENR_CWEN52(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN52_SHIFT))&ADC_CWENR_CWEN52_MASK)
#define ADC_CWENR_CWEN84_MASK                    0x100000u
#define ADC_CWENR_CWEN84_SHIFT                   20u
#define ADC_CWENR_CWEN84_WIDTH                   1u
#define ADC_CWENR_CWEN84(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN84_SHIFT))&ADC_CWENR_CWEN84_MASK)
#define ADC_CWENR_CWEN53_MASK                    0x200000u
#define ADC_CWENR_CWEN53_SHIFT                   21u
#define ADC_CWENR_CWEN53_WIDTH                   1u
#define ADC_CWENR_CWEN53(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN53_SHIFT))&ADC_CWENR_CWEN53_MASK)
#define ADC_CWENR_CWEN85_MASK                    0x200000u
#define ADC_CWENR_CWEN85_SHIFT                   21u
#define ADC_CWENR_CWEN85_WIDTH                   1u
#define ADC_CWENR_CWEN85(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN85_SHIFT))&ADC_CWENR_CWEN85_MASK)
#define ADC_CWENR_CWEN54_MASK                    0x400000u
#define ADC_CWENR_CWEN54_SHIFT                   22u
#define ADC_CWENR_CWEN54_WIDTH                   1u
#define ADC_CWENR_CWEN54(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN54_SHIFT))&ADC_CWENR_CWEN54_MASK)
#define ADC_CWENR_CWEN86_MASK                    0x400000u
#define ADC_CWENR_CWEN86_SHIFT                   22u
#define ADC_CWENR_CWEN86_WIDTH                   1u
#define ADC_CWENR_CWEN86(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN86_SHIFT))&ADC_CWENR_CWEN86_MASK)
#define ADC_CWENR_CWEN55_MASK                    0x800000u
#define ADC_CWENR_CWEN55_SHIFT                   23u
#define ADC_CWENR_CWEN55_WIDTH                   1u
#define ADC_CWENR_CWEN55(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN55_SHIFT))&ADC_CWENR_CWEN55_MASK)
#define ADC_CWENR_CWEN87_MASK                    0x800000u
#define ADC_CWENR_CWEN87_SHIFT                   23u
#define ADC_CWENR_CWEN87_WIDTH                   1u
#define ADC_CWENR_CWEN87(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN87_SHIFT))&ADC_CWENR_CWEN87_MASK)
#define ADC_CWENR_CWEN88_MASK                    0x1000000u
#define ADC_CWENR_CWEN88_SHIFT                   24u
#define ADC_CWENR_CWEN88_WIDTH                   1u
#define ADC_CWENR_CWEN88(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN88_SHIFT))&ADC_CWENR_CWEN88_MASK)
#define ADC_CWENR_CWEN56_MASK                    0x1000000u
#define ADC_CWENR_CWEN56_SHIFT                   24u
#define ADC_CWENR_CWEN56_WIDTH                   1u
#define ADC_CWENR_CWEN56(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN56_SHIFT))&ADC_CWENR_CWEN56_MASK)
#define ADC_CWENR_CWEN89_MASK                    0x2000000u
#define ADC_CWENR_CWEN89_SHIFT                   25u
#define ADC_CWENR_CWEN89_WIDTH                   1u
#define ADC_CWENR_CWEN89(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN89_SHIFT))&ADC_CWENR_CWEN89_MASK)
#define ADC_CWENR_CWEN57_MASK                    0x2000000u
#define ADC_CWENR_CWEN57_SHIFT                   25u
#define ADC_CWENR_CWEN57_WIDTH                   1u
#define ADC_CWENR_CWEN57(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN57_SHIFT))&ADC_CWENR_CWEN57_MASK)
#define ADC_CWENR_CWEN90_MASK                    0x4000000u
#define ADC_CWENR_CWEN90_SHIFT                   26u
#define ADC_CWENR_CWEN90_WIDTH                   1u
#define ADC_CWENR_CWEN90(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN90_SHIFT))&ADC_CWENR_CWEN90_MASK)
#define ADC_CWENR_CWEN58_MASK                    0x4000000u
#define ADC_CWENR_CWEN58_SHIFT                   26u
#define ADC_CWENR_CWEN58_WIDTH                   1u
#define ADC_CWENR_CWEN58(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN58_SHIFT))&ADC_CWENR_CWEN58_MASK)
#define ADC_CWENR_CWEN59_MASK                    0x8000000u
#define ADC_CWENR_CWEN59_SHIFT                   27u
#define ADC_CWENR_CWEN59_WIDTH                   1u
#define ADC_CWENR_CWEN59(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN59_SHIFT))&ADC_CWENR_CWEN59_MASK)
#define ADC_CWENR_CWEN91_MASK                    0x8000000u
#define ADC_CWENR_CWEN91_SHIFT                   27u
#define ADC_CWENR_CWEN91_WIDTH                   1u
#define ADC_CWENR_CWEN91(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN91_SHIFT))&ADC_CWENR_CWEN91_MASK)
#define ADC_CWENR_CWEN60_MASK                    0x10000000u
#define ADC_CWENR_CWEN60_SHIFT                   28u
#define ADC_CWENR_CWEN60_WIDTH                   1u
#define ADC_CWENR_CWEN60(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN60_SHIFT))&ADC_CWENR_CWEN60_MASK)
#define ADC_CWENR_CWEN92_MASK                    0x10000000u
#define ADC_CWENR_CWEN92_SHIFT                   28u
#define ADC_CWENR_CWEN92_WIDTH                   1u
#define ADC_CWENR_CWEN92(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN92_SHIFT))&ADC_CWENR_CWEN92_MASK)
#define ADC_CWENR_CWEN61_MASK                    0x20000000u
#define ADC_CWENR_CWEN61_SHIFT                   29u
#define ADC_CWENR_CWEN61_WIDTH                   1u
#define ADC_CWENR_CWEN61(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN61_SHIFT))&ADC_CWENR_CWEN61_MASK)
#define ADC_CWENR_CWEN93_MASK                    0x20000000u
#define ADC_CWENR_CWEN93_SHIFT                   29u
#define ADC_CWENR_CWEN93_WIDTH                   1u
#define ADC_CWENR_CWEN93(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN93_SHIFT))&ADC_CWENR_CWEN93_MASK)
#define ADC_CWENR_CWEN94_MASK                    0x40000000u
#define ADC_CWENR_CWEN94_SHIFT                   30u
#define ADC_CWENR_CWEN94_WIDTH                   1u
#define ADC_CWENR_CWEN94(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN94_SHIFT))&ADC_CWENR_CWEN94_MASK)
#define ADC_CWENR_CWEN62_MASK                    0x40000000u
#define ADC_CWENR_CWEN62_SHIFT                   30u
#define ADC_CWENR_CWEN62_WIDTH                   1u
#define ADC_CWENR_CWEN62(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN62_SHIFT))&ADC_CWENR_CWEN62_MASK)
#define ADC_CWENR_CWEN95_MASK                    0x80000000u
#define ADC_CWENR_CWEN95_SHIFT                   31u
#define ADC_CWENR_CWEN95_WIDTH                   1u
#define ADC_CWENR_CWEN95(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN95_SHIFT))&ADC_CWENR_CWEN95_MASK)
#define ADC_CWENR_CWEN63_MASK                    0x80000000u
#define ADC_CWENR_CWEN63_SHIFT                   31u
#define ADC_CWENR_CWEN63_WIDTH                   1u
#define ADC_CWENR_CWEN63(x)                      (((uint32_t)(((uint32_t)(x))<<ADC_CWENR_CWEN63_SHIFT))&ADC_CWENR_CWEN63_MASK)
/* AWORR Bit Fields */
#define ADC_AWORR_AWOR_CH0_MASK                  0x1u
#define ADC_AWORR_AWOR_CH0_SHIFT                 0u
#define ADC_AWORR_AWOR_CH0_WIDTH                 1u
#define ADC_AWORR_AWOR_CH0(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH0_SHIFT))&ADC_AWORR_AWOR_CH0_MASK)
#define ADC_AWORR_AWOR_CH64_MASK                 0x1u
#define ADC_AWORR_AWOR_CH64_SHIFT                0u
#define ADC_AWORR_AWOR_CH64_WIDTH                1u
#define ADC_AWORR_AWOR_CH64(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH64_SHIFT))&ADC_AWORR_AWOR_CH64_MASK)
#define ADC_AWORR_AWOR_CH32_MASK                 0x1u
#define ADC_AWORR_AWOR_CH32_SHIFT                0u
#define ADC_AWORR_AWOR_CH32_WIDTH                1u
#define ADC_AWORR_AWOR_CH32(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH32_SHIFT))&ADC_AWORR_AWOR_CH32_MASK)
#define ADC_AWORR_AWOR_CH65_MASK                 0x2u
#define ADC_AWORR_AWOR_CH65_SHIFT                1u
#define ADC_AWORR_AWOR_CH65_WIDTH                1u
#define ADC_AWORR_AWOR_CH65(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH65_SHIFT))&ADC_AWORR_AWOR_CH65_MASK)
#define ADC_AWORR_AWOR_CH33_MASK                 0x2u
#define ADC_AWORR_AWOR_CH33_SHIFT                1u
#define ADC_AWORR_AWOR_CH33_WIDTH                1u
#define ADC_AWORR_AWOR_CH33(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH33_SHIFT))&ADC_AWORR_AWOR_CH33_MASK)
#define ADC_AWORR_AWOR_CH1_MASK                  0x2u
#define ADC_AWORR_AWOR_CH1_SHIFT                 1u
#define ADC_AWORR_AWOR_CH1_WIDTH                 1u
#define ADC_AWORR_AWOR_CH1(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH1_SHIFT))&ADC_AWORR_AWOR_CH1_MASK)
#define ADC_AWORR_AWOR_CH34_MASK                 0x4u
#define ADC_AWORR_AWOR_CH34_SHIFT                2u
#define ADC_AWORR_AWOR_CH34_WIDTH                1u
#define ADC_AWORR_AWOR_CH34(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH34_SHIFT))&ADC_AWORR_AWOR_CH34_MASK)
#define ADC_AWORR_AWOR_CH2_MASK                  0x4u
#define ADC_AWORR_AWOR_CH2_SHIFT                 2u
#define ADC_AWORR_AWOR_CH2_WIDTH                 1u
#define ADC_AWORR_AWOR_CH2(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH2_SHIFT))&ADC_AWORR_AWOR_CH2_MASK)
#define ADC_AWORR_AWOR_CH66_MASK                 0x4u
#define ADC_AWORR_AWOR_CH66_SHIFT                2u
#define ADC_AWORR_AWOR_CH66_WIDTH                1u
#define ADC_AWORR_AWOR_CH66(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH66_SHIFT))&ADC_AWORR_AWOR_CH66_MASK)
#define ADC_AWORR_AWOR_CH67_MASK                 0x8u
#define ADC_AWORR_AWOR_CH67_SHIFT                3u
#define ADC_AWORR_AWOR_CH67_WIDTH                1u
#define ADC_AWORR_AWOR_CH67(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH67_SHIFT))&ADC_AWORR_AWOR_CH67_MASK)
#define ADC_AWORR_AWOR_CH3_MASK                  0x8u
#define ADC_AWORR_AWOR_CH3_SHIFT                 3u
#define ADC_AWORR_AWOR_CH3_WIDTH                 1u
#define ADC_AWORR_AWOR_CH3(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH3_SHIFT))&ADC_AWORR_AWOR_CH3_MASK)
#define ADC_AWORR_AWOR_CH35_MASK                 0x8u
#define ADC_AWORR_AWOR_CH35_SHIFT                3u
#define ADC_AWORR_AWOR_CH35_WIDTH                1u
#define ADC_AWORR_AWOR_CH35(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH35_SHIFT))&ADC_AWORR_AWOR_CH35_MASK)
#define ADC_AWORR_AWOR_CH4_MASK                  0x10u
#define ADC_AWORR_AWOR_CH4_SHIFT                 4u
#define ADC_AWORR_AWOR_CH4_WIDTH                 1u
#define ADC_AWORR_AWOR_CH4(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH4_SHIFT))&ADC_AWORR_AWOR_CH4_MASK)
#define ADC_AWORR_AWOR_CH36_MASK                 0x10u
#define ADC_AWORR_AWOR_CH36_SHIFT                4u
#define ADC_AWORR_AWOR_CH36_WIDTH                1u
#define ADC_AWORR_AWOR_CH36(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH36_SHIFT))&ADC_AWORR_AWOR_CH36_MASK)
#define ADC_AWORR_AWOR_CH68_MASK                 0x10u
#define ADC_AWORR_AWOR_CH68_SHIFT                4u
#define ADC_AWORR_AWOR_CH68_WIDTH                1u
#define ADC_AWORR_AWOR_CH68(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH68_SHIFT))&ADC_AWORR_AWOR_CH68_MASK)
#define ADC_AWORR_AWOR_CH69_MASK                 0x20u
#define ADC_AWORR_AWOR_CH69_SHIFT                5u
#define ADC_AWORR_AWOR_CH69_WIDTH                1u
#define ADC_AWORR_AWOR_CH69(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH69_SHIFT))&ADC_AWORR_AWOR_CH69_MASK)
#define ADC_AWORR_AWOR_CH37_MASK                 0x20u
#define ADC_AWORR_AWOR_CH37_SHIFT                5u
#define ADC_AWORR_AWOR_CH37_WIDTH                1u
#define ADC_AWORR_AWOR_CH37(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH37_SHIFT))&ADC_AWORR_AWOR_CH37_MASK)
#define ADC_AWORR_AWOR_CH5_MASK                  0x20u
#define ADC_AWORR_AWOR_CH5_SHIFT                 5u
#define ADC_AWORR_AWOR_CH5_WIDTH                 1u
#define ADC_AWORR_AWOR_CH5(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH5_SHIFT))&ADC_AWORR_AWOR_CH5_MASK)
#define ADC_AWORR_AWOR_CH70_MASK                 0x40u
#define ADC_AWORR_AWOR_CH70_SHIFT                6u
#define ADC_AWORR_AWOR_CH70_WIDTH                1u
#define ADC_AWORR_AWOR_CH70(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH70_SHIFT))&ADC_AWORR_AWOR_CH70_MASK)
#define ADC_AWORR_AWOR_CH38_MASK                 0x40u
#define ADC_AWORR_AWOR_CH38_SHIFT                6u
#define ADC_AWORR_AWOR_CH38_WIDTH                1u
#define ADC_AWORR_AWOR_CH38(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH38_SHIFT))&ADC_AWORR_AWOR_CH38_MASK)
#define ADC_AWORR_AWOR_CH6_MASK                  0x40u
#define ADC_AWORR_AWOR_CH6_SHIFT                 6u
#define ADC_AWORR_AWOR_CH6_WIDTH                 1u
#define ADC_AWORR_AWOR_CH6(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH6_SHIFT))&ADC_AWORR_AWOR_CH6_MASK)
#define ADC_AWORR_AWOR_CH7_MASK                  0x80u
#define ADC_AWORR_AWOR_CH7_SHIFT                 7u
#define ADC_AWORR_AWOR_CH7_WIDTH                 1u
#define ADC_AWORR_AWOR_CH7(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH7_SHIFT))&ADC_AWORR_AWOR_CH7_MASK)
#define ADC_AWORR_AWOR_CH39_MASK                 0x80u
#define ADC_AWORR_AWOR_CH39_SHIFT                7u
#define ADC_AWORR_AWOR_CH39_WIDTH                1u
#define ADC_AWORR_AWOR_CH39(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH39_SHIFT))&ADC_AWORR_AWOR_CH39_MASK)
#define ADC_AWORR_AWOR_CH71_MASK                 0x80u
#define ADC_AWORR_AWOR_CH71_SHIFT                7u
#define ADC_AWORR_AWOR_CH71_WIDTH                1u
#define ADC_AWORR_AWOR_CH71(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH71_SHIFT))&ADC_AWORR_AWOR_CH71_MASK)
#define ADC_AWORR_AWOR_CH8_MASK                  0x100u
#define ADC_AWORR_AWOR_CH8_SHIFT                 8u
#define ADC_AWORR_AWOR_CH8_WIDTH                 1u
#define ADC_AWORR_AWOR_CH8(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH8_SHIFT))&ADC_AWORR_AWOR_CH8_MASK)
#define ADC_AWORR_AWOR_CH40_MASK                 0x100u
#define ADC_AWORR_AWOR_CH40_SHIFT                8u
#define ADC_AWORR_AWOR_CH40_WIDTH                1u
#define ADC_AWORR_AWOR_CH40(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH40_SHIFT))&ADC_AWORR_AWOR_CH40_MASK)
#define ADC_AWORR_AWOR_CH72_MASK                 0x100u
#define ADC_AWORR_AWOR_CH72_SHIFT                8u
#define ADC_AWORR_AWOR_CH72_WIDTH                1u
#define ADC_AWORR_AWOR_CH72(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH72_SHIFT))&ADC_AWORR_AWOR_CH72_MASK)
#define ADC_AWORR_AWOR_CH73_MASK                 0x200u
#define ADC_AWORR_AWOR_CH73_SHIFT                9u
#define ADC_AWORR_AWOR_CH73_WIDTH                1u
#define ADC_AWORR_AWOR_CH73(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH73_SHIFT))&ADC_AWORR_AWOR_CH73_MASK)
#define ADC_AWORR_AWOR_CH41_MASK                 0x200u
#define ADC_AWORR_AWOR_CH41_SHIFT                9u
#define ADC_AWORR_AWOR_CH41_WIDTH                1u
#define ADC_AWORR_AWOR_CH41(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH41_SHIFT))&ADC_AWORR_AWOR_CH41_MASK)
#define ADC_AWORR_AWOR_CH9_MASK                  0x200u
#define ADC_AWORR_AWOR_CH9_SHIFT                 9u
#define ADC_AWORR_AWOR_CH9_WIDTH                 1u
#define ADC_AWORR_AWOR_CH9(x)                    (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH9_SHIFT))&ADC_AWORR_AWOR_CH9_MASK)
#define ADC_AWORR_AWOR_CH42_MASK                 0x400u
#define ADC_AWORR_AWOR_CH42_SHIFT                10u
#define ADC_AWORR_AWOR_CH42_WIDTH                1u
#define ADC_AWORR_AWOR_CH42(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH42_SHIFT))&ADC_AWORR_AWOR_CH42_MASK)
#define ADC_AWORR_AWOR_CH74_MASK                 0x400u
#define ADC_AWORR_AWOR_CH74_SHIFT                10u
#define ADC_AWORR_AWOR_CH74_WIDTH                1u
#define ADC_AWORR_AWOR_CH74(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH74_SHIFT))&ADC_AWORR_AWOR_CH74_MASK)
#define ADC_AWORR_AWOR_CH10_MASK                 0x400u
#define ADC_AWORR_AWOR_CH10_SHIFT                10u
#define ADC_AWORR_AWOR_CH10_WIDTH                1u
#define ADC_AWORR_AWOR_CH10(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH10_SHIFT))&ADC_AWORR_AWOR_CH10_MASK)
#define ADC_AWORR_AWOR_CH11_MASK                 0x800u
#define ADC_AWORR_AWOR_CH11_SHIFT                11u
#define ADC_AWORR_AWOR_CH11_WIDTH                1u
#define ADC_AWORR_AWOR_CH11(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH11_SHIFT))&ADC_AWORR_AWOR_CH11_MASK)
#define ADC_AWORR_AWOR_CH75_MASK                 0x800u
#define ADC_AWORR_AWOR_CH75_SHIFT                11u
#define ADC_AWORR_AWOR_CH75_WIDTH                1u
#define ADC_AWORR_AWOR_CH75(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH75_SHIFT))&ADC_AWORR_AWOR_CH75_MASK)
#define ADC_AWORR_AWOR_CH43_MASK                 0x800u
#define ADC_AWORR_AWOR_CH43_SHIFT                11u
#define ADC_AWORR_AWOR_CH43_WIDTH                1u
#define ADC_AWORR_AWOR_CH43(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH43_SHIFT))&ADC_AWORR_AWOR_CH43_MASK)
#define ADC_AWORR_AWOR_CH76_MASK                 0x1000u
#define ADC_AWORR_AWOR_CH76_SHIFT                12u
#define ADC_AWORR_AWOR_CH76_WIDTH                1u
#define ADC_AWORR_AWOR_CH76(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH76_SHIFT))&ADC_AWORR_AWOR_CH76_MASK)
#define ADC_AWORR_AWOR_CH12_MASK                 0x1000u
#define ADC_AWORR_AWOR_CH12_SHIFT                12u
#define ADC_AWORR_AWOR_CH12_WIDTH                1u
#define ADC_AWORR_AWOR_CH12(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH12_SHIFT))&ADC_AWORR_AWOR_CH12_MASK)
#define ADC_AWORR_AWOR_CH44_MASK                 0x1000u
#define ADC_AWORR_AWOR_CH44_SHIFT                12u
#define ADC_AWORR_AWOR_CH44_WIDTH                1u
#define ADC_AWORR_AWOR_CH44(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH44_SHIFT))&ADC_AWORR_AWOR_CH44_MASK)
#define ADC_AWORR_AWOR_CH45_MASK                 0x2000u
#define ADC_AWORR_AWOR_CH45_SHIFT                13u
#define ADC_AWORR_AWOR_CH45_WIDTH                1u
#define ADC_AWORR_AWOR_CH45(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH45_SHIFT))&ADC_AWORR_AWOR_CH45_MASK)
#define ADC_AWORR_AWOR_CH77_MASK                 0x2000u
#define ADC_AWORR_AWOR_CH77_SHIFT                13u
#define ADC_AWORR_AWOR_CH77_WIDTH                1u
#define ADC_AWORR_AWOR_CH77(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH77_SHIFT))&ADC_AWORR_AWOR_CH77_MASK)
#define ADC_AWORR_AWOR_CH13_MASK                 0x2000u
#define ADC_AWORR_AWOR_CH13_SHIFT                13u
#define ADC_AWORR_AWOR_CH13_WIDTH                1u
#define ADC_AWORR_AWOR_CH13(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH13_SHIFT))&ADC_AWORR_AWOR_CH13_MASK)
#define ADC_AWORR_AWOR_CH14_MASK                 0x4000u
#define ADC_AWORR_AWOR_CH14_SHIFT                14u
#define ADC_AWORR_AWOR_CH14_WIDTH                1u
#define ADC_AWORR_AWOR_CH14(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH14_SHIFT))&ADC_AWORR_AWOR_CH14_MASK)
#define ADC_AWORR_AWOR_CH46_MASK                 0x4000u
#define ADC_AWORR_AWOR_CH46_SHIFT                14u
#define ADC_AWORR_AWOR_CH46_WIDTH                1u
#define ADC_AWORR_AWOR_CH46(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH46_SHIFT))&ADC_AWORR_AWOR_CH46_MASK)
#define ADC_AWORR_AWOR_CH78_MASK                 0x4000u
#define ADC_AWORR_AWOR_CH78_SHIFT                14u
#define ADC_AWORR_AWOR_CH78_WIDTH                1u
#define ADC_AWORR_AWOR_CH78(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH78_SHIFT))&ADC_AWORR_AWOR_CH78_MASK)
#define ADC_AWORR_AWOR_CH47_MASK                 0x8000u
#define ADC_AWORR_AWOR_CH47_SHIFT                15u
#define ADC_AWORR_AWOR_CH47_WIDTH                1u
#define ADC_AWORR_AWOR_CH47(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH47_SHIFT))&ADC_AWORR_AWOR_CH47_MASK)
#define ADC_AWORR_AWOR_CH15_MASK                 0x8000u
#define ADC_AWORR_AWOR_CH15_SHIFT                15u
#define ADC_AWORR_AWOR_CH15_WIDTH                1u
#define ADC_AWORR_AWOR_CH15(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH15_SHIFT))&ADC_AWORR_AWOR_CH15_MASK)
#define ADC_AWORR_AWOR_CH79_MASK                 0x8000u
#define ADC_AWORR_AWOR_CH79_SHIFT                15u
#define ADC_AWORR_AWOR_CH79_WIDTH                1u
#define ADC_AWORR_AWOR_CH79(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH79_SHIFT))&ADC_AWORR_AWOR_CH79_MASK)
#define ADC_AWORR_AWOR_CH80_MASK                 0x10000u
#define ADC_AWORR_AWOR_CH80_SHIFT                16u
#define ADC_AWORR_AWOR_CH80_WIDTH                1u
#define ADC_AWORR_AWOR_CH80(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH80_SHIFT))&ADC_AWORR_AWOR_CH80_MASK)
#define ADC_AWORR_AWOR_CH48_MASK                 0x10000u
#define ADC_AWORR_AWOR_CH48_SHIFT                16u
#define ADC_AWORR_AWOR_CH48_WIDTH                1u
#define ADC_AWORR_AWOR_CH48(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH48_SHIFT))&ADC_AWORR_AWOR_CH48_MASK)
#define ADC_AWORR_AWOR_CH81_MASK                 0x20000u
#define ADC_AWORR_AWOR_CH81_SHIFT                17u
#define ADC_AWORR_AWOR_CH81_WIDTH                1u
#define ADC_AWORR_AWOR_CH81(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH81_SHIFT))&ADC_AWORR_AWOR_CH81_MASK)
#define ADC_AWORR_AWOR_CH49_MASK                 0x20000u
#define ADC_AWORR_AWOR_CH49_SHIFT                17u
#define ADC_AWORR_AWOR_CH49_WIDTH                1u
#define ADC_AWORR_AWOR_CH49(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH49_SHIFT))&ADC_AWORR_AWOR_CH49_MASK)
#define ADC_AWORR_AWOR_CH50_MASK                 0x40000u
#define ADC_AWORR_AWOR_CH50_SHIFT                18u
#define ADC_AWORR_AWOR_CH50_WIDTH                1u
#define ADC_AWORR_AWOR_CH50(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH50_SHIFT))&ADC_AWORR_AWOR_CH50_MASK)
#define ADC_AWORR_AWOR_CH82_MASK                 0x40000u
#define ADC_AWORR_AWOR_CH82_SHIFT                18u
#define ADC_AWORR_AWOR_CH82_WIDTH                1u
#define ADC_AWORR_AWOR_CH82(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH82_SHIFT))&ADC_AWORR_AWOR_CH82_MASK)
#define ADC_AWORR_AWOR_CH51_MASK                 0x80000u
#define ADC_AWORR_AWOR_CH51_SHIFT                19u
#define ADC_AWORR_AWOR_CH51_WIDTH                1u
#define ADC_AWORR_AWOR_CH51(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH51_SHIFT))&ADC_AWORR_AWOR_CH51_MASK)
#define ADC_AWORR_AWOR_CH83_MASK                 0x80000u
#define ADC_AWORR_AWOR_CH83_SHIFT                19u
#define ADC_AWORR_AWOR_CH83_WIDTH                1u
#define ADC_AWORR_AWOR_CH83(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH83_SHIFT))&ADC_AWORR_AWOR_CH83_MASK)
#define ADC_AWORR_AWOR_CH52_MASK                 0x100000u
#define ADC_AWORR_AWOR_CH52_SHIFT                20u
#define ADC_AWORR_AWOR_CH52_WIDTH                1u
#define ADC_AWORR_AWOR_CH52(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH52_SHIFT))&ADC_AWORR_AWOR_CH52_MASK)
#define ADC_AWORR_AWOR_CH84_MASK                 0x100000u
#define ADC_AWORR_AWOR_CH84_SHIFT                20u
#define ADC_AWORR_AWOR_CH84_WIDTH                1u
#define ADC_AWORR_AWOR_CH84(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH84_SHIFT))&ADC_AWORR_AWOR_CH84_MASK)
#define ADC_AWORR_AWOR_CH53_MASK                 0x200000u
#define ADC_AWORR_AWOR_CH53_SHIFT                21u
#define ADC_AWORR_AWOR_CH53_WIDTH                1u
#define ADC_AWORR_AWOR_CH53(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH53_SHIFT))&ADC_AWORR_AWOR_CH53_MASK)
#define ADC_AWORR_AWOR_CH85_MASK                 0x200000u
#define ADC_AWORR_AWOR_CH85_SHIFT                21u
#define ADC_AWORR_AWOR_CH85_WIDTH                1u
#define ADC_AWORR_AWOR_CH85(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH85_SHIFT))&ADC_AWORR_AWOR_CH85_MASK)
#define ADC_AWORR_AWOR_CH54_MASK                 0x400000u
#define ADC_AWORR_AWOR_CH54_SHIFT                22u
#define ADC_AWORR_AWOR_CH54_WIDTH                1u
#define ADC_AWORR_AWOR_CH54(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH54_SHIFT))&ADC_AWORR_AWOR_CH54_MASK)
#define ADC_AWORR_AWOR_CH86_MASK                 0x400000u
#define ADC_AWORR_AWOR_CH86_SHIFT                22u
#define ADC_AWORR_AWOR_CH86_WIDTH                1u
#define ADC_AWORR_AWOR_CH86(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH86_SHIFT))&ADC_AWORR_AWOR_CH86_MASK)
#define ADC_AWORR_AWOR_CH55_MASK                 0x800000u
#define ADC_AWORR_AWOR_CH55_SHIFT                23u
#define ADC_AWORR_AWOR_CH55_WIDTH                1u
#define ADC_AWORR_AWOR_CH55(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH55_SHIFT))&ADC_AWORR_AWOR_CH55_MASK)
#define ADC_AWORR_AWOR_CH87_MASK                 0x800000u
#define ADC_AWORR_AWOR_CH87_SHIFT                23u
#define ADC_AWORR_AWOR_CH87_WIDTH                1u
#define ADC_AWORR_AWOR_CH87(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH87_SHIFT))&ADC_AWORR_AWOR_CH87_MASK)
#define ADC_AWORR_AWOR_CH88_MASK                 0x1000000u
#define ADC_AWORR_AWOR_CH88_SHIFT                24u
#define ADC_AWORR_AWOR_CH88_WIDTH                1u
#define ADC_AWORR_AWOR_CH88(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH88_SHIFT))&ADC_AWORR_AWOR_CH88_MASK)
#define ADC_AWORR_AWOR_CH56_MASK                 0x1000000u
#define ADC_AWORR_AWOR_CH56_SHIFT                24u
#define ADC_AWORR_AWOR_CH56_WIDTH                1u
#define ADC_AWORR_AWOR_CH56(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH56_SHIFT))&ADC_AWORR_AWOR_CH56_MASK)
#define ADC_AWORR_AWOR_CH89_MASK                 0x2000000u
#define ADC_AWORR_AWOR_CH89_SHIFT                25u
#define ADC_AWORR_AWOR_CH89_WIDTH                1u
#define ADC_AWORR_AWOR_CH89(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH89_SHIFT))&ADC_AWORR_AWOR_CH89_MASK)
#define ADC_AWORR_AWOR_CH57_MASK                 0x2000000u
#define ADC_AWORR_AWOR_CH57_SHIFT                25u
#define ADC_AWORR_AWOR_CH57_WIDTH                1u
#define ADC_AWORR_AWOR_CH57(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH57_SHIFT))&ADC_AWORR_AWOR_CH57_MASK)
#define ADC_AWORR_AWOR_CH90_MASK                 0x4000000u
#define ADC_AWORR_AWOR_CH90_SHIFT                26u
#define ADC_AWORR_AWOR_CH90_WIDTH                1u
#define ADC_AWORR_AWOR_CH90(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH90_SHIFT))&ADC_AWORR_AWOR_CH90_MASK)
#define ADC_AWORR_AWOR_CH58_MASK                 0x4000000u
#define ADC_AWORR_AWOR_CH58_SHIFT                26u
#define ADC_AWORR_AWOR_CH58_WIDTH                1u
#define ADC_AWORR_AWOR_CH58(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH58_SHIFT))&ADC_AWORR_AWOR_CH58_MASK)
#define ADC_AWORR_AWOR_CH59_MASK                 0x8000000u
#define ADC_AWORR_AWOR_CH59_SHIFT                27u
#define ADC_AWORR_AWOR_CH59_WIDTH                1u
#define ADC_AWORR_AWOR_CH59(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH59_SHIFT))&ADC_AWORR_AWOR_CH59_MASK)
#define ADC_AWORR_AWOR_CH91_MASK                 0x8000000u
#define ADC_AWORR_AWOR_CH91_SHIFT                27u
#define ADC_AWORR_AWOR_CH91_WIDTH                1u
#define ADC_AWORR_AWOR_CH91(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH91_SHIFT))&ADC_AWORR_AWOR_CH91_MASK)
#define ADC_AWORR_AWOR_CH60_MASK                 0x10000000u
#define ADC_AWORR_AWOR_CH60_SHIFT                28u
#define ADC_AWORR_AWOR_CH60_WIDTH                1u
#define ADC_AWORR_AWOR_CH60(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH60_SHIFT))&ADC_AWORR_AWOR_CH60_MASK)
#define ADC_AWORR_AWOR_CH92_MASK                 0x10000000u
#define ADC_AWORR_AWOR_CH92_SHIFT                28u
#define ADC_AWORR_AWOR_CH92_WIDTH                1u
#define ADC_AWORR_AWOR_CH92(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH92_SHIFT))&ADC_AWORR_AWOR_CH92_MASK)
#define ADC_AWORR_AWOR_CH61_MASK                 0x20000000u
#define ADC_AWORR_AWOR_CH61_SHIFT                29u
#define ADC_AWORR_AWOR_CH61_WIDTH                1u
#define ADC_AWORR_AWOR_CH61(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH61_SHIFT))&ADC_AWORR_AWOR_CH61_MASK)
#define ADC_AWORR_AWOR_CH93_MASK                 0x20000000u
#define ADC_AWORR_AWOR_CH93_SHIFT                29u
#define ADC_AWORR_AWOR_CH93_WIDTH                1u
#define ADC_AWORR_AWOR_CH93(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH93_SHIFT))&ADC_AWORR_AWOR_CH93_MASK)
#define ADC_AWORR_AWOR_CH94_MASK                 0x40000000u
#define ADC_AWORR_AWOR_CH94_SHIFT                30u
#define ADC_AWORR_AWOR_CH94_WIDTH                1u
#define ADC_AWORR_AWOR_CH94(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH94_SHIFT))&ADC_AWORR_AWOR_CH94_MASK)
#define ADC_AWORR_AWOR_CH62_MASK                 0x40000000u
#define ADC_AWORR_AWOR_CH62_SHIFT                30u
#define ADC_AWORR_AWOR_CH62_WIDTH                1u
#define ADC_AWORR_AWOR_CH62(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH62_SHIFT))&ADC_AWORR_AWOR_CH62_MASK)
#define ADC_AWORR_AWOR_CH95_MASK                 0x80000000u
#define ADC_AWORR_AWOR_CH95_SHIFT                31u
#define ADC_AWORR_AWOR_CH95_WIDTH                1u
#define ADC_AWORR_AWOR_CH95(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH95_SHIFT))&ADC_AWORR_AWOR_CH95_MASK)
#define ADC_AWORR_AWOR_CH63_MASK                 0x80000000u
#define ADC_AWORR_AWOR_CH63_SHIFT                31u
#define ADC_AWORR_AWOR_CH63_WIDTH                1u
#define ADC_AWORR_AWOR_CH63(x)                   (((uint32_t)(((uint32_t)(x))<<ADC_AWORR_AWOR_CH63_SHIFT))&ADC_AWORR_AWOR_CH63_MASK)
/* CALBISTREG Bit Fields */
#define ADC_CALBISTREG_TEST_EN_MASK              0x1u
#define ADC_CALBISTREG_TEST_EN_SHIFT             0u
#define ADC_CALBISTREG_TEST_EN_WIDTH             1u
#define ADC_CALBISTREG_TEST_EN(x)                (((uint32_t)(((uint32_t)(x))<<ADC_CALBISTREG_TEST_EN_SHIFT))&ADC_CALBISTREG_TEST_EN_MASK)
#define ADC_CALBISTREG_TEST_FAIL_MASK            0x8u
#define ADC_CALBISTREG_TEST_FAIL_SHIFT           3u
#define ADC_CALBISTREG_TEST_FAIL_WIDTH           1u
#define ADC_CALBISTREG_TEST_FAIL(x)              (((uint32_t)(((uint32_t)(x))<<ADC_CALBISTREG_TEST_FAIL_SHIFT))&ADC_CALBISTREG_TEST_FAIL_MASK)
#define ADC_CALBISTREG_AVG_EN_MASK               0x10u
#define ADC_CALBISTREG_AVG_EN_SHIFT              4u
#define ADC_CALBISTREG_AVG_EN_WIDTH              1u
#define ADC_CALBISTREG_AVG_EN(x)                 (((uint32_t)(((uint32_t)(x))<<ADC_CALBISTREG_AVG_EN_SHIFT))&ADC_CALBISTREG_AVG_EN_MASK)
#define ADC_CALBISTREG_NR_SMPL_MASK              0x60u
#define ADC_CALBISTREG_NR_SMPL_SHIFT             5u
#define ADC_CALBISTREG_NR_SMPL_WIDTH             2u
#define ADC_CALBISTREG_NR_SMPL(x)                (((uint32_t)(((uint32_t)(x))<<ADC_CALBISTREG_NR_SMPL_SHIFT))&ADC_CALBISTREG_NR_SMPL_MASK)
#define ADC_CALBISTREG_C_T_BUSY_MASK             0x8000u
#define ADC_CALBISTREG_C_T_BUSY_SHIFT            15u
#define ADC_CALBISTREG_C_T_BUSY_WIDTH            1u
#define ADC_CALBISTREG_C_T_BUSY(x)               (((uint32_t)(((uint32_t)(x))<<ADC_CALBISTREG_C_T_BUSY_SHIFT))&ADC_CALBISTREG_C_T_BUSY_MASK)
#define ADC_CALBISTREG_TSAMP_MASK                0x18000000u
#define ADC_CALBISTREG_TSAMP_SHIFT               27u
#define ADC_CALBISTREG_TSAMP_WIDTH               2u
#define ADC_CALBISTREG_TSAMP(x)                  (((uint32_t)(((uint32_t)(x))<<ADC_CALBISTREG_TSAMP_SHIFT))&ADC_CALBISTREG_TSAMP_MASK)
/* OFSGNUSR Bit Fields */
#define ADC_OFSGNUSR_OFFSET_USER_MASK            0xFFu
#define ADC_OFSGNUSR_OFFSET_USER_SHIFT           0u
#define ADC_OFSGNUSR_OFFSET_USER_WIDTH           8u
#define ADC_OFSGNUSR_OFFSET_USER(x)              (((uint32_t)(((uint32_t)(x))<<ADC_OFSGNUSR_OFFSET_USER_SHIFT))&ADC_OFSGNUSR_OFFSET_USER_MASK)
#define ADC_OFSGNUSR_GAIN_USER_MASK              0x3FF0000u
#define ADC_OFSGNUSR_GAIN_USER_SHIFT             16u
#define ADC_OFSGNUSR_GAIN_USER_WIDTH             10u
#define ADC_OFSGNUSR_GAIN_USER(x)                (((uint32_t)(((uint32_t)(x))<<ADC_OFSGNUSR_GAIN_USER_SHIFT))&ADC_OFSGNUSR_GAIN_USER_MASK)

/*!
 * @}
 */ /* end of group ADC_Register_Masks */


/*!
 * @}
 */ /* end of group ADC_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- AIPS Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup AIPS_Peripheral_Access_Layer AIPS Peripheral Access Layer
 * @{
 */


/** AIPS - Size of Registers Arrays */
#define AIPS_PACR_COUNT                          8u
#define AIPS_OPACR_COUNT                         32u

/** AIPS - Register Layout Typedef */
typedef struct {
  __IO uint32_t MPRA;                              /**< Master Privilege Register A, offset: 0x0 */
  __IO uint32_t MPRB;                              /**< Master Privilege Register B, offset: 0x4 */
       uint8_t RESERVED_0[248];
  __IO uint32_t PACR[AIPS_PACR_COUNT];             /**< Peripheral Access Control Register, array offset: 0x100, array step: 0x4 */
       uint8_t RESERVED_1[32];
  __IO uint32_t OPACR[AIPS_OPACR_COUNT];           /**< Off-Platform Peripheral Access Control Register, array offset: 0x140, array step: 0x4 */
} AIPS_Type, *AIPS_MemMapPtr;

 /** Number of instances of the AIPS module. */
#define AIPS_INSTANCE_COUNT                      (2u)


/* AIPS - Peripheral instance base addresses */
/** Peripheral AIPS_A base address */
#define AIPS_A_BASE                              (0xFC000000u)
/** Peripheral AIPS_A base pointer */
#define AIPS_A                                   ((AIPS_Type *)AIPS_A_BASE)
/** Peripheral AIPS_B base address */
#define AIPS_B_BASE                              (0xF8000000u)
/** Peripheral AIPS_B base pointer */
#define AIPS_B                                   ((AIPS_Type *)AIPS_B_BASE)
/** Array initializer of AIPS peripheral base addresses */
#define AIPS_BASE_ADDRS                          { AIPS_A_BASE, AIPS_B_BASE }
/** Array initializer of AIPS peripheral base pointers */
#define AIPS_BASE_PTRS                           { AIPS_A, AIPS_B }

/* ----------------------------------------------------------------------------
   -- AIPS Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup AIPS_Register_Masks AIPS Register Masks
 * @{
 */

/* MPRA Bit Fields */
#define AIPS_MPRA_MPL7_MASK                      0x1u
#define AIPS_MPRA_MPL7_SHIFT                     0u
#define AIPS_MPRA_MPL7_WIDTH                     1u
#define AIPS_MPRA_MPL7(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MPL7_SHIFT))&AIPS_MPRA_MPL7_MASK)
#define AIPS_MPRA_MTW7_MASK                      0x2u
#define AIPS_MPRA_MTW7_SHIFT                     1u
#define AIPS_MPRA_MTW7_WIDTH                     1u
#define AIPS_MPRA_MTW7(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTW7_SHIFT))&AIPS_MPRA_MTW7_MASK)
#define AIPS_MPRA_MTR7_MASK                      0x4u
#define AIPS_MPRA_MTR7_SHIFT                     2u
#define AIPS_MPRA_MTR7_WIDTH                     1u
#define AIPS_MPRA_MTR7(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTR7_SHIFT))&AIPS_MPRA_MTR7_MASK)
#define AIPS_MPRA_MPL6_MASK                      0x10u
#define AIPS_MPRA_MPL6_SHIFT                     4u
#define AIPS_MPRA_MPL6_WIDTH                     1u
#define AIPS_MPRA_MPL6(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MPL6_SHIFT))&AIPS_MPRA_MPL6_MASK)
#define AIPS_MPRA_MTW6_MASK                      0x20u
#define AIPS_MPRA_MTW6_SHIFT                     5u
#define AIPS_MPRA_MTW6_WIDTH                     1u
#define AIPS_MPRA_MTW6(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTW6_SHIFT))&AIPS_MPRA_MTW6_MASK)
#define AIPS_MPRA_MTR6_MASK                      0x40u
#define AIPS_MPRA_MTR6_SHIFT                     6u
#define AIPS_MPRA_MTR6_WIDTH                     1u
#define AIPS_MPRA_MTR6(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTR6_SHIFT))&AIPS_MPRA_MTR6_MASK)
#define AIPS_MPRA_MPL5_MASK                      0x100u
#define AIPS_MPRA_MPL5_SHIFT                     8u
#define AIPS_MPRA_MPL5_WIDTH                     1u
#define AIPS_MPRA_MPL5(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MPL5_SHIFT))&AIPS_MPRA_MPL5_MASK)
#define AIPS_MPRA_MTW5_MASK                      0x200u
#define AIPS_MPRA_MTW5_SHIFT                     9u
#define AIPS_MPRA_MTW5_WIDTH                     1u
#define AIPS_MPRA_MTW5(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTW5_SHIFT))&AIPS_MPRA_MTW5_MASK)
#define AIPS_MPRA_MTR5_MASK                      0x400u
#define AIPS_MPRA_MTR5_SHIFT                     10u
#define AIPS_MPRA_MTR5_WIDTH                     1u
#define AIPS_MPRA_MTR5(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTR5_SHIFT))&AIPS_MPRA_MTR5_MASK)
#define AIPS_MPRA_MPL4_MASK                      0x1000u
#define AIPS_MPRA_MPL4_SHIFT                     12u
#define AIPS_MPRA_MPL4_WIDTH                     1u
#define AIPS_MPRA_MPL4(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MPL4_SHIFT))&AIPS_MPRA_MPL4_MASK)
#define AIPS_MPRA_MTW4_MASK                      0x2000u
#define AIPS_MPRA_MTW4_SHIFT                     13u
#define AIPS_MPRA_MTW4_WIDTH                     1u
#define AIPS_MPRA_MTW4(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTW4_SHIFT))&AIPS_MPRA_MTW4_MASK)
#define AIPS_MPRA_MTR4_MASK                      0x4000u
#define AIPS_MPRA_MTR4_SHIFT                     14u
#define AIPS_MPRA_MTR4_WIDTH                     1u
#define AIPS_MPRA_MTR4(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTR4_SHIFT))&AIPS_MPRA_MTR4_MASK)
#define AIPS_MPRA_MPL3_MASK                      0x10000u
#define AIPS_MPRA_MPL3_SHIFT                     16u
#define AIPS_MPRA_MPL3_WIDTH                     1u
#define AIPS_MPRA_MPL3(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MPL3_SHIFT))&AIPS_MPRA_MPL3_MASK)
#define AIPS_MPRA_MTW3_MASK                      0x20000u
#define AIPS_MPRA_MTW3_SHIFT                     17u
#define AIPS_MPRA_MTW3_WIDTH                     1u
#define AIPS_MPRA_MTW3(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTW3_SHIFT))&AIPS_MPRA_MTW3_MASK)
#define AIPS_MPRA_MTR3_MASK                      0x40000u
#define AIPS_MPRA_MTR3_SHIFT                     18u
#define AIPS_MPRA_MTR3_WIDTH                     1u
#define AIPS_MPRA_MTR3(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTR3_SHIFT))&AIPS_MPRA_MTR3_MASK)
#define AIPS_MPRA_MPL2_MASK                      0x100000u
#define AIPS_MPRA_MPL2_SHIFT                     20u
#define AIPS_MPRA_MPL2_WIDTH                     1u
#define AIPS_MPRA_MPL2(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MPL2_SHIFT))&AIPS_MPRA_MPL2_MASK)
#define AIPS_MPRA_MTW2_MASK                      0x200000u
#define AIPS_MPRA_MTW2_SHIFT                     21u
#define AIPS_MPRA_MTW2_WIDTH                     1u
#define AIPS_MPRA_MTW2(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTW2_SHIFT))&AIPS_MPRA_MTW2_MASK)
#define AIPS_MPRA_MTR2_MASK                      0x400000u
#define AIPS_MPRA_MTR2_SHIFT                     22u
#define AIPS_MPRA_MTR2_WIDTH                     1u
#define AIPS_MPRA_MTR2(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTR2_SHIFT))&AIPS_MPRA_MTR2_MASK)
#define AIPS_MPRA_MPL1_MASK                      0x1000000u
#define AIPS_MPRA_MPL1_SHIFT                     24u
#define AIPS_MPRA_MPL1_WIDTH                     1u
#define AIPS_MPRA_MPL1(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MPL1_SHIFT))&AIPS_MPRA_MPL1_MASK)
#define AIPS_MPRA_MTW1_MASK                      0x2000000u
#define AIPS_MPRA_MTW1_SHIFT                     25u
#define AIPS_MPRA_MTW1_WIDTH                     1u
#define AIPS_MPRA_MTW1(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTW1_SHIFT))&AIPS_MPRA_MTW1_MASK)
#define AIPS_MPRA_MTR1_MASK                      0x4000000u
#define AIPS_MPRA_MTR1_SHIFT                     26u
#define AIPS_MPRA_MTR1_WIDTH                     1u
#define AIPS_MPRA_MTR1(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTR1_SHIFT))&AIPS_MPRA_MTR1_MASK)
#define AIPS_MPRA_MPL0_MASK                      0x10000000u
#define AIPS_MPRA_MPL0_SHIFT                     28u
#define AIPS_MPRA_MPL0_WIDTH                     1u
#define AIPS_MPRA_MPL0(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MPL0_SHIFT))&AIPS_MPRA_MPL0_MASK)
#define AIPS_MPRA_MTW0_MASK                      0x20000000u
#define AIPS_MPRA_MTW0_SHIFT                     29u
#define AIPS_MPRA_MTW0_WIDTH                     1u
#define AIPS_MPRA_MTW0(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTW0_SHIFT))&AIPS_MPRA_MTW0_MASK)
#define AIPS_MPRA_MTR0_MASK                      0x40000000u
#define AIPS_MPRA_MTR0_SHIFT                     30u
#define AIPS_MPRA_MTR0_WIDTH                     1u
#define AIPS_MPRA_MTR0(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRA_MTR0_SHIFT))&AIPS_MPRA_MTR0_MASK)
/* MPRB Bit Fields */
#define AIPS_MPRB_MP14_MASK                      0x10u
#define AIPS_MPRB_MP14_SHIFT                     4u
#define AIPS_MPRB_MP14_WIDTH                     1u
#define AIPS_MPRB_MP14(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MP14_SHIFT))&AIPS_MPRB_MP14_MASK)
#define AIPS_MPRB_MTW14_MASK                     0x20u
#define AIPS_MPRB_MTW14_SHIFT                    5u
#define AIPS_MPRB_MTW14_WIDTH                    1u
#define AIPS_MPRB_MTW14(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTW14_SHIFT))&AIPS_MPRB_MTW14_MASK)
#define AIPS_MPRB_MTR14_MASK                     0x40u
#define AIPS_MPRB_MTR14_SHIFT                    6u
#define AIPS_MPRB_MTR14_WIDTH                    1u
#define AIPS_MPRB_MTR14(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTR14_SHIFT))&AIPS_MPRB_MTR14_MASK)
#define AIPS_MPRB_MPL13_MASK                     0x100u
#define AIPS_MPRB_MPL13_SHIFT                    8u
#define AIPS_MPRB_MPL13_WIDTH                    1u
#define AIPS_MPRB_MPL13(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MPL13_SHIFT))&AIPS_MPRB_MPL13_MASK)
#define AIPS_MPRB_MTW13_MASK                     0x200u
#define AIPS_MPRB_MTW13_SHIFT                    9u
#define AIPS_MPRB_MTW13_WIDTH                    1u
#define AIPS_MPRB_MTW13(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTW13_SHIFT))&AIPS_MPRB_MTW13_MASK)
#define AIPS_MPRB_MTR13_MASK                     0x400u
#define AIPS_MPRB_MTR13_SHIFT                    10u
#define AIPS_MPRB_MTR13_WIDTH                    1u
#define AIPS_MPRB_MTR13(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTR13_SHIFT))&AIPS_MPRB_MTR13_MASK)
#define AIPS_MPRB_MPL12_MASK                     0x1000u
#define AIPS_MPRB_MPL12_SHIFT                    12u
#define AIPS_MPRB_MPL12_WIDTH                    1u
#define AIPS_MPRB_MPL12(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MPL12_SHIFT))&AIPS_MPRB_MPL12_MASK)
#define AIPS_MPRB_MTW12_MASK                     0x2000u
#define AIPS_MPRB_MTW12_SHIFT                    13u
#define AIPS_MPRB_MTW12_WIDTH                    1u
#define AIPS_MPRB_MTW12(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTW12_SHIFT))&AIPS_MPRB_MTW12_MASK)
#define AIPS_MPRB_MTR12_MASK                     0x4000u
#define AIPS_MPRB_MTR12_SHIFT                    14u
#define AIPS_MPRB_MTR12_WIDTH                    1u
#define AIPS_MPRB_MTR12(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTR12_SHIFT))&AIPS_MPRB_MTR12_MASK)
#define AIPS_MPRB_MPL11_MASK                     0x10000u
#define AIPS_MPRB_MPL11_SHIFT                    16u
#define AIPS_MPRB_MPL11_WIDTH                    1u
#define AIPS_MPRB_MPL11(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MPL11_SHIFT))&AIPS_MPRB_MPL11_MASK)
#define AIPS_MPRB_MTW11_MASK                     0x20000u
#define AIPS_MPRB_MTW11_SHIFT                    17u
#define AIPS_MPRB_MTW11_WIDTH                    1u
#define AIPS_MPRB_MTW11(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTW11_SHIFT))&AIPS_MPRB_MTW11_MASK)
#define AIPS_MPRB_MTR11_MASK                     0x40000u
#define AIPS_MPRB_MTR11_SHIFT                    18u
#define AIPS_MPRB_MTR11_WIDTH                    1u
#define AIPS_MPRB_MTR11(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTR11_SHIFT))&AIPS_MPRB_MTR11_MASK)
#define AIPS_MPRB_MPL10_MASK                     0x100000u
#define AIPS_MPRB_MPL10_SHIFT                    20u
#define AIPS_MPRB_MPL10_WIDTH                    1u
#define AIPS_MPRB_MPL10(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MPL10_SHIFT))&AIPS_MPRB_MPL10_MASK)
#define AIPS_MPRB_MTW10_MASK                     0x200000u
#define AIPS_MPRB_MTW10_SHIFT                    21u
#define AIPS_MPRB_MTW10_WIDTH                    1u
#define AIPS_MPRB_MTW10(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTW10_SHIFT))&AIPS_MPRB_MTW10_MASK)
#define AIPS_MPRB_MTR10_MASK                     0x400000u
#define AIPS_MPRB_MTR10_SHIFT                    22u
#define AIPS_MPRB_MTR10_WIDTH                    1u
#define AIPS_MPRB_MTR10(x)                       (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTR10_SHIFT))&AIPS_MPRB_MTR10_MASK)
#define AIPS_MPRB_MPL9_MASK                      0x1000000u
#define AIPS_MPRB_MPL9_SHIFT                     24u
#define AIPS_MPRB_MPL9_WIDTH                     1u
#define AIPS_MPRB_MPL9(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MPL9_SHIFT))&AIPS_MPRB_MPL9_MASK)
#define AIPS_MPRB_MTW9_MASK                      0x2000000u
#define AIPS_MPRB_MTW9_SHIFT                     25u
#define AIPS_MPRB_MTW9_WIDTH                     1u
#define AIPS_MPRB_MTW9(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTW9_SHIFT))&AIPS_MPRB_MTW9_MASK)
#define AIPS_MPRB_MTR9_MASK                      0x4000000u
#define AIPS_MPRB_MTR9_SHIFT                     26u
#define AIPS_MPRB_MTR9_WIDTH                     1u
#define AIPS_MPRB_MTR9(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTR9_SHIFT))&AIPS_MPRB_MTR9_MASK)
#define AIPS_MPRB_MPL8_MASK                      0x10000000u
#define AIPS_MPRB_MPL8_SHIFT                     28u
#define AIPS_MPRB_MPL8_WIDTH                     1u
#define AIPS_MPRB_MPL8(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MPL8_SHIFT))&AIPS_MPRB_MPL8_MASK)
#define AIPS_MPRB_MTW8_MASK                      0x20000000u
#define AIPS_MPRB_MTW8_SHIFT                     29u
#define AIPS_MPRB_MTW8_WIDTH                     1u
#define AIPS_MPRB_MTW8(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTW8_SHIFT))&AIPS_MPRB_MTW8_MASK)
#define AIPS_MPRB_MTR8_MASK                      0x40000000u
#define AIPS_MPRB_MTR8_SHIFT                     30u
#define AIPS_MPRB_MTR8_WIDTH                     1u
#define AIPS_MPRB_MTR8(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_MPRB_MTR8_SHIFT))&AIPS_MPRB_MTR8_MASK)
/* PACR Bit Fields */
#define AIPS_PACR_TP7_MASK                       0x1u
#define AIPS_PACR_TP7_SHIFT                      0u
#define AIPS_PACR_TP7_WIDTH                      1u
#define AIPS_PACR_TP7(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_TP7_SHIFT))&AIPS_PACR_TP7_MASK)
#define AIPS_PACR_WP7_MASK                       0x2u
#define AIPS_PACR_WP7_SHIFT                      1u
#define AIPS_PACR_WP7_WIDTH                      1u
#define AIPS_PACR_WP7(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_WP7_SHIFT))&AIPS_PACR_WP7_MASK)
#define AIPS_PACR_SP7_MASK                       0x4u
#define AIPS_PACR_SP7_SHIFT                      2u
#define AIPS_PACR_SP7_WIDTH                      1u
#define AIPS_PACR_SP7(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_SP7_SHIFT))&AIPS_PACR_SP7_MASK)
#define AIPS_PACR_TP6_MASK                       0x10u
#define AIPS_PACR_TP6_SHIFT                      4u
#define AIPS_PACR_TP6_WIDTH                      1u
#define AIPS_PACR_TP6(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_TP6_SHIFT))&AIPS_PACR_TP6_MASK)
#define AIPS_PACR_WP6_MASK                       0x20u
#define AIPS_PACR_WP6_SHIFT                      5u
#define AIPS_PACR_WP6_WIDTH                      1u
#define AIPS_PACR_WP6(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_WP6_SHIFT))&AIPS_PACR_WP6_MASK)
#define AIPS_PACR_SP6_MASK                       0x40u
#define AIPS_PACR_SP6_SHIFT                      6u
#define AIPS_PACR_SP6_WIDTH                      1u
#define AIPS_PACR_SP6(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_SP6_SHIFT))&AIPS_PACR_SP6_MASK)
#define AIPS_PACR_TP5_MASK                       0x100u
#define AIPS_PACR_TP5_SHIFT                      8u
#define AIPS_PACR_TP5_WIDTH                      1u
#define AIPS_PACR_TP5(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_TP5_SHIFT))&AIPS_PACR_TP5_MASK)
#define AIPS_PACR_WP5_MASK                       0x200u
#define AIPS_PACR_WP5_SHIFT                      9u
#define AIPS_PACR_WP5_WIDTH                      1u
#define AIPS_PACR_WP5(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_WP5_SHIFT))&AIPS_PACR_WP5_MASK)
#define AIPS_PACR_SP5_MASK                       0x400u
#define AIPS_PACR_SP5_SHIFT                      10u
#define AIPS_PACR_SP5_WIDTH                      1u
#define AIPS_PACR_SP5(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_SP5_SHIFT))&AIPS_PACR_SP5_MASK)
#define AIPS_PACR_TP4_MASK                       0x1000u
#define AIPS_PACR_TP4_SHIFT                      12u
#define AIPS_PACR_TP4_WIDTH                      1u
#define AIPS_PACR_TP4(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_TP4_SHIFT))&AIPS_PACR_TP4_MASK)
#define AIPS_PACR_WP4_MASK                       0x2000u
#define AIPS_PACR_WP4_SHIFT                      13u
#define AIPS_PACR_WP4_WIDTH                      1u
#define AIPS_PACR_WP4(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_WP4_SHIFT))&AIPS_PACR_WP4_MASK)
#define AIPS_PACR_SP4_MASK                       0x4000u
#define AIPS_PACR_SP4_SHIFT                      14u
#define AIPS_PACR_SP4_WIDTH                      1u
#define AIPS_PACR_SP4(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_SP4_SHIFT))&AIPS_PACR_SP4_MASK)
#define AIPS_PACR_TP3_MASK                       0x10000u
#define AIPS_PACR_TP3_SHIFT                      16u
#define AIPS_PACR_TP3_WIDTH                      1u
#define AIPS_PACR_TP3(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_TP3_SHIFT))&AIPS_PACR_TP3_MASK)
#define AIPS_PACR_WP3_MASK                       0x20000u
#define AIPS_PACR_WP3_SHIFT                      17u
#define AIPS_PACR_WP3_WIDTH                      1u
#define AIPS_PACR_WP3(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_WP3_SHIFT))&AIPS_PACR_WP3_MASK)
#define AIPS_PACR_SP3_MASK                       0x40000u
#define AIPS_PACR_SP3_SHIFT                      18u
#define AIPS_PACR_SP3_WIDTH                      1u
#define AIPS_PACR_SP3(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_SP3_SHIFT))&AIPS_PACR_SP3_MASK)
#define AIPS_PACR_TP2_MASK                       0x100000u
#define AIPS_PACR_TP2_SHIFT                      20u
#define AIPS_PACR_TP2_WIDTH                      1u
#define AIPS_PACR_TP2(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_TP2_SHIFT))&AIPS_PACR_TP2_MASK)
#define AIPS_PACR_WP2_MASK                       0x200000u
#define AIPS_PACR_WP2_SHIFT                      21u
#define AIPS_PACR_WP2_WIDTH                      1u
#define AIPS_PACR_WP2(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_WP2_SHIFT))&AIPS_PACR_WP2_MASK)
#define AIPS_PACR_SP2_MASK                       0x400000u
#define AIPS_PACR_SP2_SHIFT                      22u
#define AIPS_PACR_SP2_WIDTH                      1u
#define AIPS_PACR_SP2(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_SP2_SHIFT))&AIPS_PACR_SP2_MASK)
#define AIPS_PACR_TP1_MASK                       0x1000000u
#define AIPS_PACR_TP1_SHIFT                      24u
#define AIPS_PACR_TP1_WIDTH                      1u
#define AIPS_PACR_TP1(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_TP1_SHIFT))&AIPS_PACR_TP1_MASK)
#define AIPS_PACR_WP1_MASK                       0x2000000u
#define AIPS_PACR_WP1_SHIFT                      25u
#define AIPS_PACR_WP1_WIDTH                      1u
#define AIPS_PACR_WP1(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_WP1_SHIFT))&AIPS_PACR_WP1_MASK)
#define AIPS_PACR_SP1_MASK                       0x4000000u
#define AIPS_PACR_SP1_SHIFT                      26u
#define AIPS_PACR_SP1_WIDTH                      1u
#define AIPS_PACR_SP1(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_SP1_SHIFT))&AIPS_PACR_SP1_MASK)
#define AIPS_PACR_TP0_MASK                       0x10000000u
#define AIPS_PACR_TP0_SHIFT                      28u
#define AIPS_PACR_TP0_WIDTH                      1u
#define AIPS_PACR_TP0(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_TP0_SHIFT))&AIPS_PACR_TP0_MASK)
#define AIPS_PACR_WP0_MASK                       0x20000000u
#define AIPS_PACR_WP0_SHIFT                      29u
#define AIPS_PACR_WP0_WIDTH                      1u
#define AIPS_PACR_WP0(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_WP0_SHIFT))&AIPS_PACR_WP0_MASK)
#define AIPS_PACR_SP0_MASK                       0x40000000u
#define AIPS_PACR_SP0_SHIFT                      30u
#define AIPS_PACR_SP0_WIDTH                      1u
#define AIPS_PACR_SP0(x)                         (((uint32_t)(((uint32_t)(x))<<AIPS_PACR_SP0_SHIFT))&AIPS_PACR_SP0_MASK)
/* OPACR Bit Fields */
#define AIPS_OPACR_TP7_MASK                      0x1u
#define AIPS_OPACR_TP7_SHIFT                     0u
#define AIPS_OPACR_TP7_WIDTH                     1u
#define AIPS_OPACR_TP7(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_TP7_SHIFT))&AIPS_OPACR_TP7_MASK)
#define AIPS_OPACR_WP7_MASK                      0x2u
#define AIPS_OPACR_WP7_SHIFT                     1u
#define AIPS_OPACR_WP7_WIDTH                     1u
#define AIPS_OPACR_WP7(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_WP7_SHIFT))&AIPS_OPACR_WP7_MASK)
#define AIPS_OPACR_SP7_MASK                      0x4u
#define AIPS_OPACR_SP7_SHIFT                     2u
#define AIPS_OPACR_SP7_WIDTH                     1u
#define AIPS_OPACR_SP7(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_SP7_SHIFT))&AIPS_OPACR_SP7_MASK)
#define AIPS_OPACR_TP6_MASK                      0x10u
#define AIPS_OPACR_TP6_SHIFT                     4u
#define AIPS_OPACR_TP6_WIDTH                     1u
#define AIPS_OPACR_TP6(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_TP6_SHIFT))&AIPS_OPACR_TP6_MASK)
#define AIPS_OPACR_WP6_MASK                      0x20u
#define AIPS_OPACR_WP6_SHIFT                     5u
#define AIPS_OPACR_WP6_WIDTH                     1u
#define AIPS_OPACR_WP6(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_WP6_SHIFT))&AIPS_OPACR_WP6_MASK)
#define AIPS_OPACR_SP6_MASK                      0x40u
#define AIPS_OPACR_SP6_SHIFT                     6u
#define AIPS_OPACR_SP6_WIDTH                     1u
#define AIPS_OPACR_SP6(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_SP6_SHIFT))&AIPS_OPACR_SP6_MASK)
#define AIPS_OPACR_TP5_MASK                      0x100u
#define AIPS_OPACR_TP5_SHIFT                     8u
#define AIPS_OPACR_TP5_WIDTH                     1u
#define AIPS_OPACR_TP5(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_TP5_SHIFT))&AIPS_OPACR_TP5_MASK)
#define AIPS_OPACR_WP5_MASK                      0x200u
#define AIPS_OPACR_WP5_SHIFT                     9u
#define AIPS_OPACR_WP5_WIDTH                     1u
#define AIPS_OPACR_WP5(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_WP5_SHIFT))&AIPS_OPACR_WP5_MASK)
#define AIPS_OPACR_SP5_MASK                      0x400u
#define AIPS_OPACR_SP5_SHIFT                     10u
#define AIPS_OPACR_SP5_WIDTH                     1u
#define AIPS_OPACR_SP5(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_SP5_SHIFT))&AIPS_OPACR_SP5_MASK)
#define AIPS_OPACR_TP4_MASK                      0x1000u
#define AIPS_OPACR_TP4_SHIFT                     12u
#define AIPS_OPACR_TP4_WIDTH                     1u
#define AIPS_OPACR_TP4(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_TP4_SHIFT))&AIPS_OPACR_TP4_MASK)
#define AIPS_OPACR_WP4_MASK                      0x2000u
#define AIPS_OPACR_WP4_SHIFT                     13u
#define AIPS_OPACR_WP4_WIDTH                     1u
#define AIPS_OPACR_WP4(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_WP4_SHIFT))&AIPS_OPACR_WP4_MASK)
#define AIPS_OPACR_SP4_MASK                      0x4000u
#define AIPS_OPACR_SP4_SHIFT                     14u
#define AIPS_OPACR_SP4_WIDTH                     1u
#define AIPS_OPACR_SP4(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_SP4_SHIFT))&AIPS_OPACR_SP4_MASK)
#define AIPS_OPACR_TP3_MASK                      0x10000u
#define AIPS_OPACR_TP3_SHIFT                     16u
#define AIPS_OPACR_TP3_WIDTH                     1u
#define AIPS_OPACR_TP3(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_TP3_SHIFT))&AIPS_OPACR_TP3_MASK)
#define AIPS_OPACR_WP3_MASK                      0x20000u
#define AIPS_OPACR_WP3_SHIFT                     17u
#define AIPS_OPACR_WP3_WIDTH                     1u
#define AIPS_OPACR_WP3(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_WP3_SHIFT))&AIPS_OPACR_WP3_MASK)
#define AIPS_OPACR_SP3_MASK                      0x40000u
#define AIPS_OPACR_SP3_SHIFT                     18u
#define AIPS_OPACR_SP3_WIDTH                     1u
#define AIPS_OPACR_SP3(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_SP3_SHIFT))&AIPS_OPACR_SP3_MASK)
#define AIPS_OPACR_TP2_MASK                      0x100000u
#define AIPS_OPACR_TP2_SHIFT                     20u
#define AIPS_OPACR_TP2_WIDTH                     1u
#define AIPS_OPACR_TP2(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_TP2_SHIFT))&AIPS_OPACR_TP2_MASK)
#define AIPS_OPACR_WP2_MASK                      0x200000u
#define AIPS_OPACR_WP2_SHIFT                     21u
#define AIPS_OPACR_WP2_WIDTH                     1u
#define AIPS_OPACR_WP2(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_WP2_SHIFT))&AIPS_OPACR_WP2_MASK)
#define AIPS_OPACR_SP2_MASK                      0x400000u
#define AIPS_OPACR_SP2_SHIFT                     22u
#define AIPS_OPACR_SP2_WIDTH                     1u
#define AIPS_OPACR_SP2(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_SP2_SHIFT))&AIPS_OPACR_SP2_MASK)
#define AIPS_OPACR_TP1_MASK                      0x1000000u
#define AIPS_OPACR_TP1_SHIFT                     24u
#define AIPS_OPACR_TP1_WIDTH                     1u
#define AIPS_OPACR_TP1(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_TP1_SHIFT))&AIPS_OPACR_TP1_MASK)
#define AIPS_OPACR_WP1_MASK                      0x2000000u
#define AIPS_OPACR_WP1_SHIFT                     25u
#define AIPS_OPACR_WP1_WIDTH                     1u
#define AIPS_OPACR_WP1(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_WP1_SHIFT))&AIPS_OPACR_WP1_MASK)
#define AIPS_OPACR_SP1_MASK                      0x4000000u
#define AIPS_OPACR_SP1_SHIFT                     26u
#define AIPS_OPACR_SP1_WIDTH                     1u
#define AIPS_OPACR_SP1(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_SP1_SHIFT))&AIPS_OPACR_SP1_MASK)
#define AIPS_OPACR_TP0_MASK                      0x10000000u
#define AIPS_OPACR_TP0_SHIFT                     28u
#define AIPS_OPACR_TP0_WIDTH                     1u
#define AIPS_OPACR_TP0(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_TP0_SHIFT))&AIPS_OPACR_TP0_MASK)
#define AIPS_OPACR_WP0_MASK                      0x20000000u
#define AIPS_OPACR_WP0_SHIFT                     29u
#define AIPS_OPACR_WP0_WIDTH                     1u
#define AIPS_OPACR_WP0(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_WP0_SHIFT))&AIPS_OPACR_WP0_MASK)
#define AIPS_OPACR_SP0_MASK                      0x40000000u
#define AIPS_OPACR_SP0_SHIFT                     30u
#define AIPS_OPACR_SP0_WIDTH                     1u
#define AIPS_OPACR_SP0(x)                        (((uint32_t)(((uint32_t)(x))<<AIPS_OPACR_SP0_SHIFT))&AIPS_OPACR_SP0_MASK)

/*!
 * @}
 */ /* end of group AIPS_Register_Masks */


/*!
 * @}
 */ /* end of group AIPS_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- AXBS Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup AXBS_Peripheral_Access_Layer AXBS Peripheral Access Layer
 * @{
 */


/** AXBS - Size of Registers Arrays */
#define AXBS_PORT_COUNT                          8u
#define AXBS_MGPCR_COUNT                         8u

/** AXBS - Register Layout Typedef */
typedef struct {
  struct {                                         /* offset: 0x0, array step: 0x100 */
    __IO uint32_t PRS;                               /**< Priority Registers Slave, array offset: 0x0, array step: 0x100 */
         uint8_t RESERVED_0[12];
    __IO uint32_t CRS;                               /**< Control Register, array offset: 0x10, array step: 0x100 */
         uint8_t RESERVED_1[236];
  } PORT[AXBS_PORT_COUNT];
  struct {                                         /* offset: 0x800, array step: 0x100 */
    __IO uint32_t MGPCR;                             /**< Master General Purpose Control Register, array offset: 0x800, array step: 0x100 */
         uint8_t RESERVED_0[252];
  } MGPCR[AXBS_MGPCR_COUNT];
} AXBS_Type, *AXBS_MemMapPtr;

 /** Number of instances of the AXBS module. */
#define AXBS_INSTANCE_COUNT                      (2u)


/* AXBS - Peripheral instance base addresses */
/** Peripheral AXBS_0 base address */
#define AXBS_0_BASE                              (0xFC004000u)
/** Peripheral AXBS_0 base pointer */
#define AXBS_0                                   ((AXBS_Type *)AXBS_0_BASE)
/** Peripheral AXBS_1 base address */
#define AXBS_1_BASE                              (0xFC008000u)
/** Peripheral AXBS_1 base pointer */
#define AXBS_1                                   ((AXBS_Type *)AXBS_1_BASE)
/** Array initializer of AXBS peripheral base addresses */
#define AXBS_BASE_ADDRS                          { AXBS_0_BASE, AXBS_1_BASE }
/** Array initializer of AXBS peripheral base pointers */
#define AXBS_BASE_PTRS                           { AXBS_0, AXBS_1 }

/* ----------------------------------------------------------------------------
   -- AXBS Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup AXBS_Register_Masks AXBS Register Masks
 * @{
 */

/* PRS Bit Fields */
#define AXBS_PRS_M0_MASK                         0x7u
#define AXBS_PRS_M0_SHIFT                        0u
#define AXBS_PRS_M0_WIDTH                        3u
#define AXBS_PRS_M0(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M0_SHIFT))&AXBS_PRS_M0_MASK)
#define AXBS_PRS_M1_MASK                         0x70u
#define AXBS_PRS_M1_SHIFT                        4u
#define AXBS_PRS_M1_WIDTH                        3u
#define AXBS_PRS_M1(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M1_SHIFT))&AXBS_PRS_M1_MASK)
#define AXBS_PRS_M2_MASK                         0x700u
#define AXBS_PRS_M2_SHIFT                        8u
#define AXBS_PRS_M2_WIDTH                        3u
#define AXBS_PRS_M2(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M2_SHIFT))&AXBS_PRS_M2_MASK)
#define AXBS_PRS_M3_MASK                         0x7000u
#define AXBS_PRS_M3_SHIFT                        12u
#define AXBS_PRS_M3_WIDTH                        3u
#define AXBS_PRS_M3(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M3_SHIFT))&AXBS_PRS_M3_MASK)
#define AXBS_PRS_M4_MASK                         0x70000u
#define AXBS_PRS_M4_SHIFT                        16u
#define AXBS_PRS_M4_WIDTH                        3u
#define AXBS_PRS_M4(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M4_SHIFT))&AXBS_PRS_M4_MASK)
#define AXBS_PRS_M5_MASK                         0x700000u
#define AXBS_PRS_M5_SHIFT                        20u
#define AXBS_PRS_M5_WIDTH                        3u
#define AXBS_PRS_M5(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M5_SHIFT))&AXBS_PRS_M5_MASK)
#define AXBS_PRS_M6_MASK                         0x7000000u
#define AXBS_PRS_M6_SHIFT                        24u
#define AXBS_PRS_M6_WIDTH                        3u
#define AXBS_PRS_M6(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M6_SHIFT))&AXBS_PRS_M6_MASK)
#define AXBS_PRS_M7_MASK                         0x70000000u
#define AXBS_PRS_M7_SHIFT                        28u
#define AXBS_PRS_M7_WIDTH                        3u
#define AXBS_PRS_M7(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_PRS_M7_SHIFT))&AXBS_PRS_M7_MASK)
/* CRS Bit Fields */
#define AXBS_CRS_PARK_MASK                       0x7u
#define AXBS_CRS_PARK_SHIFT                      0u
#define AXBS_CRS_PARK_WIDTH                      3u
#define AXBS_CRS_PARK(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_PARK_SHIFT))&AXBS_CRS_PARK_MASK)
#define AXBS_CRS_PCTL_MASK                       0x30u
#define AXBS_CRS_PCTL_SHIFT                      4u
#define AXBS_CRS_PCTL_WIDTH                      2u
#define AXBS_CRS_PCTL(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_PCTL_SHIFT))&AXBS_CRS_PCTL_MASK)
#define AXBS_CRS_ARB_MASK                        0x300u
#define AXBS_CRS_ARB_SHIFT                       8u
#define AXBS_CRS_ARB_WIDTH                       2u
#define AXBS_CRS_ARB(x)                          (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_ARB_SHIFT))&AXBS_CRS_ARB_MASK)
#define AXBS_CRS_HPE0_MASK                       0x10000u
#define AXBS_CRS_HPE0_SHIFT                      16u
#define AXBS_CRS_HPE0_WIDTH                      1u
#define AXBS_CRS_HPE0(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE0_SHIFT))&AXBS_CRS_HPE0_MASK)
#define AXBS_CRS_HPE1_MASK                       0x20000u
#define AXBS_CRS_HPE1_SHIFT                      17u
#define AXBS_CRS_HPE1_WIDTH                      1u
#define AXBS_CRS_HPE1(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE1_SHIFT))&AXBS_CRS_HPE1_MASK)
#define AXBS_CRS_HPE2_MASK                       0x40000u
#define AXBS_CRS_HPE2_SHIFT                      18u
#define AXBS_CRS_HPE2_WIDTH                      1u
#define AXBS_CRS_HPE2(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE2_SHIFT))&AXBS_CRS_HPE2_MASK)
#define AXBS_CRS_HPE3_MASK                       0x80000u
#define AXBS_CRS_HPE3_SHIFT                      19u
#define AXBS_CRS_HPE3_WIDTH                      1u
#define AXBS_CRS_HPE3(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE3_SHIFT))&AXBS_CRS_HPE3_MASK)
#define AXBS_CRS_HPE4_MASK                       0x100000u
#define AXBS_CRS_HPE4_SHIFT                      20u
#define AXBS_CRS_HPE4_WIDTH                      1u
#define AXBS_CRS_HPE4(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE4_SHIFT))&AXBS_CRS_HPE4_MASK)
#define AXBS_CRS_HPE5_MASK                       0x200000u
#define AXBS_CRS_HPE5_SHIFT                      21u
#define AXBS_CRS_HPE5_WIDTH                      1u
#define AXBS_CRS_HPE5(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE5_SHIFT))&AXBS_CRS_HPE5_MASK)
#define AXBS_CRS_HPE6_MASK                       0x400000u
#define AXBS_CRS_HPE6_SHIFT                      22u
#define AXBS_CRS_HPE6_WIDTH                      1u
#define AXBS_CRS_HPE6(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE6_SHIFT))&AXBS_CRS_HPE6_MASK)
#define AXBS_CRS_HPE7_MASK                       0x800000u
#define AXBS_CRS_HPE7_SHIFT                      23u
#define AXBS_CRS_HPE7_WIDTH                      1u
#define AXBS_CRS_HPE7(x)                         (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HPE7_SHIFT))&AXBS_CRS_HPE7_MASK)
#define AXBS_CRS_HLP_MASK                        0x40000000u
#define AXBS_CRS_HLP_SHIFT                       30u
#define AXBS_CRS_HLP_WIDTH                       1u
#define AXBS_CRS_HLP(x)                          (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_HLP_SHIFT))&AXBS_CRS_HLP_MASK)
#define AXBS_CRS_RO_MASK                         0x80000000u
#define AXBS_CRS_RO_SHIFT                        31u
#define AXBS_CRS_RO_WIDTH                        1u
#define AXBS_CRS_RO(x)                           (((uint32_t)(((uint32_t)(x))<<AXBS_CRS_RO_SHIFT))&AXBS_CRS_RO_MASK)
/* MGPCR Bit Fields */
#define AXBS_MGPCR_AULB_MASK                     0x7u
#define AXBS_MGPCR_AULB_SHIFT                    0u
#define AXBS_MGPCR_AULB_WIDTH                    3u
#define AXBS_MGPCR_AULB(x)                       (((uint32_t)(((uint32_t)(x))<<AXBS_MGPCR_AULB_SHIFT))&AXBS_MGPCR_AULB_MASK)

/*!
 * @}
 */ /* end of group AXBS_Register_Masks */


/*!
 * @}
 */ /* end of group AXBS_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- BCTU Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup BCTU_Peripheral_Access_Layer BCTU Peripheral Access Layer
 * @{
 */


/** BCTU - Size of Registers Arrays */
#define BCTU_TRGCFG__COUNT                       96u
#define BCTU_LISTCHR__COUNT                      16u

/** BCTU - Register Layout Typedef */
typedef struct {
  __IO uint32_t MCR;                               /**< Module Configuration Register, offset: 0x0 */
       uint8_t RESERVED_0[4];
  __IO uint32_t MSR;                               /**< Module Status register, offset: 0x8 */
       uint8_t RESERVED_1[12];
  __IO uint32_t TRGCFG_[BCTU_TRGCFG__COUNT];       /**< Trigger Configuration Register, array offset: 0x18, array step: 0x4 */
       uint8_t RESERVED_2[144];
  __IO uint32_t WRPROT;                            /**< Write Protection register, offset: 0x228 */
  __IO uint32_t SFTRGR1;                           /**< Software Trigger register 1, offset: 0x22C */
  __IO uint32_t SFTRGR2;                           /**< Software Trigger register 2, offset: 0x230 */
  __IO uint32_t SFTRGR3;                           /**< Software Trigger register 3, offset: 0x234 */
       uint8_t RESERVED_3[4];
  __I  uint32_t ADC0DR;                            /**< ADC0 Result Data register, offset: 0x23C */
  __I  uint32_t ADC1DR;                            /**< ADC1 result data register, offset: 0x240 */
       uint8_t RESERVED_4[8];
  __I  uint32_t LISTSTAR;                          /**< LIST Size Status register, offset: 0x24C */
  __IO uint32_t LISTCHR_[BCTU_LISTCHR__COUNT];     /**< LIST Channel Address Register, array offset: 0x250, array step: 0x4 */
} BCTU_Type, *BCTU_MemMapPtr;

 /** Number of instances of the BCTU module. */
#define BCTU_INSTANCE_COUNT                      (1u)


/* BCTU - Peripheral instance base addresses */
/** Peripheral BCTU base address */
#define BCTU_BASE                                (0xFFC10000u)
/** Peripheral BCTU base pointer */
#define BCTU                                     ((BCTU_Type *)BCTU_BASE)
/** Array initializer of BCTU peripheral base addresses */
#define BCTU_BASE_ADDRS                          { BCTU_BASE }
/** Array initializer of BCTU peripheral base pointers */
#define BCTU_BASE_PTRS                           { BCTU }

/* ----------------------------------------------------------------------------
   -- BCTU Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup BCTU_Register_Masks BCTU Register Masks
 * @{
 */

/* MCR Bit Fields */
#define BCTU_MCR_IEN0_MASK                       0x1u
#define BCTU_MCR_IEN0_SHIFT                      0u
#define BCTU_MCR_IEN0_WIDTH                      1u
#define BCTU_MCR_IEN0(x)                         (((uint32_t)(((uint32_t)(x))<<BCTU_MCR_IEN0_SHIFT))&BCTU_MCR_IEN0_MASK)
#define BCTU_MCR_IEN1_MASK                       0x2u
#define BCTU_MCR_IEN1_SHIFT                      1u
#define BCTU_MCR_IEN1_WIDTH                      1u
#define BCTU_MCR_IEN1(x)                         (((uint32_t)(((uint32_t)(x))<<BCTU_MCR_IEN1_SHIFT))&BCTU_MCR_IEN1_MASK)
#define BCTU_MCR_LIST_IEN_MASK                   0x20u
#define BCTU_MCR_LIST_IEN_SHIFT                  5u
#define BCTU_MCR_LIST_IEN_WIDTH                  1u
#define BCTU_MCR_LIST_IEN(x)                     (((uint32_t)(((uint32_t)(x))<<BCTU_MCR_LIST_IEN_SHIFT))&BCTU_MCR_LIST_IEN_MASK)
#define BCTU_MCR_TRGEN_MASK                      0x80u
#define BCTU_MCR_TRGEN_SHIFT                     7u
#define BCTU_MCR_TRGEN_WIDTH                     1u
#define BCTU_MCR_TRGEN(x)                        (((uint32_t)(((uint32_t)(x))<<BCTU_MCR_TRGEN_SHIFT))&BCTU_MCR_TRGEN_MASK)
#define BCTU_MCR_DMA0_MASK                       0x10000u
#define BCTU_MCR_DMA0_SHIFT                      16u
#define BCTU_MCR_DMA0_WIDTH                      1u
#define BCTU_MCR_DMA0(x)                         (((uint32_t)(((uint32_t)(x))<<BCTU_MCR_DMA0_SHIFT))&BCTU_MCR_DMA0_MASK)
#define BCTU_MCR_DMA1_MASK                       0x20000u
#define BCTU_MCR_DMA1_SHIFT                      17u
#define BCTU_MCR_DMA1_WIDTH                      1u
#define BCTU_MCR_DMA1(x)                         (((uint32_t)(((uint32_t)(x))<<BCTU_MCR_DMA1_SHIFT))&BCTU_MCR_DMA1_MASK)
#define BCTU_MCR_GTRGEN_MASK                     0x4000000u
#define BCTU_MCR_GTRGEN_SHIFT                    26u
#define BCTU_MCR_GTRGEN_WIDTH                    1u
#define BCTU_MCR_GTRGEN(x)                       (((uint32_t)(((uint32_t)(x))<<BCTU_MCR_GTRGEN_SHIFT))&BCTU_MCR_GTRGEN_MASK)
#define BCTU_MCR_FRZ_MASK                        0x20000000u
#define BCTU_MCR_FRZ_SHIFT                       29u
#define BCTU_MCR_FRZ_WIDTH                       1u
#define BCTU_MCR_FRZ(x)                          (((uint32_t)(((uint32_t)(x))<<BCTU_MCR_FRZ_SHIFT))&BCTU_MCR_FRZ_MASK)
#define BCTU_MCR_MDIS_MASK                       0x40000000u
#define BCTU_MCR_MDIS_SHIFT                      30u
#define BCTU_MCR_MDIS_WIDTH                      1u
#define BCTU_MCR_MDIS(x)                         (((uint32_t)(((uint32_t)(x))<<BCTU_MCR_MDIS_SHIFT))&BCTU_MCR_MDIS_MASK)
/* MSR Bit Fields */
#define BCTU_MSR_NDATA0_MASK                     0x1u
#define BCTU_MSR_NDATA0_SHIFT                    0u
#define BCTU_MSR_NDATA0_WIDTH                    1u
#define BCTU_MSR_NDATA0(x)                       (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_NDATA0_SHIFT))&BCTU_MSR_NDATA0_MASK)
#define BCTU_MSR_NDATA1_MASK                     0x2u
#define BCTU_MSR_NDATA1_SHIFT                    1u
#define BCTU_MSR_NDATA1_WIDTH                    1u
#define BCTU_MSR_NDATA1(x)                       (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_NDATA1_SHIFT))&BCTU_MSR_NDATA1_MASK)
#define BCTU_MSR_DATAOVR0_MASK                   0x10u
#define BCTU_MSR_DATAOVR0_SHIFT                  4u
#define BCTU_MSR_DATAOVR0_WIDTH                  1u
#define BCTU_MSR_DATAOVR0(x)                     (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_DATAOVR0_SHIFT))&BCTU_MSR_DATAOVR0_MASK)
#define BCTU_MSR_DATAOVR1_MASK                   0x20u
#define BCTU_MSR_DATAOVR1_SHIFT                  5u
#define BCTU_MSR_DATAOVR1_WIDTH                  1u
#define BCTU_MSR_DATAOVR1(x)                     (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_DATAOVR1_SHIFT))&BCTU_MSR_DATAOVR1_MASK)
#define BCTU_MSR_LIST0_Last_MASK                 0x100u
#define BCTU_MSR_LIST0_Last_SHIFT                8u
#define BCTU_MSR_LIST0_Last_WIDTH                1u
#define BCTU_MSR_LIST0_Last(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_LIST0_Last_SHIFT))&BCTU_MSR_LIST0_Last_MASK)
#define BCTU_MSR_LIST1_Last_MASK                 0x200u
#define BCTU_MSR_LIST1_Last_SHIFT                9u
#define BCTU_MSR_LIST1_Last_WIDTH                1u
#define BCTU_MSR_LIST1_Last(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_LIST1_Last_SHIFT))&BCTU_MSR_LIST1_Last_MASK)
#define BCTU_MSR_TRGF_MASK                       0x8000u
#define BCTU_MSR_TRGF_SHIFT                      15u
#define BCTU_MSR_TRGF_WIDTH                      1u
#define BCTU_MSR_TRGF(x)                         (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_TRGF_SHIFT))&BCTU_MSR_TRGF_MASK)
#define BCTU_MSR_NDATA0_CLR_MASK                 0x10000u
#define BCTU_MSR_NDATA0_CLR_SHIFT                16u
#define BCTU_MSR_NDATA0_CLR_WIDTH                1u
#define BCTU_MSR_NDATA0_CLR(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_NDATA0_CLR_SHIFT))&BCTU_MSR_NDATA0_CLR_MASK)
#define BCTU_MSR_NDATA1_CLR_MASK                 0x20000u
#define BCTU_MSR_NDATA1_CLR_SHIFT                17u
#define BCTU_MSR_NDATA1_CLR_WIDTH                1u
#define BCTU_MSR_NDATA1_CLR(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_NDATA1_CLR_SHIFT))&BCTU_MSR_NDATA1_CLR_MASK)
#define BCTU_MSR_DATAOVR0_CLR_MASK               0x100000u
#define BCTU_MSR_DATAOVR0_CLR_SHIFT              20u
#define BCTU_MSR_DATAOVR0_CLR_WIDTH              1u
#define BCTU_MSR_DATAOVR0_CLR(x)                 (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_DATAOVR0_CLR_SHIFT))&BCTU_MSR_DATAOVR0_CLR_MASK)
#define BCTU_MSR_DATAOVR1_CLR_MASK               0x200000u
#define BCTU_MSR_DATAOVR1_CLR_SHIFT              21u
#define BCTU_MSR_DATAOVR1_CLR_WIDTH              1u
#define BCTU_MSR_DATAOVR1_CLR(x)                 (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_DATAOVR1_CLR_SHIFT))&BCTU_MSR_DATAOVR1_CLR_MASK)
#define BCTU_MSR_LIST0_Last_CLR_MASK             0x1000000u
#define BCTU_MSR_LIST0_Last_CLR_SHIFT            24u
#define BCTU_MSR_LIST0_Last_CLR_WIDTH            1u
#define BCTU_MSR_LIST0_Last_CLR(x)               (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_LIST0_Last_CLR_SHIFT))&BCTU_MSR_LIST0_Last_CLR_MASK)
#define BCTU_MSR_LIST1_Last_CLR_MASK             0x2000000u
#define BCTU_MSR_LIST1_Last_CLR_SHIFT            25u
#define BCTU_MSR_LIST1_Last_CLR_WIDTH            1u
#define BCTU_MSR_LIST1_Last_CLR(x)               (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_LIST1_Last_CLR_SHIFT))&BCTU_MSR_LIST1_Last_CLR_MASK)
#define BCTU_MSR_TRGF_CLR_MASK                   0x80000000u
#define BCTU_MSR_TRGF_CLR_SHIFT                  31u
#define BCTU_MSR_TRGF_CLR_WIDTH                  1u
#define BCTU_MSR_TRGF_CLR(x)                     (((uint32_t)(((uint32_t)(x))<<BCTU_MSR_TRGF_CLR_SHIFT))&BCTU_MSR_TRGF_CLR_MASK)
/* TRGCFG_ Bit Fields */
#define BCTU_TRGCFG__CHANNEL_VALUE_OR_LADDR_MASK 0x7Fu
#define BCTU_TRGCFG__CHANNEL_VALUE_OR_LADDR_SHIFT 0u
#define BCTU_TRGCFG__CHANNEL_VALUE_OR_LADDR_WIDTH 7u
#define BCTU_TRGCFG__CHANNEL_VALUE_OR_LADDR(x)   (((uint32_t)(((uint32_t)(x))<<BCTU_TRGCFG__CHANNEL_VALUE_OR_LADDR_SHIFT))&BCTU_TRGCFG__CHANNEL_VALUE_OR_LADDR_MASK)
#define BCTU_TRGCFG__ADC_SEL0_MASK               0x100u
#define BCTU_TRGCFG__ADC_SEL0_SHIFT              8u
#define BCTU_TRGCFG__ADC_SEL0_WIDTH              1u
#define BCTU_TRGCFG__ADC_SEL0(x)                 (((uint32_t)(((uint32_t)(x))<<BCTU_TRGCFG__ADC_SEL0_SHIFT))&BCTU_TRGCFG__ADC_SEL0_MASK)
#define BCTU_TRGCFG__ADC_SEL1_MASK               0x200u
#define BCTU_TRGCFG__ADC_SEL1_SHIFT              9u
#define BCTU_TRGCFG__ADC_SEL1_WIDTH              1u
#define BCTU_TRGCFG__ADC_SEL1(x)                 (((uint32_t)(((uint32_t)(x))<<BCTU_TRGCFG__ADC_SEL1_SHIFT))&BCTU_TRGCFG__ADC_SEL1_MASK)
#define BCTU_TRGCFG__TRS_MASK                    0x2000u
#define BCTU_TRGCFG__TRS_SHIFT                   13u
#define BCTU_TRGCFG__TRS_WIDTH                   1u
#define BCTU_TRGCFG__TRS(x)                      (((uint32_t)(((uint32_t)(x))<<BCTU_TRGCFG__TRS_SHIFT))&BCTU_TRGCFG__TRS_MASK)
#define BCTU_TRGCFG__TRG_FLAG_MASK               0x4000u
#define BCTU_TRGCFG__TRG_FLAG_SHIFT              14u
#define BCTU_TRGCFG__TRG_FLAG_WIDTH              1u
#define BCTU_TRGCFG__TRG_FLAG(x)                 (((uint32_t)(((uint32_t)(x))<<BCTU_TRGCFG__TRG_FLAG_SHIFT))&BCTU_TRGCFG__TRG_FLAG_MASK)
#define BCTU_TRGCFG__TRIGEN_MASK                 0x8000u
#define BCTU_TRGCFG__TRIGEN_SHIFT                15u
#define BCTU_TRGCFG__TRIGEN_WIDTH                1u
#define BCTU_TRGCFG__TRIGEN(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_TRGCFG__TRIGEN_SHIFT))&BCTU_TRGCFG__TRIGEN_MASK)
#define BCTU_TRGCFG__LOOP_MASK                   0x80000000u
#define BCTU_TRGCFG__LOOP_SHIFT                  31u
#define BCTU_TRGCFG__LOOP_WIDTH                  1u
#define BCTU_TRGCFG__LOOP(x)                     (((uint32_t)(((uint32_t)(x))<<BCTU_TRGCFG__LOOP_SHIFT))&BCTU_TRGCFG__LOOP_MASK)
/* WRPROT Bit Fields */
#define BCTU_WRPROT_PROTEC_CODE_MASK             0xFu
#define BCTU_WRPROT_PROTEC_CODE_SHIFT            0u
#define BCTU_WRPROT_PROTEC_CODE_WIDTH            4u
#define BCTU_WRPROT_PROTEC_CODE(x)               (((uint32_t)(((uint32_t)(x))<<BCTU_WRPROT_PROTEC_CODE_SHIFT))&BCTU_WRPROT_PROTEC_CODE_MASK)
/* SFTRGR1 Bit Fields */
#define BCTU_SFTRGR1_SFTRG0_MASK                 0x1u
#define BCTU_SFTRGR1_SFTRG0_SHIFT                0u
#define BCTU_SFTRGR1_SFTRG0_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG0(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG0_SHIFT))&BCTU_SFTRGR1_SFTRG0_MASK)
#define BCTU_SFTRGR1_SFTRG1_MASK                 0x2u
#define BCTU_SFTRGR1_SFTRG1_SHIFT                1u
#define BCTU_SFTRGR1_SFTRG1_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG1(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG1_SHIFT))&BCTU_SFTRGR1_SFTRG1_MASK)
#define BCTU_SFTRGR1_SFTRG2_MASK                 0x4u
#define BCTU_SFTRGR1_SFTRG2_SHIFT                2u
#define BCTU_SFTRGR1_SFTRG2_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG2(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG2_SHIFT))&BCTU_SFTRGR1_SFTRG2_MASK)
#define BCTU_SFTRGR1_SFTRG3_MASK                 0x8u
#define BCTU_SFTRGR1_SFTRG3_SHIFT                3u
#define BCTU_SFTRGR1_SFTRG3_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG3(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG3_SHIFT))&BCTU_SFTRGR1_SFTRG3_MASK)
#define BCTU_SFTRGR1_SFTRG4_MASK                 0x10u
#define BCTU_SFTRGR1_SFTRG4_SHIFT                4u
#define BCTU_SFTRGR1_SFTRG4_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG4(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG4_SHIFT))&BCTU_SFTRGR1_SFTRG4_MASK)
#define BCTU_SFTRGR1_SFTRG5_MASK                 0x20u
#define BCTU_SFTRGR1_SFTRG5_SHIFT                5u
#define BCTU_SFTRGR1_SFTRG5_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG5(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG5_SHIFT))&BCTU_SFTRGR1_SFTRG5_MASK)
#define BCTU_SFTRGR1_SFTRG6_MASK                 0x40u
#define BCTU_SFTRGR1_SFTRG6_SHIFT                6u
#define BCTU_SFTRGR1_SFTRG6_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG6(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG6_SHIFT))&BCTU_SFTRGR1_SFTRG6_MASK)
#define BCTU_SFTRGR1_SFTRG7_MASK                 0x80u
#define BCTU_SFTRGR1_SFTRG7_SHIFT                7u
#define BCTU_SFTRGR1_SFTRG7_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG7(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG7_SHIFT))&BCTU_SFTRGR1_SFTRG7_MASK)
#define BCTU_SFTRGR1_SFTRG8_MASK                 0x100u
#define BCTU_SFTRGR1_SFTRG8_SHIFT                8u
#define BCTU_SFTRGR1_SFTRG8_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG8(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG8_SHIFT))&BCTU_SFTRGR1_SFTRG8_MASK)
#define BCTU_SFTRGR1_SFTRG9_MASK                 0x200u
#define BCTU_SFTRGR1_SFTRG9_SHIFT                9u
#define BCTU_SFTRGR1_SFTRG9_WIDTH                1u
#define BCTU_SFTRGR1_SFTRG9(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG9_SHIFT))&BCTU_SFTRGR1_SFTRG9_MASK)
#define BCTU_SFTRGR1_SFTRG10_MASK                0x400u
#define BCTU_SFTRGR1_SFTRG10_SHIFT               10u
#define BCTU_SFTRGR1_SFTRG10_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG10(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG10_SHIFT))&BCTU_SFTRGR1_SFTRG10_MASK)
#define BCTU_SFTRGR1_SFTRG11_MASK                0x800u
#define BCTU_SFTRGR1_SFTRG11_SHIFT               11u
#define BCTU_SFTRGR1_SFTRG11_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG11(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG11_SHIFT))&BCTU_SFTRGR1_SFTRG11_MASK)
#define BCTU_SFTRGR1_SFTRG12_MASK                0x1000u
#define BCTU_SFTRGR1_SFTRG12_SHIFT               12u
#define BCTU_SFTRGR1_SFTRG12_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG12(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG12_SHIFT))&BCTU_SFTRGR1_SFTRG12_MASK)
#define BCTU_SFTRGR1_SFTRG13_MASK                0x2000u
#define BCTU_SFTRGR1_SFTRG13_SHIFT               13u
#define BCTU_SFTRGR1_SFTRG13_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG13(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG13_SHIFT))&BCTU_SFTRGR1_SFTRG13_MASK)
#define BCTU_SFTRGR1_SFTRG14_MASK                0x4000u
#define BCTU_SFTRGR1_SFTRG14_SHIFT               14u
#define BCTU_SFTRGR1_SFTRG14_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG14(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG14_SHIFT))&BCTU_SFTRGR1_SFTRG14_MASK)
#define BCTU_SFTRGR1_SFTRG15_MASK                0x8000u
#define BCTU_SFTRGR1_SFTRG15_SHIFT               15u
#define BCTU_SFTRGR1_SFTRG15_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG15(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG15_SHIFT))&BCTU_SFTRGR1_SFTRG15_MASK)
#define BCTU_SFTRGR1_SFTRG16_MASK                0x10000u
#define BCTU_SFTRGR1_SFTRG16_SHIFT               16u
#define BCTU_SFTRGR1_SFTRG16_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG16(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG16_SHIFT))&BCTU_SFTRGR1_SFTRG16_MASK)
#define BCTU_SFTRGR1_SFTRG17_MASK                0x20000u
#define BCTU_SFTRGR1_SFTRG17_SHIFT               17u
#define BCTU_SFTRGR1_SFTRG17_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG17(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG17_SHIFT))&BCTU_SFTRGR1_SFTRG17_MASK)
#define BCTU_SFTRGR1_SFTRG18_MASK                0x40000u
#define BCTU_SFTRGR1_SFTRG18_SHIFT               18u
#define BCTU_SFTRGR1_SFTRG18_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG18(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG18_SHIFT))&BCTU_SFTRGR1_SFTRG18_MASK)
#define BCTU_SFTRGR1_SFTRG19_MASK                0x80000u
#define BCTU_SFTRGR1_SFTRG19_SHIFT               19u
#define BCTU_SFTRGR1_SFTRG19_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG19(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG19_SHIFT))&BCTU_SFTRGR1_SFTRG19_MASK)
#define BCTU_SFTRGR1_SFTRG20_MASK                0x100000u
#define BCTU_SFTRGR1_SFTRG20_SHIFT               20u
#define BCTU_SFTRGR1_SFTRG20_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG20(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG20_SHIFT))&BCTU_SFTRGR1_SFTRG20_MASK)
#define BCTU_SFTRGR1_SFTRG21_MASK                0x200000u
#define BCTU_SFTRGR1_SFTRG21_SHIFT               21u
#define BCTU_SFTRGR1_SFTRG21_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG21(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG21_SHIFT))&BCTU_SFTRGR1_SFTRG21_MASK)
#define BCTU_SFTRGR1_SFTRG22_MASK                0x400000u
#define BCTU_SFTRGR1_SFTRG22_SHIFT               22u
#define BCTU_SFTRGR1_SFTRG22_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG22(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG22_SHIFT))&BCTU_SFTRGR1_SFTRG22_MASK)
#define BCTU_SFTRGR1_SFTRG23_MASK                0x800000u
#define BCTU_SFTRGR1_SFTRG23_SHIFT               23u
#define BCTU_SFTRGR1_SFTRG23_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG23(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG23_SHIFT))&BCTU_SFTRGR1_SFTRG23_MASK)
#define BCTU_SFTRGR1_SFTRG24_MASK                0x1000000u
#define BCTU_SFTRGR1_SFTRG24_SHIFT               24u
#define BCTU_SFTRGR1_SFTRG24_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG24(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG24_SHIFT))&BCTU_SFTRGR1_SFTRG24_MASK)
#define BCTU_SFTRGR1_SFTRG25_MASK                0x2000000u
#define BCTU_SFTRGR1_SFTRG25_SHIFT               25u
#define BCTU_SFTRGR1_SFTRG25_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG25(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG25_SHIFT))&BCTU_SFTRGR1_SFTRG25_MASK)
#define BCTU_SFTRGR1_SFTRG26_MASK                0x4000000u
#define BCTU_SFTRGR1_SFTRG26_SHIFT               26u
#define BCTU_SFTRGR1_SFTRG26_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG26(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG26_SHIFT))&BCTU_SFTRGR1_SFTRG26_MASK)
#define BCTU_SFTRGR1_SFTRG27_MASK                0x8000000u
#define BCTU_SFTRGR1_SFTRG27_SHIFT               27u
#define BCTU_SFTRGR1_SFTRG27_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG27(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG27_SHIFT))&BCTU_SFTRGR1_SFTRG27_MASK)
#define BCTU_SFTRGR1_SFTRG28_MASK                0x10000000u
#define BCTU_SFTRGR1_SFTRG28_SHIFT               28u
#define BCTU_SFTRGR1_SFTRG28_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG28(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG28_SHIFT))&BCTU_SFTRGR1_SFTRG28_MASK)
#define BCTU_SFTRGR1_SFTRG29_MASK                0x20000000u
#define BCTU_SFTRGR1_SFTRG29_SHIFT               29u
#define BCTU_SFTRGR1_SFTRG29_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG29(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG29_SHIFT))&BCTU_SFTRGR1_SFTRG29_MASK)
#define BCTU_SFTRGR1_SFTRG30_MASK                0x40000000u
#define BCTU_SFTRGR1_SFTRG30_SHIFT               30u
#define BCTU_SFTRGR1_SFTRG30_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG30(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG30_SHIFT))&BCTU_SFTRGR1_SFTRG30_MASK)
#define BCTU_SFTRGR1_SFTRG31_MASK                0x80000000u
#define BCTU_SFTRGR1_SFTRG31_SHIFT               31u
#define BCTU_SFTRGR1_SFTRG31_WIDTH               1u
#define BCTU_SFTRGR1_SFTRG31(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR1_SFTRG31_SHIFT))&BCTU_SFTRGR1_SFTRG31_MASK)
/* SFTRGR2 Bit Fields */
#define BCTU_SFTRGR2_SFTRG32_MASK                0x1u
#define BCTU_SFTRGR2_SFTRG32_SHIFT               0u
#define BCTU_SFTRGR2_SFTRG32_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG32(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG32_SHIFT))&BCTU_SFTRGR2_SFTRG32_MASK)
#define BCTU_SFTRGR2_SFTRG33_MASK                0x2u
#define BCTU_SFTRGR2_SFTRG33_SHIFT               1u
#define BCTU_SFTRGR2_SFTRG33_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG33(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG33_SHIFT))&BCTU_SFTRGR2_SFTRG33_MASK)
#define BCTU_SFTRGR2_SFTRG34_MASK                0x4u
#define BCTU_SFTRGR2_SFTRG34_SHIFT               2u
#define BCTU_SFTRGR2_SFTRG34_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG34(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG34_SHIFT))&BCTU_SFTRGR2_SFTRG34_MASK)
#define BCTU_SFTRGR2_SFTRG35_MASK                0x8u
#define BCTU_SFTRGR2_SFTRG35_SHIFT               3u
#define BCTU_SFTRGR2_SFTRG35_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG35(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG35_SHIFT))&BCTU_SFTRGR2_SFTRG35_MASK)
#define BCTU_SFTRGR2_SFTRG36_MASK                0x10u
#define BCTU_SFTRGR2_SFTRG36_SHIFT               4u
#define BCTU_SFTRGR2_SFTRG36_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG36(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG36_SHIFT))&BCTU_SFTRGR2_SFTRG36_MASK)
#define BCTU_SFTRGR2_SFTRG37_MASK                0x20u
#define BCTU_SFTRGR2_SFTRG37_SHIFT               5u
#define BCTU_SFTRGR2_SFTRG37_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG37(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG37_SHIFT))&BCTU_SFTRGR2_SFTRG37_MASK)
#define BCTU_SFTRGR2_SFTRG38_MASK                0x40u
#define BCTU_SFTRGR2_SFTRG38_SHIFT               6u
#define BCTU_SFTRGR2_SFTRG38_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG38(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG38_SHIFT))&BCTU_SFTRGR2_SFTRG38_MASK)
#define BCTU_SFTRGR2_SFTRG39_MASK                0x80u
#define BCTU_SFTRGR2_SFTRG39_SHIFT               7u
#define BCTU_SFTRGR2_SFTRG39_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG39(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG39_SHIFT))&BCTU_SFTRGR2_SFTRG39_MASK)
#define BCTU_SFTRGR2_SFTRG40_MASK                0x100u
#define BCTU_SFTRGR2_SFTRG40_SHIFT               8u
#define BCTU_SFTRGR2_SFTRG40_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG40(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG40_SHIFT))&BCTU_SFTRGR2_SFTRG40_MASK)
#define BCTU_SFTRGR2_SFTRG41_MASK                0x200u
#define BCTU_SFTRGR2_SFTRG41_SHIFT               9u
#define BCTU_SFTRGR2_SFTRG41_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG41(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG41_SHIFT))&BCTU_SFTRGR2_SFTRG41_MASK)
#define BCTU_SFTRGR2_SFTRG42_MASK                0x400u
#define BCTU_SFTRGR2_SFTRG42_SHIFT               10u
#define BCTU_SFTRGR2_SFTRG42_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG42(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG42_SHIFT))&BCTU_SFTRGR2_SFTRG42_MASK)
#define BCTU_SFTRGR2_SFTRG43_MASK                0x800u
#define BCTU_SFTRGR2_SFTRG43_SHIFT               11u
#define BCTU_SFTRGR2_SFTRG43_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG43(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG43_SHIFT))&BCTU_SFTRGR2_SFTRG43_MASK)
#define BCTU_SFTRGR2_SFTRG44_MASK                0x1000u
#define BCTU_SFTRGR2_SFTRG44_SHIFT               12u
#define BCTU_SFTRGR2_SFTRG44_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG44(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG44_SHIFT))&BCTU_SFTRGR2_SFTRG44_MASK)
#define BCTU_SFTRGR2_SFTRG45_MASK                0x2000u
#define BCTU_SFTRGR2_SFTRG45_SHIFT               13u
#define BCTU_SFTRGR2_SFTRG45_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG45(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG45_SHIFT))&BCTU_SFTRGR2_SFTRG45_MASK)
#define BCTU_SFTRGR2_SFTRG46_MASK                0x4000u
#define BCTU_SFTRGR2_SFTRG46_SHIFT               14u
#define BCTU_SFTRGR2_SFTRG46_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG46(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG46_SHIFT))&BCTU_SFTRGR2_SFTRG46_MASK)
#define BCTU_SFTRGR2_SFTRG47_MASK                0x8000u
#define BCTU_SFTRGR2_SFTRG47_SHIFT               15u
#define BCTU_SFTRGR2_SFTRG47_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG47(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG47_SHIFT))&BCTU_SFTRGR2_SFTRG47_MASK)
#define BCTU_SFTRGR2_SFTRG48_MASK                0x10000u
#define BCTU_SFTRGR2_SFTRG48_SHIFT               16u
#define BCTU_SFTRGR2_SFTRG48_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG48(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG48_SHIFT))&BCTU_SFTRGR2_SFTRG48_MASK)
#define BCTU_SFTRGR2_SFTRG49_MASK                0x20000u
#define BCTU_SFTRGR2_SFTRG49_SHIFT               17u
#define BCTU_SFTRGR2_SFTRG49_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG49(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG49_SHIFT))&BCTU_SFTRGR2_SFTRG49_MASK)
#define BCTU_SFTRGR2_SFTRG50_MASK                0x40000u
#define BCTU_SFTRGR2_SFTRG50_SHIFT               18u
#define BCTU_SFTRGR2_SFTRG50_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG50(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG50_SHIFT))&BCTU_SFTRGR2_SFTRG50_MASK)
#define BCTU_SFTRGR2_SFTRG51_MASK                0x80000u
#define BCTU_SFTRGR2_SFTRG51_SHIFT               19u
#define BCTU_SFTRGR2_SFTRG51_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG51(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG51_SHIFT))&BCTU_SFTRGR2_SFTRG51_MASK)
#define BCTU_SFTRGR2_SFTRG52_MASK                0x100000u
#define BCTU_SFTRGR2_SFTRG52_SHIFT               20u
#define BCTU_SFTRGR2_SFTRG52_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG52(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG52_SHIFT))&BCTU_SFTRGR2_SFTRG52_MASK)
#define BCTU_SFTRGR2_SFTRG53_MASK                0x200000u
#define BCTU_SFTRGR2_SFTRG53_SHIFT               21u
#define BCTU_SFTRGR2_SFTRG53_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG53(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG53_SHIFT))&BCTU_SFTRGR2_SFTRG53_MASK)
#define BCTU_SFTRGR2_SFTRG54_MASK                0x400000u
#define BCTU_SFTRGR2_SFTRG54_SHIFT               22u
#define BCTU_SFTRGR2_SFTRG54_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG54(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG54_SHIFT))&BCTU_SFTRGR2_SFTRG54_MASK)
#define BCTU_SFTRGR2_SFTRG55_MASK                0x800000u
#define BCTU_SFTRGR2_SFTRG55_SHIFT               23u
#define BCTU_SFTRGR2_SFTRG55_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG55(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG55_SHIFT))&BCTU_SFTRGR2_SFTRG55_MASK)
#define BCTU_SFTRGR2_SFTRG56_MASK                0x1000000u
#define BCTU_SFTRGR2_SFTRG56_SHIFT               24u
#define BCTU_SFTRGR2_SFTRG56_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG56(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG56_SHIFT))&BCTU_SFTRGR2_SFTRG56_MASK)
#define BCTU_SFTRGR2_SFTRG57_MASK                0x2000000u
#define BCTU_SFTRGR2_SFTRG57_SHIFT               25u
#define BCTU_SFTRGR2_SFTRG57_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG57(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG57_SHIFT))&BCTU_SFTRGR2_SFTRG57_MASK)
#define BCTU_SFTRGR2_SFTRG58_MASK                0x4000000u
#define BCTU_SFTRGR2_SFTRG58_SHIFT               26u
#define BCTU_SFTRGR2_SFTRG58_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG58(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG58_SHIFT))&BCTU_SFTRGR2_SFTRG58_MASK)
#define BCTU_SFTRGR2_SFTRG59_MASK                0x8000000u
#define BCTU_SFTRGR2_SFTRG59_SHIFT               27u
#define BCTU_SFTRGR2_SFTRG59_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG59(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG59_SHIFT))&BCTU_SFTRGR2_SFTRG59_MASK)
#define BCTU_SFTRGR2_SFTRG60_MASK                0x10000000u
#define BCTU_SFTRGR2_SFTRG60_SHIFT               28u
#define BCTU_SFTRGR2_SFTRG60_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG60(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG60_SHIFT))&BCTU_SFTRGR2_SFTRG60_MASK)
#define BCTU_SFTRGR2_SFTRG61_MASK                0x20000000u
#define BCTU_SFTRGR2_SFTRG61_SHIFT               29u
#define BCTU_SFTRGR2_SFTRG61_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG61(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG61_SHIFT))&BCTU_SFTRGR2_SFTRG61_MASK)
#define BCTU_SFTRGR2_SFTRG62_MASK                0x40000000u
#define BCTU_SFTRGR2_SFTRG62_SHIFT               30u
#define BCTU_SFTRGR2_SFTRG62_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG62(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG62_SHIFT))&BCTU_SFTRGR2_SFTRG62_MASK)
#define BCTU_SFTRGR2_SFTRG63_MASK                0x80000000u
#define BCTU_SFTRGR2_SFTRG63_SHIFT               31u
#define BCTU_SFTRGR2_SFTRG63_WIDTH               1u
#define BCTU_SFTRGR2_SFTRG63(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR2_SFTRG63_SHIFT))&BCTU_SFTRGR2_SFTRG63_MASK)
/* SFTRGR3 Bit Fields */
#define BCTU_SFTRGR3_SFTRG64_MASK                0x1u
#define BCTU_SFTRGR3_SFTRG64_SHIFT               0u
#define BCTU_SFTRGR3_SFTRG64_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG64(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG64_SHIFT))&BCTU_SFTRGR3_SFTRG64_MASK)
#define BCTU_SFTRGR3_SFTRG65_MASK                0x2u
#define BCTU_SFTRGR3_SFTRG65_SHIFT               1u
#define BCTU_SFTRGR3_SFTRG65_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG65(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG65_SHIFT))&BCTU_SFTRGR3_SFTRG65_MASK)
#define BCTU_SFTRGR3_SFTRG66_MASK                0x4u
#define BCTU_SFTRGR3_SFTRG66_SHIFT               2u
#define BCTU_SFTRGR3_SFTRG66_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG66(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG66_SHIFT))&BCTU_SFTRGR3_SFTRG66_MASK)
#define BCTU_SFTRGR3_SFTRG67_MASK                0x8u
#define BCTU_SFTRGR3_SFTRG67_SHIFT               3u
#define BCTU_SFTRGR3_SFTRG67_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG67(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG67_SHIFT))&BCTU_SFTRGR3_SFTRG67_MASK)
#define BCTU_SFTRGR3_SFTRG68_MASK                0x10u
#define BCTU_SFTRGR3_SFTRG68_SHIFT               4u
#define BCTU_SFTRGR3_SFTRG68_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG68(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG68_SHIFT))&BCTU_SFTRGR3_SFTRG68_MASK)
#define BCTU_SFTRGR3_SFTRG69_MASK                0x20u
#define BCTU_SFTRGR3_SFTRG69_SHIFT               5u
#define BCTU_SFTRGR3_SFTRG69_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG69(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG69_SHIFT))&BCTU_SFTRGR3_SFTRG69_MASK)
#define BCTU_SFTRGR3_SFTRG70_MASK                0x40u
#define BCTU_SFTRGR3_SFTRG70_SHIFT               6u
#define BCTU_SFTRGR3_SFTRG70_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG70(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG70_SHIFT))&BCTU_SFTRGR3_SFTRG70_MASK)
#define BCTU_SFTRGR3_SFTRG71_MASK                0x80u
#define BCTU_SFTRGR3_SFTRG71_SHIFT               7u
#define BCTU_SFTRGR3_SFTRG71_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG71(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG71_SHIFT))&BCTU_SFTRGR3_SFTRG71_MASK)
#define BCTU_SFTRGR3_SFTRG72_MASK                0x100u
#define BCTU_SFTRGR3_SFTRG72_SHIFT               8u
#define BCTU_SFTRGR3_SFTRG72_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG72(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG72_SHIFT))&BCTU_SFTRGR3_SFTRG72_MASK)
#define BCTU_SFTRGR3_SFTRG73_MASK                0x200u
#define BCTU_SFTRGR3_SFTRG73_SHIFT               9u
#define BCTU_SFTRGR3_SFTRG73_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG73(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG73_SHIFT))&BCTU_SFTRGR3_SFTRG73_MASK)
#define BCTU_SFTRGR3_SFTRG74_MASK                0x400u
#define BCTU_SFTRGR3_SFTRG74_SHIFT               10u
#define BCTU_SFTRGR3_SFTRG74_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG74(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG74_SHIFT))&BCTU_SFTRGR3_SFTRG74_MASK)
#define BCTU_SFTRGR3_SFTRG75_MASK                0x800u
#define BCTU_SFTRGR3_SFTRG75_SHIFT               11u
#define BCTU_SFTRGR3_SFTRG75_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG75(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG75_SHIFT))&BCTU_SFTRGR3_SFTRG75_MASK)
#define BCTU_SFTRGR3_SFTRG76_MASK                0x1000u
#define BCTU_SFTRGR3_SFTRG76_SHIFT               12u
#define BCTU_SFTRGR3_SFTRG76_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG76(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG76_SHIFT))&BCTU_SFTRGR3_SFTRG76_MASK)
#define BCTU_SFTRGR3_SFTRG77_MASK                0x2000u
#define BCTU_SFTRGR3_SFTRG77_SHIFT               13u
#define BCTU_SFTRGR3_SFTRG77_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG77(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG77_SHIFT))&BCTU_SFTRGR3_SFTRG77_MASK)
#define BCTU_SFTRGR3_SFTRG78_MASK                0x4000u
#define BCTU_SFTRGR3_SFTRG78_SHIFT               14u
#define BCTU_SFTRGR3_SFTRG78_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG78(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG78_SHIFT))&BCTU_SFTRGR3_SFTRG78_MASK)
#define BCTU_SFTRGR3_SFTRG79_MASK                0x8000u
#define BCTU_SFTRGR3_SFTRG79_SHIFT               15u
#define BCTU_SFTRGR3_SFTRG79_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG79(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG79_SHIFT))&BCTU_SFTRGR3_SFTRG79_MASK)
#define BCTU_SFTRGR3_SFTRG80_MASK                0x10000u
#define BCTU_SFTRGR3_SFTRG80_SHIFT               16u
#define BCTU_SFTRGR3_SFTRG80_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG80(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG80_SHIFT))&BCTU_SFTRGR3_SFTRG80_MASK)
#define BCTU_SFTRGR3_SFTRG81_MASK                0x20000u
#define BCTU_SFTRGR3_SFTRG81_SHIFT               17u
#define BCTU_SFTRGR3_SFTRG81_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG81(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG81_SHIFT))&BCTU_SFTRGR3_SFTRG81_MASK)
#define BCTU_SFTRGR3_SFTRG82_MASK                0x40000u
#define BCTU_SFTRGR3_SFTRG82_SHIFT               18u
#define BCTU_SFTRGR3_SFTRG82_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG82(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG82_SHIFT))&BCTU_SFTRGR3_SFTRG82_MASK)
#define BCTU_SFTRGR3_SFTRG83_MASK                0x80000u
#define BCTU_SFTRGR3_SFTRG83_SHIFT               19u
#define BCTU_SFTRGR3_SFTRG83_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG83(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG83_SHIFT))&BCTU_SFTRGR3_SFTRG83_MASK)
#define BCTU_SFTRGR3_SFTRG84_MASK                0x100000u
#define BCTU_SFTRGR3_SFTRG84_SHIFT               20u
#define BCTU_SFTRGR3_SFTRG84_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG84(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG84_SHIFT))&BCTU_SFTRGR3_SFTRG84_MASK)
#define BCTU_SFTRGR3_SFTRG85_MASK                0x200000u
#define BCTU_SFTRGR3_SFTRG85_SHIFT               21u
#define BCTU_SFTRGR3_SFTRG85_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG85(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG85_SHIFT))&BCTU_SFTRGR3_SFTRG85_MASK)
#define BCTU_SFTRGR3_SFTRG86_MASK                0x400000u
#define BCTU_SFTRGR3_SFTRG86_SHIFT               22u
#define BCTU_SFTRGR3_SFTRG86_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG86(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG86_SHIFT))&BCTU_SFTRGR3_SFTRG86_MASK)
#define BCTU_SFTRGR3_SFTRG87_MASK                0x800000u
#define BCTU_SFTRGR3_SFTRG87_SHIFT               23u
#define BCTU_SFTRGR3_SFTRG87_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG87(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG87_SHIFT))&BCTU_SFTRGR3_SFTRG87_MASK)
#define BCTU_SFTRGR3_SFTRG88_MASK                0x1000000u
#define BCTU_SFTRGR3_SFTRG88_SHIFT               24u
#define BCTU_SFTRGR3_SFTRG88_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG88(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG88_SHIFT))&BCTU_SFTRGR3_SFTRG88_MASK)
#define BCTU_SFTRGR3_SFTRG89_MASK                0x2000000u
#define BCTU_SFTRGR3_SFTRG89_SHIFT               25u
#define BCTU_SFTRGR3_SFTRG89_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG89(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG89_SHIFT))&BCTU_SFTRGR3_SFTRG89_MASK)
#define BCTU_SFTRGR3_SFTRG90_MASK                0x4000000u
#define BCTU_SFTRGR3_SFTRG90_SHIFT               26u
#define BCTU_SFTRGR3_SFTRG90_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG90(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG90_SHIFT))&BCTU_SFTRGR3_SFTRG90_MASK)
#define BCTU_SFTRGR3_SFTRG91_MASK                0x8000000u
#define BCTU_SFTRGR3_SFTRG91_SHIFT               27u
#define BCTU_SFTRGR3_SFTRG91_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG91(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG91_SHIFT))&BCTU_SFTRGR3_SFTRG91_MASK)
#define BCTU_SFTRGR3_SFTRG92_MASK                0x10000000u
#define BCTU_SFTRGR3_SFTRG92_SHIFT               28u
#define BCTU_SFTRGR3_SFTRG92_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG92(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG92_SHIFT))&BCTU_SFTRGR3_SFTRG92_MASK)
#define BCTU_SFTRGR3_SFTRG93_MASK                0x20000000u
#define BCTU_SFTRGR3_SFTRG93_SHIFT               29u
#define BCTU_SFTRGR3_SFTRG93_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG93(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG93_SHIFT))&BCTU_SFTRGR3_SFTRG93_MASK)
#define BCTU_SFTRGR3_SFTRG94_MASK                0x40000000u
#define BCTU_SFTRGR3_SFTRG94_SHIFT               30u
#define BCTU_SFTRGR3_SFTRG94_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG94(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG94_SHIFT))&BCTU_SFTRGR3_SFTRG94_MASK)
#define BCTU_SFTRGR3_SFTRG95_MASK                0x80000000u
#define BCTU_SFTRGR3_SFTRG95_SHIFT               31u
#define BCTU_SFTRGR3_SFTRG95_WIDTH               1u
#define BCTU_SFTRGR3_SFTRG95(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_SFTRGR3_SFTRG95_SHIFT))&BCTU_SFTRGR3_SFTRG95_MASK)
/* ADC0DR Bit Fields */
#define BCTU_ADC0DR_ADC_DATA_MASK                0xFFFu
#define BCTU_ADC0DR_ADC_DATA_SHIFT               0u
#define BCTU_ADC0DR_ADC_DATA_WIDTH               12u
#define BCTU_ADC0DR_ADC_DATA(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_ADC0DR_ADC_DATA_SHIFT))&BCTU_ADC0DR_ADC_DATA_MASK)
#define BCTU_ADC0DR_LAST_MASK                    0x4000u
#define BCTU_ADC0DR_LAST_SHIFT                   14u
#define BCTU_ADC0DR_LAST_WIDTH                   1u
#define BCTU_ADC0DR_LAST(x)                      (((uint32_t)(((uint32_t)(x))<<BCTU_ADC0DR_LAST_SHIFT))&BCTU_ADC0DR_LAST_MASK)
#define BCTU_ADC0DR_LIST_MASK                    0x8000u
#define BCTU_ADC0DR_LIST_SHIFT                   15u
#define BCTU_ADC0DR_LIST_WIDTH                   1u
#define BCTU_ADC0DR_LIST(x)                      (((uint32_t)(((uint32_t)(x))<<BCTU_ADC0DR_LIST_SHIFT))&BCTU_ADC0DR_LIST_MASK)
#define BCTU_ADC0DR_CH_MASK                      0xFF0000u
#define BCTU_ADC0DR_CH_SHIFT                     16u
#define BCTU_ADC0DR_CH_WIDTH                     8u
#define BCTU_ADC0DR_CH(x)                        (((uint32_t)(((uint32_t)(x))<<BCTU_ADC0DR_CH_SHIFT))&BCTU_ADC0DR_CH_MASK)
#define BCTU_ADC0DR_TRG_SRC_MASK                 0xFF000000u
#define BCTU_ADC0DR_TRG_SRC_SHIFT                24u
#define BCTU_ADC0DR_TRG_SRC_WIDTH                8u
#define BCTU_ADC0DR_TRG_SRC(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_ADC0DR_TRG_SRC_SHIFT))&BCTU_ADC0DR_TRG_SRC_MASK)
/* ADC1DR Bit Fields */
#define BCTU_ADC1DR_ADC_DATA_MASK                0xFFFu
#define BCTU_ADC1DR_ADC_DATA_SHIFT               0u
#define BCTU_ADC1DR_ADC_DATA_WIDTH               12u
#define BCTU_ADC1DR_ADC_DATA(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_ADC1DR_ADC_DATA_SHIFT))&BCTU_ADC1DR_ADC_DATA_MASK)
#define BCTU_ADC1DR_LAST_MASK                    0x4000u
#define BCTU_ADC1DR_LAST_SHIFT                   14u
#define BCTU_ADC1DR_LAST_WIDTH                   1u
#define BCTU_ADC1DR_LAST(x)                      (((uint32_t)(((uint32_t)(x))<<BCTU_ADC1DR_LAST_SHIFT))&BCTU_ADC1DR_LAST_MASK)
#define BCTU_ADC1DR_LIST_MASK                    0x8000u
#define BCTU_ADC1DR_LIST_SHIFT                   15u
#define BCTU_ADC1DR_LIST_WIDTH                   1u
#define BCTU_ADC1DR_LIST(x)                      (((uint32_t)(((uint32_t)(x))<<BCTU_ADC1DR_LIST_SHIFT))&BCTU_ADC1DR_LIST_MASK)
#define BCTU_ADC1DR_CH_MASK                      0xFF0000u
#define BCTU_ADC1DR_CH_SHIFT                     16u
#define BCTU_ADC1DR_CH_WIDTH                     8u
#define BCTU_ADC1DR_CH(x)                        (((uint32_t)(((uint32_t)(x))<<BCTU_ADC1DR_CH_SHIFT))&BCTU_ADC1DR_CH_MASK)
#define BCTU_ADC1DR_TRG_SRC_MASK                 0xFF000000u
#define BCTU_ADC1DR_TRG_SRC_SHIFT                24u
#define BCTU_ADC1DR_TRG_SRC_WIDTH                8u
#define BCTU_ADC1DR_TRG_SRC(x)                   (((uint32_t)(((uint32_t)(x))<<BCTU_ADC1DR_TRG_SRC_SHIFT))&BCTU_ADC1DR_TRG_SRC_MASK)
/* LISTSTAR Bit Fields */
#define BCTU_LISTSTAR_LISTSZ_MASK                0xFFu
#define BCTU_LISTSTAR_LISTSZ_SHIFT               0u
#define BCTU_LISTSTAR_LISTSZ_WIDTH               8u
#define BCTU_LISTSTAR_LISTSZ(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_LISTSTAR_LISTSZ_SHIFT))&BCTU_LISTSTAR_LISTSZ_MASK)
/* LISTCHR_ Bit Fields */
#define BCTU_LISTCHR__ADC_CHL_y_plus_1_MASK      0x7Fu
#define BCTU_LISTCHR__ADC_CHL_y_plus_1_SHIFT     0u
#define BCTU_LISTCHR__ADC_CHL_y_plus_1_WIDTH     7u
#define BCTU_LISTCHR__ADC_CHL_y_plus_1(x)        (((uint32_t)(((uint32_t)(x))<<BCTU_LISTCHR__ADC_CHL_y_plus_1_SHIFT))&BCTU_LISTCHR__ADC_CHL_y_plus_1_MASK)
#define BCTU_LISTCHR__LAST_y_plus_1_MASK         0x8000u
#define BCTU_LISTCHR__LAST_y_plus_1_SHIFT        15u
#define BCTU_LISTCHR__LAST_y_plus_1_WIDTH        1u
#define BCTU_LISTCHR__LAST_y_plus_1(x)           (((uint32_t)(((uint32_t)(x))<<BCTU_LISTCHR__LAST_y_plus_1_SHIFT))&BCTU_LISTCHR__LAST_y_plus_1_MASK)
#define BCTU_LISTCHR__ADC_CH_y_MASK              0x7F0000u
#define BCTU_LISTCHR__ADC_CH_y_SHIFT             16u
#define BCTU_LISTCHR__ADC_CH_y_WIDTH             7u
#define BCTU_LISTCHR__ADC_CH_y(x)                (((uint32_t)(((uint32_t)(x))<<BCTU_LISTCHR__ADC_CH_y_SHIFT))&BCTU_LISTCHR__ADC_CH_y_MASK)
#define BCTU_LISTCHR__LAST_y_MASK                0x80000000u
#define BCTU_LISTCHR__LAST_y_SHIFT               31u
#define BCTU_LISTCHR__LAST_y_WIDTH               1u
#define BCTU_LISTCHR__LAST_y(x)                  (((uint32_t)(((uint32_t)(x))<<BCTU_LISTCHR__LAST_y_SHIFT))&BCTU_LISTCHR__LAST_y_MASK)

/*!
 * @}
 */ /* end of group BCTU_Register_Masks */


/*!
 * @}
 */ /* end of group BCTU_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- C55FMC Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup C55FMC_Peripheral_Access_Layer C55FMC Peripheral Access Layer
 * @{
 */


/** C55FMC - Size of Registers Arrays */
#define C55FMC_UM_COUNT                          9u

/** C55FMC - Register Layout Typedef */
typedef struct {
  __IO uint32_t MCR;                               /**< Module Configuration Register, offset: 0x0 */
       uint8_t RESERVED_0[4];
  __I  uint32_t MCRE;                              /**< Extended Module Configuration Register, offset: 0x8 */
       uint8_t RESERVED_1[4];
  __IO uint32_t LOCK0;                             /**< Lock 0 register, offset: 0x10 */
  __IO uint32_t LOCK1;                             /**< Lock 1 register, offset: 0x14 */
  __IO uint32_t LOCK2;                             /**< Lock 2 register, offset: 0x18 */
  __IO uint32_t LOCK3;                             /**< Lock 3 register, offset: 0x1C */
       uint8_t RESERVED_2[24];
  __IO uint32_t SEL0;                              /**< Select 0 register, offset: 0x38 */
  __IO uint32_t SEL1;                              /**< Select 1 register, offset: 0x3C */
  __IO uint32_t SEL2;                              /**< Select 2 register, offset: 0x40 */
  __IO uint32_t SEL3;                              /**< Select 3 register, offset: 0x44 */
       uint8_t RESERVED_3[8];
  __I  uint32_t ADR;                               /**< Address register, offset: 0x50 */
  __IO uint32_t UT0;                               /**< UTest 0 register, offset: 0x54 */
  __IO uint32_t UM[C55FMC_UM_COUNT];               /**< UMISR register, array offset: 0x58, array step: 0x4 */
  __IO uint32_t UM9;                               /**< UMISR register, offset: 0x7C */
  __I  uint32_t OPP0;                              /**< Over-Program Protection 0 register, offset: 0x80 */
  __I  uint32_t OPP1;                              /**< Over-Program Protection 1 register, offset: 0x84 */
  __I  uint32_t OPP2;                              /**< Over-Program Protection 2 register, offset: 0x88 */
  __I  uint32_t OPP3;                              /**< Over-Program Protection 3 register, offset: 0x8C */
  __IO uint32_t TMD;                               /**< Test Mode Disable Password Check register, offset: 0x90 */
} C55FMC_Type, *C55FMC_MemMapPtr;

 /** Number of instances of the C55FMC module. */
#define C55FMC_INSTANCE_COUNT                    (1u)


/* C55FMC - Peripheral instance base addresses */
/** Peripheral C55FMC base address */
#define C55FMC_BASE                              (0xFFFE0000u)
/** Peripheral C55FMC base pointer */
#define C55FMC                                   ((C55FMC_Type *)C55FMC_BASE)
/** Array initializer of C55FMC peripheral base addresses */
#define C55FMC_BASE_ADDRS                        { C55FMC_BASE }
/** Array initializer of C55FMC peripheral base pointers */
#define C55FMC_BASE_PTRS                         { C55FMC }

/* ----------------------------------------------------------------------------
   -- C55FMC Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup C55FMC_Register_Masks C55FMC Register Masks
 * @{
 */

/* MCR Bit Fields */
#define C55FMC_MCR_EHV_MASK                      0x1u
#define C55FMC_MCR_EHV_SHIFT                     0u
#define C55FMC_MCR_EHV_WIDTH                     1u
#define C55FMC_MCR_EHV(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_EHV_SHIFT))&C55FMC_MCR_EHV_MASK)
#define C55FMC_MCR_ESUS_MASK                     0x2u
#define C55FMC_MCR_ESUS_SHIFT                    1u
#define C55FMC_MCR_ESUS_WIDTH                    1u
#define C55FMC_MCR_ESUS(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_ESUS_SHIFT))&C55FMC_MCR_ESUS_MASK)
#define C55FMC_MCR_ERS_MASK                      0x4u
#define C55FMC_MCR_ERS_SHIFT                     2u
#define C55FMC_MCR_ERS_WIDTH                     1u
#define C55FMC_MCR_ERS(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_ERS_SHIFT))&C55FMC_MCR_ERS_MASK)
#define C55FMC_MCR_PSUS_MASK                     0x8u
#define C55FMC_MCR_PSUS_SHIFT                    3u
#define C55FMC_MCR_PSUS_WIDTH                    1u
#define C55FMC_MCR_PSUS(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_PSUS_SHIFT))&C55FMC_MCR_PSUS_MASK)
#define C55FMC_MCR_PGM_MASK                      0x10u
#define C55FMC_MCR_PGM_SHIFT                     4u
#define C55FMC_MCR_PGM_WIDTH                     1u
#define C55FMC_MCR_PGM(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_PGM_SHIFT))&C55FMC_MCR_PGM_MASK)
#define C55FMC_MCR_PECIE_MASK                    0x100u
#define C55FMC_MCR_PECIE_SHIFT                   8u
#define C55FMC_MCR_PECIE_WIDTH                   1u
#define C55FMC_MCR_PECIE(x)                      (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_PECIE_SHIFT))&C55FMC_MCR_PECIE_MASK)
#define C55FMC_MCR_PEG_MASK                      0x200u
#define C55FMC_MCR_PEG_SHIFT                     9u
#define C55FMC_MCR_PEG_WIDTH                     1u
#define C55FMC_MCR_PEG(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_PEG_SHIFT))&C55FMC_MCR_PEG_MASK)
#define C55FMC_MCR_DONE_MASK                     0x400u
#define C55FMC_MCR_DONE_SHIFT                    10u
#define C55FMC_MCR_DONE_WIDTH                    1u
#define C55FMC_MCR_DONE(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_DONE_SHIFT))&C55FMC_MCR_DONE_MASK)
#define C55FMC_MCR_PEAS_MASK                     0x800u
#define C55FMC_MCR_PEAS_SHIFT                    11u
#define C55FMC_MCR_PEAS_WIDTH                    1u
#define C55FMC_MCR_PEAS(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_PEAS_SHIFT))&C55FMC_MCR_PEAS_MASK)
#define C55FMC_MCR_SBC_MASK                      0x2000u
#define C55FMC_MCR_SBC_SHIFT                     13u
#define C55FMC_MCR_SBC_WIDTH                     1u
#define C55FMC_MCR_SBC(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_SBC_SHIFT))&C55FMC_MCR_SBC_MASK)
#define C55FMC_MCR_RWE_MASK                      0x4000u
#define C55FMC_MCR_RWE_SHIFT                     14u
#define C55FMC_MCR_RWE_WIDTH                     1u
#define C55FMC_MCR_RWE(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_RWE_SHIFT))&C55FMC_MCR_RWE_MASK)
#define C55FMC_MCR_EER_MASK                      0x8000u
#define C55FMC_MCR_EER_SHIFT                     15u
#define C55FMC_MCR_EER_WIDTH                     1u
#define C55FMC_MCR_EER(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_EER_SHIFT))&C55FMC_MCR_EER_MASK)
#define C55FMC_MCR_EEE_MASK                      0x10000000u
#define C55FMC_MCR_EEE_SHIFT                     28u
#define C55FMC_MCR_EEE_WIDTH                     1u
#define C55FMC_MCR_EEE(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_EEE_SHIFT))&C55FMC_MCR_EEE_MASK)
#define C55FMC_MCR_AEE_MASK                      0x20000000u
#define C55FMC_MCR_AEE_SHIFT                     29u
#define C55FMC_MCR_AEE_WIDTH                     1u
#define C55FMC_MCR_AEE(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_AEE_SHIFT))&C55FMC_MCR_AEE_MASK)
#define C55FMC_MCR_RRE_MASK                      0x40000000u
#define C55FMC_MCR_RRE_SHIFT                     30u
#define C55FMC_MCR_RRE_WIDTH                     1u
#define C55FMC_MCR_RRE(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_RRE_SHIFT))&C55FMC_MCR_RRE_MASK)
#define C55FMC_MCR_RVE_MASK                      0x80000000u
#define C55FMC_MCR_RVE_SHIFT                     31u
#define C55FMC_MCR_RVE_WIDTH                     1u
#define C55FMC_MCR_RVE(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCR_RVE_SHIFT))&C55FMC_MCR_RVE_MASK)
/* MCRE Bit Fields */
#define C55FMC_MCRE_n16Kl_MASK                   0x7u
#define C55FMC_MCRE_n16Kl_SHIFT                  0u
#define C55FMC_MCRE_n16Kl_WIDTH                  3u
#define C55FMC_MCRE_n16Kl(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n16Kl_SHIFT))&C55FMC_MCRE_n16Kl_MASK)
#define C55FMC_MCRE_n32Kl_MASK                   0x18u
#define C55FMC_MCRE_n32Kl_SHIFT                  3u
#define C55FMC_MCRE_n32Kl_WIDTH                  2u
#define C55FMC_MCRE_n32Kl(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n32Kl_SHIFT))&C55FMC_MCRE_n32Kl_MASK)
#define C55FMC_MCRE_n64Kl_MASK                   0xE0u
#define C55FMC_MCRE_n64Kl_SHIFT                  5u
#define C55FMC_MCRE_n64Kl_WIDTH                  3u
#define C55FMC_MCRE_n64Kl(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n64Kl_SHIFT))&C55FMC_MCRE_n64Kl_MASK)
#define C55FMC_MCRE_n16Km_MASK                   0x700u
#define C55FMC_MCRE_n16Km_SHIFT                  8u
#define C55FMC_MCRE_n16Km_WIDTH                  3u
#define C55FMC_MCRE_n16Km(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n16Km_SHIFT))&C55FMC_MCRE_n16Km_MASK)
#define C55FMC_MCRE_n32Km_MASK                   0x1800u
#define C55FMC_MCRE_n32Km_SHIFT                  11u
#define C55FMC_MCRE_n32Km_WIDTH                  2u
#define C55FMC_MCRE_n32Km(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n32Km_SHIFT))&C55FMC_MCRE_n32Km_MASK)
#define C55FMC_MCRE_n64Km_MASK                   0xE000u
#define C55FMC_MCRE_n64Km_SHIFT                  13u
#define C55FMC_MCRE_n64Km_WIDTH                  3u
#define C55FMC_MCRE_n64Km(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n64Km_SHIFT))&C55FMC_MCRE_n64Km_MASK)
#define C55FMC_MCRE_n16Kh_MASK                   0x70000u
#define C55FMC_MCRE_n16Kh_SHIFT                  16u
#define C55FMC_MCRE_n16Kh_WIDTH                  3u
#define C55FMC_MCRE_n16Kh(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n16Kh_SHIFT))&C55FMC_MCRE_n16Kh_MASK)
#define C55FMC_MCRE_n32Kh_MASK                   0x180000u
#define C55FMC_MCRE_n32Kh_SHIFT                  19u
#define C55FMC_MCRE_n32Kh_WIDTH                  2u
#define C55FMC_MCRE_n32Kh(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n32Kh_SHIFT))&C55FMC_MCRE_n32Kh_MASK)
#define C55FMC_MCRE_n64Kh_MASK                   0xE00000u
#define C55FMC_MCRE_n64Kh_SHIFT                  21u
#define C55FMC_MCRE_n64Kh_WIDTH                  3u
#define C55FMC_MCRE_n64Kh(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n64Kh_SHIFT))&C55FMC_MCRE_n64Kh_MASK)
#define C55FMC_MCRE_n256K_MASK                   0x1F000000u
#define C55FMC_MCRE_n256K_SHIFT                  24u
#define C55FMC_MCRE_n256K_WIDTH                  5u
#define C55FMC_MCRE_n256K(x)                     (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_n256K_SHIFT))&C55FMC_MCRE_n256K_MASK)
#define C55FMC_MCRE_HT_MASK                      0x80000000u
#define C55FMC_MCRE_HT_SHIFT                     31u
#define C55FMC_MCRE_HT_WIDTH                     1u
#define C55FMC_MCRE_HT(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_MCRE_HT_SHIFT))&C55FMC_MCRE_HT_MASK)
/* LOCK0 Bit Fields */
#define C55FMC_LOCK0_MIDLOCK_MASK                0xFFFFu
#define C55FMC_LOCK0_MIDLOCK_SHIFT               0u
#define C55FMC_LOCK0_MIDLOCK_WIDTH               16u
#define C55FMC_LOCK0_MIDLOCK(x)                  (((uint32_t)(((uint32_t)(x))<<C55FMC_LOCK0_MIDLOCK_SHIFT))&C55FMC_LOCK0_MIDLOCK_MASK)
#define C55FMC_LOCK0_LOWLOCK_MASK                0x3FFF0000u
#define C55FMC_LOCK0_LOWLOCK_SHIFT               16u
#define C55FMC_LOCK0_LOWLOCK_WIDTH               14u
#define C55FMC_LOCK0_LOWLOCK(x)                  (((uint32_t)(((uint32_t)(x))<<C55FMC_LOCK0_LOWLOCK_SHIFT))&C55FMC_LOCK0_LOWLOCK_MASK)
#define C55FMC_LOCK0_TSLOCK_MASK                 0x80000000u
#define C55FMC_LOCK0_TSLOCK_SHIFT                31u
#define C55FMC_LOCK0_TSLOCK_WIDTH                1u
#define C55FMC_LOCK0_TSLOCK(x)                   (((uint32_t)(((uint32_t)(x))<<C55FMC_LOCK0_TSLOCK_SHIFT))&C55FMC_LOCK0_TSLOCK_MASK)
/* LOCK1 Bit Fields */
#define C55FMC_LOCK1_HIGHLOCK_MASK               0xFFFFu
#define C55FMC_LOCK1_HIGHLOCK_SHIFT              0u
#define C55FMC_LOCK1_HIGHLOCK_WIDTH              16u
#define C55FMC_LOCK1_HIGHLOCK(x)                 (((uint32_t)(((uint32_t)(x))<<C55FMC_LOCK1_HIGHLOCK_SHIFT))&C55FMC_LOCK1_HIGHLOCK_MASK)
/* LOCK2 Bit Fields */
#define C55FMC_LOCK2_A256KLOCK_MASK              0xFFFFFFFFu
#define C55FMC_LOCK2_A256KLOCK_SHIFT             0u
#define C55FMC_LOCK2_A256KLOCK_WIDTH             32u
#define C55FMC_LOCK2_A256KLOCK(x)                (((uint32_t)(((uint32_t)(x))<<C55FMC_LOCK2_A256KLOCK_SHIFT))&C55FMC_LOCK2_A256KLOCK_MASK)
/* LOCK3 Bit Fields */
#define C55FMC_LOCK3_A256KLOCK_MASK              0xFFFFu
#define C55FMC_LOCK3_A256KLOCK_SHIFT             0u
#define C55FMC_LOCK3_A256KLOCK_WIDTH             16u
#define C55FMC_LOCK3_A256KLOCK(x)                (((uint32_t)(((uint32_t)(x))<<C55FMC_LOCK3_A256KLOCK_SHIFT))&C55FMC_LOCK3_A256KLOCK_MASK)
/* SEL0 Bit Fields */
#define C55FMC_SEL0_MIDSEL_MASK                  0xFFFFu
#define C55FMC_SEL0_MIDSEL_SHIFT                 0u
#define C55FMC_SEL0_MIDSEL_WIDTH                 16u
#define C55FMC_SEL0_MIDSEL(x)                    (((uint32_t)(((uint32_t)(x))<<C55FMC_SEL0_MIDSEL_SHIFT))&C55FMC_SEL0_MIDSEL_MASK)
#define C55FMC_SEL0_LOWSEL_MASK                  0x3FFF0000u
#define C55FMC_SEL0_LOWSEL_SHIFT                 16u
#define C55FMC_SEL0_LOWSEL_WIDTH                 14u
#define C55FMC_SEL0_LOWSEL(x)                    (((uint32_t)(((uint32_t)(x))<<C55FMC_SEL0_LOWSEL_SHIFT))&C55FMC_SEL0_LOWSEL_MASK)
/* SEL1 Bit Fields */
#define C55FMC_SEL1_HIGHSEL_MASK                 0xFFFFu
#define C55FMC_SEL1_HIGHSEL_SHIFT                0u
#define C55FMC_SEL1_HIGHSEL_WIDTH                16u
#define C55FMC_SEL1_HIGHSEL(x)                   (((uint32_t)(((uint32_t)(x))<<C55FMC_SEL1_HIGHSEL_SHIFT))&C55FMC_SEL1_HIGHSEL_MASK)
/* SEL2 Bit Fields */
#define C55FMC_SEL2_A256KSEL_MASK                0xFFFFFFFFu
#define C55FMC_SEL2_A256KSEL_SHIFT               0u
#define C55FMC_SEL2_A256KSEL_WIDTH               32u
#define C55FMC_SEL2_A256KSEL(x)                  (((uint32_t)(((uint32_t)(x))<<C55FMC_SEL2_A256KSEL_SHIFT))&C55FMC_SEL2_A256KSEL_MASK)
/* SEL3 Bit Fields */
#define C55FMC_SEL3_A256KSEL_MASK                0xFFFFu
#define C55FMC_SEL3_A256KSEL_SHIFT               0u
#define C55FMC_SEL3_A256KSEL_WIDTH               16u
#define C55FMC_SEL3_A256KSEL(x)                  (((uint32_t)(((uint32_t)(x))<<C55FMC_SEL3_A256KSEL_SHIFT))&C55FMC_SEL3_A256KSEL_MASK)
/* ADR Bit Fields */
#define C55FMC_ADR_ADDR_MASK                     0xFFFFF8u
#define C55FMC_ADR_ADDR_SHIFT                    3u
#define C55FMC_ADR_ADDR_WIDTH                    21u
#define C55FMC_ADR_ADDR(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_ADR_ADDR_SHIFT))&C55FMC_ADR_ADDR_MASK)
#define C55FMC_ADR_a16k_MASK                     0x1000000u
#define C55FMC_ADR_a16k_SHIFT                    24u
#define C55FMC_ADR_a16k_WIDTH                    1u
#define C55FMC_ADR_a16k(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_ADR_a16k_SHIFT))&C55FMC_ADR_a16k_MASK)
#define C55FMC_ADR_a32k_MASK                     0x2000000u
#define C55FMC_ADR_a32k_SHIFT                    25u
#define C55FMC_ADR_a32k_WIDTH                    1u
#define C55FMC_ADR_a32k(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_ADR_a32k_SHIFT))&C55FMC_ADR_a32k_MASK)
#define C55FMC_ADR_a64k_MASK                     0x4000000u
#define C55FMC_ADR_a64k_SHIFT                    26u
#define C55FMC_ADR_a64k_WIDTH                    1u
#define C55FMC_ADR_a64k(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_ADR_a64k_SHIFT))&C55FMC_ADR_a64k_MASK)
#define C55FMC_ADR_a256k_MASK                    0x8000000u
#define C55FMC_ADR_a256k_SHIFT                   27u
#define C55FMC_ADR_a256k_WIDTH                   1u
#define C55FMC_ADR_a256k(x)                      (((uint32_t)(((uint32_t)(x))<<C55FMC_ADR_a256k_SHIFT))&C55FMC_ADR_a256k_MASK)
#define C55FMC_ADR_aL_MASK                       0x10000000u
#define C55FMC_ADR_aL_SHIFT                      28u
#define C55FMC_ADR_aL_WIDTH                      1u
#define C55FMC_ADR_aL(x)                         (((uint32_t)(((uint32_t)(x))<<C55FMC_ADR_aL_SHIFT))&C55FMC_ADR_aL_MASK)
#define C55FMC_ADR_aM_MASK                       0x20000000u
#define C55FMC_ADR_aM_SHIFT                      29u
#define C55FMC_ADR_aM_WIDTH                      1u
#define C55FMC_ADR_aM(x)                         (((uint32_t)(((uint32_t)(x))<<C55FMC_ADR_aM_SHIFT))&C55FMC_ADR_aM_MASK)
#define C55FMC_ADR_aH_MASK                       0x40000000u
#define C55FMC_ADR_aH_SHIFT                      30u
#define C55FMC_ADR_aH_WIDTH                      1u
#define C55FMC_ADR_aH(x)                         (((uint32_t)(((uint32_t)(x))<<C55FMC_ADR_aH_SHIFT))&C55FMC_ADR_aH_MASK)
#define C55FMC_ADR_SAD_MASK                      0x80000000u
#define C55FMC_ADR_SAD_SHIFT                     31u
#define C55FMC_ADR_SAD_WIDTH                     1u
#define C55FMC_ADR_SAD(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_ADR_SAD_SHIFT))&C55FMC_ADR_SAD_MASK)
/* UT0 Bit Fields */
#define C55FMC_UT0_AID_MASK                      0x1u
#define C55FMC_UT0_AID_SHIFT                     0u
#define C55FMC_UT0_AID_WIDTH                     1u
#define C55FMC_UT0_AID(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_AID_SHIFT))&C55FMC_UT0_AID_MASK)
#define C55FMC_UT0_AIE_MASK                      0x2u
#define C55FMC_UT0_AIE_SHIFT                     1u
#define C55FMC_UT0_AIE_WIDTH                     1u
#define C55FMC_UT0_AIE(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_AIE_SHIFT))&C55FMC_UT0_AIE_MASK)
#define C55FMC_UT0_AIS_MASK                      0x4u
#define C55FMC_UT0_AIS_SHIFT                     2u
#define C55FMC_UT0_AIS_WIDTH                     1u
#define C55FMC_UT0_AIS(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_AIS_SHIFT))&C55FMC_UT0_AIS_MASK)
#define C55FMC_UT0_MRV_MASK                      0x10u
#define C55FMC_UT0_MRV_SHIFT                     4u
#define C55FMC_UT0_MRV_WIDTH                     1u
#define C55FMC_UT0_MRV(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_MRV_SHIFT))&C55FMC_UT0_MRV_MASK)
#define C55FMC_UT0_MRE_MASK                      0x20u
#define C55FMC_UT0_MRE_SHIFT                     5u
#define C55FMC_UT0_MRE_WIDTH                     1u
#define C55FMC_UT0_MRE(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_MRE_SHIFT))&C55FMC_UT0_MRE_MASK)
#define C55FMC_UT0_AISUS_MASK                    0x40u
#define C55FMC_UT0_AISUS_SHIFT                   6u
#define C55FMC_UT0_AISUS_WIDTH                   1u
#define C55FMC_UT0_AISUS(x)                      (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_AISUS_SHIFT))&C55FMC_UT0_AISUS_MASK)
#define C55FMC_UT0_AIBPE_MASK                    0x100u
#define C55FMC_UT0_AIBPE_SHIFT                   8u
#define C55FMC_UT0_AIBPE_WIDTH                   1u
#define C55FMC_UT0_AIBPE(x)                      (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_AIBPE_SHIFT))&C55FMC_UT0_AIBPE_MASK)
#define C55FMC_UT0_NAIBP_MASK                    0x200u
#define C55FMC_UT0_NAIBP_SHIFT                   9u
#define C55FMC_UT0_NAIBP_WIDTH                   1u
#define C55FMC_UT0_NAIBP(x)                      (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_NAIBP_SHIFT))&C55FMC_UT0_NAIBP_MASK)
#define C55FMC_UT0_CPE_MASK                      0x10000u
#define C55FMC_UT0_CPE_SHIFT                     16u
#define C55FMC_UT0_CPE_WIDTH                     1u
#define C55FMC_UT0_CPE(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_CPE_SHIFT))&C55FMC_UT0_CPE_MASK)
#define C55FMC_UT0_CPA_MASK                      0x20000u
#define C55FMC_UT0_CPA_SHIFT                     17u
#define C55FMC_UT0_CPA_WIDTH                     1u
#define C55FMC_UT0_CPA(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_CPA_SHIFT))&C55FMC_UT0_CPA_MASK)
#define C55FMC_UT0_CPR_MASK                      0x40000u
#define C55FMC_UT0_CPR_SHIFT                     18u
#define C55FMC_UT0_CPR_WIDTH                     1u
#define C55FMC_UT0_CPR(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_CPR_SHIFT))&C55FMC_UT0_CPR_MASK)
#define C55FMC_UT0_SBCE_MASK                     0x40000000u
#define C55FMC_UT0_SBCE_SHIFT                    30u
#define C55FMC_UT0_SBCE_WIDTH                    1u
#define C55FMC_UT0_SBCE(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_SBCE_SHIFT))&C55FMC_UT0_SBCE_MASK)
#define C55FMC_UT0_UTE_MASK                      0x80000000u
#define C55FMC_UT0_UTE_SHIFT                     31u
#define C55FMC_UT0_UTE_WIDTH                     1u
#define C55FMC_UT0_UTE(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UT0_UTE_SHIFT))&C55FMC_UT0_UTE_MASK)
/* UM Bit Fields */
#define C55FMC_UM_MISR_MASK                      0xFFFFFFFFu
#define C55FMC_UM_MISR_SHIFT                     0u
#define C55FMC_UM_MISR_WIDTH                     32u
#define C55FMC_UM_MISR(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_UM_MISR_SHIFT))&C55FMC_UM_MISR_MASK)
/* UM9 Bit Fields */
#define C55FMC_UM9_MISR_MASK                     0x1u
#define C55FMC_UM9_MISR_SHIFT                    0u
#define C55FMC_UM9_MISR_WIDTH                    1u
#define C55FMC_UM9_MISR(x)                       (((uint32_t)(((uint32_t)(x))<<C55FMC_UM9_MISR_SHIFT))&C55FMC_UM9_MISR_MASK)
/* OPP0 Bit Fields */
#define C55FMC_OPP0_MIDOPP_MASK                  0xFFFFu
#define C55FMC_OPP0_MIDOPP_SHIFT                 0u
#define C55FMC_OPP0_MIDOPP_WIDTH                 16u
#define C55FMC_OPP0_MIDOPP(x)                    (((uint32_t)(((uint32_t)(x))<<C55FMC_OPP0_MIDOPP_SHIFT))&C55FMC_OPP0_MIDOPP_MASK)
#define C55FMC_OPP0_LOWOPP_MASK                  0x3FFF0000u
#define C55FMC_OPP0_LOWOPP_SHIFT                 16u
#define C55FMC_OPP0_LOWOPP_WIDTH                 14u
#define C55FMC_OPP0_LOWOPP(x)                    (((uint32_t)(((uint32_t)(x))<<C55FMC_OPP0_LOWOPP_SHIFT))&C55FMC_OPP0_LOWOPP_MASK)
/* OPP1 Bit Fields */
#define C55FMC_OPP1_HIGHOPP_MASK                 0xFFFFu
#define C55FMC_OPP1_HIGHOPP_SHIFT                0u
#define C55FMC_OPP1_HIGHOPP_WIDTH                16u
#define C55FMC_OPP1_HIGHOPP(x)                   (((uint32_t)(((uint32_t)(x))<<C55FMC_OPP1_HIGHOPP_SHIFT))&C55FMC_OPP1_HIGHOPP_MASK)
/* OPP2 Bit Fields */
#define C55FMC_OPP2_A256KOPP_MASK                0xFFFFFFFFu
#define C55FMC_OPP2_A256KOPP_SHIFT               0u
#define C55FMC_OPP2_A256KOPP_WIDTH               32u
#define C55FMC_OPP2_A256KOPP(x)                  (((uint32_t)(((uint32_t)(x))<<C55FMC_OPP2_A256KOPP_SHIFT))&C55FMC_OPP2_A256KOPP_MASK)
/* OPP3 Bit Fields */
#define C55FMC_OPP3_A256KOPP_MASK                0xFFFFu
#define C55FMC_OPP3_A256KOPP_SHIFT               0u
#define C55FMC_OPP3_A256KOPP_WIDTH               16u
#define C55FMC_OPP3_A256KOPP(x)                  (((uint32_t)(((uint32_t)(x))<<C55FMC_OPP3_A256KOPP_SHIFT))&C55FMC_OPP3_A256KOPP_MASK)
/* TMD Bit Fields */
#define C55FMC_TMD_PWD_MASK                      0xFFFFFFFFu
#define C55FMC_TMD_PWD_SHIFT                     0u
#define C55FMC_TMD_PWD_WIDTH                     32u
#define C55FMC_TMD_PWD(x)                        (((uint32_t)(((uint32_t)(x))<<C55FMC_TMD_PWD_SHIFT))&C55FMC_TMD_PWD_MASK)

/*!
 * @}
 */ /* end of group C55FMC_Register_Masks */


/*!
 * @}
 */ /* end of group C55FMC_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- CAN Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CAN_Peripheral_Access_Layer CAN Peripheral Access Layer
 * @{
 */


/** CAN - Size of Registers Arrays */
#define CAN_RAMn_COUNT                           384u
#define CAN_RXIMR_COUNT                          96u
#define CAN_WMB_COUNT                            4u

/** CAN - Register Layout Typedef */
typedef struct {
  __IO uint32_t MCR;                               /**< Module Configuration Register, offset: 0x0 */
  __IO uint32_t CTRL1;                             /**< Control 1 register, offset: 0x4 */
  __IO uint32_t TIMER;                             /**< Free Running Timer, offset: 0x8 */
       uint8_t RESERVED_0[4];
  __IO uint32_t RXMGMASK;                          /**< Rx Mailboxes Global Mask Register, offset: 0x10 */
  __IO uint32_t RX14MASK;                          /**< Rx 14 Mask register, offset: 0x14 */
  __IO uint32_t RX15MASK;                          /**< Rx 15 Mask register, offset: 0x18 */
  __IO uint32_t ECR;                               /**< Error Counter, offset: 0x1C */
  __IO uint32_t ESR1;                              /**< Error and Status 1 register, offset: 0x20 */
  __IO uint32_t IMASK2;                            /**< Interrupt Masks 2 register, offset: 0x24 */
  __IO uint32_t IMASK1;                            /**< Interrupt Masks 1 register, offset: 0x28 */
  __IO uint32_t IFLAG2;                            /**< Interrupt Flags 2 register, offset: 0x2C */
  __IO uint32_t IFLAG1;                            /**< Interrupt Flags 1 register, offset: 0x30 */
  __IO uint32_t CTRL2;                             /**< Control 2 register, offset: 0x34 */
  __I  uint32_t ESR2;                              /**< Error and Status 2 register, offset: 0x38 */
       uint8_t RESERVED_1[8];
  __I  uint32_t CRCR;                              /**< CRC Register, offset: 0x44 */
  __IO uint32_t RXFGMASK;                          /**< Rx FIFO Global Mask register, offset: 0x48 */
  __I  uint32_t RXFIR;                             /**< Rx FIFO Information Register, offset: 0x4C */
  __IO uint32_t CBT;                               /**< CAN Bit Timing Register, offset: 0x50 */
       uint8_t RESERVED_2[24];
  __IO uint32_t IMASK3;                            /**< Interrupt Masks 3 Register, offset: 0x6C */
       uint8_t RESERVED_3[4];
  __IO uint32_t IFLAG3;                            /**< Interrupt Flags 3 Register, offset: 0x74 */
       uint8_t RESERVED_4[8];
  __IO uint32_t RAMn[CAN_RAMn_COUNT];              /**< Embedded RAM, array offset: 0x80, array step: 0x4 */
       uint8_t RESERVED_5[512];
  __IO uint32_t RXIMR[CAN_RXIMR_COUNT];            /**< Rx Individual Mask Registers, array offset: 0x880, array step: 0x4 */
       uint8_t RESERVED_6[256];
  __IO uint32_t CTRL1_PN;                          /**< Pretended Networking Control 1 Register, offset: 0xB00 */
  __IO uint32_t CTRL2_PN;                          /**< Pretended Networking Control 2 Register, offset: 0xB04 */
  __IO uint32_t WU_MTC;                            /**< Pretended Networking Wake Up Match Register, offset: 0xB08 */
  __IO uint32_t FLT_ID1;                           /**< Pretended Networking ID Filter 1 Register, offset: 0xB0C */
  __IO uint32_t FLT_DLC;                           /**< Pretended Networking DLC Filter Register, offset: 0xB10 */
  __IO uint32_t PL1_LO;                            /**< Pretended Networking Payload Low Filter 1 Register, offset: 0xB14 */
  __IO uint32_t PL1_HI;                            /**< Pretended Networking Payload High Filter 1 Register, offset: 0xB18 */
  __IO uint32_t FLT_ID2_IDMASK;                    /**< Pretended Networking ID Filter 2 Register / ID Mask Register, offset: 0xB1C */
  __IO uint32_t PL2_PLMASK_LO;                     /**< Pretended Networking Payload Low Filter 2 Register / Payload Low Mask Register, offset: 0xB20 */
  __IO uint32_t PL2_PLMASK_HI;                     /**< Pretended Networking Payload High Filter 2 low order bits / Payload High Mask Register, offset: 0xB24 */
       uint8_t RESERVED_7[24];
  struct {                                         /* offset: 0xB40, array step: 0x10 */
    __I  uint32_t WMBn_CS;                           /**< Wake Up Message Buffer Register for C/S, array offset: 0xB40, array step: 0x10 */
    __I  uint32_t WMBn_ID;                           /**< Wake Up Message Buffer Register for ID, array offset: 0xB44, array step: 0x10 */
    __I  uint32_t WMBn_D03;                          /**< Wake Up Message Buffer Register for Data 0-3, array offset: 0xB48, array step: 0x10 */
    __I  uint32_t WMBn_D47;                          /**< Wake Up Message Buffer Register Data 4-7, array offset: 0xB4C, array step: 0x10 */
  } WMB[CAN_WMB_COUNT];
       uint8_t RESERVED_8[128];
  __IO uint32_t FDCTRL;                            /**< CAN FD Control Register, offset: 0xC00 */
  __IO uint32_t FDCBT;                             /**< CAN FD Bit Timing Register, offset: 0xC04 */
  __I  uint32_t FDCRC;                             /**< CAN FD CRC Register, offset: 0xC08 */
} CAN_Type, *CAN_MemMapPtr;

 /** Number of instances of the CAN module. */
#define CAN_INSTANCE_COUNT                       (8u)


/* CAN - Peripheral instance base addresses */
/** Peripheral CAN_0 base address */
#define CAN_0_BASE                               (0xFFEC0000u)
/** Peripheral CAN_0 base pointer */
#define CAN_0                                    ((CAN_Type *)CAN_0_BASE)
/** Peripheral CAN_1 base address */
#define CAN_1_BASE                               (0xFBEC0000u)
/** Peripheral CAN_1 base pointer */
#define CAN_1                                    ((CAN_Type *)CAN_1_BASE)
/** Peripheral CAN_2 base address */
#define CAN_2_BASE                               (0xFFEC4000u)
/** Peripheral CAN_2 base pointer */
#define CAN_2                                    ((CAN_Type *)CAN_2_BASE)
/** Peripheral CAN_3 base address */
#define CAN_3_BASE                               (0xFBEC4000u)
/** Peripheral CAN_3 base pointer */
#define CAN_3                                    ((CAN_Type *)CAN_3_BASE)
/** Peripheral CAN_4 base address */
#define CAN_4_BASE                               (0xFFEC8000u)
/** Peripheral CAN_4 base pointer */
#define CAN_4                                    ((CAN_Type *)CAN_4_BASE)
/** Peripheral CAN_5 base address */
#define CAN_5_BASE                               (0xFBEC8000u)
/** Peripheral CAN_5 base pointer */
#define CAN_5                                    ((CAN_Type *)CAN_5_BASE)
/** Peripheral CAN_6 base address */
#define CAN_6_BASE                               (0xFFECC000u)
/** Peripheral CAN_6 base pointer */
#define CAN_6                                    ((CAN_Type *)CAN_6_BASE)
/** Peripheral CAN_7 base address */
#define CAN_7_BASE                               (0xFBECC000u)
/** Peripheral CAN_7 base pointer */
#define CAN_7                                    ((CAN_Type *)CAN_7_BASE)
/** Array initializer of CAN peripheral base addresses */
#define CAN_BASE_ADDRS                           { CAN_0_BASE, CAN_1_BASE, CAN_2_BASE, CAN_3_BASE, CAN_4_BASE, CAN_5_BASE, CAN_6_BASE, CAN_7_BASE }
/** Array initializer of CAN peripheral base pointers */
#define CAN_BASE_PTRS                            { CAN_0, CAN_1, CAN_2, CAN_3, CAN_4, CAN_5, CAN_6, CAN_7 }

/** Interrupt vectors for the CAN peripheral type */
#define CAN_Rx_Warning_IRQS                      { CAN0_ORed_IRQn, CAN1_ORed_IRQn, \
                                                   CAN2_ORed_IRQn, CAN3_ORed_IRQn, \
                                                   CAN4_ORed_IRQn, CAN5_ORed_IRQn, \
                                                   CAN6_ORed_IRQn, CAN7_ORed_IRQn }

#define CAN_Tx_Warning_IRQS                      { CAN0_ORed_IRQn, CAN1_ORed_IRQn, \
                                                   CAN2_ORed_IRQn, CAN3_ORed_IRQn, \
                                                   CAN4_ORed_IRQn, CAN5_ORed_IRQn, \
                                                   CAN6_ORed_IRQn, CAN7_ORed_IRQn }

#define CAN_Wake_Up_IRQS                         { CAN0_Wake_Up_IRQn, NotAvail_IRQn, \
                                                   NotAvail_IRQn, NotAvail_IRQn, \
                                                   NotAvail_IRQn, NotAvail_IRQn, \
                                                   NotAvail_IRQn, NotAvail_IRQn }

#define CAN_Error_IRQS                           { CAN0_Error_IRQn, CAN1_Error_IRQn, \
                                                   CAN2_Error_IRQn, CAN3_Error_IRQn, \
                                                   CAN4_Error_IRQn, CAN5_Error_IRQn, \
                                                   CAN6_Error_IRQn, CAN7_Error_IRQn }

#define CAN_Bus_Off_IRQS                         { CAN0_ORed_IRQn, CAN1_ORed_IRQn, \
                                                   CAN2_ORed_IRQn, CAN3_ORed_IRQn, \
                                                   CAN4_ORed_IRQn, CAN5_ORed_IRQn, \
                                                   CAN6_ORed_IRQn, CAN7_ORed_IRQn }

#define CAN_ORed_00_03_MB_IRQS                   { CAN0_ORed_00_03_MB_IRQn, CAN1_ORed_00_03_MB_IRQn, \
                                                   CAN2_ORed_00_03_MB_IRQn, CAN3_ORed_00_03_MB_IRQn, \
                                                   CAN4_ORed_00_03_MB_IRQn, CAN5_ORed_00_03_MB_IRQn, \
                                                   CAN6_ORed_00_03_MB_IRQn, CAN7_ORed_00_03_MB_IRQn }

#define CAN_ORed_04_07_MB_IRQS                   { CAN0_ORed_04_07_MB_IRQn, CAN1_ORed_04_07_MB_IRQn, \
                                                   CAN2_ORed_04_07_MB_IRQn, CAN3_ORed_04_07_MB_IRQn, \
                                                   CAN4_ORed_04_07_MB_IRQn, CAN5_ORed_04_07_MB_IRQn, \
                                                   CAN6_ORed_04_07_MB_IRQn, CAN7_ORed_04_07_MB_IRQn }

#define CAN_ORed_08_11_MB_IRQS                   { CAN0_ORed_08_11_MB_IRQn, CAN1_ORed_08_11_MB_IRQn, \
                                                   CAN2_ORed_08_11_MB_IRQn, CAN3_ORed_08_11_MB_IRQn, \
                                                   CAN4_ORed_08_11_MB_IRQn, CAN5_ORed_08_11_MB_IRQn, \
                                                   CAN6_ORed_08_11_MB_IRQn, CAN7_ORed_08_11_MB_IRQn }

#define CAN_ORed_12_15_MB_IRQS                   { CAN0_ORed_12_15_MB_IRQn, CAN1_ORed_12_15_MB_IRQn, \
                                                   CAN2_ORed_12_15_MB_IRQn, CAN3_ORed_12_15_MB_IRQn, \
                                                   CAN4_ORed_12_15_MB_IRQn, CAN5_ORed_12_15_MB_IRQn, \
                                                   CAN6_ORed_12_15_MB_IRQn, CAN7_ORed_12_15_MB_IRQn }

#define CAN_ORed_16_31_MB_IRQS                   { CAN0_ORed_16_31_MB_IRQn, CAN1_ORed_16_31_MB_IRQn, \
                                                   CAN2_ORed_16_31_MB_IRQn, CAN3_ORed_16_31_MB_IRQn, \
                                                   CAN4_ORed_16_31_MB_IRQn, CAN5_ORed_16_31_MB_IRQn, \
                                                   CAN6_ORed_16_31_MB_IRQn, CAN7_ORed_16_31_MB_IRQn }

#define CAN_ORed_32_63_MB_IRQS                   { CAN0_ORed_32_63_MB_IRQn, CAN1_ORed_32_63_MB_IRQn, \
                                                   CAN2_ORed_32_63_MB_IRQn, CAN3_ORed_32_63_MB_IRQn, \
                                                   CAN4_ORed_32_63_MB_IRQn, CAN5_ORed_32_63_MB_IRQn, \
                                                   CAN6_ORed_32_63_MB_IRQn, CAN7_ORed_32_63_MB_IRQn }

#define CAN_ORed_64_95_MB_IRQS                   { CAN0_ORed_64_95_MB_IRQn, CAN1_ORed_64_95_MB_IRQn, \
                                                   CAN2_ORed_64_95_MB_IRQn, CAN3_ORed_64_95_MB_IRQn, \
                                                   CAN4_ORed_64_95_MB_IRQn, CAN5_ORed_64_95_MB_IRQn, \
                                                   CAN6_ORed_64_95_MB_IRQn, CAN7_ORed_64_95_MB_IRQn }

/* ----------------------------------------------------------------------------
   -- CAN Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CAN_Register_Masks CAN Register Masks
 * @{
 */

/* MCR Bit Fields */
#define CAN_MCR_MAXMB_MASK                       0x7Fu
#define CAN_MCR_MAXMB_SHIFT                      0u
#define CAN_MCR_MAXMB_WIDTH                      7u
#define CAN_MCR_MAXMB(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_MCR_MAXMB_SHIFT))&CAN_MCR_MAXMB_MASK)
#define CAN_MCR_IDAM_MASK                        0x300u
#define CAN_MCR_IDAM_SHIFT                       8u
#define CAN_MCR_IDAM_WIDTH                       2u
#define CAN_MCR_IDAM(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_IDAM_SHIFT))&CAN_MCR_IDAM_MASK)
#define CAN_MCR_FDEN_MASK                        0x800u
#define CAN_MCR_FDEN_SHIFT                       11u
#define CAN_MCR_FDEN_WIDTH                       1u
#define CAN_MCR_FDEN(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_FDEN_SHIFT))&CAN_MCR_FDEN_MASK)
#define CAN_MCR_AEN_MASK                         0x1000u
#define CAN_MCR_AEN_SHIFT                        12u
#define CAN_MCR_AEN_WIDTH                        1u
#define CAN_MCR_AEN(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_MCR_AEN_SHIFT))&CAN_MCR_AEN_MASK)
#define CAN_MCR_LPRIOEN_MASK                     0x2000u
#define CAN_MCR_LPRIOEN_SHIFT                    13u
#define CAN_MCR_LPRIOEN_WIDTH                    1u
#define CAN_MCR_LPRIOEN(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_MCR_LPRIOEN_SHIFT))&CAN_MCR_LPRIOEN_MASK)
#define CAN_MCR_PNET_EN_MASK                     0x4000u
#define CAN_MCR_PNET_EN_SHIFT                    14u
#define CAN_MCR_PNET_EN_WIDTH                    1u
#define CAN_MCR_PNET_EN(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_MCR_PNET_EN_SHIFT))&CAN_MCR_PNET_EN_MASK)
#define CAN_MCR_DMA_MASK                         0x8000u
#define CAN_MCR_DMA_SHIFT                        15u
#define CAN_MCR_DMA_WIDTH                        1u
#define CAN_MCR_DMA(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_MCR_DMA_SHIFT))&CAN_MCR_DMA_MASK)
#define CAN_MCR_IRMQ_MASK                        0x10000u
#define CAN_MCR_IRMQ_SHIFT                       16u
#define CAN_MCR_IRMQ_WIDTH                       1u
#define CAN_MCR_IRMQ(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_IRMQ_SHIFT))&CAN_MCR_IRMQ_MASK)
#define CAN_MCR_SRXDIS_MASK                      0x20000u
#define CAN_MCR_SRXDIS_SHIFT                     17u
#define CAN_MCR_SRXDIS_WIDTH                     1u
#define CAN_MCR_SRXDIS(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_SRXDIS_SHIFT))&CAN_MCR_SRXDIS_MASK)
#define CAN_MCR_WAKSRC_MASK                      0x80000u
#define CAN_MCR_WAKSRC_SHIFT                     19u
#define CAN_MCR_WAKSRC_WIDTH                     1u
#define CAN_MCR_WAKSRC(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_WAKSRC_SHIFT))&CAN_MCR_WAKSRC_MASK)
#define CAN_MCR_LPMACK_MASK                      0x100000u
#define CAN_MCR_LPMACK_SHIFT                     20u
#define CAN_MCR_LPMACK_WIDTH                     1u
#define CAN_MCR_LPMACK(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_LPMACK_SHIFT))&CAN_MCR_LPMACK_MASK)
#define CAN_MCR_WRNEN_MASK                       0x200000u
#define CAN_MCR_WRNEN_SHIFT                      21u
#define CAN_MCR_WRNEN_WIDTH                      1u
#define CAN_MCR_WRNEN(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_MCR_WRNEN_SHIFT))&CAN_MCR_WRNEN_MASK)
#define CAN_MCR_SLFWAK_MASK                      0x400000u
#define CAN_MCR_SLFWAK_SHIFT                     22u
#define CAN_MCR_SLFWAK_WIDTH                     1u
#define CAN_MCR_SLFWAK(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_SLFWAK_SHIFT))&CAN_MCR_SLFWAK_MASK)
#define CAN_MCR_SUPV_MASK                        0x800000u
#define CAN_MCR_SUPV_SHIFT                       23u
#define CAN_MCR_SUPV_WIDTH                       1u
#define CAN_MCR_SUPV(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_SUPV_SHIFT))&CAN_MCR_SUPV_MASK)
#define CAN_MCR_FRZACK_MASK                      0x1000000u
#define CAN_MCR_FRZACK_SHIFT                     24u
#define CAN_MCR_FRZACK_WIDTH                     1u
#define CAN_MCR_FRZACK(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_FRZACK_SHIFT))&CAN_MCR_FRZACK_MASK)
#define CAN_MCR_SOFTRST_MASK                     0x2000000u
#define CAN_MCR_SOFTRST_SHIFT                    25u
#define CAN_MCR_SOFTRST_WIDTH                    1u
#define CAN_MCR_SOFTRST(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_MCR_SOFTRST_SHIFT))&CAN_MCR_SOFTRST_MASK)
#define CAN_MCR_WAKMSK_MASK                      0x4000000u
#define CAN_MCR_WAKMSK_SHIFT                     26u
#define CAN_MCR_WAKMSK_WIDTH                     1u
#define CAN_MCR_WAKMSK(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_WAKMSK_SHIFT))&CAN_MCR_WAKMSK_MASK)
#define CAN_MCR_NOTRDY_MASK                      0x8000000u
#define CAN_MCR_NOTRDY_SHIFT                     27u
#define CAN_MCR_NOTRDY_WIDTH                     1u
#define CAN_MCR_NOTRDY(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_MCR_NOTRDY_SHIFT))&CAN_MCR_NOTRDY_MASK)
#define CAN_MCR_HALT_MASK                        0x10000000u
#define CAN_MCR_HALT_SHIFT                       28u
#define CAN_MCR_HALT_WIDTH                       1u
#define CAN_MCR_HALT(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_HALT_SHIFT))&CAN_MCR_HALT_MASK)
#define CAN_MCR_RFEN_MASK                        0x20000000u
#define CAN_MCR_RFEN_SHIFT                       29u
#define CAN_MCR_RFEN_WIDTH                       1u
#define CAN_MCR_RFEN(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_RFEN_SHIFT))&CAN_MCR_RFEN_MASK)
#define CAN_MCR_FRZ_MASK                         0x40000000u
#define CAN_MCR_FRZ_SHIFT                        30u
#define CAN_MCR_FRZ_WIDTH                        1u
#define CAN_MCR_FRZ(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_MCR_FRZ_SHIFT))&CAN_MCR_FRZ_MASK)
#define CAN_MCR_MDIS_MASK                        0x80000000u
#define CAN_MCR_MDIS_SHIFT                       31u
#define CAN_MCR_MDIS_WIDTH                       1u
#define CAN_MCR_MDIS(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_MCR_MDIS_SHIFT))&CAN_MCR_MDIS_MASK)
/* CTRL1 Bit Fields */
#define CAN_CTRL1_PROPSEG_MASK                   0x7u
#define CAN_CTRL1_PROPSEG_SHIFT                  0u
#define CAN_CTRL1_PROPSEG_WIDTH                  3u
#define CAN_CTRL1_PROPSEG(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PROPSEG_SHIFT))&CAN_CTRL1_PROPSEG_MASK)
#define CAN_CTRL1_LOM_MASK                       0x8u
#define CAN_CTRL1_LOM_SHIFT                      3u
#define CAN_CTRL1_LOM_WIDTH                      1u
#define CAN_CTRL1_LOM(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_LOM_SHIFT))&CAN_CTRL1_LOM_MASK)
#define CAN_CTRL1_LBUF_MASK                      0x10u
#define CAN_CTRL1_LBUF_SHIFT                     4u
#define CAN_CTRL1_LBUF_WIDTH                     1u
#define CAN_CTRL1_LBUF(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_LBUF_SHIFT))&CAN_CTRL1_LBUF_MASK)
#define CAN_CTRL1_TSYN_MASK                      0x20u
#define CAN_CTRL1_TSYN_SHIFT                     5u
#define CAN_CTRL1_TSYN_WIDTH                     1u
#define CAN_CTRL1_TSYN(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_TSYN_SHIFT))&CAN_CTRL1_TSYN_MASK)
#define CAN_CTRL1_BOFFREC_MASK                   0x40u
#define CAN_CTRL1_BOFFREC_SHIFT                  6u
#define CAN_CTRL1_BOFFREC_WIDTH                  1u
#define CAN_CTRL1_BOFFREC(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_BOFFREC_SHIFT))&CAN_CTRL1_BOFFREC_MASK)
#define CAN_CTRL1_SMP_MASK                       0x80u
#define CAN_CTRL1_SMP_SHIFT                      7u
#define CAN_CTRL1_SMP_WIDTH                      1u
#define CAN_CTRL1_SMP(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_SMP_SHIFT))&CAN_CTRL1_SMP_MASK)
#define CAN_CTRL1_RWRNMSK_MASK                   0x400u
#define CAN_CTRL1_RWRNMSK_SHIFT                  10u
#define CAN_CTRL1_RWRNMSK_WIDTH                  1u
#define CAN_CTRL1_RWRNMSK(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_RWRNMSK_SHIFT))&CAN_CTRL1_RWRNMSK_MASK)
#define CAN_CTRL1_TWRNMSK_MASK                   0x800u
#define CAN_CTRL1_TWRNMSK_SHIFT                  11u
#define CAN_CTRL1_TWRNMSK_WIDTH                  1u
#define CAN_CTRL1_TWRNMSK(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_TWRNMSK_SHIFT))&CAN_CTRL1_TWRNMSK_MASK)
#define CAN_CTRL1_LPB_MASK                       0x1000u
#define CAN_CTRL1_LPB_SHIFT                      12u
#define CAN_CTRL1_LPB_WIDTH                      1u
#define CAN_CTRL1_LPB(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_LPB_SHIFT))&CAN_CTRL1_LPB_MASK)
#define CAN_CTRL1_CLKSRC_MASK                    0x2000u
#define CAN_CTRL1_CLKSRC_SHIFT                   13u
#define CAN_CTRL1_CLKSRC_WIDTH                   1u
#define CAN_CTRL1_CLKSRC(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_CLKSRC_SHIFT))&CAN_CTRL1_CLKSRC_MASK)
#define CAN_CTRL1_ERRMSK_MASK                    0x4000u
#define CAN_CTRL1_ERRMSK_SHIFT                   14u
#define CAN_CTRL1_ERRMSK_WIDTH                   1u
#define CAN_CTRL1_ERRMSK(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_ERRMSK_SHIFT))&CAN_CTRL1_ERRMSK_MASK)
#define CAN_CTRL1_BOFFMSK_MASK                   0x8000u
#define CAN_CTRL1_BOFFMSK_SHIFT                  15u
#define CAN_CTRL1_BOFFMSK_WIDTH                  1u
#define CAN_CTRL1_BOFFMSK(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_BOFFMSK_SHIFT))&CAN_CTRL1_BOFFMSK_MASK)
#define CAN_CTRL1_PSEG2_MASK                     0x70000u
#define CAN_CTRL1_PSEG2_SHIFT                    16u
#define CAN_CTRL1_PSEG2_WIDTH                    3u
#define CAN_CTRL1_PSEG2(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PSEG2_SHIFT))&CAN_CTRL1_PSEG2_MASK)
#define CAN_CTRL1_PSEG1_MASK                     0x380000u
#define CAN_CTRL1_PSEG1_SHIFT                    19u
#define CAN_CTRL1_PSEG1_WIDTH                    3u
#define CAN_CTRL1_PSEG1(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PSEG1_SHIFT))&CAN_CTRL1_PSEG1_MASK)
#define CAN_CTRL1_RJW_MASK                       0xC00000u
#define CAN_CTRL1_RJW_SHIFT                      22u
#define CAN_CTRL1_RJW_WIDTH                      2u
#define CAN_CTRL1_RJW(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_RJW_SHIFT))&CAN_CTRL1_RJW_MASK)
#define CAN_CTRL1_PRESDIV_MASK                   0xFF000000u
#define CAN_CTRL1_PRESDIV_SHIFT                  24u
#define CAN_CTRL1_PRESDIV_WIDTH                  8u
#define CAN_CTRL1_PRESDIV(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PRESDIV_SHIFT))&CAN_CTRL1_PRESDIV_MASK)
/* TIMER Bit Fields */
#define CAN_TIMER_TIMER_MASK                     0xFFFFu
#define CAN_TIMER_TIMER_SHIFT                    0u
#define CAN_TIMER_TIMER_WIDTH                    16u
#define CAN_TIMER_TIMER(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_TIMER_TIMER_SHIFT))&CAN_TIMER_TIMER_MASK)
/* RXMGMASK Bit Fields */
#define CAN_RXMGMASK_MG_MASK                     0xFFFFFFFFu
#define CAN_RXMGMASK_MG_SHIFT                    0u
#define CAN_RXMGMASK_MG_WIDTH                    32u
#define CAN_RXMGMASK_MG(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_RXMGMASK_MG_SHIFT))&CAN_RXMGMASK_MG_MASK)
/* RX14MASK Bit Fields */
#define CAN_RX14MASK_RX14M_MASK                  0xFFFFFFFFu
#define CAN_RX14MASK_RX14M_SHIFT                 0u
#define CAN_RX14MASK_RX14M_WIDTH                 32u
#define CAN_RX14MASK_RX14M(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_RX14MASK_RX14M_SHIFT))&CAN_RX14MASK_RX14M_MASK)
/* RX15MASK Bit Fields */
#define CAN_RX15MASK_RX15M_MASK                  0xFFFFFFFFu
#define CAN_RX15MASK_RX15M_SHIFT                 0u
#define CAN_RX15MASK_RX15M_WIDTH                 32u
#define CAN_RX15MASK_RX15M(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_RX15MASK_RX15M_SHIFT))&CAN_RX15MASK_RX15M_MASK)
/* ECR Bit Fields */
#define CAN_ECR_TXERRCNT_MASK                    0xFFu
#define CAN_ECR_TXERRCNT_SHIFT                   0u
#define CAN_ECR_TXERRCNT_WIDTH                   8u
#define CAN_ECR_TXERRCNT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ECR_TXERRCNT_SHIFT))&CAN_ECR_TXERRCNT_MASK)
#define CAN_ECR_RXERRCNT_MASK                    0xFF00u
#define CAN_ECR_RXERRCNT_SHIFT                   8u
#define CAN_ECR_RXERRCNT_WIDTH                   8u
#define CAN_ECR_RXERRCNT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ECR_RXERRCNT_SHIFT))&CAN_ECR_RXERRCNT_MASK)
#define CAN_ECR_TXERRCNT_FAST_MASK               0xFF0000u
#define CAN_ECR_TXERRCNT_FAST_SHIFT              16u
#define CAN_ECR_TXERRCNT_FAST_WIDTH              8u
#define CAN_ECR_TXERRCNT_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_ECR_TXERRCNT_FAST_SHIFT))&CAN_ECR_TXERRCNT_FAST_MASK)
#define CAN_ECR_RXERRCNT_FAST_MASK               0xFF000000u
#define CAN_ECR_RXERRCNT_FAST_SHIFT              24u
#define CAN_ECR_RXERRCNT_FAST_WIDTH              8u
#define CAN_ECR_RXERRCNT_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_ECR_RXERRCNT_FAST_SHIFT))&CAN_ECR_RXERRCNT_FAST_MASK)
/* ESR1 Bit Fields */
#define CAN_ESR1_WAKINT_MASK                     0x1u
#define CAN_ESR1_WAKINT_SHIFT                    0u
#define CAN_ESR1_WAKINT_WIDTH                    1u
#define CAN_ESR1_WAKINT(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_WAKINT_SHIFT))&CAN_ESR1_WAKINT_MASK)
#define CAN_ESR1_ERRINT_MASK                     0x2u
#define CAN_ESR1_ERRINT_SHIFT                    1u
#define CAN_ESR1_ERRINT_WIDTH                    1u
#define CAN_ESR1_ERRINT(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_ERRINT_SHIFT))&CAN_ESR1_ERRINT_MASK)
#define CAN_ESR1_BOFFINT_MASK                    0x4u
#define CAN_ESR1_BOFFINT_SHIFT                   2u
#define CAN_ESR1_BOFFINT_WIDTH                   1u
#define CAN_ESR1_BOFFINT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BOFFINT_SHIFT))&CAN_ESR1_BOFFINT_MASK)
#define CAN_ESR1_RX_MASK                         0x8u
#define CAN_ESR1_RX_SHIFT                        3u
#define CAN_ESR1_RX_WIDTH                        1u
#define CAN_ESR1_RX(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_RX_SHIFT))&CAN_ESR1_RX_MASK)
#define CAN_ESR1_FLTCONF_MASK                    0x30u
#define CAN_ESR1_FLTCONF_SHIFT                   4u
#define CAN_ESR1_FLTCONF_WIDTH                   2u
#define CAN_ESR1_FLTCONF(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_FLTCONF_SHIFT))&CAN_ESR1_FLTCONF_MASK)
#define CAN_ESR1_TX_MASK                         0x40u
#define CAN_ESR1_TX_SHIFT                        6u
#define CAN_ESR1_TX_WIDTH                        1u
#define CAN_ESR1_TX(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_TX_SHIFT))&CAN_ESR1_TX_MASK)
#define CAN_ESR1_IDLE_MASK                       0x80u
#define CAN_ESR1_IDLE_SHIFT                      7u
#define CAN_ESR1_IDLE_WIDTH                      1u
#define CAN_ESR1_IDLE(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_IDLE_SHIFT))&CAN_ESR1_IDLE_MASK)
#define CAN_ESR1_RXWRN_MASK                      0x100u
#define CAN_ESR1_RXWRN_SHIFT                     8u
#define CAN_ESR1_RXWRN_WIDTH                     1u
#define CAN_ESR1_RXWRN(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_RXWRN_SHIFT))&CAN_ESR1_RXWRN_MASK)
#define CAN_ESR1_TXWRN_MASK                      0x200u
#define CAN_ESR1_TXWRN_SHIFT                     9u
#define CAN_ESR1_TXWRN_WIDTH                     1u
#define CAN_ESR1_TXWRN(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_TXWRN_SHIFT))&CAN_ESR1_TXWRN_MASK)
#define CAN_ESR1_STFERR_MASK                     0x400u
#define CAN_ESR1_STFERR_SHIFT                    10u
#define CAN_ESR1_STFERR_WIDTH                    1u
#define CAN_ESR1_STFERR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_STFERR_SHIFT))&CAN_ESR1_STFERR_MASK)
#define CAN_ESR1_FRMERR_MASK                     0x800u
#define CAN_ESR1_FRMERR_SHIFT                    11u
#define CAN_ESR1_FRMERR_WIDTH                    1u
#define CAN_ESR1_FRMERR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_FRMERR_SHIFT))&CAN_ESR1_FRMERR_MASK)
#define CAN_ESR1_CRCERR_MASK                     0x1000u
#define CAN_ESR1_CRCERR_SHIFT                    12u
#define CAN_ESR1_CRCERR_WIDTH                    1u
#define CAN_ESR1_CRCERR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_CRCERR_SHIFT))&CAN_ESR1_CRCERR_MASK)
#define CAN_ESR1_ACKERR_MASK                     0x2000u
#define CAN_ESR1_ACKERR_SHIFT                    13u
#define CAN_ESR1_ACKERR_WIDTH                    1u
#define CAN_ESR1_ACKERR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_ACKERR_SHIFT))&CAN_ESR1_ACKERR_MASK)
#define CAN_ESR1_BIT0ERR_MASK                    0x4000u
#define CAN_ESR1_BIT0ERR_SHIFT                   14u
#define CAN_ESR1_BIT0ERR_WIDTH                   1u
#define CAN_ESR1_BIT0ERR(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BIT0ERR_SHIFT))&CAN_ESR1_BIT0ERR_MASK)
#define CAN_ESR1_BIT1ERR_MASK                    0x8000u
#define CAN_ESR1_BIT1ERR_SHIFT                   15u
#define CAN_ESR1_BIT1ERR_WIDTH                   1u
#define CAN_ESR1_BIT1ERR(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BIT1ERR_SHIFT))&CAN_ESR1_BIT1ERR_MASK)
#define CAN_ESR1_RWRNINT_MASK                    0x10000u
#define CAN_ESR1_RWRNINT_SHIFT                   16u
#define CAN_ESR1_RWRNINT_WIDTH                   1u
#define CAN_ESR1_RWRNINT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_RWRNINT_SHIFT))&CAN_ESR1_RWRNINT_MASK)
#define CAN_ESR1_TWRNINT_MASK                    0x20000u
#define CAN_ESR1_TWRNINT_SHIFT                   17u
#define CAN_ESR1_TWRNINT_WIDTH                   1u
#define CAN_ESR1_TWRNINT(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_TWRNINT_SHIFT))&CAN_ESR1_TWRNINT_MASK)
#define CAN_ESR1_SYNCH_MASK                      0x40000u
#define CAN_ESR1_SYNCH_SHIFT                     18u
#define CAN_ESR1_SYNCH_WIDTH                     1u
#define CAN_ESR1_SYNCH(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_SYNCH_SHIFT))&CAN_ESR1_SYNCH_MASK)
#define CAN_ESR1_BOFFDONEINT_MASK                0x80000u
#define CAN_ESR1_BOFFDONEINT_SHIFT               19u
#define CAN_ESR1_BOFFDONEINT_WIDTH               1u
#define CAN_ESR1_BOFFDONEINT(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BOFFDONEINT_SHIFT))&CAN_ESR1_BOFFDONEINT_MASK)
#define CAN_ESR1_ERRINT_FAST_MASK                0x100000u
#define CAN_ESR1_ERRINT_FAST_SHIFT               20u
#define CAN_ESR1_ERRINT_FAST_WIDTH               1u
#define CAN_ESR1_ERRINT_FAST(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_ERRINT_FAST_SHIFT))&CAN_ESR1_ERRINT_FAST_MASK)
#define CAN_ESR1_ERROVR_MASK                     0x200000u
#define CAN_ESR1_ERROVR_SHIFT                    21u
#define CAN_ESR1_ERROVR_WIDTH                    1u
#define CAN_ESR1_ERROVR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_ERROVR_SHIFT))&CAN_ESR1_ERROVR_MASK)
#define CAN_ESR1_STFERR_FAST_MASK                0x4000000u
#define CAN_ESR1_STFERR_FAST_SHIFT               26u
#define CAN_ESR1_STFERR_FAST_WIDTH               1u
#define CAN_ESR1_STFERR_FAST(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_STFERR_FAST_SHIFT))&CAN_ESR1_STFERR_FAST_MASK)
#define CAN_ESR1_FRMERR_FAST_MASK                0x8000000u
#define CAN_ESR1_FRMERR_FAST_SHIFT               27u
#define CAN_ESR1_FRMERR_FAST_WIDTH               1u
#define CAN_ESR1_FRMERR_FAST(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_FRMERR_FAST_SHIFT))&CAN_ESR1_FRMERR_FAST_MASK)
#define CAN_ESR1_CRCERR_FAST_MASK                0x10000000u
#define CAN_ESR1_CRCERR_FAST_SHIFT               28u
#define CAN_ESR1_CRCERR_FAST_WIDTH               1u
#define CAN_ESR1_CRCERR_FAST(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_CRCERR_FAST_SHIFT))&CAN_ESR1_CRCERR_FAST_MASK)
#define CAN_ESR1_BIT0ERR_FAST_MASK               0x40000000u
#define CAN_ESR1_BIT0ERR_FAST_SHIFT              30u
#define CAN_ESR1_BIT0ERR_FAST_WIDTH              1u
#define CAN_ESR1_BIT0ERR_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BIT0ERR_FAST_SHIFT))&CAN_ESR1_BIT0ERR_FAST_MASK)
#define CAN_ESR1_BIT1ERR_FAST_MASK               0x80000000u
#define CAN_ESR1_BIT1ERR_FAST_SHIFT              31u
#define CAN_ESR1_BIT1ERR_FAST_WIDTH              1u
#define CAN_ESR1_BIT1ERR_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_ESR1_BIT1ERR_FAST_SHIFT))&CAN_ESR1_BIT1ERR_FAST_MASK)
/* IMASK2 Bit Fields */
#define CAN_IMASK2_BUF63TO32M_MASK               0xFFFFFFFFu
#define CAN_IMASK2_BUF63TO32M_SHIFT              0u
#define CAN_IMASK2_BUF63TO32M_WIDTH              32u
#define CAN_IMASK2_BUF63TO32M(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_IMASK2_BUF63TO32M_SHIFT))&CAN_IMASK2_BUF63TO32M_MASK)
/* IMASK1 Bit Fields */
#define CAN_IMASK1_BUF31TO0M_MASK                0xFFFFFFFFu
#define CAN_IMASK1_BUF31TO0M_SHIFT               0u
#define CAN_IMASK1_BUF31TO0M_WIDTH               32u
#define CAN_IMASK1_BUF31TO0M(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_IMASK1_BUF31TO0M_SHIFT))&CAN_IMASK1_BUF31TO0M_MASK)
/* IFLAG2 Bit Fields */
#define CAN_IFLAG2_BUF63TO32I_MASK               0xFFFFFFFFu
#define CAN_IFLAG2_BUF63TO32I_SHIFT              0u
#define CAN_IFLAG2_BUF63TO32I_WIDTH              32u
#define CAN_IFLAG2_BUF63TO32I(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG2_BUF63TO32I_SHIFT))&CAN_IFLAG2_BUF63TO32I_MASK)
/* IFLAG1 Bit Fields */
#define CAN_IFLAG1_BUF0I_MASK                    0x1u
#define CAN_IFLAG1_BUF0I_SHIFT                   0u
#define CAN_IFLAG1_BUF0I_WIDTH                   1u
#define CAN_IFLAG1_BUF0I(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF0I_SHIFT))&CAN_IFLAG1_BUF0I_MASK)
#define CAN_IFLAG1_BUF4TO1I_MASK                 0x1Eu
#define CAN_IFLAG1_BUF4TO1I_SHIFT                1u
#define CAN_IFLAG1_BUF4TO1I_WIDTH                4u
#define CAN_IFLAG1_BUF4TO1I(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF4TO1I_SHIFT))&CAN_IFLAG1_BUF4TO1I_MASK)
#define CAN_IFLAG1_BUF5I_MASK                    0x20u
#define CAN_IFLAG1_BUF5I_SHIFT                   5u
#define CAN_IFLAG1_BUF5I_WIDTH                   1u
#define CAN_IFLAG1_BUF5I(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF5I_SHIFT))&CAN_IFLAG1_BUF5I_MASK)
#define CAN_IFLAG1_BUF6I_MASK                    0x40u
#define CAN_IFLAG1_BUF6I_SHIFT                   6u
#define CAN_IFLAG1_BUF6I_WIDTH                   1u
#define CAN_IFLAG1_BUF6I(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF6I_SHIFT))&CAN_IFLAG1_BUF6I_MASK)
#define CAN_IFLAG1_BUF7I_MASK                    0x80u
#define CAN_IFLAG1_BUF7I_SHIFT                   7u
#define CAN_IFLAG1_BUF7I_WIDTH                   1u
#define CAN_IFLAG1_BUF7I(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF7I_SHIFT))&CAN_IFLAG1_BUF7I_MASK)
#define CAN_IFLAG1_BUF31TO8I_MASK                0xFFFFFF00u
#define CAN_IFLAG1_BUF31TO8I_SHIFT               8u
#define CAN_IFLAG1_BUF31TO8I_WIDTH               24u
#define CAN_IFLAG1_BUF31TO8I(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG1_BUF31TO8I_SHIFT))&CAN_IFLAG1_BUF31TO8I_MASK)
/* CTRL2 Bit Fields */
#define CAN_CTRL2_EDFLTDIS_MASK                  0x800u
#define CAN_CTRL2_EDFLTDIS_SHIFT                 11u
#define CAN_CTRL2_EDFLTDIS_WIDTH                 1u
#define CAN_CTRL2_EDFLTDIS(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_EDFLTDIS_SHIFT))&CAN_CTRL2_EDFLTDIS_MASK)
#define CAN_CTRL2_ISOCANFDEN_MASK                0x1000u
#define CAN_CTRL2_ISOCANFDEN_SHIFT               12u
#define CAN_CTRL2_ISOCANFDEN_WIDTH               1u
#define CAN_CTRL2_ISOCANFDEN(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_ISOCANFDEN_SHIFT))&CAN_CTRL2_ISOCANFDEN_MASK)
#define CAN_CTRL2_PREXCEN_MASK                   0x4000u
#define CAN_CTRL2_PREXCEN_SHIFT                  14u
#define CAN_CTRL2_PREXCEN_WIDTH                  1u
#define CAN_CTRL2_PREXCEN(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_PREXCEN_SHIFT))&CAN_CTRL2_PREXCEN_MASK)
#define CAN_CTRL2_TIMER_SRC_MASK                 0x8000u
#define CAN_CTRL2_TIMER_SRC_SHIFT                15u
#define CAN_CTRL2_TIMER_SRC_WIDTH                1u
#define CAN_CTRL2_TIMER_SRC(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_TIMER_SRC_SHIFT))&CAN_CTRL2_TIMER_SRC_MASK)
#define CAN_CTRL2_EACEN_MASK                     0x10000u
#define CAN_CTRL2_EACEN_SHIFT                    16u
#define CAN_CTRL2_EACEN_WIDTH                    1u
#define CAN_CTRL2_EACEN(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_EACEN_SHIFT))&CAN_CTRL2_EACEN_MASK)
#define CAN_CTRL2_RRS_MASK                       0x20000u
#define CAN_CTRL2_RRS_SHIFT                      17u
#define CAN_CTRL2_RRS_WIDTH                      1u
#define CAN_CTRL2_RRS(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_RRS_SHIFT))&CAN_CTRL2_RRS_MASK)
#define CAN_CTRL2_MRP_MASK                       0x40000u
#define CAN_CTRL2_MRP_SHIFT                      18u
#define CAN_CTRL2_MRP_WIDTH                      1u
#define CAN_CTRL2_MRP(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_MRP_SHIFT))&CAN_CTRL2_MRP_MASK)
#define CAN_CTRL2_TASD_MASK                      0xF80000u
#define CAN_CTRL2_TASD_SHIFT                     19u
#define CAN_CTRL2_TASD_WIDTH                     5u
#define CAN_CTRL2_TASD(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_TASD_SHIFT))&CAN_CTRL2_TASD_MASK)
#define CAN_CTRL2_RFFN_MASK                      0xF000000u
#define CAN_CTRL2_RFFN_SHIFT                     24u
#define CAN_CTRL2_RFFN_WIDTH                     4u
#define CAN_CTRL2_RFFN(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_RFFN_SHIFT))&CAN_CTRL2_RFFN_MASK)
#define CAN_CTRL2_BOFFDONEMSK_MASK               0x40000000u
#define CAN_CTRL2_BOFFDONEMSK_SHIFT              30u
#define CAN_CTRL2_BOFFDONEMSK_WIDTH              1u
#define CAN_CTRL2_BOFFDONEMSK(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_BOFFDONEMSK_SHIFT))&CAN_CTRL2_BOFFDONEMSK_MASK)
#define CAN_CTRL2_ERRMSK_FAST_MASK               0x80000000u
#define CAN_CTRL2_ERRMSK_FAST_SHIFT              31u
#define CAN_CTRL2_ERRMSK_FAST_WIDTH              1u
#define CAN_CTRL2_ERRMSK_FAST(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_ERRMSK_FAST_SHIFT))&CAN_CTRL2_ERRMSK_FAST_MASK)
/* ESR2 Bit Fields */
#define CAN_ESR2_IMB_MASK                        0x2000u
#define CAN_ESR2_IMB_SHIFT                       13u
#define CAN_ESR2_IMB_WIDTH                       1u
#define CAN_ESR2_IMB(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_ESR2_IMB_SHIFT))&CAN_ESR2_IMB_MASK)
#define CAN_ESR2_VPS_MASK                        0x4000u
#define CAN_ESR2_VPS_SHIFT                       14u
#define CAN_ESR2_VPS_WIDTH                       1u
#define CAN_ESR2_VPS(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_ESR2_VPS_SHIFT))&CAN_ESR2_VPS_MASK)
#define CAN_ESR2_LPTM_MASK                       0x7F0000u
#define CAN_ESR2_LPTM_SHIFT                      16u
#define CAN_ESR2_LPTM_WIDTH                      7u
#define CAN_ESR2_LPTM(x)                         (((uint32_t)(((uint32_t)(x))<<CAN_ESR2_LPTM_SHIFT))&CAN_ESR2_LPTM_MASK)
/* CRCR Bit Fields */
#define CAN_CRCR_TXCRC_MASK                      0x7FFFu
#define CAN_CRCR_TXCRC_SHIFT                     0u
#define CAN_CRCR_TXCRC_WIDTH                     15u
#define CAN_CRCR_TXCRC(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CRCR_TXCRC_SHIFT))&CAN_CRCR_TXCRC_MASK)
#define CAN_CRCR_MBCRC_MASK                      0x7F0000u
#define CAN_CRCR_MBCRC_SHIFT                     16u
#define CAN_CRCR_MBCRC_WIDTH                     7u
#define CAN_CRCR_MBCRC(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CRCR_MBCRC_SHIFT))&CAN_CRCR_MBCRC_MASK)
/* RXFGMASK Bit Fields */
#define CAN_RXFGMASK_FGM_MASK                    0xFFFFFFFFu
#define CAN_RXFGMASK_FGM_SHIFT                   0u
#define CAN_RXFGMASK_FGM_WIDTH                   32u
#define CAN_RXFGMASK_FGM(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_RXFGMASK_FGM_SHIFT))&CAN_RXFGMASK_FGM_MASK)
/* RXFIR Bit Fields */
#define CAN_RXFIR_IDHIT_MASK                     0x1FFu
#define CAN_RXFIR_IDHIT_SHIFT                    0u
#define CAN_RXFIR_IDHIT_WIDTH                    9u
#define CAN_RXFIR_IDHIT(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_RXFIR_IDHIT_SHIFT))&CAN_RXFIR_IDHIT_MASK)
/* CBT Bit Fields */
#define CAN_CBT_EPSEG2_MASK                      0x1Fu
#define CAN_CBT_EPSEG2_SHIFT                     0u
#define CAN_CBT_EPSEG2_WIDTH                     5u
#define CAN_CBT_EPSEG2(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CBT_EPSEG2_SHIFT))&CAN_CBT_EPSEG2_MASK)
#define CAN_CBT_EPSEG1_MASK                      0x3E0u
#define CAN_CBT_EPSEG1_SHIFT                     5u
#define CAN_CBT_EPSEG1_WIDTH                     5u
#define CAN_CBT_EPSEG1(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_CBT_EPSEG1_SHIFT))&CAN_CBT_EPSEG1_MASK)
#define CAN_CBT_EPROPSEG_MASK                    0xFC00u
#define CAN_CBT_EPROPSEG_SHIFT                   10u
#define CAN_CBT_EPROPSEG_WIDTH                   6u
#define CAN_CBT_EPROPSEG(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CBT_EPROPSEG_SHIFT))&CAN_CBT_EPROPSEG_MASK)
#define CAN_CBT_ERJW_MASK                        0x1F0000u
#define CAN_CBT_ERJW_SHIFT                       16u
#define CAN_CBT_ERJW_WIDTH                       5u
#define CAN_CBT_ERJW(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_CBT_ERJW_SHIFT))&CAN_CBT_ERJW_MASK)
#define CAN_CBT_EPRESDIV_MASK                    0x7FE00000u
#define CAN_CBT_EPRESDIV_SHIFT                   21u
#define CAN_CBT_EPRESDIV_WIDTH                   10u
#define CAN_CBT_EPRESDIV(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CBT_EPRESDIV_SHIFT))&CAN_CBT_EPRESDIV_MASK)
#define CAN_CBT_BTF_MASK                         0x80000000u
#define CAN_CBT_BTF_SHIFT                        31u
#define CAN_CBT_BTF_WIDTH                        1u
#define CAN_CBT_BTF(x)                           (((uint32_t)(((uint32_t)(x))<<CAN_CBT_BTF_SHIFT))&CAN_CBT_BTF_MASK)
/* IMASK3 Bit Fields */
#define CAN_IMASK3_BUF95TO64M_MASK               0xFFFFFFFFu
#define CAN_IMASK3_BUF95TO64M_SHIFT              0u
#define CAN_IMASK3_BUF95TO64M_WIDTH              32u
#define CAN_IMASK3_BUF95TO64M(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_IMASK3_BUF95TO64M_SHIFT))&CAN_IMASK3_BUF95TO64M_MASK)
/* IFLAG3 Bit Fields */
#define CAN_IFLAG3_BUF95TO64_MASK                0xFFFFFFFFu
#define CAN_IFLAG3_BUF95TO64_SHIFT               0u
#define CAN_IFLAG3_BUF95TO64_WIDTH               32u
#define CAN_IFLAG3_BUF95TO64(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_IFLAG3_BUF95TO64_SHIFT))&CAN_IFLAG3_BUF95TO64_MASK)
/* RAMn Bit Fields */
#define CAN_RAMn_DATA_BYTE_3_MASK                0xFFu
#define CAN_RAMn_DATA_BYTE_3_SHIFT               0u
#define CAN_RAMn_DATA_BYTE_3_WIDTH               8u
#define CAN_RAMn_DATA_BYTE_3(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_RAMn_DATA_BYTE_3_SHIFT))&CAN_RAMn_DATA_BYTE_3_MASK)
#define CAN_RAMn_DATA_BYTE_2_MASK                0xFF00u
#define CAN_RAMn_DATA_BYTE_2_SHIFT               8u
#define CAN_RAMn_DATA_BYTE_2_WIDTH               8u
#define CAN_RAMn_DATA_BYTE_2(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_RAMn_DATA_BYTE_2_SHIFT))&CAN_RAMn_DATA_BYTE_2_MASK)
#define CAN_RAMn_DATA_BYTE_1_MASK                0xFF0000u
#define CAN_RAMn_DATA_BYTE_1_SHIFT               16u
#define CAN_RAMn_DATA_BYTE_1_WIDTH               8u
#define CAN_RAMn_DATA_BYTE_1(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_RAMn_DATA_BYTE_1_SHIFT))&CAN_RAMn_DATA_BYTE_1_MASK)
#define CAN_RAMn_DATA_BYTE_0_MASK                0xFF000000u
#define CAN_RAMn_DATA_BYTE_0_SHIFT               24u
#define CAN_RAMn_DATA_BYTE_0_WIDTH               8u
#define CAN_RAMn_DATA_BYTE_0(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_RAMn_DATA_BYTE_0_SHIFT))&CAN_RAMn_DATA_BYTE_0_MASK)
/* RXIMR Bit Fields */
#define CAN_RXIMR_MI_MASK                        0xFFFFFFFFu
#define CAN_RXIMR_MI_SHIFT                       0u
#define CAN_RXIMR_MI_WIDTH                       32u
#define CAN_RXIMR_MI(x)                          (((uint32_t)(((uint32_t)(x))<<CAN_RXIMR_MI_SHIFT))&CAN_RXIMR_MI_MASK)
/* CTRL1_PN Bit Fields */
#define CAN_CTRL1_PN_FCS_MASK                    0x3u
#define CAN_CTRL1_PN_FCS_SHIFT                   0u
#define CAN_CTRL1_PN_FCS_WIDTH                   2u
#define CAN_CTRL1_PN_FCS(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PN_FCS_SHIFT))&CAN_CTRL1_PN_FCS_MASK)
#define CAN_CTRL1_PN_IDFS_MASK                   0xCu
#define CAN_CTRL1_PN_IDFS_SHIFT                  2u
#define CAN_CTRL1_PN_IDFS_WIDTH                  2u
#define CAN_CTRL1_PN_IDFS(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PN_IDFS_SHIFT))&CAN_CTRL1_PN_IDFS_MASK)
#define CAN_CTRL1_PN_PLFS_MASK                   0x30u
#define CAN_CTRL1_PN_PLFS_SHIFT                  4u
#define CAN_CTRL1_PN_PLFS_WIDTH                  2u
#define CAN_CTRL1_PN_PLFS(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PN_PLFS_SHIFT))&CAN_CTRL1_PN_PLFS_MASK)
#define CAN_CTRL1_PN_NMATCH_MASK                 0xFF00u
#define CAN_CTRL1_PN_NMATCH_SHIFT                8u
#define CAN_CTRL1_PN_NMATCH_WIDTH                8u
#define CAN_CTRL1_PN_NMATCH(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PN_NMATCH_SHIFT))&CAN_CTRL1_PN_NMATCH_MASK)
#define CAN_CTRL1_PN_WUMF_MSK_MASK               0x10000u
#define CAN_CTRL1_PN_WUMF_MSK_SHIFT              16u
#define CAN_CTRL1_PN_WUMF_MSK_WIDTH              1u
#define CAN_CTRL1_PN_WUMF_MSK(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PN_WUMF_MSK_SHIFT))&CAN_CTRL1_PN_WUMF_MSK_MASK)
#define CAN_CTRL1_PN_WTOF_MSK_MASK               0x20000u
#define CAN_CTRL1_PN_WTOF_MSK_SHIFT              17u
#define CAN_CTRL1_PN_WTOF_MSK_WIDTH              1u
#define CAN_CTRL1_PN_WTOF_MSK(x)                 (((uint32_t)(((uint32_t)(x))<<CAN_CTRL1_PN_WTOF_MSK_SHIFT))&CAN_CTRL1_PN_WTOF_MSK_MASK)
/* CTRL2_PN Bit Fields */
#define CAN_CTRL2_PN_MATCHTO_MASK                0xFFFFu
#define CAN_CTRL2_PN_MATCHTO_SHIFT               0u
#define CAN_CTRL2_PN_MATCHTO_WIDTH               16u
#define CAN_CTRL2_PN_MATCHTO(x)                  (((uint32_t)(((uint32_t)(x))<<CAN_CTRL2_PN_MATCHTO_SHIFT))&CAN_CTRL2_PN_MATCHTO_MASK)
/* WU_MTC Bit Fields */
#define CAN_WU_MTC_MCOUNTER_MASK                 0xFF00u
#define CAN_WU_MTC_MCOUNTER_SHIFT                8u
#define CAN_WU_MTC_MCOUNTER_WIDTH                8u
#define CAN_WU_MTC_MCOUNTER(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_WU_MTC_MCOUNTER_SHIFT))&CAN_WU_MTC_MCOUNTER_MASK)
#define CAN_WU_MTC_WUMF_MASK                     0x10000u
#define CAN_WU_MTC_WUMF_SHIFT                    16u
#define CAN_WU_MTC_WUMF_WIDTH                    1u
#define CAN_WU_MTC_WUMF(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_WU_MTC_WUMF_SHIFT))&CAN_WU_MTC_WUMF_MASK)
#define CAN_WU_MTC_WTOF_MASK                     0x20000u
#define CAN_WU_MTC_WTOF_SHIFT                    17u
#define CAN_WU_MTC_WTOF_WIDTH                    1u
#define CAN_WU_MTC_WTOF(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_WU_MTC_WTOF_SHIFT))&CAN_WU_MTC_WTOF_MASK)
/* FLT_ID1 Bit Fields */
#define CAN_FLT_ID1_FLT_ID1_MASK                 0x1FFFFFFFu
#define CAN_FLT_ID1_FLT_ID1_SHIFT                0u
#define CAN_FLT_ID1_FLT_ID1_WIDTH                29u
#define CAN_FLT_ID1_FLT_ID1(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_FLT_ID1_FLT_ID1_SHIFT))&CAN_FLT_ID1_FLT_ID1_MASK)
#define CAN_FLT_ID1_FLT_RTR_MASK                 0x20000000u
#define CAN_FLT_ID1_FLT_RTR_SHIFT                29u
#define CAN_FLT_ID1_FLT_RTR_WIDTH                1u
#define CAN_FLT_ID1_FLT_RTR(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_FLT_ID1_FLT_RTR_SHIFT))&CAN_FLT_ID1_FLT_RTR_MASK)
#define CAN_FLT_ID1_FLT_IDE_MASK                 0x40000000u
#define CAN_FLT_ID1_FLT_IDE_SHIFT                30u
#define CAN_FLT_ID1_FLT_IDE_WIDTH                1u
#define CAN_FLT_ID1_FLT_IDE(x)                   (((uint32_t)(((uint32_t)(x))<<CAN_FLT_ID1_FLT_IDE_SHIFT))&CAN_FLT_ID1_FLT_IDE_MASK)
/* FLT_DLC Bit Fields */
#define CAN_FLT_DLC_FLT_DLC_HI_MASK              0xFu
#define CAN_FLT_DLC_FLT_DLC_HI_SHIFT             0u
#define CAN_FLT_DLC_FLT_DLC_HI_WIDTH             4u
#define CAN_FLT_DLC_FLT_DLC_HI(x)                (((uint32_t)(((uint32_t)(x))<<CAN_FLT_DLC_FLT_DLC_HI_SHIFT))&CAN_FLT_DLC_FLT_DLC_HI_MASK)
#define CAN_FLT_DLC_FLT_DLC_LO_MASK              0xF0000u
#define CAN_FLT_DLC_FLT_DLC_LO_SHIFT             16u
#define CAN_FLT_DLC_FLT_DLC_LO_WIDTH             4u
#define CAN_FLT_DLC_FLT_DLC_LO(x)                (((uint32_t)(((uint32_t)(x))<<CAN_FLT_DLC_FLT_DLC_LO_SHIFT))&CAN_FLT_DLC_FLT_DLC_LO_MASK)
/* PL1_LO Bit Fields */
#define CAN_PL1_LO_Data_byte_3_MASK              0xFFu
#define CAN_PL1_LO_Data_byte_3_SHIFT             0u
#define CAN_PL1_LO_Data_byte_3_WIDTH             8u
#define CAN_PL1_LO_Data_byte_3(x)                (((uint32_t)(((uint32_t)(x))<<CAN_PL1_LO_Data_byte_3_SHIFT))&CAN_PL1_LO_Data_byte_3_MASK)
#define CAN_PL1_LO_Data_byte_2_MASK              0xFF00u
#define CAN_PL1_LO_Data_byte_2_SHIFT             8u
#define CAN_PL1_LO_Data_byte_2_WIDTH             8u
#define CAN_PL1_LO_Data_byte_2(x)                (((uint32_t)(((uint32_t)(x))<<CAN_PL1_LO_Data_byte_2_SHIFT))&CAN_PL1_LO_Data_byte_2_MASK)
#define CAN_PL1_LO_Data_byte_1_MASK              0xFF0000u
#define CAN_PL1_LO_Data_byte_1_SHIFT             16u
#define CAN_PL1_LO_Data_byte_1_WIDTH             8u
#define CAN_PL1_LO_Data_byte_1(x)                (((uint32_t)(((uint32_t)(x))<<CAN_PL1_LO_Data_byte_1_SHIFT))&CAN_PL1_LO_Data_byte_1_MASK)
#define CAN_PL1_LO_Data_byte_0_MASK              0xFF000000u
#define CAN_PL1_LO_Data_byte_0_SHIFT             24u
#define CAN_PL1_LO_Data_byte_0_WIDTH             8u
#define CAN_PL1_LO_Data_byte_0(x)                (((uint32_t)(((uint32_t)(x))<<CAN_PL1_LO_Data_byte_0_SHIFT))&CAN_PL1_LO_Data_byte_0_MASK)
/* PL1_HI Bit Fields */
#define CAN_PL1_HI_Data_byte_7_MASK              0xFFu
#define CAN_PL1_HI_Data_byte_7_SHIFT             0u
#define CAN_PL1_HI_Data_byte_7_WIDTH             8u
#define CAN_PL1_HI_Data_byte_7(x)                (((uint32_t)(((uint32_t)(x))<<CAN_PL1_HI_Data_byte_7_SHIFT))&CAN_PL1_HI_Data_byte_7_MASK)
#define CAN_PL1_HI_Data_byte_6_MASK              0xFF00u
#define CAN_PL1_HI_Data_byte_6_SHIFT             8u
#define CAN_PL1_HI_Data_byte_6_WIDTH             8u
#define CAN_PL1_HI_Data_byte_6(x)                (((uint32_t)(((uint32_t)(x))<<CAN_PL1_HI_Data_byte_6_SHIFT))&CAN_PL1_HI_Data_byte_6_MASK)
#define CAN_PL1_HI_Data_byte_5_MASK              0xFF0000u
#define CAN_PL1_HI_Data_byte_5_SHIFT             16u
#define CAN_PL1_HI_Data_byte_5_WIDTH             8u
#define CAN_PL1_HI_Data_byte_5(x)                (((uint32_t)(((uint32_t)(x))<<CAN_PL1_HI_Data_byte_5_SHIFT))&CAN_PL1_HI_Data_byte_5_MASK)
#define CAN_PL1_HI_Data_byte_4_MASK              0xFF000000u
#define CAN_PL1_HI_Data_byte_4_SHIFT             24u
#define CAN_PL1_HI_Data_byte_4_WIDTH             8u
#define CAN_PL1_HI_Data_byte_4(x)                (((uint32_t)(((uint32_t)(x))<<CAN_PL1_HI_Data_byte_4_SHIFT))&CAN_PL1_HI_Data_byte_4_MASK)
/* FLT_ID2_IDMASK Bit Fields */
#define CAN_FLT_ID2_IDMASK_FLT_ID2_IDMASK_MASK   0x1FFFFFFFu
#define CAN_FLT_ID2_IDMASK_FLT_ID2_IDMASK_SHIFT  0u
#define CAN_FLT_ID2_IDMASK_FLT_ID2_IDMASK_WIDTH  29u
#define CAN_FLT_ID2_IDMASK_FLT_ID2_IDMASK(x)     (((uint32_t)(((uint32_t)(x))<<CAN_FLT_ID2_IDMASK_FLT_ID2_IDMASK_SHIFT))&CAN_FLT_ID2_IDMASK_FLT_ID2_IDMASK_MASK)
#define CAN_FLT_ID2_IDMASK_RTR_MSK_MASK          0x20000000u
#define CAN_FLT_ID2_IDMASK_RTR_MSK_SHIFT         29u
#define CAN_FLT_ID2_IDMASK_RTR_MSK_WIDTH         1u
#define CAN_FLT_ID2_IDMASK_RTR_MSK(x)            (((uint32_t)(((uint32_t)(x))<<CAN_FLT_ID2_IDMASK_RTR_MSK_SHIFT))&CAN_FLT_ID2_IDMASK_RTR_MSK_MASK)
#define CAN_FLT_ID2_IDMASK_IDE_MSK_MASK          0x40000000u
#define CAN_FLT_ID2_IDMASK_IDE_MSK_SHIFT         30u
#define CAN_FLT_ID2_IDMASK_IDE_MSK_WIDTH         1u
#define CAN_FLT_ID2_IDMASK_IDE_MSK(x)            (((uint32_t)(((uint32_t)(x))<<CAN_FLT_ID2_IDMASK_IDE_MSK_SHIFT))&CAN_FLT_ID2_IDMASK_IDE_MSK_MASK)
/* PL2_PLMASK_LO Bit Fields */
#define CAN_PL2_PLMASK_LO_Data_byte_3_MASK       0xFFu
#define CAN_PL2_PLMASK_LO_Data_byte_3_SHIFT      0u
#define CAN_PL2_PLMASK_LO_Data_byte_3_WIDTH      8u
#define CAN_PL2_PLMASK_LO_Data_byte_3(x)         (((uint32_t)(((uint32_t)(x))<<CAN_PL2_PLMASK_LO_Data_byte_3_SHIFT))&CAN_PL2_PLMASK_LO_Data_byte_3_MASK)
#define CAN_PL2_PLMASK_LO_Data_byte_2_MASK       0xFF00u
#define CAN_PL2_PLMASK_LO_Data_byte_2_SHIFT      8u
#define CAN_PL2_PLMASK_LO_Data_byte_2_WIDTH      8u
#define CAN_PL2_PLMASK_LO_Data_byte_2(x)         (((uint32_t)(((uint32_t)(x))<<CAN_PL2_PLMASK_LO_Data_byte_2_SHIFT))&CAN_PL2_PLMASK_LO_Data_byte_2_MASK)
#define CAN_PL2_PLMASK_LO_Data_byte_1_MASK       0xFF0000u
#define CAN_PL2_PLMASK_LO_Data_byte_1_SHIFT      16u
#define CAN_PL2_PLMASK_LO_Data_byte_1_WIDTH      8u
#define CAN_PL2_PLMASK_LO_Data_byte_1(x)         (((uint32_t)(((uint32_t)(x))<<CAN_PL2_PLMASK_LO_Data_byte_1_SHIFT))&CAN_PL2_PLMASK_LO_Data_byte_1_MASK)
#define CAN_PL2_PLMASK_LO_Data_byte_0_MASK       0xFF000000u
#define CAN_PL2_PLMASK_LO_Data_byte_0_SHIFT      24u
#define CAN_PL2_PLMASK_LO_Data_byte_0_WIDTH      8u
#define CAN_PL2_PLMASK_LO_Data_byte_0(x)         (((uint32_t)(((uint32_t)(x))<<CAN_PL2_PLMASK_LO_Data_byte_0_SHIFT))&CAN_PL2_PLMASK_LO_Data_byte_0_MASK)
/* PL2_PLMASK_HI Bit Fields */
#define CAN_PL2_PLMASK_HI_Data_byte_7_MASK       0xFFu
#define CAN_PL2_PLMASK_HI_Data_byte_7_SHIFT      0u
#define CAN_PL2_PLMASK_HI_Data_byte_7_WIDTH      8u
#define CAN_PL2_PLMASK_HI_Data_byte_7(x)         (((uint32_t)(((uint32_t)(x))<<CAN_PL2_PLMASK_HI_Data_byte_7_SHIFT))&CAN_PL2_PLMASK_HI_Data_byte_7_MASK)
#define CAN_PL2_PLMASK_HI_Data_byte_6_MASK       0xFF00u
#define CAN_PL2_PLMASK_HI_Data_byte_6_SHIFT      8u
#define CAN_PL2_PLMASK_HI_Data_byte_6_WIDTH      8u
#define CAN_PL2_PLMASK_HI_Data_byte_6(x)         (((uint32_t)(((uint32_t)(x))<<CAN_PL2_PLMASK_HI_Data_byte_6_SHIFT))&CAN_PL2_PLMASK_HI_Data_byte_6_MASK)
#define CAN_PL2_PLMASK_HI_Data_byte_5_MASK       0xFF0000u
#define CAN_PL2_PLMASK_HI_Data_byte_5_SHIFT      16u
#define CAN_PL2_PLMASK_HI_Data_byte_5_WIDTH      8u
#define CAN_PL2_PLMASK_HI_Data_byte_5(x)         (((uint32_t)(((uint32_t)(x))<<CAN_PL2_PLMASK_HI_Data_byte_5_SHIFT))&CAN_PL2_PLMASK_HI_Data_byte_5_MASK)
#define CAN_PL2_PLMASK_HI_Data_byte_4_MASK       0xFF000000u
#define CAN_PL2_PLMASK_HI_Data_byte_4_SHIFT      24u
#define CAN_PL2_PLMASK_HI_Data_byte_4_WIDTH      8u
#define CAN_PL2_PLMASK_HI_Data_byte_4(x)         (((uint32_t)(((uint32_t)(x))<<CAN_PL2_PLMASK_HI_Data_byte_4_SHIFT))&CAN_PL2_PLMASK_HI_Data_byte_4_MASK)
/* WMBn_CS Bit Fields */
#define CAN_WMBn_CS_DLC_MASK                     0xF0000u
#define CAN_WMBn_CS_DLC_SHIFT                    16u
#define CAN_WMBn_CS_DLC_WIDTH                    4u
#define CAN_WMBn_CS_DLC(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_CS_DLC_SHIFT))&CAN_WMBn_CS_DLC_MASK)
#define CAN_WMBn_CS_RTR_MASK                     0x100000u
#define CAN_WMBn_CS_RTR_SHIFT                    20u
#define CAN_WMBn_CS_RTR_WIDTH                    1u
#define CAN_WMBn_CS_RTR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_CS_RTR_SHIFT))&CAN_WMBn_CS_RTR_MASK)
#define CAN_WMBn_CS_IDE_MASK                     0x200000u
#define CAN_WMBn_CS_IDE_SHIFT                    21u
#define CAN_WMBn_CS_IDE_WIDTH                    1u
#define CAN_WMBn_CS_IDE(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_CS_IDE_SHIFT))&CAN_WMBn_CS_IDE_MASK)
#define CAN_WMBn_CS_SRR_MASK                     0x400000u
#define CAN_WMBn_CS_SRR_SHIFT                    22u
#define CAN_WMBn_CS_SRR_WIDTH                    1u
#define CAN_WMBn_CS_SRR(x)                       (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_CS_SRR_SHIFT))&CAN_WMBn_CS_SRR_MASK)
/* WMBn_ID Bit Fields */
#define CAN_WMBn_ID_ID_MASK                      0x1FFFFFFFu
#define CAN_WMBn_ID_ID_SHIFT                     0u
#define CAN_WMBn_ID_ID_WIDTH                     29u
#define CAN_WMBn_ID_ID(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_ID_ID_SHIFT))&CAN_WMBn_ID_ID_MASK)
/* WMBn_D03 Bit Fields */
#define CAN_WMBn_D03_Data_byte_3_MASK            0xFFu
#define CAN_WMBn_D03_Data_byte_3_SHIFT           0u
#define CAN_WMBn_D03_Data_byte_3_WIDTH           8u
#define CAN_WMBn_D03_Data_byte_3(x)              (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_D03_Data_byte_3_SHIFT))&CAN_WMBn_D03_Data_byte_3_MASK)
#define CAN_WMBn_D03_Data_byte_2_MASK            0xFF00u
#define CAN_WMBn_D03_Data_byte_2_SHIFT           8u
#define CAN_WMBn_D03_Data_byte_2_WIDTH           8u
#define CAN_WMBn_D03_Data_byte_2(x)              (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_D03_Data_byte_2_SHIFT))&CAN_WMBn_D03_Data_byte_2_MASK)
#define CAN_WMBn_D03_Data_byte_1_MASK            0xFF0000u
#define CAN_WMBn_D03_Data_byte_1_SHIFT           16u
#define CAN_WMBn_D03_Data_byte_1_WIDTH           8u
#define CAN_WMBn_D03_Data_byte_1(x)              (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_D03_Data_byte_1_SHIFT))&CAN_WMBn_D03_Data_byte_1_MASK)
#define CAN_WMBn_D03_Data_byte_0_MASK            0xFF000000u
#define CAN_WMBn_D03_Data_byte_0_SHIFT           24u
#define CAN_WMBn_D03_Data_byte_0_WIDTH           8u
#define CAN_WMBn_D03_Data_byte_0(x)              (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_D03_Data_byte_0_SHIFT))&CAN_WMBn_D03_Data_byte_0_MASK)
/* WMBn_D47 Bit Fields */
#define CAN_WMBn_D47_Data_byte_7_MASK            0xFFu
#define CAN_WMBn_D47_Data_byte_7_SHIFT           0u
#define CAN_WMBn_D47_Data_byte_7_WIDTH           8u
#define CAN_WMBn_D47_Data_byte_7(x)              (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_D47_Data_byte_7_SHIFT))&CAN_WMBn_D47_Data_byte_7_MASK)
#define CAN_WMBn_D47_Data_byte_6_MASK            0xFF00u
#define CAN_WMBn_D47_Data_byte_6_SHIFT           8u
#define CAN_WMBn_D47_Data_byte_6_WIDTH           8u
#define CAN_WMBn_D47_Data_byte_6(x)              (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_D47_Data_byte_6_SHIFT))&CAN_WMBn_D47_Data_byte_6_MASK)
#define CAN_WMBn_D47_Data_byte_5_MASK            0xFF0000u
#define CAN_WMBn_D47_Data_byte_5_SHIFT           16u
#define CAN_WMBn_D47_Data_byte_5_WIDTH           8u
#define CAN_WMBn_D47_Data_byte_5(x)              (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_D47_Data_byte_5_SHIFT))&CAN_WMBn_D47_Data_byte_5_MASK)
#define CAN_WMBn_D47_Data_byte_4_MASK            0xFF000000u
#define CAN_WMBn_D47_Data_byte_4_SHIFT           24u
#define CAN_WMBn_D47_Data_byte_4_WIDTH           8u
#define CAN_WMBn_D47_Data_byte_4(x)              (((uint32_t)(((uint32_t)(x))<<CAN_WMBn_D47_Data_byte_4_SHIFT))&CAN_WMBn_D47_Data_byte_4_MASK)
/* FDCTRL Bit Fields */
#define CAN_FDCTRL_TDCVAL_MASK                   0x3Fu
#define CAN_FDCTRL_TDCVAL_SHIFT                  0u
#define CAN_FDCTRL_TDCVAL_WIDTH                  6u
#define CAN_FDCTRL_TDCVAL(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_TDCVAL_SHIFT))&CAN_FDCTRL_TDCVAL_MASK)
#define CAN_FDCTRL_TDCOFF_MASK                   0x1F00u
#define CAN_FDCTRL_TDCOFF_SHIFT                  8u
#define CAN_FDCTRL_TDCOFF_WIDTH                  5u
#define CAN_FDCTRL_TDCOFF(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_TDCOFF_SHIFT))&CAN_FDCTRL_TDCOFF_MASK)
#define CAN_FDCTRL_TDCFAIL_MASK                  0x4000u
#define CAN_FDCTRL_TDCFAIL_SHIFT                 14u
#define CAN_FDCTRL_TDCFAIL_WIDTH                 1u
#define CAN_FDCTRL_TDCFAIL(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_TDCFAIL_SHIFT))&CAN_FDCTRL_TDCFAIL_MASK)
#define CAN_FDCTRL_TDCEN_MASK                    0x8000u
#define CAN_FDCTRL_TDCEN_SHIFT                   15u
#define CAN_FDCTRL_TDCEN_WIDTH                   1u
#define CAN_FDCTRL_TDCEN(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_TDCEN_SHIFT))&CAN_FDCTRL_TDCEN_MASK)
#define CAN_FDCTRL_MBDSR0_MASK                   0x30000u
#define CAN_FDCTRL_MBDSR0_SHIFT                  16u
#define CAN_FDCTRL_MBDSR0_WIDTH                  2u
#define CAN_FDCTRL_MBDSR0(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_MBDSR0_SHIFT))&CAN_FDCTRL_MBDSR0_MASK)
#define CAN_FDCTRL_MBDSR1_MASK                   0x180000u
#define CAN_FDCTRL_MBDSR1_SHIFT                  19u
#define CAN_FDCTRL_MBDSR1_WIDTH                  2u
#define CAN_FDCTRL_MBDSR1(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_MBDSR1_SHIFT))&CAN_FDCTRL_MBDSR1_MASK)
#define CAN_FDCTRL_MBDSR2_MASK                   0xC00000u
#define CAN_FDCTRL_MBDSR2_SHIFT                  22u
#define CAN_FDCTRL_MBDSR2_WIDTH                  2u
#define CAN_FDCTRL_MBDSR2(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_MBDSR2_SHIFT))&CAN_FDCTRL_MBDSR2_MASK)
#define CAN_FDCTRL_FDRATE_MASK                   0x80000000u
#define CAN_FDCTRL_FDRATE_SHIFT                  31u
#define CAN_FDCTRL_FDRATE_WIDTH                  1u
#define CAN_FDCTRL_FDRATE(x)                     (((uint32_t)(((uint32_t)(x))<<CAN_FDCTRL_FDRATE_SHIFT))&CAN_FDCTRL_FDRATE_MASK)
/* FDCBT Bit Fields */
#define CAN_FDCBT_FPSEG2_MASK                    0x7u
#define CAN_FDCBT_FPSEG2_SHIFT                   0u
#define CAN_FDCBT_FPSEG2_WIDTH                   3u
#define CAN_FDCBT_FPSEG2(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FPSEG2_SHIFT))&CAN_FDCBT_FPSEG2_MASK)
#define CAN_FDCBT_FPSEG1_MASK                    0xE0u
#define CAN_FDCBT_FPSEG1_SHIFT                   5u
#define CAN_FDCBT_FPSEG1_WIDTH                   3u
#define CAN_FDCBT_FPSEG1(x)                      (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FPSEG1_SHIFT))&CAN_FDCBT_FPSEG1_MASK)
#define CAN_FDCBT_FPROPSEG_MASK                  0x7C00u
#define CAN_FDCBT_FPROPSEG_SHIFT                 10u
#define CAN_FDCBT_FPROPSEG_WIDTH                 5u
#define CAN_FDCBT_FPROPSEG(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FPROPSEG_SHIFT))&CAN_FDCBT_FPROPSEG_MASK)
#define CAN_FDCBT_FRJW_MASK                      0x70000u
#define CAN_FDCBT_FRJW_SHIFT                     16u
#define CAN_FDCBT_FRJW_WIDTH                     3u
#define CAN_FDCBT_FRJW(x)                        (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FRJW_SHIFT))&CAN_FDCBT_FRJW_MASK)
#define CAN_FDCBT_FPRESDIV_MASK                  0x3FF00000u
#define CAN_FDCBT_FPRESDIV_SHIFT                 20u
#define CAN_FDCBT_FPRESDIV_WIDTH                 10u
#define CAN_FDCBT_FPRESDIV(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCBT_FPRESDIV_SHIFT))&CAN_FDCBT_FPRESDIV_MASK)
/* FDCRC Bit Fields */
#define CAN_FDCRC_FD_TXCRC_MASK                  0x1FFFFFu
#define CAN_FDCRC_FD_TXCRC_SHIFT                 0u
#define CAN_FDCRC_FD_TXCRC_WIDTH                 21u
#define CAN_FDCRC_FD_TXCRC(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCRC_FD_TXCRC_SHIFT))&CAN_FDCRC_FD_TXCRC_MASK)
#define CAN_FDCRC_FD_MBCRC_MASK                  0x7F000000u
#define CAN_FDCRC_FD_MBCRC_SHIFT                 24u
#define CAN_FDCRC_FD_MBCRC_WIDTH                 7u
#define CAN_FDCRC_FD_MBCRC(x)                    (((uint32_t)(((uint32_t)(x))<<CAN_FDCRC_FD_MBCRC_SHIFT))&CAN_FDCRC_FD_MBCRC_MASK)

/*!
 * @}
 */ /* end of group CAN_Register_Masks */


/*!
 * @}
 */ /* end of group CAN_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- CMP Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CMP_Peripheral_Access_Layer CMP Peripheral Access Layer
 * @{
 */


/** CMP - Size of Registers Arrays */

/** CMP - Register Layout Typedef */
typedef struct {
  __IO uint32_t C0;                                /**< CMP Control Register 0, offset: 0x0 */
  __IO uint32_t C1;                                /**< CMP Control Register 1, offset: 0x4 */
  __IO uint32_t C2;                                /**< CMP Control Register 2, offset: 0x8 */
} CMP_Type, *CMP_MemMapPtr;

 /** Number of instances of the CMP module. */
#define CMP_INSTANCE_COUNT                       (3u)


/* CMP - Peripheral instance base addresses */
/** Peripheral CMP_0 base address */
#define CMP_0_BASE                               (0xFFCDC000u)
/** Peripheral CMP_0 base pointer */
#define CMP_0                                    ((CMP_Type *)CMP_0_BASE)
/** Peripheral CMP_1 base address */
#define CMP_1_BASE                               (0xFFCE0000u)
/** Peripheral CMP_1 base pointer */
#define CMP_1                                    ((CMP_Type *)CMP_1_BASE)
/** Peripheral CMP_2 base address */
#define CMP_2_BASE                               (0xFFCE4000u)
/** Peripheral CMP_2 base pointer */
#define CMP_2                                    ((CMP_Type *)CMP_2_BASE)
/** Array initializer of CMP peripheral base addresses */
#define CMP_BASE_ADDRS                           { CMP_0_BASE, CMP_1_BASE, CMP_2_BASE }
/** Array initializer of CMP peripheral base pointers */
#define CMP_BASE_PTRS                            { CMP_0, CMP_1, CMP_2 }

/* ----------------------------------------------------------------------------
   -- CMP Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CMP_Register_Masks CMP Register Masks
 * @{
 */

/* C0 Bit Fields */
#define CMP_C0_HYSTCTR_MASK                      0x3u
#define CMP_C0_HYSTCTR_SHIFT                     0u
#define CMP_C0_HYSTCTR_WIDTH                     2u
#define CMP_C0_HYSTCTR(x)                        (((uint32_t)(((uint32_t)(x))<<CMP_C0_HYSTCTR_SHIFT))&CMP_C0_HYSTCTR_MASK)
#define CMP_C0_FILTER_CNT_MASK                   0x70u
#define CMP_C0_FILTER_CNT_SHIFT                  4u
#define CMP_C0_FILTER_CNT_WIDTH                  3u
#define CMP_C0_FILTER_CNT(x)                     (((uint32_t)(((uint32_t)(x))<<CMP_C0_FILTER_CNT_SHIFT))&CMP_C0_FILTER_CNT_MASK)
#define CMP_C0_EN_MASK                           0x100u
#define CMP_C0_EN_SHIFT                          8u
#define CMP_C0_EN_WIDTH                          1u
#define CMP_C0_EN(x)                             (((uint32_t)(((uint32_t)(x))<<CMP_C0_EN_SHIFT))&CMP_C0_EN_MASK)
#define CMP_C0_OPE_MASK                          0x200u
#define CMP_C0_OPE_SHIFT                         9u
#define CMP_C0_OPE_WIDTH                         1u
#define CMP_C0_OPE(x)                            (((uint32_t)(((uint32_t)(x))<<CMP_C0_OPE_SHIFT))&CMP_C0_OPE_MASK)
#define CMP_C0_COS_MASK                          0x400u
#define CMP_C0_COS_SHIFT                         10u
#define CMP_C0_COS_WIDTH                         1u
#define CMP_C0_COS(x)                            (((uint32_t)(((uint32_t)(x))<<CMP_C0_COS_SHIFT))&CMP_C0_COS_MASK)
#define CMP_C0_INVT_MASK                         0x800u
#define CMP_C0_INVT_SHIFT                        11u
#define CMP_C0_INVT_WIDTH                        1u
#define CMP_C0_INVT(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C0_INVT_SHIFT))&CMP_C0_INVT_MASK)
#define CMP_C0_PMODE_MASK                        0x1000u
#define CMP_C0_PMODE_SHIFT                       12u
#define CMP_C0_PMODE_WIDTH                       1u
#define CMP_C0_PMODE(x)                          (((uint32_t)(((uint32_t)(x))<<CMP_C0_PMODE_SHIFT))&CMP_C0_PMODE_MASK)
#define CMP_C0_WE_MASK                           0x4000u
#define CMP_C0_WE_SHIFT                          14u
#define CMP_C0_WE_WIDTH                          1u
#define CMP_C0_WE(x)                             (((uint32_t)(((uint32_t)(x))<<CMP_C0_WE_SHIFT))&CMP_C0_WE_MASK)
#define CMP_C0_SE_MASK                           0x8000u
#define CMP_C0_SE_SHIFT                          15u
#define CMP_C0_SE_WIDTH                          1u
#define CMP_C0_SE(x)                             (((uint32_t)(((uint32_t)(x))<<CMP_C0_SE_SHIFT))&CMP_C0_SE_MASK)
#define CMP_C0_FPR_MASK                          0xFF0000u
#define CMP_C0_FPR_SHIFT                         16u
#define CMP_C0_FPR_WIDTH                         8u
#define CMP_C0_FPR(x)                            (((uint32_t)(((uint32_t)(x))<<CMP_C0_FPR_SHIFT))&CMP_C0_FPR_MASK)
#define CMP_C0_COUT_MASK                         0x1000000u
#define CMP_C0_COUT_SHIFT                        24u
#define CMP_C0_COUT_WIDTH                        1u
#define CMP_C0_COUT(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C0_COUT_SHIFT))&CMP_C0_COUT_MASK)
#define CMP_C0_CFF_MASK                          0x2000000u
#define CMP_C0_CFF_SHIFT                         25u
#define CMP_C0_CFF_WIDTH                         1u
#define CMP_C0_CFF(x)                            (((uint32_t)(((uint32_t)(x))<<CMP_C0_CFF_SHIFT))&CMP_C0_CFF_MASK)
#define CMP_C0_CFR_MASK                          0x4000000u
#define CMP_C0_CFR_SHIFT                         26u
#define CMP_C0_CFR_WIDTH                         1u
#define CMP_C0_CFR(x)                            (((uint32_t)(((uint32_t)(x))<<CMP_C0_CFR_SHIFT))&CMP_C0_CFR_MASK)
#define CMP_C0_IEF_MASK                          0x8000000u
#define CMP_C0_IEF_SHIFT                         27u
#define CMP_C0_IEF_WIDTH                         1u
#define CMP_C0_IEF(x)                            (((uint32_t)(((uint32_t)(x))<<CMP_C0_IEF_SHIFT))&CMP_C0_IEF_MASK)
#define CMP_C0_IER_MASK                          0x10000000u
#define CMP_C0_IER_SHIFT                         28u
#define CMP_C0_IER_WIDTH                         1u
#define CMP_C0_IER(x)                            (((uint32_t)(((uint32_t)(x))<<CMP_C0_IER_SHIFT))&CMP_C0_IER_MASK)
#define CMP_C0_DMAEN_MASK                        0x40000000u
#define CMP_C0_DMAEN_SHIFT                       30u
#define CMP_C0_DMAEN_WIDTH                       1u
#define CMP_C0_DMAEN(x)                          (((uint32_t)(((uint32_t)(x))<<CMP_C0_DMAEN_SHIFT))&CMP_C0_DMAEN_MASK)
/* C1 Bit Fields */
#define CMP_C1_VOSEL_MASK                        0x3Fu
#define CMP_C1_VOSEL_SHIFT                       0u
#define CMP_C1_VOSEL_WIDTH                       6u
#define CMP_C1_VOSEL(x)                          (((uint32_t)(((uint32_t)(x))<<CMP_C1_VOSEL_SHIFT))&CMP_C1_VOSEL_MASK)
#define CMP_C1_VRSEL_MASK                        0x40u
#define CMP_C1_VRSEL_SHIFT                       6u
#define CMP_C1_VRSEL_WIDTH                       1u
#define CMP_C1_VRSEL(x)                          (((uint32_t)(((uint32_t)(x))<<CMP_C1_VRSEL_SHIFT))&CMP_C1_VRSEL_MASK)
#define CMP_C1_DACEN_MASK                        0x80u
#define CMP_C1_DACEN_SHIFT                       7u
#define CMP_C1_DACEN_WIDTH                       1u
#define CMP_C1_DACEN(x)                          (((uint32_t)(((uint32_t)(x))<<CMP_C1_DACEN_SHIFT))&CMP_C1_DACEN_MASK)
#define CMP_C1_MSEL_MASK                         0x700u
#define CMP_C1_MSEL_SHIFT                        8u
#define CMP_C1_MSEL_WIDTH                        3u
#define CMP_C1_MSEL(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_MSEL_SHIFT))&CMP_C1_MSEL_MASK)
#define CMP_C1_PSEL_MASK                         0x3800u
#define CMP_C1_PSEL_SHIFT                        11u
#define CMP_C1_PSEL_WIDTH                        3u
#define CMP_C1_PSEL(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_PSEL_SHIFT))&CMP_C1_PSEL_MASK)
#define CMP_C1_DMSEL_MASK                        0x4000u
#define CMP_C1_DMSEL_SHIFT                       14u
#define CMP_C1_DMSEL_WIDTH                       1u
#define CMP_C1_DMSEL(x)                          (((uint32_t)(((uint32_t)(x))<<CMP_C1_DMSEL_SHIFT))&CMP_C1_DMSEL_MASK)
#define CMP_C1_DPSEL_MASK                        0x8000u
#define CMP_C1_DPSEL_SHIFT                       15u
#define CMP_C1_DPSEL_WIDTH                       1u
#define CMP_C1_DPSEL(x)                          (((uint32_t)(((uint32_t)(x))<<CMP_C1_DPSEL_SHIFT))&CMP_C1_DPSEL_MASK)
#define CMP_C1_CHN0_MASK                         0x10000u
#define CMP_C1_CHN0_SHIFT                        16u
#define CMP_C1_CHN0_WIDTH                        1u
#define CMP_C1_CHN0(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_CHN0_SHIFT))&CMP_C1_CHN0_MASK)
#define CMP_C1_CHN1_MASK                         0x20000u
#define CMP_C1_CHN1_SHIFT                        17u
#define CMP_C1_CHN1_WIDTH                        1u
#define CMP_C1_CHN1(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_CHN1_SHIFT))&CMP_C1_CHN1_MASK)
#define CMP_C1_CHN2_MASK                         0x40000u
#define CMP_C1_CHN2_SHIFT                        18u
#define CMP_C1_CHN2_WIDTH                        1u
#define CMP_C1_CHN2(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_CHN2_SHIFT))&CMP_C1_CHN2_MASK)
#define CMP_C1_CHN3_MASK                         0x80000u
#define CMP_C1_CHN3_SHIFT                        19u
#define CMP_C1_CHN3_WIDTH                        1u
#define CMP_C1_CHN3(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_CHN3_SHIFT))&CMP_C1_CHN3_MASK)
#define CMP_C1_CHN4_MASK                         0x100000u
#define CMP_C1_CHN4_SHIFT                        20u
#define CMP_C1_CHN4_WIDTH                        1u
#define CMP_C1_CHN4(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_CHN4_SHIFT))&CMP_C1_CHN4_MASK)
#define CMP_C1_CHN5_MASK                         0x200000u
#define CMP_C1_CHN5_SHIFT                        21u
#define CMP_C1_CHN5_WIDTH                        1u
#define CMP_C1_CHN5(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_CHN5_SHIFT))&CMP_C1_CHN5_MASK)
#define CMP_C1_CHN6_MASK                         0x400000u
#define CMP_C1_CHN6_SHIFT                        22u
#define CMP_C1_CHN6_WIDTH                        1u
#define CMP_C1_CHN6(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_CHN6_SHIFT))&CMP_C1_CHN6_MASK)
#define CMP_C1_CHN7_MASK                         0x800000u
#define CMP_C1_CHN7_SHIFT                        23u
#define CMP_C1_CHN7_WIDTH                        1u
#define CMP_C1_CHN7(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C1_CHN7_SHIFT))&CMP_C1_CHN7_MASK)
/* C2 Bit Fields */
#define CMP_C2_ACOn_MASK                         0xFFu
#define CMP_C2_ACOn_SHIFT                        0u
#define CMP_C2_ACOn_WIDTH                        8u
#define CMP_C2_ACOn(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_ACOn_SHIFT))&CMP_C2_ACOn_MASK)
#define CMP_C2_NSAM_MASK                         0xC000u
#define CMP_C2_NSAM_SHIFT                        14u
#define CMP_C2_NSAM_WIDTH                        2u
#define CMP_C2_NSAM(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_NSAM_SHIFT))&CMP_C2_NSAM_MASK)
#define CMP_C2_CH0F_MASK                         0x10000u
#define CMP_C2_CH0F_SHIFT                        16u
#define CMP_C2_CH0F_WIDTH                        1u
#define CMP_C2_CH0F(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_CH0F_SHIFT))&CMP_C2_CH0F_MASK)
#define CMP_C2_CH1F_MASK                         0x20000u
#define CMP_C2_CH1F_SHIFT                        17u
#define CMP_C2_CH1F_WIDTH                        1u
#define CMP_C2_CH1F(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_CH1F_SHIFT))&CMP_C2_CH1F_MASK)
#define CMP_C2_CH2F_MASK                         0x40000u
#define CMP_C2_CH2F_SHIFT                        18u
#define CMP_C2_CH2F_WIDTH                        1u
#define CMP_C2_CH2F(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_CH2F_SHIFT))&CMP_C2_CH2F_MASK)
#define CMP_C2_CH3F_MASK                         0x80000u
#define CMP_C2_CH3F_SHIFT                        19u
#define CMP_C2_CH3F_WIDTH                        1u
#define CMP_C2_CH3F(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_CH3F_SHIFT))&CMP_C2_CH3F_MASK)
#define CMP_C2_CH4F_MASK                         0x100000u
#define CMP_C2_CH4F_SHIFT                        20u
#define CMP_C2_CH4F_WIDTH                        1u
#define CMP_C2_CH4F(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_CH4F_SHIFT))&CMP_C2_CH4F_MASK)
#define CMP_C2_CH5F_MASK                         0x200000u
#define CMP_C2_CH5F_SHIFT                        21u
#define CMP_C2_CH5F_WIDTH                        1u
#define CMP_C2_CH5F(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_CH5F_SHIFT))&CMP_C2_CH5F_MASK)
#define CMP_C2_CH6F_MASK                         0x400000u
#define CMP_C2_CH6F_SHIFT                        22u
#define CMP_C2_CH6F_WIDTH                        1u
#define CMP_C2_CH6F(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_CH6F_SHIFT))&CMP_C2_CH6F_MASK)
#define CMP_C2_CH7F_MASK                         0x800000u
#define CMP_C2_CH7F_SHIFT                        23u
#define CMP_C2_CH7F_WIDTH                        1u
#define CMP_C2_CH7F(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_CH7F_SHIFT))&CMP_C2_CH7F_MASK)
#define CMP_C2_FXMXCH_MASK                       0xE000000u
#define CMP_C2_FXMXCH_SHIFT                      25u
#define CMP_C2_FXMXCH_WIDTH                      3u
#define CMP_C2_FXMXCH(x)                         (((uint32_t)(((uint32_t)(x))<<CMP_C2_FXMXCH_SHIFT))&CMP_C2_FXMXCH_MASK)
#define CMP_C2_FXDACI_MASK                       0x10000000u
#define CMP_C2_FXDACI_SHIFT                      28u
#define CMP_C2_FXDACI_WIDTH                      1u
#define CMP_C2_FXDACI(x)                         (((uint32_t)(((uint32_t)(x))<<CMP_C2_FXDACI_SHIFT))&CMP_C2_FXDACI_MASK)
#define CMP_C2_FXMP_MASK                         0x20000000u
#define CMP_C2_FXMP_SHIFT                        29u
#define CMP_C2_FXMP_WIDTH                        1u
#define CMP_C2_FXMP(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_FXMP_SHIFT))&CMP_C2_FXMP_MASK)
#define CMP_C2_RRIE_MASK                         0x40000000u
#define CMP_C2_RRIE_SHIFT                        30u
#define CMP_C2_RRIE_WIDTH                        1u
#define CMP_C2_RRIE(x)                           (((uint32_t)(((uint32_t)(x))<<CMP_C2_RRIE_SHIFT))&CMP_C2_RRIE_MASK)
#define CMP_C2_RRE_MASK                          0x80000000u
#define CMP_C2_RRE_SHIFT                         31u
#define CMP_C2_RRE_WIDTH                         1u
#define CMP_C2_RRE(x)                            (((uint32_t)(((uint32_t)(x))<<CMP_C2_RRE_SHIFT))&CMP_C2_RRE_MASK)

/*!
 * @}
 */ /* end of group CMP_Register_Masks */


/*!
 * @}
 */ /* end of group CMP_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- CMU Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CMU_Peripheral_Access_Layer CMU Peripheral Access Layer
 * @{
 */


/** CMU - Size of Registers Arrays */

/** CMU - Register Layout Typedef */
typedef struct {
  __IO uint32_t CSR;                               /**< CMU Control Status Register, offset: 0x0 */
  __I  uint32_t FDR;                               /**< CMU Frequency Display Register, offset: 0x4 */
  __IO uint32_t HFREFR;                            /**< CMU High Frequency Reference Register CLKMN1, offset: 0x8 */
  __IO uint32_t LFREFR;                            /**< CMU Low Frequency Reference Register CLKMN1, offset: 0xC */
  __IO uint32_t ISR;                               /**< CMU Interrupt Status Register, offset: 0x10 */
       uint8_t RESERVED_0[4];
  __IO uint32_t MDR;                               /**< CMU Measurement Duration Register, offset: 0x18 */
} CMU_Type, *CMU_MemMapPtr;

 /** Number of instances of the CMU module. */
#define CMU_INSTANCE_COUNT                       (1u)


/* CMU - Peripheral instance base addresses */
/** Peripheral CMU base address */
#define CMU_BASE                                 (0xFFFB0140u)
/** Peripheral CMU base pointer */
#define CMU                                      ((CMU_Type *)CMU_BASE)
/** Array initializer of CMU peripheral base addresses */
#define CMU_BASE_ADDRS                           { CMU_BASE }
/** Array initializer of CMU peripheral base pointers */
#define CMU_BASE_PTRS                            { CMU }

/* ----------------------------------------------------------------------------
   -- CMU Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CMU_Register_Masks CMU Register Masks
 * @{
 */

/* CSR Bit Fields */
#define CMU_CSR_CME_MASK                         0x1u
#define CMU_CSR_CME_SHIFT                        0u
#define CMU_CSR_CME_WIDTH                        1u
#define CMU_CSR_CME(x)                           (((uint32_t)(((uint32_t)(x))<<CMU_CSR_CME_SHIFT))&CMU_CSR_CME_MASK)
#define CMU_CSR_RCDIV_MASK                       0x6u
#define CMU_CSR_RCDIV_SHIFT                      1u
#define CMU_CSR_RCDIV_WIDTH                      2u
#define CMU_CSR_RCDIV(x)                         (((uint32_t)(((uint32_t)(x))<<CMU_CSR_RCDIV_SHIFT))&CMU_CSR_RCDIV_MASK)
#define CMU_CSR_CKSEL1_MASK                      0x300u
#define CMU_CSR_CKSEL1_SHIFT                     8u
#define CMU_CSR_CKSEL1_WIDTH                     2u
#define CMU_CSR_CKSEL1(x)                        (((uint32_t)(((uint32_t)(x))<<CMU_CSR_CKSEL1_SHIFT))&CMU_CSR_CKSEL1_MASK)
#define CMU_CSR_SFM_MASK                         0x800000u
#define CMU_CSR_SFM_SHIFT                        23u
#define CMU_CSR_SFM_WIDTH                        1u
#define CMU_CSR_SFM(x)                           (((uint32_t)(((uint32_t)(x))<<CMU_CSR_SFM_SHIFT))&CMU_CSR_SFM_MASK)
/* FDR Bit Fields */
#define CMU_FDR_FD_MASK                          0xFFFFFu
#define CMU_FDR_FD_SHIFT                         0u
#define CMU_FDR_FD_WIDTH                         20u
#define CMU_FDR_FD(x)                            (((uint32_t)(((uint32_t)(x))<<CMU_FDR_FD_SHIFT))&CMU_FDR_FD_MASK)
/* HFREFR Bit Fields */
#define CMU_HFREFR_HFREF_MASK                    0xFFFu
#define CMU_HFREFR_HFREF_SHIFT                   0u
#define CMU_HFREFR_HFREF_WIDTH                   12u
#define CMU_HFREFR_HFREF(x)                      (((uint32_t)(((uint32_t)(x))<<CMU_HFREFR_HFREF_SHIFT))&CMU_HFREFR_HFREF_MASK)
/* LFREFR Bit Fields */
#define CMU_LFREFR_LFREF_MASK                    0xFFFu
#define CMU_LFREFR_LFREF_SHIFT                   0u
#define CMU_LFREFR_LFREF_WIDTH                   12u
#define CMU_LFREFR_LFREF(x)                      (((uint32_t)(((uint32_t)(x))<<CMU_LFREFR_LFREF_SHIFT))&CMU_LFREFR_LFREF_MASK)
/* ISR Bit Fields */
#define CMU_ISR_OLRI_MASK                        0x1u
#define CMU_ISR_OLRI_SHIFT                       0u
#define CMU_ISR_OLRI_WIDTH                       1u
#define CMU_ISR_OLRI(x)                          (((uint32_t)(((uint32_t)(x))<<CMU_ISR_OLRI_SHIFT))&CMU_ISR_OLRI_MASK)
#define CMU_ISR_FLLI_MASK                        0x2u
#define CMU_ISR_FLLI_SHIFT                       1u
#define CMU_ISR_FLLI_WIDTH                       1u
#define CMU_ISR_FLLI(x)                          (((uint32_t)(((uint32_t)(x))<<CMU_ISR_FLLI_SHIFT))&CMU_ISR_FLLI_MASK)
#define CMU_ISR_FHHI_MASK                        0x4u
#define CMU_ISR_FHHI_SHIFT                       2u
#define CMU_ISR_FHHI_WIDTH                       1u
#define CMU_ISR_FHHI(x)                          (((uint32_t)(((uint32_t)(x))<<CMU_ISR_FHHI_SHIFT))&CMU_ISR_FHHI_MASK)
/* MDR Bit Fields */
#define CMU_MDR_MD_MASK                          0xFFFFFu
#define CMU_MDR_MD_SHIFT                         0u
#define CMU_MDR_MD_WIDTH                         20u
#define CMU_MDR_MD(x)                            (((uint32_t)(((uint32_t)(x))<<CMU_MDR_MD_SHIFT))&CMU_MDR_MD_MASK)

/*!
 * @}
 */ /* end of group CMU_Register_Masks */


/*!
 * @}
 */ /* end of group CMU_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- CRC Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CRC_Peripheral_Access_Layer CRC Peripheral Access Layer
 * @{
 */


/** CRC - Size of Registers Arrays */

/** CRC - Register Layout Typedef */
typedef struct {
  __IO uint32_t CFG;                               /**< Configuration Register, offset: 0x0 */
  __IO uint32_t INP;                               /**< Input Register, offset: 0x4 */
  __IO uint32_t CSTAT;                             /**< Current Status Register, offset: 0x8 */
  __I  uint32_t OUTP;                              /**< Output Register, offset: 0xC */
} CRC_Type, *CRC_MemMapPtr;

 /** Number of instances of the CRC module. */
#define CRC_INSTANCE_COUNT                       (1u)


/* CRC - Peripheral instance base addresses */
/** Peripheral CRC base address */
#define CRC_BASE                                 (0xFFF64000u)
/** Peripheral CRC base pointer */
#define CRC                                      ((CRC_Type *)CRC_BASE)
/** Array initializer of CRC peripheral base addresses */
#define CRC_BASE_ADDRS                           { CRC_BASE }
/** Array initializer of CRC peripheral base pointers */
#define CRC_BASE_PTRS                            { CRC }

/* ----------------------------------------------------------------------------
   -- CRC Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup CRC_Register_Masks CRC Register Masks
 * @{
 */

/* CFG Bit Fields */
#define CRC_CFG_INV_MASK                         0x1u
#define CRC_CFG_INV_SHIFT                        0u
#define CRC_CFG_INV_WIDTH                        1u
#define CRC_CFG_INV(x)                           (((uint32_t)(((uint32_t)(x))<<CRC_CFG_INV_SHIFT))&CRC_CFG_INV_MASK)
#define CRC_CFG_SWAP_MASK                        0x2u
#define CRC_CFG_SWAP_SHIFT                       1u
#define CRC_CFG_SWAP_WIDTH                       1u
#define CRC_CFG_SWAP(x)                          (((uint32_t)(((uint32_t)(x))<<CRC_CFG_SWAP_SHIFT))&CRC_CFG_SWAP_MASK)
#define CRC_CFG_POLYG_MASK                       0xCu
#define CRC_CFG_POLYG_SHIFT                      2u
#define CRC_CFG_POLYG_WIDTH                      2u
#define CRC_CFG_POLYG(x)                         (((uint32_t)(((uint32_t)(x))<<CRC_CFG_POLYG_SHIFT))&CRC_CFG_POLYG_MASK)
#define CRC_CFG_SWAP_BITWISE_MASK                0x10u
#define CRC_CFG_SWAP_BITWISE_SHIFT               4u
#define CRC_CFG_SWAP_BITWISE_WIDTH               1u
#define CRC_CFG_SWAP_BITWISE(x)                  (((uint32_t)(((uint32_t)(x))<<CRC_CFG_SWAP_BITWISE_SHIFT))&CRC_CFG_SWAP_BITWISE_MASK)
#define CRC_CFG_SWAP_BYTEWISE_MASK               0x20u
#define CRC_CFG_SWAP_BYTEWISE_SHIFT              5u
#define CRC_CFG_SWAP_BYTEWISE_WIDTH              1u
#define CRC_CFG_SWAP_BYTEWISE(x)                 (((uint32_t)(((uint32_t)(x))<<CRC_CFG_SWAP_BYTEWISE_SHIFT))&CRC_CFG_SWAP_BYTEWISE_MASK)
/* INP Bit Fields */
#define CRC_INP_INP_MASK                         0xFFFFFFFFu
#define CRC_INP_INP_SHIFT                        0u
#define CRC_INP_INP_WIDTH                        32u
#define CRC_INP_INP(x)                           (((uint32_t)(((uint32_t)(x))<<CRC_INP_INP_SHIFT))&CRC_INP_INP_MASK)
/* CSTAT Bit Fields */
#define CRC_CSTAT_CSTAT_MASK                     0xFFFFFFFFu
#define CRC_CSTAT_CSTAT_SHIFT                    0u
#define CRC_CSTAT_CSTAT_WIDTH                    32u
#define CRC_CSTAT_CSTAT(x)                       (((uint32_t)(((uint32_t)(x))<<CRC_CSTAT_CSTAT_SHIFT))&CRC_CSTAT_CSTAT_MASK)
/* OUTP Bit Fields */
#define CRC_OUTP_OUTP_MASK                       0xFFFFFFFFu
#define CRC_OUTP_OUTP_SHIFT                      0u
#define CRC_OUTP_OUTP_WIDTH                      32u
#define CRC_OUTP_OUTP(x)                         (((uint32_t)(((uint32_t)(x))<<CRC_OUTP_OUTP_SHIFT))&CRC_OUTP_OUTP_MASK)

/*!
 * @}
 */ /* end of group CRC_Register_Masks */


/*!
 * @}
 */ /* end of group CRC_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- DMA Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DMA_Peripheral_Access_Layer DMA Peripheral Access Layer
 * @{
 */


/** DMA - Size of Registers Arrays */
#define DMA_DCHPRI_COUNT                         32u
#define DMA_DCHMID_COUNT                         32u
#define DMA_TCD_COUNT                            32u

/** DMA - Register Layout Typedef */
typedef struct {
  __IO uint32_t CR;                                /**< Control Register, offset: 0x0 */
  __I  uint32_t ES;                                /**< Error Status Register, offset: 0x4 */
       uint8_t RESERVED_0[4];
  __IO uint32_t ERQ;                               /**< Enable Request Register, offset: 0xC */
       uint8_t RESERVED_1[4];
  __IO uint32_t EEI;                               /**< Enable Error Interrupt Register, offset: 0x14 */
  __O  uint8_t SERQ;                               /**< Set Enable Request Register, offset: 0x18 */
  __O  uint8_t CERQ;                               /**< Clear Enable Request Register, offset: 0x19 */
  __O  uint8_t SEEI;                               /**< Set Enable Error Interrupt Register, offset: 0x1A */
  __O  uint8_t CEEI;                               /**< Clear Enable Error Interrupt Register, offset: 0x1B */
  __O  uint8_t CINT;                               /**< Clear Interrupt Request Register, offset: 0x1C */
  __O  uint8_t CERR;                               /**< Clear Error Register, offset: 0x1D */
  __O  uint8_t SSRT;                               /**< Set START Bit Register, offset: 0x1E */
  __O  uint8_t CDNE;                               /**< Clear DONE Status Bit Register, offset: 0x1F */
       uint8_t RESERVED_2[4];
  __IO uint32_t INT;                               /**< Interrupt Request Register, offset: 0x24 */
       uint8_t RESERVED_3[4];
  __IO uint32_t ERR;                               /**< Error Register, offset: 0x2C */
       uint8_t RESERVED_4[4];
  __I  uint32_t HRS;                               /**< Hardware Request Status Register, offset: 0x34 */
       uint8_t RESERVED_5[200];
  __IO uint8_t DCHPRI[DMA_DCHPRI_COUNT];           /**< Channel n Priority Register, array offset: 0x100, array step: 0x1 */
       uint8_t RESERVED_6[32];
  __IO uint8_t DCHMID[DMA_DCHMID_COUNT];           /**< Channel n Master ID Register, array offset: 0x140, array step: 0x1 */
       uint8_t RESERVED_7[3744];
  struct {                                         /* offset: 0x1000, array step: 0x20 */
    __IO uint32_t SADDR;                             /**< TCD Source Address, array offset: 0x1000, array step: 0x20 */
    __IO uint16_t ATTR;                              /**< TCD Transfer Attributes, array offset: 0x1004, array step: 0x20 */
    __IO uint16_t SOFF;                              /**< TCD Signed Source Address Offset, array offset: 0x1006, array step: 0x20 */
    union {                                          /* offset: 0x1008, array step: 0x20 */
      __IO uint32_t MLNO;                              /**< TCD Minor Byte Count (Minor Loop Mapping Disabled), array offset: 0x1008, array step: 0x20 */
      __IO uint32_t MLOFFNO;                           /**< TCD Signed Minor Loop Offset (Minor Loop Mapping Enabled and Offset Disabled), array offset: 0x1008, array step: 0x20 */
      __IO uint32_t MLOFFYES;                          /**< TCD Signed Minor Loop Offset (Minor Loop Mapping and Offset Enabled), array offset: 0x1008, array step: 0x20 */
    } NBYTES;
    __IO uint32_t SLAST;                             /**< TCD Last Source Address Adjustment, array offset: 0x100C, array step: 0x20 */
    __IO uint32_t DADDR;                             /**< TCD Destination Address, array offset: 0x1010, array step: 0x20 */
    union {                                          /* offset: 0x1014, array step: 0x20 */
      __IO uint16_t ELINKNO;                           /**< TCD Current Minor Loop Link, Major Loop Count (Channel Linking Disabled), array offset: 0x1014, array step: 0x20 */
      __IO uint16_t ELINKYES;                          /**< TCD Current Minor Loop Link, Major Loop Count (Channel Linking Enabled), array offset: 0x1014, array step: 0x20 */
    } CITER;
    __IO uint16_t DOFF;                              /**< TCD Signed Destination Address Offset, array offset: 0x1016, array step: 0x20 */
    __IO uint32_t DLASTSGA;                          /**< TCD Last Destination Address Adjustment/Scatter Gather Address, array offset: 0x1018, array step: 0x20 */
    union {                                          /* offset: 0x101C, array step: 0x20 */
      __IO uint16_t ELINKNO;                           /**< TCD Beginning Minor Loop Link, Major Loop Count (Channel Linking Disabled), array offset: 0x101C, array step: 0x20 */
      __IO uint16_t ELINKYES;                          /**< TCD Beginning Minor Loop Link, Major Loop Count (Channel Linking Enabled), array offset: 0x101C, array step: 0x20 */
    } BITER;
    __IO uint16_t CSR;                               /**< TCD Control and Status, array offset: 0x101E, array step: 0x20 */
  } TCD[DMA_TCD_COUNT];
} DMA_Type, *DMA_MemMapPtr;

 /** Number of instances of the DMA module. */
#define DMA_INSTANCE_COUNT                       (1u)


/* DMA - Peripheral instance base addresses */
/** Peripheral DMA base address */
#define DMA_BASE                                 (0xFC0A0000u)
/** Peripheral DMA base pointer */
#define DMA                                      ((DMA_Type *)DMA_BASE)
/** Array initializer of DMA peripheral base addresses */
#define DMA_BASE_ADDRS                           { DMA_BASE }
/** Array initializer of DMA peripheral base pointers */
#define DMA_BASE_PTRS                            { DMA }
 /** Number of interrupt vector arrays for the DMA module. */
#define DMA_IRQS_ARR_COUNT                       (2u)
 /** Number of interrupt channels for the CHN type of DMA module. */
#define DMA_CHN_IRQS_CH_COUNT                    (32u)
 /** Number of interrupt channels for the ERROR type of DMA module. */
#define DMA_ERROR_IRQS_CH_COUNT                  (1u)
/** Interrupt vectors for the DMA peripheral type */
#define DMA_CHN_IRQS                             { DMA0_IRQn, DMA1_IRQn, DMA2_IRQn, DMA3_IRQn, DMA4_IRQn, DMA5_IRQn, DMA6_IRQn, DMA7_IRQn, DMA8_IRQn, DMA9_IRQn, DMA10_IRQn, DMA11_IRQn, DMA12_IRQn, DMA13_IRQn, DMA14_IRQn, DMA15_IRQn, DMA16_IRQn, DMA17_IRQn, DMA18_IRQn, DMA19_IRQn, DMA20_IRQn, DMA21_IRQn, DMA22_IRQn, DMA23_IRQn, DMA24_IRQn, DMA25_IRQn, DMA26_IRQn, DMA27_IRQn, DMA28_IRQn, DMA29_IRQn, DMA30_IRQn, DMA31_IRQn }
#define DMA_ERROR_IRQS                           { DMA_Error_IRQn }

/* ----------------------------------------------------------------------------
   -- DMA Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DMA_Register_Masks DMA Register Masks
 * @{
 */

/* CR Bit Fields */
#define DMA_CR_EDBG_MASK                         0x2u
#define DMA_CR_EDBG_SHIFT                        1u
#define DMA_CR_EDBG_WIDTH                        1u
#define DMA_CR_EDBG(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_EDBG_SHIFT))&DMA_CR_EDBG_MASK)
#define DMA_CR_ERCA_MASK                         0x4u
#define DMA_CR_ERCA_SHIFT                        2u
#define DMA_CR_ERCA_WIDTH                        1u
#define DMA_CR_ERCA(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_ERCA_SHIFT))&DMA_CR_ERCA_MASK)
#define DMA_CR_ERGA_MASK                         0x8u
#define DMA_CR_ERGA_SHIFT                        3u
#define DMA_CR_ERGA_WIDTH                        1u
#define DMA_CR_ERGA(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_ERGA_SHIFT))&DMA_CR_ERGA_MASK)
#define DMA_CR_HOE_MASK                          0x10u
#define DMA_CR_HOE_SHIFT                         4u
#define DMA_CR_HOE_WIDTH                         1u
#define DMA_CR_HOE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_CR_HOE_SHIFT))&DMA_CR_HOE_MASK)
#define DMA_CR_HALT_MASK                         0x20u
#define DMA_CR_HALT_SHIFT                        5u
#define DMA_CR_HALT_WIDTH                        1u
#define DMA_CR_HALT(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_HALT_SHIFT))&DMA_CR_HALT_MASK)
#define DMA_CR_CLM_MASK                          0x40u
#define DMA_CR_CLM_SHIFT                         6u
#define DMA_CR_CLM_WIDTH                         1u
#define DMA_CR_CLM(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_CR_CLM_SHIFT))&DMA_CR_CLM_MASK)
#define DMA_CR_EMLM_MASK                         0x80u
#define DMA_CR_EMLM_SHIFT                        7u
#define DMA_CR_EMLM_WIDTH                        1u
#define DMA_CR_EMLM(x)                           (((uint32_t)(((uint32_t)(x))<<DMA_CR_EMLM_SHIFT))&DMA_CR_EMLM_MASK)
#define DMA_CR_GRP0PRI_MASK                      0x100u
#define DMA_CR_GRP0PRI_SHIFT                     8u
#define DMA_CR_GRP0PRI_WIDTH                     1u
#define DMA_CR_GRP0PRI(x)                        (((uint32_t)(((uint32_t)(x))<<DMA_CR_GRP0PRI_SHIFT))&DMA_CR_GRP0PRI_MASK)
#define DMA_CR_GRP1PRI_MASK                      0x400u
#define DMA_CR_GRP1PRI_SHIFT                     10u
#define DMA_CR_GRP1PRI_WIDTH                     1u
#define DMA_CR_GRP1PRI(x)                        (((uint32_t)(((uint32_t)(x))<<DMA_CR_GRP1PRI_SHIFT))&DMA_CR_GRP1PRI_MASK)
#define DMA_CR_ECX_MASK                          0x10000u
#define DMA_CR_ECX_SHIFT                         16u
#define DMA_CR_ECX_WIDTH                         1u
#define DMA_CR_ECX(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_CR_ECX_SHIFT))&DMA_CR_ECX_MASK)
#define DMA_CR_CX_MASK                           0x20000u
#define DMA_CR_CX_SHIFT                          17u
#define DMA_CR_CX_WIDTH                          1u
#define DMA_CR_CX(x)                             (((uint32_t)(((uint32_t)(x))<<DMA_CR_CX_SHIFT))&DMA_CR_CX_MASK)
/* ES Bit Fields */
#define DMA_ES_DBE_MASK                          0x1u
#define DMA_ES_DBE_SHIFT                         0u
#define DMA_ES_DBE_WIDTH                         1u
#define DMA_ES_DBE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_DBE_SHIFT))&DMA_ES_DBE_MASK)
#define DMA_ES_SBE_MASK                          0x2u
#define DMA_ES_SBE_SHIFT                         1u
#define DMA_ES_SBE_WIDTH                         1u
#define DMA_ES_SBE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_SBE_SHIFT))&DMA_ES_SBE_MASK)
#define DMA_ES_SGE_MASK                          0x4u
#define DMA_ES_SGE_SHIFT                         2u
#define DMA_ES_SGE_WIDTH                         1u
#define DMA_ES_SGE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_SGE_SHIFT))&DMA_ES_SGE_MASK)
#define DMA_ES_NCE_MASK                          0x8u
#define DMA_ES_NCE_SHIFT                         3u
#define DMA_ES_NCE_WIDTH                         1u
#define DMA_ES_NCE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_NCE_SHIFT))&DMA_ES_NCE_MASK)
#define DMA_ES_DOE_MASK                          0x10u
#define DMA_ES_DOE_SHIFT                         4u
#define DMA_ES_DOE_WIDTH                         1u
#define DMA_ES_DOE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_DOE_SHIFT))&DMA_ES_DOE_MASK)
#define DMA_ES_DAE_MASK                          0x20u
#define DMA_ES_DAE_SHIFT                         5u
#define DMA_ES_DAE_WIDTH                         1u
#define DMA_ES_DAE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_DAE_SHIFT))&DMA_ES_DAE_MASK)
#define DMA_ES_SOE_MASK                          0x40u
#define DMA_ES_SOE_SHIFT                         6u
#define DMA_ES_SOE_WIDTH                         1u
#define DMA_ES_SOE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_SOE_SHIFT))&DMA_ES_SOE_MASK)
#define DMA_ES_SAE_MASK                          0x80u
#define DMA_ES_SAE_SHIFT                         7u
#define DMA_ES_SAE_WIDTH                         1u
#define DMA_ES_SAE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_SAE_SHIFT))&DMA_ES_SAE_MASK)
#define DMA_ES_ERRCHN_MASK                       0x1F00u
#define DMA_ES_ERRCHN_SHIFT                      8u
#define DMA_ES_ERRCHN_WIDTH                      5u
#define DMA_ES_ERRCHN(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ES_ERRCHN_SHIFT))&DMA_ES_ERRCHN_MASK)
#define DMA_ES_CPE_MASK                          0x4000u
#define DMA_ES_CPE_SHIFT                         14u
#define DMA_ES_CPE_WIDTH                         1u
#define DMA_ES_CPE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_CPE_SHIFT))&DMA_ES_CPE_MASK)
#define DMA_ES_GPE_MASK                          0x8000u
#define DMA_ES_GPE_SHIFT                         15u
#define DMA_ES_GPE_WIDTH                         1u
#define DMA_ES_GPE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_GPE_SHIFT))&DMA_ES_GPE_MASK)
#define DMA_ES_ECX_MASK                          0x10000u
#define DMA_ES_ECX_SHIFT                         16u
#define DMA_ES_ECX_WIDTH                         1u
#define DMA_ES_ECX(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_ECX_SHIFT))&DMA_ES_ECX_MASK)
#define DMA_ES_UCE_MASK                          0x20000u
#define DMA_ES_UCE_SHIFT                         17u
#define DMA_ES_UCE_WIDTH                         1u
#define DMA_ES_UCE(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_UCE_SHIFT))&DMA_ES_UCE_MASK)
#define DMA_ES_VLD_MASK                          0x80000000u
#define DMA_ES_VLD_SHIFT                         31u
#define DMA_ES_VLD_WIDTH                         1u
#define DMA_ES_VLD(x)                            (((uint32_t)(((uint32_t)(x))<<DMA_ES_VLD_SHIFT))&DMA_ES_VLD_MASK)
/* ERQ Bit Fields */
#define DMA_ERQ_ERQ0_MASK                        0x1u
#define DMA_ERQ_ERQ0_SHIFT                       0u
#define DMA_ERQ_ERQ0_WIDTH                       1u
#define DMA_ERQ_ERQ0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ0_SHIFT))&DMA_ERQ_ERQ0_MASK)
#define DMA_ERQ_ERQ1_MASK                        0x2u
#define DMA_ERQ_ERQ1_SHIFT                       1u
#define DMA_ERQ_ERQ1_WIDTH                       1u
#define DMA_ERQ_ERQ1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ1_SHIFT))&DMA_ERQ_ERQ1_MASK)
#define DMA_ERQ_ERQ2_MASK                        0x4u
#define DMA_ERQ_ERQ2_SHIFT                       2u
#define DMA_ERQ_ERQ2_WIDTH                       1u
#define DMA_ERQ_ERQ2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ2_SHIFT))&DMA_ERQ_ERQ2_MASK)
#define DMA_ERQ_ERQ3_MASK                        0x8u
#define DMA_ERQ_ERQ3_SHIFT                       3u
#define DMA_ERQ_ERQ3_WIDTH                       1u
#define DMA_ERQ_ERQ3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ3_SHIFT))&DMA_ERQ_ERQ3_MASK)
#define DMA_ERQ_ERQ4_MASK                        0x10u
#define DMA_ERQ_ERQ4_SHIFT                       4u
#define DMA_ERQ_ERQ4_WIDTH                       1u
#define DMA_ERQ_ERQ4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ4_SHIFT))&DMA_ERQ_ERQ4_MASK)
#define DMA_ERQ_ERQ5_MASK                        0x20u
#define DMA_ERQ_ERQ5_SHIFT                       5u
#define DMA_ERQ_ERQ5_WIDTH                       1u
#define DMA_ERQ_ERQ5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ5_SHIFT))&DMA_ERQ_ERQ5_MASK)
#define DMA_ERQ_ERQ6_MASK                        0x40u
#define DMA_ERQ_ERQ6_SHIFT                       6u
#define DMA_ERQ_ERQ6_WIDTH                       1u
#define DMA_ERQ_ERQ6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ6_SHIFT))&DMA_ERQ_ERQ6_MASK)
#define DMA_ERQ_ERQ7_MASK                        0x80u
#define DMA_ERQ_ERQ7_SHIFT                       7u
#define DMA_ERQ_ERQ7_WIDTH                       1u
#define DMA_ERQ_ERQ7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ7_SHIFT))&DMA_ERQ_ERQ7_MASK)
#define DMA_ERQ_ERQ8_MASK                        0x100u
#define DMA_ERQ_ERQ8_SHIFT                       8u
#define DMA_ERQ_ERQ8_WIDTH                       1u
#define DMA_ERQ_ERQ8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ8_SHIFT))&DMA_ERQ_ERQ8_MASK)
#define DMA_ERQ_ERQ9_MASK                        0x200u
#define DMA_ERQ_ERQ9_SHIFT                       9u
#define DMA_ERQ_ERQ9_WIDTH                       1u
#define DMA_ERQ_ERQ9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ9_SHIFT))&DMA_ERQ_ERQ9_MASK)
#define DMA_ERQ_ERQ10_MASK                       0x400u
#define DMA_ERQ_ERQ10_SHIFT                      10u
#define DMA_ERQ_ERQ10_WIDTH                      1u
#define DMA_ERQ_ERQ10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ10_SHIFT))&DMA_ERQ_ERQ10_MASK)
#define DMA_ERQ_ERQ11_MASK                       0x800u
#define DMA_ERQ_ERQ11_SHIFT                      11u
#define DMA_ERQ_ERQ11_WIDTH                      1u
#define DMA_ERQ_ERQ11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ11_SHIFT))&DMA_ERQ_ERQ11_MASK)
#define DMA_ERQ_ERQ12_MASK                       0x1000u
#define DMA_ERQ_ERQ12_SHIFT                      12u
#define DMA_ERQ_ERQ12_WIDTH                      1u
#define DMA_ERQ_ERQ12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ12_SHIFT))&DMA_ERQ_ERQ12_MASK)
#define DMA_ERQ_ERQ13_MASK                       0x2000u
#define DMA_ERQ_ERQ13_SHIFT                      13u
#define DMA_ERQ_ERQ13_WIDTH                      1u
#define DMA_ERQ_ERQ13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ13_SHIFT))&DMA_ERQ_ERQ13_MASK)
#define DMA_ERQ_ERQ14_MASK                       0x4000u
#define DMA_ERQ_ERQ14_SHIFT                      14u
#define DMA_ERQ_ERQ14_WIDTH                      1u
#define DMA_ERQ_ERQ14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ14_SHIFT))&DMA_ERQ_ERQ14_MASK)
#define DMA_ERQ_ERQ15_MASK                       0x8000u
#define DMA_ERQ_ERQ15_SHIFT                      15u
#define DMA_ERQ_ERQ15_WIDTH                      1u
#define DMA_ERQ_ERQ15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ15_SHIFT))&DMA_ERQ_ERQ15_MASK)
#define DMA_ERQ_ERQ16_MASK                       0x10000u
#define DMA_ERQ_ERQ16_SHIFT                      16u
#define DMA_ERQ_ERQ16_WIDTH                      1u
#define DMA_ERQ_ERQ16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ16_SHIFT))&DMA_ERQ_ERQ16_MASK)
#define DMA_ERQ_ERQ17_MASK                       0x20000u
#define DMA_ERQ_ERQ17_SHIFT                      17u
#define DMA_ERQ_ERQ17_WIDTH                      1u
#define DMA_ERQ_ERQ17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ17_SHIFT))&DMA_ERQ_ERQ17_MASK)
#define DMA_ERQ_ERQ18_MASK                       0x40000u
#define DMA_ERQ_ERQ18_SHIFT                      18u
#define DMA_ERQ_ERQ18_WIDTH                      1u
#define DMA_ERQ_ERQ18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ18_SHIFT))&DMA_ERQ_ERQ18_MASK)
#define DMA_ERQ_ERQ19_MASK                       0x80000u
#define DMA_ERQ_ERQ19_SHIFT                      19u
#define DMA_ERQ_ERQ19_WIDTH                      1u
#define DMA_ERQ_ERQ19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ19_SHIFT))&DMA_ERQ_ERQ19_MASK)
#define DMA_ERQ_ERQ20_MASK                       0x100000u
#define DMA_ERQ_ERQ20_SHIFT                      20u
#define DMA_ERQ_ERQ20_WIDTH                      1u
#define DMA_ERQ_ERQ20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ20_SHIFT))&DMA_ERQ_ERQ20_MASK)
#define DMA_ERQ_ERQ21_MASK                       0x200000u
#define DMA_ERQ_ERQ21_SHIFT                      21u
#define DMA_ERQ_ERQ21_WIDTH                      1u
#define DMA_ERQ_ERQ21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ21_SHIFT))&DMA_ERQ_ERQ21_MASK)
#define DMA_ERQ_ERQ22_MASK                       0x400000u
#define DMA_ERQ_ERQ22_SHIFT                      22u
#define DMA_ERQ_ERQ22_WIDTH                      1u
#define DMA_ERQ_ERQ22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ22_SHIFT))&DMA_ERQ_ERQ22_MASK)
#define DMA_ERQ_ERQ23_MASK                       0x800000u
#define DMA_ERQ_ERQ23_SHIFT                      23u
#define DMA_ERQ_ERQ23_WIDTH                      1u
#define DMA_ERQ_ERQ23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ23_SHIFT))&DMA_ERQ_ERQ23_MASK)
#define DMA_ERQ_ERQ24_MASK                       0x1000000u
#define DMA_ERQ_ERQ24_SHIFT                      24u
#define DMA_ERQ_ERQ24_WIDTH                      1u
#define DMA_ERQ_ERQ24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ24_SHIFT))&DMA_ERQ_ERQ24_MASK)
#define DMA_ERQ_ERQ25_MASK                       0x2000000u
#define DMA_ERQ_ERQ25_SHIFT                      25u
#define DMA_ERQ_ERQ25_WIDTH                      1u
#define DMA_ERQ_ERQ25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ25_SHIFT))&DMA_ERQ_ERQ25_MASK)
#define DMA_ERQ_ERQ26_MASK                       0x4000000u
#define DMA_ERQ_ERQ26_SHIFT                      26u
#define DMA_ERQ_ERQ26_WIDTH                      1u
#define DMA_ERQ_ERQ26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ26_SHIFT))&DMA_ERQ_ERQ26_MASK)
#define DMA_ERQ_ERQ27_MASK                       0x8000000u
#define DMA_ERQ_ERQ27_SHIFT                      27u
#define DMA_ERQ_ERQ27_WIDTH                      1u
#define DMA_ERQ_ERQ27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ27_SHIFT))&DMA_ERQ_ERQ27_MASK)
#define DMA_ERQ_ERQ28_MASK                       0x10000000u
#define DMA_ERQ_ERQ28_SHIFT                      28u
#define DMA_ERQ_ERQ28_WIDTH                      1u
#define DMA_ERQ_ERQ28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ28_SHIFT))&DMA_ERQ_ERQ28_MASK)
#define DMA_ERQ_ERQ29_MASK                       0x20000000u
#define DMA_ERQ_ERQ29_SHIFT                      29u
#define DMA_ERQ_ERQ29_WIDTH                      1u
#define DMA_ERQ_ERQ29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ29_SHIFT))&DMA_ERQ_ERQ29_MASK)
#define DMA_ERQ_ERQ30_MASK                       0x40000000u
#define DMA_ERQ_ERQ30_SHIFT                      30u
#define DMA_ERQ_ERQ30_WIDTH                      1u
#define DMA_ERQ_ERQ30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ30_SHIFT))&DMA_ERQ_ERQ30_MASK)
#define DMA_ERQ_ERQ31_MASK                       0x80000000u
#define DMA_ERQ_ERQ31_SHIFT                      31u
#define DMA_ERQ_ERQ31_WIDTH                      1u
#define DMA_ERQ_ERQ31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERQ_ERQ31_SHIFT))&DMA_ERQ_ERQ31_MASK)
/* EEI Bit Fields */
#define DMA_EEI_EEI0_MASK                        0x1u
#define DMA_EEI_EEI0_SHIFT                       0u
#define DMA_EEI_EEI0_WIDTH                       1u
#define DMA_EEI_EEI0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI0_SHIFT))&DMA_EEI_EEI0_MASK)
#define DMA_EEI_EEI1_MASK                        0x2u
#define DMA_EEI_EEI1_SHIFT                       1u
#define DMA_EEI_EEI1_WIDTH                       1u
#define DMA_EEI_EEI1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI1_SHIFT))&DMA_EEI_EEI1_MASK)
#define DMA_EEI_EEI2_MASK                        0x4u
#define DMA_EEI_EEI2_SHIFT                       2u
#define DMA_EEI_EEI2_WIDTH                       1u
#define DMA_EEI_EEI2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI2_SHIFT))&DMA_EEI_EEI2_MASK)
#define DMA_EEI_EEI3_MASK                        0x8u
#define DMA_EEI_EEI3_SHIFT                       3u
#define DMA_EEI_EEI3_WIDTH                       1u
#define DMA_EEI_EEI3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI3_SHIFT))&DMA_EEI_EEI3_MASK)
#define DMA_EEI_EEI4_MASK                        0x10u
#define DMA_EEI_EEI4_SHIFT                       4u
#define DMA_EEI_EEI4_WIDTH                       1u
#define DMA_EEI_EEI4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI4_SHIFT))&DMA_EEI_EEI4_MASK)
#define DMA_EEI_EEI5_MASK                        0x20u
#define DMA_EEI_EEI5_SHIFT                       5u
#define DMA_EEI_EEI5_WIDTH                       1u
#define DMA_EEI_EEI5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI5_SHIFT))&DMA_EEI_EEI5_MASK)
#define DMA_EEI_EEI6_MASK                        0x40u
#define DMA_EEI_EEI6_SHIFT                       6u
#define DMA_EEI_EEI6_WIDTH                       1u
#define DMA_EEI_EEI6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI6_SHIFT))&DMA_EEI_EEI6_MASK)
#define DMA_EEI_EEI7_MASK                        0x80u
#define DMA_EEI_EEI7_SHIFT                       7u
#define DMA_EEI_EEI7_WIDTH                       1u
#define DMA_EEI_EEI7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI7_SHIFT))&DMA_EEI_EEI7_MASK)
#define DMA_EEI_EEI8_MASK                        0x100u
#define DMA_EEI_EEI8_SHIFT                       8u
#define DMA_EEI_EEI8_WIDTH                       1u
#define DMA_EEI_EEI8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI8_SHIFT))&DMA_EEI_EEI8_MASK)
#define DMA_EEI_EEI9_MASK                        0x200u
#define DMA_EEI_EEI9_SHIFT                       9u
#define DMA_EEI_EEI9_WIDTH                       1u
#define DMA_EEI_EEI9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI9_SHIFT))&DMA_EEI_EEI9_MASK)
#define DMA_EEI_EEI10_MASK                       0x400u
#define DMA_EEI_EEI10_SHIFT                      10u
#define DMA_EEI_EEI10_WIDTH                      1u
#define DMA_EEI_EEI10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI10_SHIFT))&DMA_EEI_EEI10_MASK)
#define DMA_EEI_EEI11_MASK                       0x800u
#define DMA_EEI_EEI11_SHIFT                      11u
#define DMA_EEI_EEI11_WIDTH                      1u
#define DMA_EEI_EEI11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI11_SHIFT))&DMA_EEI_EEI11_MASK)
#define DMA_EEI_EEI12_MASK                       0x1000u
#define DMA_EEI_EEI12_SHIFT                      12u
#define DMA_EEI_EEI12_WIDTH                      1u
#define DMA_EEI_EEI12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI12_SHIFT))&DMA_EEI_EEI12_MASK)
#define DMA_EEI_EEI13_MASK                       0x2000u
#define DMA_EEI_EEI13_SHIFT                      13u
#define DMA_EEI_EEI13_WIDTH                      1u
#define DMA_EEI_EEI13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI13_SHIFT))&DMA_EEI_EEI13_MASK)
#define DMA_EEI_EEI14_MASK                       0x4000u
#define DMA_EEI_EEI14_SHIFT                      14u
#define DMA_EEI_EEI14_WIDTH                      1u
#define DMA_EEI_EEI14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI14_SHIFT))&DMA_EEI_EEI14_MASK)
#define DMA_EEI_EEI15_MASK                       0x8000u
#define DMA_EEI_EEI15_SHIFT                      15u
#define DMA_EEI_EEI15_WIDTH                      1u
#define DMA_EEI_EEI15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI15_SHIFT))&DMA_EEI_EEI15_MASK)
#define DMA_EEI_EEI16_MASK                       0x10000u
#define DMA_EEI_EEI16_SHIFT                      16u
#define DMA_EEI_EEI16_WIDTH                      1u
#define DMA_EEI_EEI16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI16_SHIFT))&DMA_EEI_EEI16_MASK)
#define DMA_EEI_EEI17_MASK                       0x20000u
#define DMA_EEI_EEI17_SHIFT                      17u
#define DMA_EEI_EEI17_WIDTH                      1u
#define DMA_EEI_EEI17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI17_SHIFT))&DMA_EEI_EEI17_MASK)
#define DMA_EEI_EEI18_MASK                       0x40000u
#define DMA_EEI_EEI18_SHIFT                      18u
#define DMA_EEI_EEI18_WIDTH                      1u
#define DMA_EEI_EEI18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI18_SHIFT))&DMA_EEI_EEI18_MASK)
#define DMA_EEI_EEI19_MASK                       0x80000u
#define DMA_EEI_EEI19_SHIFT                      19u
#define DMA_EEI_EEI19_WIDTH                      1u
#define DMA_EEI_EEI19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI19_SHIFT))&DMA_EEI_EEI19_MASK)
#define DMA_EEI_EEI20_MASK                       0x100000u
#define DMA_EEI_EEI20_SHIFT                      20u
#define DMA_EEI_EEI20_WIDTH                      1u
#define DMA_EEI_EEI20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI20_SHIFT))&DMA_EEI_EEI20_MASK)
#define DMA_EEI_EEI21_MASK                       0x200000u
#define DMA_EEI_EEI21_SHIFT                      21u
#define DMA_EEI_EEI21_WIDTH                      1u
#define DMA_EEI_EEI21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI21_SHIFT))&DMA_EEI_EEI21_MASK)
#define DMA_EEI_EEI22_MASK                       0x400000u
#define DMA_EEI_EEI22_SHIFT                      22u
#define DMA_EEI_EEI22_WIDTH                      1u
#define DMA_EEI_EEI22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI22_SHIFT))&DMA_EEI_EEI22_MASK)
#define DMA_EEI_EEI23_MASK                       0x800000u
#define DMA_EEI_EEI23_SHIFT                      23u
#define DMA_EEI_EEI23_WIDTH                      1u
#define DMA_EEI_EEI23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI23_SHIFT))&DMA_EEI_EEI23_MASK)
#define DMA_EEI_EEI24_MASK                       0x1000000u
#define DMA_EEI_EEI24_SHIFT                      24u
#define DMA_EEI_EEI24_WIDTH                      1u
#define DMA_EEI_EEI24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI24_SHIFT))&DMA_EEI_EEI24_MASK)
#define DMA_EEI_EEI25_MASK                       0x2000000u
#define DMA_EEI_EEI25_SHIFT                      25u
#define DMA_EEI_EEI25_WIDTH                      1u
#define DMA_EEI_EEI25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI25_SHIFT))&DMA_EEI_EEI25_MASK)
#define DMA_EEI_EEI26_MASK                       0x4000000u
#define DMA_EEI_EEI26_SHIFT                      26u
#define DMA_EEI_EEI26_WIDTH                      1u
#define DMA_EEI_EEI26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI26_SHIFT))&DMA_EEI_EEI26_MASK)
#define DMA_EEI_EEI27_MASK                       0x8000000u
#define DMA_EEI_EEI27_SHIFT                      27u
#define DMA_EEI_EEI27_WIDTH                      1u
#define DMA_EEI_EEI27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI27_SHIFT))&DMA_EEI_EEI27_MASK)
#define DMA_EEI_EEI28_MASK                       0x10000000u
#define DMA_EEI_EEI28_SHIFT                      28u
#define DMA_EEI_EEI28_WIDTH                      1u
#define DMA_EEI_EEI28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI28_SHIFT))&DMA_EEI_EEI28_MASK)
#define DMA_EEI_EEI29_MASK                       0x20000000u
#define DMA_EEI_EEI29_SHIFT                      29u
#define DMA_EEI_EEI29_WIDTH                      1u
#define DMA_EEI_EEI29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI29_SHIFT))&DMA_EEI_EEI29_MASK)
#define DMA_EEI_EEI30_MASK                       0x40000000u
#define DMA_EEI_EEI30_SHIFT                      30u
#define DMA_EEI_EEI30_WIDTH                      1u
#define DMA_EEI_EEI30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI30_SHIFT))&DMA_EEI_EEI30_MASK)
#define DMA_EEI_EEI31_MASK                       0x80000000u
#define DMA_EEI_EEI31_SHIFT                      31u
#define DMA_EEI_EEI31_WIDTH                      1u
#define DMA_EEI_EEI31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_EEI_EEI31_SHIFT))&DMA_EEI_EEI31_MASK)
/* SERQ Bit Fields */
#define DMA_SERQ_SERQ_MASK                       0x1Fu
#define DMA_SERQ_SERQ_SHIFT                      0u
#define DMA_SERQ_SERQ_WIDTH                      5u
#define DMA_SERQ_SERQ(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SERQ_SERQ_SHIFT))&DMA_SERQ_SERQ_MASK)
#define DMA_SERQ_SAER_MASK                       0x40u
#define DMA_SERQ_SAER_SHIFT                      6u
#define DMA_SERQ_SAER_WIDTH                      1u
#define DMA_SERQ_SAER(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SERQ_SAER_SHIFT))&DMA_SERQ_SAER_MASK)
#define DMA_SERQ_NOP_MASK                        0x80u
#define DMA_SERQ_NOP_SHIFT                       7u
#define DMA_SERQ_NOP_WIDTH                       1u
#define DMA_SERQ_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_SERQ_NOP_SHIFT))&DMA_SERQ_NOP_MASK)
/* CERQ Bit Fields */
#define DMA_CERQ_CERQ_MASK                       0x1Fu
#define DMA_CERQ_CERQ_SHIFT                      0u
#define DMA_CERQ_CERQ_WIDTH                      5u
#define DMA_CERQ_CERQ(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CERQ_CERQ_SHIFT))&DMA_CERQ_CERQ_MASK)
#define DMA_CERQ_CAER_MASK                       0x40u
#define DMA_CERQ_CAER_SHIFT                      6u
#define DMA_CERQ_CAER_WIDTH                      1u
#define DMA_CERQ_CAER(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CERQ_CAER_SHIFT))&DMA_CERQ_CAER_MASK)
#define DMA_CERQ_NOP_MASK                        0x80u
#define DMA_CERQ_NOP_SHIFT                       7u
#define DMA_CERQ_NOP_WIDTH                       1u
#define DMA_CERQ_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CERQ_NOP_SHIFT))&DMA_CERQ_NOP_MASK)
/* SEEI Bit Fields */
#define DMA_SEEI_SEEI_MASK                       0x1Fu
#define DMA_SEEI_SEEI_SHIFT                      0u
#define DMA_SEEI_SEEI_WIDTH                      5u
#define DMA_SEEI_SEEI(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SEEI_SEEI_SHIFT))&DMA_SEEI_SEEI_MASK)
#define DMA_SEEI_SAEE_MASK                       0x40u
#define DMA_SEEI_SAEE_SHIFT                      6u
#define DMA_SEEI_SAEE_WIDTH                      1u
#define DMA_SEEI_SAEE(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SEEI_SAEE_SHIFT))&DMA_SEEI_SAEE_MASK)
#define DMA_SEEI_NOP_MASK                        0x80u
#define DMA_SEEI_NOP_SHIFT                       7u
#define DMA_SEEI_NOP_WIDTH                       1u
#define DMA_SEEI_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_SEEI_NOP_SHIFT))&DMA_SEEI_NOP_MASK)
/* CEEI Bit Fields */
#define DMA_CEEI_CEEI_MASK                       0x1Fu
#define DMA_CEEI_CEEI_SHIFT                      0u
#define DMA_CEEI_CEEI_WIDTH                      5u
#define DMA_CEEI_CEEI(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CEEI_CEEI_SHIFT))&DMA_CEEI_CEEI_MASK)
#define DMA_CEEI_CAEE_MASK                       0x40u
#define DMA_CEEI_CAEE_SHIFT                      6u
#define DMA_CEEI_CAEE_WIDTH                      1u
#define DMA_CEEI_CAEE(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CEEI_CAEE_SHIFT))&DMA_CEEI_CAEE_MASK)
#define DMA_CEEI_NOP_MASK                        0x80u
#define DMA_CEEI_NOP_SHIFT                       7u
#define DMA_CEEI_NOP_WIDTH                       1u
#define DMA_CEEI_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CEEI_NOP_SHIFT))&DMA_CEEI_NOP_MASK)
/* CINT Bit Fields */
#define DMA_CINT_CINT_MASK                       0x1Fu
#define DMA_CINT_CINT_SHIFT                      0u
#define DMA_CINT_CINT_WIDTH                      5u
#define DMA_CINT_CINT(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CINT_CINT_SHIFT))&DMA_CINT_CINT_MASK)
#define DMA_CINT_CAIR_MASK                       0x40u
#define DMA_CINT_CAIR_SHIFT                      6u
#define DMA_CINT_CAIR_WIDTH                      1u
#define DMA_CINT_CAIR(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CINT_CAIR_SHIFT))&DMA_CINT_CAIR_MASK)
#define DMA_CINT_NOP_MASK                        0x80u
#define DMA_CINT_NOP_SHIFT                       7u
#define DMA_CINT_NOP_WIDTH                       1u
#define DMA_CINT_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CINT_NOP_SHIFT))&DMA_CINT_NOP_MASK)
/* CERR Bit Fields */
#define DMA_CERR_CERR_MASK                       0x1Fu
#define DMA_CERR_CERR_SHIFT                      0u
#define DMA_CERR_CERR_WIDTH                      5u
#define DMA_CERR_CERR(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CERR_CERR_SHIFT))&DMA_CERR_CERR_MASK)
#define DMA_CERR_CAEI_MASK                       0x40u
#define DMA_CERR_CAEI_SHIFT                      6u
#define DMA_CERR_CAEI_WIDTH                      1u
#define DMA_CERR_CAEI(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CERR_CAEI_SHIFT))&DMA_CERR_CAEI_MASK)
#define DMA_CERR_NOP_MASK                        0x80u
#define DMA_CERR_NOP_SHIFT                       7u
#define DMA_CERR_NOP_WIDTH                       1u
#define DMA_CERR_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CERR_NOP_SHIFT))&DMA_CERR_NOP_MASK)
/* SSRT Bit Fields */
#define DMA_SSRT_SSRT_MASK                       0x1Fu
#define DMA_SSRT_SSRT_SHIFT                      0u
#define DMA_SSRT_SSRT_WIDTH                      5u
#define DMA_SSRT_SSRT(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SSRT_SSRT_SHIFT))&DMA_SSRT_SSRT_MASK)
#define DMA_SSRT_SAST_MASK                       0x40u
#define DMA_SSRT_SAST_SHIFT                      6u
#define DMA_SSRT_SAST_WIDTH                      1u
#define DMA_SSRT_SAST(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_SSRT_SAST_SHIFT))&DMA_SSRT_SAST_MASK)
#define DMA_SSRT_NOP_MASK                        0x80u
#define DMA_SSRT_NOP_SHIFT                       7u
#define DMA_SSRT_NOP_WIDTH                       1u
#define DMA_SSRT_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_SSRT_NOP_SHIFT))&DMA_SSRT_NOP_MASK)
/* CDNE Bit Fields */
#define DMA_CDNE_CDNE_MASK                       0x1Fu
#define DMA_CDNE_CDNE_SHIFT                      0u
#define DMA_CDNE_CDNE_WIDTH                      5u
#define DMA_CDNE_CDNE(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CDNE_CDNE_SHIFT))&DMA_CDNE_CDNE_MASK)
#define DMA_CDNE_CADN_MASK                       0x40u
#define DMA_CDNE_CADN_SHIFT                      6u
#define DMA_CDNE_CADN_WIDTH                      1u
#define DMA_CDNE_CADN(x)                         (((uint8_t)(((uint8_t)(x))<<DMA_CDNE_CADN_SHIFT))&DMA_CDNE_CADN_MASK)
#define DMA_CDNE_NOP_MASK                        0x80u
#define DMA_CDNE_NOP_SHIFT                       7u
#define DMA_CDNE_NOP_WIDTH                       1u
#define DMA_CDNE_NOP(x)                          (((uint8_t)(((uint8_t)(x))<<DMA_CDNE_NOP_SHIFT))&DMA_CDNE_NOP_MASK)
/* INT Bit Fields */
#define DMA_INT_INT0_MASK                        0x1u
#define DMA_INT_INT0_SHIFT                       0u
#define DMA_INT_INT0_WIDTH                       1u
#define DMA_INT_INT0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT0_SHIFT))&DMA_INT_INT0_MASK)
#define DMA_INT_INT1_MASK                        0x2u
#define DMA_INT_INT1_SHIFT                       1u
#define DMA_INT_INT1_WIDTH                       1u
#define DMA_INT_INT1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT1_SHIFT))&DMA_INT_INT1_MASK)
#define DMA_INT_INT2_MASK                        0x4u
#define DMA_INT_INT2_SHIFT                       2u
#define DMA_INT_INT2_WIDTH                       1u
#define DMA_INT_INT2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT2_SHIFT))&DMA_INT_INT2_MASK)
#define DMA_INT_INT3_MASK                        0x8u
#define DMA_INT_INT3_SHIFT                       3u
#define DMA_INT_INT3_WIDTH                       1u
#define DMA_INT_INT3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT3_SHIFT))&DMA_INT_INT3_MASK)
#define DMA_INT_INT4_MASK                        0x10u
#define DMA_INT_INT4_SHIFT                       4u
#define DMA_INT_INT4_WIDTH                       1u
#define DMA_INT_INT4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT4_SHIFT))&DMA_INT_INT4_MASK)
#define DMA_INT_INT5_MASK                        0x20u
#define DMA_INT_INT5_SHIFT                       5u
#define DMA_INT_INT5_WIDTH                       1u
#define DMA_INT_INT5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT5_SHIFT))&DMA_INT_INT5_MASK)
#define DMA_INT_INT6_MASK                        0x40u
#define DMA_INT_INT6_SHIFT                       6u
#define DMA_INT_INT6_WIDTH                       1u
#define DMA_INT_INT6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT6_SHIFT))&DMA_INT_INT6_MASK)
#define DMA_INT_INT7_MASK                        0x80u
#define DMA_INT_INT7_SHIFT                       7u
#define DMA_INT_INT7_WIDTH                       1u
#define DMA_INT_INT7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT7_SHIFT))&DMA_INT_INT7_MASK)
#define DMA_INT_INT8_MASK                        0x100u
#define DMA_INT_INT8_SHIFT                       8u
#define DMA_INT_INT8_WIDTH                       1u
#define DMA_INT_INT8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT8_SHIFT))&DMA_INT_INT8_MASK)
#define DMA_INT_INT9_MASK                        0x200u
#define DMA_INT_INT9_SHIFT                       9u
#define DMA_INT_INT9_WIDTH                       1u
#define DMA_INT_INT9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT9_SHIFT))&DMA_INT_INT9_MASK)
#define DMA_INT_INT10_MASK                       0x400u
#define DMA_INT_INT10_SHIFT                      10u
#define DMA_INT_INT10_WIDTH                      1u
#define DMA_INT_INT10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT10_SHIFT))&DMA_INT_INT10_MASK)
#define DMA_INT_INT11_MASK                       0x800u
#define DMA_INT_INT11_SHIFT                      11u
#define DMA_INT_INT11_WIDTH                      1u
#define DMA_INT_INT11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT11_SHIFT))&DMA_INT_INT11_MASK)
#define DMA_INT_INT12_MASK                       0x1000u
#define DMA_INT_INT12_SHIFT                      12u
#define DMA_INT_INT12_WIDTH                      1u
#define DMA_INT_INT12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT12_SHIFT))&DMA_INT_INT12_MASK)
#define DMA_INT_INT13_MASK                       0x2000u
#define DMA_INT_INT13_SHIFT                      13u
#define DMA_INT_INT13_WIDTH                      1u
#define DMA_INT_INT13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT13_SHIFT))&DMA_INT_INT13_MASK)
#define DMA_INT_INT14_MASK                       0x4000u
#define DMA_INT_INT14_SHIFT                      14u
#define DMA_INT_INT14_WIDTH                      1u
#define DMA_INT_INT14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT14_SHIFT))&DMA_INT_INT14_MASK)
#define DMA_INT_INT15_MASK                       0x8000u
#define DMA_INT_INT15_SHIFT                      15u
#define DMA_INT_INT15_WIDTH                      1u
#define DMA_INT_INT15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT15_SHIFT))&DMA_INT_INT15_MASK)
#define DMA_INT_INT16_MASK                       0x10000u
#define DMA_INT_INT16_SHIFT                      16u
#define DMA_INT_INT16_WIDTH                      1u
#define DMA_INT_INT16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT16_SHIFT))&DMA_INT_INT16_MASK)
#define DMA_INT_INT17_MASK                       0x20000u
#define DMA_INT_INT17_SHIFT                      17u
#define DMA_INT_INT17_WIDTH                      1u
#define DMA_INT_INT17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT17_SHIFT))&DMA_INT_INT17_MASK)
#define DMA_INT_INT18_MASK                       0x40000u
#define DMA_INT_INT18_SHIFT                      18u
#define DMA_INT_INT18_WIDTH                      1u
#define DMA_INT_INT18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT18_SHIFT))&DMA_INT_INT18_MASK)
#define DMA_INT_INT19_MASK                       0x80000u
#define DMA_INT_INT19_SHIFT                      19u
#define DMA_INT_INT19_WIDTH                      1u
#define DMA_INT_INT19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT19_SHIFT))&DMA_INT_INT19_MASK)
#define DMA_INT_INT20_MASK                       0x100000u
#define DMA_INT_INT20_SHIFT                      20u
#define DMA_INT_INT20_WIDTH                      1u
#define DMA_INT_INT20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT20_SHIFT))&DMA_INT_INT20_MASK)
#define DMA_INT_INT21_MASK                       0x200000u
#define DMA_INT_INT21_SHIFT                      21u
#define DMA_INT_INT21_WIDTH                      1u
#define DMA_INT_INT21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT21_SHIFT))&DMA_INT_INT21_MASK)
#define DMA_INT_INT22_MASK                       0x400000u
#define DMA_INT_INT22_SHIFT                      22u
#define DMA_INT_INT22_WIDTH                      1u
#define DMA_INT_INT22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT22_SHIFT))&DMA_INT_INT22_MASK)
#define DMA_INT_INT23_MASK                       0x800000u
#define DMA_INT_INT23_SHIFT                      23u
#define DMA_INT_INT23_WIDTH                      1u
#define DMA_INT_INT23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT23_SHIFT))&DMA_INT_INT23_MASK)
#define DMA_INT_INT24_MASK                       0x1000000u
#define DMA_INT_INT24_SHIFT                      24u
#define DMA_INT_INT24_WIDTH                      1u
#define DMA_INT_INT24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT24_SHIFT))&DMA_INT_INT24_MASK)
#define DMA_INT_INT25_MASK                       0x2000000u
#define DMA_INT_INT25_SHIFT                      25u
#define DMA_INT_INT25_WIDTH                      1u
#define DMA_INT_INT25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT25_SHIFT))&DMA_INT_INT25_MASK)
#define DMA_INT_INT26_MASK                       0x4000000u
#define DMA_INT_INT26_SHIFT                      26u
#define DMA_INT_INT26_WIDTH                      1u
#define DMA_INT_INT26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT26_SHIFT))&DMA_INT_INT26_MASK)
#define DMA_INT_INT27_MASK                       0x8000000u
#define DMA_INT_INT27_SHIFT                      27u
#define DMA_INT_INT27_WIDTH                      1u
#define DMA_INT_INT27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT27_SHIFT))&DMA_INT_INT27_MASK)
#define DMA_INT_INT28_MASK                       0x10000000u
#define DMA_INT_INT28_SHIFT                      28u
#define DMA_INT_INT28_WIDTH                      1u
#define DMA_INT_INT28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT28_SHIFT))&DMA_INT_INT28_MASK)
#define DMA_INT_INT29_MASK                       0x20000000u
#define DMA_INT_INT29_SHIFT                      29u
#define DMA_INT_INT29_WIDTH                      1u
#define DMA_INT_INT29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT29_SHIFT))&DMA_INT_INT29_MASK)
#define DMA_INT_INT30_MASK                       0x40000000u
#define DMA_INT_INT30_SHIFT                      30u
#define DMA_INT_INT30_WIDTH                      1u
#define DMA_INT_INT30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT30_SHIFT))&DMA_INT_INT30_MASK)
#define DMA_INT_INT31_MASK                       0x80000000u
#define DMA_INT_INT31_SHIFT                      31u
#define DMA_INT_INT31_WIDTH                      1u
#define DMA_INT_INT31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_INT_INT31_SHIFT))&DMA_INT_INT31_MASK)
/* ERR Bit Fields */
#define DMA_ERR_ERR0_MASK                        0x1u
#define DMA_ERR_ERR0_SHIFT                       0u
#define DMA_ERR_ERR0_WIDTH                       1u
#define DMA_ERR_ERR0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR0_SHIFT))&DMA_ERR_ERR0_MASK)
#define DMA_ERR_ERR1_MASK                        0x2u
#define DMA_ERR_ERR1_SHIFT                       1u
#define DMA_ERR_ERR1_WIDTH                       1u
#define DMA_ERR_ERR1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR1_SHIFT))&DMA_ERR_ERR1_MASK)
#define DMA_ERR_ERR2_MASK                        0x4u
#define DMA_ERR_ERR2_SHIFT                       2u
#define DMA_ERR_ERR2_WIDTH                       1u
#define DMA_ERR_ERR2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR2_SHIFT))&DMA_ERR_ERR2_MASK)
#define DMA_ERR_ERR3_MASK                        0x8u
#define DMA_ERR_ERR3_SHIFT                       3u
#define DMA_ERR_ERR3_WIDTH                       1u
#define DMA_ERR_ERR3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR3_SHIFT))&DMA_ERR_ERR3_MASK)
#define DMA_ERR_ERR4_MASK                        0x10u
#define DMA_ERR_ERR4_SHIFT                       4u
#define DMA_ERR_ERR4_WIDTH                       1u
#define DMA_ERR_ERR4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR4_SHIFT))&DMA_ERR_ERR4_MASK)
#define DMA_ERR_ERR5_MASK                        0x20u
#define DMA_ERR_ERR5_SHIFT                       5u
#define DMA_ERR_ERR5_WIDTH                       1u
#define DMA_ERR_ERR5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR5_SHIFT))&DMA_ERR_ERR5_MASK)
#define DMA_ERR_ERR6_MASK                        0x40u
#define DMA_ERR_ERR6_SHIFT                       6u
#define DMA_ERR_ERR6_WIDTH                       1u
#define DMA_ERR_ERR6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR6_SHIFT))&DMA_ERR_ERR6_MASK)
#define DMA_ERR_ERR7_MASK                        0x80u
#define DMA_ERR_ERR7_SHIFT                       7u
#define DMA_ERR_ERR7_WIDTH                       1u
#define DMA_ERR_ERR7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR7_SHIFT))&DMA_ERR_ERR7_MASK)
#define DMA_ERR_ERR8_MASK                        0x100u
#define DMA_ERR_ERR8_SHIFT                       8u
#define DMA_ERR_ERR8_WIDTH                       1u
#define DMA_ERR_ERR8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR8_SHIFT))&DMA_ERR_ERR8_MASK)
#define DMA_ERR_ERR9_MASK                        0x200u
#define DMA_ERR_ERR9_SHIFT                       9u
#define DMA_ERR_ERR9_WIDTH                       1u
#define DMA_ERR_ERR9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR9_SHIFT))&DMA_ERR_ERR9_MASK)
#define DMA_ERR_ERR10_MASK                       0x400u
#define DMA_ERR_ERR10_SHIFT                      10u
#define DMA_ERR_ERR10_WIDTH                      1u
#define DMA_ERR_ERR10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR10_SHIFT))&DMA_ERR_ERR10_MASK)
#define DMA_ERR_ERR11_MASK                       0x800u
#define DMA_ERR_ERR11_SHIFT                      11u
#define DMA_ERR_ERR11_WIDTH                      1u
#define DMA_ERR_ERR11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR11_SHIFT))&DMA_ERR_ERR11_MASK)
#define DMA_ERR_ERR12_MASK                       0x1000u
#define DMA_ERR_ERR12_SHIFT                      12u
#define DMA_ERR_ERR12_WIDTH                      1u
#define DMA_ERR_ERR12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR12_SHIFT))&DMA_ERR_ERR12_MASK)
#define DMA_ERR_ERR13_MASK                       0x2000u
#define DMA_ERR_ERR13_SHIFT                      13u
#define DMA_ERR_ERR13_WIDTH                      1u
#define DMA_ERR_ERR13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR13_SHIFT))&DMA_ERR_ERR13_MASK)
#define DMA_ERR_ERR14_MASK                       0x4000u
#define DMA_ERR_ERR14_SHIFT                      14u
#define DMA_ERR_ERR14_WIDTH                      1u
#define DMA_ERR_ERR14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR14_SHIFT))&DMA_ERR_ERR14_MASK)
#define DMA_ERR_ERR15_MASK                       0x8000u
#define DMA_ERR_ERR15_SHIFT                      15u
#define DMA_ERR_ERR15_WIDTH                      1u
#define DMA_ERR_ERR15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR15_SHIFT))&DMA_ERR_ERR15_MASK)
#define DMA_ERR_ERR16_MASK                       0x10000u
#define DMA_ERR_ERR16_SHIFT                      16u
#define DMA_ERR_ERR16_WIDTH                      1u
#define DMA_ERR_ERR16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR16_SHIFT))&DMA_ERR_ERR16_MASK)
#define DMA_ERR_ERR17_MASK                       0x20000u
#define DMA_ERR_ERR17_SHIFT                      17u
#define DMA_ERR_ERR17_WIDTH                      1u
#define DMA_ERR_ERR17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR17_SHIFT))&DMA_ERR_ERR17_MASK)
#define DMA_ERR_ERR18_MASK                       0x40000u
#define DMA_ERR_ERR18_SHIFT                      18u
#define DMA_ERR_ERR18_WIDTH                      1u
#define DMA_ERR_ERR18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR18_SHIFT))&DMA_ERR_ERR18_MASK)
#define DMA_ERR_ERR19_MASK                       0x80000u
#define DMA_ERR_ERR19_SHIFT                      19u
#define DMA_ERR_ERR19_WIDTH                      1u
#define DMA_ERR_ERR19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR19_SHIFT))&DMA_ERR_ERR19_MASK)
#define DMA_ERR_ERR20_MASK                       0x100000u
#define DMA_ERR_ERR20_SHIFT                      20u
#define DMA_ERR_ERR20_WIDTH                      1u
#define DMA_ERR_ERR20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR20_SHIFT))&DMA_ERR_ERR20_MASK)
#define DMA_ERR_ERR21_MASK                       0x200000u
#define DMA_ERR_ERR21_SHIFT                      21u
#define DMA_ERR_ERR21_WIDTH                      1u
#define DMA_ERR_ERR21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR21_SHIFT))&DMA_ERR_ERR21_MASK)
#define DMA_ERR_ERR22_MASK                       0x400000u
#define DMA_ERR_ERR22_SHIFT                      22u
#define DMA_ERR_ERR22_WIDTH                      1u
#define DMA_ERR_ERR22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR22_SHIFT))&DMA_ERR_ERR22_MASK)
#define DMA_ERR_ERR23_MASK                       0x800000u
#define DMA_ERR_ERR23_SHIFT                      23u
#define DMA_ERR_ERR23_WIDTH                      1u
#define DMA_ERR_ERR23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR23_SHIFT))&DMA_ERR_ERR23_MASK)
#define DMA_ERR_ERR24_MASK                       0x1000000u
#define DMA_ERR_ERR24_SHIFT                      24u
#define DMA_ERR_ERR24_WIDTH                      1u
#define DMA_ERR_ERR24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR24_SHIFT))&DMA_ERR_ERR24_MASK)
#define DMA_ERR_ERR25_MASK                       0x2000000u
#define DMA_ERR_ERR25_SHIFT                      25u
#define DMA_ERR_ERR25_WIDTH                      1u
#define DMA_ERR_ERR25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR25_SHIFT))&DMA_ERR_ERR25_MASK)
#define DMA_ERR_ERR26_MASK                       0x4000000u
#define DMA_ERR_ERR26_SHIFT                      26u
#define DMA_ERR_ERR26_WIDTH                      1u
#define DMA_ERR_ERR26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR26_SHIFT))&DMA_ERR_ERR26_MASK)
#define DMA_ERR_ERR27_MASK                       0x8000000u
#define DMA_ERR_ERR27_SHIFT                      27u
#define DMA_ERR_ERR27_WIDTH                      1u
#define DMA_ERR_ERR27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR27_SHIFT))&DMA_ERR_ERR27_MASK)
#define DMA_ERR_ERR28_MASK                       0x10000000u
#define DMA_ERR_ERR28_SHIFT                      28u
#define DMA_ERR_ERR28_WIDTH                      1u
#define DMA_ERR_ERR28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR28_SHIFT))&DMA_ERR_ERR28_MASK)
#define DMA_ERR_ERR29_MASK                       0x20000000u
#define DMA_ERR_ERR29_SHIFT                      29u
#define DMA_ERR_ERR29_WIDTH                      1u
#define DMA_ERR_ERR29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR29_SHIFT))&DMA_ERR_ERR29_MASK)
#define DMA_ERR_ERR30_MASK                       0x40000000u
#define DMA_ERR_ERR30_SHIFT                      30u
#define DMA_ERR_ERR30_WIDTH                      1u
#define DMA_ERR_ERR30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR30_SHIFT))&DMA_ERR_ERR30_MASK)
#define DMA_ERR_ERR31_MASK                       0x80000000u
#define DMA_ERR_ERR31_SHIFT                      31u
#define DMA_ERR_ERR31_WIDTH                      1u
#define DMA_ERR_ERR31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_ERR_ERR31_SHIFT))&DMA_ERR_ERR31_MASK)
/* HRS Bit Fields */
#define DMA_HRS_HRS0_MASK                        0x1u
#define DMA_HRS_HRS0_SHIFT                       0u
#define DMA_HRS_HRS0_WIDTH                       1u
#define DMA_HRS_HRS0(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS0_SHIFT))&DMA_HRS_HRS0_MASK)
#define DMA_HRS_HRS1_MASK                        0x2u
#define DMA_HRS_HRS1_SHIFT                       1u
#define DMA_HRS_HRS1_WIDTH                       1u
#define DMA_HRS_HRS1(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS1_SHIFT))&DMA_HRS_HRS1_MASK)
#define DMA_HRS_HRS2_MASK                        0x4u
#define DMA_HRS_HRS2_SHIFT                       2u
#define DMA_HRS_HRS2_WIDTH                       1u
#define DMA_HRS_HRS2(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS2_SHIFT))&DMA_HRS_HRS2_MASK)
#define DMA_HRS_HRS3_MASK                        0x8u
#define DMA_HRS_HRS3_SHIFT                       3u
#define DMA_HRS_HRS3_WIDTH                       1u
#define DMA_HRS_HRS3(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS3_SHIFT))&DMA_HRS_HRS3_MASK)
#define DMA_HRS_HRS4_MASK                        0x10u
#define DMA_HRS_HRS4_SHIFT                       4u
#define DMA_HRS_HRS4_WIDTH                       1u
#define DMA_HRS_HRS4(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS4_SHIFT))&DMA_HRS_HRS4_MASK)
#define DMA_HRS_HRS5_MASK                        0x20u
#define DMA_HRS_HRS5_SHIFT                       5u
#define DMA_HRS_HRS5_WIDTH                       1u
#define DMA_HRS_HRS5(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS5_SHIFT))&DMA_HRS_HRS5_MASK)
#define DMA_HRS_HRS6_MASK                        0x40u
#define DMA_HRS_HRS6_SHIFT                       6u
#define DMA_HRS_HRS6_WIDTH                       1u
#define DMA_HRS_HRS6(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS6_SHIFT))&DMA_HRS_HRS6_MASK)
#define DMA_HRS_HRS7_MASK                        0x80u
#define DMA_HRS_HRS7_SHIFT                       7u
#define DMA_HRS_HRS7_WIDTH                       1u
#define DMA_HRS_HRS7(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS7_SHIFT))&DMA_HRS_HRS7_MASK)
#define DMA_HRS_HRS8_MASK                        0x100u
#define DMA_HRS_HRS8_SHIFT                       8u
#define DMA_HRS_HRS8_WIDTH                       1u
#define DMA_HRS_HRS8(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS8_SHIFT))&DMA_HRS_HRS8_MASK)
#define DMA_HRS_HRS9_MASK                        0x200u
#define DMA_HRS_HRS9_SHIFT                       9u
#define DMA_HRS_HRS9_WIDTH                       1u
#define DMA_HRS_HRS9(x)                          (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS9_SHIFT))&DMA_HRS_HRS9_MASK)
#define DMA_HRS_HRS10_MASK                       0x400u
#define DMA_HRS_HRS10_SHIFT                      10u
#define DMA_HRS_HRS10_WIDTH                      1u
#define DMA_HRS_HRS10(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS10_SHIFT))&DMA_HRS_HRS10_MASK)
#define DMA_HRS_HRS11_MASK                       0x800u
#define DMA_HRS_HRS11_SHIFT                      11u
#define DMA_HRS_HRS11_WIDTH                      1u
#define DMA_HRS_HRS11(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS11_SHIFT))&DMA_HRS_HRS11_MASK)
#define DMA_HRS_HRS12_MASK                       0x1000u
#define DMA_HRS_HRS12_SHIFT                      12u
#define DMA_HRS_HRS12_WIDTH                      1u
#define DMA_HRS_HRS12(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS12_SHIFT))&DMA_HRS_HRS12_MASK)
#define DMA_HRS_HRS13_MASK                       0x2000u
#define DMA_HRS_HRS13_SHIFT                      13u
#define DMA_HRS_HRS13_WIDTH                      1u
#define DMA_HRS_HRS13(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS13_SHIFT))&DMA_HRS_HRS13_MASK)
#define DMA_HRS_HRS14_MASK                       0x4000u
#define DMA_HRS_HRS14_SHIFT                      14u
#define DMA_HRS_HRS14_WIDTH                      1u
#define DMA_HRS_HRS14(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS14_SHIFT))&DMA_HRS_HRS14_MASK)
#define DMA_HRS_HRS15_MASK                       0x8000u
#define DMA_HRS_HRS15_SHIFT                      15u
#define DMA_HRS_HRS15_WIDTH                      1u
#define DMA_HRS_HRS15(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS15_SHIFT))&DMA_HRS_HRS15_MASK)
#define DMA_HRS_HRS16_MASK                       0x10000u
#define DMA_HRS_HRS16_SHIFT                      16u
#define DMA_HRS_HRS16_WIDTH                      1u
#define DMA_HRS_HRS16(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS16_SHIFT))&DMA_HRS_HRS16_MASK)
#define DMA_HRS_HRS17_MASK                       0x20000u
#define DMA_HRS_HRS17_SHIFT                      17u
#define DMA_HRS_HRS17_WIDTH                      1u
#define DMA_HRS_HRS17(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS17_SHIFT))&DMA_HRS_HRS17_MASK)
#define DMA_HRS_HRS18_MASK                       0x40000u
#define DMA_HRS_HRS18_SHIFT                      18u
#define DMA_HRS_HRS18_WIDTH                      1u
#define DMA_HRS_HRS18(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS18_SHIFT))&DMA_HRS_HRS18_MASK)
#define DMA_HRS_HRS19_MASK                       0x80000u
#define DMA_HRS_HRS19_SHIFT                      19u
#define DMA_HRS_HRS19_WIDTH                      1u
#define DMA_HRS_HRS19(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS19_SHIFT))&DMA_HRS_HRS19_MASK)
#define DMA_HRS_HRS20_MASK                       0x100000u
#define DMA_HRS_HRS20_SHIFT                      20u
#define DMA_HRS_HRS20_WIDTH                      1u
#define DMA_HRS_HRS20(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS20_SHIFT))&DMA_HRS_HRS20_MASK)
#define DMA_HRS_HRS21_MASK                       0x200000u
#define DMA_HRS_HRS21_SHIFT                      21u
#define DMA_HRS_HRS21_WIDTH                      1u
#define DMA_HRS_HRS21(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS21_SHIFT))&DMA_HRS_HRS21_MASK)
#define DMA_HRS_HRS22_MASK                       0x400000u
#define DMA_HRS_HRS22_SHIFT                      22u
#define DMA_HRS_HRS22_WIDTH                      1u
#define DMA_HRS_HRS22(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS22_SHIFT))&DMA_HRS_HRS22_MASK)
#define DMA_HRS_HRS23_MASK                       0x800000u
#define DMA_HRS_HRS23_SHIFT                      23u
#define DMA_HRS_HRS23_WIDTH                      1u
#define DMA_HRS_HRS23(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS23_SHIFT))&DMA_HRS_HRS23_MASK)
#define DMA_HRS_HRS24_MASK                       0x1000000u
#define DMA_HRS_HRS24_SHIFT                      24u
#define DMA_HRS_HRS24_WIDTH                      1u
#define DMA_HRS_HRS24(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS24_SHIFT))&DMA_HRS_HRS24_MASK)
#define DMA_HRS_HRS25_MASK                       0x2000000u
#define DMA_HRS_HRS25_SHIFT                      25u
#define DMA_HRS_HRS25_WIDTH                      1u
#define DMA_HRS_HRS25(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS25_SHIFT))&DMA_HRS_HRS25_MASK)
#define DMA_HRS_HRS26_MASK                       0x4000000u
#define DMA_HRS_HRS26_SHIFT                      26u
#define DMA_HRS_HRS26_WIDTH                      1u
#define DMA_HRS_HRS26(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS26_SHIFT))&DMA_HRS_HRS26_MASK)
#define DMA_HRS_HRS27_MASK                       0x8000000u
#define DMA_HRS_HRS27_SHIFT                      27u
#define DMA_HRS_HRS27_WIDTH                      1u
#define DMA_HRS_HRS27(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS27_SHIFT))&DMA_HRS_HRS27_MASK)
#define DMA_HRS_HRS28_MASK                       0x10000000u
#define DMA_HRS_HRS28_SHIFT                      28u
#define DMA_HRS_HRS28_WIDTH                      1u
#define DMA_HRS_HRS28(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS28_SHIFT))&DMA_HRS_HRS28_MASK)
#define DMA_HRS_HRS29_MASK                       0x20000000u
#define DMA_HRS_HRS29_SHIFT                      29u
#define DMA_HRS_HRS29_WIDTH                      1u
#define DMA_HRS_HRS29(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS29_SHIFT))&DMA_HRS_HRS29_MASK)
#define DMA_HRS_HRS30_MASK                       0x40000000u
#define DMA_HRS_HRS30_SHIFT                      30u
#define DMA_HRS_HRS30_WIDTH                      1u
#define DMA_HRS_HRS30(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS30_SHIFT))&DMA_HRS_HRS30_MASK)
#define DMA_HRS_HRS31_MASK                       0x80000000u
#define DMA_HRS_HRS31_SHIFT                      31u
#define DMA_HRS_HRS31_WIDTH                      1u
#define DMA_HRS_HRS31(x)                         (((uint32_t)(((uint32_t)(x))<<DMA_HRS_HRS31_SHIFT))&DMA_HRS_HRS31_MASK)
/* DCHPRI Bit Fields */
#define DMA_DCHPRI_CHPRI_MASK                    0xFu
#define DMA_DCHPRI_CHPRI_SHIFT                   0u
#define DMA_DCHPRI_CHPRI_WIDTH                   4u
#define DMA_DCHPRI_CHPRI(x)                      (((uint8_t)(((uint8_t)(x))<<DMA_DCHPRI_CHPRI_SHIFT))&DMA_DCHPRI_CHPRI_MASK)
#define DMA_DCHPRI_GRPPRI_MASK                   0x30u
#define DMA_DCHPRI_GRPPRI_SHIFT                  4u
#define DMA_DCHPRI_GRPPRI_WIDTH                  2u
#define DMA_DCHPRI_GRPPRI(x)                     (((uint8_t)(((uint8_t)(x))<<DMA_DCHPRI_GRPPRI_SHIFT))&DMA_DCHPRI_GRPPRI_MASK)
#define DMA_DCHPRI_DPA_MASK                      0x40u
#define DMA_DCHPRI_DPA_SHIFT                     6u
#define DMA_DCHPRI_DPA_WIDTH                     1u
#define DMA_DCHPRI_DPA(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHPRI_DPA_SHIFT))&DMA_DCHPRI_DPA_MASK)
#define DMA_DCHPRI_ECP_MASK                      0x80u
#define DMA_DCHPRI_ECP_SHIFT                     7u
#define DMA_DCHPRI_ECP_WIDTH                     1u
#define DMA_DCHPRI_ECP(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHPRI_ECP_SHIFT))&DMA_DCHPRI_ECP_MASK)
/* DCHMID Bit Fields */
#define DMA_DCHMID_MID_MASK                      0xFu
#define DMA_DCHMID_MID_SHIFT                     0u
#define DMA_DCHMID_MID_WIDTH                     4u
#define DMA_DCHMID_MID(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHMID_MID_SHIFT))&DMA_DCHMID_MID_MASK)
#define DMA_DCHMID_PAL_MASK                      0x40u
#define DMA_DCHMID_PAL_SHIFT                     6u
#define DMA_DCHMID_PAL_WIDTH                     1u
#define DMA_DCHMID_PAL(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHMID_PAL_SHIFT))&DMA_DCHMID_PAL_MASK)
#define DMA_DCHMID_EMI_MASK                      0x80u
#define DMA_DCHMID_EMI_SHIFT                     7u
#define DMA_DCHMID_EMI_WIDTH                     1u
#define DMA_DCHMID_EMI(x)                        (((uint8_t)(((uint8_t)(x))<<DMA_DCHMID_EMI_SHIFT))&DMA_DCHMID_EMI_MASK)
/* TCD_SADDR Bit Fields */
#define DMA_TCD_SADDR_SADDR_MASK                 0xFFFFFFFFu
#define DMA_TCD_SADDR_SADDR_SHIFT                0u
#define DMA_TCD_SADDR_SADDR_WIDTH                32u
#define DMA_TCD_SADDR_SADDR(x)                   (((uint32_t)(((uint32_t)(x))<<DMA_TCD_SADDR_SADDR_SHIFT))&DMA_TCD_SADDR_SADDR_MASK)
/* TCD_ATTR Bit Fields */
#define DMA_TCD_ATTR_DSIZE_MASK                  0x7u
#define DMA_TCD_ATTR_DSIZE_SHIFT                 0u
#define DMA_TCD_ATTR_DSIZE_WIDTH                 3u
#define DMA_TCD_ATTR_DSIZE(x)                    (((uint16_t)(((uint16_t)(x))<<DMA_TCD_ATTR_DSIZE_SHIFT))&DMA_TCD_ATTR_DSIZE_MASK)
#define DMA_TCD_ATTR_DMOD_MASK                   0xF8u
#define DMA_TCD_ATTR_DMOD_SHIFT                  3u
#define DMA_TCD_ATTR_DMOD_WIDTH                  5u
#define DMA_TCD_ATTR_DMOD(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_ATTR_DMOD_SHIFT))&DMA_TCD_ATTR_DMOD_MASK)
#define DMA_TCD_ATTR_SSIZE_MASK                  0x700u
#define DMA_TCD_ATTR_SSIZE_SHIFT                 8u
#define DMA_TCD_ATTR_SSIZE_WIDTH                 3u
#define DMA_TCD_ATTR_SSIZE(x)                    (((uint16_t)(((uint16_t)(x))<<DMA_TCD_ATTR_SSIZE_SHIFT))&DMA_TCD_ATTR_SSIZE_MASK)
#define DMA_TCD_ATTR_SMOD_MASK                   0xF800u
#define DMA_TCD_ATTR_SMOD_SHIFT                  11u
#define DMA_TCD_ATTR_SMOD_WIDTH                  5u
#define DMA_TCD_ATTR_SMOD(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_ATTR_SMOD_SHIFT))&DMA_TCD_ATTR_SMOD_MASK)
/* TCD_SOFF Bit Fields */
#define DMA_TCD_SOFF_SOFF_MASK                   0xFFFFu
#define DMA_TCD_SOFF_SOFF_SHIFT                  0u
#define DMA_TCD_SOFF_SOFF_WIDTH                  16u
#define DMA_TCD_SOFF_SOFF(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_SOFF_SOFF_SHIFT))&DMA_TCD_SOFF_SOFF_MASK)
/* TCD_NBYTES_MLNO Bit Fields */
#define DMA_TCD_NBYTES_MLNO_NBYTES_MASK          0xFFFFFFFFu
#define DMA_TCD_NBYTES_MLNO_NBYTES_SHIFT         0u
#define DMA_TCD_NBYTES_MLNO_NBYTES_WIDTH         32u
#define DMA_TCD_NBYTES_MLNO_NBYTES(x)            (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLNO_NBYTES_SHIFT))&DMA_TCD_NBYTES_MLNO_NBYTES_MASK)
/* TCD_NBYTES_MLOFFNO Bit Fields */
#define DMA_TCD_NBYTES_MLOFFNO_NBYTES_MASK       0x3FFFFFFFu
#define DMA_TCD_NBYTES_MLOFFNO_NBYTES_SHIFT      0u
#define DMA_TCD_NBYTES_MLOFFNO_NBYTES_WIDTH      30u
#define DMA_TCD_NBYTES_MLOFFNO_NBYTES(x)         (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFNO_NBYTES_SHIFT))&DMA_TCD_NBYTES_MLOFFNO_NBYTES_MASK)
#define DMA_TCD_NBYTES_MLOFFNO_DMLOE_MASK        0x40000000u
#define DMA_TCD_NBYTES_MLOFFNO_DMLOE_SHIFT       30u
#define DMA_TCD_NBYTES_MLOFFNO_DMLOE_WIDTH       1u
#define DMA_TCD_NBYTES_MLOFFNO_DMLOE(x)          (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFNO_DMLOE_SHIFT))&DMA_TCD_NBYTES_MLOFFNO_DMLOE_MASK)
#define DMA_TCD_NBYTES_MLOFFNO_SMLOE_MASK        0x80000000u
#define DMA_TCD_NBYTES_MLOFFNO_SMLOE_SHIFT       31u
#define DMA_TCD_NBYTES_MLOFFNO_SMLOE_WIDTH       1u
#define DMA_TCD_NBYTES_MLOFFNO_SMLOE(x)          (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFNO_SMLOE_SHIFT))&DMA_TCD_NBYTES_MLOFFNO_SMLOE_MASK)
/* TCD_NBYTES_MLOFFYES Bit Fields */
#define DMA_TCD_NBYTES_MLOFFYES_NBYTES_MASK      0x3FFu
#define DMA_TCD_NBYTES_MLOFFYES_NBYTES_SHIFT     0u
#define DMA_TCD_NBYTES_MLOFFYES_NBYTES_WIDTH     10u
#define DMA_TCD_NBYTES_MLOFFYES_NBYTES(x)        (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFYES_NBYTES_SHIFT))&DMA_TCD_NBYTES_MLOFFYES_NBYTES_MASK)
#define DMA_TCD_NBYTES_MLOFFYES_MLOFF_MASK       0x3FFFFC00u
#define DMA_TCD_NBYTES_MLOFFYES_MLOFF_SHIFT      10u
#define DMA_TCD_NBYTES_MLOFFYES_MLOFF_WIDTH      20u
#define DMA_TCD_NBYTES_MLOFFYES_MLOFF(x)         (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFYES_MLOFF_SHIFT))&DMA_TCD_NBYTES_MLOFFYES_MLOFF_MASK)
#define DMA_TCD_NBYTES_MLOFFYES_DMLOE_MASK       0x40000000u
#define DMA_TCD_NBYTES_MLOFFYES_DMLOE_SHIFT      30u
#define DMA_TCD_NBYTES_MLOFFYES_DMLOE_WIDTH      1u
#define DMA_TCD_NBYTES_MLOFFYES_DMLOE(x)         (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFYES_DMLOE_SHIFT))&DMA_TCD_NBYTES_MLOFFYES_DMLOE_MASK)
#define DMA_TCD_NBYTES_MLOFFYES_SMLOE_MASK       0x80000000u
#define DMA_TCD_NBYTES_MLOFFYES_SMLOE_SHIFT      31u
#define DMA_TCD_NBYTES_MLOFFYES_SMLOE_WIDTH      1u
#define DMA_TCD_NBYTES_MLOFFYES_SMLOE(x)         (((uint32_t)(((uint32_t)(x))<<DMA_TCD_NBYTES_MLOFFYES_SMLOE_SHIFT))&DMA_TCD_NBYTES_MLOFFYES_SMLOE_MASK)
/* TCD_SLAST Bit Fields */
#define DMA_TCD_SLAST_SLAST_MASK                 0xFFFFFFFFu
#define DMA_TCD_SLAST_SLAST_SHIFT                0u
#define DMA_TCD_SLAST_SLAST_WIDTH                32u
#define DMA_TCD_SLAST_SLAST(x)                   (((uint32_t)(((uint32_t)(x))<<DMA_TCD_SLAST_SLAST_SHIFT))&DMA_TCD_SLAST_SLAST_MASK)
/* TCD_DADDR Bit Fields */
#define DMA_TCD_DADDR_DADDR_MASK                 0xFFFFFFFFu
#define DMA_TCD_DADDR_DADDR_SHIFT                0u
#define DMA_TCD_DADDR_DADDR_WIDTH                32u
#define DMA_TCD_DADDR_DADDR(x)                   (((uint32_t)(((uint32_t)(x))<<DMA_TCD_DADDR_DADDR_SHIFT))&DMA_TCD_DADDR_DADDR_MASK)
/* TCD_CITER_ELINKNO Bit Fields */
#define DMA_TCD_CITER_ELINKNO_CITER_MASK         0x7FFFu
#define DMA_TCD_CITER_ELINKNO_CITER_SHIFT        0u
#define DMA_TCD_CITER_ELINKNO_CITER_WIDTH        15u
#define DMA_TCD_CITER_ELINKNO_CITER(x)           (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKNO_CITER_SHIFT))&DMA_TCD_CITER_ELINKNO_CITER_MASK)
#define DMA_TCD_CITER_ELINKNO_ELINK_MASK         0x8000u
#define DMA_TCD_CITER_ELINKNO_ELINK_SHIFT        15u
#define DMA_TCD_CITER_ELINKNO_ELINK_WIDTH        1u
#define DMA_TCD_CITER_ELINKNO_ELINK(x)           (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKNO_ELINK_SHIFT))&DMA_TCD_CITER_ELINKNO_ELINK_MASK)
/* TCD_CITER_ELINKYES Bit Fields */
#define DMA_TCD_CITER_ELINKYES_CITER_MASK        0x1FFu
#define DMA_TCD_CITER_ELINKYES_CITER_SHIFT       0u
#define DMA_TCD_CITER_ELINKYES_CITER_WIDTH       9u
#define DMA_TCD_CITER_ELINKYES_CITER(x)          (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKYES_CITER_SHIFT))&DMA_TCD_CITER_ELINKYES_CITER_MASK)
#define DMA_TCD_CITER_ELINKYES_LINKCH_MASK       0x3E00u
#define DMA_TCD_CITER_ELINKYES_LINKCH_SHIFT      9u
#define DMA_TCD_CITER_ELINKYES_LINKCH_WIDTH      5u
#define DMA_TCD_CITER_ELINKYES_LINKCH(x)         (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKYES_LINKCH_SHIFT))&DMA_TCD_CITER_ELINKYES_LINKCH_MASK)
#define DMA_TCD_CITER_ELINKYES_ELINK_MASK        0x8000u
#define DMA_TCD_CITER_ELINKYES_ELINK_SHIFT       15u
#define DMA_TCD_CITER_ELINKYES_ELINK_WIDTH       1u
#define DMA_TCD_CITER_ELINKYES_ELINK(x)          (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CITER_ELINKYES_ELINK_SHIFT))&DMA_TCD_CITER_ELINKYES_ELINK_MASK)
/* TCD_DOFF Bit Fields */
#define DMA_TCD_DOFF_DOFF_MASK                   0xFFFFu
#define DMA_TCD_DOFF_DOFF_SHIFT                  0u
#define DMA_TCD_DOFF_DOFF_WIDTH                  16u
#define DMA_TCD_DOFF_DOFF(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_DOFF_DOFF_SHIFT))&DMA_TCD_DOFF_DOFF_MASK)
/* TCD_DLASTSGA Bit Fields */
#define DMA_TCD_DLASTSGA_DLASTSGA_MASK           0xFFFFFFFFu
#define DMA_TCD_DLASTSGA_DLASTSGA_SHIFT          0u
#define DMA_TCD_DLASTSGA_DLASTSGA_WIDTH          32u
#define DMA_TCD_DLASTSGA_DLASTSGA(x)             (((uint32_t)(((uint32_t)(x))<<DMA_TCD_DLASTSGA_DLASTSGA_SHIFT))&DMA_TCD_DLASTSGA_DLASTSGA_MASK)
/* TCD_BITER_ELINKNO Bit Fields */
#define DMA_TCD_BITER_ELINKNO_BITER_MASK         0x7FFFu
#define DMA_TCD_BITER_ELINKNO_BITER_SHIFT        0u
#define DMA_TCD_BITER_ELINKNO_BITER_WIDTH        15u
#define DMA_TCD_BITER_ELINKNO_BITER(x)           (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKNO_BITER_SHIFT))&DMA_TCD_BITER_ELINKNO_BITER_MASK)
#define DMA_TCD_BITER_ELINKNO_ELINK_MASK         0x8000u
#define DMA_TCD_BITER_ELINKNO_ELINK_SHIFT        15u
#define DMA_TCD_BITER_ELINKNO_ELINK_WIDTH        1u
#define DMA_TCD_BITER_ELINKNO_ELINK(x)           (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKNO_ELINK_SHIFT))&DMA_TCD_BITER_ELINKNO_ELINK_MASK)
/* TCD_BITER_ELINKYES Bit Fields */
#define DMA_TCD_BITER_ELINKYES_BITER_MASK        0x1FFu
#define DMA_TCD_BITER_ELINKYES_BITER_SHIFT       0u
#define DMA_TCD_BITER_ELINKYES_BITER_WIDTH       9u
#define DMA_TCD_BITER_ELINKYES_BITER(x)          (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKYES_BITER_SHIFT))&DMA_TCD_BITER_ELINKYES_BITER_MASK)
#define DMA_TCD_BITER_ELINKYES_LINKCH_MASK       0x3E00u
#define DMA_TCD_BITER_ELINKYES_LINKCH_SHIFT      9u
#define DMA_TCD_BITER_ELINKYES_LINKCH_WIDTH      5u
#define DMA_TCD_BITER_ELINKYES_LINKCH(x)         (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKYES_LINKCH_SHIFT))&DMA_TCD_BITER_ELINKYES_LINKCH_MASK)
#define DMA_TCD_BITER_ELINKYES_ELINK_MASK        0x8000u
#define DMA_TCD_BITER_ELINKYES_ELINK_SHIFT       15u
#define DMA_TCD_BITER_ELINKYES_ELINK_WIDTH       1u
#define DMA_TCD_BITER_ELINKYES_ELINK(x)          (((uint16_t)(((uint16_t)(x))<<DMA_TCD_BITER_ELINKYES_ELINK_SHIFT))&DMA_TCD_BITER_ELINKYES_ELINK_MASK)
/* TCD_CSR Bit Fields */
#define DMA_TCD_CSR_START_MASK                   0x1u
#define DMA_TCD_CSR_START_SHIFT                  0u
#define DMA_TCD_CSR_START_WIDTH                  1u
#define DMA_TCD_CSR_START(x)                     (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_START_SHIFT))&DMA_TCD_CSR_START_MASK)
#define DMA_TCD_CSR_INTMAJOR_MASK                0x2u
#define DMA_TCD_CSR_INTMAJOR_SHIFT               1u
#define DMA_TCD_CSR_INTMAJOR_WIDTH               1u
#define DMA_TCD_CSR_INTMAJOR(x)                  (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_INTMAJOR_SHIFT))&DMA_TCD_CSR_INTMAJOR_MASK)
#define DMA_TCD_CSR_INTHALF_MASK                 0x4u
#define DMA_TCD_CSR_INTHALF_SHIFT                2u
#define DMA_TCD_CSR_INTHALF_WIDTH                1u
#define DMA_TCD_CSR_INTHALF(x)                   (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_INTHALF_SHIFT))&DMA_TCD_CSR_INTHALF_MASK)
#define DMA_TCD_CSR_DREQ_MASK                    0x8u
#define DMA_TCD_CSR_DREQ_SHIFT                   3u
#define DMA_TCD_CSR_DREQ_WIDTH                   1u
#define DMA_TCD_CSR_DREQ(x)                      (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_DREQ_SHIFT))&DMA_TCD_CSR_DREQ_MASK)
#define DMA_TCD_CSR_ESG_MASK                     0x10u
#define DMA_TCD_CSR_ESG_SHIFT                    4u
#define DMA_TCD_CSR_ESG_WIDTH                    1u
#define DMA_TCD_CSR_ESG(x)                       (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_ESG_SHIFT))&DMA_TCD_CSR_ESG_MASK)
#define DMA_TCD_CSR_MAJORELINK_MASK              0x20u
#define DMA_TCD_CSR_MAJORELINK_SHIFT             5u
#define DMA_TCD_CSR_MAJORELINK_WIDTH             1u
#define DMA_TCD_CSR_MAJORELINK(x)                (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_MAJORELINK_SHIFT))&DMA_TCD_CSR_MAJORELINK_MASK)
#define DMA_TCD_CSR_ACTIVE_MASK                  0x40u
#define DMA_TCD_CSR_ACTIVE_SHIFT                 6u
#define DMA_TCD_CSR_ACTIVE_WIDTH                 1u
#define DMA_TCD_CSR_ACTIVE(x)                    (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_ACTIVE_SHIFT))&DMA_TCD_CSR_ACTIVE_MASK)
#define DMA_TCD_CSR_DONE_MASK                    0x80u
#define DMA_TCD_CSR_DONE_SHIFT                   7u
#define DMA_TCD_CSR_DONE_WIDTH                   1u
#define DMA_TCD_CSR_DONE(x)                      (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_DONE_SHIFT))&DMA_TCD_CSR_DONE_MASK)
#define DMA_TCD_CSR_MAJORLINKCH_MASK             0x1F00u
#define DMA_TCD_CSR_MAJORLINKCH_SHIFT            8u
#define DMA_TCD_CSR_MAJORLINKCH_WIDTH            5u
#define DMA_TCD_CSR_MAJORLINKCH(x)               (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_MAJORLINKCH_SHIFT))&DMA_TCD_CSR_MAJORLINKCH_MASK)
#define DMA_TCD_CSR_BWC_MASK                     0xC000u
#define DMA_TCD_CSR_BWC_SHIFT                    14u
#define DMA_TCD_CSR_BWC_WIDTH                    2u
#define DMA_TCD_CSR_BWC(x)                       (((uint16_t)(((uint16_t)(x))<<DMA_TCD_CSR_BWC_SHIFT))&DMA_TCD_CSR_BWC_MASK)

/*!
 * @}
 */ /* end of group DMA_Register_Masks */


/*!
 * @}
 */ /* end of group DMA_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- DMAMUX Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DMAMUX_Peripheral_Access_Layer DMAMUX Peripheral Access Layer
 * @{
 */


/** DMAMUX - Size of Registers Arrays */
#define DMAMUX_CHCFG_COUNT                       16u

/** DMAMUX - Register Layout Typedef */
typedef struct {
  __IO uint8_t CHCFG[DMAMUX_CHCFG_COUNT];          /**< Channel Configuration register, array offset: 0x0, array step: 0x1 */
} DMAMUX_Type, *DMAMUX_MemMapPtr;

 /** Number of instances of the DMAMUX module. */
#define DMAMUX_INSTANCE_COUNT                    (2u)


/* DMAMUX - Peripheral instance base addresses */
/** Peripheral DMAMUX_0 base address */
#define DMAMUX_0_BASE                            (0xFFF6C000u)
/** Peripheral DMAMUX_0 base pointer */
#define DMAMUX_0                                 ((DMAMUX_Type *)DMAMUX_0_BASE)
/** Peripheral DMAMUX_1 base address */
#define DMAMUX_1_BASE                            (0xFFF6C200u)
/** Peripheral DMAMUX_1 base pointer */
#define DMAMUX_1                                 ((DMAMUX_Type *)DMAMUX_1_BASE)
/** Array initializer of DMAMUX peripheral base addresses */
#define DMAMUX_BASE_ADDRS                        { DMAMUX_0_BASE, DMAMUX_1_BASE }
/** Array initializer of DMAMUX peripheral base pointers */
#define DMAMUX_BASE_PTRS                         { DMAMUX_0, DMAMUX_1 }

/* ----------------------------------------------------------------------------
   -- DMAMUX Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DMAMUX_Register_Masks DMAMUX Register Masks
 * @{
 */

/* CHCFG Bit Fields */
#define DMAMUX_CHCFG_SOURCE_MASK                 0x3Fu
#define DMAMUX_CHCFG_SOURCE_SHIFT                0u
#define DMAMUX_CHCFG_SOURCE_WIDTH                6u
#define DMAMUX_CHCFG_SOURCE(x)                   (((uint8_t)(((uint8_t)(x))<<DMAMUX_CHCFG_SOURCE_SHIFT))&DMAMUX_CHCFG_SOURCE_MASK)
#define DMAMUX_CHCFG_TRIG_MASK                   0x40u
#define DMAMUX_CHCFG_TRIG_SHIFT                  6u
#define DMAMUX_CHCFG_TRIG_WIDTH                  1u
#define DMAMUX_CHCFG_TRIG(x)                     (((uint8_t)(((uint8_t)(x))<<DMAMUX_CHCFG_TRIG_SHIFT))&DMAMUX_CHCFG_TRIG_MASK)
#define DMAMUX_CHCFG_ENBL_MASK                   0x80u
#define DMAMUX_CHCFG_ENBL_SHIFT                  7u
#define DMAMUX_CHCFG_ENBL_WIDTH                  1u
#define DMAMUX_CHCFG_ENBL(x)                     (((uint8_t)(((uint8_t)(x))<<DMAMUX_CHCFG_ENBL_SHIFT))&DMAMUX_CHCFG_ENBL_MASK)

/*!
 * @}
 */ /* end of group DMAMUX_Register_Masks */


/*!
 * @}
 */ /* end of group DMAMUX_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- DSPI Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DSPI_Peripheral_Access_Layer DSPI Peripheral Access Layer
 * @{
 */


/** DSPI - Size of Registers Arrays */
#define DSPI_CTAR_COUNT                          6u
#define DSPI_CTAR_SLAVE_COUNT                    2u
#define DSPI_TXFR_COUNT                          4u
#define DSPI_RXFR_COUNT                          4u
#define DSPI_CTARE_COUNT                         6u

/** DSPI - Register Layout Typedef */
typedef struct {
  __IO uint32_t MCR;                               /**< Module Configuration Register, offset: 0x0 */
       uint8_t RESERVED_0[4];
  __IO uint32_t TCR;                               /**< Transfer Count Register, offset: 0x8 */
  union {                                          /* offset: 0xC */
    __IO uint32_t CTAR[DSPI_CTAR_COUNT];             /**< Clock and Transfer Attributes Register (In Master Mode), array offset: 0xC, array step: 0x4 */
    __IO uint32_t CTAR_SLAVE[DSPI_CTAR_SLAVE_COUNT]; /**< Clock and Transfer Attributes Register (In Slave Mode), array offset: 0xC, array step: 0x4 */
  } MODE;
       uint8_t RESERVED_1[8];
  __IO uint32_t SR;                                /**< Status Register, offset: 0x2C */
  __IO uint32_t RSER;                              /**< DMA/Interrupt Request Select and Enable Register, offset: 0x30 */
  union {                                          /* offset: 0x34 */
    struct {                                         /* offset: 0x34 */
      __IO uint16_t CMD;                               /**< DSPI_1_PUSHR_FIFO_CMD register...DSPI_2_PUSHR_FIFO_CMD register., offset: 0x34 */
      __IO uint16_t TX;                                /**< DSPI_1_PUSHR_FIFO_TX register...DSPI_2_PUSHR_FIFO_TX register., offset: 0x36 */
    } FIFO;
    __IO uint32_t PUSHR;                             /**< PUSH TX FIFO Register In Master Mode, offset: 0x34 */
    __IO uint32_t SLAVE;                             /**< PUSH TX FIFO Register In Slave Mode, offset: 0x34 */
  } PUSHR;
  __I  uint32_t POPR;                              /**< POP RX FIFO Register, offset: 0x38 */
  __I  uint32_t TXFR[DSPI_TXFR_COUNT];             /**< Transmit FIFO Registers, array offset: 0x3C, array step: 0x4 */
       uint8_t RESERVED_2[48];
  __I  uint32_t RXFR[DSPI_RXFR_COUNT];             /**< Receive FIFO Registers, array offset: 0x7C, array step: 0x4 */
       uint8_t RESERVED_3[48];
  __IO uint32_t DSICR0;                            /**< DSI Configuration Register 0, offset: 0xBC */
  __I  uint32_t SDR0;                              /**< DSI Serialization Data Register 0, offset: 0xC0 */
  __IO uint32_t ASDR0;                             /**< DSI Alternate Serialization Data Register 0, offset: 0xC4 */
  __I  uint32_t COMPR0;                            /**< DSI Transmit Comparison Register 0, offset: 0xC8 */
  __I  uint32_t DDR0;                              /**< DSI Deserialization Data Register 0, offset: 0xCC */
  __IO uint32_t DSICR1;                            /**< DSI Configuration Register 1, offset: 0xD0 */
  __IO uint32_t SSR0;                              /**< DSI Serialization Source Select Register 0, offset: 0xD4 */
  __IO uint32_t PISR0;                             /**< DSI Parallel Input Select Register 0, offset: 0xD8 */
  __IO uint32_t PISR1;                             /**< DSI Parallel Input Select Register 1, offset: 0xDC */
  __IO uint32_t PISR2;                             /**< DSI Parallel Input Select Register 2, offset: 0xE0 */
  __IO uint32_t PISR3;                             /**< DSI Parallel Input Select Register 3, offset: 0xE4 */
  __IO uint32_t DIMR0;                             /**< DSI Deserialized Data Interrupt Mask Register 0, offset: 0xE8 */
  __IO uint32_t DPIR0;                             /**< DSI Deserialized Data Polarity Interrupt Register 0, offset: 0xEC */
  __I  uint32_t SDR1;                              /**< DSI Serialization Data Register 1, offset: 0xF0 */
  __IO uint32_t ASDR1;                             /**< DSI Alternate Serialization Data Register 1, offset: 0xF4 */
  __I  uint32_t COMPR1;                            /**< DSI Transmit Comparison Register 1, offset: 0xF8 */
  __I  uint32_t DDR1;                              /**< DSI Deserialization Data Register 1, offset: 0xFC */
  __IO uint32_t SSR1;                              /**< DSI Serialization Source Select Register 1, offset: 0x100 */
  __IO uint32_t PISR4;                             /**< DSI Parallel Input Select Register 4, offset: 0x104 */
  __IO uint32_t PISR5;                             /**< DSI Parallel Input Select Register 5, offset: 0x108 */
  __IO uint32_t PISR6;                             /**< DSI Parallel Input Select Register 6, offset: 0x10C */
  __IO uint32_t PISR7;                             /**< DSI Parallel Input Select Register 7, offset: 0x110 */
  __IO uint32_t DIMR1;                             /**< DSI Deserialized Data Interrupt Mask Register 1, offset: 0x114 */
  __IO uint32_t DPIR1;                             /**< DSI Deserialized Data Polarity Interrupt Register 1, offset: 0x118 */
  __IO uint32_t CTARE[DSPI_CTARE_COUNT];           /**< Clock and Transfer Attributes Register Extended, array offset: 0x11C, array step: 0x4 */
       uint8_t RESERVED_4[8];
  __I  uint32_t SREX;                              /**< Status Register Extended, offset: 0x13C */
  __IO uint32_t TRIG;                              /**< Trigger Register, offset: 0x140 */
       uint8_t RESERVED_5[12];
  __IO uint32_t TSL;                               /**< Time Slot Length Register, offset: 0x150 */
  __IO uint32_t TS_CONF;                           /**< Time Slot Configuration Register, offset: 0x154 */
} DSPI_Type, *DSPI_MemMapPtr;

 /** Number of instances of the DSPI module. */
#define DSPI_INSTANCE_COUNT                      (4u)


/* DSPI - Peripheral instance base addresses */
/** Peripheral DSPI_0 base address */
#define DSPI_0_BASE                              (0xFFE70000u)
/** Peripheral DSPI_0 base pointer */
#define DSPI_0                                   ((DSPI_Type *)DSPI_0_BASE)
/** Peripheral DSPI_1 base address */
#define DSPI_1_BASE                              (0xFBE70000u)
/** Peripheral DSPI_1 base pointer */
#define DSPI_1                                   ((DSPI_Type *)DSPI_1_BASE)
/** Peripheral DSPI_2 base address */
#define DSPI_2_BASE                              (0xFFE74000u)
/** Peripheral DSPI_2 base pointer */
#define DSPI_2                                   ((DSPI_Type *)DSPI_2_BASE)
/** Peripheral DSPI_3 base address */
#define DSPI_3_BASE                              (0xFBE74000u)
/** Peripheral DSPI_3 base pointer */
#define DSPI_3                                   ((DSPI_Type *)DSPI_3_BASE)
/** Array initializer of DSPI peripheral base addresses */
#define DSPI_BASE_ADDRS                          { DSPI_0_BASE, DSPI_1_BASE, DSPI_2_BASE, DSPI_3_BASE }
/** Array initializer of DSPI peripheral base pointers */
#define DSPI_BASE_PTRS                           { DSPI_0, DSPI_1, DSPI_2, DSPI_3 }

/* ----------------------------------------------------------------------------
   -- DSPI Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup DSPI_Register_Masks DSPI Register Masks
 * @{
 */

/* MCR Bit Fields */
#define DSPI_MCR_HALT_MASK                       0x1u
#define DSPI_MCR_HALT_SHIFT                      0u
#define DSPI_MCR_HALT_WIDTH                      1u
#define DSPI_MCR_HALT(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_HALT_SHIFT))&DSPI_MCR_HALT_MASK)
#define DSPI_MCR_PES_MASK                        0x2u
#define DSPI_MCR_PES_SHIFT                       1u
#define DSPI_MCR_PES_WIDTH                       1u
#define DSPI_MCR_PES(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_PES_SHIFT))&DSPI_MCR_PES_MASK)
#define DSPI_MCR_FCPCS_MASK                      0x4u
#define DSPI_MCR_FCPCS_SHIFT                     2u
#define DSPI_MCR_FCPCS_WIDTH                     1u
#define DSPI_MCR_FCPCS(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_FCPCS_SHIFT))&DSPI_MCR_FCPCS_MASK)
#define DSPI_MCR_XSPI_MASK                       0x8u
#define DSPI_MCR_XSPI_SHIFT                      3u
#define DSPI_MCR_XSPI_WIDTH                      1u
#define DSPI_MCR_XSPI(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_XSPI_SHIFT))&DSPI_MCR_XSPI_MASK)
#define DSPI_MCR_SMPL_PT_MASK                    0x300u
#define DSPI_MCR_SMPL_PT_SHIFT                   8u
#define DSPI_MCR_SMPL_PT_WIDTH                   2u
#define DSPI_MCR_SMPL_PT(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_SMPL_PT_SHIFT))&DSPI_MCR_SMPL_PT_MASK)
#define DSPI_MCR_CLR_RXF_MASK                    0x400u
#define DSPI_MCR_CLR_RXF_SHIFT                   10u
#define DSPI_MCR_CLR_RXF_WIDTH                   1u
#define DSPI_MCR_CLR_RXF(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_CLR_RXF_SHIFT))&DSPI_MCR_CLR_RXF_MASK)
#define DSPI_MCR_CLR_TXF_MASK                    0x800u
#define DSPI_MCR_CLR_TXF_SHIFT                   11u
#define DSPI_MCR_CLR_TXF_WIDTH                   1u
#define DSPI_MCR_CLR_TXF(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_CLR_TXF_SHIFT))&DSPI_MCR_CLR_TXF_MASK)
#define DSPI_MCR_DIS_RXF_MASK                    0x1000u
#define DSPI_MCR_DIS_RXF_SHIFT                   12u
#define DSPI_MCR_DIS_RXF_WIDTH                   1u
#define DSPI_MCR_DIS_RXF(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_DIS_RXF_SHIFT))&DSPI_MCR_DIS_RXF_MASK)
#define DSPI_MCR_DIS_TXF_MASK                    0x2000u
#define DSPI_MCR_DIS_TXF_SHIFT                   13u
#define DSPI_MCR_DIS_TXF_WIDTH                   1u
#define DSPI_MCR_DIS_TXF(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_DIS_TXF_SHIFT))&DSPI_MCR_DIS_TXF_MASK)
#define DSPI_MCR_MDIS_MASK                       0x4000u
#define DSPI_MCR_MDIS_SHIFT                      14u
#define DSPI_MCR_MDIS_WIDTH                      1u
#define DSPI_MCR_MDIS(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_MDIS_SHIFT))&DSPI_MCR_MDIS_MASK)
#define DSPI_MCR_PCSIS_MASK                      0x3F0000u
#define DSPI_MCR_PCSIS_SHIFT                     16u
#define DSPI_MCR_PCSIS_WIDTH                     6u
#define DSPI_MCR_PCSIS(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_PCSIS_SHIFT))&DSPI_MCR_PCSIS_MASK)
#define DSPI_MCR_ROOE_MASK                       0x1000000u
#define DSPI_MCR_ROOE_SHIFT                      24u
#define DSPI_MCR_ROOE_WIDTH                      1u
#define DSPI_MCR_ROOE(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_ROOE_SHIFT))&DSPI_MCR_ROOE_MASK)
#define DSPI_MCR_PCSSE_MASK                      0x2000000u
#define DSPI_MCR_PCSSE_SHIFT                     25u
#define DSPI_MCR_PCSSE_WIDTH                     1u
#define DSPI_MCR_PCSSE(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_PCSSE_SHIFT))&DSPI_MCR_PCSSE_MASK)
#define DSPI_MCR_MTFE_MASK                       0x4000000u
#define DSPI_MCR_MTFE_SHIFT                      26u
#define DSPI_MCR_MTFE_WIDTH                      1u
#define DSPI_MCR_MTFE(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_MTFE_SHIFT))&DSPI_MCR_MTFE_MASK)
#define DSPI_MCR_FRZ_MASK                        0x8000000u
#define DSPI_MCR_FRZ_SHIFT                       27u
#define DSPI_MCR_FRZ_WIDTH                       1u
#define DSPI_MCR_FRZ(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_FRZ_SHIFT))&DSPI_MCR_FRZ_MASK)
#define DSPI_MCR_DCONF_MASK                      0x30000000u
#define DSPI_MCR_DCONF_SHIFT                     28u
#define DSPI_MCR_DCONF_WIDTH                     2u
#define DSPI_MCR_DCONF(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_DCONF_SHIFT))&DSPI_MCR_DCONF_MASK)
#define DSPI_MCR_CONT_SCKE_MASK                  0x40000000u
#define DSPI_MCR_CONT_SCKE_SHIFT                 30u
#define DSPI_MCR_CONT_SCKE_WIDTH                 1u
#define DSPI_MCR_CONT_SCKE(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_CONT_SCKE_SHIFT))&DSPI_MCR_CONT_SCKE_MASK)
#define DSPI_MCR_MSTR_MASK                       0x80000000u
#define DSPI_MCR_MSTR_SHIFT                      31u
#define DSPI_MCR_MSTR_WIDTH                      1u
#define DSPI_MCR_MSTR(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_MCR_MSTR_SHIFT))&DSPI_MCR_MSTR_MASK)
/* TCR Bit Fields */
#define DSPI_TCR_SPI_TCNT_MASK                   0xFFFF0000u
#define DSPI_TCR_SPI_TCNT_SHIFT                  16u
#define DSPI_TCR_SPI_TCNT_WIDTH                  16u
#define DSPI_TCR_SPI_TCNT(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_TCR_SPI_TCNT_SHIFT))&DSPI_TCR_SPI_TCNT_MASK)
/* CTAR Bit Fields */
#define DSPI_CTAR_BR_MASK                        0xFu
#define DSPI_CTAR_BR_SHIFT                       0u
#define DSPI_CTAR_BR_WIDTH                       4u
#define DSPI_CTAR_BR(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_BR_SHIFT))&DSPI_CTAR_BR_MASK)
#define DSPI_CTAR_DT_MASK                        0xF0u
#define DSPI_CTAR_DT_SHIFT                       4u
#define DSPI_CTAR_DT_WIDTH                       4u
#define DSPI_CTAR_DT(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_DT_SHIFT))&DSPI_CTAR_DT_MASK)
#define DSPI_CTAR_ASC_MASK                       0xF00u
#define DSPI_CTAR_ASC_SHIFT                      8u
#define DSPI_CTAR_ASC_WIDTH                      4u
#define DSPI_CTAR_ASC(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_ASC_SHIFT))&DSPI_CTAR_ASC_MASK)
#define DSPI_CTAR_CSSCK_MASK                     0xF000u
#define DSPI_CTAR_CSSCK_SHIFT                    12u
#define DSPI_CTAR_CSSCK_WIDTH                    4u
#define DSPI_CTAR_CSSCK(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_CSSCK_SHIFT))&DSPI_CTAR_CSSCK_MASK)
#define DSPI_CTAR_PBR_MASK                       0x30000u
#define DSPI_CTAR_PBR_SHIFT                      16u
#define DSPI_CTAR_PBR_WIDTH                      2u
#define DSPI_CTAR_PBR(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_PBR_SHIFT))&DSPI_CTAR_PBR_MASK)
#define DSPI_CTAR_PDT_MASK                       0xC0000u
#define DSPI_CTAR_PDT_SHIFT                      18u
#define DSPI_CTAR_PDT_WIDTH                      2u
#define DSPI_CTAR_PDT(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_PDT_SHIFT))&DSPI_CTAR_PDT_MASK)
#define DSPI_CTAR_PASC_MASK                      0x300000u
#define DSPI_CTAR_PASC_SHIFT                     20u
#define DSPI_CTAR_PASC_WIDTH                     2u
#define DSPI_CTAR_PASC(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_PASC_SHIFT))&DSPI_CTAR_PASC_MASK)
#define DSPI_CTAR_PCSSCK_MASK                    0xC00000u
#define DSPI_CTAR_PCSSCK_SHIFT                   22u
#define DSPI_CTAR_PCSSCK_WIDTH                   2u
#define DSPI_CTAR_PCSSCK(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_PCSSCK_SHIFT))&DSPI_CTAR_PCSSCK_MASK)
#define DSPI_CTAR_LSBFE_MASK                     0x1000000u
#define DSPI_CTAR_LSBFE_SHIFT                    24u
#define DSPI_CTAR_LSBFE_WIDTH                    1u
#define DSPI_CTAR_LSBFE(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_LSBFE_SHIFT))&DSPI_CTAR_LSBFE_MASK)
#define DSPI_CTAR_CPHA_MASK                      0x2000000u
#define DSPI_CTAR_CPHA_SHIFT                     25u
#define DSPI_CTAR_CPHA_WIDTH                     1u
#define DSPI_CTAR_CPHA(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_CPHA_SHIFT))&DSPI_CTAR_CPHA_MASK)
#define DSPI_CTAR_CPOL_MASK                      0x4000000u
#define DSPI_CTAR_CPOL_SHIFT                     26u
#define DSPI_CTAR_CPOL_WIDTH                     1u
#define DSPI_CTAR_CPOL(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_CPOL_SHIFT))&DSPI_CTAR_CPOL_MASK)
#define DSPI_CTAR_FMSZ_MASK                      0x78000000u
#define DSPI_CTAR_FMSZ_SHIFT                     27u
#define DSPI_CTAR_FMSZ_WIDTH                     4u
#define DSPI_CTAR_FMSZ(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_FMSZ_SHIFT))&DSPI_CTAR_FMSZ_MASK)
#define DSPI_CTAR_DBR_MASK                       0x80000000u
#define DSPI_CTAR_DBR_SHIFT                      31u
#define DSPI_CTAR_DBR_WIDTH                      1u
#define DSPI_CTAR_DBR(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_DBR_SHIFT))&DSPI_CTAR_DBR_MASK)
/* CTAR_SLAVE Bit Fields */
#define DSPI_CTAR_SLAVE_FMSZ5_MASK               0x400000u
#define DSPI_CTAR_SLAVE_FMSZ5_SHIFT              22u
#define DSPI_CTAR_SLAVE_FMSZ5_WIDTH              1u
#define DSPI_CTAR_SLAVE_FMSZ5(x)                 (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_SLAVE_FMSZ5_SHIFT))&DSPI_CTAR_SLAVE_FMSZ5_MASK)
#define DSPI_CTAR_SLAVE_PP_MASK                  0x800000u
#define DSPI_CTAR_SLAVE_PP_SHIFT                 23u
#define DSPI_CTAR_SLAVE_PP_WIDTH                 1u
#define DSPI_CTAR_SLAVE_PP(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_SLAVE_PP_SHIFT))&DSPI_CTAR_SLAVE_PP_MASK)
#define DSPI_CTAR_SLAVE_PE_MASK                  0x1000000u
#define DSPI_CTAR_SLAVE_PE_SHIFT                 24u
#define DSPI_CTAR_SLAVE_PE_WIDTH                 1u
#define DSPI_CTAR_SLAVE_PE(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_SLAVE_PE_SHIFT))&DSPI_CTAR_SLAVE_PE_MASK)
#define DSPI_CTAR_SLAVE_CPHA_MASK                0x2000000u
#define DSPI_CTAR_SLAVE_CPHA_SHIFT               25u
#define DSPI_CTAR_SLAVE_CPHA_WIDTH               1u
#define DSPI_CTAR_SLAVE_CPHA(x)                  (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_SLAVE_CPHA_SHIFT))&DSPI_CTAR_SLAVE_CPHA_MASK)
#define DSPI_CTAR_SLAVE_CPOL_MASK                0x4000000u
#define DSPI_CTAR_SLAVE_CPOL_SHIFT               26u
#define DSPI_CTAR_SLAVE_CPOL_WIDTH               1u
#define DSPI_CTAR_SLAVE_CPOL(x)                  (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_SLAVE_CPOL_SHIFT))&DSPI_CTAR_SLAVE_CPOL_MASK)
#define DSPI_CTAR_SLAVE_FMSZ_MASK                0xF8000000u
#define DSPI_CTAR_SLAVE_FMSZ_SHIFT               27u
#define DSPI_CTAR_SLAVE_FMSZ_WIDTH               5u
#define DSPI_CTAR_SLAVE_FMSZ(x)                  (((uint32_t)(((uint32_t)(x))<<DSPI_CTAR_SLAVE_FMSZ_SHIFT))&DSPI_CTAR_SLAVE_FMSZ_MASK)
/* SR Bit Fields */
#define DSPI_SR_POPNXTPTR_MASK                   0xFu
#define DSPI_SR_POPNXTPTR_SHIFT                  0u
#define DSPI_SR_POPNXTPTR_WIDTH                  4u
#define DSPI_SR_POPNXTPTR(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_SR_POPNXTPTR_SHIFT))&DSPI_SR_POPNXTPTR_MASK)
#define DSPI_SR_RXCTR_MASK                       0xF0u
#define DSPI_SR_RXCTR_SHIFT                      4u
#define DSPI_SR_RXCTR_WIDTH                      4u
#define DSPI_SR_RXCTR(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_SR_RXCTR_SHIFT))&DSPI_SR_RXCTR_MASK)
#define DSPI_SR_TXNXTPTR_MASK                    0xF00u
#define DSPI_SR_TXNXTPTR_SHIFT                   8u
#define DSPI_SR_TXNXTPTR_WIDTH                   4u
#define DSPI_SR_TXNXTPTR(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_SR_TXNXTPTR_SHIFT))&DSPI_SR_TXNXTPTR_MASK)
#define DSPI_SR_TXCTR_MASK                       0xF000u
#define DSPI_SR_TXCTR_SHIFT                      12u
#define DSPI_SR_TXCTR_WIDTH                      4u
#define DSPI_SR_TXCTR(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_SR_TXCTR_SHIFT))&DSPI_SR_TXCTR_MASK)
#define DSPI_SR_CMDFFF_MASK                      0x10000u
#define DSPI_SR_CMDFFF_SHIFT                     16u
#define DSPI_SR_CMDFFF_WIDTH                     1u
#define DSPI_SR_CMDFFF(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_SR_CMDFFF_SHIFT))&DSPI_SR_CMDFFF_MASK)
#define DSPI_SR_RFDF_MASK                        0x20000u
#define DSPI_SR_RFDF_SHIFT                       17u
#define DSPI_SR_RFDF_WIDTH                       1u
#define DSPI_SR_RFDF(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SR_RFDF_SHIFT))&DSPI_SR_RFDF_MASK)
#define DSPI_SR_TFIWF_MASK                       0x40000u
#define DSPI_SR_TFIWF_SHIFT                      18u
#define DSPI_SR_TFIWF_WIDTH                      1u
#define DSPI_SR_TFIWF(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_SR_TFIWF_SHIFT))&DSPI_SR_TFIWF_MASK)
#define DSPI_SR_RFOF_MASK                        0x80000u
#define DSPI_SR_RFOF_SHIFT                       19u
#define DSPI_SR_RFOF_WIDTH                       1u
#define DSPI_SR_RFOF(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SR_RFOF_SHIFT))&DSPI_SR_RFOF_MASK)
#define DSPI_SR_DDIF_MASK                        0x100000u
#define DSPI_SR_DDIF_SHIFT                       20u
#define DSPI_SR_DDIF_WIDTH                       1u
#define DSPI_SR_DDIF(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SR_DDIF_SHIFT))&DSPI_SR_DDIF_MASK)
#define DSPI_SR_SPEF_MASK                        0x200000u
#define DSPI_SR_SPEF_SHIFT                       21u
#define DSPI_SR_SPEF_WIDTH                       1u
#define DSPI_SR_SPEF(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SR_SPEF_SHIFT))&DSPI_SR_SPEF_MASK)
#define DSPI_SR_DPEF_MASK                        0x400000u
#define DSPI_SR_DPEF_SHIFT                       22u
#define DSPI_SR_DPEF_WIDTH                       1u
#define DSPI_SR_DPEF(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SR_DPEF_SHIFT))&DSPI_SR_DPEF_MASK)
#define DSPI_SR_CMDTCF_MASK                      0x800000u
#define DSPI_SR_CMDTCF_SHIFT                     23u
#define DSPI_SR_CMDTCF_WIDTH                     1u
#define DSPI_SR_CMDTCF(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_SR_CMDTCF_SHIFT))&DSPI_SR_CMDTCF_MASK)
#define DSPI_SR_BSYF_MASK                        0x1000000u
#define DSPI_SR_BSYF_SHIFT                       24u
#define DSPI_SR_BSYF_WIDTH                       1u
#define DSPI_SR_BSYF(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SR_BSYF_SHIFT))&DSPI_SR_BSYF_MASK)
#define DSPI_SR_TFFF_MASK                        0x2000000u
#define DSPI_SR_TFFF_SHIFT                       25u
#define DSPI_SR_TFFF_WIDTH                       1u
#define DSPI_SR_TFFF(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SR_TFFF_SHIFT))&DSPI_SR_TFFF_MASK)
#define DSPI_SR_DSITCF_MASK                      0x4000000u
#define DSPI_SR_DSITCF_SHIFT                     26u
#define DSPI_SR_DSITCF_WIDTH                     1u
#define DSPI_SR_DSITCF(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_SR_DSITCF_SHIFT))&DSPI_SR_DSITCF_MASK)
#define DSPI_SR_TFUF_MASK                        0x8000000u
#define DSPI_SR_TFUF_SHIFT                       27u
#define DSPI_SR_TFUF_WIDTH                       1u
#define DSPI_SR_TFUF(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SR_TFUF_SHIFT))&DSPI_SR_TFUF_MASK)
#define DSPI_SR_EOQF_MASK                        0x10000000u
#define DSPI_SR_EOQF_SHIFT                       28u
#define DSPI_SR_EOQF_WIDTH                       1u
#define DSPI_SR_EOQF(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SR_EOQF_SHIFT))&DSPI_SR_EOQF_MASK)
#define DSPI_SR_SPITCF_MASK                      0x20000000u
#define DSPI_SR_SPITCF_SHIFT                     29u
#define DSPI_SR_SPITCF_WIDTH                     1u
#define DSPI_SR_SPITCF(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_SR_SPITCF_SHIFT))&DSPI_SR_SPITCF_MASK)
#define DSPI_SR_TXRXS_MASK                       0x40000000u
#define DSPI_SR_TXRXS_SHIFT                      30u
#define DSPI_SR_TXRXS_WIDTH                      1u
#define DSPI_SR_TXRXS(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_SR_TXRXS_SHIFT))&DSPI_SR_TXRXS_MASK)
#define DSPI_SR_TCF_MASK                         0x80000000u
#define DSPI_SR_TCF_SHIFT                        31u
#define DSPI_SR_TCF_WIDTH                        1u
#define DSPI_SR_TCF(x)                           (((uint32_t)(((uint32_t)(x))<<DSPI_SR_TCF_SHIFT))&DSPI_SR_TCF_MASK)
/* RSER Bit Fields */
#define DSPI_RSER_DDIF_DIRS_MASK                 0x4000u
#define DSPI_RSER_DDIF_DIRS_SHIFT                14u
#define DSPI_RSER_DDIF_DIRS_WIDTH                1u
#define DSPI_RSER_DDIF_DIRS(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_DDIF_DIRS_SHIFT))&DSPI_RSER_DDIF_DIRS_MASK)
#define DSPI_RSER_CMDFFF_DIRS_MASK               0x8000u
#define DSPI_RSER_CMDFFF_DIRS_SHIFT              15u
#define DSPI_RSER_CMDFFF_DIRS_WIDTH              1u
#define DSPI_RSER_CMDFFF_DIRS(x)                 (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_CMDFFF_DIRS_SHIFT))&DSPI_RSER_CMDFFF_DIRS_MASK)
#define DSPI_RSER_RFDF_DIRS_MASK                 0x10000u
#define DSPI_RSER_RFDF_DIRS_SHIFT                16u
#define DSPI_RSER_RFDF_DIRS_WIDTH                1u
#define DSPI_RSER_RFDF_DIRS(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_RFDF_DIRS_SHIFT))&DSPI_RSER_RFDF_DIRS_MASK)
#define DSPI_RSER_RFDF_RE_MASK                   0x20000u
#define DSPI_RSER_RFDF_RE_SHIFT                  17u
#define DSPI_RSER_RFDF_RE_WIDTH                  1u
#define DSPI_RSER_RFDF_RE(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_RFDF_RE_SHIFT))&DSPI_RSER_RFDF_RE_MASK)
#define DSPI_RSER_TFIWF_RE_MASK                  0x40000u
#define DSPI_RSER_TFIWF_RE_SHIFT                 18u
#define DSPI_RSER_TFIWF_RE_WIDTH                 1u
#define DSPI_RSER_TFIWF_RE(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_TFIWF_RE_SHIFT))&DSPI_RSER_TFIWF_RE_MASK)
#define DSPI_RSER_RFOF_RE_MASK                   0x80000u
#define DSPI_RSER_RFOF_RE_SHIFT                  19u
#define DSPI_RSER_RFOF_RE_WIDTH                  1u
#define DSPI_RSER_RFOF_RE(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_RFOF_RE_SHIFT))&DSPI_RSER_RFOF_RE_MASK)
#define DSPI_RSER_DDIF_RE_MASK                   0x100000u
#define DSPI_RSER_DDIF_RE_SHIFT                  20u
#define DSPI_RSER_DDIF_RE_WIDTH                  1u
#define DSPI_RSER_DDIF_RE(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_DDIF_RE_SHIFT))&DSPI_RSER_DDIF_RE_MASK)
#define DSPI_RSER_SPEF_RE_MASK                   0x200000u
#define DSPI_RSER_SPEF_RE_SHIFT                  21u
#define DSPI_RSER_SPEF_RE_WIDTH                  1u
#define DSPI_RSER_SPEF_RE(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_SPEF_RE_SHIFT))&DSPI_RSER_SPEF_RE_MASK)
#define DSPI_RSER_DPEF_RE_MASK                   0x400000u
#define DSPI_RSER_DPEF_RE_SHIFT                  22u
#define DSPI_RSER_DPEF_RE_WIDTH                  1u
#define DSPI_RSER_DPEF_RE(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_DPEF_RE_SHIFT))&DSPI_RSER_DPEF_RE_MASK)
#define DSPI_RSER_CMDTCF_RE_MASK                 0x800000u
#define DSPI_RSER_CMDTCF_RE_SHIFT                23u
#define DSPI_RSER_CMDTCF_RE_WIDTH                1u
#define DSPI_RSER_CMDTCF_RE(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_CMDTCF_RE_SHIFT))&DSPI_RSER_CMDTCF_RE_MASK)
#define DSPI_RSER_TFFF_DIRS_MASK                 0x1000000u
#define DSPI_RSER_TFFF_DIRS_SHIFT                24u
#define DSPI_RSER_TFFF_DIRS_WIDTH                1u
#define DSPI_RSER_TFFF_DIRS(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_TFFF_DIRS_SHIFT))&DSPI_RSER_TFFF_DIRS_MASK)
#define DSPI_RSER_TFFF_RE_MASK                   0x2000000u
#define DSPI_RSER_TFFF_RE_SHIFT                  25u
#define DSPI_RSER_TFFF_RE_WIDTH                  1u
#define DSPI_RSER_TFFF_RE(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_TFFF_RE_SHIFT))&DSPI_RSER_TFFF_RE_MASK)
#define DSPI_RSER_DSITCF_RE_MASK                 0x4000000u
#define DSPI_RSER_DSITCF_RE_SHIFT                26u
#define DSPI_RSER_DSITCF_RE_WIDTH                1u
#define DSPI_RSER_DSITCF_RE(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_DSITCF_RE_SHIFT))&DSPI_RSER_DSITCF_RE_MASK)
#define DSPI_RSER_TFUF_RE_MASK                   0x8000000u
#define DSPI_RSER_TFUF_RE_SHIFT                  27u
#define DSPI_RSER_TFUF_RE_WIDTH                  1u
#define DSPI_RSER_TFUF_RE(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_TFUF_RE_SHIFT))&DSPI_RSER_TFUF_RE_MASK)
#define DSPI_RSER_EOQF_RE_MASK                   0x10000000u
#define DSPI_RSER_EOQF_RE_SHIFT                  28u
#define DSPI_RSER_EOQF_RE_WIDTH                  1u
#define DSPI_RSER_EOQF_RE(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_EOQF_RE_SHIFT))&DSPI_RSER_EOQF_RE_MASK)
#define DSPI_RSER_SPITCF_RE_MASK                 0x20000000u
#define DSPI_RSER_SPITCF_RE_SHIFT                29u
#define DSPI_RSER_SPITCF_RE_WIDTH                1u
#define DSPI_RSER_SPITCF_RE(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_SPITCF_RE_SHIFT))&DSPI_RSER_SPITCF_RE_MASK)
#define DSPI_RSER_CMDFFF_RE_MASK                 0x40000000u
#define DSPI_RSER_CMDFFF_RE_SHIFT                30u
#define DSPI_RSER_CMDFFF_RE_WIDTH                1u
#define DSPI_RSER_CMDFFF_RE(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_CMDFFF_RE_SHIFT))&DSPI_RSER_CMDFFF_RE_MASK)
#define DSPI_RSER_TCF_RE_MASK                    0x80000000u
#define DSPI_RSER_TCF_RE_SHIFT                   31u
#define DSPI_RSER_TCF_RE_WIDTH                   1u
#define DSPI_RSER_TCF_RE(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_RSER_TCF_RE_SHIFT))&DSPI_RSER_TCF_RE_MASK)
/* PUSHR_FIFO_CMD Bit Fields */
#define DSPI_PUSHR_FIFO_CMD_PUSHR_FIFO_CMD_MASK  0xFFFFu
#define DSPI_PUSHR_FIFO_CMD_PUSHR_FIFO_CMD_SHIFT 0u
#define DSPI_PUSHR_FIFO_CMD_PUSHR_FIFO_CMD_WIDTH 16u
#define DSPI_PUSHR_FIFO_CMD_PUSHR_FIFO_CMD(x)    (((uint16_t)(((uint16_t)(x))<<DSPI_PUSHR_FIFO_CMD_PUSHR_FIFO_CMD_SHIFT))&DSPI_PUSHR_FIFO_CMD_PUSHR_FIFO_CMD_MASK)
/* PUSHR_FIFO_TX Bit Fields */
#define DSPI_PUSHR_FIFO_TX_PUSHR_FIFO_TX_MASK    0xFFFFu
#define DSPI_PUSHR_FIFO_TX_PUSHR_FIFO_TX_SHIFT   0u
#define DSPI_PUSHR_FIFO_TX_PUSHR_FIFO_TX_WIDTH   16u
#define DSPI_PUSHR_FIFO_TX_PUSHR_FIFO_TX(x)      (((uint16_t)(((uint16_t)(x))<<DSPI_PUSHR_FIFO_TX_PUSHR_FIFO_TX_SHIFT))&DSPI_PUSHR_FIFO_TX_PUSHR_FIFO_TX_MASK)
/* PUSHR Bit Fields */
#define DSPI_PUSHR_TXDATA_MASK                   0xFFFFu
#define DSPI_PUSHR_TXDATA_SHIFT                  0u
#define DSPI_PUSHR_TXDATA_WIDTH                  16u
#define DSPI_PUSHR_TXDATA(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_PUSHR_TXDATA_SHIFT))&DSPI_PUSHR_TXDATA_MASK)
#define DSPI_PUSHR_PCS_MASK                      0x3F0000u
#define DSPI_PUSHR_PCS_SHIFT                     16u
#define DSPI_PUSHR_PCS_WIDTH                     6u
#define DSPI_PUSHR_PCS(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_PUSHR_PCS_SHIFT))&DSPI_PUSHR_PCS_MASK)
#define DSPI_PUSHR_PP_MCSC_MASK                  0x1000000u
#define DSPI_PUSHR_PP_MCSC_SHIFT                 24u
#define DSPI_PUSHR_PP_MCSC_WIDTH                 1u
#define DSPI_PUSHR_PP_MCSC(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_PUSHR_PP_MCSC_SHIFT))&DSPI_PUSHR_PP_MCSC_MASK)
#define DSPI_PUSHR_PE_MASC_MASK                  0x2000000u
#define DSPI_PUSHR_PE_MASC_SHIFT                 25u
#define DSPI_PUSHR_PE_MASC_WIDTH                 1u
#define DSPI_PUSHR_PE_MASC(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_PUSHR_PE_MASC_SHIFT))&DSPI_PUSHR_PE_MASC_MASK)
#define DSPI_PUSHR_CTCNT_MASK                    0x4000000u
#define DSPI_PUSHR_CTCNT_SHIFT                   26u
#define DSPI_PUSHR_CTCNT_WIDTH                   1u
#define DSPI_PUSHR_CTCNT(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PUSHR_CTCNT_SHIFT))&DSPI_PUSHR_CTCNT_MASK)
#define DSPI_PUSHR_EOQ_MASK                      0x8000000u
#define DSPI_PUSHR_EOQ_SHIFT                     27u
#define DSPI_PUSHR_EOQ_WIDTH                     1u
#define DSPI_PUSHR_EOQ(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_PUSHR_EOQ_SHIFT))&DSPI_PUSHR_EOQ_MASK)
#define DSPI_PUSHR_CTAS_MASK                     0x70000000u
#define DSPI_PUSHR_CTAS_SHIFT                    28u
#define DSPI_PUSHR_CTAS_WIDTH                    3u
#define DSPI_PUSHR_CTAS(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PUSHR_CTAS_SHIFT))&DSPI_PUSHR_CTAS_MASK)
#define DSPI_PUSHR_CONT_MASK                     0x80000000u
#define DSPI_PUSHR_CONT_SHIFT                    31u
#define DSPI_PUSHR_CONT_WIDTH                    1u
#define DSPI_PUSHR_CONT(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PUSHR_CONT_SHIFT))&DSPI_PUSHR_CONT_MASK)
/* PUSHR_SLAVE Bit Fields */
#define DSPI_PUSHR_SLAVE_TXDATA_MASK             0xFFFFu
#define DSPI_PUSHR_SLAVE_TXDATA_SHIFT            0u
#define DSPI_PUSHR_SLAVE_TXDATA_WIDTH            16u
#define DSPI_PUSHR_SLAVE_TXDATA(x)               (((uint32_t)(((uint32_t)(x))<<DSPI_PUSHR_SLAVE_TXDATA_SHIFT))&DSPI_PUSHR_SLAVE_TXDATA_MASK)
/* POPR Bit Fields */
#define DSPI_POPR_RXDATA_MASK                    0xFFFFFFFFu
#define DSPI_POPR_RXDATA_SHIFT                   0u
#define DSPI_POPR_RXDATA_WIDTH                   32u
#define DSPI_POPR_RXDATA(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_POPR_RXDATA_SHIFT))&DSPI_POPR_RXDATA_MASK)
/* TXFR Bit Fields */
#define DSPI_TXFR_TXDATA_MASK                    0xFFFFu
#define DSPI_TXFR_TXDATA_SHIFT                   0u
#define DSPI_TXFR_TXDATA_WIDTH                   16u
#define DSPI_TXFR_TXDATA(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TXFR_TXDATA_SHIFT))&DSPI_TXFR_TXDATA_MASK)
#define DSPI_TXFR_TXCMD_TXDATA_MASK              0xFFFF0000u
#define DSPI_TXFR_TXCMD_TXDATA_SHIFT             16u
#define DSPI_TXFR_TXCMD_TXDATA_WIDTH             16u
#define DSPI_TXFR_TXCMD_TXDATA(x)                (((uint32_t)(((uint32_t)(x))<<DSPI_TXFR_TXCMD_TXDATA_SHIFT))&DSPI_TXFR_TXCMD_TXDATA_MASK)
/* RXFR Bit Fields */
#define DSPI_RXFR_RXDATA_MASK                    0xFFFFFFFFu
#define DSPI_RXFR_RXDATA_SHIFT                   0u
#define DSPI_RXFR_RXDATA_WIDTH                   32u
#define DSPI_RXFR_RXDATA(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_RXFR_RXDATA_SHIFT))&DSPI_RXFR_RXDATA_MASK)
/* DSICR0 Bit Fields */
#define DSPI_DSICR0_DPCSx_MASK                   0xFFu
#define DSPI_DSICR0_DPCSx_SHIFT                  0u
#define DSPI_DSICR0_DPCSx_WIDTH                  8u
#define DSPI_DSICR0_DPCSx(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_DPCSx_SHIFT))&DSPI_DSICR0_DPCSx_MASK)
#define DSPI_DSICR0_PP_MASK                      0x100u
#define DSPI_DSICR0_PP_SHIFT                     8u
#define DSPI_DSICR0_PP_WIDTH                     1u
#define DSPI_DSICR0_PP(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_PP_SHIFT))&DSPI_DSICR0_PP_MASK)
#define DSPI_DSICR0_PE_MASK                      0x200u
#define DSPI_DSICR0_PE_SHIFT                     9u
#define DSPI_DSICR0_PE_WIDTH                     1u
#define DSPI_DSICR0_PE(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_PE_SHIFT))&DSPI_DSICR0_PE_MASK)
#define DSPI_DSICR0_PES_MASK                     0x400u
#define DSPI_DSICR0_PES_SHIFT                    10u
#define DSPI_DSICR0_PES_WIDTH                    1u
#define DSPI_DSICR0_PES(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_PES_SHIFT))&DSPI_DSICR0_PES_MASK)
#define DSPI_DSICR0_DMS_MASK                     0x800u
#define DSPI_DSICR0_DMS_SHIFT                    11u
#define DSPI_DSICR0_DMS_WIDTH                    1u
#define DSPI_DSICR0_DMS(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_DMS_SHIFT))&DSPI_DSICR0_DMS_MASK)
#define DSPI_DSICR0_DSICTAS_MASK                 0x7000u
#define DSPI_DSICR0_DSICTAS_SHIFT                12u
#define DSPI_DSICR0_DSICTAS_WIDTH                3u
#define DSPI_DSICR0_DSICTAS(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_DSICTAS_SHIFT))&DSPI_DSICR0_DSICTAS_MASK)
#define DSPI_DSICR0_DCONT_MASK                   0x8000u
#define DSPI_DSICR0_DCONT_SHIFT                  15u
#define DSPI_DSICR0_DCONT_WIDTH                  1u
#define DSPI_DSICR0_DCONT(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_DCONT_SHIFT))&DSPI_DSICR0_DCONT_MASK)
#define DSPI_DSICR0_CID_MASK                     0x10000u
#define DSPI_DSICR0_CID_SHIFT                    16u
#define DSPI_DSICR0_CID_WIDTH                    1u
#define DSPI_DSICR0_CID(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_CID_SHIFT))&DSPI_DSICR0_CID_MASK)
#define DSPI_DSICR0_TRRE_MASK                    0x20000u
#define DSPI_DSICR0_TRRE_SHIFT                   17u
#define DSPI_DSICR0_TRRE_WIDTH                   1u
#define DSPI_DSICR0_TRRE(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_TRRE_SHIFT))&DSPI_DSICR0_TRRE_MASK)
#define DSPI_DSICR0_TPOL_MASK                    0x40000u
#define DSPI_DSICR0_TPOL_SHIFT                   18u
#define DSPI_DSICR0_TPOL_WIDTH                   1u
#define DSPI_DSICR0_TPOL(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_TPOL_SHIFT))&DSPI_DSICR0_TPOL_MASK)
#define DSPI_DSICR0_TXSS_MASK                    0x80000u
#define DSPI_DSICR0_TXSS_SHIFT                   19u
#define DSPI_DSICR0_TXSS_WIDTH                   1u
#define DSPI_DSICR0_TXSS(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_TXSS_SHIFT))&DSPI_DSICR0_TXSS_MASK)
#define DSPI_DSICR0_TSBC_MASK                    0x100000u
#define DSPI_DSICR0_TSBC_SHIFT                   20u
#define DSPI_DSICR0_TSBC_WIDTH                   1u
#define DSPI_DSICR0_TSBC(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_TSBC_SHIFT))&DSPI_DSICR0_TSBC_MASK)
#define DSPI_DSICR0_ITSB_MASK                    0x200000u
#define DSPI_DSICR0_ITSB_SHIFT                   21u
#define DSPI_DSICR0_ITSB_WIDTH                   1u
#define DSPI_DSICR0_ITSB(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_ITSB_SHIFT))&DSPI_DSICR0_ITSB_MASK)
#define DSPI_DSICR0_FMSZ5_MASK                   0x800000u
#define DSPI_DSICR0_FMSZ5_SHIFT                  23u
#define DSPI_DSICR0_FMSZ5_WIDTH                  1u
#define DSPI_DSICR0_FMSZ5(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_FMSZ5_SHIFT))&DSPI_DSICR0_FMSZ5_MASK)
#define DSPI_DSICR0_MTOCNT_MASK                  0x3F000000u
#define DSPI_DSICR0_MTOCNT_SHIFT                 24u
#define DSPI_DSICR0_MTOCNT_WIDTH                 6u
#define DSPI_DSICR0_MTOCNT(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_MTOCNT_SHIFT))&DSPI_DSICR0_MTOCNT_MASK)
#define DSPI_DSICR0_FMSZ4_MASK                   0x40000000u
#define DSPI_DSICR0_FMSZ4_SHIFT                  30u
#define DSPI_DSICR0_FMSZ4_WIDTH                  1u
#define DSPI_DSICR0_FMSZ4(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_FMSZ4_SHIFT))&DSPI_DSICR0_FMSZ4_MASK)
#define DSPI_DSICR0_MTOE_MASK                    0x80000000u
#define DSPI_DSICR0_MTOE_SHIFT                   31u
#define DSPI_DSICR0_MTOE_WIDTH                   1u
#define DSPI_DSICR0_MTOE(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR0_MTOE_SHIFT))&DSPI_DSICR0_MTOE_MASK)
/* SDR0 Bit Fields */
#define DSPI_SDR0_SER_DATA_MASK                  0xFFFFFFFFu
#define DSPI_SDR0_SER_DATA_SHIFT                 0u
#define DSPI_SDR0_SER_DATA_WIDTH                 32u
#define DSPI_SDR0_SER_DATA(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_SDR0_SER_DATA_SHIFT))&DSPI_SDR0_SER_DATA_MASK)
/* ASDR0 Bit Fields */
#define DSPI_ASDR0_ASER_DATA_MASK                0xFFFFFFFFu
#define DSPI_ASDR0_ASER_DATA_SHIFT               0u
#define DSPI_ASDR0_ASER_DATA_WIDTH               32u
#define DSPI_ASDR0_ASER_DATA(x)                  (((uint32_t)(((uint32_t)(x))<<DSPI_ASDR0_ASER_DATA_SHIFT))&DSPI_ASDR0_ASER_DATA_MASK)
/* COMPR0 Bit Fields */
#define DSPI_COMPR0_COMP_DATA_MASK               0xFFFFFFFFu
#define DSPI_COMPR0_COMP_DATA_SHIFT              0u
#define DSPI_COMPR0_COMP_DATA_WIDTH              32u
#define DSPI_COMPR0_COMP_DATA(x)                 (((uint32_t)(((uint32_t)(x))<<DSPI_COMPR0_COMP_DATA_SHIFT))&DSPI_COMPR0_COMP_DATA_MASK)
/* DDR0 Bit Fields */
#define DSPI_DDR0_DESER_DATA_MASK                0xFFFFFFFFu
#define DSPI_DDR0_DESER_DATA_SHIFT               0u
#define DSPI_DDR0_DESER_DATA_WIDTH               32u
#define DSPI_DDR0_DESER_DATA(x)                  (((uint32_t)(((uint32_t)(x))<<DSPI_DDR0_DESER_DATA_SHIFT))&DSPI_DDR0_DESER_DATA_MASK)
/* DSICR1 Bit Fields */
#define DSPI_DSICR1_DPCS1_x_MASK                 0xFFu
#define DSPI_DSICR1_DPCS1_x_SHIFT                0u
#define DSPI_DSICR1_DPCS1_x_WIDTH                8u
#define DSPI_DSICR1_DPCS1_x(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR1_DPCS1_x_SHIFT))&DSPI_DSICR1_DPCS1_x_MASK)
#define DSPI_DSICR1_DSE0_MASK                    0x10000u
#define DSPI_DSICR1_DSE0_SHIFT                   16u
#define DSPI_DSICR1_DSE0_WIDTH                   1u
#define DSPI_DSICR1_DSE0(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR1_DSE0_SHIFT))&DSPI_DSICR1_DSE0_MASK)
#define DSPI_DSICR1_DSE1_MASK                    0x20000u
#define DSPI_DSICR1_DSE1_SHIFT                   17u
#define DSPI_DSICR1_DSE1_WIDTH                   1u
#define DSPI_DSICR1_DSE1(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR1_DSE1_SHIFT))&DSPI_DSICR1_DSE1_MASK)
#define DSPI_DSICR1_DSI64E_MASK                  0x40000u
#define DSPI_DSICR1_DSI64E_SHIFT                 18u
#define DSPI_DSICR1_DSI64E_WIDTH                 1u
#define DSPI_DSICR1_DSI64E(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR1_DSI64E_SHIFT))&DSPI_DSICR1_DSI64E_MASK)
#define DSPI_DSICR1_CSE_MASK                     0x80000u
#define DSPI_DSICR1_CSE_SHIFT                    19u
#define DSPI_DSICR1_CSE_WIDTH                    1u
#define DSPI_DSICR1_CSE(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR1_CSE_SHIFT))&DSPI_DSICR1_CSE_MASK)
#define DSPI_DSICR1_CSI_PRTY_MASK                0x100000u
#define DSPI_DSICR1_CSI_PRTY_SHIFT               20u
#define DSPI_DSICR1_CSI_PRTY_WIDTH               1u
#define DSPI_DSICR1_CSI_PRTY(x)                  (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR1_CSI_PRTY_SHIFT))&DSPI_DSICR1_CSI_PRTY_MASK)
#define DSPI_DSICR1_TSBCNT_MASK                  0x3F000000u
#define DSPI_DSICR1_TSBCNT_SHIFT                 24u
#define DSPI_DSICR1_TSBCNT_WIDTH                 6u
#define DSPI_DSICR1_TSBCNT(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_DSICR1_TSBCNT_SHIFT))&DSPI_DSICR1_TSBCNT_MASK)
/* SSR0 Bit Fields */
#define DSPI_SSR0_SS_MASK                        0xFFFFFFFFu
#define DSPI_SSR0_SS_SHIFT                       0u
#define DSPI_SSR0_SS_WIDTH                       32u
#define DSPI_SSR0_SS(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SSR0_SS_SHIFT))&DSPI_SSR0_SS_MASK)
/* PISR0 Bit Fields */
#define DSPI_PISR0_IPS0_MASK                     0xFu
#define DSPI_PISR0_IPS0_SHIFT                    0u
#define DSPI_PISR0_IPS0_WIDTH                    4u
#define DSPI_PISR0_IPS0(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR0_IPS0_SHIFT))&DSPI_PISR0_IPS0_MASK)
#define DSPI_PISR0_IPS1_MASK                     0xF0u
#define DSPI_PISR0_IPS1_SHIFT                    4u
#define DSPI_PISR0_IPS1_WIDTH                    4u
#define DSPI_PISR0_IPS1(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR0_IPS1_SHIFT))&DSPI_PISR0_IPS1_MASK)
#define DSPI_PISR0_IPS2_MASK                     0xF00u
#define DSPI_PISR0_IPS2_SHIFT                    8u
#define DSPI_PISR0_IPS2_WIDTH                    4u
#define DSPI_PISR0_IPS2(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR0_IPS2_SHIFT))&DSPI_PISR0_IPS2_MASK)
#define DSPI_PISR0_IPS3_MASK                     0xF000u
#define DSPI_PISR0_IPS3_SHIFT                    12u
#define DSPI_PISR0_IPS3_WIDTH                    4u
#define DSPI_PISR0_IPS3(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR0_IPS3_SHIFT))&DSPI_PISR0_IPS3_MASK)
#define DSPI_PISR0_IPS4_MASK                     0xF0000u
#define DSPI_PISR0_IPS4_SHIFT                    16u
#define DSPI_PISR0_IPS4_WIDTH                    4u
#define DSPI_PISR0_IPS4(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR0_IPS4_SHIFT))&DSPI_PISR0_IPS4_MASK)
#define DSPI_PISR0_IPS5_MASK                     0xF00000u
#define DSPI_PISR0_IPS5_SHIFT                    20u
#define DSPI_PISR0_IPS5_WIDTH                    4u
#define DSPI_PISR0_IPS5(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR0_IPS5_SHIFT))&DSPI_PISR0_IPS5_MASK)
#define DSPI_PISR0_IPS6_MASK                     0xF000000u
#define DSPI_PISR0_IPS6_SHIFT                    24u
#define DSPI_PISR0_IPS6_WIDTH                    4u
#define DSPI_PISR0_IPS6(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR0_IPS6_SHIFT))&DSPI_PISR0_IPS6_MASK)
#define DSPI_PISR0_IPS7_MASK                     0xF0000000u
#define DSPI_PISR0_IPS7_SHIFT                    28u
#define DSPI_PISR0_IPS7_WIDTH                    4u
#define DSPI_PISR0_IPS7(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR0_IPS7_SHIFT))&DSPI_PISR0_IPS7_MASK)
/* PISR1 Bit Fields */
#define DSPI_PISR1_IPS8_MASK                     0xFu
#define DSPI_PISR1_IPS8_SHIFT                    0u
#define DSPI_PISR1_IPS8_WIDTH                    4u
#define DSPI_PISR1_IPS8(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR1_IPS8_SHIFT))&DSPI_PISR1_IPS8_MASK)
#define DSPI_PISR1_IPS9_MASK                     0xF0u
#define DSPI_PISR1_IPS9_SHIFT                    4u
#define DSPI_PISR1_IPS9_WIDTH                    4u
#define DSPI_PISR1_IPS9(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_PISR1_IPS9_SHIFT))&DSPI_PISR1_IPS9_MASK)
#define DSPI_PISR1_IPS10_MASK                    0xF00u
#define DSPI_PISR1_IPS10_SHIFT                   8u
#define DSPI_PISR1_IPS10_WIDTH                   4u
#define DSPI_PISR1_IPS10(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR1_IPS10_SHIFT))&DSPI_PISR1_IPS10_MASK)
#define DSPI_PISR1_IPS11_MASK                    0xF000u
#define DSPI_PISR1_IPS11_SHIFT                   12u
#define DSPI_PISR1_IPS11_WIDTH                   4u
#define DSPI_PISR1_IPS11(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR1_IPS11_SHIFT))&DSPI_PISR1_IPS11_MASK)
#define DSPI_PISR1_IPS12_MASK                    0xF0000u
#define DSPI_PISR1_IPS12_SHIFT                   16u
#define DSPI_PISR1_IPS12_WIDTH                   4u
#define DSPI_PISR1_IPS12(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR1_IPS12_SHIFT))&DSPI_PISR1_IPS12_MASK)
#define DSPI_PISR1_IPS13_MASK                    0xF00000u
#define DSPI_PISR1_IPS13_SHIFT                   20u
#define DSPI_PISR1_IPS13_WIDTH                   4u
#define DSPI_PISR1_IPS13(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR1_IPS13_SHIFT))&DSPI_PISR1_IPS13_MASK)
#define DSPI_PISR1_IPS14_MASK                    0xF000000u
#define DSPI_PISR1_IPS14_SHIFT                   24u
#define DSPI_PISR1_IPS14_WIDTH                   4u
#define DSPI_PISR1_IPS14(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR1_IPS14_SHIFT))&DSPI_PISR1_IPS14_MASK)
#define DSPI_PISR1_IPS15_MASK                    0xF0000000u
#define DSPI_PISR1_IPS15_SHIFT                   28u
#define DSPI_PISR1_IPS15_WIDTH                   4u
#define DSPI_PISR1_IPS15(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR1_IPS15_SHIFT))&DSPI_PISR1_IPS15_MASK)
/* PISR2 Bit Fields */
#define DSPI_PISR2_IPS16_MASK                    0xFu
#define DSPI_PISR2_IPS16_SHIFT                   0u
#define DSPI_PISR2_IPS16_WIDTH                   4u
#define DSPI_PISR2_IPS16(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR2_IPS16_SHIFT))&DSPI_PISR2_IPS16_MASK)
#define DSPI_PISR2_IPS17_MASK                    0xF0u
#define DSPI_PISR2_IPS17_SHIFT                   4u
#define DSPI_PISR2_IPS17_WIDTH                   4u
#define DSPI_PISR2_IPS17(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR2_IPS17_SHIFT))&DSPI_PISR2_IPS17_MASK)
#define DSPI_PISR2_IPS18_MASK                    0xF00u
#define DSPI_PISR2_IPS18_SHIFT                   8u
#define DSPI_PISR2_IPS18_WIDTH                   4u
#define DSPI_PISR2_IPS18(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR2_IPS18_SHIFT))&DSPI_PISR2_IPS18_MASK)
#define DSPI_PISR2_IPS19_MASK                    0xF000u
#define DSPI_PISR2_IPS19_SHIFT                   12u
#define DSPI_PISR2_IPS19_WIDTH                   4u
#define DSPI_PISR2_IPS19(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR2_IPS19_SHIFT))&DSPI_PISR2_IPS19_MASK)
#define DSPI_PISR2_IPS20_MASK                    0xF0000u
#define DSPI_PISR2_IPS20_SHIFT                   16u
#define DSPI_PISR2_IPS20_WIDTH                   4u
#define DSPI_PISR2_IPS20(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR2_IPS20_SHIFT))&DSPI_PISR2_IPS20_MASK)
#define DSPI_PISR2_IPS21_MASK                    0xF00000u
#define DSPI_PISR2_IPS21_SHIFT                   20u
#define DSPI_PISR2_IPS21_WIDTH                   4u
#define DSPI_PISR2_IPS21(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR2_IPS21_SHIFT))&DSPI_PISR2_IPS21_MASK)
#define DSPI_PISR2_IPS22_MASK                    0xF000000u
#define DSPI_PISR2_IPS22_SHIFT                   24u
#define DSPI_PISR2_IPS22_WIDTH                   4u
#define DSPI_PISR2_IPS22(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR2_IPS22_SHIFT))&DSPI_PISR2_IPS22_MASK)
#define DSPI_PISR2_IPS23_MASK                    0xF0000000u
#define DSPI_PISR2_IPS23_SHIFT                   28u
#define DSPI_PISR2_IPS23_WIDTH                   4u
#define DSPI_PISR2_IPS23(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR2_IPS23_SHIFT))&DSPI_PISR2_IPS23_MASK)
/* PISR3 Bit Fields */
#define DSPI_PISR3_IPS24_MASK                    0xFu
#define DSPI_PISR3_IPS24_SHIFT                   0u
#define DSPI_PISR3_IPS24_WIDTH                   4u
#define DSPI_PISR3_IPS24(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR3_IPS24_SHIFT))&DSPI_PISR3_IPS24_MASK)
#define DSPI_PISR3_IPS25_MASK                    0xF0u
#define DSPI_PISR3_IPS25_SHIFT                   4u
#define DSPI_PISR3_IPS25_WIDTH                   4u
#define DSPI_PISR3_IPS25(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR3_IPS25_SHIFT))&DSPI_PISR3_IPS25_MASK)
#define DSPI_PISR3_IPS26_MASK                    0xF00u
#define DSPI_PISR3_IPS26_SHIFT                   8u
#define DSPI_PISR3_IPS26_WIDTH                   4u
#define DSPI_PISR3_IPS26(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR3_IPS26_SHIFT))&DSPI_PISR3_IPS26_MASK)
#define DSPI_PISR3_IPS27_MASK                    0xF000u
#define DSPI_PISR3_IPS27_SHIFT                   12u
#define DSPI_PISR3_IPS27_WIDTH                   4u
#define DSPI_PISR3_IPS27(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR3_IPS27_SHIFT))&DSPI_PISR3_IPS27_MASK)
#define DSPI_PISR3_IPS28_MASK                    0xF0000u
#define DSPI_PISR3_IPS28_SHIFT                   16u
#define DSPI_PISR3_IPS28_WIDTH                   4u
#define DSPI_PISR3_IPS28(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR3_IPS28_SHIFT))&DSPI_PISR3_IPS28_MASK)
#define DSPI_PISR3_IPS29_MASK                    0xF00000u
#define DSPI_PISR3_IPS29_SHIFT                   20u
#define DSPI_PISR3_IPS29_WIDTH                   4u
#define DSPI_PISR3_IPS29(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR3_IPS29_SHIFT))&DSPI_PISR3_IPS29_MASK)
#define DSPI_PISR3_IPS30_MASK                    0xF000000u
#define DSPI_PISR3_IPS30_SHIFT                   24u
#define DSPI_PISR3_IPS30_WIDTH                   4u
#define DSPI_PISR3_IPS30(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR3_IPS30_SHIFT))&DSPI_PISR3_IPS30_MASK)
#define DSPI_PISR3_IPS31_MASK                    0xF0000000u
#define DSPI_PISR3_IPS31_SHIFT                   28u
#define DSPI_PISR3_IPS31_WIDTH                   4u
#define DSPI_PISR3_IPS31(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR3_IPS31_SHIFT))&DSPI_PISR3_IPS31_MASK)
/* DIMR0 Bit Fields */
#define DSPI_DIMR0_MASK_MASK                     0xFFFFFFFFu
#define DSPI_DIMR0_MASK_SHIFT                    0u
#define DSPI_DIMR0_MASK_WIDTH                    32u
#define DSPI_DIMR0_MASK(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_DIMR0_MASK_SHIFT))&DSPI_DIMR0_MASK_MASK)
/* DPIR0 Bit Fields */
#define DSPI_DPIR0_DP_MASK                       0xFFFFFFFFu
#define DSPI_DPIR0_DP_SHIFT                      0u
#define DSPI_DPIR0_DP_WIDTH                      32u
#define DSPI_DPIR0_DP(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_DPIR0_DP_SHIFT))&DSPI_DPIR0_DP_MASK)
/* SDR1 Bit Fields */
#define DSPI_SDR1_SER_DATA_MASK                  0xFFFFFFFFu
#define DSPI_SDR1_SER_DATA_SHIFT                 0u
#define DSPI_SDR1_SER_DATA_WIDTH                 32u
#define DSPI_SDR1_SER_DATA(x)                    (((uint32_t)(((uint32_t)(x))<<DSPI_SDR1_SER_DATA_SHIFT))&DSPI_SDR1_SER_DATA_MASK)
/* ASDR1 Bit Fields */
#define DSPI_ASDR1_ASER_DATA_MASK                0xFFFFFFFFu
#define DSPI_ASDR1_ASER_DATA_SHIFT               0u
#define DSPI_ASDR1_ASER_DATA_WIDTH               32u
#define DSPI_ASDR1_ASER_DATA(x)                  (((uint32_t)(((uint32_t)(x))<<DSPI_ASDR1_ASER_DATA_SHIFT))&DSPI_ASDR1_ASER_DATA_MASK)
/* COMPR1 Bit Fields */
#define DSPI_COMPR1_COMP_DATA_MASK               0xFFFFFFFFu
#define DSPI_COMPR1_COMP_DATA_SHIFT              0u
#define DSPI_COMPR1_COMP_DATA_WIDTH              32u
#define DSPI_COMPR1_COMP_DATA(x)                 (((uint32_t)(((uint32_t)(x))<<DSPI_COMPR1_COMP_DATA_SHIFT))&DSPI_COMPR1_COMP_DATA_MASK)
/* DDR1 Bit Fields */
#define DSPI_DDR1_DESER_DATA_MASK                0xFFFFFFFFu
#define DSPI_DDR1_DESER_DATA_SHIFT               0u
#define DSPI_DDR1_DESER_DATA_WIDTH               32u
#define DSPI_DDR1_DESER_DATA(x)                  (((uint32_t)(((uint32_t)(x))<<DSPI_DDR1_DESER_DATA_SHIFT))&DSPI_DDR1_DESER_DATA_MASK)
/* SSR1 Bit Fields */
#define DSPI_SSR1_SS_MASK                        0xFFFFFFFFu
#define DSPI_SSR1_SS_SHIFT                       0u
#define DSPI_SSR1_SS_WIDTH                       32u
#define DSPI_SSR1_SS(x)                          (((uint32_t)(((uint32_t)(x))<<DSPI_SSR1_SS_SHIFT))&DSPI_SSR1_SS_MASK)
/* PISR4 Bit Fields */
#define DSPI_PISR4_IPS32_MASK                    0xFu
#define DSPI_PISR4_IPS32_SHIFT                   0u
#define DSPI_PISR4_IPS32_WIDTH                   4u
#define DSPI_PISR4_IPS32(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR4_IPS32_SHIFT))&DSPI_PISR4_IPS32_MASK)
#define DSPI_PISR4_IPS33_MASK                    0xF0u
#define DSPI_PISR4_IPS33_SHIFT                   4u
#define DSPI_PISR4_IPS33_WIDTH                   4u
#define DSPI_PISR4_IPS33(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR4_IPS33_SHIFT))&DSPI_PISR4_IPS33_MASK)
#define DSPI_PISR4_IPS34_MASK                    0xF00u
#define DSPI_PISR4_IPS34_SHIFT                   8u
#define DSPI_PISR4_IPS34_WIDTH                   4u
#define DSPI_PISR4_IPS34(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR4_IPS34_SHIFT))&DSPI_PISR4_IPS34_MASK)
#define DSPI_PISR4_IPS35_MASK                    0xF000u
#define DSPI_PISR4_IPS35_SHIFT                   12u
#define DSPI_PISR4_IPS35_WIDTH                   4u
#define DSPI_PISR4_IPS35(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR4_IPS35_SHIFT))&DSPI_PISR4_IPS35_MASK)
#define DSPI_PISR4_IPS36_MASK                    0xF0000u
#define DSPI_PISR4_IPS36_SHIFT                   16u
#define DSPI_PISR4_IPS36_WIDTH                   4u
#define DSPI_PISR4_IPS36(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR4_IPS36_SHIFT))&DSPI_PISR4_IPS36_MASK)
#define DSPI_PISR4_IPS37_MASK                    0xF00000u
#define DSPI_PISR4_IPS37_SHIFT                   20u
#define DSPI_PISR4_IPS37_WIDTH                   4u
#define DSPI_PISR4_IPS37(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR4_IPS37_SHIFT))&DSPI_PISR4_IPS37_MASK)
#define DSPI_PISR4_IPS38_MASK                    0xF000000u
#define DSPI_PISR4_IPS38_SHIFT                   24u
#define DSPI_PISR4_IPS38_WIDTH                   4u
#define DSPI_PISR4_IPS38(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR4_IPS38_SHIFT))&DSPI_PISR4_IPS38_MASK)
#define DSPI_PISR4_IPS39_MASK                    0xF0000000u
#define DSPI_PISR4_IPS39_SHIFT                   28u
#define DSPI_PISR4_IPS39_WIDTH                   4u
#define DSPI_PISR4_IPS39(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR4_IPS39_SHIFT))&DSPI_PISR4_IPS39_MASK)
/* PISR5 Bit Fields */
#define DSPI_PISR5_IPS40_MASK                    0xFu
#define DSPI_PISR5_IPS40_SHIFT                   0u
#define DSPI_PISR5_IPS40_WIDTH                   4u
#define DSPI_PISR5_IPS40(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR5_IPS40_SHIFT))&DSPI_PISR5_IPS40_MASK)
#define DSPI_PISR5_IPS41_MASK                    0xF0u
#define DSPI_PISR5_IPS41_SHIFT                   4u
#define DSPI_PISR5_IPS41_WIDTH                   4u
#define DSPI_PISR5_IPS41(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR5_IPS41_SHIFT))&DSPI_PISR5_IPS41_MASK)
#define DSPI_PISR5_IPS42_MASK                    0xF00u
#define DSPI_PISR5_IPS42_SHIFT                   8u
#define DSPI_PISR5_IPS42_WIDTH                   4u
#define DSPI_PISR5_IPS42(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR5_IPS42_SHIFT))&DSPI_PISR5_IPS42_MASK)
#define DSPI_PISR5_IPS43_MASK                    0xF000u
#define DSPI_PISR5_IPS43_SHIFT                   12u
#define DSPI_PISR5_IPS43_WIDTH                   4u
#define DSPI_PISR5_IPS43(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR5_IPS43_SHIFT))&DSPI_PISR5_IPS43_MASK)
#define DSPI_PISR5_IPS44_MASK                    0xF0000u
#define DSPI_PISR5_IPS44_SHIFT                   16u
#define DSPI_PISR5_IPS44_WIDTH                   4u
#define DSPI_PISR5_IPS44(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR5_IPS44_SHIFT))&DSPI_PISR5_IPS44_MASK)
#define DSPI_PISR5_IPS45_MASK                    0xF00000u
#define DSPI_PISR5_IPS45_SHIFT                   20u
#define DSPI_PISR5_IPS45_WIDTH                   4u
#define DSPI_PISR5_IPS45(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR5_IPS45_SHIFT))&DSPI_PISR5_IPS45_MASK)
#define DSPI_PISR5_IPS46_MASK                    0xF000000u
#define DSPI_PISR5_IPS46_SHIFT                   24u
#define DSPI_PISR5_IPS46_WIDTH                   4u
#define DSPI_PISR5_IPS46(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR5_IPS46_SHIFT))&DSPI_PISR5_IPS46_MASK)
#define DSPI_PISR5_IPS47_MASK                    0xF0000000u
#define DSPI_PISR5_IPS47_SHIFT                   28u
#define DSPI_PISR5_IPS47_WIDTH                   4u
#define DSPI_PISR5_IPS47(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR5_IPS47_SHIFT))&DSPI_PISR5_IPS47_MASK)
/* PISR6 Bit Fields */
#define DSPI_PISR6_IPS48_MASK                    0xFu
#define DSPI_PISR6_IPS48_SHIFT                   0u
#define DSPI_PISR6_IPS48_WIDTH                   4u
#define DSPI_PISR6_IPS48(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR6_IPS48_SHIFT))&DSPI_PISR6_IPS48_MASK)
#define DSPI_PISR6_IPS49_MASK                    0xF0u
#define DSPI_PISR6_IPS49_SHIFT                   4u
#define DSPI_PISR6_IPS49_WIDTH                   4u
#define DSPI_PISR6_IPS49(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR6_IPS49_SHIFT))&DSPI_PISR6_IPS49_MASK)
#define DSPI_PISR6_IPS50_MASK                    0xF00u
#define DSPI_PISR6_IPS50_SHIFT                   8u
#define DSPI_PISR6_IPS50_WIDTH                   4u
#define DSPI_PISR6_IPS50(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR6_IPS50_SHIFT))&DSPI_PISR6_IPS50_MASK)
#define DSPI_PISR6_IPS51_MASK                    0xF000u
#define DSPI_PISR6_IPS51_SHIFT                   12u
#define DSPI_PISR6_IPS51_WIDTH                   4u
#define DSPI_PISR6_IPS51(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR6_IPS51_SHIFT))&DSPI_PISR6_IPS51_MASK)
#define DSPI_PISR6_IPS52_MASK                    0xF0000u
#define DSPI_PISR6_IPS52_SHIFT                   16u
#define DSPI_PISR6_IPS52_WIDTH                   4u
#define DSPI_PISR6_IPS52(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR6_IPS52_SHIFT))&DSPI_PISR6_IPS52_MASK)
#define DSPI_PISR6_IPS53_MASK                    0xF00000u
#define DSPI_PISR6_IPS53_SHIFT                   20u
#define DSPI_PISR6_IPS53_WIDTH                   4u
#define DSPI_PISR6_IPS53(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR6_IPS53_SHIFT))&DSPI_PISR6_IPS53_MASK)
#define DSPI_PISR6_IPS54_MASK                    0xF000000u
#define DSPI_PISR6_IPS54_SHIFT                   24u
#define DSPI_PISR6_IPS54_WIDTH                   4u
#define DSPI_PISR6_IPS54(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR6_IPS54_SHIFT))&DSPI_PISR6_IPS54_MASK)
#define DSPI_PISR6_IPS55_MASK                    0xF0000000u
#define DSPI_PISR6_IPS55_SHIFT                   28u
#define DSPI_PISR6_IPS55_WIDTH                   4u
#define DSPI_PISR6_IPS55(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR6_IPS55_SHIFT))&DSPI_PISR6_IPS55_MASK)
/* PISR7 Bit Fields */
#define DSPI_PISR7_IPS56_MASK                    0xFu
#define DSPI_PISR7_IPS56_SHIFT                   0u
#define DSPI_PISR7_IPS56_WIDTH                   4u
#define DSPI_PISR7_IPS56(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR7_IPS56_SHIFT))&DSPI_PISR7_IPS56_MASK)
#define DSPI_PISR7_IPS57_MASK                    0xF0u
#define DSPI_PISR7_IPS57_SHIFT                   4u
#define DSPI_PISR7_IPS57_WIDTH                   4u
#define DSPI_PISR7_IPS57(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR7_IPS57_SHIFT))&DSPI_PISR7_IPS57_MASK)
#define DSPI_PISR7_IPS58_MASK                    0xF00u
#define DSPI_PISR7_IPS58_SHIFT                   8u
#define DSPI_PISR7_IPS58_WIDTH                   4u
#define DSPI_PISR7_IPS58(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR7_IPS58_SHIFT))&DSPI_PISR7_IPS58_MASK)
#define DSPI_PISR7_IPS59_MASK                    0xF000u
#define DSPI_PISR7_IPS59_SHIFT                   12u
#define DSPI_PISR7_IPS59_WIDTH                   4u
#define DSPI_PISR7_IPS59(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR7_IPS59_SHIFT))&DSPI_PISR7_IPS59_MASK)
#define DSPI_PISR7_IPS60_MASK                    0xF0000u
#define DSPI_PISR7_IPS60_SHIFT                   16u
#define DSPI_PISR7_IPS60_WIDTH                   4u
#define DSPI_PISR7_IPS60(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR7_IPS60_SHIFT))&DSPI_PISR7_IPS60_MASK)
#define DSPI_PISR7_IPS61_MASK                    0xF00000u
#define DSPI_PISR7_IPS61_SHIFT                   20u
#define DSPI_PISR7_IPS61_WIDTH                   4u
#define DSPI_PISR7_IPS61(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR7_IPS61_SHIFT))&DSPI_PISR7_IPS61_MASK)
#define DSPI_PISR7_IPS62_MASK                    0xF000000u
#define DSPI_PISR7_IPS62_SHIFT                   24u
#define DSPI_PISR7_IPS62_WIDTH                   4u
#define DSPI_PISR7_IPS62(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR7_IPS62_SHIFT))&DSPI_PISR7_IPS62_MASK)
#define DSPI_PISR7_IPS63_MASK                    0xF0000000u
#define DSPI_PISR7_IPS63_SHIFT                   28u
#define DSPI_PISR7_IPS63_WIDTH                   4u
#define DSPI_PISR7_IPS63(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_PISR7_IPS63_SHIFT))&DSPI_PISR7_IPS63_MASK)
/* DIMR1 Bit Fields */
#define DSPI_DIMR1_MASK_MASK                     0xFFFFFFFFu
#define DSPI_DIMR1_MASK_SHIFT                    0u
#define DSPI_DIMR1_MASK_WIDTH                    32u
#define DSPI_DIMR1_MASK(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_DIMR1_MASK_SHIFT))&DSPI_DIMR1_MASK_MASK)
/* DPIR1 Bit Fields */
#define DSPI_DPIR1_DP_MASK                       0xFFFFFFFFu
#define DSPI_DPIR1_DP_SHIFT                      0u
#define DSPI_DPIR1_DP_WIDTH                      32u
#define DSPI_DPIR1_DP(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_DPIR1_DP_SHIFT))&DSPI_DPIR1_DP_MASK)
/* CTARE Bit Fields */
#define DSPI_CTARE_DTCP_MASK                     0x7FFu
#define DSPI_CTARE_DTCP_SHIFT                    0u
#define DSPI_CTARE_DTCP_WIDTH                    11u
#define DSPI_CTARE_DTCP(x)                       (((uint32_t)(((uint32_t)(x))<<DSPI_CTARE_DTCP_SHIFT))&DSPI_CTARE_DTCP_MASK)
#define DSPI_CTARE_FMSZE_MASK                    0x10000u
#define DSPI_CTARE_FMSZE_SHIFT                   16u
#define DSPI_CTARE_FMSZE_WIDTH                   1u
#define DSPI_CTARE_FMSZE(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_CTARE_FMSZE_SHIFT))&DSPI_CTARE_FMSZE_MASK)
/* SREX Bit Fields */
#define DSPI_SREX_CMDNXTPTR_MASK                 0xFu
#define DSPI_SREX_CMDNXTPTR_SHIFT                0u
#define DSPI_SREX_CMDNXTPTR_WIDTH                4u
#define DSPI_SREX_CMDNXTPTR(x)                   (((uint32_t)(((uint32_t)(x))<<DSPI_SREX_CMDNXTPTR_SHIFT))&DSPI_SREX_CMDNXTPTR_MASK)
#define DSPI_SREX_CMDCTR_MASK                    0x1F0u
#define DSPI_SREX_CMDCTR_SHIFT                   4u
#define DSPI_SREX_CMDCTR_WIDTH                   5u
#define DSPI_SREX_CMDCTR(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_SREX_CMDCTR_SHIFT))&DSPI_SREX_CMDCTR_MASK)
#define DSPI_SREX_TXCTR4_MASK                    0x800u
#define DSPI_SREX_TXCTR4_SHIFT                   11u
#define DSPI_SREX_TXCTR4_WIDTH                   1u
#define DSPI_SREX_TXCTR4(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_SREX_TXCTR4_SHIFT))&DSPI_SREX_TXCTR4_MASK)
#define DSPI_SREX_RXCTR4_MASK                    0x4000u
#define DSPI_SREX_RXCTR4_SHIFT                   14u
#define DSPI_SREX_RXCTR4_WIDTH                   1u
#define DSPI_SREX_RXCTR4(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_SREX_RXCTR4_SHIFT))&DSPI_SREX_RXCTR4_MASK)
/* TRIG Bit Fields */
#define DSPI_TRIG_TRRE_MASK                      0x1u
#define DSPI_TRIG_TRRE_SHIFT                     0u
#define DSPI_TRIG_TRRE_WIDTH                     1u
#define DSPI_TRIG_TRRE(x)                        (((uint32_t)(((uint32_t)(x))<<DSPI_TRIG_TRRE_SHIFT))&DSPI_TRIG_TRRE_MASK)
#define DSPI_TRIG_CID_MASK                       0x2u
#define DSPI_TRIG_CID_SHIFT                      1u
#define DSPI_TRIG_CID_WIDTH                      1u
#define DSPI_TRIG_CID(x)                         (((uint32_t)(((uint32_t)(x))<<DSPI_TRIG_CID_SHIFT))&DSPI_TRIG_CID_MASK)
#define DSPI_TRIG_ASDR_WT_MASK                   0x4u
#define DSPI_TRIG_ASDR_WT_SHIFT                  2u
#define DSPI_TRIG_ASDR_WT_WIDTH                  1u
#define DSPI_TRIG_ASDR_WT(x)                     (((uint32_t)(((uint32_t)(x))<<DSPI_TRIG_ASDR_WT_SHIFT))&DSPI_TRIG_ASDR_WT_MASK)
#define DSPI_TRIG_ENABLE_MASK                    0x80000000u
#define DSPI_TRIG_ENABLE_SHIFT                   31u
#define DSPI_TRIG_ENABLE_WIDTH                   1u
#define DSPI_TRIG_ENABLE(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TRIG_ENABLE_SHIFT))&DSPI_TRIG_ENABLE_MASK)
/* TSL Bit Fields */
#define DSPI_TSL_TS0_LEN_MASK                    0x7Fu
#define DSPI_TSL_TS0_LEN_SHIFT                   0u
#define DSPI_TSL_TS0_LEN_WIDTH                   7u
#define DSPI_TSL_TS0_LEN(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TSL_TS0_LEN_SHIFT))&DSPI_TSL_TS0_LEN_MASK)
#define DSPI_TSL_TS1_LEN_MASK                    0x7F00u
#define DSPI_TSL_TS1_LEN_SHIFT                   8u
#define DSPI_TSL_TS1_LEN_WIDTH                   7u
#define DSPI_TSL_TS1_LEN(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TSL_TS1_LEN_SHIFT))&DSPI_TSL_TS1_LEN_MASK)
#define DSPI_TSL_TS2_LEN_MASK                    0x7F0000u
#define DSPI_TSL_TS2_LEN_SHIFT                   16u
#define DSPI_TSL_TS2_LEN_WIDTH                   7u
#define DSPI_TSL_TS2_LEN(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TSL_TS2_LEN_SHIFT))&DSPI_TSL_TS2_LEN_MASK)
#define DSPI_TSL_TS3_LEN_MASK                    0x7F000000u
#define DSPI_TSL_TS3_LEN_SHIFT                   24u
#define DSPI_TSL_TS3_LEN_WIDTH                   7u
#define DSPI_TSL_TS3_LEN(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TSL_TS3_LEN_SHIFT))&DSPI_TSL_TS3_LEN_MASK)
/* TS_CONF Bit Fields */
#define DSPI_TS_CONF_TS0_MASK                    0xFu
#define DSPI_TS_CONF_TS0_SHIFT                   0u
#define DSPI_TS_CONF_TS0_WIDTH                   4u
#define DSPI_TS_CONF_TS0(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TS_CONF_TS0_SHIFT))&DSPI_TS_CONF_TS0_MASK)
#define DSPI_TS_CONF_TS1_MASK                    0xF0u
#define DSPI_TS_CONF_TS1_SHIFT                   4u
#define DSPI_TS_CONF_TS1_WIDTH                   4u
#define DSPI_TS_CONF_TS1(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TS_CONF_TS1_SHIFT))&DSPI_TS_CONF_TS1_MASK)
#define DSPI_TS_CONF_TS2_MASK                    0xF00u
#define DSPI_TS_CONF_TS2_SHIFT                   8u
#define DSPI_TS_CONF_TS2_WIDTH                   4u
#define DSPI_TS_CONF_TS2(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TS_CONF_TS2_SHIFT))&DSPI_TS_CONF_TS2_MASK)
#define DSPI_TS_CONF_TS3_MASK                    0xF000u
#define DSPI_TS_CONF_TS3_SHIFT                   12u
#define DSPI_TS_CONF_TS3_WIDTH                   4u
#define DSPI_TS_CONF_TS3(x)                      (((uint32_t)(((uint32_t)(x))<<DSPI_TS_CONF_TS3_SHIFT))&DSPI_TS_CONF_TS3_MASK)

/*!
 * @}
 */ /* end of group DSPI_Register_Masks */


/*!
 * @}
 */ /* end of group DSPI_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- ENET Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ENET_Peripheral_Access_Layer ENET Peripheral Access Layer
 * @{
 */


/** ENET - Size of Registers Arrays */
#define ENET_TXIC_COUNT                          3u
#define ENET_RXIC_COUNT                          3u
#define ENET_RCMR_COUNT                          2u
#define ENET_DMACFG_COUNT                        2u
#define ENET_CHANNEL_COUNT                       4u

/** ENET - Register Layout Typedef */
typedef struct {
       uint8_t RESERVED_0[4];
  __IO uint32_t EIR;                               /**< Interrupt Event Register, offset: 0x4 */
  __IO uint32_t EIMR;                              /**< Interrupt Mask Register, offset: 0x8 */
       uint8_t RESERVED_1[4];
  __IO uint32_t RDAR;                              /**< Receive Descriptor Active Register - Ring 0, offset: 0x10 */
  __IO uint32_t TDAR;                              /**< Transmit Descriptor Active Register - Ring 0, offset: 0x14 */
       uint8_t RESERVED_2[12];
  __IO uint32_t ECR;                               /**< Ethernet Control Register, offset: 0x24 */
       uint8_t RESERVED_3[24];
  __IO uint32_t MMFR;                              /**< MII Management Frame Register, offset: 0x40 */
  __IO uint32_t MSCR;                              /**< MII Speed Control Register, offset: 0x44 */
       uint8_t RESERVED_4[28];
  __IO uint32_t MIBC;                              /**< MIB Control Register, offset: 0x64 */
       uint8_t RESERVED_5[28];
  __IO uint32_t RCR;                               /**< Receive Control Register, offset: 0x84 */
       uint8_t RESERVED_6[60];
  __IO uint32_t TCR;                               /**< Transmit Control Register, offset: 0xC4 */
       uint8_t RESERVED_7[28];
  __IO uint32_t PALR;                              /**< Physical Address Lower Register, offset: 0xE4 */
  __IO uint32_t PAUR;                              /**< Physical Address Upper Register, offset: 0xE8 */
  __IO uint32_t OPD;                               /**< Opcode/Pause Duration Register, offset: 0xEC */
  __IO uint32_t TXIC[ENET_TXIC_COUNT];             /**< Transmit Interrupt Coalescing Register, array offset: 0xF0, array step: 0x4 */
       uint8_t RESERVED_8[4];
  __IO uint32_t RXIC[ENET_RXIC_COUNT];             /**< Receive Interrupt Coalescing Register, array offset: 0x100, array step: 0x4 */
       uint8_t RESERVED_9[12];
  __IO uint32_t IAUR;                              /**< Descriptor Individual Upper Address Register, offset: 0x118 */
  __IO uint32_t IALR;                              /**< Descriptor Individual Lower Address Register, offset: 0x11C */
  __IO uint32_t GAUR;                              /**< Descriptor Group Upper Address Register, offset: 0x120 */
  __IO uint32_t GALR;                              /**< Descriptor Group Lower Address Register, offset: 0x124 */
       uint8_t RESERVED_10[28];
  __IO uint32_t TFWR;                              /**< Transmit FIFO Watermark Register, offset: 0x144 */
       uint8_t RESERVED_11[24];
  __IO uint32_t RDSR1;                             /**< Receive Descriptor Ring 1 Start Register, offset: 0x160 */
  __IO uint32_t TDSR1;                             /**< Transmit Buffer Descriptor Ring 1 Start Register, offset: 0x164 */
  __IO uint32_t MRBR1;                             /**< Maximum Receive Buffer Size Register - Ring 1, offset: 0x168 */
  __IO uint32_t RDSR2;                             /**< Receive Descriptor Ring 2 Start Register, offset: 0x16C */
  __IO uint32_t TDSR2;                             /**< Transmit Buffer Descriptor Ring 2 Start Register, offset: 0x170 */
  __IO uint32_t MRBR2;                             /**< Maximum Receive Buffer Size Register - Ring 2, offset: 0x174 */
       uint8_t RESERVED_12[8];
  __IO uint32_t RDSR;                              /**< Receive Descriptor Ring 0 Start Register, offset: 0x180 */
  __IO uint32_t TDSR;                              /**< Transmit Buffer Descriptor Ring 0 Start Register, offset: 0x184 */
  __IO uint32_t MRBR;                              /**< Maximum Receive Buffer Size Register - Ring 0, offset: 0x188 */
       uint8_t RESERVED_13[4];
  __IO uint32_t RSFL;                              /**< Receive FIFO Section Full Threshold, offset: 0x190 */
  __IO uint32_t RSEM;                              /**< Receive FIFO Section Empty Threshold, offset: 0x194 */
  __IO uint32_t RAEM;                              /**< Receive FIFO Almost Empty Threshold, offset: 0x198 */
  __IO uint32_t RAFL;                              /**< Receive FIFO Almost Full Threshold, offset: 0x19C */
  __IO uint32_t TSEM;                              /**< Transmit FIFO Section Empty Threshold, offset: 0x1A0 */
  __IO uint32_t TAEM;                              /**< Transmit FIFO Almost Empty Threshold, offset: 0x1A4 */
  __IO uint32_t TAFL;                              /**< Transmit FIFO Almost Full Threshold, offset: 0x1A8 */
  __IO uint32_t TIPG;                              /**< Transmit Inter-Packet Gap, offset: 0x1AC */
  __IO uint32_t FTRL;                              /**< Frame Truncation Length, offset: 0x1B0 */
       uint8_t RESERVED_14[12];
  __IO uint32_t TACC;                              /**< Transmit Accelerator Function Configuration, offset: 0x1C0 */
  __IO uint32_t RACC;                              /**< Receive Accelerator Function Configuration, offset: 0x1C4 */
  __IO uint32_t RCMR[ENET_RCMR_COUNT];             /**< Receive Classification Match Register for Class n, array offset: 0x1C8, array step: 0x4 */
       uint8_t RESERVED_15[8];
  __IO uint32_t DMACFG[ENET_DMACFG_COUNT];         /**< DMA Class Based Configuration, array offset: 0x1D8, array step: 0x4 */
  __IO uint32_t RDAR1;                             /**< Receive Descriptor Active Register - Ring 1, offset: 0x1E0 */
  __IO uint32_t TDAR1;                             /**< Transmit Descriptor Active Register - Ring 1, offset: 0x1E4 */
  __IO uint32_t RDAR2;                             /**< Receive Descriptor Active Register - Ring 2, offset: 0x1E8 */
  __IO uint32_t TDAR2;                             /**< Transmit Descriptor Active Register - Ring 2, offset: 0x1EC */
  __IO uint32_t QOS;                               /**< QOS Scheme, offset: 0x1F0 */
       uint8_t RESERVED_16[12];
  __I  uint32_t RMON_T_DROP;                       /**< Reserved Statistic Register, offset: 0x200 */
  __I  uint32_t RMON_T_PACKETS;                    /**< Tx Packet Count Statistic Register, offset: 0x204 */
  __I  uint32_t RMON_T_BC_PKT;                     /**< Tx Broadcast Packets Statistic Register, offset: 0x208 */
  __I  uint32_t RMON_T_MC_PKT;                     /**< Tx Multicast Packets Statistic Register, offset: 0x20C */
  __I  uint32_t RMON_T_CRC_ALIGN;                  /**< Tx Packets with CRC/Align Error Statistic Register, offset: 0x210 */
  __I  uint32_t RMON_T_UNDERSIZE;                  /**< Tx Packets Less Than Bytes and Good CRC Statistic Register, offset: 0x214 */
  __I  uint32_t RMON_T_OVERSIZE;                   /**< Tx Packets GT MAX_FL bytes and Good CRC Statistic Register, offset: 0x218 */
  __I  uint32_t RMON_T_FRAG;                       /**< Tx Packets Less Than 64 Bytes and Bad CRC Statistic Register, offset: 0x21C */
  __I  uint32_t RMON_T_JAB;                        /**< Tx Packets Greater Than MAX_FL bytes and Bad CRC Statistic Register, offset: 0x220 */
  __I  uint32_t RMON_T_COL;                        /**< Tx Collision Count Statistic Register, offset: 0x224 */
  __I  uint32_t RMON_T_P64;                        /**< Tx 64-Byte Packets Statistic Register, offset: 0x228 */
  __I  uint32_t RMON_T_P65TO127;                   /**< Tx 65- to 127-byte Packets Statistic Register, offset: 0x22C */
  __I  uint32_t RMON_T_P128TO255;                  /**< Tx 128- to 255-byte Packets Statistic Register, offset: 0x230 */
  __I  uint32_t RMON_T_P256TO511;                  /**< Tx 256- to 511-byte Packets Statistic Register, offset: 0x234 */
  __I  uint32_t RMON_T_P512TO1023;                 /**< Tx 512- to 1023-byte Packets Statistic Register, offset: 0x238 */
  __I  uint32_t RMON_T_P1024TO2047;                /**< Tx 1024- to 2047-byte Packets Statistic Register, offset: 0x23C */
  __I  uint32_t RMON_T_P_GTE2048;                  /**< Tx Packets Greater Than 2048 Bytes Statistic Register, offset: 0x240 */
  __I  uint32_t RMON_T_OCTETS;                     /**< Tx Octets Statistic Register, offset: 0x244 */
  __I  uint32_t IEEE_T_DROP;                       /**< Reserved Statistic Register, offset: 0x248 */
  __I  uint32_t IEEE_T_FRAME_OK;                   /**< Frames Transmitted OK Statistic Register, offset: 0x24C */
  __I  uint32_t IEEE_T_1COL;                       /**< Frames Transmitted with Single Collision Statistic Register, offset: 0x250 */
  __I  uint32_t IEEE_T_MCOL;                       /**< Frames Transmitted with Multiple Collisions Statistic Register, offset: 0x254 */
  __I  uint32_t IEEE_T_DEF;                        /**< Frames Transmitted after Deferral Delay Statistic Register, offset: 0x258 */
  __I  uint32_t IEEE_T_LCOL;                       /**< Frames Transmitted with Late Collision Statistic Register, offset: 0x25C */
  __I  uint32_t IEEE_T_EXCOL;                      /**< Frames Transmitted with Excessive Collisions Statistic Register, offset: 0x260 */
  __I  uint32_t IEEE_T_MACERR;                     /**< Frames Transmitted with Tx FIFO Underrun Statistic Register, offset: 0x264 */
  __I  uint32_t IEEE_T_CSERR;                      /**< Frames Transmitted with Carrier Sense Error Statistic Register, offset: 0x268 */
  __I  uint32_t IEEE_T_SQE;                        /**< Reserved Statistic Register, offset: 0x26C */
  __I  uint32_t IEEE_T_FDXFC;                      /**< Flow Control Pause Frames Transmitted Statistic Register, offset: 0x270 */
  __I  uint32_t IEEE_T_OCTETS_OK;                  /**< Octet Count for Frames Transmitted w/o Error Statistic Register, offset: 0x274 */
       uint8_t RESERVED_17[12];
  __I  uint32_t RMON_R_PACKETS;                    /**< Rx Packet Count Statistic Register, offset: 0x284 */
  __I  uint32_t RMON_R_BC_PKT;                     /**< Rx Broadcast Packets Statistic Register, offset: 0x288 */
  __I  uint32_t RMON_R_MC_PKT;                     /**< Rx Multicast Packets Statistic Register, offset: 0x28C */
  __I  uint32_t RMON_R_CRC_ALIGN;                  /**< Rx Packets with CRC/Align Error Statistic Register, offset: 0x290 */
  __I  uint32_t RMON_R_UNDERSIZE;                  /**< Rx Packets with Less Than 64 Bytes and Good CRC Statistic Register, offset: 0x294 */
  __I  uint32_t RMON_R_OVERSIZE;                   /**< Rx Packets Greater Than MAX_FL and Good CRC Statistic Register, offset: 0x298 */
  __I  uint32_t RMON_R_FRAG;                       /**< Rx Packets Less Than 64 Bytes and Bad CRC Statistic Register, offset: 0x29C */
  __I  uint32_t RMON_R_JAB;                        /**< Rx Packets Greater Than MAX_FL Bytes and Bad CRC Statistic Register, offset: 0x2A0 */
  __I  uint32_t RMON_R_RESVD_0;                    /**< Reserved Statistic Register, offset: 0x2A4 */
  __I  uint32_t RMON_R_P64;                        /**< Rx 64-Byte Packets Statistic Register, offset: 0x2A8 */
  __I  uint32_t RMON_R_P65TO127;                   /**< Rx 65- to 127-Byte Packets Statistic Register, offset: 0x2AC */
  __I  uint32_t RMON_R_P128TO255;                  /**< Rx 128- to 255-Byte Packets Statistic Register, offset: 0x2B0 */
  __I  uint32_t RMON_R_P256TO511;                  /**< Rx 256- to 511-Byte Packets Statistic Register, offset: 0x2B4 */
  __I  uint32_t RMON_R_P512TO1023;                 /**< Rx 512- to 1023-Byte Packets Statistic Register, offset: 0x2B8 */
  __I  uint32_t RMON_R_P1024TO2047;                /**< Rx 1024- to 2047-Byte Packets Statistic Register, offset: 0x2BC */
  __I  uint32_t RMON_R_P_GTE2048;                  /**< Rx Packets Greater than 2048 Bytes Statistic Register, offset: 0x2C0 */
  __I  uint32_t RMON_R_OCTETS;                     /**< Rx Octets Statistic Register, offset: 0x2C4 */
  __I  uint32_t IEEE_R_DROP;                       /**< Frames not Counted Correctly Statistic Register, offset: 0x2C8 */
  __I  uint32_t IEEE_R_FRAME_OK;                   /**< Frames Received OK Statistic Register, offset: 0x2CC */
  __I  uint32_t IEEE_R_CRC;                        /**< Frames Received with CRC Error Statistic Register, offset: 0x2D0 */
  __I  uint32_t IEEE_R_ALIGN;                      /**< Frames Received with Alignment Error Statistic Register, offset: 0x2D4 */
  __I  uint32_t IEEE_R_MACERR;                     /**< Receive FIFO Overflow Count Statistic Register, offset: 0x2D8 */
  __I  uint32_t IEEE_R_FDXFC;                      /**< Flow Control Pause Frames Received Statistic Register, offset: 0x2DC */
  __I  uint32_t IEEE_R_OCTETS_OK;                  /**< Octet Count for Frames Received without Error Statistic Register, offset: 0x2E0 */
       uint8_t RESERVED_18[284];
  __IO uint32_t ATCR;                              /**< Adjustable Timer Control Register, offset: 0x400 */
  __IO uint32_t ATVR;                              /**< Timer Value Register, offset: 0x404 */
  __IO uint32_t ATOFF;                             /**< Timer Offset Register, offset: 0x408 */
  __IO uint32_t ATPER;                             /**< Timer Period Register, offset: 0x40C */
  __IO uint32_t ATCOR;                             /**< Timer Correction Register, offset: 0x410 */
  __IO uint32_t ATINC;                             /**< Time-Stamping Clock Period Register, offset: 0x414 */
  __I  uint32_t ATSTMP;                            /**< Timestamp of Last Transmitted Frame, offset: 0x418 */
       uint8_t RESERVED_19[356];
  __IO uint32_t MDATA;                             /**< Pattern Match Data Register, offset: 0x580 */
  __IO uint32_t MMASK;                             /**< Match Entry Mask Register, offset: 0x584 */
  __IO uint32_t MCONFIG;                           /**< Match Entry Rules Configuration Register, offset: 0x588 */
  __IO uint32_t MENTRYRW;                          /**< Match Entry Read/Write Command Register, offset: 0x58C */
  __IO uint32_t RXPCTL;                            /**< Receive Parser Control Register, offset: 0x590 */
  __IO uint32_t MAXFRMOFF;                         /**< Maximum Frame Offset, offset: 0x594 */
  __I  uint32_t RXPARST;                           /**< Receive Parser Status, offset: 0x598 */
       uint8_t RESERVED_20[4];
  __I  uint32_t PARSDSCD;                          /**< Parser Discard Count, offset: 0x5A0 */
  __I  uint32_t PRSACPT0;                          /**< Parser Accept Count 0, offset: 0x5A4 */
  __I  uint32_t PRSRJCT0;                          /**< Parser Reject Count 0, offset: 0x5A8 */
  __I  uint32_t PRSACPT1;                          /**< Parser Accept Count 1, offset: 0x5AC */
  __I  uint32_t PRSRJCT1;                          /**< Parser Reject Count 1, offset: 0x5B0 */
  __I  uint32_t PRSACPT2;                          /**< Parser Accept Count 2, offset: 0x5B4 */
  __I  uint32_t PRSRJCT2;                          /**< Parser Reject Count 2, offset: 0x5B8 */
       uint8_t RESERVED_21[72];
  __IO uint32_t TGSR;                              /**< Timer Global Status Register, offset: 0x604 */
  struct {                                         /* offset: 0x608, array step: 0x8 */
    __IO uint32_t TCSR;                              /**< Timer Control Status Register, array offset: 0x608, array step: 0x8 */
    __IO uint32_t TCCR;                              /**< Timer Compare Capture Register, array offset: 0x60C, array step: 0x8 */
  } CHANNEL[ENET_CHANNEL_COUNT];
} ENET_Type, *ENET_MemMapPtr;

 /** Number of instances of the ENET module. */
#define ENET_INSTANCE_COUNT                      (2u)


/* ENET - Peripheral instance base addresses */
/** Peripheral ENET_0 base address */
#define ENET_0_BASE                              (0xFFCC4000u)
/** Peripheral ENET_0 base pointer */
#define ENET_0                                   ((ENET_Type *)ENET_0_BASE)
/** Peripheral ENET_1 base address */
#define ENET_1_BASE                              (0xFFCC8000u)
/** Peripheral ENET_1 base pointer */
#define ENET_1                                   ((ENET_Type *)ENET_1_BASE)
/** Array initializer of ENET peripheral base addresses */
#define ENET_BASE_ADDRS                          { ENET_0_BASE, ENET_1_BASE }
/** Array initializer of ENET peripheral base pointers */
#define ENET_BASE_PTRS                           { ENET_0, ENET_1 }

/** Interrupt vectors for the ENET peripheral type */
#define ENET_Err_IRQS                     { ENET0_GROUP0_IRQn, ENET1_GROUP0_IRQn }
#define ENET_Rx_IRQS                      { ENET0_GROUP1_IRQn, ENET1_GROUP1_IRQn }
#define ENET_Tx_IRQS                      { ENET0_GROUP2_IRQn, ENET1_GROUP2_IRQn }
#define ENET_Parser_IRQS                  { ENET0_GROUP3_IRQn, ENET1_GROUP3_IRQn }
#define ENET_Timer_IRQS                   { ENET0_GROUP4_IRQn, ENET1_GROUP4_IRQn }
#define ENET_Rx_1_IRQS                    { ENET0_GROUP5_IRQn, ENET1_GROUP5_IRQn }
#define ENET_Tx_1_IRQS                    { ENET0_GROUP6_IRQn, ENET1_GROUP6_IRQn }
#define ENET_Rx_2_IRQS                    { ENET0_GROUP7_IRQn, ENET1_GROUP7_IRQn }
#define ENET_Tx_2_IRQS                    { ENET0_GROUP8_IRQn, ENET1_GROUP8_IRQn }

/* ----------------------------------------------------------------------------
   -- ENET Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ENET_Register_Masks ENET Register Masks
 * @{
 */

/* EIR Bit Fields */
#define ENET_EIR_RXB1_MASK                       0x1u
#define ENET_EIR_RXB1_SHIFT                      0u
#define ENET_EIR_RXB1_WIDTH                      1u
#define ENET_EIR_RXB1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXB1_SHIFT))&ENET_EIR_RXB1_MASK)
#define ENET_EIR_RXF1_MASK                       0x2u
#define ENET_EIR_RXF1_SHIFT                      1u
#define ENET_EIR_RXF1_WIDTH                      1u
#define ENET_EIR_RXF1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXF1_SHIFT))&ENET_EIR_RXF1_MASK)
#define ENET_EIR_TXB1_MASK                       0x4u
#define ENET_EIR_TXB1_SHIFT                      2u
#define ENET_EIR_TXB1_WIDTH                      1u
#define ENET_EIR_TXB1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXB1_SHIFT))&ENET_EIR_TXB1_MASK)
#define ENET_EIR_TXF1_MASK                       0x8u
#define ENET_EIR_TXF1_SHIFT                      3u
#define ENET_EIR_TXF1_WIDTH                      1u
#define ENET_EIR_TXF1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXF1_SHIFT))&ENET_EIR_TXF1_MASK)
#define ENET_EIR_RXB2_MASK                       0x10u
#define ENET_EIR_RXB2_SHIFT                      4u
#define ENET_EIR_RXB2_WIDTH                      1u
#define ENET_EIR_RXB2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXB2_SHIFT))&ENET_EIR_RXB2_MASK)
#define ENET_EIR_RXF2_MASK                       0x20u
#define ENET_EIR_RXF2_SHIFT                      5u
#define ENET_EIR_RXF2_WIDTH                      1u
#define ENET_EIR_RXF2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXF2_SHIFT))&ENET_EIR_RXF2_MASK)
#define ENET_EIR_TXB2_MASK                       0x40u
#define ENET_EIR_TXB2_SHIFT                      6u
#define ENET_EIR_TXB2_WIDTH                      1u
#define ENET_EIR_TXB2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXB2_SHIFT))&ENET_EIR_TXB2_MASK)
#define ENET_EIR_TXF2_MASK                       0x80u
#define ENET_EIR_TXF2_SHIFT                      7u
#define ENET_EIR_TXF2_WIDTH                      1u
#define ENET_EIR_TXF2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXF2_SHIFT))&ENET_EIR_TXF2_MASK)
#define ENET_EIR_PARSRF_MASK                     0x200u
#define ENET_EIR_PARSRF_SHIFT                    9u
#define ENET_EIR_PARSRF_WIDTH                    1u
#define ENET_EIR_PARSRF(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_EIR_PARSRF_SHIFT))&ENET_EIR_PARSRF_MASK)
#define ENET_EIR_PARSERR_MASK                    0x400u
#define ENET_EIR_PARSERR_SHIFT                   10u
#define ENET_EIR_PARSERR_WIDTH                   1u
#define ENET_EIR_PARSERR(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_EIR_PARSERR_SHIFT))&ENET_EIR_PARSERR_MASK)
#define ENET_EIR_RXFLUSH_0_MASK                  0x1000u
#define ENET_EIR_RXFLUSH_0_SHIFT                 12u
#define ENET_EIR_RXFLUSH_0_WIDTH                 1u
#define ENET_EIR_RXFLUSH_0(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXFLUSH_0_SHIFT))&ENET_EIR_RXFLUSH_0_MASK)
#define ENET_EIR_RXFLUSH_1_MASK                  0x2000u
#define ENET_EIR_RXFLUSH_1_SHIFT                 13u
#define ENET_EIR_RXFLUSH_1_WIDTH                 1u
#define ENET_EIR_RXFLUSH_1(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXFLUSH_1_SHIFT))&ENET_EIR_RXFLUSH_1_MASK)
#define ENET_EIR_RXFLUSH_2_MASK                  0x4000u
#define ENET_EIR_RXFLUSH_2_SHIFT                 14u
#define ENET_EIR_RXFLUSH_2_WIDTH                 1u
#define ENET_EIR_RXFLUSH_2(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXFLUSH_2_SHIFT))&ENET_EIR_RXFLUSH_2_MASK)
#define ENET_EIR_TS_TIMER_MASK                   0x8000u
#define ENET_EIR_TS_TIMER_SHIFT                  15u
#define ENET_EIR_TS_TIMER_WIDTH                  1u
#define ENET_EIR_TS_TIMER(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TS_TIMER_SHIFT))&ENET_EIR_TS_TIMER_MASK)
#define ENET_EIR_TS_AVAIL_MASK                   0x10000u
#define ENET_EIR_TS_AVAIL_SHIFT                  16u
#define ENET_EIR_TS_AVAIL_WIDTH                  1u
#define ENET_EIR_TS_AVAIL(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TS_AVAIL_SHIFT))&ENET_EIR_TS_AVAIL_MASK)
#define ENET_EIR_WAKEUP_MASK                     0x20000u
#define ENET_EIR_WAKEUP_SHIFT                    17u
#define ENET_EIR_WAKEUP_WIDTH                    1u
#define ENET_EIR_WAKEUP(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_EIR_WAKEUP_SHIFT))&ENET_EIR_WAKEUP_MASK)
#define ENET_EIR_PLR_MASK                        0x40000u
#define ENET_EIR_PLR_SHIFT                       18u
#define ENET_EIR_PLR_WIDTH                       1u
#define ENET_EIR_PLR(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_PLR_SHIFT))&ENET_EIR_PLR_MASK)
#define ENET_EIR_UN_MASK                         0x80000u
#define ENET_EIR_UN_SHIFT                        19u
#define ENET_EIR_UN_WIDTH                        1u
#define ENET_EIR_UN(x)                           (((uint32_t)(((uint32_t)(x))<<ENET_EIR_UN_SHIFT))&ENET_EIR_UN_MASK)
#define ENET_EIR_RL_MASK                         0x100000u
#define ENET_EIR_RL_SHIFT                        20u
#define ENET_EIR_RL_WIDTH                        1u
#define ENET_EIR_RL(x)                           (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RL_SHIFT))&ENET_EIR_RL_MASK)
#define ENET_EIR_LC_MASK                         0x200000u
#define ENET_EIR_LC_SHIFT                        21u
#define ENET_EIR_LC_WIDTH                        1u
#define ENET_EIR_LC(x)                           (((uint32_t)(((uint32_t)(x))<<ENET_EIR_LC_SHIFT))&ENET_EIR_LC_MASK)
#define ENET_EIR_EBERR_MASK                      0x400000u
#define ENET_EIR_EBERR_SHIFT                     22u
#define ENET_EIR_EBERR_WIDTH                     1u
#define ENET_EIR_EBERR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIR_EBERR_SHIFT))&ENET_EIR_EBERR_MASK)
#define ENET_EIR_MII_MASK                        0x800000u
#define ENET_EIR_MII_SHIFT                       23u
#define ENET_EIR_MII_WIDTH                       1u
#define ENET_EIR_MII(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_MII_SHIFT))&ENET_EIR_MII_MASK)
#define ENET_EIR_RXB_MASK                        0x1000000u
#define ENET_EIR_RXB_SHIFT                       24u
#define ENET_EIR_RXB_WIDTH                       1u
#define ENET_EIR_RXB(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXB_SHIFT))&ENET_EIR_RXB_MASK)
#define ENET_EIR_RXF_MASK                        0x2000000u
#define ENET_EIR_RXF_SHIFT                       25u
#define ENET_EIR_RXF_WIDTH                       1u
#define ENET_EIR_RXF(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_RXF_SHIFT))&ENET_EIR_RXF_MASK)
#define ENET_EIR_TXB_MASK                        0x4000000u
#define ENET_EIR_TXB_SHIFT                       26u
#define ENET_EIR_TXB_WIDTH                       1u
#define ENET_EIR_TXB(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXB_SHIFT))&ENET_EIR_TXB_MASK)
#define ENET_EIR_TXF_MASK                        0x8000000u
#define ENET_EIR_TXF_SHIFT                       27u
#define ENET_EIR_TXF_WIDTH                       1u
#define ENET_EIR_TXF(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_TXF_SHIFT))&ENET_EIR_TXF_MASK)
#define ENET_EIR_GRA_MASK                        0x10000000u
#define ENET_EIR_GRA_SHIFT                       28u
#define ENET_EIR_GRA_WIDTH                       1u
#define ENET_EIR_GRA(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIR_GRA_SHIFT))&ENET_EIR_GRA_MASK)
#define ENET_EIR_BABT_MASK                       0x20000000u
#define ENET_EIR_BABT_SHIFT                      29u
#define ENET_EIR_BABT_WIDTH                      1u
#define ENET_EIR_BABT(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_BABT_SHIFT))&ENET_EIR_BABT_MASK)
#define ENET_EIR_BABR_MASK                       0x40000000u
#define ENET_EIR_BABR_SHIFT                      30u
#define ENET_EIR_BABR_WIDTH                      1u
#define ENET_EIR_BABR(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIR_BABR_SHIFT))&ENET_EIR_BABR_MASK)
/* EIMR Bit Fields */
#define ENET_EIMR_RXB1_MASK                      0x1u
#define ENET_EIMR_RXB1_SHIFT                     0u
#define ENET_EIMR_RXB1_WIDTH                     1u
#define ENET_EIMR_RXB1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXB1_SHIFT))&ENET_EIMR_RXB1_MASK)
#define ENET_EIMR_RXF1_MASK                      0x2u
#define ENET_EIMR_RXF1_SHIFT                     1u
#define ENET_EIMR_RXF1_WIDTH                     1u
#define ENET_EIMR_RXF1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXF1_SHIFT))&ENET_EIMR_RXF1_MASK)
#define ENET_EIMR_TXB1_MASK                      0x4u
#define ENET_EIMR_TXB1_SHIFT                     2u
#define ENET_EIMR_TXB1_WIDTH                     1u
#define ENET_EIMR_TXB1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXB1_SHIFT))&ENET_EIMR_TXB1_MASK)
#define ENET_EIMR_TXF1_MASK                      0x8u
#define ENET_EIMR_TXF1_SHIFT                     3u
#define ENET_EIMR_TXF1_WIDTH                     1u
#define ENET_EIMR_TXF1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXF1_SHIFT))&ENET_EIMR_TXF1_MASK)
#define ENET_EIMR_RXB2_MASK                      0x10u
#define ENET_EIMR_RXB2_SHIFT                     4u
#define ENET_EIMR_RXB2_WIDTH                     1u
#define ENET_EIMR_RXB2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXB2_SHIFT))&ENET_EIMR_RXB2_MASK)
#define ENET_EIMR_RXF2_MASK                      0x20u
#define ENET_EIMR_RXF2_SHIFT                     5u
#define ENET_EIMR_RXF2_WIDTH                     1u
#define ENET_EIMR_RXF2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXF2_SHIFT))&ENET_EIMR_RXF2_MASK)
#define ENET_EIMR_TXB2_MASK                      0x40u
#define ENET_EIMR_TXB2_SHIFT                     6u
#define ENET_EIMR_TXB2_WIDTH                     1u
#define ENET_EIMR_TXB2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXB2_SHIFT))&ENET_EIMR_TXB2_MASK)
#define ENET_EIMR_TXF2_MASK                      0x80u
#define ENET_EIMR_TXF2_SHIFT                     7u
#define ENET_EIMR_TXF2_WIDTH                     1u
#define ENET_EIMR_TXF2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXF2_SHIFT))&ENET_EIMR_TXF2_MASK)
#define ENET_EIMR_PARSRF_MASK                    0x200u
#define ENET_EIMR_PARSRF_SHIFT                   9u
#define ENET_EIMR_PARSRF_WIDTH                   1u
#define ENET_EIMR_PARSRF(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_PARSRF_SHIFT))&ENET_EIMR_PARSRF_MASK)
#define ENET_EIMR_PARSERR_MASK                   0x400u
#define ENET_EIMR_PARSERR_SHIFT                  10u
#define ENET_EIMR_PARSERR_WIDTH                  1u
#define ENET_EIMR_PARSERR(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_PARSERR_SHIFT))&ENET_EIMR_PARSERR_MASK)
#define ENET_EIMR_RXFLUSH_0_MASK                 0x1000u
#define ENET_EIMR_RXFLUSH_0_SHIFT                12u
#define ENET_EIMR_RXFLUSH_0_WIDTH                1u
#define ENET_EIMR_RXFLUSH_0(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXFLUSH_0_SHIFT))&ENET_EIMR_RXFLUSH_0_MASK)
#define ENET_EIMR_RXFLUSH_1_MASK                 0x2000u
#define ENET_EIMR_RXFLUSH_1_SHIFT                13u
#define ENET_EIMR_RXFLUSH_1_WIDTH                1u
#define ENET_EIMR_RXFLUSH_1(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXFLUSH_1_SHIFT))&ENET_EIMR_RXFLUSH_1_MASK)
#define ENET_EIMR_RXFLUSH_2_MASK                 0x4000u
#define ENET_EIMR_RXFLUSH_2_SHIFT                14u
#define ENET_EIMR_RXFLUSH_2_WIDTH                1u
#define ENET_EIMR_RXFLUSH_2(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXFLUSH_2_SHIFT))&ENET_EIMR_RXFLUSH_2_MASK)
#define ENET_EIMR_TS_TIMER_MASK                  0x8000u
#define ENET_EIMR_TS_TIMER_SHIFT                 15u
#define ENET_EIMR_TS_TIMER_WIDTH                 1u
#define ENET_EIMR_TS_TIMER(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TS_TIMER_SHIFT))&ENET_EIMR_TS_TIMER_MASK)
#define ENET_EIMR_TS_AVAIL_MASK                  0x10000u
#define ENET_EIMR_TS_AVAIL_SHIFT                 16u
#define ENET_EIMR_TS_AVAIL_WIDTH                 1u
#define ENET_EIMR_TS_AVAIL(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TS_AVAIL_SHIFT))&ENET_EIMR_TS_AVAIL_MASK)
#define ENET_EIMR_WAKEUP_MASK                    0x20000u
#define ENET_EIMR_WAKEUP_SHIFT                   17u
#define ENET_EIMR_WAKEUP_WIDTH                   1u
#define ENET_EIMR_WAKEUP(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_WAKEUP_SHIFT))&ENET_EIMR_WAKEUP_MASK)
#define ENET_EIMR_PLR_MASK                       0x40000u
#define ENET_EIMR_PLR_SHIFT                      18u
#define ENET_EIMR_PLR_WIDTH                      1u
#define ENET_EIMR_PLR(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_PLR_SHIFT))&ENET_EIMR_PLR_MASK)
#define ENET_EIMR_UN_MASK                        0x80000u
#define ENET_EIMR_UN_SHIFT                       19u
#define ENET_EIMR_UN_WIDTH                       1u
#define ENET_EIMR_UN(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_UN_SHIFT))&ENET_EIMR_UN_MASK)
#define ENET_EIMR_RL_MASK                        0x100000u
#define ENET_EIMR_RL_SHIFT                       20u
#define ENET_EIMR_RL_WIDTH                       1u
#define ENET_EIMR_RL(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RL_SHIFT))&ENET_EIMR_RL_MASK)
#define ENET_EIMR_LC_MASK                        0x200000u
#define ENET_EIMR_LC_SHIFT                       21u
#define ENET_EIMR_LC_WIDTH                       1u
#define ENET_EIMR_LC(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_LC_SHIFT))&ENET_EIMR_LC_MASK)
#define ENET_EIMR_EBERR_MASK                     0x400000u
#define ENET_EIMR_EBERR_SHIFT                    22u
#define ENET_EIMR_EBERR_WIDTH                    1u
#define ENET_EIMR_EBERR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_EBERR_SHIFT))&ENET_EIMR_EBERR_MASK)
#define ENET_EIMR_MII_MASK                       0x800000u
#define ENET_EIMR_MII_SHIFT                      23u
#define ENET_EIMR_MII_WIDTH                      1u
#define ENET_EIMR_MII(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_MII_SHIFT))&ENET_EIMR_MII_MASK)
#define ENET_EIMR_RXB_MASK                       0x1000000u
#define ENET_EIMR_RXB_SHIFT                      24u
#define ENET_EIMR_RXB_WIDTH                      1u
#define ENET_EIMR_RXB(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXB_SHIFT))&ENET_EIMR_RXB_MASK)
#define ENET_EIMR_RXF_MASK                       0x2000000u
#define ENET_EIMR_RXF_SHIFT                      25u
#define ENET_EIMR_RXF_WIDTH                      1u
#define ENET_EIMR_RXF(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_RXF_SHIFT))&ENET_EIMR_RXF_MASK)
#define ENET_EIMR_TXB_MASK                       0x4000000u
#define ENET_EIMR_TXB_SHIFT                      26u
#define ENET_EIMR_TXB_WIDTH                      1u
#define ENET_EIMR_TXB(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXB_SHIFT))&ENET_EIMR_TXB_MASK)
#define ENET_EIMR_TXF_MASK                       0x8000000u
#define ENET_EIMR_TXF_SHIFT                      27u
#define ENET_EIMR_TXF_WIDTH                      1u
#define ENET_EIMR_TXF(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_TXF_SHIFT))&ENET_EIMR_TXF_MASK)
#define ENET_EIMR_GRA_MASK                       0x10000000u
#define ENET_EIMR_GRA_SHIFT                      28u
#define ENET_EIMR_GRA_WIDTH                      1u
#define ENET_EIMR_GRA(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_GRA_SHIFT))&ENET_EIMR_GRA_MASK)
#define ENET_EIMR_BABT_MASK                      0x20000000u
#define ENET_EIMR_BABT_SHIFT                     29u
#define ENET_EIMR_BABT_WIDTH                     1u
#define ENET_EIMR_BABT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_BABT_SHIFT))&ENET_EIMR_BABT_MASK)
#define ENET_EIMR_BABR_MASK                      0x40000000u
#define ENET_EIMR_BABR_SHIFT                     30u
#define ENET_EIMR_BABR_WIDTH                     1u
#define ENET_EIMR_BABR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_EIMR_BABR_SHIFT))&ENET_EIMR_BABR_MASK)
/* RDAR Bit Fields */
#define ENET_RDAR_RDAR_MASK                      0x1000000u
#define ENET_RDAR_RDAR_SHIFT                     24u
#define ENET_RDAR_RDAR_WIDTH                     1u
#define ENET_RDAR_RDAR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RDAR_RDAR_SHIFT))&ENET_RDAR_RDAR_MASK)
/* TDAR Bit Fields */
#define ENET_TDAR_TDAR_MASK                      0x1000000u
#define ENET_TDAR_TDAR_SHIFT                     24u
#define ENET_TDAR_TDAR_WIDTH                     1u
#define ENET_TDAR_TDAR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TDAR_TDAR_SHIFT))&ENET_TDAR_TDAR_MASK)
/* ECR Bit Fields */
#define ENET_ECR_RESET_MASK                      0x1u
#define ENET_ECR_RESET_SHIFT                     0u
#define ENET_ECR_RESET_WIDTH                     1u
#define ENET_ECR_RESET(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ECR_RESET_SHIFT))&ENET_ECR_RESET_MASK)
#define ENET_ECR_ETHEREN_MASK                    0x2u
#define ENET_ECR_ETHEREN_SHIFT                   1u
#define ENET_ECR_ETHEREN_WIDTH                   1u
#define ENET_ECR_ETHEREN(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ECR_ETHEREN_SHIFT))&ENET_ECR_ETHEREN_MASK)
#define ENET_ECR_MAGICEN_MASK                    0x4u
#define ENET_ECR_MAGICEN_SHIFT                   2u
#define ENET_ECR_MAGICEN_WIDTH                   1u
#define ENET_ECR_MAGICEN(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ECR_MAGICEN_SHIFT))&ENET_ECR_MAGICEN_MASK)
#define ENET_ECR_SLEEP_MASK                      0x8u
#define ENET_ECR_SLEEP_SHIFT                     3u
#define ENET_ECR_SLEEP_WIDTH                     1u
#define ENET_ECR_SLEEP(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ECR_SLEEP_SHIFT))&ENET_ECR_SLEEP_MASK)
#define ENET_ECR_EN1588_MASK                     0x10u
#define ENET_ECR_EN1588_SHIFT                    4u
#define ENET_ECR_EN1588_WIDTH                    1u
#define ENET_ECR_EN1588(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ECR_EN1588_SHIFT))&ENET_ECR_EN1588_MASK)
#define ENET_ECR_DBGEN_MASK                      0x40u
#define ENET_ECR_DBGEN_SHIFT                     6u
#define ENET_ECR_DBGEN_WIDTH                     1u
#define ENET_ECR_DBGEN(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ECR_DBGEN_SHIFT))&ENET_ECR_DBGEN_MASK)
#define ENET_ECR_DBSWP_MASK                      0x100u
#define ENET_ECR_DBSWP_SHIFT                     8u
#define ENET_ECR_DBSWP_WIDTH                     1u
#define ENET_ECR_DBSWP(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ECR_DBSWP_SHIFT))&ENET_ECR_DBSWP_MASK)
#define ENET_ECR_SVLANEN_MASK                    0x200u
#define ENET_ECR_SVLANEN_SHIFT                   9u
#define ENET_ECR_SVLANEN_WIDTH                   1u
#define ENET_ECR_SVLANEN(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ECR_SVLANEN_SHIFT))&ENET_ECR_SVLANEN_MASK)
#define ENET_ECR_VLANUSE2ND_MASK                 0x400u
#define ENET_ECR_VLANUSE2ND_SHIFT                10u
#define ENET_ECR_VLANUSE2ND_WIDTH                1u
#define ENET_ECR_VLANUSE2ND(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_ECR_VLANUSE2ND_SHIFT))&ENET_ECR_VLANUSE2ND_MASK)
#define ENET_ECR_SVLANDBL_MASK                   0x800u
#define ENET_ECR_SVLANDBL_SHIFT                  11u
#define ENET_ECR_SVLANDBL_WIDTH                  1u
#define ENET_ECR_SVLANDBL(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ECR_SVLANDBL_SHIFT))&ENET_ECR_SVLANDBL_MASK)
/* MMFR Bit Fields */
#define ENET_MMFR_DATA_MASK                      0xFFFFu
#define ENET_MMFR_DATA_SHIFT                     0u
#define ENET_MMFR_DATA_WIDTH                     16u
#define ENET_MMFR_DATA(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_DATA_SHIFT))&ENET_MMFR_DATA_MASK)
#define ENET_MMFR_TA_MASK                        0x30000u
#define ENET_MMFR_TA_SHIFT                       16u
#define ENET_MMFR_TA_WIDTH                       2u
#define ENET_MMFR_TA(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_TA_SHIFT))&ENET_MMFR_TA_MASK)
#define ENET_MMFR_RA_MASK                        0x7C0000u
#define ENET_MMFR_RA_SHIFT                       18u
#define ENET_MMFR_RA_WIDTH                       5u
#define ENET_MMFR_RA(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_RA_SHIFT))&ENET_MMFR_RA_MASK)
#define ENET_MMFR_PA_MASK                        0xF800000u
#define ENET_MMFR_PA_SHIFT                       23u
#define ENET_MMFR_PA_WIDTH                       5u
#define ENET_MMFR_PA(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_PA_SHIFT))&ENET_MMFR_PA_MASK)
#define ENET_MMFR_OP_MASK                        0x30000000u
#define ENET_MMFR_OP_SHIFT                       28u
#define ENET_MMFR_OP_WIDTH                       2u
#define ENET_MMFR_OP(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_OP_SHIFT))&ENET_MMFR_OP_MASK)
#define ENET_MMFR_ST_MASK                        0xC0000000u
#define ENET_MMFR_ST_SHIFT                       30u
#define ENET_MMFR_ST_WIDTH                       2u
#define ENET_MMFR_ST(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_MMFR_ST_SHIFT))&ENET_MMFR_ST_MASK)
/* MSCR Bit Fields */
#define ENET_MSCR_MII_SPEED_MASK                 0x7Eu
#define ENET_MSCR_MII_SPEED_SHIFT                1u
#define ENET_MSCR_MII_SPEED_WIDTH                6u
#define ENET_MSCR_MII_SPEED(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_MSCR_MII_SPEED_SHIFT))&ENET_MSCR_MII_SPEED_MASK)
#define ENET_MSCR_DIS_PRE_MASK                   0x80u
#define ENET_MSCR_DIS_PRE_SHIFT                  7u
#define ENET_MSCR_DIS_PRE_WIDTH                  1u
#define ENET_MSCR_DIS_PRE(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_MSCR_DIS_PRE_SHIFT))&ENET_MSCR_DIS_PRE_MASK)
#define ENET_MSCR_HOLDTIME_MASK                  0x700u
#define ENET_MSCR_HOLDTIME_SHIFT                 8u
#define ENET_MSCR_HOLDTIME_WIDTH                 3u
#define ENET_MSCR_HOLDTIME(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_MSCR_HOLDTIME_SHIFT))&ENET_MSCR_HOLDTIME_MASK)
/* MIBC Bit Fields */
#define ENET_MIBC_MIB_CLEAR_MASK                 0x20000000u
#define ENET_MIBC_MIB_CLEAR_SHIFT                29u
#define ENET_MIBC_MIB_CLEAR_WIDTH                1u
#define ENET_MIBC_MIB_CLEAR(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_MIBC_MIB_CLEAR_SHIFT))&ENET_MIBC_MIB_CLEAR_MASK)
#define ENET_MIBC_MIB_IDLE_MASK                  0x40000000u
#define ENET_MIBC_MIB_IDLE_SHIFT                 30u
#define ENET_MIBC_MIB_IDLE_WIDTH                 1u
#define ENET_MIBC_MIB_IDLE(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_MIBC_MIB_IDLE_SHIFT))&ENET_MIBC_MIB_IDLE_MASK)
#define ENET_MIBC_MIB_DIS_MASK                   0x80000000u
#define ENET_MIBC_MIB_DIS_SHIFT                  31u
#define ENET_MIBC_MIB_DIS_WIDTH                  1u
#define ENET_MIBC_MIB_DIS(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_MIBC_MIB_DIS_SHIFT))&ENET_MIBC_MIB_DIS_MASK)
/* RCR Bit Fields */
#define ENET_RCR_LOOP_MASK                       0x1u
#define ENET_RCR_LOOP_SHIFT                      0u
#define ENET_RCR_LOOP_WIDTH                      1u
#define ENET_RCR_LOOP(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_RCR_LOOP_SHIFT))&ENET_RCR_LOOP_MASK)
#define ENET_RCR_DRT_MASK                        0x2u
#define ENET_RCR_DRT_SHIFT                       1u
#define ENET_RCR_DRT_WIDTH                       1u
#define ENET_RCR_DRT(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_RCR_DRT_SHIFT))&ENET_RCR_DRT_MASK)
#define ENET_RCR_MII_MODE_MASK                   0x4u
#define ENET_RCR_MII_MODE_SHIFT                  2u
#define ENET_RCR_MII_MODE_WIDTH                  1u
#define ENET_RCR_MII_MODE(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RCR_MII_MODE_SHIFT))&ENET_RCR_MII_MODE_MASK)
#define ENET_RCR_PROM_MASK                       0x8u
#define ENET_RCR_PROM_SHIFT                      3u
#define ENET_RCR_PROM_WIDTH                      1u
#define ENET_RCR_PROM(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_RCR_PROM_SHIFT))&ENET_RCR_PROM_MASK)
#define ENET_RCR_BC_REJ_MASK                     0x10u
#define ENET_RCR_BC_REJ_SHIFT                    4u
#define ENET_RCR_BC_REJ_WIDTH                    1u
#define ENET_RCR_BC_REJ(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RCR_BC_REJ_SHIFT))&ENET_RCR_BC_REJ_MASK)
#define ENET_RCR_FCE_MASK                        0x20u
#define ENET_RCR_FCE_SHIFT                       5u
#define ENET_RCR_FCE_WIDTH                       1u
#define ENET_RCR_FCE(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_RCR_FCE_SHIFT))&ENET_RCR_FCE_MASK)
#define ENET_RCR_RMII_MODE_MASK                  0x100u
#define ENET_RCR_RMII_MODE_SHIFT                 8u
#define ENET_RCR_RMII_MODE_WIDTH                 1u
#define ENET_RCR_RMII_MODE(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_RCR_RMII_MODE_SHIFT))&ENET_RCR_RMII_MODE_MASK)
#define ENET_RCR_RMII_10T_MASK                   0x200u
#define ENET_RCR_RMII_10T_SHIFT                  9u
#define ENET_RCR_RMII_10T_WIDTH                  1u
#define ENET_RCR_RMII_10T(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RCR_RMII_10T_SHIFT))&ENET_RCR_RMII_10T_MASK)
#define ENET_RCR_PADEN_MASK                      0x1000u
#define ENET_RCR_PADEN_SHIFT                     12u
#define ENET_RCR_PADEN_WIDTH                     1u
#define ENET_RCR_PADEN(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCR_PADEN_SHIFT))&ENET_RCR_PADEN_MASK)
#define ENET_RCR_PAUFWD_MASK                     0x2000u
#define ENET_RCR_PAUFWD_SHIFT                    13u
#define ENET_RCR_PAUFWD_WIDTH                    1u
#define ENET_RCR_PAUFWD(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RCR_PAUFWD_SHIFT))&ENET_RCR_PAUFWD_MASK)
#define ENET_RCR_CRCFWD_MASK                     0x4000u
#define ENET_RCR_CRCFWD_SHIFT                    14u
#define ENET_RCR_CRCFWD_WIDTH                    1u
#define ENET_RCR_CRCFWD(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RCR_CRCFWD_SHIFT))&ENET_RCR_CRCFWD_MASK)
#define ENET_RCR_CFEN_MASK                       0x8000u
#define ENET_RCR_CFEN_SHIFT                      15u
#define ENET_RCR_CFEN_WIDTH                      1u
#define ENET_RCR_CFEN(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_RCR_CFEN_SHIFT))&ENET_RCR_CFEN_MASK)
#define ENET_RCR_MAX_FL_MASK                     0x3FFF0000u
#define ENET_RCR_MAX_FL_SHIFT                    16u
#define ENET_RCR_MAX_FL_WIDTH                    14u
#define ENET_RCR_MAX_FL(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RCR_MAX_FL_SHIFT))&ENET_RCR_MAX_FL_MASK)
#define ENET_RCR_NLC_MASK                        0x40000000u
#define ENET_RCR_NLC_SHIFT                       30u
#define ENET_RCR_NLC_WIDTH                       1u
#define ENET_RCR_NLC(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_RCR_NLC_SHIFT))&ENET_RCR_NLC_MASK)
#define ENET_RCR_GRS_MASK                        0x80000000u
#define ENET_RCR_GRS_SHIFT                       31u
#define ENET_RCR_GRS_WIDTH                       1u
#define ENET_RCR_GRS(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_RCR_GRS_SHIFT))&ENET_RCR_GRS_MASK)
/* TCR Bit Fields */
#define ENET_TCR_GTS_MASK                        0x1u
#define ENET_TCR_GTS_SHIFT                       0u
#define ENET_TCR_GTS_WIDTH                       1u
#define ENET_TCR_GTS(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_TCR_GTS_SHIFT))&ENET_TCR_GTS_MASK)
#define ENET_TCR_FDEN_MASK                       0x4u
#define ENET_TCR_FDEN_SHIFT                      2u
#define ENET_TCR_FDEN_WIDTH                      1u
#define ENET_TCR_FDEN(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TCR_FDEN_SHIFT))&ENET_TCR_FDEN_MASK)
#define ENET_TCR_TFC_PAUSE_MASK                  0x8u
#define ENET_TCR_TFC_PAUSE_SHIFT                 3u
#define ENET_TCR_TFC_PAUSE_WIDTH                 1u
#define ENET_TCR_TFC_PAUSE(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_TCR_TFC_PAUSE_SHIFT))&ENET_TCR_TFC_PAUSE_MASK)
#define ENET_TCR_RFC_PAUSE_MASK                  0x10u
#define ENET_TCR_RFC_PAUSE_SHIFT                 4u
#define ENET_TCR_RFC_PAUSE_WIDTH                 1u
#define ENET_TCR_RFC_PAUSE(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_TCR_RFC_PAUSE_SHIFT))&ENET_TCR_RFC_PAUSE_MASK)
#define ENET_TCR_ADDSEL_MASK                     0xE0u
#define ENET_TCR_ADDSEL_SHIFT                    5u
#define ENET_TCR_ADDSEL_WIDTH                    3u
#define ENET_TCR_ADDSEL(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TCR_ADDSEL_SHIFT))&ENET_TCR_ADDSEL_MASK)
#define ENET_TCR_ADDINS_MASK                     0x100u
#define ENET_TCR_ADDINS_SHIFT                    8u
#define ENET_TCR_ADDINS_WIDTH                    1u
#define ENET_TCR_ADDINS(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TCR_ADDINS_SHIFT))&ENET_TCR_ADDINS_MASK)
#define ENET_TCR_CRCFWD_MASK                     0x200u
#define ENET_TCR_CRCFWD_SHIFT                    9u
#define ENET_TCR_CRCFWD_WIDTH                    1u
#define ENET_TCR_CRCFWD(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TCR_CRCFWD_SHIFT))&ENET_TCR_CRCFWD_MASK)
/* PALR Bit Fields */
#define ENET_PALR_PADDR1_MASK                    0xFFFFFFFFu
#define ENET_PALR_PADDR1_SHIFT                   0u
#define ENET_PALR_PADDR1_WIDTH                   32u
#define ENET_PALR_PADDR1(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_PALR_PADDR1_SHIFT))&ENET_PALR_PADDR1_MASK)
/* PAUR Bit Fields */
#define ENET_PAUR_TYPE_MASK                      0xFFFFu
#define ENET_PAUR_TYPE_SHIFT                     0u
#define ENET_PAUR_TYPE_WIDTH                     16u
#define ENET_PAUR_TYPE(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_PAUR_TYPE_SHIFT))&ENET_PAUR_TYPE_MASK)
#define ENET_PAUR_PADDR2_MASK                    0xFFFF0000u
#define ENET_PAUR_PADDR2_SHIFT                   16u
#define ENET_PAUR_PADDR2_WIDTH                   16u
#define ENET_PAUR_PADDR2(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_PAUR_PADDR2_SHIFT))&ENET_PAUR_PADDR2_MASK)
/* OPD Bit Fields */
#define ENET_OPD_PAUSE_DUR_MASK                  0xFFFFu
#define ENET_OPD_PAUSE_DUR_SHIFT                 0u
#define ENET_OPD_PAUSE_DUR_WIDTH                 16u
#define ENET_OPD_PAUSE_DUR(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_OPD_PAUSE_DUR_SHIFT))&ENET_OPD_PAUSE_DUR_MASK)
#define ENET_OPD_OPCODE_MASK                     0xFFFF0000u
#define ENET_OPD_OPCODE_SHIFT                    16u
#define ENET_OPD_OPCODE_WIDTH                    16u
#define ENET_OPD_OPCODE(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_OPD_OPCODE_SHIFT))&ENET_OPD_OPCODE_MASK)
/* TXIC Bit Fields */
#define ENET_TXIC_ICTT_MASK                      0xFFFFu
#define ENET_TXIC_ICTT_SHIFT                     0u
#define ENET_TXIC_ICTT_WIDTH                     16u
#define ENET_TXIC_ICTT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TXIC_ICTT_SHIFT))&ENET_TXIC_ICTT_MASK)
#define ENET_TXIC_ICFT_MASK                      0xFF00000u
#define ENET_TXIC_ICFT_SHIFT                     20u
#define ENET_TXIC_ICFT_WIDTH                     8u
#define ENET_TXIC_ICFT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TXIC_ICFT_SHIFT))&ENET_TXIC_ICFT_MASK)
#define ENET_TXIC_ICCS_MASK                      0x40000000u
#define ENET_TXIC_ICCS_SHIFT                     30u
#define ENET_TXIC_ICCS_WIDTH                     1u
#define ENET_TXIC_ICCS(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TXIC_ICCS_SHIFT))&ENET_TXIC_ICCS_MASK)
#define ENET_TXIC_ICEN_MASK                      0x80000000u
#define ENET_TXIC_ICEN_SHIFT                     31u
#define ENET_TXIC_ICEN_WIDTH                     1u
#define ENET_TXIC_ICEN(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TXIC_ICEN_SHIFT))&ENET_TXIC_ICEN_MASK)
/* RXIC Bit Fields */
#define ENET_RXIC_ICTT_MASK                      0xFFFFu
#define ENET_RXIC_ICTT_SHIFT                     0u
#define ENET_RXIC_ICTT_WIDTH                     16u
#define ENET_RXIC_ICTT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RXIC_ICTT_SHIFT))&ENET_RXIC_ICTT_MASK)
#define ENET_RXIC_ICFT_MASK                      0xFF00000u
#define ENET_RXIC_ICFT_SHIFT                     20u
#define ENET_RXIC_ICFT_WIDTH                     8u
#define ENET_RXIC_ICFT(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RXIC_ICFT_SHIFT))&ENET_RXIC_ICFT_MASK)
#define ENET_RXIC_ICCS_MASK                      0x40000000u
#define ENET_RXIC_ICCS_SHIFT                     30u
#define ENET_RXIC_ICCS_WIDTH                     1u
#define ENET_RXIC_ICCS(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RXIC_ICCS_SHIFT))&ENET_RXIC_ICCS_MASK)
#define ENET_RXIC_ICEN_MASK                      0x80000000u
#define ENET_RXIC_ICEN_SHIFT                     31u
#define ENET_RXIC_ICEN_WIDTH                     1u
#define ENET_RXIC_ICEN(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RXIC_ICEN_SHIFT))&ENET_RXIC_ICEN_MASK)
/* IAUR Bit Fields */
#define ENET_IAUR_IADDR1_MASK                    0xFFFFFFFFu
#define ENET_IAUR_IADDR1_SHIFT                   0u
#define ENET_IAUR_IADDR1_WIDTH                   32u
#define ENET_IAUR_IADDR1(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_IAUR_IADDR1_SHIFT))&ENET_IAUR_IADDR1_MASK)
/* IALR Bit Fields */
#define ENET_IALR_IADDR2_MASK                    0xFFFFFFFFu
#define ENET_IALR_IADDR2_SHIFT                   0u
#define ENET_IALR_IADDR2_WIDTH                   32u
#define ENET_IALR_IADDR2(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_IALR_IADDR2_SHIFT))&ENET_IALR_IADDR2_MASK)
/* GAUR Bit Fields */
#define ENET_GAUR_GADDR1_MASK                    0xFFFFFFFFu
#define ENET_GAUR_GADDR1_SHIFT                   0u
#define ENET_GAUR_GADDR1_WIDTH                   32u
#define ENET_GAUR_GADDR1(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_GAUR_GADDR1_SHIFT))&ENET_GAUR_GADDR1_MASK)
/* GALR Bit Fields */
#define ENET_GALR_GADDR2_MASK                    0xFFFFFFFFu
#define ENET_GALR_GADDR2_SHIFT                   0u
#define ENET_GALR_GADDR2_WIDTH                   32u
#define ENET_GALR_GADDR2(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_GALR_GADDR2_SHIFT))&ENET_GALR_GADDR2_MASK)
/* TFWR Bit Fields */
#define ENET_TFWR_TFWR_MASK                      0x3Fu
#define ENET_TFWR_TFWR_SHIFT                     0u
#define ENET_TFWR_TFWR_WIDTH                     6u
#define ENET_TFWR_TFWR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TFWR_TFWR_SHIFT))&ENET_TFWR_TFWR_MASK)
#define ENET_TFWR_STRFWD_MASK                    0x100u
#define ENET_TFWR_STRFWD_SHIFT                   8u
#define ENET_TFWR_STRFWD_WIDTH                   1u
#define ENET_TFWR_STRFWD(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_TFWR_STRFWD_SHIFT))&ENET_TFWR_STRFWD_MASK)
/* RDSR1 Bit Fields */
#define ENET_RDSR1_R_DES_START_MASK              0xFFFFFFF8u
#define ENET_RDSR1_R_DES_START_SHIFT             3u
#define ENET_RDSR1_R_DES_START_WIDTH             29u
#define ENET_RDSR1_R_DES_START(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RDSR1_R_DES_START_SHIFT))&ENET_RDSR1_R_DES_START_MASK)
/* TDSR1 Bit Fields */
#define ENET_TDSR1_X_DES_START_MASK              0xFFFFFFF8u
#define ENET_TDSR1_X_DES_START_SHIFT             3u
#define ENET_TDSR1_X_DES_START_WIDTH             29u
#define ENET_TDSR1_X_DES_START(x)                (((uint32_t)(((uint32_t)(x))<<ENET_TDSR1_X_DES_START_SHIFT))&ENET_TDSR1_X_DES_START_MASK)
/* MRBR1 Bit Fields */
#define ENET_MRBR1_R_BUF_SIZE_MASK               0x3FF0u
#define ENET_MRBR1_R_BUF_SIZE_SHIFT              4u
#define ENET_MRBR1_R_BUF_SIZE_WIDTH              10u
#define ENET_MRBR1_R_BUF_SIZE(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_MRBR1_R_BUF_SIZE_SHIFT))&ENET_MRBR1_R_BUF_SIZE_MASK)
/* RDSR2 Bit Fields */
#define ENET_RDSR2_R_DES_START_MASK              0xFFFFFFF8u
#define ENET_RDSR2_R_DES_START_SHIFT             3u
#define ENET_RDSR2_R_DES_START_WIDTH             29u
#define ENET_RDSR2_R_DES_START(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RDSR2_R_DES_START_SHIFT))&ENET_RDSR2_R_DES_START_MASK)
/* TDSR2 Bit Fields */
#define ENET_TDSR2_X_DES_START_MASK              0xFFFFFFF8u
#define ENET_TDSR2_X_DES_START_SHIFT             3u
#define ENET_TDSR2_X_DES_START_WIDTH             29u
#define ENET_TDSR2_X_DES_START(x)                (((uint32_t)(((uint32_t)(x))<<ENET_TDSR2_X_DES_START_SHIFT))&ENET_TDSR2_X_DES_START_MASK)
/* MRBR2 Bit Fields */
#define ENET_MRBR2_R_BUF_SIZE_MASK               0x3FF0u
#define ENET_MRBR2_R_BUF_SIZE_SHIFT              4u
#define ENET_MRBR2_R_BUF_SIZE_WIDTH              10u
#define ENET_MRBR2_R_BUF_SIZE(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_MRBR2_R_BUF_SIZE_SHIFT))&ENET_MRBR2_R_BUF_SIZE_MASK)
/* RDSR Bit Fields */
#define ENET_RDSR_R_DES_START_MASK               0xFFFFFFF8u
#define ENET_RDSR_R_DES_START_SHIFT              3u
#define ENET_RDSR_R_DES_START_WIDTH              29u
#define ENET_RDSR_R_DES_START(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RDSR_R_DES_START_SHIFT))&ENET_RDSR_R_DES_START_MASK)
/* TDSR Bit Fields */
#define ENET_TDSR_X_DES_START_MASK               0xFFFFFFF8u
#define ENET_TDSR_X_DES_START_SHIFT              3u
#define ENET_TDSR_X_DES_START_WIDTH              29u
#define ENET_TDSR_X_DES_START(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_TDSR_X_DES_START_SHIFT))&ENET_TDSR_X_DES_START_MASK)
/* MRBR Bit Fields */
#define ENET_MRBR_R_BUF_SIZE_MASK                0x3FF0u
#define ENET_MRBR_R_BUF_SIZE_SHIFT               4u
#define ENET_MRBR_R_BUF_SIZE_WIDTH               10u
#define ENET_MRBR_R_BUF_SIZE(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_MRBR_R_BUF_SIZE_SHIFT))&ENET_MRBR_R_BUF_SIZE_MASK)
/* RSFL Bit Fields */
#define ENET_RSFL_RX_SECTION_FULL_MASK           0xFFu
#define ENET_RSFL_RX_SECTION_FULL_SHIFT          0u
#define ENET_RSFL_RX_SECTION_FULL_WIDTH          8u
#define ENET_RSFL_RX_SECTION_FULL(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RSFL_RX_SECTION_FULL_SHIFT))&ENET_RSFL_RX_SECTION_FULL_MASK)
/* RSEM Bit Fields */
#define ENET_RSEM_RX_SECTION_EMPTY_MASK          0xFFu
#define ENET_RSEM_RX_SECTION_EMPTY_SHIFT         0u
#define ENET_RSEM_RX_SECTION_EMPTY_WIDTH         8u
#define ENET_RSEM_RX_SECTION_EMPTY(x)            (((uint32_t)(((uint32_t)(x))<<ENET_RSEM_RX_SECTION_EMPTY_SHIFT))&ENET_RSEM_RX_SECTION_EMPTY_MASK)
#define ENET_RSEM_STAT_SECTION_EMPTY_MASK        0x1F0000u
#define ENET_RSEM_STAT_SECTION_EMPTY_SHIFT       16u
#define ENET_RSEM_STAT_SECTION_EMPTY_WIDTH       5u
#define ENET_RSEM_STAT_SECTION_EMPTY(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RSEM_STAT_SECTION_EMPTY_SHIFT))&ENET_RSEM_STAT_SECTION_EMPTY_MASK)
/* RAEM Bit Fields */
#define ENET_RAEM_RX_ALMOST_EMPTY_MASK           0xFFu
#define ENET_RAEM_RX_ALMOST_EMPTY_SHIFT          0u
#define ENET_RAEM_RX_ALMOST_EMPTY_WIDTH          8u
#define ENET_RAEM_RX_ALMOST_EMPTY(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RAEM_RX_ALMOST_EMPTY_SHIFT))&ENET_RAEM_RX_ALMOST_EMPTY_MASK)
/* RAFL Bit Fields */
#define ENET_RAFL_RX_ALMOST_FULL_MASK            0xFFu
#define ENET_RAFL_RX_ALMOST_FULL_SHIFT           0u
#define ENET_RAFL_RX_ALMOST_FULL_WIDTH           8u
#define ENET_RAFL_RX_ALMOST_FULL(x)              (((uint32_t)(((uint32_t)(x))<<ENET_RAFL_RX_ALMOST_FULL_SHIFT))&ENET_RAFL_RX_ALMOST_FULL_MASK)
/* TSEM Bit Fields */
#define ENET_TSEM_TX_SECTION_EMPTY_MASK          0xFFu
#define ENET_TSEM_TX_SECTION_EMPTY_SHIFT         0u
#define ENET_TSEM_TX_SECTION_EMPTY_WIDTH         8u
#define ENET_TSEM_TX_SECTION_EMPTY(x)            (((uint32_t)(((uint32_t)(x))<<ENET_TSEM_TX_SECTION_EMPTY_SHIFT))&ENET_TSEM_TX_SECTION_EMPTY_MASK)
/* TAEM Bit Fields */
#define ENET_TAEM_TX_ALMOST_EMPTY_MASK           0xFFu
#define ENET_TAEM_TX_ALMOST_EMPTY_SHIFT          0u
#define ENET_TAEM_TX_ALMOST_EMPTY_WIDTH          8u
#define ENET_TAEM_TX_ALMOST_EMPTY(x)             (((uint32_t)(((uint32_t)(x))<<ENET_TAEM_TX_ALMOST_EMPTY_SHIFT))&ENET_TAEM_TX_ALMOST_EMPTY_MASK)
/* TAFL Bit Fields */
#define ENET_TAFL_TX_ALMOST_FULL_MASK            0xFFu
#define ENET_TAFL_TX_ALMOST_FULL_SHIFT           0u
#define ENET_TAFL_TX_ALMOST_FULL_WIDTH           8u
#define ENET_TAFL_TX_ALMOST_FULL(x)              (((uint32_t)(((uint32_t)(x))<<ENET_TAFL_TX_ALMOST_FULL_SHIFT))&ENET_TAFL_TX_ALMOST_FULL_MASK)
/* TIPG Bit Fields */
#define ENET_TIPG_IPG_MASK                       0x1Fu
#define ENET_TIPG_IPG_SHIFT                      0u
#define ENET_TIPG_IPG_WIDTH                      5u
#define ENET_TIPG_IPG(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TIPG_IPG_SHIFT))&ENET_TIPG_IPG_MASK)
/* FTRL Bit Fields */
#define ENET_FTRL_TRUNC_FL_MASK                  0x3FFFu
#define ENET_FTRL_TRUNC_FL_SHIFT                 0u
#define ENET_FTRL_TRUNC_FL_WIDTH                 14u
#define ENET_FTRL_TRUNC_FL(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_FTRL_TRUNC_FL_SHIFT))&ENET_FTRL_TRUNC_FL_MASK)
/* TACC Bit Fields */
#define ENET_TACC_SHIFT16_MASK                   0x1u
#define ENET_TACC_SHIFT16_SHIFT                  0u
#define ENET_TACC_SHIFT16_WIDTH                  1u
#define ENET_TACC_SHIFT16(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_TACC_SHIFT16_SHIFT))&ENET_TACC_SHIFT16_MASK)
#define ENET_TACC_IPCHK_MASK                     0x8u
#define ENET_TACC_IPCHK_SHIFT                    3u
#define ENET_TACC_IPCHK_WIDTH                    1u
#define ENET_TACC_IPCHK(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TACC_IPCHK_SHIFT))&ENET_TACC_IPCHK_MASK)
#define ENET_TACC_PROCHK_MASK                    0x10u
#define ENET_TACC_PROCHK_SHIFT                   4u
#define ENET_TACC_PROCHK_WIDTH                   1u
#define ENET_TACC_PROCHK(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_TACC_PROCHK_SHIFT))&ENET_TACC_PROCHK_MASK)
/* RACC Bit Fields */
#define ENET_RACC_PADREM_MASK                    0x1u
#define ENET_RACC_PADREM_SHIFT                   0u
#define ENET_RACC_PADREM_WIDTH                   1u
#define ENET_RACC_PADREM(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_RACC_PADREM_SHIFT))&ENET_RACC_PADREM_MASK)
#define ENET_RACC_IPDIS_MASK                     0x2u
#define ENET_RACC_IPDIS_SHIFT                    1u
#define ENET_RACC_IPDIS_WIDTH                    1u
#define ENET_RACC_IPDIS(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RACC_IPDIS_SHIFT))&ENET_RACC_IPDIS_MASK)
#define ENET_RACC_PRODIS_MASK                    0x4u
#define ENET_RACC_PRODIS_SHIFT                   2u
#define ENET_RACC_PRODIS_WIDTH                   1u
#define ENET_RACC_PRODIS(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_RACC_PRODIS_SHIFT))&ENET_RACC_PRODIS_MASK)
#define ENET_RACC_LINEDIS_MASK                   0x40u
#define ENET_RACC_LINEDIS_SHIFT                  6u
#define ENET_RACC_LINEDIS_WIDTH                  1u
#define ENET_RACC_LINEDIS(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RACC_LINEDIS_SHIFT))&ENET_RACC_LINEDIS_MASK)
#define ENET_RACC_SHIFT16_MASK                   0x80u
#define ENET_RACC_SHIFT16_SHIFT                  7u
#define ENET_RACC_SHIFT16_WIDTH                  1u
#define ENET_RACC_SHIFT16(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RACC_SHIFT16_SHIFT))&ENET_RACC_SHIFT16_MASK)
/* RCMR Bit Fields */
#define ENET_RCMR_CMP0_MASK                      0x7u
#define ENET_RCMR_CMP0_SHIFT                     0u
#define ENET_RCMR_CMP0_WIDTH                     3u
#define ENET_RCMR_CMP0(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_CMP0_SHIFT))&ENET_RCMR_CMP0_MASK)
#define ENET_RCMR_CMP1_MASK                      0x70u
#define ENET_RCMR_CMP1_SHIFT                     4u
#define ENET_RCMR_CMP1_WIDTH                     3u
#define ENET_RCMR_CMP1(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_CMP1_SHIFT))&ENET_RCMR_CMP1_MASK)
#define ENET_RCMR_CMP2_MASK                      0x700u
#define ENET_RCMR_CMP2_SHIFT                     8u
#define ENET_RCMR_CMP2_WIDTH                     3u
#define ENET_RCMR_CMP2(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_CMP2_SHIFT))&ENET_RCMR_CMP2_MASK)
#define ENET_RCMR_CMP3_MASK                      0x7000u
#define ENET_RCMR_CMP3_SHIFT                     12u
#define ENET_RCMR_CMP3_WIDTH                     3u
#define ENET_RCMR_CMP3(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_CMP3_SHIFT))&ENET_RCMR_CMP3_MASK)
#define ENET_RCMR_MATCHEN_MASK                   0x10000u
#define ENET_RCMR_MATCHEN_SHIFT                  16u
#define ENET_RCMR_MATCHEN_WIDTH                  1u
#define ENET_RCMR_MATCHEN(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_RCMR_MATCHEN_SHIFT))&ENET_RCMR_MATCHEN_MASK)
/* DMACFG Bit Fields */
#define ENET_DMACFG_IDLE_SLOPE_MASK              0xFFFFu
#define ENET_DMACFG_IDLE_SLOPE_SHIFT             0u
#define ENET_DMACFG_IDLE_SLOPE_WIDTH             16u
#define ENET_DMACFG_IDLE_SLOPE(x)                (((uint32_t)(((uint32_t)(x))<<ENET_DMACFG_IDLE_SLOPE_SHIFT))&ENET_DMACFG_IDLE_SLOPE_MASK)
#define ENET_DMACFG_DMA_CLASS_EN_MASK            0x10000u
#define ENET_DMACFG_DMA_CLASS_EN_SHIFT           16u
#define ENET_DMACFG_DMA_CLASS_EN_WIDTH           1u
#define ENET_DMACFG_DMA_CLASS_EN(x)              (((uint32_t)(((uint32_t)(x))<<ENET_DMACFG_DMA_CLASS_EN_SHIFT))&ENET_DMACFG_DMA_CLASS_EN_MASK)
#define ENET_DMACFG_CALC_NOIPG_MASK              0x20000u
#define ENET_DMACFG_CALC_NOIPG_SHIFT             17u
#define ENET_DMACFG_CALC_NOIPG_WIDTH             1u
#define ENET_DMACFG_CALC_NOIPG(x)                (((uint32_t)(((uint32_t)(x))<<ENET_DMACFG_CALC_NOIPG_SHIFT))&ENET_DMACFG_CALC_NOIPG_MASK)
/* RDAR1 Bit Fields */
#define ENET_RDAR1_RDAR_MASK                     0x1000000u
#define ENET_RDAR1_RDAR_SHIFT                    24u
#define ENET_RDAR1_RDAR_WIDTH                    1u
#define ENET_RDAR1_RDAR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RDAR1_RDAR_SHIFT))&ENET_RDAR1_RDAR_MASK)
/* TDAR1 Bit Fields */
#define ENET_TDAR1_TDAR_MASK                     0x1000000u
#define ENET_TDAR1_TDAR_SHIFT                    24u
#define ENET_TDAR1_TDAR_WIDTH                    1u
#define ENET_TDAR1_TDAR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TDAR1_TDAR_SHIFT))&ENET_TDAR1_TDAR_MASK)
/* RDAR2 Bit Fields */
#define ENET_RDAR2_RDAR_MASK                     0x1000000u
#define ENET_RDAR2_RDAR_SHIFT                    24u
#define ENET_RDAR2_RDAR_WIDTH                    1u
#define ENET_RDAR2_RDAR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_RDAR2_RDAR_SHIFT))&ENET_RDAR2_RDAR_MASK)
/* TDAR2 Bit Fields */
#define ENET_TDAR2_TDAR_MASK                     0x1000000u
#define ENET_TDAR2_TDAR_SHIFT                    24u
#define ENET_TDAR2_TDAR_WIDTH                    1u
#define ENET_TDAR2_TDAR(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TDAR2_TDAR_SHIFT))&ENET_TDAR2_TDAR_MASK)
/* QOS Bit Fields */
#define ENET_QOS_TX_SCHEME_MASK                  0x7u
#define ENET_QOS_TX_SCHEME_SHIFT                 0u
#define ENET_QOS_TX_SCHEME_WIDTH                 3u
#define ENET_QOS_TX_SCHEME(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_QOS_TX_SCHEME_SHIFT))&ENET_QOS_TX_SCHEME_MASK)
#define ENET_QOS_RX_FLUSH0_MASK                  0x8u
#define ENET_QOS_RX_FLUSH0_SHIFT                 3u
#define ENET_QOS_RX_FLUSH0_WIDTH                 1u
#define ENET_QOS_RX_FLUSH0(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_QOS_RX_FLUSH0_SHIFT))&ENET_QOS_RX_FLUSH0_MASK)
#define ENET_QOS_RX_FLUSH1_MASK                  0x10u
#define ENET_QOS_RX_FLUSH1_SHIFT                 4u
#define ENET_QOS_RX_FLUSH1_WIDTH                 1u
#define ENET_QOS_RX_FLUSH1(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_QOS_RX_FLUSH1_SHIFT))&ENET_QOS_RX_FLUSH1_MASK)
#define ENET_QOS_RX_FLUSH2_MASK                  0x20u
#define ENET_QOS_RX_FLUSH2_SHIFT                 5u
#define ENET_QOS_RX_FLUSH2_WIDTH                 1u
#define ENET_QOS_RX_FLUSH2(x)                    (((uint32_t)(((uint32_t)(x))<<ENET_QOS_RX_FLUSH2_SHIFT))&ENET_QOS_RX_FLUSH2_MASK)
/* RMON_T_PACKETS Bit Fields */
#define ENET_RMON_T_PACKETS_TXPKTS_MASK          0xFFFFu
#define ENET_RMON_T_PACKETS_TXPKTS_SHIFT         0u
#define ENET_RMON_T_PACKETS_TXPKTS_WIDTH         16u
#define ENET_RMON_T_PACKETS_TXPKTS(x)            (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_PACKETS_TXPKTS_SHIFT))&ENET_RMON_T_PACKETS_TXPKTS_MASK)
/* RMON_T_BC_PKT Bit Fields */
#define ENET_RMON_T_BC_PKT_TXPKTS_MASK           0xFFFFu
#define ENET_RMON_T_BC_PKT_TXPKTS_SHIFT          0u
#define ENET_RMON_T_BC_PKT_TXPKTS_WIDTH          16u
#define ENET_RMON_T_BC_PKT_TXPKTS(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_BC_PKT_TXPKTS_SHIFT))&ENET_RMON_T_BC_PKT_TXPKTS_MASK)
/* RMON_T_MC_PKT Bit Fields */
#define ENET_RMON_T_MC_PKT_TXPKTS_MASK           0xFFFFu
#define ENET_RMON_T_MC_PKT_TXPKTS_SHIFT          0u
#define ENET_RMON_T_MC_PKT_TXPKTS_WIDTH          16u
#define ENET_RMON_T_MC_PKT_TXPKTS(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_MC_PKT_TXPKTS_SHIFT))&ENET_RMON_T_MC_PKT_TXPKTS_MASK)
/* RMON_T_CRC_ALIGN Bit Fields */
#define ENET_RMON_T_CRC_ALIGN_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_CRC_ALIGN_TXPKTS_SHIFT       0u
#define ENET_RMON_T_CRC_ALIGN_TXPKTS_WIDTH       16u
#define ENET_RMON_T_CRC_ALIGN_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_CRC_ALIGN_TXPKTS_SHIFT))&ENET_RMON_T_CRC_ALIGN_TXPKTS_MASK)
/* RMON_T_UNDERSIZE Bit Fields */
#define ENET_RMON_T_UNDERSIZE_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_UNDERSIZE_TXPKTS_SHIFT       0u
#define ENET_RMON_T_UNDERSIZE_TXPKTS_WIDTH       16u
#define ENET_RMON_T_UNDERSIZE_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_UNDERSIZE_TXPKTS_SHIFT))&ENET_RMON_T_UNDERSIZE_TXPKTS_MASK)
/* RMON_T_OVERSIZE Bit Fields */
#define ENET_RMON_T_OVERSIZE_TXPKTS_MASK         0xFFFFu
#define ENET_RMON_T_OVERSIZE_TXPKTS_SHIFT        0u
#define ENET_RMON_T_OVERSIZE_TXPKTS_WIDTH        16u
#define ENET_RMON_T_OVERSIZE_TXPKTS(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_OVERSIZE_TXPKTS_SHIFT))&ENET_RMON_T_OVERSIZE_TXPKTS_MASK)
/* RMON_T_FRAG Bit Fields */
#define ENET_RMON_T_FRAG_TXPKTS_MASK             0xFFFFu
#define ENET_RMON_T_FRAG_TXPKTS_SHIFT            0u
#define ENET_RMON_T_FRAG_TXPKTS_WIDTH            16u
#define ENET_RMON_T_FRAG_TXPKTS(x)               (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_FRAG_TXPKTS_SHIFT))&ENET_RMON_T_FRAG_TXPKTS_MASK)
/* RMON_T_JAB Bit Fields */
#define ENET_RMON_T_JAB_TXPKTS_MASK              0xFFFFu
#define ENET_RMON_T_JAB_TXPKTS_SHIFT             0u
#define ENET_RMON_T_JAB_TXPKTS_WIDTH             16u
#define ENET_RMON_T_JAB_TXPKTS(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_JAB_TXPKTS_SHIFT))&ENET_RMON_T_JAB_TXPKTS_MASK)
/* RMON_T_COL Bit Fields */
#define ENET_RMON_T_COL_TXPKTS_MASK              0xFFFFu
#define ENET_RMON_T_COL_TXPKTS_SHIFT             0u
#define ENET_RMON_T_COL_TXPKTS_WIDTH             16u
#define ENET_RMON_T_COL_TXPKTS(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_COL_TXPKTS_SHIFT))&ENET_RMON_T_COL_TXPKTS_MASK)
/* RMON_T_P64 Bit Fields */
#define ENET_RMON_T_P64_TXPKTS_MASK              0xFFFFu
#define ENET_RMON_T_P64_TXPKTS_SHIFT             0u
#define ENET_RMON_T_P64_TXPKTS_WIDTH             16u
#define ENET_RMON_T_P64_TXPKTS(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P64_TXPKTS_SHIFT))&ENET_RMON_T_P64_TXPKTS_MASK)
/* RMON_T_P65TO127 Bit Fields */
#define ENET_RMON_T_P65TO127_TXPKTS_MASK         0xFFFFu
#define ENET_RMON_T_P65TO127_TXPKTS_SHIFT        0u
#define ENET_RMON_T_P65TO127_TXPKTS_WIDTH        16u
#define ENET_RMON_T_P65TO127_TXPKTS(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P65TO127_TXPKTS_SHIFT))&ENET_RMON_T_P65TO127_TXPKTS_MASK)
/* RMON_T_P128TO255 Bit Fields */
#define ENET_RMON_T_P128TO255_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_P128TO255_TXPKTS_SHIFT       0u
#define ENET_RMON_T_P128TO255_TXPKTS_WIDTH       16u
#define ENET_RMON_T_P128TO255_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P128TO255_TXPKTS_SHIFT))&ENET_RMON_T_P128TO255_TXPKTS_MASK)
/* RMON_T_P256TO511 Bit Fields */
#define ENET_RMON_T_P256TO511_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_P256TO511_TXPKTS_SHIFT       0u
#define ENET_RMON_T_P256TO511_TXPKTS_WIDTH       16u
#define ENET_RMON_T_P256TO511_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P256TO511_TXPKTS_SHIFT))&ENET_RMON_T_P256TO511_TXPKTS_MASK)
/* RMON_T_P512TO1023 Bit Fields */
#define ENET_RMON_T_P512TO1023_TXPKTS_MASK       0xFFFFu
#define ENET_RMON_T_P512TO1023_TXPKTS_SHIFT      0u
#define ENET_RMON_T_P512TO1023_TXPKTS_WIDTH      16u
#define ENET_RMON_T_P512TO1023_TXPKTS(x)         (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P512TO1023_TXPKTS_SHIFT))&ENET_RMON_T_P512TO1023_TXPKTS_MASK)
/* RMON_T_P1024TO2047 Bit Fields */
#define ENET_RMON_T_P1024TO2047_TXPKTS_MASK      0xFFFFu
#define ENET_RMON_T_P1024TO2047_TXPKTS_SHIFT     0u
#define ENET_RMON_T_P1024TO2047_TXPKTS_WIDTH     16u
#define ENET_RMON_T_P1024TO2047_TXPKTS(x)        (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P1024TO2047_TXPKTS_SHIFT))&ENET_RMON_T_P1024TO2047_TXPKTS_MASK)
/* RMON_T_P_GTE2048 Bit Fields */
#define ENET_RMON_T_P_GTE2048_TXPKTS_MASK        0xFFFFu
#define ENET_RMON_T_P_GTE2048_TXPKTS_SHIFT       0u
#define ENET_RMON_T_P_GTE2048_TXPKTS_WIDTH       16u
#define ENET_RMON_T_P_GTE2048_TXPKTS(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_P_GTE2048_TXPKTS_SHIFT))&ENET_RMON_T_P_GTE2048_TXPKTS_MASK)
/* RMON_T_OCTETS Bit Fields */
#define ENET_RMON_T_OCTETS_TXOCTS_MASK           0xFFFFFFFFu
#define ENET_RMON_T_OCTETS_TXOCTS_SHIFT          0u
#define ENET_RMON_T_OCTETS_TXOCTS_WIDTH          32u
#define ENET_RMON_T_OCTETS_TXOCTS(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RMON_T_OCTETS_TXOCTS_SHIFT))&ENET_RMON_T_OCTETS_TXOCTS_MASK)
/* IEEE_T_FRAME_OK Bit Fields */
#define ENET_IEEE_T_FRAME_OK_COUNT_MASK          0xFFFFu
#define ENET_IEEE_T_FRAME_OK_COUNT_SHIFT         0u
#define ENET_IEEE_T_FRAME_OK_COUNT_WIDTH         16u
#define ENET_IEEE_T_FRAME_OK_COUNT(x)            (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_FRAME_OK_COUNT_SHIFT))&ENET_IEEE_T_FRAME_OK_COUNT_MASK)
/* IEEE_T_1COL Bit Fields */
#define ENET_IEEE_T_1COL_COUNT_MASK              0xFFFFu
#define ENET_IEEE_T_1COL_COUNT_SHIFT             0u
#define ENET_IEEE_T_1COL_COUNT_WIDTH             16u
#define ENET_IEEE_T_1COL_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_1COL_COUNT_SHIFT))&ENET_IEEE_T_1COL_COUNT_MASK)
/* IEEE_T_MCOL Bit Fields */
#define ENET_IEEE_T_MCOL_COUNT_MASK              0xFFFFu
#define ENET_IEEE_T_MCOL_COUNT_SHIFT             0u
#define ENET_IEEE_T_MCOL_COUNT_WIDTH             16u
#define ENET_IEEE_T_MCOL_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_MCOL_COUNT_SHIFT))&ENET_IEEE_T_MCOL_COUNT_MASK)
/* IEEE_T_DEF Bit Fields */
#define ENET_IEEE_T_DEF_COUNT_MASK               0xFFFFu
#define ENET_IEEE_T_DEF_COUNT_SHIFT              0u
#define ENET_IEEE_T_DEF_COUNT_WIDTH              16u
#define ENET_IEEE_T_DEF_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_DEF_COUNT_SHIFT))&ENET_IEEE_T_DEF_COUNT_MASK)
/* IEEE_T_LCOL Bit Fields */
#define ENET_IEEE_T_LCOL_COUNT_MASK              0xFFFFu
#define ENET_IEEE_T_LCOL_COUNT_SHIFT             0u
#define ENET_IEEE_T_LCOL_COUNT_WIDTH             16u
#define ENET_IEEE_T_LCOL_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_LCOL_COUNT_SHIFT))&ENET_IEEE_T_LCOL_COUNT_MASK)
/* IEEE_T_EXCOL Bit Fields */
#define ENET_IEEE_T_EXCOL_COUNT_MASK             0xFFFFu
#define ENET_IEEE_T_EXCOL_COUNT_SHIFT            0u
#define ENET_IEEE_T_EXCOL_COUNT_WIDTH            16u
#define ENET_IEEE_T_EXCOL_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_EXCOL_COUNT_SHIFT))&ENET_IEEE_T_EXCOL_COUNT_MASK)
/* IEEE_T_MACERR Bit Fields */
#define ENET_IEEE_T_MACERR_COUNT_MASK            0xFFFFu
#define ENET_IEEE_T_MACERR_COUNT_SHIFT           0u
#define ENET_IEEE_T_MACERR_COUNT_WIDTH           16u
#define ENET_IEEE_T_MACERR_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_MACERR_COUNT_SHIFT))&ENET_IEEE_T_MACERR_COUNT_MASK)
/* IEEE_T_CSERR Bit Fields */
#define ENET_IEEE_T_CSERR_COUNT_MASK             0xFFFFu
#define ENET_IEEE_T_CSERR_COUNT_SHIFT            0u
#define ENET_IEEE_T_CSERR_COUNT_WIDTH            16u
#define ENET_IEEE_T_CSERR_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_CSERR_COUNT_SHIFT))&ENET_IEEE_T_CSERR_COUNT_MASK)
/* IEEE_T_SQE Bit Fields */
#define ENET_IEEE_T_SQE_COUNT_MASK               0xFFFFu
#define ENET_IEEE_T_SQE_COUNT_SHIFT              0u
#define ENET_IEEE_T_SQE_COUNT_WIDTH              16u
#define ENET_IEEE_T_SQE_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_SQE_COUNT_SHIFT))&ENET_IEEE_T_SQE_COUNT_MASK)
/* IEEE_T_FDXFC Bit Fields */
#define ENET_IEEE_T_FDXFC_COUNT_MASK             0xFFFFu
#define ENET_IEEE_T_FDXFC_COUNT_SHIFT            0u
#define ENET_IEEE_T_FDXFC_COUNT_WIDTH            16u
#define ENET_IEEE_T_FDXFC_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_FDXFC_COUNT_SHIFT))&ENET_IEEE_T_FDXFC_COUNT_MASK)
/* IEEE_T_OCTETS_OK Bit Fields */
#define ENET_IEEE_T_OCTETS_OK_COUNT_MASK         0xFFFFFFFFu
#define ENET_IEEE_T_OCTETS_OK_COUNT_SHIFT        0u
#define ENET_IEEE_T_OCTETS_OK_COUNT_WIDTH        32u
#define ENET_IEEE_T_OCTETS_OK_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_T_OCTETS_OK_COUNT_SHIFT))&ENET_IEEE_T_OCTETS_OK_COUNT_MASK)
/* RMON_R_PACKETS Bit Fields */
#define ENET_RMON_R_PACKETS_COUNT_MASK           0xFFFFu
#define ENET_RMON_R_PACKETS_COUNT_SHIFT          0u
#define ENET_RMON_R_PACKETS_COUNT_WIDTH          16u
#define ENET_RMON_R_PACKETS_COUNT(x)             (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_PACKETS_COUNT_SHIFT))&ENET_RMON_R_PACKETS_COUNT_MASK)
/* RMON_R_BC_PKT Bit Fields */
#define ENET_RMON_R_BC_PKT_COUNT_MASK            0xFFFFu
#define ENET_RMON_R_BC_PKT_COUNT_SHIFT           0u
#define ENET_RMON_R_BC_PKT_COUNT_WIDTH           16u
#define ENET_RMON_R_BC_PKT_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_BC_PKT_COUNT_SHIFT))&ENET_RMON_R_BC_PKT_COUNT_MASK)
/* RMON_R_MC_PKT Bit Fields */
#define ENET_RMON_R_MC_PKT_COUNT_MASK            0xFFFFu
#define ENET_RMON_R_MC_PKT_COUNT_SHIFT           0u
#define ENET_RMON_R_MC_PKT_COUNT_WIDTH           16u
#define ENET_RMON_R_MC_PKT_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_MC_PKT_COUNT_SHIFT))&ENET_RMON_R_MC_PKT_COUNT_MASK)
/* RMON_R_CRC_ALIGN Bit Fields */
#define ENET_RMON_R_CRC_ALIGN_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_CRC_ALIGN_COUNT_SHIFT        0u
#define ENET_RMON_R_CRC_ALIGN_COUNT_WIDTH        16u
#define ENET_RMON_R_CRC_ALIGN_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_CRC_ALIGN_COUNT_SHIFT))&ENET_RMON_R_CRC_ALIGN_COUNT_MASK)
/* RMON_R_UNDERSIZE Bit Fields */
#define ENET_RMON_R_UNDERSIZE_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_UNDERSIZE_COUNT_SHIFT        0u
#define ENET_RMON_R_UNDERSIZE_COUNT_WIDTH        16u
#define ENET_RMON_R_UNDERSIZE_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_UNDERSIZE_COUNT_SHIFT))&ENET_RMON_R_UNDERSIZE_COUNT_MASK)
/* RMON_R_OVERSIZE Bit Fields */
#define ENET_RMON_R_OVERSIZE_COUNT_MASK          0xFFFFu
#define ENET_RMON_R_OVERSIZE_COUNT_SHIFT         0u
#define ENET_RMON_R_OVERSIZE_COUNT_WIDTH         16u
#define ENET_RMON_R_OVERSIZE_COUNT(x)            (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_OVERSIZE_COUNT_SHIFT))&ENET_RMON_R_OVERSIZE_COUNT_MASK)
/* RMON_R_FRAG Bit Fields */
#define ENET_RMON_R_FRAG_COUNT_MASK              0xFFFFu
#define ENET_RMON_R_FRAG_COUNT_SHIFT             0u
#define ENET_RMON_R_FRAG_COUNT_WIDTH             16u
#define ENET_RMON_R_FRAG_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_FRAG_COUNT_SHIFT))&ENET_RMON_R_FRAG_COUNT_MASK)
/* RMON_R_JAB Bit Fields */
#define ENET_RMON_R_JAB_COUNT_MASK               0xFFFFu
#define ENET_RMON_R_JAB_COUNT_SHIFT              0u
#define ENET_RMON_R_JAB_COUNT_WIDTH              16u
#define ENET_RMON_R_JAB_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_JAB_COUNT_SHIFT))&ENET_RMON_R_JAB_COUNT_MASK)
/* RMON_R_P64 Bit Fields */
#define ENET_RMON_R_P64_COUNT_MASK               0xFFFFu
#define ENET_RMON_R_P64_COUNT_SHIFT              0u
#define ENET_RMON_R_P64_COUNT_WIDTH              16u
#define ENET_RMON_R_P64_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P64_COUNT_SHIFT))&ENET_RMON_R_P64_COUNT_MASK)
/* RMON_R_P65TO127 Bit Fields */
#define ENET_RMON_R_P65TO127_COUNT_MASK          0xFFFFu
#define ENET_RMON_R_P65TO127_COUNT_SHIFT         0u
#define ENET_RMON_R_P65TO127_COUNT_WIDTH         16u
#define ENET_RMON_R_P65TO127_COUNT(x)            (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P65TO127_COUNT_SHIFT))&ENET_RMON_R_P65TO127_COUNT_MASK)
/* RMON_R_P128TO255 Bit Fields */
#define ENET_RMON_R_P128TO255_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_P128TO255_COUNT_SHIFT        0u
#define ENET_RMON_R_P128TO255_COUNT_WIDTH        16u
#define ENET_RMON_R_P128TO255_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P128TO255_COUNT_SHIFT))&ENET_RMON_R_P128TO255_COUNT_MASK)
/* RMON_R_P256TO511 Bit Fields */
#define ENET_RMON_R_P256TO511_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_P256TO511_COUNT_SHIFT        0u
#define ENET_RMON_R_P256TO511_COUNT_WIDTH        16u
#define ENET_RMON_R_P256TO511_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P256TO511_COUNT_SHIFT))&ENET_RMON_R_P256TO511_COUNT_MASK)
/* RMON_R_P512TO1023 Bit Fields */
#define ENET_RMON_R_P512TO1023_COUNT_MASK        0xFFFFu
#define ENET_RMON_R_P512TO1023_COUNT_SHIFT       0u
#define ENET_RMON_R_P512TO1023_COUNT_WIDTH       16u
#define ENET_RMON_R_P512TO1023_COUNT(x)          (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P512TO1023_COUNT_SHIFT))&ENET_RMON_R_P512TO1023_COUNT_MASK)
/* RMON_R_P1024TO2047 Bit Fields */
#define ENET_RMON_R_P1024TO2047_COUNT_MASK       0xFFFFu
#define ENET_RMON_R_P1024TO2047_COUNT_SHIFT      0u
#define ENET_RMON_R_P1024TO2047_COUNT_WIDTH      16u
#define ENET_RMON_R_P1024TO2047_COUNT(x)         (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P1024TO2047_COUNT_SHIFT))&ENET_RMON_R_P1024TO2047_COUNT_MASK)
/* RMON_R_P_GTE2048 Bit Fields */
#define ENET_RMON_R_P_GTE2048_COUNT_MASK         0xFFFFu
#define ENET_RMON_R_P_GTE2048_COUNT_SHIFT        0u
#define ENET_RMON_R_P_GTE2048_COUNT_WIDTH        16u
#define ENET_RMON_R_P_GTE2048_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_P_GTE2048_COUNT_SHIFT))&ENET_RMON_R_P_GTE2048_COUNT_MASK)
/* RMON_R_OCTETS Bit Fields */
#define ENET_RMON_R_OCTETS_COUNT_MASK            0xFFFFFFFFu
#define ENET_RMON_R_OCTETS_COUNT_SHIFT           0u
#define ENET_RMON_R_OCTETS_COUNT_WIDTH           32u
#define ENET_RMON_R_OCTETS_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_RMON_R_OCTETS_COUNT_SHIFT))&ENET_RMON_R_OCTETS_COUNT_MASK)
/* IEEE_R_DROP Bit Fields */
#define ENET_IEEE_R_DROP_COUNT_MASK              0xFFFFu
#define ENET_IEEE_R_DROP_COUNT_SHIFT             0u
#define ENET_IEEE_R_DROP_COUNT_WIDTH             16u
#define ENET_IEEE_R_DROP_COUNT(x)                (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_DROP_COUNT_SHIFT))&ENET_IEEE_R_DROP_COUNT_MASK)
/* IEEE_R_FRAME_OK Bit Fields */
#define ENET_IEEE_R_FRAME_OK_COUNT_MASK          0xFFFFu
#define ENET_IEEE_R_FRAME_OK_COUNT_SHIFT         0u
#define ENET_IEEE_R_FRAME_OK_COUNT_WIDTH         16u
#define ENET_IEEE_R_FRAME_OK_COUNT(x)            (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_FRAME_OK_COUNT_SHIFT))&ENET_IEEE_R_FRAME_OK_COUNT_MASK)
/* IEEE_R_CRC Bit Fields */
#define ENET_IEEE_R_CRC_COUNT_MASK               0xFFFFu
#define ENET_IEEE_R_CRC_COUNT_SHIFT              0u
#define ENET_IEEE_R_CRC_COUNT_WIDTH              16u
#define ENET_IEEE_R_CRC_COUNT(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_CRC_COUNT_SHIFT))&ENET_IEEE_R_CRC_COUNT_MASK)
/* IEEE_R_ALIGN Bit Fields */
#define ENET_IEEE_R_ALIGN_COUNT_MASK             0xFFFFu
#define ENET_IEEE_R_ALIGN_COUNT_SHIFT            0u
#define ENET_IEEE_R_ALIGN_COUNT_WIDTH            16u
#define ENET_IEEE_R_ALIGN_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_ALIGN_COUNT_SHIFT))&ENET_IEEE_R_ALIGN_COUNT_MASK)
/* IEEE_R_MACERR Bit Fields */
#define ENET_IEEE_R_MACERR_COUNT_MASK            0xFFFFu
#define ENET_IEEE_R_MACERR_COUNT_SHIFT           0u
#define ENET_IEEE_R_MACERR_COUNT_WIDTH           16u
#define ENET_IEEE_R_MACERR_COUNT(x)              (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_MACERR_COUNT_SHIFT))&ENET_IEEE_R_MACERR_COUNT_MASK)
/* IEEE_R_FDXFC Bit Fields */
#define ENET_IEEE_R_FDXFC_COUNT_MASK             0xFFFFu
#define ENET_IEEE_R_FDXFC_COUNT_SHIFT            0u
#define ENET_IEEE_R_FDXFC_COUNT_WIDTH            16u
#define ENET_IEEE_R_FDXFC_COUNT(x)               (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_FDXFC_COUNT_SHIFT))&ENET_IEEE_R_FDXFC_COUNT_MASK)
/* IEEE_R_OCTETS_OK Bit Fields */
#define ENET_IEEE_R_OCTETS_OK_COUNT_MASK         0xFFFFFFFFu
#define ENET_IEEE_R_OCTETS_OK_COUNT_SHIFT        0u
#define ENET_IEEE_R_OCTETS_OK_COUNT_WIDTH        32u
#define ENET_IEEE_R_OCTETS_OK_COUNT(x)           (((uint32_t)(((uint32_t)(x))<<ENET_IEEE_R_OCTETS_OK_COUNT_SHIFT))&ENET_IEEE_R_OCTETS_OK_COUNT_MASK)
/* ATCR Bit Fields */
#define ENET_ATCR_EN_MASK                        0x1u
#define ENET_ATCR_EN_SHIFT                       0u
#define ENET_ATCR_EN_WIDTH                       1u
#define ENET_ATCR_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_EN_SHIFT))&ENET_ATCR_EN_MASK)
#define ENET_ATCR_OFFEN_MASK                     0x4u
#define ENET_ATCR_OFFEN_SHIFT                    2u
#define ENET_ATCR_OFFEN_WIDTH                    1u
#define ENET_ATCR_OFFEN(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_OFFEN_SHIFT))&ENET_ATCR_OFFEN_MASK)
#define ENET_ATCR_OFFRST_MASK                    0x8u
#define ENET_ATCR_OFFRST_SHIFT                   3u
#define ENET_ATCR_OFFRST_WIDTH                   1u
#define ENET_ATCR_OFFRST(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_OFFRST_SHIFT))&ENET_ATCR_OFFRST_MASK)
#define ENET_ATCR_PEREN_MASK                     0x10u
#define ENET_ATCR_PEREN_SHIFT                    4u
#define ENET_ATCR_PEREN_WIDTH                    1u
#define ENET_ATCR_PEREN(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_PEREN_SHIFT))&ENET_ATCR_PEREN_MASK)
#define ENET_ATCR_PINPER_MASK                    0x80u
#define ENET_ATCR_PINPER_SHIFT                   7u
#define ENET_ATCR_PINPER_WIDTH                   1u
#define ENET_ATCR_PINPER(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_PINPER_SHIFT))&ENET_ATCR_PINPER_MASK)
#define ENET_ATCR_RESTART_MASK                   0x200u
#define ENET_ATCR_RESTART_SHIFT                  9u
#define ENET_ATCR_RESTART_WIDTH                  1u
#define ENET_ATCR_RESTART(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_RESTART_SHIFT))&ENET_ATCR_RESTART_MASK)
#define ENET_ATCR_CAPTURE_MASK                   0x800u
#define ENET_ATCR_CAPTURE_SHIFT                  11u
#define ENET_ATCR_CAPTURE_WIDTH                  1u
#define ENET_ATCR_CAPTURE(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_CAPTURE_SHIFT))&ENET_ATCR_CAPTURE_MASK)
#define ENET_ATCR_SLAVE_MASK                     0x2000u
#define ENET_ATCR_SLAVE_SHIFT                    13u
#define ENET_ATCR_SLAVE_WIDTH                    1u
#define ENET_ATCR_SLAVE(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ATCR_SLAVE_SHIFT))&ENET_ATCR_SLAVE_MASK)
/* ATVR Bit Fields */
#define ENET_ATVR_ATIME_MASK                     0xFFFFFFFFu
#define ENET_ATVR_ATIME_SHIFT                    0u
#define ENET_ATVR_ATIME_WIDTH                    32u
#define ENET_ATVR_ATIME(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_ATVR_ATIME_SHIFT))&ENET_ATVR_ATIME_MASK)
/* ATOFF Bit Fields */
#define ENET_ATOFF_OFFSET_MASK                   0xFFFFFFFFu
#define ENET_ATOFF_OFFSET_SHIFT                  0u
#define ENET_ATOFF_OFFSET_WIDTH                  32u
#define ENET_ATOFF_OFFSET(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ATOFF_OFFSET_SHIFT))&ENET_ATOFF_OFFSET_MASK)
/* ATPER Bit Fields */
#define ENET_ATPER_PERIOD_MASK                   0xFFFFFFFFu
#define ENET_ATPER_PERIOD_SHIFT                  0u
#define ENET_ATPER_PERIOD_WIDTH                  32u
#define ENET_ATPER_PERIOD(x)                     (((uint32_t)(((uint32_t)(x))<<ENET_ATPER_PERIOD_SHIFT))&ENET_ATPER_PERIOD_MASK)
/* ATCOR Bit Fields */
#define ENET_ATCOR_COR_MASK                      0x7FFFFFFFu
#define ENET_ATCOR_COR_SHIFT                     0u
#define ENET_ATCOR_COR_WIDTH                     31u
#define ENET_ATCOR_COR(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ATCOR_COR_SHIFT))&ENET_ATCOR_COR_MASK)
/* ATINC Bit Fields */
#define ENET_ATINC_INC_MASK                      0x7Fu
#define ENET_ATINC_INC_SHIFT                     0u
#define ENET_ATINC_INC_WIDTH                     7u
#define ENET_ATINC_INC(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_ATINC_INC_SHIFT))&ENET_ATINC_INC_MASK)
#define ENET_ATINC_INC_CORR_MASK                 0x7F00u
#define ENET_ATINC_INC_CORR_SHIFT                8u
#define ENET_ATINC_INC_CORR_WIDTH                7u
#define ENET_ATINC_INC_CORR(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_ATINC_INC_CORR_SHIFT))&ENET_ATINC_INC_CORR_MASK)
/* ATSTMP Bit Fields */
#define ENET_ATSTMP_TIMESTAMP_MASK               0xFFFFFFFFu
#define ENET_ATSTMP_TIMESTAMP_SHIFT              0u
#define ENET_ATSTMP_TIMESTAMP_WIDTH              32u
#define ENET_ATSTMP_TIMESTAMP(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_ATSTMP_TIMESTAMP_SHIFT))&ENET_ATSTMP_TIMESTAMP_MASK)
/* MDATA Bit Fields */
#define ENET_MDATA_MATCHDATA_MASK                0xFFFFFFFFu
#define ENET_MDATA_MATCHDATA_SHIFT               0u
#define ENET_MDATA_MATCHDATA_WIDTH               32u
#define ENET_MDATA_MATCHDATA(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_MDATA_MATCHDATA_SHIFT))&ENET_MDATA_MATCHDATA_MASK)
/* MMASK Bit Fields */
#define ENET_MMASK_MATCHMASK_MASK                0xFFFFFFFFu
#define ENET_MMASK_MATCHMASK_SHIFT               0u
#define ENET_MMASK_MATCHMASK_WIDTH               32u
#define ENET_MMASK_MATCHMASK(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_MMASK_MATCHMASK_SHIFT))&ENET_MMASK_MATCHMASK_MASK)
/* MCONFIG Bit Fields */
#define ENET_MCONFIG_FRMOFF_MASK                 0xFCu
#define ENET_MCONFIG_FRMOFF_SHIFT                2u
#define ENET_MCONFIG_FRMOFF_WIDTH                6u
#define ENET_MCONFIG_FRMOFF(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_FRMOFF_SHIFT))&ENET_MCONFIG_FRMOFF_MASK)
#define ENET_MCONFIG_OK_INDEX_MASK               0xFF0000u
#define ENET_MCONFIG_OK_INDEX_SHIFT              16u
#define ENET_MCONFIG_OK_INDEX_WIDTH              8u
#define ENET_MCONFIG_OK_INDEX(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_OK_INDEX_SHIFT))&ENET_MCONFIG_OK_INDEX_MASK)
#define ENET_MCONFIG_IM_MASK                     0x20000000u
#define ENET_MCONFIG_IM_SHIFT                    29u
#define ENET_MCONFIG_IM_WIDTH                    1u
#define ENET_MCONFIG_IM(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_IM_SHIFT))&ENET_MCONFIG_IM_MASK)
#define ENET_MCONFIG_RF_MASK                     0x40000000u
#define ENET_MCONFIG_RF_SHIFT                    30u
#define ENET_MCONFIG_RF_WIDTH                    1u
#define ENET_MCONFIG_RF(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_RF_SHIFT))&ENET_MCONFIG_RF_MASK)
#define ENET_MCONFIG_AF_MASK                     0x80000000u
#define ENET_MCONFIG_AF_SHIFT                    31u
#define ENET_MCONFIG_AF_WIDTH                    1u
#define ENET_MCONFIG_AF(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_MCONFIG_AF_SHIFT))&ENET_MCONFIG_AF_MASK)
/* MENTRYRW Bit Fields */
#define ENET_MENTRYRW_ENTRYADD_MASK              0xFFu
#define ENET_MENTRYRW_ENTRYADD_SHIFT             0u
#define ENET_MENTRYRW_ENTRYADD_WIDTH             8u
#define ENET_MENTRYRW_ENTRYADD(x)                (((uint32_t)(((uint32_t)(x))<<ENET_MENTRYRW_ENTRYADD_SHIFT))&ENET_MENTRYRW_ENTRYADD_MASK)
#define ENET_MENTRYRW_WR_MASK                    0x100u
#define ENET_MENTRYRW_WR_SHIFT                   8u
#define ENET_MENTRYRW_WR_WIDTH                   1u
#define ENET_MENTRYRW_WR(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_MENTRYRW_WR_SHIFT))&ENET_MENTRYRW_WR_MASK)
#define ENET_MENTRYRW_RD_MASK                    0x200u
#define ENET_MENTRYRW_RD_SHIFT                   9u
#define ENET_MENTRYRW_RD_WIDTH                   1u
#define ENET_MENTRYRW_RD(x)                      (((uint32_t)(((uint32_t)(x))<<ENET_MENTRYRW_RD_SHIFT))&ENET_MENTRYRW_RD_MASK)
/* RXPCTL Bit Fields */
#define ENET_RXPCTL_ENPARSER_MASK                0x1u
#define ENET_RXPCTL_ENPARSER_SHIFT               0u
#define ENET_RXPCTL_ENPARSER_WIDTH               1u
#define ENET_RXPCTL_ENPARSER(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_ENPARSER_SHIFT))&ENET_RXPCTL_ENPARSER_MASK)
#define ENET_RXPCTL_INVBYTORD_MASK               0x2u
#define ENET_RXPCTL_INVBYTORD_SHIFT              1u
#define ENET_RXPCTL_INVBYTORD_WIDTH              1u
#define ENET_RXPCTL_INVBYTORD(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_INVBYTORD_SHIFT))&ENET_RXPCTL_INVBYTORD_MASK)
#define ENET_RXPCTL_PRSRSCLR_MASK                0x10u
#define ENET_RXPCTL_PRSRSCLR_SHIFT               4u
#define ENET_RXPCTL_PRSRSCLR_WIDTH               1u
#define ENET_RXPCTL_PRSRSCLR(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_PRSRSCLR_SHIFT))&ENET_RXPCTL_PRSRSCLR_MASK)
#define ENET_RXPCTL_MAXINDEX_MASK                0xFF00u
#define ENET_RXPCTL_MAXINDEX_SHIFT               8u
#define ENET_RXPCTL_MAXINDEX_WIDTH               8u
#define ENET_RXPCTL_MAXINDEX(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_MAXINDEX_SHIFT))&ENET_RXPCTL_MAXINDEX_MASK)
#define ENET_RXPCTL_ENDERRQ_MASK                 0xFF0000u
#define ENET_RXPCTL_ENDERRQ_SHIFT                16u
#define ENET_RXPCTL_ENDERRQ_WIDTH                8u
#define ENET_RXPCTL_ENDERRQ(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_ENDERRQ_SHIFT))&ENET_RXPCTL_ENDERRQ_MASK)
#define ENET_RXPCTL_ACPTEERR_MASK                0x1000000u
#define ENET_RXPCTL_ACPTEERR_SHIFT               24u
#define ENET_RXPCTL_ACPTEERR_WIDTH               1u
#define ENET_RXPCTL_ACPTEERR(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPCTL_ACPTEERR_SHIFT))&ENET_RXPCTL_ACPTEERR_MASK)
/* MAXFRMOFF Bit Fields */
#define ENET_MAXFRMOFF_MXFRMOFF_MASK             0x3Fu
#define ENET_MAXFRMOFF_MXFRMOFF_SHIFT            0u
#define ENET_MAXFRMOFF_MXFRMOFF_WIDTH            6u
#define ENET_MAXFRMOFF_MXFRMOFF(x)               (((uint32_t)(((uint32_t)(x))<<ENET_MAXFRMOFF_MXFRMOFF_SHIFT))&ENET_MAXFRMOFF_MXFRMOFF_MASK)
/* RXPARST Bit Fields */
#define ENET_RXPARST_MXINDERR_MASK               0x1u
#define ENET_RXPARST_MXINDERR_SHIFT              0u
#define ENET_RXPARST_MXINDERR_WIDTH              1u
#define ENET_RXPARST_MXINDERR(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_MXINDERR_SHIFT))&ENET_RXPARST_MXINDERR_MASK)
#define ENET_RXPARST_TBLDPTERR_MASK              0x2u
#define ENET_RXPARST_TBLDPTERR_SHIFT             1u
#define ENET_RXPARST_TBLDPTERR_WIDTH             1u
#define ENET_RXPARST_TBLDPTERR(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_TBLDPTERR_SHIFT))&ENET_RXPARST_TBLDPTERR_MASK)
#define ENET_RXPARST_NOMTCERR_MASK               0x4u
#define ENET_RXPARST_NOMTCERR_SHIFT              2u
#define ENET_RXPARST_NOMTCERR_WIDTH              1u
#define ENET_RXPARST_NOMTCERR(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_NOMTCERR_SHIFT))&ENET_RXPARST_NOMTCERR_MASK)
#define ENET_RXPARST_FMOFFERR_MASK               0x8u
#define ENET_RXPARST_FMOFFERR_SHIFT              3u
#define ENET_RXPARST_FMOFFERR_WIDTH              1u
#define ENET_RXPARST_FMOFFERR(x)                 (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_FMOFFERR_SHIFT))&ENET_RXPARST_FMOFFERR_MASK)
#define ENET_RXPARST_PRSENDERR_MASK              0x10u
#define ENET_RXPARST_PRSENDERR_SHIFT             4u
#define ENET_RXPARST_PRSENDERR_WIDTH             1u
#define ENET_RXPARST_PRSENDERR(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_PRSENDERR_SHIFT))&ENET_RXPARST_PRSENDERR_MASK)
#define ENET_RXPARST_INVMAXIDX_MASK              0x20u
#define ENET_RXPARST_INVMAXIDX_SHIFT             5u
#define ENET_RXPARST_INVMAXIDX_WIDTH             1u
#define ENET_RXPARST_INVMAXIDX(x)                (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_INVMAXIDX_SHIFT))&ENET_RXPARST_INVMAXIDX_MASK)
#define ENET_RXPARST_RXPRSDN_MASK                0x100u
#define ENET_RXPARST_RXPRSDN_SHIFT               8u
#define ENET_RXPARST_RXPRSDN_WIDTH               1u
#define ENET_RXPARST_RXPRSDN(x)                  (((uint32_t)(((uint32_t)(x))<<ENET_RXPARST_RXPRSDN_SHIFT))&ENET_RXPARST_RXPRSDN_MASK)
/* PARSDSCD Bit Fields */
#define ENET_PARSDSCD_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PARSDSCD_COUNT_SHIFT                0u
#define ENET_PARSDSCD_COUNT_WIDTH                32u
#define ENET_PARSDSCD_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PARSDSCD_COUNT_SHIFT))&ENET_PARSDSCD_COUNT_MASK)
/* PRSACPT0 Bit Fields */
#define ENET_PRSACPT0_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSACPT0_COUNT_SHIFT                0u
#define ENET_PRSACPT0_COUNT_WIDTH                32u
#define ENET_PRSACPT0_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSACPT0_COUNT_SHIFT))&ENET_PRSACPT0_COUNT_MASK)
/* PRSRJCT0 Bit Fields */
#define ENET_PRSRJCT0_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSRJCT0_COUNT_SHIFT                0u
#define ENET_PRSRJCT0_COUNT_WIDTH                32u
#define ENET_PRSRJCT0_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSRJCT0_COUNT_SHIFT))&ENET_PRSRJCT0_COUNT_MASK)
/* PRSACPT1 Bit Fields */
#define ENET_PRSACPT1_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSACPT1_COUNT_SHIFT                0u
#define ENET_PRSACPT1_COUNT_WIDTH                32u
#define ENET_PRSACPT1_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSACPT1_COUNT_SHIFT))&ENET_PRSACPT1_COUNT_MASK)
/* PRSRJCT1 Bit Fields */
#define ENET_PRSRJCT1_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSRJCT1_COUNT_SHIFT                0u
#define ENET_PRSRJCT1_COUNT_WIDTH                32u
#define ENET_PRSRJCT1_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSRJCT1_COUNT_SHIFT))&ENET_PRSRJCT1_COUNT_MASK)
/* PRSACPT2 Bit Fields */
#define ENET_PRSACPT2_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSACPT2_COUNT_SHIFT                0u
#define ENET_PRSACPT2_COUNT_WIDTH                32u
#define ENET_PRSACPT2_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSACPT2_COUNT_SHIFT))&ENET_PRSACPT2_COUNT_MASK)
/* PRSRJCT2 Bit Fields */
#define ENET_PRSRJCT2_COUNT_MASK                 0xFFFFFFFFu
#define ENET_PRSRJCT2_COUNT_SHIFT                0u
#define ENET_PRSRJCT2_COUNT_WIDTH                32u
#define ENET_PRSRJCT2_COUNT(x)                   (((uint32_t)(((uint32_t)(x))<<ENET_PRSRJCT2_COUNT_SHIFT))&ENET_PRSRJCT2_COUNT_MASK)
/* TGSR Bit Fields */
#define ENET_TGSR_TF0_MASK                       0x1u
#define ENET_TGSR_TF0_SHIFT                      0u
#define ENET_TGSR_TF0_WIDTH                      1u
#define ENET_TGSR_TF0(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TGSR_TF0_SHIFT))&ENET_TGSR_TF0_MASK)
#define ENET_TGSR_TF1_MASK                       0x2u
#define ENET_TGSR_TF1_SHIFT                      1u
#define ENET_TGSR_TF1_WIDTH                      1u
#define ENET_TGSR_TF1(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TGSR_TF1_SHIFT))&ENET_TGSR_TF1_MASK)
#define ENET_TGSR_TF2_MASK                       0x4u
#define ENET_TGSR_TF2_SHIFT                      2u
#define ENET_TGSR_TF2_WIDTH                      1u
#define ENET_TGSR_TF2(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TGSR_TF2_SHIFT))&ENET_TGSR_TF2_MASK)
#define ENET_TGSR_TF3_MASK                       0x8u
#define ENET_TGSR_TF3_SHIFT                      3u
#define ENET_TGSR_TF3_WIDTH                      1u
#define ENET_TGSR_TF3(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TGSR_TF3_SHIFT))&ENET_TGSR_TF3_MASK)
/* TCSR Bit Fields */
#define ENET_TCSR_TDRE_MASK                      0x1u
#define ENET_TCSR_TDRE_SHIFT                     0u
#define ENET_TCSR_TDRE_WIDTH                     1u
#define ENET_TCSR_TDRE(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TCSR_TDRE_SHIFT))&ENET_TCSR_TDRE_MASK)
#define ENET_TCSR_TMODE_MASK                     0x3Cu
#define ENET_TCSR_TMODE_SHIFT                    2u
#define ENET_TCSR_TMODE_WIDTH                    4u
#define ENET_TCSR_TMODE(x)                       (((uint32_t)(((uint32_t)(x))<<ENET_TCSR_TMODE_SHIFT))&ENET_TCSR_TMODE_MASK)
#define ENET_TCSR_TIE_MASK                       0x40u
#define ENET_TCSR_TIE_SHIFT                      6u
#define ENET_TCSR_TIE_WIDTH                      1u
#define ENET_TCSR_TIE(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TCSR_TIE_SHIFT))&ENET_TCSR_TIE_MASK)
#define ENET_TCSR_TF_MASK                        0x80u
#define ENET_TCSR_TF_SHIFT                       7u
#define ENET_TCSR_TF_WIDTH                       1u
#define ENET_TCSR_TF(x)                          (((uint32_t)(((uint32_t)(x))<<ENET_TCSR_TF_SHIFT))&ENET_TCSR_TF_MASK)
#define ENET_TCSR_TPWC_MASK                      0xF800u
#define ENET_TCSR_TPWC_SHIFT                     11u
#define ENET_TCSR_TPWC_WIDTH                     5u
#define ENET_TCSR_TPWC(x)                        (((uint32_t)(((uint32_t)(x))<<ENET_TCSR_TPWC_SHIFT))&ENET_TCSR_TPWC_MASK)
/* TCCR Bit Fields */
#define ENET_TCCR_TCC_MASK                       0xFFFFFFFFu
#define ENET_TCCR_TCC_SHIFT                      0u
#define ENET_TCCR_TCC_WIDTH                      32u
#define ENET_TCCR_TCC(x)                         (((uint32_t)(((uint32_t)(x))<<ENET_TCCR_TCC_SHIFT))&ENET_TCCR_TCC_MASK)

/*!
 * @}
 */ /* end of group ENET_Register_Masks */


/*!
 * @}
 */ /* end of group ENET_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- ESW Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ESW_Peripheral_Access_Layer ESW Peripheral Access Layer
 * @{
 */


/** ESW - Size of Registers Arrays */
#define ESW_RCMR_COUNT                           2u
#define ESW_VRES_COUNT                           32u

/** ESW - Register Layout Typedef */
typedef struct {
  __I  uint32_t REV;                               /**< Revision, offset: 0x0 */
  __IO uint32_t SCR;                               /**< Scratch register, offset: 0x4 */
  __IO uint32_t PER;                               /**< Port enable register, offset: 0x8 */
  __IO uint32_t DUCR;                              /**< Default unicast resolution, offset: 0xC */
  __IO uint32_t VLANV;                             /**< VLAN verify, offset: 0x10 */
  __IO uint32_t DBCR;                              /**< Default broadcast resolution, offset: 0x14 */
  __IO uint32_t DMCR;                              /**< Default multicast resolution, offset: 0x18 */
  __IO uint32_t BKLR;                              /**< Blocking and learning enable, offset: 0x1C */
  __IO uint32_t BMPC;                              /**< Bridge management port configuration, offset: 0x20 */
  __IO uint32_t MODE;                              /**< Mode configuration, offset: 0x24 */
  __IO uint32_t VIMSEL;                            /**< VLAN input manipulation select, offset: 0x28 */
  __IO uint32_t VOMSEL;                            /**< VLAN output manipulation select, offset: 0x2C */
  __IO uint32_t VIMEN;                             /**< VLAN input manipulation enable, offset: 0x30 */
  __IO uint32_t VID;                               /**< VLAN tag ID, offset: 0x34 */
  __IO uint32_t RCMR[ESW_RCMR_COUNT];              /**< Receive Classification Match Register for Class n, array offset: 0x38, array step: 0x4 */
  __IO uint32_t MCR;                               /**< Mirror control register, offset: 0x40 */
  __IO uint32_t EGMAP;                             /**< Egress port definitions, offset: 0x44 */
  __IO uint32_t INGMAP;                            /**< Ingress port definitions, offset: 0x48 */
  __IO uint32_t INGSAL;                            /**< Ingress source MAC address low, offset: 0x4C */
  __IO uint32_t INGSAH;                            /**< Ingress source MAC address high, offset: 0x50 */
  __IO uint32_t INGDAL;                            /**< Ingress destination MAC address low, offset: 0x54 */
  __IO uint32_t INGDAH;                            /**< Ingress destination MAC address high, offset: 0x58 */
  __IO uint32_t EGSAL;                             /**< Egress source MAC address low, offset: 0x5C */
  __IO uint32_t EGSAH;                             /**< Egress source MAC address high, offset: 0x60 */
  __IO uint32_t EGDAL;                             /**< Egress destination MAC address low, offset: 0x64 */
  __IO uint32_t EGDAH;                             /**< Egress destination MAC address high, offset: 0x68 */
  __IO uint32_t MCVAL;                             /**< Mirror count value, offset: 0x6C */
       uint8_t RESERVED_0[16];
  __IO uint32_t MMSR;                              /**< Memory manager status, offset: 0x80 */
  __IO uint32_t LMT;                               /**< Low memory threshold, offset: 0x84 */
  __IO uint32_t LFC;                               /**< Lowest number of free cells, offset: 0x88 */
  __I  uint32_t PCSR;                              /**< Port congestion status, offset: 0x8C */
  __I  uint32_t IOSR;                              /**< Switch input and output interface status, offset: 0x90 */
  __IO uint32_t QWT;                               /**< Queue weights, offset: 0x94 */
       uint8_t RESERVED_1[4];
  __IO uint32_t P0BCT;                             /**< Port 0 Backpressure Congestion Threshold, offset: 0x9C */
       uint8_t RESERVED_2[28];
  __IO uint32_t FFEN;                              /**< Port 0 forced forwarding enable, offset: 0xBC */
  __IO uint32_t PSNP1;                             /**< Port snooping registers, offset: 0xC0 */
  __IO uint32_t PSNP2;                             /**< Port snooping registers, offset: 0xC4 */
  __IO uint32_t PSNP3;                             /**< Port snooping registers, offset: 0xC8 */
  __IO uint32_t PSNP4;                             /**< Port snooping registers, offset: 0xCC */
  __IO uint32_t PSNP5;                             /**< Port snooping registers, offset: 0xD0 */
  __IO uint32_t PSNP6;                             /**< Port snooping registers, offset: 0xD4 */
  __IO uint32_t PSNP7;                             /**< Port snooping registers, offset: 0xD8 */
  __IO uint32_t PSNP8;                             /**< Port snooping registers, offset: 0xDC */
  __IO uint32_t IPSNP1;                            /**< IP snooping registers, offset: 0xE0 */
  __IO uint32_t IPSNP2;                            /**< IP snooping registers, offset: 0xE4 */
  __IO uint32_t IPSNP3;                            /**< IP snooping registers, offset: 0xE8 */
  __IO uint32_t IPSNP4;                            /**< IP snooping registers, offset: 0xEC */
  __IO uint32_t IPSNP5;                            /**< IP snooping registers, offset: 0xF0 */
  __IO uint32_t IPSNP6;                            /**< IP snooping registers, offset: 0xF4 */
  __IO uint32_t IPSNP7;                            /**< IP snooping registers, offset: 0xF8 */
  __IO uint32_t IPSNP8;                            /**< IP snooping registers, offset: 0xFC */
  __IO uint32_t P0VRES;                            /**< Port 0 VLAN priority resolution map, offset: 0x100 */
  __IO uint32_t P1VRES;                            /**< Port 1 VLAN priority resolution map, offset: 0x104 */
  __IO uint32_t P2VRES;                            /**< Port 2 VLAN priority resolution map, offset: 0x108 */
       uint8_t RESERVED_3[52];
  __IO uint32_t IPRES;                             /**< IPv4/v6 priority resolution table, offset: 0x140 */
       uint8_t RESERVED_4[60];
  __IO uint32_t P0RES;                             /**< Port 0 priority resolution configuration, offset: 0x180 */
  __IO uint32_t P1RES;                             /**< Port 1 priority resolution configuration, offset: 0x184 */
  __IO uint32_t P2RES;                             /**< Port 2 priority resolution configuration, offset: 0x188 */
       uint8_t RESERVED_5[36];
  __IO uint32_t HSEED0;                            /**< Hash seed 0, offset: 0x1B0 */
  __IO uint32_t HSEED1;                            /**< Hash seed 1, offset: 0x1B4 */
       uint8_t RESERVED_6[72];
  __IO uint32_t P0ID;                              /**< Port 0 VLAN ID, offset: 0x200 */
  __IO uint32_t P1ID;                              /**< Port 1 VLAN ID, offset: 0x204 */
  __IO uint32_t P2ID;                              /**< Port 2 VLAN ID, offset: 0x208 */
       uint8_t RESERVED_7[116];
  __IO uint32_t VRES[ESW_VRES_COUNT];              /**< VLAN table, array offset: 0x280, array step: 0x4 */
  __I  uint32_t DISCN;                             /**< Number of discarded frames, offset: 0x300 */
  __I  uint32_t DISCB;                             /**< Bytes of discarded frames, offset: 0x304 */
  __I  uint32_t NDISCN;                            /**< Number of non-discarded frames, offset: 0x308 */
  __I  uint32_t NDISCB;                            /**< Bytes of non-discarded frames, offset: 0x30C */
  __I  uint32_t P0OQC;                             /**< Port 0 output queue congestion, offset: 0x310 */
  __I  uint32_t P0MVID;                            /**< Port 0 mismatching VLAN ID, offset: 0x314 */
  __I  uint32_t P0MVTAG;                           /**< Port 0 missing VLAN tag, offset: 0x318 */
  __I  uint32_t P0BL;                              /**< Port 0 blocked, offset: 0x31C */
  __I  uint32_t P1OQC;                             /**< Port 1 output queue congestion, offset: 0x320 */
  __I  uint32_t P1MVID;                            /**< Port 1 mismatching VLAN ID, offset: 0x324 */
  __I  uint32_t P1MVTAG;                           /**< Port 1 missing VLAN tag, offset: 0x328 */
  __I  uint32_t P1BL;                              /**< Port 1 blocked, offset: 0x32C */
  __I  uint32_t P2OQC;                             /**< Port 2 output queue congestion, offset: 0x330 */
  __I  uint32_t P2MVID;                            /**< Port 2 mismatching VLAN ID, offset: 0x334 */
  __I  uint32_t P2MVTAG;                           /**< Port 2 missing VLAN tag, offset: 0x338 */
  __I  uint32_t P2BL;                              /**< Port 2 blocked, offset: 0x33C */
       uint8_t RESERVED_8[192];
  __IO uint32_t ISR;                               /**< Interrupt status register, offset: 0x400 */
  __IO uint32_t IMR;                               /**< Interrupt mask register, offset: 0x404 */
  __IO uint32_t RDSR;                              /**< Receive descriptor ring pointer, offset: 0x408 */
  __IO uint32_t TDSR;                              /**< Transmit descriptor ring pointer, offset: 0x40C */
  __IO uint32_t MRBR;                              /**< Maximum receive buffer size, offset: 0x410 */
  __IO uint32_t RDAR;                              /**< Receive descriptor active - class0, offset: 0x414 */
  __IO uint32_t TDAR;                              /**< Transmit descriptor active - class0, offset: 0x418 */
  __IO uint32_t RDAR1;                             /**< Receive descriptor active - class1, offset: 0x41C */
  __IO uint32_t TDAR1;                             /**< Transmit descriptor active - class1, offset: 0x420 */
  __IO uint32_t RDAR2;                             /**< Receive descriptor active - class2, offset: 0x424 */
  __IO uint32_t TDAR2;                             /**< Transmit descriptor active - class2, offset: 0x428 */
       uint8_t RESERVED_9[84];
  __IO uint32_t TXIC;                              /**< Transmit interrupt coalesing register, offset: 0x480 */
  __IO uint32_t RXIC;                              /**< Receive interrupt coalesing register, offset: 0x484 */
       uint8_t RESERVED_10[8];
  __IO uint32_t SEL_CLASS;                         /**< Select register class for read/write access, offset: 0x490 */
       uint8_t RESERVED_11[108];
  __I  uint32_t LREC0;                             /**< Learning records A0 & B1, offset: 0x500 */
  __I  uint32_t LREC1;                             /**< Learning record B1, offset: 0x504 */
  __I  uint32_t LSR;                               /**< Learning data available status, offset: 0x508 */
} ESW_Type, *ESW_MemMapPtr;

 /** Number of instances of the ESW module. */
#define ESW_INSTANCE_COUNT                       (1u)


/* ESW - Peripheral instance base addresses */
/** Peripheral ESW base address */
#define ESW_BASE                                 (0xFFCCC000u)
/** Peripheral ESW base pointer */
#define ESW                                      ((ESW_Type *)ESW_BASE)
/** Array initializer of ESW peripheral base addresses */
#define ESW_BASE_ADDRS                           { ESW_BASE }
/** Array initializer of ESW peripheral base pointers */
#define ESW_BASE_PTRS                            { ESW }

/* ----------------------------------------------------------------------------
   -- ESW Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup ESW_Register_Masks ESW Register Masks
 * @{
 */

/* REV Bit Fields */
#define ESW_REV_REVISION_MASK                    0xFFFFFFFFu
#define ESW_REV_REVISION_SHIFT                   0u
#define ESW_REV_REVISION_WIDTH                   32u
#define ESW_REV_REVISION(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_REV_REVISION_SHIFT))&ESW_REV_REVISION_MASK)
/* SCR Bit Fields */
#define ESW_SCR_SCRATCH_MASK                     0xFFFFFFFFu
#define ESW_SCR_SCRATCH_SHIFT                    0u
#define ESW_SCR_SCRATCH_WIDTH                    32u
#define ESW_SCR_SCRATCH(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_SCR_SCRATCH_SHIFT))&ESW_SCR_SCRATCH_MASK)
/* PER Bit Fields */
#define ESW_PER_TE0_MASK                         0x1u
#define ESW_PER_TE0_SHIFT                        0u
#define ESW_PER_TE0_WIDTH                        1u
#define ESW_PER_TE0(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_PER_TE0_SHIFT))&ESW_PER_TE0_MASK)
#define ESW_PER_TE1_MASK                         0x2u
#define ESW_PER_TE1_SHIFT                        1u
#define ESW_PER_TE1_WIDTH                        1u
#define ESW_PER_TE1(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_PER_TE1_SHIFT))&ESW_PER_TE1_MASK)
#define ESW_PER_TE2_MASK                         0x4u
#define ESW_PER_TE2_SHIFT                        2u
#define ESW_PER_TE2_WIDTH                        1u
#define ESW_PER_TE2(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_PER_TE2_SHIFT))&ESW_PER_TE2_MASK)
#define ESW_PER_RE0_MASK                         0x10000u
#define ESW_PER_RE0_SHIFT                        16u
#define ESW_PER_RE0_WIDTH                        1u
#define ESW_PER_RE0(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_PER_RE0_SHIFT))&ESW_PER_RE0_MASK)
#define ESW_PER_RE1_MASK                         0x20000u
#define ESW_PER_RE1_SHIFT                        17u
#define ESW_PER_RE1_WIDTH                        1u
#define ESW_PER_RE1(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_PER_RE1_SHIFT))&ESW_PER_RE1_MASK)
#define ESW_PER_RE2_MASK                         0x40000u
#define ESW_PER_RE2_SHIFT                        18u
#define ESW_PER_RE2_WIDTH                        1u
#define ESW_PER_RE2(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_PER_RE2_SHIFT))&ESW_PER_RE2_MASK)
/* DUCR Bit Fields */
#define ESW_DUCR_UP0_MASK                        0x1u
#define ESW_DUCR_UP0_SHIFT                       0u
#define ESW_DUCR_UP0_WIDTH                       1u
#define ESW_DUCR_UP0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DUCR_UP0_SHIFT))&ESW_DUCR_UP0_MASK)
#define ESW_DUCR_UP1_MASK                        0x2u
#define ESW_DUCR_UP1_SHIFT                       1u
#define ESW_DUCR_UP1_WIDTH                       1u
#define ESW_DUCR_UP1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DUCR_UP1_SHIFT))&ESW_DUCR_UP1_MASK)
#define ESW_DUCR_UP2_MASK                        0x4u
#define ESW_DUCR_UP2_SHIFT                       2u
#define ESW_DUCR_UP2_WIDTH                       1u
#define ESW_DUCR_UP2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DUCR_UP2_SHIFT))&ESW_DUCR_UP2_MASK)
#define ESW_DUCR_TP0_MASK                        0x10000u
#define ESW_DUCR_TP0_SHIFT                       16u
#define ESW_DUCR_TP0_WIDTH                       1u
#define ESW_DUCR_TP0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DUCR_TP0_SHIFT))&ESW_DUCR_TP0_MASK)
#define ESW_DUCR_TP1_MASK                        0x20000u
#define ESW_DUCR_TP1_SHIFT                       17u
#define ESW_DUCR_TP1_WIDTH                       1u
#define ESW_DUCR_TP1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DUCR_TP1_SHIFT))&ESW_DUCR_TP1_MASK)
#define ESW_DUCR_TP2_MASK                        0x40000u
#define ESW_DUCR_TP2_SHIFT                       18u
#define ESW_DUCR_TP2_WIDTH                       1u
#define ESW_DUCR_TP2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DUCR_TP2_SHIFT))&ESW_DUCR_TP2_MASK)
/* VLANV Bit Fields */
#define ESW_VLANV_VV0_MASK                       0x1u
#define ESW_VLANV_VV0_SHIFT                      0u
#define ESW_VLANV_VV0_WIDTH                      1u
#define ESW_VLANV_VV0(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_VV0_SHIFT))&ESW_VLANV_VV0_MASK)
#define ESW_VLANV_VV1_MASK                       0x2u
#define ESW_VLANV_VV1_SHIFT                      1u
#define ESW_VLANV_VV1_WIDTH                      1u
#define ESW_VLANV_VV1(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_VV1_SHIFT))&ESW_VLANV_VV1_MASK)
#define ESW_VLANV_VV2_MASK                       0x4u
#define ESW_VLANV_VV2_SHIFT                      2u
#define ESW_VLANV_VV2_WIDTH                      1u
#define ESW_VLANV_VV2(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_VV2_SHIFT))&ESW_VLANV_VV2_MASK)
#define ESW_VLANV_DUV0_MASK                      0x100u
#define ESW_VLANV_DUV0_SHIFT                     8u
#define ESW_VLANV_DUV0_WIDTH                     1u
#define ESW_VLANV_DUV0(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_DUV0_SHIFT))&ESW_VLANV_DUV0_MASK)
#define ESW_VLANV_DUV1_MASK                      0x200u
#define ESW_VLANV_DUV1_SHIFT                     9u
#define ESW_VLANV_DUV1_WIDTH                     1u
#define ESW_VLANV_DUV1(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_DUV1_SHIFT))&ESW_VLANV_DUV1_MASK)
#define ESW_VLANV_DUV2_MASK                      0x400u
#define ESW_VLANV_DUV2_SHIFT                     10u
#define ESW_VLANV_DUV2_WIDTH                     1u
#define ESW_VLANV_DUV2(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_DUV2_SHIFT))&ESW_VLANV_DUV2_MASK)
#define ESW_VLANV_DU0_MASK                       0x10000u
#define ESW_VLANV_DU0_SHIFT                      16u
#define ESW_VLANV_DU0_WIDTH                      1u
#define ESW_VLANV_DU0(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_DU0_SHIFT))&ESW_VLANV_DU0_MASK)
#define ESW_VLANV_DU1_MASK                       0x20000u
#define ESW_VLANV_DU1_SHIFT                      17u
#define ESW_VLANV_DU1_WIDTH                      1u
#define ESW_VLANV_DU1(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_DU1_SHIFT))&ESW_VLANV_DU1_MASK)
#define ESW_VLANV_DU2_MASK                       0x40000u
#define ESW_VLANV_DU2_SHIFT                      18u
#define ESW_VLANV_DU2_WIDTH                      1u
#define ESW_VLANV_DU2(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_DU2_SHIFT))&ESW_VLANV_DU2_MASK)
#define ESW_VLANV_DUT0_MASK                      0x1000000u
#define ESW_VLANV_DUT0_SHIFT                     24u
#define ESW_VLANV_DUT0_WIDTH                     1u
#define ESW_VLANV_DUT0(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_DUT0_SHIFT))&ESW_VLANV_DUT0_MASK)
#define ESW_VLANV_DUT1_MASK                      0x2000000u
#define ESW_VLANV_DUT1_SHIFT                     25u
#define ESW_VLANV_DUT1_WIDTH                     1u
#define ESW_VLANV_DUT1(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_DUT1_SHIFT))&ESW_VLANV_DUT1_MASK)
#define ESW_VLANV_DUT2_MASK                      0x4000000u
#define ESW_VLANV_DUT2_SHIFT                     26u
#define ESW_VLANV_DUT2_WIDTH                     1u
#define ESW_VLANV_DUT2(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VLANV_DUT2_SHIFT))&ESW_VLANV_DUT2_MASK)
/* DBCR Bit Fields */
#define ESW_DBCR_UP0_MASK                        0x1u
#define ESW_DBCR_UP0_SHIFT                       0u
#define ESW_DBCR_UP0_WIDTH                       1u
#define ESW_DBCR_UP0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DBCR_UP0_SHIFT))&ESW_DBCR_UP0_MASK)
#define ESW_DBCR_UP1_MASK                        0x2u
#define ESW_DBCR_UP1_SHIFT                       1u
#define ESW_DBCR_UP1_WIDTH                       1u
#define ESW_DBCR_UP1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DBCR_UP1_SHIFT))&ESW_DBCR_UP1_MASK)
#define ESW_DBCR_UP2_MASK                        0x4u
#define ESW_DBCR_UP2_SHIFT                       2u
#define ESW_DBCR_UP2_WIDTH                       1u
#define ESW_DBCR_UP2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DBCR_UP2_SHIFT))&ESW_DBCR_UP2_MASK)
#define ESW_DBCR_TP0_MASK                        0x10000u
#define ESW_DBCR_TP0_SHIFT                       16u
#define ESW_DBCR_TP0_WIDTH                       1u
#define ESW_DBCR_TP0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DBCR_TP0_SHIFT))&ESW_DBCR_TP0_MASK)
#define ESW_DBCR_TP1_MASK                        0x20000u
#define ESW_DBCR_TP1_SHIFT                       17u
#define ESW_DBCR_TP1_WIDTH                       1u
#define ESW_DBCR_TP1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DBCR_TP1_SHIFT))&ESW_DBCR_TP1_MASK)
#define ESW_DBCR_TP2_MASK                        0x40000u
#define ESW_DBCR_TP2_SHIFT                       18u
#define ESW_DBCR_TP2_WIDTH                       1u
#define ESW_DBCR_TP2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DBCR_TP2_SHIFT))&ESW_DBCR_TP2_MASK)
/* DMCR Bit Fields */
#define ESW_DMCR_UP0_MASK                        0x1u
#define ESW_DMCR_UP0_SHIFT                       0u
#define ESW_DMCR_UP0_WIDTH                       1u
#define ESW_DMCR_UP0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DMCR_UP0_SHIFT))&ESW_DMCR_UP0_MASK)
#define ESW_DMCR_UP1_MASK                        0x2u
#define ESW_DMCR_UP1_SHIFT                       1u
#define ESW_DMCR_UP1_WIDTH                       1u
#define ESW_DMCR_UP1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DMCR_UP1_SHIFT))&ESW_DMCR_UP1_MASK)
#define ESW_DMCR_UP2_MASK                        0x4u
#define ESW_DMCR_UP2_SHIFT                       2u
#define ESW_DMCR_UP2_WIDTH                       1u
#define ESW_DMCR_UP2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DMCR_UP2_SHIFT))&ESW_DMCR_UP2_MASK)
#define ESW_DMCR_TP0_MASK                        0x10000u
#define ESW_DMCR_TP0_SHIFT                       16u
#define ESW_DMCR_TP0_WIDTH                       1u
#define ESW_DMCR_TP0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DMCR_TP0_SHIFT))&ESW_DMCR_TP0_MASK)
#define ESW_DMCR_TP1_MASK                        0x20000u
#define ESW_DMCR_TP1_SHIFT                       17u
#define ESW_DMCR_TP1_WIDTH                       1u
#define ESW_DMCR_TP1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DMCR_TP1_SHIFT))&ESW_DMCR_TP1_MASK)
#define ESW_DMCR_TP2_MASK                        0x40000u
#define ESW_DMCR_TP2_SHIFT                       18u
#define ESW_DMCR_TP2_WIDTH                       1u
#define ESW_DMCR_TP2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_DMCR_TP2_SHIFT))&ESW_DMCR_TP2_MASK)
/* BKLR Bit Fields */
#define ESW_BKLR_BE0_MASK                        0x1u
#define ESW_BKLR_BE0_SHIFT                       0u
#define ESW_BKLR_BE0_WIDTH                       1u
#define ESW_BKLR_BE0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_BKLR_BE0_SHIFT))&ESW_BKLR_BE0_MASK)
#define ESW_BKLR_BE1_MASK                        0x2u
#define ESW_BKLR_BE1_SHIFT                       1u
#define ESW_BKLR_BE1_WIDTH                       1u
#define ESW_BKLR_BE1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_BKLR_BE1_SHIFT))&ESW_BKLR_BE1_MASK)
#define ESW_BKLR_BE2_MASK                        0x4u
#define ESW_BKLR_BE2_SHIFT                       2u
#define ESW_BKLR_BE2_WIDTH                       1u
#define ESW_BKLR_BE2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_BKLR_BE2_SHIFT))&ESW_BKLR_BE2_MASK)
#define ESW_BKLR_LD0_MASK                        0x10000u
#define ESW_BKLR_LD0_SHIFT                       16u
#define ESW_BKLR_LD0_WIDTH                       1u
#define ESW_BKLR_LD0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_BKLR_LD0_SHIFT))&ESW_BKLR_LD0_MASK)
#define ESW_BKLR_LD1_MASK                        0x20000u
#define ESW_BKLR_LD1_SHIFT                       17u
#define ESW_BKLR_LD1_WIDTH                       1u
#define ESW_BKLR_LD1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_BKLR_LD1_SHIFT))&ESW_BKLR_LD1_MASK)
#define ESW_BKLR_LD2_MASK                        0x40000u
#define ESW_BKLR_LD2_SHIFT                       18u
#define ESW_BKLR_LD2_WIDTH                       1u
#define ESW_BKLR_LD2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_BKLR_LD2_SHIFT))&ESW_BKLR_LD2_MASK)
/* BMPC Bit Fields */
#define ESW_BMPC_PORT_MASK                       0xFu
#define ESW_BMPC_PORT_SHIFT                      0u
#define ESW_BMPC_PORT_WIDTH                      4u
#define ESW_BMPC_PORT(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_BMPC_PORT_SHIFT))&ESW_BMPC_PORT_MASK)
#define ESW_BMPC_MSGTX_MASK                      0x20u
#define ESW_BMPC_MSGTX_SHIFT                     5u
#define ESW_BMPC_MSGTX_WIDTH                     1u
#define ESW_BMPC_MSGTX(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_BMPC_MSGTX_SHIFT))&ESW_BMPC_MSGTX_MASK)
#define ESW_BMPC_EN_MASK                         0x40u
#define ESW_BMPC_EN_SHIFT                        6u
#define ESW_BMPC_EN_WIDTH                        1u
#define ESW_BMPC_EN(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_BMPC_EN_SHIFT))&ESW_BMPC_EN_MASK)
#define ESW_BMPC_DIS_MASK                        0x80u
#define ESW_BMPC_DIS_SHIFT                       7u
#define ESW_BMPC_DIS_WIDTH                       1u
#define ESW_BMPC_DIS(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_BMPC_DIS_SHIFT))&ESW_BMPC_DIS_MASK)
#define ESW_BMPC_PRIORITY_MASK                   0xE000u
#define ESW_BMPC_PRIORITY_SHIFT                  13u
#define ESW_BMPC_PRIORITY_WIDTH                  3u
#define ESW_BMPC_PRIORITY(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_BMPC_PRIORITY_SHIFT))&ESW_BMPC_PRIORITY_MASK)
#define ESW_BMPC_PORTMASK_MASK                   0x70000u
#define ESW_BMPC_PORTMASK_SHIFT                  16u
#define ESW_BMPC_PORTMASK_WIDTH                  3u
#define ESW_BMPC_PORTMASK(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_BMPC_PORTMASK_SHIFT))&ESW_BMPC_PORTMASK_MASK)
/* MODE Bit Fields */
#define ESW_MODE_SWRST_MASK                      0x1u
#define ESW_MODE_SWRST_SHIFT                     0u
#define ESW_MODE_SWRST_WIDTH                     1u
#define ESW_MODE_SWRST(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_MODE_SWRST_SHIFT))&ESW_MODE_SWRST_MASK)
#define ESW_MODE_SWEN_MASK                       0x2u
#define ESW_MODE_SWEN_SHIFT                      1u
#define ESW_MODE_SWEN_WIDTH                      1u
#define ESW_MODE_SWEN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_MODE_SWEN_SHIFT))&ESW_MODE_SWEN_MASK)
#define ESW_MODE_VIDTMEN_MASK                    0x4u
#define ESW_MODE_VIDTMEN_SHIFT                   2u
#define ESW_MODE_VIDTMEN_WIDTH                   1u
#define ESW_MODE_VIDTMEN(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_MODE_VIDTMEN_SHIFT))&ESW_MODE_VIDTMEN_MASK)
#define ESW_MODE_LFFC_MASK                       0x8u
#define ESW_MODE_LFFC_SHIFT                      3u
#define ESW_MODE_LFFC_WIDTH                      1u
#define ESW_MODE_LFFC(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_MODE_LFFC_SHIFT))&ESW_MODE_LFFC_MASK)
#define ESW_MODE_LBPDUD_MASK                     0x10u
#define ESW_MODE_LBPDUD_SHIFT                    4u
#define ESW_MODE_LBPDUD_WIDTH                    1u
#define ESW_MODE_LBPDUD(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_MODE_LBPDUD_SHIFT))&ESW_MODE_LBPDUD_MASK)
#define ESW_MODE_CRCTRAN_MASK                    0x100u
#define ESW_MODE_CRCTRAN_SHIFT                   8u
#define ESW_MODE_CRCTRAN_WIDTH                   1u
#define ESW_MODE_CRCTRAN(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_MODE_CRCTRAN_SHIFT))&ESW_MODE_CRCTRAN_MASK)
#define ESW_MODE_P0CT_MASK                       0x200u
#define ESW_MODE_P0CT_SHIFT                      9u
#define ESW_MODE_P0CT_WIDTH                      1u
#define ESW_MODE_P0CT(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_MODE_P0CT_SHIFT))&ESW_MODE_P0CT_MASK)
#define ESW_MODE_UCASTD_MASK                     0x700000u
#define ESW_MODE_UCASTD_SHIFT                    20u
#define ESW_MODE_UCASTD_WIDTH                    3u
#define ESW_MODE_UCASTD(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_MODE_UCASTD_SHIFT))&ESW_MODE_UCASTD_MASK)
#define ESW_MODE_STATRST_MASK                    0x80000000u
#define ESW_MODE_STATRST_SHIFT                   31u
#define ESW_MODE_STATRST_WIDTH                   1u
#define ESW_MODE_STATRST(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_MODE_STATRST_SHIFT))&ESW_MODE_STATRST_MASK)
/* VIMSEL Bit Fields */
#define ESW_VIMSEL_IM0_MASK                      0x3u
#define ESW_VIMSEL_IM0_SHIFT                     0u
#define ESW_VIMSEL_IM0_WIDTH                     2u
#define ESW_VIMSEL_IM0(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VIMSEL_IM0_SHIFT))&ESW_VIMSEL_IM0_MASK)
#define ESW_VIMSEL_IM1_MASK                      0xCu
#define ESW_VIMSEL_IM1_SHIFT                     2u
#define ESW_VIMSEL_IM1_WIDTH                     2u
#define ESW_VIMSEL_IM1(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VIMSEL_IM1_SHIFT))&ESW_VIMSEL_IM1_MASK)
#define ESW_VIMSEL_IM2_MASK                      0x30u
#define ESW_VIMSEL_IM2_SHIFT                     4u
#define ESW_VIMSEL_IM2_WIDTH                     2u
#define ESW_VIMSEL_IM2(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VIMSEL_IM2_SHIFT))&ESW_VIMSEL_IM2_MASK)
/* VOMSEL Bit Fields */
#define ESW_VOMSEL_OM0_MASK                      0x3u
#define ESW_VOMSEL_OM0_SHIFT                     0u
#define ESW_VOMSEL_OM0_WIDTH                     2u
#define ESW_VOMSEL_OM0(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VOMSEL_OM0_SHIFT))&ESW_VOMSEL_OM0_MASK)
#define ESW_VOMSEL_OM1_MASK                      0xCu
#define ESW_VOMSEL_OM1_SHIFT                     2u
#define ESW_VOMSEL_OM1_WIDTH                     2u
#define ESW_VOMSEL_OM1(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VOMSEL_OM1_SHIFT))&ESW_VOMSEL_OM1_MASK)
#define ESW_VOMSEL_OM2_MASK                      0x30u
#define ESW_VOMSEL_OM2_SHIFT                     4u
#define ESW_VOMSEL_OM2_WIDTH                     2u
#define ESW_VOMSEL_OM2(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VOMSEL_OM2_SHIFT))&ESW_VOMSEL_OM2_MASK)
/* VIMEN Bit Fields */
#define ESW_VIMEN_EN0_MASK                       0x1u
#define ESW_VIMEN_EN0_SHIFT                      0u
#define ESW_VIMEN_EN0_WIDTH                      1u
#define ESW_VIMEN_EN0(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_VIMEN_EN0_SHIFT))&ESW_VIMEN_EN0_MASK)
#define ESW_VIMEN_EN1_MASK                       0x2u
#define ESW_VIMEN_EN1_SHIFT                      1u
#define ESW_VIMEN_EN1_WIDTH                      1u
#define ESW_VIMEN_EN1(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_VIMEN_EN1_SHIFT))&ESW_VIMEN_EN1_MASK)
#define ESW_VIMEN_EN2_MASK                       0x4u
#define ESW_VIMEN_EN2_SHIFT                      2u
#define ESW_VIMEN_EN2_WIDTH                      1u
#define ESW_VIMEN_EN2(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_VIMEN_EN2_SHIFT))&ESW_VIMEN_EN2_MASK)
#define ESW_VIMEN_ETO0_MASK                      0x10000u
#define ESW_VIMEN_ETO0_SHIFT                     16u
#define ESW_VIMEN_ETO0_WIDTH                     1u
#define ESW_VIMEN_ETO0(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VIMEN_ETO0_SHIFT))&ESW_VIMEN_ETO0_MASK)
#define ESW_VIMEN_ETO1_MASK                      0x20000u
#define ESW_VIMEN_ETO1_SHIFT                     17u
#define ESW_VIMEN_ETO1_WIDTH                     1u
#define ESW_VIMEN_ETO1(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VIMEN_ETO1_SHIFT))&ESW_VIMEN_ETO1_MASK)
#define ESW_VIMEN_ETO2_MASK                      0x40000u
#define ESW_VIMEN_ETO2_SHIFT                     18u
#define ESW_VIMEN_ETO2_WIDTH                     1u
#define ESW_VIMEN_ETO2(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VIMEN_ETO2_SHIFT))&ESW_VIMEN_ETO2_MASK)
/* VID Bit Fields */
#define ESW_VID_TAG_MASK                         0xFFFFFFFFu
#define ESW_VID_TAG_SHIFT                        0u
#define ESW_VID_TAG_WIDTH                        32u
#define ESW_VID_TAG(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_VID_TAG_SHIFT))&ESW_VID_TAG_MASK)
/* RCMR Bit Fields */
#define ESW_RCMR_CMP0_MASK                       0x7u
#define ESW_RCMR_CMP0_SHIFT                      0u
#define ESW_RCMR_CMP0_WIDTH                      3u
#define ESW_RCMR_CMP0(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_RCMR_CMP0_SHIFT))&ESW_RCMR_CMP0_MASK)
#define ESW_RCMR_CMP1_MASK                       0x70u
#define ESW_RCMR_CMP1_SHIFT                      4u
#define ESW_RCMR_CMP1_WIDTH                      3u
#define ESW_RCMR_CMP1(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_RCMR_CMP1_SHIFT))&ESW_RCMR_CMP1_MASK)
#define ESW_RCMR_CMP2_MASK                       0x700u
#define ESW_RCMR_CMP2_SHIFT                      8u
#define ESW_RCMR_CMP2_WIDTH                      3u
#define ESW_RCMR_CMP2(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_RCMR_CMP2_SHIFT))&ESW_RCMR_CMP2_MASK)
#define ESW_RCMR_CMP3_MASK                       0x7000u
#define ESW_RCMR_CMP3_SHIFT                      12u
#define ESW_RCMR_CMP3_WIDTH                      3u
#define ESW_RCMR_CMP3(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_RCMR_CMP3_SHIFT))&ESW_RCMR_CMP3_MASK)
#define ESW_RCMR_MATCHEN_MASK                    0x10000u
#define ESW_RCMR_MATCHEN_SHIFT                   16u
#define ESW_RCMR_MATCHEN_WIDTH                   1u
#define ESW_RCMR_MATCHEN(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_RCMR_MATCHEN_SHIFT))&ESW_RCMR_MATCHEN_MASK)
/* MCR Bit Fields */
#define ESW_MCR_PORT_MASK                        0xFu
#define ESW_MCR_PORT_SHIFT                       0u
#define ESW_MCR_PORT_WIDTH                       4u
#define ESW_MCR_PORT(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_MCR_PORT_SHIFT))&ESW_MCR_PORT_MASK)
#define ESW_MCR_MEN_MASK                         0x10u
#define ESW_MCR_MEN_SHIFT                        4u
#define ESW_MCR_MEN_WIDTH                        1u
#define ESW_MCR_MEN(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_MCR_MEN_SHIFT))&ESW_MCR_MEN_MASK)
#define ESW_MCR_INGMAP_MASK                      0x20u
#define ESW_MCR_INGMAP_SHIFT                     5u
#define ESW_MCR_INGMAP_WIDTH                     1u
#define ESW_MCR_INGMAP(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_MCR_INGMAP_SHIFT))&ESW_MCR_INGMAP_MASK)
#define ESW_MCR_EGMAP_MASK                       0x40u
#define ESW_MCR_EGMAP_SHIFT                      6u
#define ESW_MCR_EGMAP_WIDTH                      1u
#define ESW_MCR_EGMAP(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_MCR_EGMAP_SHIFT))&ESW_MCR_EGMAP_MASK)
#define ESW_MCR_INGSA_MASK                       0x80u
#define ESW_MCR_INGSA_SHIFT                      7u
#define ESW_MCR_INGSA_WIDTH                      1u
#define ESW_MCR_INGSA(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_MCR_INGSA_SHIFT))&ESW_MCR_INGSA_MASK)
#define ESW_MCR_INGDA_MASK                       0x100u
#define ESW_MCR_INGDA_SHIFT                      8u
#define ESW_MCR_INGDA_WIDTH                      1u
#define ESW_MCR_INGDA(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_MCR_INGDA_SHIFT))&ESW_MCR_INGDA_MASK)
#define ESW_MCR_EGSA_MASK                        0x200u
#define ESW_MCR_EGSA_SHIFT                       9u
#define ESW_MCR_EGSA_WIDTH                       1u
#define ESW_MCR_EGSA(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_MCR_EGSA_SHIFT))&ESW_MCR_EGSA_MASK)
#define ESW_MCR_EGDA_MASK                        0x400u
#define ESW_MCR_EGDA_SHIFT                       10u
#define ESW_MCR_EGDA_WIDTH                       1u
#define ESW_MCR_EGDA(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_MCR_EGDA_SHIFT))&ESW_MCR_EGDA_MASK)
/* EGMAP Bit Fields */
#define ESW_EGMAP_EG0_MASK                       0x1u
#define ESW_EGMAP_EG0_SHIFT                      0u
#define ESW_EGMAP_EG0_WIDTH                      1u
#define ESW_EGMAP_EG0(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_EGMAP_EG0_SHIFT))&ESW_EGMAP_EG0_MASK)
#define ESW_EGMAP_EG1_MASK                       0x2u
#define ESW_EGMAP_EG1_SHIFT                      1u
#define ESW_EGMAP_EG1_WIDTH                      1u
#define ESW_EGMAP_EG1(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_EGMAP_EG1_SHIFT))&ESW_EGMAP_EG1_MASK)
#define ESW_EGMAP_EG2_MASK                       0x4u
#define ESW_EGMAP_EG2_SHIFT                      2u
#define ESW_EGMAP_EG2_WIDTH                      1u
#define ESW_EGMAP_EG2(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_EGMAP_EG2_SHIFT))&ESW_EGMAP_EG2_MASK)
/* INGMAP Bit Fields */
#define ESW_INGMAP_ING0_MASK                     0x1u
#define ESW_INGMAP_ING0_SHIFT                    0u
#define ESW_INGMAP_ING0_WIDTH                    1u
#define ESW_INGMAP_ING0(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_INGMAP_ING0_SHIFT))&ESW_INGMAP_ING0_MASK)
#define ESW_INGMAP_ING1_MASK                     0x2u
#define ESW_INGMAP_ING1_SHIFT                    1u
#define ESW_INGMAP_ING1_WIDTH                    1u
#define ESW_INGMAP_ING1(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_INGMAP_ING1_SHIFT))&ESW_INGMAP_ING1_MASK)
#define ESW_INGMAP_ING2_MASK                     0x4u
#define ESW_INGMAP_ING2_SHIFT                    2u
#define ESW_INGMAP_ING2_WIDTH                    1u
#define ESW_INGMAP_ING2(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_INGMAP_ING2_SHIFT))&ESW_INGMAP_ING2_MASK)
/* INGSAL Bit Fields */
#define ESW_INGSAL_ADDLOW_MASK                   0xFFFFFFFFu
#define ESW_INGSAL_ADDLOW_SHIFT                  0u
#define ESW_INGSAL_ADDLOW_WIDTH                  32u
#define ESW_INGSAL_ADDLOW(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_INGSAL_ADDLOW_SHIFT))&ESW_INGSAL_ADDLOW_MASK)
/* INGSAH Bit Fields */
#define ESW_INGSAH_ADDHIGH_MASK                  0xFFFFu
#define ESW_INGSAH_ADDHIGH_SHIFT                 0u
#define ESW_INGSAH_ADDHIGH_WIDTH                 16u
#define ESW_INGSAH_ADDHIGH(x)                    (((uint32_t)(((uint32_t)(x))<<ESW_INGSAH_ADDHIGH_SHIFT))&ESW_INGSAH_ADDHIGH_MASK)
/* INGDAL Bit Fields */
#define ESW_INGDAL_ADDLOW_MASK                   0xFFFFFFFFu
#define ESW_INGDAL_ADDLOW_SHIFT                  0u
#define ESW_INGDAL_ADDLOW_WIDTH                  32u
#define ESW_INGDAL_ADDLOW(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_INGDAL_ADDLOW_SHIFT))&ESW_INGDAL_ADDLOW_MASK)
/* INGDAH Bit Fields */
#define ESW_INGDAH_ADDHIGH_MASK                  0xFFFFu
#define ESW_INGDAH_ADDHIGH_SHIFT                 0u
#define ESW_INGDAH_ADDHIGH_WIDTH                 16u
#define ESW_INGDAH_ADDHIGH(x)                    (((uint32_t)(((uint32_t)(x))<<ESW_INGDAH_ADDHIGH_SHIFT))&ESW_INGDAH_ADDHIGH_MASK)
/* EGSAL Bit Fields */
#define ESW_EGSAL_ADDLOW_MASK                    0xFFFFFFFFu
#define ESW_EGSAL_ADDLOW_SHIFT                   0u
#define ESW_EGSAL_ADDLOW_WIDTH                   32u
#define ESW_EGSAL_ADDLOW(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_EGSAL_ADDLOW_SHIFT))&ESW_EGSAL_ADDLOW_MASK)
/* EGSAH Bit Fields */
#define ESW_EGSAH_ADDHIGH_MASK                   0xFFFFu
#define ESW_EGSAH_ADDHIGH_SHIFT                  0u
#define ESW_EGSAH_ADDHIGH_WIDTH                  16u
#define ESW_EGSAH_ADDHIGH(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_EGSAH_ADDHIGH_SHIFT))&ESW_EGSAH_ADDHIGH_MASK)
/* EGDAL Bit Fields */
#define ESW_EGDAL_ADDLOW_MASK                    0xFFFFFFFFu
#define ESW_EGDAL_ADDLOW_SHIFT                   0u
#define ESW_EGDAL_ADDLOW_WIDTH                   32u
#define ESW_EGDAL_ADDLOW(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_EGDAL_ADDLOW_SHIFT))&ESW_EGDAL_ADDLOW_MASK)
/* EGDAH Bit Fields */
#define ESW_EGDAH_ADDHIGH_MASK                   0xFFFFu
#define ESW_EGDAH_ADDHIGH_SHIFT                  0u
#define ESW_EGDAH_ADDHIGH_WIDTH                  16u
#define ESW_EGDAH_ADDHIGH(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_EGDAH_ADDHIGH_SHIFT))&ESW_EGDAH_ADDHIGH_MASK)
/* MCVAL Bit Fields */
#define ESW_MCVAL_COUNT_MASK                     0xFFu
#define ESW_MCVAL_COUNT_SHIFT                    0u
#define ESW_MCVAL_COUNT_WIDTH                    8u
#define ESW_MCVAL_COUNT(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_MCVAL_COUNT_SHIFT))&ESW_MCVAL_COUNT_MASK)
/* MMSR Bit Fields */
#define ESW_MMSR_BUSY_MASK                       0x1u
#define ESW_MMSR_BUSY_SHIFT                      0u
#define ESW_MMSR_BUSY_WIDTH                      1u
#define ESW_MMSR_BUSY(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_MMSR_BUSY_SHIFT))&ESW_MMSR_BUSY_MASK)
#define ESW_MMSR_NOCELL_MASK                     0x2u
#define ESW_MMSR_NOCELL_SHIFT                    1u
#define ESW_MMSR_NOCELL_WIDTH                    1u
#define ESW_MMSR_NOCELL(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_MMSR_NOCELL_SHIFT))&ESW_MMSR_NOCELL_MASK)
#define ESW_MMSR_MEMFULL_MASK                    0x4u
#define ESW_MMSR_MEMFULL_SHIFT                   2u
#define ESW_MMSR_MEMFULL_WIDTH                   1u
#define ESW_MMSR_MEMFULL(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_MMSR_MEMFULL_SHIFT))&ESW_MMSR_MEMFULL_MASK)
#define ESW_MMSR_MFLATCH_MASK                    0x8u
#define ESW_MMSR_MFLATCH_SHIFT                   3u
#define ESW_MMSR_MFLATCH_WIDTH                   1u
#define ESW_MMSR_MFLATCH(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_MMSR_MFLATCH_SHIFT))&ESW_MMSR_MFLATCH_MASK)
#define ESW_MMSR_DQGRNT_MASK                     0x40u
#define ESW_MMSR_DQGRNT_SHIFT                    6u
#define ESW_MMSR_DQGRNT_WIDTH                    1u
#define ESW_MMSR_DQGRNT(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_MMSR_DQGRNT_SHIFT))&ESW_MMSR_DQGRNT_MASK)
#define ESW_MMSR_CELLS_AVAIL_MASK                0xFF0000u
#define ESW_MMSR_CELLS_AVAIL_SHIFT               16u
#define ESW_MMSR_CELLS_AVAIL_WIDTH               8u
#define ESW_MMSR_CELLS_AVAIL(x)                  (((uint32_t)(((uint32_t)(x))<<ESW_MMSR_CELLS_AVAIL_SHIFT))&ESW_MMSR_CELLS_AVAIL_MASK)
/* LMT Bit Fields */
#define ESW_LMT_THRESH_MASK                      0xFFu
#define ESW_LMT_THRESH_SHIFT                     0u
#define ESW_LMT_THRESH_WIDTH                     8u
#define ESW_LMT_THRESH(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_LMT_THRESH_SHIFT))&ESW_LMT_THRESH_MASK)
/* LFC Bit Fields */
#define ESW_LFC_COUNT_MASK                       0xFFFFFFFFu
#define ESW_LFC_COUNT_SHIFT                      0u
#define ESW_LFC_COUNT_WIDTH                      32u
#define ESW_LFC_COUNT(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_LFC_COUNT_SHIFT))&ESW_LFC_COUNT_MASK)
/* PCSR Bit Fields */
#define ESW_PCSR_PC0_MASK                        0x1u
#define ESW_PCSR_PC0_SHIFT                       0u
#define ESW_PCSR_PC0_WIDTH                       1u
#define ESW_PCSR_PC0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PCSR_PC0_SHIFT))&ESW_PCSR_PC0_MASK)
#define ESW_PCSR_PC1_MASK                        0x2u
#define ESW_PCSR_PC1_SHIFT                       1u
#define ESW_PCSR_PC1_WIDTH                       1u
#define ESW_PCSR_PC1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PCSR_PC1_SHIFT))&ESW_PCSR_PC1_MASK)
#define ESW_PCSR_PC2_MASK                        0x4u
#define ESW_PCSR_PC2_SHIFT                       2u
#define ESW_PCSR_PC2_WIDTH                       1u
#define ESW_PCSR_PC2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PCSR_PC2_SHIFT))&ESW_PCSR_PC2_MASK)
/* IOSR Bit Fields */
#define ESW_IOSR_OR0_MASK                        0x1u
#define ESW_IOSR_OR0_SHIFT                       0u
#define ESW_IOSR_OR0_WIDTH                       1u
#define ESW_IOSR_OR0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IOSR_OR0_SHIFT))&ESW_IOSR_OR0_MASK)
#define ESW_IOSR_OR1_MASK                        0x2u
#define ESW_IOSR_OR1_SHIFT                       1u
#define ESW_IOSR_OR1_WIDTH                       1u
#define ESW_IOSR_OR1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IOSR_OR1_SHIFT))&ESW_IOSR_OR1_MASK)
#define ESW_IOSR_OR2_MASK                        0x4u
#define ESW_IOSR_OR2_SHIFT                       2u
#define ESW_IOSR_OR2_WIDTH                       1u
#define ESW_IOSR_OR2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IOSR_OR2_SHIFT))&ESW_IOSR_OR2_MASK)
#define ESW_IOSR_IR0_MASK                        0x10000u
#define ESW_IOSR_IR0_SHIFT                       16u
#define ESW_IOSR_IR0_WIDTH                       1u
#define ESW_IOSR_IR0(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IOSR_IR0_SHIFT))&ESW_IOSR_IR0_MASK)
#define ESW_IOSR_IR1_MASK                        0x20000u
#define ESW_IOSR_IR1_SHIFT                       17u
#define ESW_IOSR_IR1_WIDTH                       1u
#define ESW_IOSR_IR1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IOSR_IR1_SHIFT))&ESW_IOSR_IR1_MASK)
#define ESW_IOSR_IR2_MASK                        0x40000u
#define ESW_IOSR_IR2_SHIFT                       18u
#define ESW_IOSR_IR2_WIDTH                       1u
#define ESW_IOSR_IR2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IOSR_IR2_SHIFT))&ESW_IOSR_IR2_MASK)
/* QWT Bit Fields */
#define ESW_QWT_Q0WT_MASK                        0x1Fu
#define ESW_QWT_Q0WT_SHIFT                       0u
#define ESW_QWT_Q0WT_WIDTH                       5u
#define ESW_QWT_Q0WT(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_QWT_Q0WT_SHIFT))&ESW_QWT_Q0WT_MASK)
#define ESW_QWT_Q1WT_MASK                        0x1F00u
#define ESW_QWT_Q1WT_SHIFT                       8u
#define ESW_QWT_Q1WT_WIDTH                       5u
#define ESW_QWT_Q1WT(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_QWT_Q1WT_SHIFT))&ESW_QWT_Q1WT_MASK)
#define ESW_QWT_Q2WT_MASK                        0x1F0000u
#define ESW_QWT_Q2WT_SHIFT                       16u
#define ESW_QWT_Q2WT_WIDTH                       5u
#define ESW_QWT_Q2WT(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_QWT_Q2WT_SHIFT))&ESW_QWT_Q2WT_MASK)
#define ESW_QWT_Q3WT_MASK                        0x1F000000u
#define ESW_QWT_Q3WT_SHIFT                       24u
#define ESW_QWT_Q3WT_WIDTH                       5u
#define ESW_QWT_Q3WT(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_QWT_Q3WT_SHIFT))&ESW_QWT_Q3WT_MASK)
/* P0BCT Bit Fields */
#define ESW_P0BCT_THRESH_MASK                    0xFFu
#define ESW_P0BCT_THRESH_SHIFT                   0u
#define ESW_P0BCT_THRESH_WIDTH                   8u
#define ESW_P0BCT_THRESH(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_P0BCT_THRESH_SHIFT))&ESW_P0BCT_THRESH_MASK)
/* FFEN Bit Fields */
#define ESW_FFEN_FEN_MASK                        0x1u
#define ESW_FFEN_FEN_SHIFT                       0u
#define ESW_FFEN_FEN_WIDTH                       1u
#define ESW_FFEN_FEN(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_FFEN_FEN_SHIFT))&ESW_FFEN_FEN_MASK)
#define ESW_FFEN_FD_MASK                         0xCu
#define ESW_FFEN_FD_SHIFT                        2u
#define ESW_FFEN_FD_WIDTH                        2u
#define ESW_FFEN_FD(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_FFEN_FD_SHIFT))&ESW_FFEN_FD_MASK)
/* PSNP1 Bit Fields */
#define ESW_PSNP1_EN_MASK                        0x1u
#define ESW_PSNP1_EN_SHIFT                       0u
#define ESW_PSNP1_EN_WIDTH                       1u
#define ESW_PSNP1_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP1_EN_SHIFT))&ESW_PSNP1_EN_MASK)
#define ESW_PSNP1_MODE_MASK                      0x6u
#define ESW_PSNP1_MODE_SHIFT                     1u
#define ESW_PSNP1_MODE_WIDTH                     2u
#define ESW_PSNP1_MODE(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_PSNP1_MODE_SHIFT))&ESW_PSNP1_MODE_MASK)
#define ESW_PSNP1_CD_MASK                        0x8u
#define ESW_PSNP1_CD_SHIFT                       3u
#define ESW_PSNP1_CD_WIDTH                       1u
#define ESW_PSNP1_CD(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP1_CD_SHIFT))&ESW_PSNP1_CD_MASK)
#define ESW_PSNP1_CS_MASK                        0x10u
#define ESW_PSNP1_CS_SHIFT                       4u
#define ESW_PSNP1_CS_WIDTH                       1u
#define ESW_PSNP1_CS(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP1_CS_SHIFT))&ESW_PSNP1_CS_MASK)
#define ESW_PSNP1_PORT_COMPARE_MASK              0xFFFF0000u
#define ESW_PSNP1_PORT_COMPARE_SHIFT             16u
#define ESW_PSNP1_PORT_COMPARE_WIDTH             16u
#define ESW_PSNP1_PORT_COMPARE(x)                (((uint32_t)(((uint32_t)(x))<<ESW_PSNP1_PORT_COMPARE_SHIFT))&ESW_PSNP1_PORT_COMPARE_MASK)
/* PSNP2 Bit Fields */
#define ESW_PSNP2_EN_MASK                        0x1u
#define ESW_PSNP2_EN_SHIFT                       0u
#define ESW_PSNP2_EN_WIDTH                       1u
#define ESW_PSNP2_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP2_EN_SHIFT))&ESW_PSNP2_EN_MASK)
#define ESW_PSNP2_MODE_MASK                      0x6u
#define ESW_PSNP2_MODE_SHIFT                     1u
#define ESW_PSNP2_MODE_WIDTH                     2u
#define ESW_PSNP2_MODE(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_PSNP2_MODE_SHIFT))&ESW_PSNP2_MODE_MASK)
#define ESW_PSNP2_CD_MASK                        0x8u
#define ESW_PSNP2_CD_SHIFT                       3u
#define ESW_PSNP2_CD_WIDTH                       1u
#define ESW_PSNP2_CD(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP2_CD_SHIFT))&ESW_PSNP2_CD_MASK)
#define ESW_PSNP2_CS_MASK                        0x10u
#define ESW_PSNP2_CS_SHIFT                       4u
#define ESW_PSNP2_CS_WIDTH                       1u
#define ESW_PSNP2_CS(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP2_CS_SHIFT))&ESW_PSNP2_CS_MASK)
#define ESW_PSNP2_PORT_COMPARE_MASK              0xFFFF0000u
#define ESW_PSNP2_PORT_COMPARE_SHIFT             16u
#define ESW_PSNP2_PORT_COMPARE_WIDTH             16u
#define ESW_PSNP2_PORT_COMPARE(x)                (((uint32_t)(((uint32_t)(x))<<ESW_PSNP2_PORT_COMPARE_SHIFT))&ESW_PSNP2_PORT_COMPARE_MASK)
/* PSNP3 Bit Fields */
#define ESW_PSNP3_EN_MASK                        0x1u
#define ESW_PSNP3_EN_SHIFT                       0u
#define ESW_PSNP3_EN_WIDTH                       1u
#define ESW_PSNP3_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP3_EN_SHIFT))&ESW_PSNP3_EN_MASK)
#define ESW_PSNP3_MODE_MASK                      0x6u
#define ESW_PSNP3_MODE_SHIFT                     1u
#define ESW_PSNP3_MODE_WIDTH                     2u
#define ESW_PSNP3_MODE(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_PSNP3_MODE_SHIFT))&ESW_PSNP3_MODE_MASK)
#define ESW_PSNP3_CD_MASK                        0x8u
#define ESW_PSNP3_CD_SHIFT                       3u
#define ESW_PSNP3_CD_WIDTH                       1u
#define ESW_PSNP3_CD(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP3_CD_SHIFT))&ESW_PSNP3_CD_MASK)
#define ESW_PSNP3_CS_MASK                        0x10u
#define ESW_PSNP3_CS_SHIFT                       4u
#define ESW_PSNP3_CS_WIDTH                       1u
#define ESW_PSNP3_CS(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP3_CS_SHIFT))&ESW_PSNP3_CS_MASK)
#define ESW_PSNP3_PORT_COMPARE_MASK              0xFFFF0000u
#define ESW_PSNP3_PORT_COMPARE_SHIFT             16u
#define ESW_PSNP3_PORT_COMPARE_WIDTH             16u
#define ESW_PSNP3_PORT_COMPARE(x)                (((uint32_t)(((uint32_t)(x))<<ESW_PSNP3_PORT_COMPARE_SHIFT))&ESW_PSNP3_PORT_COMPARE_MASK)
/* PSNP4 Bit Fields */
#define ESW_PSNP4_EN_MASK                        0x1u
#define ESW_PSNP4_EN_SHIFT                       0u
#define ESW_PSNP4_EN_WIDTH                       1u
#define ESW_PSNP4_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP4_EN_SHIFT))&ESW_PSNP4_EN_MASK)
#define ESW_PSNP4_MODE_MASK                      0x6u
#define ESW_PSNP4_MODE_SHIFT                     1u
#define ESW_PSNP4_MODE_WIDTH                     2u
#define ESW_PSNP4_MODE(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_PSNP4_MODE_SHIFT))&ESW_PSNP4_MODE_MASK)
#define ESW_PSNP4_CD_MASK                        0x8u
#define ESW_PSNP4_CD_SHIFT                       3u
#define ESW_PSNP4_CD_WIDTH                       1u
#define ESW_PSNP4_CD(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP4_CD_SHIFT))&ESW_PSNP4_CD_MASK)
#define ESW_PSNP4_CS_MASK                        0x10u
#define ESW_PSNP4_CS_SHIFT                       4u
#define ESW_PSNP4_CS_WIDTH                       1u
#define ESW_PSNP4_CS(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP4_CS_SHIFT))&ESW_PSNP4_CS_MASK)
#define ESW_PSNP4_PORT_COMPARE_MASK              0xFFFF0000u
#define ESW_PSNP4_PORT_COMPARE_SHIFT             16u
#define ESW_PSNP4_PORT_COMPARE_WIDTH             16u
#define ESW_PSNP4_PORT_COMPARE(x)                (((uint32_t)(((uint32_t)(x))<<ESW_PSNP4_PORT_COMPARE_SHIFT))&ESW_PSNP4_PORT_COMPARE_MASK)
/* PSNP5 Bit Fields */
#define ESW_PSNP5_EN_MASK                        0x1u
#define ESW_PSNP5_EN_SHIFT                       0u
#define ESW_PSNP5_EN_WIDTH                       1u
#define ESW_PSNP5_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP5_EN_SHIFT))&ESW_PSNP5_EN_MASK)
#define ESW_PSNP5_MODE_MASK                      0x6u
#define ESW_PSNP5_MODE_SHIFT                     1u
#define ESW_PSNP5_MODE_WIDTH                     2u
#define ESW_PSNP5_MODE(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_PSNP5_MODE_SHIFT))&ESW_PSNP5_MODE_MASK)
#define ESW_PSNP5_CD_MASK                        0x8u
#define ESW_PSNP5_CD_SHIFT                       3u
#define ESW_PSNP5_CD_WIDTH                       1u
#define ESW_PSNP5_CD(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP5_CD_SHIFT))&ESW_PSNP5_CD_MASK)
#define ESW_PSNP5_CS_MASK                        0x10u
#define ESW_PSNP5_CS_SHIFT                       4u
#define ESW_PSNP5_CS_WIDTH                       1u
#define ESW_PSNP5_CS(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP5_CS_SHIFT))&ESW_PSNP5_CS_MASK)
#define ESW_PSNP5_PORT_COMPARE_MASK              0xFFFF0000u
#define ESW_PSNP5_PORT_COMPARE_SHIFT             16u
#define ESW_PSNP5_PORT_COMPARE_WIDTH             16u
#define ESW_PSNP5_PORT_COMPARE(x)                (((uint32_t)(((uint32_t)(x))<<ESW_PSNP5_PORT_COMPARE_SHIFT))&ESW_PSNP5_PORT_COMPARE_MASK)
/* PSNP6 Bit Fields */
#define ESW_PSNP6_EN_MASK                        0x1u
#define ESW_PSNP6_EN_SHIFT                       0u
#define ESW_PSNP6_EN_WIDTH                       1u
#define ESW_PSNP6_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP6_EN_SHIFT))&ESW_PSNP6_EN_MASK)
#define ESW_PSNP6_MODE_MASK                      0x6u
#define ESW_PSNP6_MODE_SHIFT                     1u
#define ESW_PSNP6_MODE_WIDTH                     2u
#define ESW_PSNP6_MODE(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_PSNP6_MODE_SHIFT))&ESW_PSNP6_MODE_MASK)
#define ESW_PSNP6_CD_MASK                        0x8u
#define ESW_PSNP6_CD_SHIFT                       3u
#define ESW_PSNP6_CD_WIDTH                       1u
#define ESW_PSNP6_CD(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP6_CD_SHIFT))&ESW_PSNP6_CD_MASK)
#define ESW_PSNP6_CS_MASK                        0x10u
#define ESW_PSNP6_CS_SHIFT                       4u
#define ESW_PSNP6_CS_WIDTH                       1u
#define ESW_PSNP6_CS(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP6_CS_SHIFT))&ESW_PSNP6_CS_MASK)
#define ESW_PSNP6_PORT_COMPARE_MASK              0xFFFF0000u
#define ESW_PSNP6_PORT_COMPARE_SHIFT             16u
#define ESW_PSNP6_PORT_COMPARE_WIDTH             16u
#define ESW_PSNP6_PORT_COMPARE(x)                (((uint32_t)(((uint32_t)(x))<<ESW_PSNP6_PORT_COMPARE_SHIFT))&ESW_PSNP6_PORT_COMPARE_MASK)
/* PSNP7 Bit Fields */
#define ESW_PSNP7_EN_MASK                        0x1u
#define ESW_PSNP7_EN_SHIFT                       0u
#define ESW_PSNP7_EN_WIDTH                       1u
#define ESW_PSNP7_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP7_EN_SHIFT))&ESW_PSNP7_EN_MASK)
#define ESW_PSNP7_MODE_MASK                      0x6u
#define ESW_PSNP7_MODE_SHIFT                     1u
#define ESW_PSNP7_MODE_WIDTH                     2u
#define ESW_PSNP7_MODE(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_PSNP7_MODE_SHIFT))&ESW_PSNP7_MODE_MASK)
#define ESW_PSNP7_CD_MASK                        0x8u
#define ESW_PSNP7_CD_SHIFT                       3u
#define ESW_PSNP7_CD_WIDTH                       1u
#define ESW_PSNP7_CD(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP7_CD_SHIFT))&ESW_PSNP7_CD_MASK)
#define ESW_PSNP7_CS_MASK                        0x10u
#define ESW_PSNP7_CS_SHIFT                       4u
#define ESW_PSNP7_CS_WIDTH                       1u
#define ESW_PSNP7_CS(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP7_CS_SHIFT))&ESW_PSNP7_CS_MASK)
#define ESW_PSNP7_PORT_COMPARE_MASK              0xFFFF0000u
#define ESW_PSNP7_PORT_COMPARE_SHIFT             16u
#define ESW_PSNP7_PORT_COMPARE_WIDTH             16u
#define ESW_PSNP7_PORT_COMPARE(x)                (((uint32_t)(((uint32_t)(x))<<ESW_PSNP7_PORT_COMPARE_SHIFT))&ESW_PSNP7_PORT_COMPARE_MASK)
/* PSNP8 Bit Fields */
#define ESW_PSNP8_EN_MASK                        0x1u
#define ESW_PSNP8_EN_SHIFT                       0u
#define ESW_PSNP8_EN_WIDTH                       1u
#define ESW_PSNP8_EN(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP8_EN_SHIFT))&ESW_PSNP8_EN_MASK)
#define ESW_PSNP8_MODE_MASK                      0x6u
#define ESW_PSNP8_MODE_SHIFT                     1u
#define ESW_PSNP8_MODE_WIDTH                     2u
#define ESW_PSNP8_MODE(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_PSNP8_MODE_SHIFT))&ESW_PSNP8_MODE_MASK)
#define ESW_PSNP8_CD_MASK                        0x8u
#define ESW_PSNP8_CD_SHIFT                       3u
#define ESW_PSNP8_CD_WIDTH                       1u
#define ESW_PSNP8_CD(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP8_CD_SHIFT))&ESW_PSNP8_CD_MASK)
#define ESW_PSNP8_CS_MASK                        0x10u
#define ESW_PSNP8_CS_SHIFT                       4u
#define ESW_PSNP8_CS_WIDTH                       1u
#define ESW_PSNP8_CS(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_PSNP8_CS_SHIFT))&ESW_PSNP8_CS_MASK)
#define ESW_PSNP8_PORT_COMPARE_MASK              0xFFFF0000u
#define ESW_PSNP8_PORT_COMPARE_SHIFT             16u
#define ESW_PSNP8_PORT_COMPARE_WIDTH             16u
#define ESW_PSNP8_PORT_COMPARE(x)                (((uint32_t)(((uint32_t)(x))<<ESW_PSNP8_PORT_COMPARE_SHIFT))&ESW_PSNP8_PORT_COMPARE_MASK)
/* IPSNP1 Bit Fields */
#define ESW_IPSNP1_EN_MASK                       0x1u
#define ESW_IPSNP1_EN_SHIFT                      0u
#define ESW_IPSNP1_EN_WIDTH                      1u
#define ESW_IPSNP1_EN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP1_EN_SHIFT))&ESW_IPSNP1_EN_MASK)
#define ESW_IPSNP1_MODE_MASK                     0x6u
#define ESW_IPSNP1_MODE_SHIFT                    1u
#define ESW_IPSNP1_MODE_WIDTH                    2u
#define ESW_IPSNP1_MODE(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP1_MODE_SHIFT))&ESW_IPSNP1_MODE_MASK)
#define ESW_IPSNP1_PROTOCOL_MASK                 0xFF00u
#define ESW_IPSNP1_PROTOCOL_SHIFT                8u
#define ESW_IPSNP1_PROTOCOL_WIDTH                8u
#define ESW_IPSNP1_PROTOCOL(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP1_PROTOCOL_SHIFT))&ESW_IPSNP1_PROTOCOL_MASK)
/* IPSNP2 Bit Fields */
#define ESW_IPSNP2_EN_MASK                       0x1u
#define ESW_IPSNP2_EN_SHIFT                      0u
#define ESW_IPSNP2_EN_WIDTH                      1u
#define ESW_IPSNP2_EN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP2_EN_SHIFT))&ESW_IPSNP2_EN_MASK)
#define ESW_IPSNP2_MODE_MASK                     0x6u
#define ESW_IPSNP2_MODE_SHIFT                    1u
#define ESW_IPSNP2_MODE_WIDTH                    2u
#define ESW_IPSNP2_MODE(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP2_MODE_SHIFT))&ESW_IPSNP2_MODE_MASK)
#define ESW_IPSNP2_PROTOCOL_MASK                 0xFF00u
#define ESW_IPSNP2_PROTOCOL_SHIFT                8u
#define ESW_IPSNP2_PROTOCOL_WIDTH                8u
#define ESW_IPSNP2_PROTOCOL(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP2_PROTOCOL_SHIFT))&ESW_IPSNP2_PROTOCOL_MASK)
/* IPSNP3 Bit Fields */
#define ESW_IPSNP3_EN_MASK                       0x1u
#define ESW_IPSNP3_EN_SHIFT                      0u
#define ESW_IPSNP3_EN_WIDTH                      1u
#define ESW_IPSNP3_EN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP3_EN_SHIFT))&ESW_IPSNP3_EN_MASK)
#define ESW_IPSNP3_MODE_MASK                     0x6u
#define ESW_IPSNP3_MODE_SHIFT                    1u
#define ESW_IPSNP3_MODE_WIDTH                    2u
#define ESW_IPSNP3_MODE(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP3_MODE_SHIFT))&ESW_IPSNP3_MODE_MASK)
#define ESW_IPSNP3_PROTOCOL_MASK                 0xFF00u
#define ESW_IPSNP3_PROTOCOL_SHIFT                8u
#define ESW_IPSNP3_PROTOCOL_WIDTH                8u
#define ESW_IPSNP3_PROTOCOL(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP3_PROTOCOL_SHIFT))&ESW_IPSNP3_PROTOCOL_MASK)
/* IPSNP4 Bit Fields */
#define ESW_IPSNP4_EN_MASK                       0x1u
#define ESW_IPSNP4_EN_SHIFT                      0u
#define ESW_IPSNP4_EN_WIDTH                      1u
#define ESW_IPSNP4_EN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP4_EN_SHIFT))&ESW_IPSNP4_EN_MASK)
#define ESW_IPSNP4_MODE_MASK                     0x6u
#define ESW_IPSNP4_MODE_SHIFT                    1u
#define ESW_IPSNP4_MODE_WIDTH                    2u
#define ESW_IPSNP4_MODE(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP4_MODE_SHIFT))&ESW_IPSNP4_MODE_MASK)
#define ESW_IPSNP4_PROTOCOL_MASK                 0xFF00u
#define ESW_IPSNP4_PROTOCOL_SHIFT                8u
#define ESW_IPSNP4_PROTOCOL_WIDTH                8u
#define ESW_IPSNP4_PROTOCOL(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP4_PROTOCOL_SHIFT))&ESW_IPSNP4_PROTOCOL_MASK)
/* IPSNP5 Bit Fields */
#define ESW_IPSNP5_EN_MASK                       0x1u
#define ESW_IPSNP5_EN_SHIFT                      0u
#define ESW_IPSNP5_EN_WIDTH                      1u
#define ESW_IPSNP5_EN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP5_EN_SHIFT))&ESW_IPSNP5_EN_MASK)
#define ESW_IPSNP5_MODE_MASK                     0x6u
#define ESW_IPSNP5_MODE_SHIFT                    1u
#define ESW_IPSNP5_MODE_WIDTH                    2u
#define ESW_IPSNP5_MODE(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP5_MODE_SHIFT))&ESW_IPSNP5_MODE_MASK)
#define ESW_IPSNP5_PROTOCOL_MASK                 0xFF00u
#define ESW_IPSNP5_PROTOCOL_SHIFT                8u
#define ESW_IPSNP5_PROTOCOL_WIDTH                8u
#define ESW_IPSNP5_PROTOCOL(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP5_PROTOCOL_SHIFT))&ESW_IPSNP5_PROTOCOL_MASK)
/* IPSNP6 Bit Fields */
#define ESW_IPSNP6_EN_MASK                       0x1u
#define ESW_IPSNP6_EN_SHIFT                      0u
#define ESW_IPSNP6_EN_WIDTH                      1u
#define ESW_IPSNP6_EN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP6_EN_SHIFT))&ESW_IPSNP6_EN_MASK)
#define ESW_IPSNP6_MODE_MASK                     0x6u
#define ESW_IPSNP6_MODE_SHIFT                    1u
#define ESW_IPSNP6_MODE_WIDTH                    2u
#define ESW_IPSNP6_MODE(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP6_MODE_SHIFT))&ESW_IPSNP6_MODE_MASK)
#define ESW_IPSNP6_PROTOCOL_MASK                 0xFF00u
#define ESW_IPSNP6_PROTOCOL_SHIFT                8u
#define ESW_IPSNP6_PROTOCOL_WIDTH                8u
#define ESW_IPSNP6_PROTOCOL(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP6_PROTOCOL_SHIFT))&ESW_IPSNP6_PROTOCOL_MASK)
/* IPSNP7 Bit Fields */
#define ESW_IPSNP7_EN_MASK                       0x1u
#define ESW_IPSNP7_EN_SHIFT                      0u
#define ESW_IPSNP7_EN_WIDTH                      1u
#define ESW_IPSNP7_EN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP7_EN_SHIFT))&ESW_IPSNP7_EN_MASK)
#define ESW_IPSNP7_MODE_MASK                     0x6u
#define ESW_IPSNP7_MODE_SHIFT                    1u
#define ESW_IPSNP7_MODE_WIDTH                    2u
#define ESW_IPSNP7_MODE(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP7_MODE_SHIFT))&ESW_IPSNP7_MODE_MASK)
#define ESW_IPSNP7_PROTOCOL_MASK                 0xFF00u
#define ESW_IPSNP7_PROTOCOL_SHIFT                8u
#define ESW_IPSNP7_PROTOCOL_WIDTH                8u
#define ESW_IPSNP7_PROTOCOL(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP7_PROTOCOL_SHIFT))&ESW_IPSNP7_PROTOCOL_MASK)
/* IPSNP8 Bit Fields */
#define ESW_IPSNP8_EN_MASK                       0x1u
#define ESW_IPSNP8_EN_SHIFT                      0u
#define ESW_IPSNP8_EN_WIDTH                      1u
#define ESW_IPSNP8_EN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP8_EN_SHIFT))&ESW_IPSNP8_EN_MASK)
#define ESW_IPSNP8_MODE_MASK                     0x6u
#define ESW_IPSNP8_MODE_SHIFT                    1u
#define ESW_IPSNP8_MODE_WIDTH                    2u
#define ESW_IPSNP8_MODE(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP8_MODE_SHIFT))&ESW_IPSNP8_MODE_MASK)
#define ESW_IPSNP8_PROTOCOL_MASK                 0xFF00u
#define ESW_IPSNP8_PROTOCOL_SHIFT                8u
#define ESW_IPSNP8_PROTOCOL_WIDTH                8u
#define ESW_IPSNP8_PROTOCOL(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_IPSNP8_PROTOCOL_SHIFT))&ESW_IPSNP8_PROTOCOL_MASK)
/* P0VRES Bit Fields */
#define ESW_P0VRES_PRI0_MASK                     0x7u
#define ESW_P0VRES_PRI0_SHIFT                    0u
#define ESW_P0VRES_PRI0_WIDTH                    3u
#define ESW_P0VRES_PRI0(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0VRES_PRI0_SHIFT))&ESW_P0VRES_PRI0_MASK)
#define ESW_P0VRES_PRI1_MASK                     0x38u
#define ESW_P0VRES_PRI1_SHIFT                    3u
#define ESW_P0VRES_PRI1_WIDTH                    3u
#define ESW_P0VRES_PRI1(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0VRES_PRI1_SHIFT))&ESW_P0VRES_PRI1_MASK)
#define ESW_P0VRES_PRI2_MASK                     0x1C0u
#define ESW_P0VRES_PRI2_SHIFT                    6u
#define ESW_P0VRES_PRI2_WIDTH                    3u
#define ESW_P0VRES_PRI2(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0VRES_PRI2_SHIFT))&ESW_P0VRES_PRI2_MASK)
#define ESW_P0VRES_PRI3_MASK                     0xE00u
#define ESW_P0VRES_PRI3_SHIFT                    9u
#define ESW_P0VRES_PRI3_WIDTH                    3u
#define ESW_P0VRES_PRI3(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0VRES_PRI3_SHIFT))&ESW_P0VRES_PRI3_MASK)
#define ESW_P0VRES_PRI4_MASK                     0x7000u
#define ESW_P0VRES_PRI4_SHIFT                    12u
#define ESW_P0VRES_PRI4_WIDTH                    3u
#define ESW_P0VRES_PRI4(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0VRES_PRI4_SHIFT))&ESW_P0VRES_PRI4_MASK)
#define ESW_P0VRES_PRI5_MASK                     0x38000u
#define ESW_P0VRES_PRI5_SHIFT                    15u
#define ESW_P0VRES_PRI5_WIDTH                    3u
#define ESW_P0VRES_PRI5(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0VRES_PRI5_SHIFT))&ESW_P0VRES_PRI5_MASK)
#define ESW_P0VRES_PRI6_MASK                     0x1C0000u
#define ESW_P0VRES_PRI6_SHIFT                    18u
#define ESW_P0VRES_PRI6_WIDTH                    3u
#define ESW_P0VRES_PRI6(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0VRES_PRI6_SHIFT))&ESW_P0VRES_PRI6_MASK)
#define ESW_P0VRES_PRI7_MASK                     0xE00000u
#define ESW_P0VRES_PRI7_SHIFT                    21u
#define ESW_P0VRES_PRI7_WIDTH                    3u
#define ESW_P0VRES_PRI7(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0VRES_PRI7_SHIFT))&ESW_P0VRES_PRI7_MASK)
/* P1VRES Bit Fields */
#define ESW_P1VRES_PRI0_MASK                     0x7u
#define ESW_P1VRES_PRI0_SHIFT                    0u
#define ESW_P1VRES_PRI0_WIDTH                    3u
#define ESW_P1VRES_PRI0(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1VRES_PRI0_SHIFT))&ESW_P1VRES_PRI0_MASK)
#define ESW_P1VRES_PRI1_MASK                     0x38u
#define ESW_P1VRES_PRI1_SHIFT                    3u
#define ESW_P1VRES_PRI1_WIDTH                    3u
#define ESW_P1VRES_PRI1(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1VRES_PRI1_SHIFT))&ESW_P1VRES_PRI1_MASK)
#define ESW_P1VRES_PRI2_MASK                     0x1C0u
#define ESW_P1VRES_PRI2_SHIFT                    6u
#define ESW_P1VRES_PRI2_WIDTH                    3u
#define ESW_P1VRES_PRI2(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1VRES_PRI2_SHIFT))&ESW_P1VRES_PRI2_MASK)
#define ESW_P1VRES_PRI3_MASK                     0xE00u
#define ESW_P1VRES_PRI3_SHIFT                    9u
#define ESW_P1VRES_PRI3_WIDTH                    3u
#define ESW_P1VRES_PRI3(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1VRES_PRI3_SHIFT))&ESW_P1VRES_PRI3_MASK)
#define ESW_P1VRES_PRI4_MASK                     0x7000u
#define ESW_P1VRES_PRI4_SHIFT                    12u
#define ESW_P1VRES_PRI4_WIDTH                    3u
#define ESW_P1VRES_PRI4(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1VRES_PRI4_SHIFT))&ESW_P1VRES_PRI4_MASK)
#define ESW_P1VRES_PRI5_MASK                     0x38000u
#define ESW_P1VRES_PRI5_SHIFT                    15u
#define ESW_P1VRES_PRI5_WIDTH                    3u
#define ESW_P1VRES_PRI5(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1VRES_PRI5_SHIFT))&ESW_P1VRES_PRI5_MASK)
#define ESW_P1VRES_PRI6_MASK                     0x1C0000u
#define ESW_P1VRES_PRI6_SHIFT                    18u
#define ESW_P1VRES_PRI6_WIDTH                    3u
#define ESW_P1VRES_PRI6(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1VRES_PRI6_SHIFT))&ESW_P1VRES_PRI6_MASK)
#define ESW_P1VRES_PRI7_MASK                     0xE00000u
#define ESW_P1VRES_PRI7_SHIFT                    21u
#define ESW_P1VRES_PRI7_WIDTH                    3u
#define ESW_P1VRES_PRI7(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1VRES_PRI7_SHIFT))&ESW_P1VRES_PRI7_MASK)
/* P2VRES Bit Fields */
#define ESW_P2VRES_PRI0_MASK                     0x7u
#define ESW_P2VRES_PRI0_SHIFT                    0u
#define ESW_P2VRES_PRI0_WIDTH                    3u
#define ESW_P2VRES_PRI0(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2VRES_PRI0_SHIFT))&ESW_P2VRES_PRI0_MASK)
#define ESW_P2VRES_PRI1_MASK                     0x38u
#define ESW_P2VRES_PRI1_SHIFT                    3u
#define ESW_P2VRES_PRI1_WIDTH                    3u
#define ESW_P2VRES_PRI1(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2VRES_PRI1_SHIFT))&ESW_P2VRES_PRI1_MASK)
#define ESW_P2VRES_PRI2_MASK                     0x1C0u
#define ESW_P2VRES_PRI2_SHIFT                    6u
#define ESW_P2VRES_PRI2_WIDTH                    3u
#define ESW_P2VRES_PRI2(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2VRES_PRI2_SHIFT))&ESW_P2VRES_PRI2_MASK)
#define ESW_P2VRES_PRI3_MASK                     0xE00u
#define ESW_P2VRES_PRI3_SHIFT                    9u
#define ESW_P2VRES_PRI3_WIDTH                    3u
#define ESW_P2VRES_PRI3(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2VRES_PRI3_SHIFT))&ESW_P2VRES_PRI3_MASK)
#define ESW_P2VRES_PRI4_MASK                     0x7000u
#define ESW_P2VRES_PRI4_SHIFT                    12u
#define ESW_P2VRES_PRI4_WIDTH                    3u
#define ESW_P2VRES_PRI4(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2VRES_PRI4_SHIFT))&ESW_P2VRES_PRI4_MASK)
#define ESW_P2VRES_PRI5_MASK                     0x38000u
#define ESW_P2VRES_PRI5_SHIFT                    15u
#define ESW_P2VRES_PRI5_WIDTH                    3u
#define ESW_P2VRES_PRI5(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2VRES_PRI5_SHIFT))&ESW_P2VRES_PRI5_MASK)
#define ESW_P2VRES_PRI6_MASK                     0x1C0000u
#define ESW_P2VRES_PRI6_SHIFT                    18u
#define ESW_P2VRES_PRI6_WIDTH                    3u
#define ESW_P2VRES_PRI6(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2VRES_PRI6_SHIFT))&ESW_P2VRES_PRI6_MASK)
#define ESW_P2VRES_PRI7_MASK                     0xE00000u
#define ESW_P2VRES_PRI7_SHIFT                    21u
#define ESW_P2VRES_PRI7_WIDTH                    3u
#define ESW_P2VRES_PRI7(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2VRES_PRI7_SHIFT))&ESW_P2VRES_PRI7_MASK)
/* IPRES Bit Fields */
#define ESW_IPRES_ADDRESS_MASK                   0xFFu
#define ESW_IPRES_ADDRESS_SHIFT                  0u
#define ESW_IPRES_ADDRESS_WIDTH                  8u
#define ESW_IPRES_ADDRESS(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_IPRES_ADDRESS_SHIFT))&ESW_IPRES_ADDRESS_MASK)
#define ESW_IPRES_IPV4SEL_MASK                   0x100u
#define ESW_IPRES_IPV4SEL_SHIFT                  8u
#define ESW_IPRES_IPV4SEL_WIDTH                  1u
#define ESW_IPRES_IPV4SEL(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_IPRES_IPV4SEL_SHIFT))&ESW_IPRES_IPV4SEL_MASK)
#define ESW_IPRES_PRI0_MASK                      0x600u
#define ESW_IPRES_PRI0_SHIFT                     9u
#define ESW_IPRES_PRI0_WIDTH                     2u
#define ESW_IPRES_PRI0(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_IPRES_PRI0_SHIFT))&ESW_IPRES_PRI0_MASK)
#define ESW_IPRES_PRI1_MASK                      0x1800u
#define ESW_IPRES_PRI1_SHIFT                     11u
#define ESW_IPRES_PRI1_WIDTH                     2u
#define ESW_IPRES_PRI1(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_IPRES_PRI1_SHIFT))&ESW_IPRES_PRI1_MASK)
#define ESW_IPRES_PRI2_MASK                      0x6000u
#define ESW_IPRES_PRI2_SHIFT                     13u
#define ESW_IPRES_PRI2_WIDTH                     2u
#define ESW_IPRES_PRI2(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_IPRES_PRI2_SHIFT))&ESW_IPRES_PRI2_MASK)
#define ESW_IPRES_READ_MASK                      0x80000000u
#define ESW_IPRES_READ_SHIFT                     31u
#define ESW_IPRES_READ_WIDTH                     1u
#define ESW_IPRES_READ(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_IPRES_READ_SHIFT))&ESW_IPRES_READ_MASK)
/* P0RES Bit Fields */
#define ESW_P0RES_VLAN_MASK                      0x1u
#define ESW_P0RES_VLAN_SHIFT                     0u
#define ESW_P0RES_VLAN_WIDTH                     1u
#define ESW_P0RES_VLAN(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_P0RES_VLAN_SHIFT))&ESW_P0RES_VLAN_MASK)
#define ESW_P0RES_IP_MASK                        0x2u
#define ESW_P0RES_IP_SHIFT                       1u
#define ESW_P0RES_IP_WIDTH                       1u
#define ESW_P0RES_IP(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_P0RES_IP_SHIFT))&ESW_P0RES_IP_MASK)
#define ESW_P0RES_MAC_MASK                       0x4u
#define ESW_P0RES_MAC_SHIFT                      2u
#define ESW_P0RES_MAC_WIDTH                      1u
#define ESW_P0RES_MAC(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_P0RES_MAC_SHIFT))&ESW_P0RES_MAC_MASK)
#define ESW_P0RES_DFLT_PRI_MASK                  0x70u
#define ESW_P0RES_DFLT_PRI_SHIFT                 4u
#define ESW_P0RES_DFLT_PRI_WIDTH                 3u
#define ESW_P0RES_DFLT_PRI(x)                    (((uint32_t)(((uint32_t)(x))<<ESW_P0RES_DFLT_PRI_SHIFT))&ESW_P0RES_DFLT_PRI_MASK)
/* P1RES Bit Fields */
#define ESW_P1RES_VLAN_MASK                      0x1u
#define ESW_P1RES_VLAN_SHIFT                     0u
#define ESW_P1RES_VLAN_WIDTH                     1u
#define ESW_P1RES_VLAN(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_P1RES_VLAN_SHIFT))&ESW_P1RES_VLAN_MASK)
#define ESW_P1RES_IP_MASK                        0x2u
#define ESW_P1RES_IP_SHIFT                       1u
#define ESW_P1RES_IP_WIDTH                       1u
#define ESW_P1RES_IP(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_P1RES_IP_SHIFT))&ESW_P1RES_IP_MASK)
#define ESW_P1RES_MAC_MASK                       0x4u
#define ESW_P1RES_MAC_SHIFT                      2u
#define ESW_P1RES_MAC_WIDTH                      1u
#define ESW_P1RES_MAC(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_P1RES_MAC_SHIFT))&ESW_P1RES_MAC_MASK)
#define ESW_P1RES_DFLT_PRI_MASK                  0x70u
#define ESW_P1RES_DFLT_PRI_SHIFT                 4u
#define ESW_P1RES_DFLT_PRI_WIDTH                 3u
#define ESW_P1RES_DFLT_PRI(x)                    (((uint32_t)(((uint32_t)(x))<<ESW_P1RES_DFLT_PRI_SHIFT))&ESW_P1RES_DFLT_PRI_MASK)
/* P2RES Bit Fields */
#define ESW_P2RES_VLAN_MASK                      0x1u
#define ESW_P2RES_VLAN_SHIFT                     0u
#define ESW_P2RES_VLAN_WIDTH                     1u
#define ESW_P2RES_VLAN(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_P2RES_VLAN_SHIFT))&ESW_P2RES_VLAN_MASK)
#define ESW_P2RES_IP_MASK                        0x2u
#define ESW_P2RES_IP_SHIFT                       1u
#define ESW_P2RES_IP_WIDTH                       1u
#define ESW_P2RES_IP(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_P2RES_IP_SHIFT))&ESW_P2RES_IP_MASK)
#define ESW_P2RES_MAC_MASK                       0x4u
#define ESW_P2RES_MAC_SHIFT                      2u
#define ESW_P2RES_MAC_WIDTH                      1u
#define ESW_P2RES_MAC(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_P2RES_MAC_SHIFT))&ESW_P2RES_MAC_MASK)
#define ESW_P2RES_DFLT_PRI_MASK                  0x70u
#define ESW_P2RES_DFLT_PRI_SHIFT                 4u
#define ESW_P2RES_DFLT_PRI_WIDTH                 3u
#define ESW_P2RES_DFLT_PRI(x)                    (((uint32_t)(((uint32_t)(x))<<ESW_P2RES_DFLT_PRI_SHIFT))&ESW_P2RES_DFLT_PRI_MASK)
/* HSEED0 Bit Fields */
#define ESW_HSEED0_HASHSEED_MASK                 0xFFFFFFFFu
#define ESW_HSEED0_HASHSEED_SHIFT                0u
#define ESW_HSEED0_HASHSEED_WIDTH                32u
#define ESW_HSEED0_HASHSEED(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_HSEED0_HASHSEED_SHIFT))&ESW_HSEED0_HASHSEED_MASK)
/* HSEED1 Bit Fields */
#define ESW_HSEED1_HASHSEED_MASK                 0xFFFFu
#define ESW_HSEED1_HASHSEED_SHIFT                0u
#define ESW_HSEED1_HASHSEED_WIDTH                16u
#define ESW_HSEED1_HASHSEED(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_HSEED1_HASHSEED_SHIFT))&ESW_HSEED1_HASHSEED_MASK)
/* P0ID Bit Fields */
#define ESW_P0ID_VLANID_MASK                     0xFFFFu
#define ESW_P0ID_VLANID_SHIFT                    0u
#define ESW_P0ID_VLANID_WIDTH                    16u
#define ESW_P0ID_VLANID(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0ID_VLANID_SHIFT))&ESW_P0ID_VLANID_MASK)
#define ESW_P0ID_VLANOW_MASK                     0xFFFF0000u
#define ESW_P0ID_VLANOW_SHIFT                    16u
#define ESW_P0ID_VLANOW_WIDTH                    16u
#define ESW_P0ID_VLANOW(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0ID_VLANOW_SHIFT))&ESW_P0ID_VLANOW_MASK)
/* P1ID Bit Fields */
#define ESW_P1ID_VLANID_MASK                     0xFFFFu
#define ESW_P1ID_VLANID_SHIFT                    0u
#define ESW_P1ID_VLANID_WIDTH                    16u
#define ESW_P1ID_VLANID(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1ID_VLANID_SHIFT))&ESW_P1ID_VLANID_MASK)
#define ESW_P1ID_VLANOW_MASK                     0xFFFF0000u
#define ESW_P1ID_VLANOW_SHIFT                    16u
#define ESW_P1ID_VLANOW_WIDTH                    16u
#define ESW_P1ID_VLANOW(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1ID_VLANOW_SHIFT))&ESW_P1ID_VLANOW_MASK)
/* P2ID Bit Fields */
#define ESW_P2ID_VLANID_MASK                     0xFFFFu
#define ESW_P2ID_VLANID_SHIFT                    0u
#define ESW_P2ID_VLANID_WIDTH                    16u
#define ESW_P2ID_VLANID(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2ID_VLANID_SHIFT))&ESW_P2ID_VLANID_MASK)
#define ESW_P2ID_VLANOW_MASK                     0xFFFF0000u
#define ESW_P2ID_VLANOW_SHIFT                    16u
#define ESW_P2ID_VLANOW_WIDTH                    16u
#define ESW_P2ID_VLANOW(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2ID_VLANOW_SHIFT))&ESW_P2ID_VLANOW_MASK)
/* VRES Bit Fields */
#define ESW_VRES_P0_MASK                         0x1u
#define ESW_VRES_P0_SHIFT                        0u
#define ESW_VRES_P0_WIDTH                        1u
#define ESW_VRES_P0(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_VRES_P0_SHIFT))&ESW_VRES_P0_MASK)
#define ESW_VRES_P1_MASK                         0x2u
#define ESW_VRES_P1_SHIFT                        1u
#define ESW_VRES_P1_WIDTH                        1u
#define ESW_VRES_P1(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_VRES_P1_SHIFT))&ESW_VRES_P1_MASK)
#define ESW_VRES_P2_MASK                         0x4u
#define ESW_VRES_P2_SHIFT                        2u
#define ESW_VRES_P2_WIDTH                        1u
#define ESW_VRES_P2(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_VRES_P2_SHIFT))&ESW_VRES_P2_MASK)
#define ESW_VRES_VLANID_MASK                     0x7FF8u
#define ESW_VRES_VLANID_SHIFT                    3u
#define ESW_VRES_VLANID_WIDTH                    12u
#define ESW_VRES_VLANID(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_VRES_VLANID_SHIFT))&ESW_VRES_VLANID_MASK)
#define ESW_VRES_TAGM0_MASK                      0x100000u
#define ESW_VRES_TAGM0_SHIFT                     20u
#define ESW_VRES_TAGM0_WIDTH                     1u
#define ESW_VRES_TAGM0(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VRES_TAGM0_SHIFT))&ESW_VRES_TAGM0_MASK)
#define ESW_VRES_TAGM1_MASK                      0x200000u
#define ESW_VRES_TAGM1_SHIFT                     21u
#define ESW_VRES_TAGM1_WIDTH                     1u
#define ESW_VRES_TAGM1(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VRES_TAGM1_SHIFT))&ESW_VRES_TAGM1_MASK)
#define ESW_VRES_TAGM2_MASK                      0x400000u
#define ESW_VRES_TAGM2_SHIFT                     22u
#define ESW_VRES_TAGM2_WIDTH                     1u
#define ESW_VRES_TAGM2(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_VRES_TAGM2_SHIFT))&ESW_VRES_TAGM2_MASK)
/* DISCN Bit Fields */
#define ESW_DISCN_COUNT_MASK                     0xFFFFFFFFu
#define ESW_DISCN_COUNT_SHIFT                    0u
#define ESW_DISCN_COUNT_WIDTH                    32u
#define ESW_DISCN_COUNT(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_DISCN_COUNT_SHIFT))&ESW_DISCN_COUNT_MASK)
/* DISCB Bit Fields */
#define ESW_DISCB_COUNT_MASK                     0xFFFFFFFFu
#define ESW_DISCB_COUNT_SHIFT                    0u
#define ESW_DISCB_COUNT_WIDTH                    32u
#define ESW_DISCB_COUNT(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_DISCB_COUNT_SHIFT))&ESW_DISCB_COUNT_MASK)
/* NDISCN Bit Fields */
#define ESW_NDISCN_COUNT_MASK                    0xFFFFFFFFu
#define ESW_NDISCN_COUNT_SHIFT                   0u
#define ESW_NDISCN_COUNT_WIDTH                   32u
#define ESW_NDISCN_COUNT(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_NDISCN_COUNT_SHIFT))&ESW_NDISCN_COUNT_MASK)
/* NDISCB Bit Fields */
#define ESW_NDISCB_COUNT_MASK                    0xFFFFFFFFu
#define ESW_NDISCB_COUNT_SHIFT                   0u
#define ESW_NDISCB_COUNT_WIDTH                   32u
#define ESW_NDISCB_COUNT(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_NDISCB_COUNT_SHIFT))&ESW_NDISCB_COUNT_MASK)
/* P0OQC Bit Fields */
#define ESW_P0OQC_COUNT_MASK                     0xFFFFFFFFu
#define ESW_P0OQC_COUNT_SHIFT                    0u
#define ESW_P0OQC_COUNT_WIDTH                    32u
#define ESW_P0OQC_COUNT(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P0OQC_COUNT_SHIFT))&ESW_P0OQC_COUNT_MASK)
/* P0MVID Bit Fields */
#define ESW_P0MVID_COUNT_MASK                    0xFFFFFFFFu
#define ESW_P0MVID_COUNT_SHIFT                   0u
#define ESW_P0MVID_COUNT_WIDTH                   32u
#define ESW_P0MVID_COUNT(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_P0MVID_COUNT_SHIFT))&ESW_P0MVID_COUNT_MASK)
/* P0MVTAG Bit Fields */
#define ESW_P0MVTAG_COUNT_MASK                   0xFFFFFFFFu
#define ESW_P0MVTAG_COUNT_SHIFT                  0u
#define ESW_P0MVTAG_COUNT_WIDTH                  32u
#define ESW_P0MVTAG_COUNT(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_P0MVTAG_COUNT_SHIFT))&ESW_P0MVTAG_COUNT_MASK)
/* P0BL Bit Fields */
#define ESW_P0BL_COUNT_MASK                      0xFFFFFFFFu
#define ESW_P0BL_COUNT_SHIFT                     0u
#define ESW_P0BL_COUNT_WIDTH                     32u
#define ESW_P0BL_COUNT(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_P0BL_COUNT_SHIFT))&ESW_P0BL_COUNT_MASK)
/* P1OQC Bit Fields */
#define ESW_P1OQC_COUNT_MASK                     0xFFFFFFFFu
#define ESW_P1OQC_COUNT_SHIFT                    0u
#define ESW_P1OQC_COUNT_WIDTH                    32u
#define ESW_P1OQC_COUNT(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P1OQC_COUNT_SHIFT))&ESW_P1OQC_COUNT_MASK)
/* P1MVID Bit Fields */
#define ESW_P1MVID_COUNT_MASK                    0xFFFFFFFFu
#define ESW_P1MVID_COUNT_SHIFT                   0u
#define ESW_P1MVID_COUNT_WIDTH                   32u
#define ESW_P1MVID_COUNT(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_P1MVID_COUNT_SHIFT))&ESW_P1MVID_COUNT_MASK)
/* P1MVTAG Bit Fields */
#define ESW_P1MVTAG_COUNT_MASK                   0xFFFFFFFFu
#define ESW_P1MVTAG_COUNT_SHIFT                  0u
#define ESW_P1MVTAG_COUNT_WIDTH                  32u
#define ESW_P1MVTAG_COUNT(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_P1MVTAG_COUNT_SHIFT))&ESW_P1MVTAG_COUNT_MASK)
/* P1BL Bit Fields */
#define ESW_P1BL_COUNT_MASK                      0xFFFFFFFFu
#define ESW_P1BL_COUNT_SHIFT                     0u
#define ESW_P1BL_COUNT_WIDTH                     32u
#define ESW_P1BL_COUNT(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_P1BL_COUNT_SHIFT))&ESW_P1BL_COUNT_MASK)
/* P2OQC Bit Fields */
#define ESW_P2OQC_COUNT_MASK                     0xFFFFFFFFu
#define ESW_P2OQC_COUNT_SHIFT                    0u
#define ESW_P2OQC_COUNT_WIDTH                    32u
#define ESW_P2OQC_COUNT(x)                       (((uint32_t)(((uint32_t)(x))<<ESW_P2OQC_COUNT_SHIFT))&ESW_P2OQC_COUNT_MASK)
/* P2MVID Bit Fields */
#define ESW_P2MVID_COUNT_MASK                    0xFFFFFFFFu
#define ESW_P2MVID_COUNT_SHIFT                   0u
#define ESW_P2MVID_COUNT_WIDTH                   32u
#define ESW_P2MVID_COUNT(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_P2MVID_COUNT_SHIFT))&ESW_P2MVID_COUNT_MASK)
/* P2MVTAG Bit Fields */
#define ESW_P2MVTAG_COUNT_MASK                   0xFFFFFFFFu
#define ESW_P2MVTAG_COUNT_SHIFT                  0u
#define ESW_P2MVTAG_COUNT_WIDTH                  32u
#define ESW_P2MVTAG_COUNT(x)                     (((uint32_t)(((uint32_t)(x))<<ESW_P2MVTAG_COUNT_SHIFT))&ESW_P2MVTAG_COUNT_MASK)
/* P2BL Bit Fields */
#define ESW_P2BL_COUNT_MASK                      0xFFFFFFFFu
#define ESW_P2BL_COUNT_SHIFT                     0u
#define ESW_P2BL_COUNT_WIDTH                     32u
#define ESW_P2BL_COUNT(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_P2BL_COUNT_SHIFT))&ESW_P2BL_COUNT_MASK)
/* ISR Bit Fields */
#define ESW_ISR_EBERR_MASK                       0x1u
#define ESW_ISR_EBERR_SHIFT                      0u
#define ESW_ISR_EBERR_WIDTH                      1u
#define ESW_ISR_EBERR(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_ISR_EBERR_SHIFT))&ESW_ISR_EBERR_MASK)
#define ESW_ISR_RXB_MASK                         0x2u
#define ESW_ISR_RXB_SHIFT                        1u
#define ESW_ISR_RXB_WIDTH                        1u
#define ESW_ISR_RXB(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_ISR_RXB_SHIFT))&ESW_ISR_RXB_MASK)
#define ESW_ISR_RXF_MASK                         0x4u
#define ESW_ISR_RXF_SHIFT                        2u
#define ESW_ISR_RXF_WIDTH                        1u
#define ESW_ISR_RXF(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_ISR_RXF_SHIFT))&ESW_ISR_RXF_MASK)
#define ESW_ISR_TXB_MASK                         0x8u
#define ESW_ISR_TXB_SHIFT                        3u
#define ESW_ISR_TXB_WIDTH                        1u
#define ESW_ISR_TXB(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_ISR_TXB_SHIFT))&ESW_ISR_TXB_MASK)
#define ESW_ISR_TXF_MASK                         0x10u
#define ESW_ISR_TXF_SHIFT                        4u
#define ESW_ISR_TXF_WIDTH                        1u
#define ESW_ISR_TXF(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_ISR_TXF_SHIFT))&ESW_ISR_TXF_MASK)
#define ESW_ISR_QM_MASK                          0x20u
#define ESW_ISR_QM_SHIFT                         5u
#define ESW_ISR_QM_WIDTH                         1u
#define ESW_ISR_QM(x)                            (((uint32_t)(((uint32_t)(x))<<ESW_ISR_QM_SHIFT))&ESW_ISR_QM_MASK)
#define ESW_ISR_OD0_MASK                         0x40u
#define ESW_ISR_OD0_SHIFT                        6u
#define ESW_ISR_OD0_WIDTH                        1u
#define ESW_ISR_OD0(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_ISR_OD0_SHIFT))&ESW_ISR_OD0_MASK)
#define ESW_ISR_OD1_MASK                         0x80u
#define ESW_ISR_OD1_SHIFT                        7u
#define ESW_ISR_OD1_WIDTH                        1u
#define ESW_ISR_OD1(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_ISR_OD1_SHIFT))&ESW_ISR_OD1_MASK)
#define ESW_ISR_OD2_MASK                         0x100u
#define ESW_ISR_OD2_SHIFT                        8u
#define ESW_ISR_OD2_WIDTH                        1u
#define ESW_ISR_OD2(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_ISR_OD2_SHIFT))&ESW_ISR_OD2_MASK)
#define ESW_ISR_LRN_MASK                         0x200u
#define ESW_ISR_LRN_SHIFT                        9u
#define ESW_ISR_LRN_WIDTH                        1u
#define ESW_ISR_LRN(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_ISR_LRN_SHIFT))&ESW_ISR_LRN_MASK)
#define ESW_ISR_RXB1_MASK                        0x400u
#define ESW_ISR_RXB1_SHIFT                       10u
#define ESW_ISR_RXB1_WIDTH                       1u
#define ESW_ISR_RXB1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_ISR_RXB1_SHIFT))&ESW_ISR_RXB1_MASK)
#define ESW_ISR_RXF1_MASK                        0x800u
#define ESW_ISR_RXF1_SHIFT                       11u
#define ESW_ISR_RXF1_WIDTH                       1u
#define ESW_ISR_RXF1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_ISR_RXF1_SHIFT))&ESW_ISR_RXF1_MASK)
#define ESW_ISR_TXB1_MASK                        0x1000u
#define ESW_ISR_TXB1_SHIFT                       12u
#define ESW_ISR_TXB1_WIDTH                       1u
#define ESW_ISR_TXB1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_ISR_TXB1_SHIFT))&ESW_ISR_TXB1_MASK)
#define ESW_ISR_TXF1_MASK                        0x2000u
#define ESW_ISR_TXF1_SHIFT                       13u
#define ESW_ISR_TXF1_WIDTH                       1u
#define ESW_ISR_TXF1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_ISR_TXF1_SHIFT))&ESW_ISR_TXF1_MASK)
#define ESW_ISR_RXB2_MASK                        0x4000u
#define ESW_ISR_RXB2_SHIFT                       14u
#define ESW_ISR_RXB2_WIDTH                       1u
#define ESW_ISR_RXB2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_ISR_RXB2_SHIFT))&ESW_ISR_RXB2_MASK)
#define ESW_ISR_RXF2_MASK                        0x8000u
#define ESW_ISR_RXF2_SHIFT                       15u
#define ESW_ISR_RXF2_WIDTH                       1u
#define ESW_ISR_RXF2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_ISR_RXF2_SHIFT))&ESW_ISR_RXF2_MASK)
#define ESW_ISR_TXB2_MASK                        0x10000u
#define ESW_ISR_TXB2_SHIFT                       16u
#define ESW_ISR_TXB2_WIDTH                       1u
#define ESW_ISR_TXB2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_ISR_TXB2_SHIFT))&ESW_ISR_TXB2_MASK)
#define ESW_ISR_TXF2_MASK                        0x20000u
#define ESW_ISR_TXF2_SHIFT                       17u
#define ESW_ISR_TXF2_WIDTH                       1u
#define ESW_ISR_TXF2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_ISR_TXF2_SHIFT))&ESW_ISR_TXF2_MASK)
#define ESW_ISR_RXFLUSH0_MASK                    0x40000u
#define ESW_ISR_RXFLUSH0_SHIFT                   18u
#define ESW_ISR_RXFLUSH0_WIDTH                   1u
#define ESW_ISR_RXFLUSH0(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_ISR_RXFLUSH0_SHIFT))&ESW_ISR_RXFLUSH0_MASK)
#define ESW_ISR_RXFLUSH1_MASK                    0x80000u
#define ESW_ISR_RXFLUSH1_SHIFT                   19u
#define ESW_ISR_RXFLUSH1_WIDTH                   1u
#define ESW_ISR_RXFLUSH1(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_ISR_RXFLUSH1_SHIFT))&ESW_ISR_RXFLUSH1_MASK)
#define ESW_ISR_RXFLUSH2_MASK                    0x100000u
#define ESW_ISR_RXFLUSH2_SHIFT                   20u
#define ESW_ISR_RXFLUSH2_WIDTH                   1u
#define ESW_ISR_RXFLUSH2(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_ISR_RXFLUSH2_SHIFT))&ESW_ISR_RXFLUSH2_MASK)
/* IMR Bit Fields */
#define ESW_IMR_EBERR_MASK                       0x1u
#define ESW_IMR_EBERR_SHIFT                      0u
#define ESW_IMR_EBERR_WIDTH                      1u
#define ESW_IMR_EBERR(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_IMR_EBERR_SHIFT))&ESW_IMR_EBERR_MASK)
#define ESW_IMR_RXB_MASK                         0x2u
#define ESW_IMR_RXB_SHIFT                        1u
#define ESW_IMR_RXB_WIDTH                        1u
#define ESW_IMR_RXB(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_IMR_RXB_SHIFT))&ESW_IMR_RXB_MASK)
#define ESW_IMR_RXF_MASK                         0x4u
#define ESW_IMR_RXF_SHIFT                        2u
#define ESW_IMR_RXF_WIDTH                        1u
#define ESW_IMR_RXF(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_IMR_RXF_SHIFT))&ESW_IMR_RXF_MASK)
#define ESW_IMR_TXB_MASK                         0x8u
#define ESW_IMR_TXB_SHIFT                        3u
#define ESW_IMR_TXB_WIDTH                        1u
#define ESW_IMR_TXB(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_IMR_TXB_SHIFT))&ESW_IMR_TXB_MASK)
#define ESW_IMR_TXF_MASK                         0x10u
#define ESW_IMR_TXF_SHIFT                        4u
#define ESW_IMR_TXF_WIDTH                        1u
#define ESW_IMR_TXF(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_IMR_TXF_SHIFT))&ESW_IMR_TXF_MASK)
#define ESW_IMR_QM_MASK                          0x20u
#define ESW_IMR_QM_SHIFT                         5u
#define ESW_IMR_QM_WIDTH                         1u
#define ESW_IMR_QM(x)                            (((uint32_t)(((uint32_t)(x))<<ESW_IMR_QM_SHIFT))&ESW_IMR_QM_MASK)
#define ESW_IMR_OD0_MASK                         0x40u
#define ESW_IMR_OD0_SHIFT                        6u
#define ESW_IMR_OD0_WIDTH                        1u
#define ESW_IMR_OD0(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_IMR_OD0_SHIFT))&ESW_IMR_OD0_MASK)
#define ESW_IMR_OD1_MASK                         0x80u
#define ESW_IMR_OD1_SHIFT                        7u
#define ESW_IMR_OD1_WIDTH                        1u
#define ESW_IMR_OD1(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_IMR_OD1_SHIFT))&ESW_IMR_OD1_MASK)
#define ESW_IMR_OD2_MASK                         0x100u
#define ESW_IMR_OD2_SHIFT                        8u
#define ESW_IMR_OD2_WIDTH                        1u
#define ESW_IMR_OD2(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_IMR_OD2_SHIFT))&ESW_IMR_OD2_MASK)
#define ESW_IMR_LRN_MASK                         0x200u
#define ESW_IMR_LRN_SHIFT                        9u
#define ESW_IMR_LRN_WIDTH                        1u
#define ESW_IMR_LRN(x)                           (((uint32_t)(((uint32_t)(x))<<ESW_IMR_LRN_SHIFT))&ESW_IMR_LRN_MASK)
#define ESW_IMR_RXB1_MASK                        0x400u
#define ESW_IMR_RXB1_SHIFT                       10u
#define ESW_IMR_RXB1_WIDTH                       1u
#define ESW_IMR_RXB1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IMR_RXB1_SHIFT))&ESW_IMR_RXB1_MASK)
#define ESW_IMR_RXF1_MASK                        0x800u
#define ESW_IMR_RXF1_SHIFT                       11u
#define ESW_IMR_RXF1_WIDTH                       1u
#define ESW_IMR_RXF1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IMR_RXF1_SHIFT))&ESW_IMR_RXF1_MASK)
#define ESW_IMR_TXB1_MASK                        0x1000u
#define ESW_IMR_TXB1_SHIFT                       12u
#define ESW_IMR_TXB1_WIDTH                       1u
#define ESW_IMR_TXB1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IMR_TXB1_SHIFT))&ESW_IMR_TXB1_MASK)
#define ESW_IMR_TXF1_MASK                        0x2000u
#define ESW_IMR_TXF1_SHIFT                       13u
#define ESW_IMR_TXF1_WIDTH                       1u
#define ESW_IMR_TXF1(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IMR_TXF1_SHIFT))&ESW_IMR_TXF1_MASK)
#define ESW_IMR_RXB2_MASK                        0x4000u
#define ESW_IMR_RXB2_SHIFT                       14u
#define ESW_IMR_RXB2_WIDTH                       1u
#define ESW_IMR_RXB2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IMR_RXB2_SHIFT))&ESW_IMR_RXB2_MASK)
#define ESW_IMR_RXF2_MASK                        0x8000u
#define ESW_IMR_RXF2_SHIFT                       15u
#define ESW_IMR_RXF2_WIDTH                       1u
#define ESW_IMR_RXF2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IMR_RXF2_SHIFT))&ESW_IMR_RXF2_MASK)
#define ESW_IMR_TXB2_MASK                        0x10000u
#define ESW_IMR_TXB2_SHIFT                       16u
#define ESW_IMR_TXB2_WIDTH                       1u
#define ESW_IMR_TXB2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IMR_TXB2_SHIFT))&ESW_IMR_TXB2_MASK)
#define ESW_IMR_TXF2_MASK                        0x20000u
#define ESW_IMR_TXF2_SHIFT                       17u
#define ESW_IMR_TXF2_WIDTH                       1u
#define ESW_IMR_TXF2(x)                          (((uint32_t)(((uint32_t)(x))<<ESW_IMR_TXF2_SHIFT))&ESW_IMR_TXF2_MASK)
#define ESW_IMR_RXFLUSH0_MASK                    0x40000u
#define ESW_IMR_RXFLUSH0_SHIFT                   18u
#define ESW_IMR_RXFLUSH0_WIDTH                   1u
#define ESW_IMR_RXFLUSH0(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_IMR_RXFLUSH0_SHIFT))&ESW_IMR_RXFLUSH0_MASK)
#define ESW_IMR_RXFLUSH1_MASK                    0x80000u
#define ESW_IMR_RXFLUSH1_SHIFT                   19u
#define ESW_IMR_RXFLUSH1_WIDTH                   1u
#define ESW_IMR_RXFLUSH1(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_IMR_RXFLUSH1_SHIFT))&ESW_IMR_RXFLUSH1_MASK)
#define ESW_IMR_RXFLUSH2_MASK                    0x100000u
#define ESW_IMR_RXFLUSH2_SHIFT                   20u
#define ESW_IMR_RXFLUSH2_WIDTH                   1u
#define ESW_IMR_RXFLUSH2(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_IMR_RXFLUSH2_SHIFT))&ESW_IMR_RXFLUSH2_MASK)
/* RDSR Bit Fields */
#define ESW_RDSR_ADDRESS_MASK                    0xFFFFFFF8u
#define ESW_RDSR_ADDRESS_SHIFT                   3u
#define ESW_RDSR_ADDRESS_WIDTH                   29u
#define ESW_RDSR_ADDRESS(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_RDSR_ADDRESS_SHIFT))&ESW_RDSR_ADDRESS_MASK)
/* TDSR Bit Fields */
#define ESW_TDSR_ADDRESS_MASK                    0xFFFFFFF8u
#define ESW_TDSR_ADDRESS_SHIFT                   3u
#define ESW_TDSR_ADDRESS_WIDTH                   29u
#define ESW_TDSR_ADDRESS(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_TDSR_ADDRESS_SHIFT))&ESW_TDSR_ADDRESS_MASK)
/* MRBR Bit Fields */
#define ESW_MRBR_SIZE_MASK                       0x3FF0u
#define ESW_MRBR_SIZE_SHIFT                      4u
#define ESW_MRBR_SIZE_WIDTH                      10u
#define ESW_MRBR_SIZE(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_MRBR_SIZE_SHIFT))&ESW_MRBR_SIZE_MASK)
/* RDAR Bit Fields */
#define ESW_RDAR_RDAR_MASK                       0x1000000u
#define ESW_RDAR_RDAR_SHIFT                      24u
#define ESW_RDAR_RDAR_WIDTH                      1u
#define ESW_RDAR_RDAR(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_RDAR_RDAR_SHIFT))&ESW_RDAR_RDAR_MASK)
/* TDAR Bit Fields */
#define ESW_TDAR_TDAR_MASK                       0x1000000u
#define ESW_TDAR_TDAR_SHIFT                      24u
#define ESW_TDAR_TDAR_WIDTH                      1u
#define ESW_TDAR_TDAR(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_TDAR_TDAR_SHIFT))&ESW_TDAR_TDAR_MASK)
/* RDAR1 Bit Fields */
#define ESW_RDAR1_RDAR_MASK                      0x1000000u
#define ESW_RDAR1_RDAR_SHIFT                     24u
#define ESW_RDAR1_RDAR_WIDTH                     1u
#define ESW_RDAR1_RDAR(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_RDAR1_RDAR_SHIFT))&ESW_RDAR1_RDAR_MASK)
/* TDAR1 Bit Fields */
#define ESW_TDAR1_TDAR_MASK                      0x1000000u
#define ESW_TDAR1_TDAR_SHIFT                     24u
#define ESW_TDAR1_TDAR_WIDTH                     1u
#define ESW_TDAR1_TDAR(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_TDAR1_TDAR_SHIFT))&ESW_TDAR1_TDAR_MASK)
/* RDAR2 Bit Fields */
#define ESW_RDAR2_RDAR_MASK                      0x1000000u
#define ESW_RDAR2_RDAR_SHIFT                     24u
#define ESW_RDAR2_RDAR_WIDTH                     1u
#define ESW_RDAR2_RDAR(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_RDAR2_RDAR_SHIFT))&ESW_RDAR2_RDAR_MASK)
/* TDAR2 Bit Fields */
#define ESW_TDAR2_TDAR_MASK                      0x1000000u
#define ESW_TDAR2_TDAR_SHIFT                     24u
#define ESW_TDAR2_TDAR_WIDTH                     1u
#define ESW_TDAR2_TDAR(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_TDAR2_TDAR_SHIFT))&ESW_TDAR2_TDAR_MASK)
/* TXIC Bit Fields */
#define ESW_TXIC_ICTT_MASK                       0xFFFFu
#define ESW_TXIC_ICTT_SHIFT                      0u
#define ESW_TXIC_ICTT_WIDTH                      16u
#define ESW_TXIC_ICTT(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_TXIC_ICTT_SHIFT))&ESW_TXIC_ICTT_MASK)
#define ESW_TXIC_ICFT_MASK                       0xFF00000u
#define ESW_TXIC_ICFT_SHIFT                      20u
#define ESW_TXIC_ICFT_WIDTH                      8u
#define ESW_TXIC_ICFT(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_TXIC_ICFT_SHIFT))&ESW_TXIC_ICFT_MASK)
#define ESW_TXIC_ICCS_MASK                       0x40000000u
#define ESW_TXIC_ICCS_SHIFT                      30u
#define ESW_TXIC_ICCS_WIDTH                      1u
#define ESW_TXIC_ICCS(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_TXIC_ICCS_SHIFT))&ESW_TXIC_ICCS_MASK)
#define ESW_TXIC_ICEN_MASK                       0x80000000u
#define ESW_TXIC_ICEN_SHIFT                      31u
#define ESW_TXIC_ICEN_WIDTH                      1u
#define ESW_TXIC_ICEN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_TXIC_ICEN_SHIFT))&ESW_TXIC_ICEN_MASK)
/* RXIC Bit Fields */
#define ESW_RXIC_ICTT_MASK                       0xFFFFu
#define ESW_RXIC_ICTT_SHIFT                      0u
#define ESW_RXIC_ICTT_WIDTH                      16u
#define ESW_RXIC_ICTT(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_RXIC_ICTT_SHIFT))&ESW_RXIC_ICTT_MASK)
#define ESW_RXIC_ICFT_MASK                       0xFF00000u
#define ESW_RXIC_ICFT_SHIFT                      20u
#define ESW_RXIC_ICFT_WIDTH                      8u
#define ESW_RXIC_ICFT(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_RXIC_ICFT_SHIFT))&ESW_RXIC_ICFT_MASK)
#define ESW_RXIC_ICCS_MASK                       0x40000000u
#define ESW_RXIC_ICCS_SHIFT                      30u
#define ESW_RXIC_ICCS_WIDTH                      1u
#define ESW_RXIC_ICCS(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_RXIC_ICCS_SHIFT))&ESW_RXIC_ICCS_MASK)
#define ESW_RXIC_ICEN_MASK                       0x80000000u
#define ESW_RXIC_ICEN_SHIFT                      31u
#define ESW_RXIC_ICEN_WIDTH                      1u
#define ESW_RXIC_ICEN(x)                         (((uint32_t)(((uint32_t)(x))<<ESW_RXIC_ICEN_SHIFT))&ESW_RXIC_ICEN_MASK)
/* SEL_CLASS Bit Fields */
#define ESW_SEL_CLASS_SEL_CLASS_MASK             0x3u
#define ESW_SEL_CLASS_SEL_CLASS_SHIFT            0u
#define ESW_SEL_CLASS_SEL_CLASS_WIDTH            2u
#define ESW_SEL_CLASS_SEL_CLASS(x)               (((uint32_t)(((uint32_t)(x))<<ESW_SEL_CLASS_SEL_CLASS_SHIFT))&ESW_SEL_CLASS_SEL_CLASS_MASK)
/* LREC0 Bit Fields */
#define ESW_LREC0_MAC_ADDR0_MASK                 0xFFFFFFFFu
#define ESW_LREC0_MAC_ADDR0_SHIFT                0u
#define ESW_LREC0_MAC_ADDR0_WIDTH                32u
#define ESW_LREC0_MAC_ADDR0(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_LREC0_MAC_ADDR0_SHIFT))&ESW_LREC0_MAC_ADDR0_MASK)
/* LREC1 Bit Fields */
#define ESW_LREC1_MAC_ADDR1_MASK                 0xFFFFu
#define ESW_LREC1_MAC_ADDR1_SHIFT                0u
#define ESW_LREC1_MAC_ADDR1_WIDTH                16u
#define ESW_LREC1_MAC_ADDR1(x)                   (((uint32_t)(((uint32_t)(x))<<ESW_LREC1_MAC_ADDR1_SHIFT))&ESW_LREC1_MAC_ADDR1_MASK)
#define ESW_LREC1_HASH_MASK                      0xFF0000u
#define ESW_LREC1_HASH_SHIFT                     16u
#define ESW_LREC1_HASH_WIDTH                     8u
#define ESW_LREC1_HASH(x)                        (((uint32_t)(((uint32_t)(x))<<ESW_LREC1_HASH_SHIFT))&ESW_LREC1_HASH_MASK)
#define ESW_LREC1_SWPORT_MASK                    0x3000000u
#define ESW_LREC1_SWPORT_SHIFT                   24u
#define ESW_LREC1_SWPORT_WIDTH                   2u
#define ESW_LREC1_SWPORT(x)                      (((uint32_t)(((uint32_t)(x))<<ESW_LREC1_SWPORT_SHIFT))&ESW_LREC1_SWPORT_MASK)
/* LSR Bit Fields */
#define ESW_LSR_DA_MASK                          0x1u
#define ESW_LSR_DA_SHIFT                         0u
#define ESW_LSR_DA_WIDTH                         1u
#define ESW_LSR_DA(x)                            (((uint32_t)(((uint32_t)(x))<<ESW_LSR_DA_SHIFT))&ESW_LSR_DA_MASK)

/*!
 * @}
 */ /* end of group ESW_Register_Masks */


/*!
 * @}
 */ /* end of group ESW_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- FCCU Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FCCU_Peripheral_Access_Layer FCCU Peripheral Access Layer
 * @{
 */


/** FCCU - Size of Registers Arrays */
#define FCCU_NCF_CFG_COUNT                       3u
#define FCCU_NCFS_CFG_COUNT                      5u
#define FCCU_NCF_S_COUNT                         3u
#define FCCU_NCF_E_COUNT                         3u
#define FCCU_NCF_TOE_COUNT                       3u
#define FCCU_IRQ_ALARM_EN_COUNT                  3u
#define FCCU_NMI_EN_COUNT                        3u
#define FCCU_EOUT_SIG_EN_COUNT                   3u

/** FCCU - Register Layout Typedef */
typedef struct {
  __IO uint32_t CTRL;                              /**< Control, offset: 0x0 */
  __O  uint32_t CTRLK;                             /**< Control Key, offset: 0x4 */
  __IO uint32_t CFG;                               /**< Configuration, offset: 0x8 */
       uint8_t RESERVED_0[16];
  __IO uint32_t NCF_CFG[FCCU_NCF_CFG_COUNT];       /**< Noncritical Fault Configuration, array offset: 0x1C, array step: 0x4 */
       uint8_t RESERVED_1[36];
  __IO uint32_t NCFS_CFG[FCCU_NCFS_CFG_COUNT];     /**< Noncritical Fault State Configuration, array offset: 0x4C, array step: 0x4 */
       uint8_t RESERVED_2[32];
  __IO uint32_t NCF_S[FCCU_NCF_S_COUNT];           /**< Noncritical Fault Status, array offset: 0x80, array step: 0x4 */
       uint8_t RESERVED_3[4];
  __O  uint32_t NCFK;                              /**< Noncritical Fault Key, offset: 0x90 */
  __IO uint32_t NCF_E[FCCU_NCF_E_COUNT];           /**< Noncritical Fault Enable, array offset: 0x94, array step: 0x4 */
       uint8_t RESERVED_4[4];
  __IO uint32_t NCF_TOE[FCCU_NCF_TOE_COUNT];       /**< Noncritical Fault Timeout Enable, array offset: 0xA4, array step: 0x4 */
       uint8_t RESERVED_5[4];
  __IO uint32_t NCF_TO;                            /**< Noncritical Fault Timeout, offset: 0xB4 */
  __IO uint32_t CFG_TO;                            /**< Configuration-State Timer Interval, offset: 0xB8 */
  __IO uint32_t EINOUT;                            /**< IO Control, offset: 0xBC */
  __I  uint32_t STAT;                              /**< Status, offset: 0xC0 */
  __I  uint32_t N2AF_STATUS;                       /**< NA Freeze Status, offset: 0xC4 */
  __I  uint32_t A2FF_STATUS;                       /**< AF Freeze Status, offset: 0xC8 */
  __I  uint32_t N2FF_STATUS;                       /**< NF Freeze Status, offset: 0xCC */
  __I  uint32_t F2A_STATUS;                        /**< FA Freeze Status, offset: 0xD0 */
       uint8_t RESERVED_6[8];
  __IO uint32_t NCFF;                              /**< Noncritical Fault Fake, offset: 0xDC */
  __IO uint32_t IRQ_STAT;                          /**< IRQ Status, offset: 0xE0 */
  __IO uint32_t IRQ_EN;                            /**< IRQ Enable, offset: 0xE4 */
  __I  uint32_t XTMR;                              /**< XTMR, offset: 0xE8 */
  __I  uint32_t MCS;                               /**< Mode Controller Status, offset: 0xEC */
  __IO uint32_t TRANS_LOCK;                        /**< Transient Configuration Lock, offset: 0xF0 */
  __IO uint32_t PERMNT_LOCK;                       /**< Permanent Configuration Lock, offset: 0xF4 */
  __IO uint32_t DELTA_T;                           /**< Delta T, offset: 0xF8 */
  __IO uint32_t IRQ_ALARM_EN[FCCU_IRQ_ALARM_EN_COUNT]; /**< IRQ Alarm Enable, array offset: 0xFC, array step: 0x4 */
       uint8_t RESERVED_7[4];
  __IO uint32_t NMI_EN[FCCU_NMI_EN_COUNT];         /**< NMI Enable, array offset: 0x10C, array step: 0x4 */
       uint8_t RESERVED_8[4];
  __IO uint32_t EOUT_SIG_EN[FCCU_EOUT_SIG_EN_COUNT]; /**< Bistable Fault-Output (EOUT) Mode Signaling Enable, array offset: 0x11C, array step: 0x4 */
} FCCU_Type, *FCCU_MemMapPtr;

 /** Number of instances of the FCCU module. */
#define FCCU_INSTANCE_COUNT                      (1u)


/* FCCU - Peripheral instance base addresses */
/** Peripheral FCCU base address */
#define FCCU_BASE                                (0xFBF58000u)
/** Peripheral FCCU base pointer */
#define FCCU                                     ((FCCU_Type *)FCCU_BASE)
/** Array initializer of FCCU peripheral base addresses */
#define FCCU_BASE_ADDRS                          { FCCU_BASE }
/** Array initializer of FCCU peripheral base pointers */
#define FCCU_BASE_PTRS                           { FCCU }

/* ----------------------------------------------------------------------------
   -- FCCU Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FCCU_Register_Masks FCCU Register Masks
 * @{
 */

/* CTRL Bit Fields */
#define FCCU_CTRL_OPR_MASK                       0x1Fu
#define FCCU_CTRL_OPR_SHIFT                      0u
#define FCCU_CTRL_OPR_WIDTH                      5u
#define FCCU_CTRL_OPR(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_OPR_SHIFT))&FCCU_CTRL_OPR_MASK)
#define FCCU_CTRL_OPS_MASK                       0xC0u
#define FCCU_CTRL_OPS_SHIFT                      6u
#define FCCU_CTRL_OPS_WIDTH                      2u
#define FCCU_CTRL_OPS(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_OPS_SHIFT))&FCCU_CTRL_OPS_MASK)
#define FCCU_CTRL_DEBUG_MASK                     0x200u
#define FCCU_CTRL_DEBUG_SHIFT                    9u
#define FCCU_CTRL_DEBUG_WIDTH                    1u
#define FCCU_CTRL_DEBUG(x)                       (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_DEBUG_SHIFT))&FCCU_CTRL_DEBUG_MASK)
#define FCCU_CTRL_FILTER_WIDTH_MASK              0x60000000u
#define FCCU_CTRL_FILTER_WIDTH_SHIFT             29u
#define FCCU_CTRL_FILTER_WIDTH_WIDTH             2u
#define FCCU_CTRL_FILTER_WIDTH(x)                (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_FILTER_WIDTH_SHIFT))&FCCU_CTRL_FILTER_WIDTH_MASK)
#define FCCU_CTRL_FILTER_BYPASS_MASK             0x80000000u
#define FCCU_CTRL_FILTER_BYPASS_SHIFT            31u
#define FCCU_CTRL_FILTER_BYPASS_WIDTH            1u
#define FCCU_CTRL_FILTER_BYPASS(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_CTRL_FILTER_BYPASS_SHIFT))&FCCU_CTRL_FILTER_BYPASS_MASK)
/* CTRLK Bit Fields */
#define FCCU_CTRLK_CTRLK_MASK                    0xFFFFFFFFu
#define FCCU_CTRLK_CTRLK_SHIFT                   0u
#define FCCU_CTRLK_CTRLK_WIDTH                   32u
#define FCCU_CTRLK_CTRLK(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_CTRLK_CTRLK_SHIFT))&FCCU_CTRLK_CTRLK_MASK)
/* CFG Bit Fields */
#define FCCU_CFG_FOP_MASK                        0x3Fu
#define FCCU_CFG_FOP_SHIFT                       0u
#define FCCU_CFG_FOP_WIDTH                       6u
#define FCCU_CFG_FOP(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FOP_SHIFT))&FCCU_CFG_FOP_MASK)
#define FCCU_CFG_FOM_MASK                        0x1C0u
#define FCCU_CFG_FOM_SHIFT                       6u
#define FCCU_CFG_FOM_WIDTH                       3u
#define FCCU_CFG_FOM(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FOM_SHIFT))&FCCU_CFG_FOM_MASK)
#define FCCU_CFG_PS_MASK                         0x200u
#define FCCU_CFG_PS_SHIFT                        9u
#define FCCU_CFG_PS_WIDTH                        1u
#define FCCU_CFG_PS(x)                           (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_PS_SHIFT))&FCCU_CFG_PS_MASK)
#define FCCU_CFG_SM_MASK                         0x400u
#define FCCU_CFG_SM_SHIFT                        10u
#define FCCU_CFG_SM_WIDTH                        1u
#define FCCU_CFG_SM(x)                           (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_SM_SHIFT))&FCCU_CFG_SM_MASK)
#define FCCU_CFG_OD_MASK                         0x1000u
#define FCCU_CFG_OD_SHIFT                        12u
#define FCCU_CFG_OD_WIDTH                        1u
#define FCCU_CFG_OD(x)                           (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_OD_SHIFT))&FCCU_CFG_OD_MASK)
#define FCCU_CFG_FOME_MASK                       0x6000u
#define FCCU_CFG_FOME_SHIFT                      13u
#define FCCU_CFG_FOME_WIDTH                      2u
#define FCCU_CFG_FOME(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FOME_SHIFT))&FCCU_CFG_FOME_MASK)
#define FCCU_CFG_FOPE_MASK                       0x8000u
#define FCCU_CFG_FOPE_SHIFT                      15u
#define FCCU_CFG_FOPE_WIDTH                      1u
#define FCCU_CFG_FOPE(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FOPE_SHIFT))&FCCU_CFG_FOPE_MASK)
#define FCCU_CFG_FCCU_SET_CLEAR_MASK             0xC00000u
#define FCCU_CFG_FCCU_SET_CLEAR_SHIFT            22u
#define FCCU_CFG_FCCU_SET_CLEAR_WIDTH            2u
#define FCCU_CFG_FCCU_SET_CLEAR(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FCCU_SET_CLEAR_SHIFT))&FCCU_CFG_FCCU_SET_CLEAR_MASK)
#define FCCU_CFG_FCCU_SET_AFTER_RESET_MASK       0x1000000u
#define FCCU_CFG_FCCU_SET_AFTER_RESET_SHIFT      24u
#define FCCU_CFG_FCCU_SET_AFTER_RESET_WIDTH      1u
#define FCCU_CFG_FCCU_SET_AFTER_RESET(x)         (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_FCCU_SET_AFTER_RESET_SHIFT))&FCCU_CFG_FCCU_SET_AFTER_RESET_MASK)
#define FCCU_CFG_TGL_MASK                        0xFE000000u
#define FCCU_CFG_TGL_SHIFT                       25u
#define FCCU_CFG_TGL_WIDTH                       7u
#define FCCU_CFG_TGL(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_TGL_SHIFT))&FCCU_CFG_TGL_MASK)
/* NCF_CFG Bit Fields */
#define FCCU_NCF_CFG_NCFCx_MASK                  0xFFFFFFFFu
#define FCCU_NCF_CFG_NCFCx_SHIFT                 0u
#define FCCU_NCF_CFG_NCFCx_WIDTH                 32u
#define FCCU_NCF_CFG_NCFCx(x)                    (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_CFG_NCFCx_SHIFT))&FCCU_NCF_CFG_NCFCx_MASK)
/* NCFS_CFG Bit Fields */
#define FCCU_NCFS_CFG_NCFSCx_MASK                0xFFFFFFFFu
#define FCCU_NCFS_CFG_NCFSCx_SHIFT               0u
#define FCCU_NCFS_CFG_NCFSCx_WIDTH               32u
#define FCCU_NCFS_CFG_NCFSCx(x)                  (((uint32_t)(((uint32_t)(x))<<FCCU_NCFS_CFG_NCFSCx_SHIFT))&FCCU_NCFS_CFG_NCFSCx_MASK)
/* NCF_S Bit Fields */
#define FCCU_NCF_S_NCFSx_MASK                    0xFFFFFFFFu
#define FCCU_NCF_S_NCFSx_SHIFT                   0u
#define FCCU_NCF_S_NCFSx_WIDTH                   32u
#define FCCU_NCF_S_NCFSx(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_S_NCFSx_SHIFT))&FCCU_NCF_S_NCFSx_MASK)
/* NCFK Bit Fields */
#define FCCU_NCFK_NCFK_MASK                      0xFFFFFFFFu
#define FCCU_NCFK_NCFK_SHIFT                     0u
#define FCCU_NCFK_NCFK_WIDTH                     32u
#define FCCU_NCFK_NCFK(x)                        (((uint32_t)(((uint32_t)(x))<<FCCU_NCFK_NCFK_SHIFT))&FCCU_NCFK_NCFK_MASK)
/* NCF_E Bit Fields */
#define FCCU_NCF_E_NCFEx_MASK                    0xFFFFFFFFu
#define FCCU_NCF_E_NCFEx_SHIFT                   0u
#define FCCU_NCF_E_NCFEx_WIDTH                   32u
#define FCCU_NCF_E_NCFEx(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_E_NCFEx_SHIFT))&FCCU_NCF_E_NCFEx_MASK)
/* NCF_TOE Bit Fields */
#define FCCU_NCF_TOE_NCFTOEx_MASK                0xFFFFFFFFu
#define FCCU_NCF_TOE_NCFTOEx_SHIFT               0u
#define FCCU_NCF_TOE_NCFTOEx_WIDTH               32u
#define FCCU_NCF_TOE_NCFTOEx(x)                  (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_TOE_NCFTOEx_SHIFT))&FCCU_NCF_TOE_NCFTOEx_MASK)
/* NCF_TO Bit Fields */
#define FCCU_NCF_TO_TO_MASK                      0xFFFFFFFFu
#define FCCU_NCF_TO_TO_SHIFT                     0u
#define FCCU_NCF_TO_TO_WIDTH                     32u
#define FCCU_NCF_TO_TO(x)                        (((uint32_t)(((uint32_t)(x))<<FCCU_NCF_TO_TO_SHIFT))&FCCU_NCF_TO_TO_MASK)
/* CFG_TO Bit Fields */
#define FCCU_CFG_TO_TO_MASK                      0x7u
#define FCCU_CFG_TO_TO_SHIFT                     0u
#define FCCU_CFG_TO_TO_WIDTH                     3u
#define FCCU_CFG_TO_TO(x)                        (((uint32_t)(((uint32_t)(x))<<FCCU_CFG_TO_TO_SHIFT))&FCCU_CFG_TO_TO_MASK)
/* EINOUT Bit Fields */
#define FCCU_EINOUT_EOUTX_MASK                   0x3u
#define FCCU_EINOUT_EOUTX_SHIFT                  0u
#define FCCU_EINOUT_EOUTX_WIDTH                  2u
#define FCCU_EINOUT_EOUTX(x)                     (((uint32_t)(((uint32_t)(x))<<FCCU_EINOUT_EOUTX_SHIFT))&FCCU_EINOUT_EOUTX_MASK)
#define FCCU_EINOUT_EIN0_MASK                    0x10u
#define FCCU_EINOUT_EIN0_SHIFT                   4u
#define FCCU_EINOUT_EIN0_WIDTH                   1u
#define FCCU_EINOUT_EIN0(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_EINOUT_EIN0_SHIFT))&FCCU_EINOUT_EIN0_MASK)
#define FCCU_EINOUT_EIN1_MASK                    0x20u
#define FCCU_EINOUT_EIN1_SHIFT                   5u
#define FCCU_EINOUT_EIN1_WIDTH                   1u
#define FCCU_EINOUT_EIN1(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_EINOUT_EIN1_SHIFT))&FCCU_EINOUT_EIN1_MASK)
/* STAT Bit Fields */
#define FCCU_STAT_STATUS_MASK                    0x7u
#define FCCU_STAT_STATUS_SHIFT                   0u
#define FCCU_STAT_STATUS_WIDTH                   3u
#define FCCU_STAT_STATUS(x)                      (((uint32_t)(((uint32_t)(x))<<FCCU_STAT_STATUS_SHIFT))&FCCU_STAT_STATUS_MASK)
#define FCCU_STAT_ESTAT_MASK                     0x8u
#define FCCU_STAT_ESTAT_SHIFT                    3u
#define FCCU_STAT_ESTAT_WIDTH                    1u
#define FCCU_STAT_ESTAT(x)                       (((uint32_t)(((uint32_t)(x))<<FCCU_STAT_ESTAT_SHIFT))&FCCU_STAT_ESTAT_MASK)
#define FCCU_STAT_PhysicErrorPin_MASK            0x30u
#define FCCU_STAT_PhysicErrorPin_SHIFT           4u
#define FCCU_STAT_PhysicErrorPin_WIDTH           2u
#define FCCU_STAT_PhysicErrorPin(x)              (((uint32_t)(((uint32_t)(x))<<FCCU_STAT_PhysicErrorPin_SHIFT))&FCCU_STAT_PhysicErrorPin_MASK)
/* N2AF_STATUS Bit Fields */
#define FCCU_N2AF_STATUS_NAFS_MASK               0xFFu
#define FCCU_N2AF_STATUS_NAFS_SHIFT              0u
#define FCCU_N2AF_STATUS_NAFS_WIDTH              8u
#define FCCU_N2AF_STATUS_NAFS(x)                 (((uint32_t)(((uint32_t)(x))<<FCCU_N2AF_STATUS_NAFS_SHIFT))&FCCU_N2AF_STATUS_NAFS_MASK)
/* A2FF_STATUS Bit Fields */
#define FCCU_A2FF_STATUS_AFFS_MASK               0xFFu
#define FCCU_A2FF_STATUS_AFFS_SHIFT              0u
#define FCCU_A2FF_STATUS_AFFS_WIDTH              8u
#define FCCU_A2FF_STATUS_AFFS(x)                 (((uint32_t)(((uint32_t)(x))<<FCCU_A2FF_STATUS_AFFS_SHIFT))&FCCU_A2FF_STATUS_AFFS_MASK)
#define FCCU_A2FF_STATUS_AF_SRC_MASK             0x300u
#define FCCU_A2FF_STATUS_AF_SRC_SHIFT            8u
#define FCCU_A2FF_STATUS_AF_SRC_WIDTH            2u
#define FCCU_A2FF_STATUS_AF_SRC(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_A2FF_STATUS_AF_SRC_SHIFT))&FCCU_A2FF_STATUS_AF_SRC_MASK)
/* N2FF_STATUS Bit Fields */
#define FCCU_N2FF_STATUS_NFFS_MASK               0xFFu
#define FCCU_N2FF_STATUS_NFFS_SHIFT              0u
#define FCCU_N2FF_STATUS_NFFS_WIDTH              8u
#define FCCU_N2FF_STATUS_NFFS(x)                 (((uint32_t)(((uint32_t)(x))<<FCCU_N2FF_STATUS_NFFS_SHIFT))&FCCU_N2FF_STATUS_NFFS_MASK)
#define FCCU_N2FF_STATUS_NF_SRC_MASK             0x300u
#define FCCU_N2FF_STATUS_NF_SRC_SHIFT            8u
#define FCCU_N2FF_STATUS_NF_SRC_WIDTH            2u
#define FCCU_N2FF_STATUS_NF_SRC(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_N2FF_STATUS_NF_SRC_SHIFT))&FCCU_N2FF_STATUS_NF_SRC_MASK)
/* F2A_STATUS Bit Fields */
#define FCCU_F2A_STATUS_FAFS_MASK                0x1FFu
#define FCCU_F2A_STATUS_FAFS_SHIFT               0u
#define FCCU_F2A_STATUS_FAFS_WIDTH               9u
#define FCCU_F2A_STATUS_FAFS(x)                  (((uint32_t)(((uint32_t)(x))<<FCCU_F2A_STATUS_FAFS_SHIFT))&FCCU_F2A_STATUS_FAFS_MASK)
/* NCFF Bit Fields */
#define FCCU_NCFF_FNCFC_MASK                     0x7Fu
#define FCCU_NCFF_FNCFC_SHIFT                    0u
#define FCCU_NCFF_FNCFC_WIDTH                    7u
#define FCCU_NCFF_FNCFC(x)                       (((uint32_t)(((uint32_t)(x))<<FCCU_NCFF_FNCFC_SHIFT))&FCCU_NCFF_FNCFC_MASK)
/* IRQ_STAT Bit Fields */
#define FCCU_IRQ_STAT_CFG_TO_STAT_MASK           0x1u
#define FCCU_IRQ_STAT_CFG_TO_STAT_SHIFT          0u
#define FCCU_IRQ_STAT_CFG_TO_STAT_WIDTH          1u
#define FCCU_IRQ_STAT_CFG_TO_STAT(x)             (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_STAT_CFG_TO_STAT_SHIFT))&FCCU_IRQ_STAT_CFG_TO_STAT_MASK)
#define FCCU_IRQ_STAT_ALRM_STAT_MASK             0x2u
#define FCCU_IRQ_STAT_ALRM_STAT_SHIFT            1u
#define FCCU_IRQ_STAT_ALRM_STAT_WIDTH            1u
#define FCCU_IRQ_STAT_ALRM_STAT(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_STAT_ALRM_STAT_SHIFT))&FCCU_IRQ_STAT_ALRM_STAT_MASK)
#define FCCU_IRQ_STAT_NMI_STAT_MASK              0x4u
#define FCCU_IRQ_STAT_NMI_STAT_SHIFT             2u
#define FCCU_IRQ_STAT_NMI_STAT_WIDTH             1u
#define FCCU_IRQ_STAT_NMI_STAT(x)                (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_STAT_NMI_STAT_SHIFT))&FCCU_IRQ_STAT_NMI_STAT_MASK)
#define FCCU_IRQ_STAT_IRQ_EOUT_MASK              0x8u
#define FCCU_IRQ_STAT_IRQ_EOUT_SHIFT             3u
#define FCCU_IRQ_STAT_IRQ_EOUT_WIDTH             1u
#define FCCU_IRQ_STAT_IRQ_EOUT(x)                (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_STAT_IRQ_EOUT_SHIFT))&FCCU_IRQ_STAT_IRQ_EOUT_MASK)
#define FCCU_IRQ_STAT_DMA_REQ_MASK               0x10u
#define FCCU_IRQ_STAT_DMA_REQ_SHIFT              4u
#define FCCU_IRQ_STAT_DMA_REQ_WIDTH              1u
#define FCCU_IRQ_STAT_DMA_REQ(x)                 (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_STAT_DMA_REQ_SHIFT))&FCCU_IRQ_STAT_DMA_REQ_MASK)
/* IRQ_EN Bit Fields */
#define FCCU_IRQ_EN_CFG_TO_IEN_MASK              0x1u
#define FCCU_IRQ_EN_CFG_TO_IEN_SHIFT             0u
#define FCCU_IRQ_EN_CFG_TO_IEN_WIDTH             1u
#define FCCU_IRQ_EN_CFG_TO_IEN(x)                (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_EN_CFG_TO_IEN_SHIFT))&FCCU_IRQ_EN_CFG_TO_IEN_MASK)
#define FCCU_IRQ_EN_IRQ_EOUT_EN_MASK             0x8u
#define FCCU_IRQ_EN_IRQ_EOUT_EN_SHIFT            3u
#define FCCU_IRQ_EN_IRQ_EOUT_EN_WIDTH            1u
#define FCCU_IRQ_EN_IRQ_EOUT_EN(x)               (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_EN_IRQ_EOUT_EN_SHIFT))&FCCU_IRQ_EN_IRQ_EOUT_EN_MASK)
/* XTMR Bit Fields */
#define FCCU_XTMR_XTMR_MASK                      0xFFFFFFFFu
#define FCCU_XTMR_XTMR_SHIFT                     0u
#define FCCU_XTMR_XTMR_WIDTH                     32u
#define FCCU_XTMR_XTMR(x)                        (((uint32_t)(((uint32_t)(x))<<FCCU_XTMR_XTMR_SHIFT))&FCCU_XTMR_XTMR_MASK)
/* MCS Bit Fields */
#define FCCU_MCS_MCS0_MASK                       0xFu
#define FCCU_MCS_MCS0_SHIFT                      0u
#define FCCU_MCS_MCS0_WIDTH                      4u
#define FCCU_MCS_MCS0(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_MCS0_SHIFT))&FCCU_MCS_MCS0_MASK)
#define FCCU_MCS_FS0_MASK                        0x40u
#define FCCU_MCS_FS0_SHIFT                       6u
#define FCCU_MCS_FS0_WIDTH                       1u
#define FCCU_MCS_FS0(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_FS0_SHIFT))&FCCU_MCS_FS0_MASK)
#define FCCU_MCS_VL0_MASK                        0x80u
#define FCCU_MCS_VL0_SHIFT                       7u
#define FCCU_MCS_VL0_WIDTH                       1u
#define FCCU_MCS_VL0(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_VL0_SHIFT))&FCCU_MCS_VL0_MASK)
#define FCCU_MCS_MCS1_MASK                       0xF00u
#define FCCU_MCS_MCS1_SHIFT                      8u
#define FCCU_MCS_MCS1_WIDTH                      4u
#define FCCU_MCS_MCS1(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_MCS1_SHIFT))&FCCU_MCS_MCS1_MASK)
#define FCCU_MCS_FS1_MASK                        0x4000u
#define FCCU_MCS_FS1_SHIFT                       14u
#define FCCU_MCS_FS1_WIDTH                       1u
#define FCCU_MCS_FS1(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_FS1_SHIFT))&FCCU_MCS_FS1_MASK)
#define FCCU_MCS_VL1_MASK                        0x8000u
#define FCCU_MCS_VL1_SHIFT                       15u
#define FCCU_MCS_VL1_WIDTH                       1u
#define FCCU_MCS_VL1(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_VL1_SHIFT))&FCCU_MCS_VL1_MASK)
#define FCCU_MCS_MCS2_MASK                       0xF0000u
#define FCCU_MCS_MCS2_SHIFT                      16u
#define FCCU_MCS_MCS2_WIDTH                      4u
#define FCCU_MCS_MCS2(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_MCS2_SHIFT))&FCCU_MCS_MCS2_MASK)
#define FCCU_MCS_FS2_MASK                        0x400000u
#define FCCU_MCS_FS2_SHIFT                       22u
#define FCCU_MCS_FS2_WIDTH                       1u
#define FCCU_MCS_FS2(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_FS2_SHIFT))&FCCU_MCS_FS2_MASK)
#define FCCU_MCS_VL2_MASK                        0x800000u
#define FCCU_MCS_VL2_SHIFT                       23u
#define FCCU_MCS_VL2_WIDTH                       1u
#define FCCU_MCS_VL2(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_VL2_SHIFT))&FCCU_MCS_VL2_MASK)
#define FCCU_MCS_MCS3_MASK                       0xF000000u
#define FCCU_MCS_MCS3_SHIFT                      24u
#define FCCU_MCS_MCS3_WIDTH                      4u
#define FCCU_MCS_MCS3(x)                         (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_MCS3_SHIFT))&FCCU_MCS_MCS3_MASK)
#define FCCU_MCS_FS3_MASK                        0x40000000u
#define FCCU_MCS_FS3_SHIFT                       30u
#define FCCU_MCS_FS3_WIDTH                       1u
#define FCCU_MCS_FS3(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_FS3_SHIFT))&FCCU_MCS_FS3_MASK)
#define FCCU_MCS_VL3_MASK                        0x80000000u
#define FCCU_MCS_VL3_SHIFT                       31u
#define FCCU_MCS_VL3_WIDTH                       1u
#define FCCU_MCS_VL3(x)                          (((uint32_t)(((uint32_t)(x))<<FCCU_MCS_VL3_SHIFT))&FCCU_MCS_VL3_MASK)
/* TRANS_LOCK Bit Fields */
#define FCCU_TRANS_LOCK_TRANSKEY_MASK            0x1FFu
#define FCCU_TRANS_LOCK_TRANSKEY_SHIFT           0u
#define FCCU_TRANS_LOCK_TRANSKEY_WIDTH           9u
#define FCCU_TRANS_LOCK_TRANSKEY(x)              (((uint32_t)(((uint32_t)(x))<<FCCU_TRANS_LOCK_TRANSKEY_SHIFT))&FCCU_TRANS_LOCK_TRANSKEY_MASK)
/* PERMNT_LOCK Bit Fields */
#define FCCU_PERMNT_LOCK_PERMNTKEY_MASK          0x1FFu
#define FCCU_PERMNT_LOCK_PERMNTKEY_SHIFT         0u
#define FCCU_PERMNT_LOCK_PERMNTKEY_WIDTH         9u
#define FCCU_PERMNT_LOCK_PERMNTKEY(x)            (((uint32_t)(((uint32_t)(x))<<FCCU_PERMNT_LOCK_PERMNTKEY_SHIFT))&FCCU_PERMNT_LOCK_PERMNTKEY_MASK)
/* DELTA_T Bit Fields */
#define FCCU_DELTA_T_DELTA_T_MASK                0x3FFFu
#define FCCU_DELTA_T_DELTA_T_SHIFT               0u
#define FCCU_DELTA_T_DELTA_T_WIDTH               14u
#define FCCU_DELTA_T_DELTA_T(x)                  (((uint32_t)(((uint32_t)(x))<<FCCU_DELTA_T_DELTA_T_SHIFT))&FCCU_DELTA_T_DELTA_T_MASK)
#define FCCU_DELTA_T_T_TRG_MASK                  0x3FFF0000u
#define FCCU_DELTA_T_T_TRG_SHIFT                 16u
#define FCCU_DELTA_T_T_TRG_WIDTH                 14u
#define FCCU_DELTA_T_T_TRG(x)                    (((uint32_t)(((uint32_t)(x))<<FCCU_DELTA_T_T_TRG_SHIFT))&FCCU_DELTA_T_T_TRG_MASK)
/* IRQ_ALARM_EN Bit Fields */
#define FCCU_IRQ_ALARM_EN_IRQENx_MASK            0xFFFFFFFFu
#define FCCU_IRQ_ALARM_EN_IRQENx_SHIFT           0u
#define FCCU_IRQ_ALARM_EN_IRQENx_WIDTH           32u
#define FCCU_IRQ_ALARM_EN_IRQENx(x)              (((uint32_t)(((uint32_t)(x))<<FCCU_IRQ_ALARM_EN_IRQENx_SHIFT))&FCCU_IRQ_ALARM_EN_IRQENx_MASK)
/* NMI_EN Bit Fields */
#define FCCU_NMI_EN_NMIENx_MASK                  0xFFFFFFFFu
#define FCCU_NMI_EN_NMIENx_SHIFT                 0u
#define FCCU_NMI_EN_NMIENx_WIDTH                 32u
#define FCCU_NMI_EN_NMIENx(x)                    (((uint32_t)(((uint32_t)(x))<<FCCU_NMI_EN_NMIENx_SHIFT))&FCCU_NMI_EN_NMIENx_MASK)
/* EOUT_SIG_EN Bit Fields */
#define FCCU_EOUT_SIG_EN_EOUTENx_MASK            0xFFFFFFFFu
#define FCCU_EOUT_SIG_EN_EOUTENx_SHIFT           0u
#define FCCU_EOUT_SIG_EN_EOUTENx_WIDTH           32u
#define FCCU_EOUT_SIG_EN_EOUTENx(x)              (((uint32_t)(((uint32_t)(x))<<FCCU_EOUT_SIG_EN_EOUTENx_SHIFT))&FCCU_EOUT_SIG_EN_EOUTENx_MASK)

/*!
 * @}
 */ /* end of group FCCU_Register_Masks */


/*!
 * @}
 */ /* end of group FCCU_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- FIRC Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FIRC_Peripheral_Access_Layer FIRC Peripheral Access Layer
 * @{
 */


/** FIRC - Size of Registers Arrays */

/** FIRC - Register Layout Typedef */
typedef struct {
  __IO uint32_t CTL;                               /**< Control register, offset: 0x0 */
} FIRC_Type, *FIRC_MemMapPtr;

 /** Number of instances of the FIRC module. */
#define FIRC_INSTANCE_COUNT                      (1u)


/* FIRC - Peripheral instance base addresses */
/** Peripheral FIRC base address */
#define FIRC_BASE                                (0xFFFB0000u)
/** Peripheral FIRC base pointer */
#define FIRC                                     ((FIRC_Type *)FIRC_BASE)
/** Array initializer of FIRC peripheral base addresses */
#define FIRC_BASE_ADDRS                          { FIRC_BASE }
/** Array initializer of FIRC peripheral base pointers */
#define FIRC_BASE_PTRS                           { FIRC }

/* ----------------------------------------------------------------------------
   -- FIRC Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FIRC_Register_Masks FIRC Register Masks
 * @{
 */

/* CTL Bit Fields */
#define FIRC_CTL_FIRCS_MASK                      0x10u
#define FIRC_CTL_FIRCS_SHIFT                     4u
#define FIRC_CTL_FIRCS_WIDTH                     1u
#define FIRC_CTL_FIRCS(x)                        (((uint32_t)(((uint32_t)(x))<<FIRC_CTL_FIRCS_SHIFT))&FIRC_CTL_FIRCS_MASK)
#define FIRC_CTL_FIRCDIV_MASK                    0x1F00u
#define FIRC_CTL_FIRCDIV_SHIFT                   8u
#define FIRC_CTL_FIRCDIV_WIDTH                   5u
#define FIRC_CTL_FIRCDIV(x)                      (((uint32_t)(((uint32_t)(x))<<FIRC_CTL_FIRCDIV_SHIFT))&FIRC_CTL_FIRCDIV_MASK)

/*!
 * @}
 */ /* end of group FIRC_Register_Masks */


/*!
 * @}
 */ /* end of group FIRC_Peripheral_Access_Layer */


/* ----------------------------------------------------------------------------
   -- FR Peripheral Access Layer
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FR_Peripheral_Access_Layer FR Peripheral Access Layer
 * @{
 */


/** FR - Size of Registers Arrays */
#define FR_NMVR_COUNT                            6u
#define FR_SSR_COUNT                             8u
#define FR_SSCR_COUNT                            4u
#define FR_MB_COUNT                              128u
#define FR_MBDOR_COUNT                           132u
#define FR_LEETR_COUNT                           6u

/** FR - Register Layout Typedef */
typedef struct {
  __I  uint16_t MVR;                               /**< Module Version Register, offset: 0x0 */
  __IO uint16_t MCR;                               /**< Module Configuration Register, offset: 0x2 */
  __IO uint16_t SYMBADHR;                          /**< System Memory Base Address High Register, offset: 0x4 */
  __IO uint16_t SYMBADLR;                          /**< System Memory Base Address Low Register, offset: 0x6 */
  __IO uint16_t STBSCR;                            /**< Strobe Signal Control Register, offset: 0x8 */
       uint8_t RESERVED_0[2];
  __IO uint16_t MBDSR;                             /**< Message Buffer Data Size Register, offset: 0xC */
  __IO uint16_t MBSSUTR;                           /**< Message Buffer Segment Size and Utilization Register, offset: 0xE */
  __IO uint16_t PEDRAR;                            /**< PE DRAM Access Register, offset: 0x10 */
  __IO uint16_t PEDRDR;                            /**< PE DRAM Data Register, offset: 0x12 */
  __IO uint16_t POCR;                              /**< Protocol Operation Control Register, offset: 0x14 */
  __IO uint16_t GIFER;                             /**< Global Interrupt Flag and Enable Register, offset: 0x16 */
  __IO uint16_t PIFR0;                             /**< Protocol Interrupt Flag Register 0, offset: 0x18 */
  __IO uint16_t PIFR1;                             /**< Protocol Interrupt Flag Register 1, offset: 0x1A */
  __IO uint16_t PIER0;                             /**< Protocol Interrupt Enable Register 0, offset: 0x1C */
  __IO uint16_t PIER1;                             /**< Protocol Interrupt Enable Register 1, offset: 0x1E */
  __IO uint16_t CHIERFR;                           /**< CHI Error Flag Register, offset: 0x20 */
  __IO uint16_t MBIVEC;                            /**< Message Buffer Interrupt Vector Register, offset: 0x22 */
  __I  uint16_t CASERCR;                           /**< Channel A Status Error Counter Register, offset: 0x24 */
  __I  uint16_t CBSERCR;                           /**< Channel B Status Error Counter Register, offset: 0x26 */
  __I  uint16_t PSR0;                              /**< Protocol Status Register 0, offset: 0x28 */
  __IO uint16_t PSR1;                              /**< Protocol Status Register 1, offset: 0x2A */
  __I  uint16_t PSR2;                              /**< Protocol Status Register 2, offset: 0x2C */
  __IO uint16_t PSR3;                              /**< Protocol Status Register 3, offset: 0x2E */
  __I  uint16_t MTCTR;                             /**< Macrotick Counter Register, offset: 0x30 */
  __I  uint16_t CYCTR;                             /**< Cycle Counter Register, offset: 0x32 */
  __I  uint16_t SLTCTAR;                           /**< Slot Counter Channel A Register, offset: 0x34 */
  __I  uint16_t SLTCTBR;                           /**< Slot Counter Channel B Register, offset: 0x36 */
  __I  uint16_t RTCORVR;                           /**< Rate Correction Value Register, offset: 0x38 */
  __I  uint16_t OFCORVR;                           /**< Offset Correction Value Register, offset: 0x3A */
  __I  uint16_t CIFR;                              /**< Combined Interrupt Flag Register, offset: 0x3C */
  __IO uint16_t SYMATOR;                           /**< System Memory Access Time-Out Register, offset: 0x3E */
  __I  uint16_t SFCNTR;                            /**< Sync Frame Counter Register, offset: 0x40 */
  __IO uint16_t SFTOR;                             /**< Sync Frame Table Offset Register, offset: 0x42 */
  __IO uint16_t SFTCCSR;                           /**< Sync Frame Table Configuration, Control, Status Register, offset: 0x44 */
  __IO uint16_t SFIDRFR;                           /**< Sync Frame ID Rejection Filter Register, offset: 0x46 */
  __IO uint16_t SFIDAFVR;                          /**< Sync Frame ID Acceptance Filter Value Register, offset: 0x48 */
  __IO uint16_t SFIDAFMR;                          /**< Sync Frame ID Acceptance Filter Mask Register, offset: 0x4A */
  __I  uint16_t NMVR[FR_NMVR_COUNT];               /**< Network Management Vector Register, array offset: 0x4C, array step: 0x2 */
  __IO uint16_t NMVLR;                             /**< Network Management Vector Length Register, offset: 0x58 */
  __IO uint16_t TICCR;                             /**< Timer Configuration and Control Register, offset: 0x5A */
  __IO uint16_t TI1CYSR;                           /**< Timer 1 Cycle Set Register, offset: 0x5C */
  __IO uint16_t TI1MTOR;                           /**< Timer 1 Macrotick Offset Register, offset: 0x5E */
  union {                                          /* offset: 0x60 */
    __IO uint16_t ABS;                               /**< Timer 2 Configuration Register 0 (Absolute Timer Configuration), offset: 0x60 */
    __IO uint16_t REL;                               /**< Timer 2 Configuration Register 0 (Relative Timer Configuration), offset: 0x60 */
  } TI2CR0;
  union {                                          /* offset: 0x62 */
    __IO uint16_t ABS;                               /**< Timer 2 Configuration Register 1 (Absolute Timer Configuration), offset: 0x62 */
    __IO uint16_t REL;                               /**< Timer 2 Configuration Register 1 (Relative Timer Configuration), offset: 0x62 */
  } TI2CR1;
  __IO uint16_t SSSR;                              /**< Slot Status Selection Register, offset: 0x64 */
  __IO uint16_t SSCCR;                             /**< Slot Status Counter Condition Register, offset: 0x66 */
  __I  uint16_t SSR[FR_SSR_COUNT];                 /**< Slot Status Register, array offset: 0x68, array step: 0x2 */
  __I  uint16_t SSCR[FR_SSCR_COUNT];               /**< Slot Status Counter Register, array offset: 0x78, array step: 0x2 */
  __IO uint16_t MTSACFR;                           /**< MTS A Configuration Register, offset: 0x80 */
  __IO uint16_t MTSBCFR;                           /**< MTS B Configuration Register, offset: 0x82 */
  __IO uint16_t RSBIR;                             /**< Receive Shadow Buffer Index Register, offset: 0x84 */
  __IO uint16_t RFWMSR;                            /**< Receive FIFO Watermark and Selection Register, offset: 0x86 */
  __IO uint16_t RFSIR;                             /**< Receive FIFO Start Index Register, offset: 0x88 */
  __IO uint16_t RFDSR;                             /**< Receive FIFO Depth and Size Register, offset: 0x8A */
  __I  uint16_t RFARIR;                            /**< Receive FIFO A Read Index Register, offset: 0x8C */
  __I  uint16_t RFBRIR;                            /**< Receive FIFO B Read Index Register, offset: 0x8E */
  __IO uint16_t RFMIDAFVR;                         /**< Receive FIFO Message ID Acceptance Filter Value Register, offset: 0x90 */
  __IO uint16_t RFMIDAFMR;                         /**< Receive FIFO Message ID Acceptance Filter Mask Register, offset: 0x92 */
  __IO uint16_t RFFIDRFVR;                         /**< Receive FIFO Frame ID Rejection Filter Value Register, offset: 0x94 */
  __IO uint16_t RFFIDRFMR;                         /**< Receive FIFO Frame ID Rejection Filter Mask Register, offset: 0x96 */
  __IO uint16_t RFRFCFR;                           /**< Receive FIFO Range Filter Configuration Register, offset: 0x98 */
  __IO uint16_t RFRFCTR;                           /**< Receive FIFO Range Filter Control Register, offset: 0x9A */
  __I  uint16_t LDTXSLAR;                          /**< Last Dynamic Transmit Slot Channel A Register, offset: 0x9C */
  __I  uint16_t LDTXSLBR;                          /**< Last Dynamic Transmit Slot Channel B Register, offset: 0x9E */
  __IO uint16_t PCR0;                              /**< Protocol Configuration Register 0, offset: 0xA0 */
  __IO uint16_t PCR1;                              /**< Protocol Configuration Register 1, offset: 0xA2 */
  __IO uint16_t PCR2;                              /**< Protocol Configuration Register 2, offset: 0xA4 */
  __IO uint16_t PCR3;                              /**< Protocol Configuration Register 3, offset: 0xA6 */
  __IO uint16_t PCR4;                              /**< Protocol Configuration Register 4, offset: 0xA8 */
  __IO uint16_t PCR5;                              /**< Protocol Configuration Register 5, offset: 0xAA */
  __IO uint16_t PCR6;                              /**< Protocol Configuration Register 6, offset: 0xAC */
  __IO uint16_t PCR7;                              /**< Protocol Configuration Register 7, offset: 0xAE */
  __IO uint16_t PCR8;                              /**< Protocol Configuration Register 8, offset: 0xB0 */
  __IO uint16_t PCR9;                              /**< Protocol Configuration Register 9, offset: 0xB2 */
  __IO uint16_t PCR10;                             /**< Protocol Configuration Register 10, offset: 0xB4 */
  __IO uint16_t PCR11;                             /**< Protocol Configuration Register 11, offset: 0xB6 */
  __IO uint16_t PCR12;                             /**< Protocol Configuration Register 12, offset: 0xB8 */
  __IO uint16_t PCR13;                             /**< Protocol Configuration Register 13, offset: 0xBA */
  __IO uint16_t PCR14;                             /**< Protocol Configuration Register 14, offset: 0xBC */
  __IO uint16_t PCR15;                             /**< Protocol Configuration Register 15, offset: 0xBE */
  __IO uint16_t PCR16;                             /**< Protocol Configuration Register 16, offset: 0xC0 */
  __IO uint16_t PCR17;                             /**< Protocol Configuration Register 17, offset: 0xC2 */
  __IO uint16_t PCR18;                             /**< Protocol Configuration Register 18, offset: 0xC4 */
  __IO uint16_t PCR19;                             /**< Protocol Configuration Register 19, offset: 0xC6 */
  __IO uint16_t PCR20;                             /**< Protocol Configuration Register 20, offset: 0xC8 */
  __IO uint16_t PCR21;                             /**< Protocol Configuration Register 21, offset: 0xCA */
  __IO uint16_t PCR22;                             /**< Protocol Configuration Register 22, offset: 0xCC */
  __IO uint16_t PCR23;                             /**< Protocol Configuration Register 23, offset: 0xCE */
  __IO uint16_t PCR24;                             /**< Protocol Configuration Register 24, offset: 0xD0 */
  __IO uint16_t PCR25;                             /**< Protocol Configuration Register 25, offset: 0xD2 */
  __IO uint16_t PCR26;                             /**< Protocol Configuration Register 26, offset: 0xD4 */
  __IO uint16_t PCR27;                             /**< Protocol Configuration Register 27, offset: 0xD6 */
  __IO uint16_t PCR28;                             /**< Protocol Configuration Register 28, offset: 0xD8 */
  __IO uint16_t PCR29;                             /**< Protocol Configuration Register 29, offset: 0xDA */
  __IO uint16_t PCR30;                             /**< Protocol Configuration Register 30, offset: 0xDC */
  __IO uint16_t STPWHR;                            /**< StopWatch Count High Register, offset: 0xDE */
  __IO uint16_t STPWLR;                            /**< Stop Watch Count Low Register, offset: 0xE0 */
  __IO uint16_t PEOER;                             /**< Protocol Event Output Enable and StopWatch Control Register, offset: 0xE2 */
       uint8_t RESERVED_1[2];
  __IO uint16_t RFSDOR;                            /**< Receive FIFO Start Data Offset Register, offset: 0xE6 */
  __IO uint16_t RFSYMBADHR;                        /**< Receive FIFO System Memory Base Address High Register, offset: 0xE8 */
  __IO uint16_t RFSYMBADLR;                        /**< Receive FIFO System Memory Base Address Low Register, offset: 0xEA */
  __IO uint16_t RFPTR;                             /**< Receive FIFO Periodic Timer Register, offset: 0xEC */
  __IO uint16_t RFFLPCR;                           /**< Receive FIFO Fill Level and POP Count Register, offset: 0xEE */
  __IO uint16_t EEIFER;                            /**< ECC Error Interrupt Flag and Enable Register, offset: 0xF0 */
  __IO uint16_t EERICR;                            /**< ECC Error Report and Injection Control Register, offset: 0xF2 */
  __I  uint16_t EERAR;                             /**< ECC Error Report Address Register, offset: 0xF4 */
  __I  uint16_t EERDR;                             /**< ECC Error Report Data Register, offset: 0xF6 */
  __I  uint16_t EERCR;                             /**< ECC Error Report Code Register, offset: 0xF8 */
  __IO uint16_t EEIAR;                             /**< ECC Error Injection Address Register, offset: 0xFA */
  __IO uint16_t EEIDR;                             /**< ECC Error Injection Data Register, offset: 0xFC */
  __IO uint16_t EEICR;                             /**< ECC Error Injection Code Register, offset: 0xFE */
       uint8_t RESERVED_2[1792];
  struct {                                         /* offset: 0x800, array step: 0x8 */
    __IO uint16_t CCSR;                              /**< Message Buffer Configuration, Control, Status Register, array offset: 0x800, array step: 0x8 */
    __IO uint16_t CCFR;                              /**< Message Buffer Cycle Counter Filter Register, array offset: 0x802, array step: 0x8 */
    __IO uint16_t FIDR;                              /**< Message Buffer Frame ID Register, array offset: 0x804, array step: 0x8 */
    __IO uint16_t IDXR;                              /**< Message Buffer Index Register, array offset: 0x806, array step: 0x8 */
  } MB[FR_MB_COUNT];
       uint8_t RESERVED_3[1024];
  __IO uint16_t MBDOR[FR_MBDOR_COUNT];             /**< Message Buffer Data Field Offset Register, array offset: 0x1000, array step: 0x2 */
  __IO uint16_t LEETR[FR_LEETR_COUNT];             /**< LRAM ECC Error Test Register, array offset: 0x1108, array step: 0x2 */
} FR_Type, *FR_MemMapPtr;

 /** Number of instances of the FR module. */
#define FR_INSTANCE_COUNT                        (1u)


/* FR - Peripheral instance base addresses */
/** Peripheral FR base address */
#define FR_BASE                                  (0xFFE50000u)
/** Peripheral FR base pointer */
#define FR                                       ((FR_Type *)FR_BASE)
/** Array initializer of FR peripheral base addresses */
#define FR_BASE_ADDRS                            { FR_BASE }
/** Array initializer of FR peripheral base pointers */
#define FR_BASE_PTRS                             { FR }

/* ----------------------------------------------------------------------------
   -- FR Register Masks
   ---------------------------------------------------------------------------- */

/*!
 * @addtogroup FR_Register_Masks FR Register Masks
 * @{
 */

/* MVR Bit Fields */
#define FR_MVR_PEVER_MASK                        0xFFu
#define FR_MVR_PEVER_SHIFT                       0u
#define FR_MVR_PEVER_WIDTH                       8u
#define FR_MVR_PEVER(x)                          (((uint16_t)(((uint16_t)(x))<<FR_MVR_PEVER_SHIFT))&FR_MVR_PEVER_MASK)
#define FR_MVR_CHIVER_MASK                       0xFF00u
#define FR_MVR_CHIVER_SHIFT                      8u
#define FR_MVR_CHIVER_WIDTH                      8u
#define FR_MVR_CHIVER(x)                         (((uint16_t)(((uint16_t)(x))<<FR_MVR_CHIVER_SHIFT))&FR_MVR_CHIVER_MASK)
/* MCR Bit Fields */
#define FR_MCR_BITRATE_MASK                      0xEu
#define FR_MCR_BITRATE_SHIFT                     1u
#define FR_MCR_BITRATE_WIDTH                     3u
#define FR_MCR_BITRATE(x)                        (((uint16_t)(((uint16_t)(x))<<FR_MCR_BITRATE_SHIFT))&FR_MCR_BITRATE_MASK)
#define FR_MCR_CLKSEL_MASK                       0x10u
#define FR_MCR_CLKSEL_SHIFT                      4u
#define FR_MCR_CLKSEL_WIDTH                      1u
#define FR_MCR_CLKSEL(x)                         (((uint16_t)(((uint16_t)(x))<<FR_MCR_CLKSEL_SHIFT))&FR_MCR_CLKSEL_MASK)
#define FR_MCR_FAM_MASK                          0x40u
#define FR_MCR_FAM_SHIFT                         6u
#define FR_MCR_FAM_WIDTH                         1u
#define FR_MCR_FAM(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_FAM_SHIFT))&FR_MCR_FAM_MASK)
#define FR_MCR_FUM_MASK                          0x80u
#define FR_MCR_FUM_SHIFT                         7u
#define FR_MCR_FUM_WIDTH                         1u
#define FR_MCR_FUM(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_FUM_SHIFT))&FR_MCR_FUM_MASK)
#define FR_MCR_ECCE_MASK                         0x200u
#define FR_MCR_ECCE_SHIFT                        9u
#define FR_MCR_ECCE_WIDTH                        1u
#define FR_MCR_ECCE(x)                           (((uint16_t)(((uint16_t)(x))<<FR_MCR_ECCE_SHIFT))&FR_MCR_ECCE_MASK)
#define FR_MCR_SFFE_MASK                         0x400u
#define FR_MCR_SFFE_SHIFT                        10u
#define FR_MCR_SFFE_WIDTH                        1u
#define FR_MCR_SFFE(x)                           (((uint16_t)(((uint16_t)(x))<<FR_MCR_SFFE_SHIFT))&FR_MCR_SFFE_MASK)
#define FR_MCR_CHA_MASK                          0x800u
#define FR_MCR_CHA_SHIFT                         11u
#define FR_MCR_CHA_WIDTH                         1u
#define FR_MCR_CHA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_CHA_SHIFT))&FR_MCR_CHA_MASK)
#define FR_MCR_CHB_MASK                          0x1000u
#define FR_MCR_CHB_SHIFT                         12u
#define FR_MCR_CHB_WIDTH                         1u
#define FR_MCR_CHB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_CHB_SHIFT))&FR_MCR_CHB_MASK)
#define FR_MCR_SCM_MASK                          0x2000u
#define FR_MCR_SCM_SHIFT                         13u
#define FR_MCR_SCM_WIDTH                         1u
#define FR_MCR_SCM(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_SCM_SHIFT))&FR_MCR_SCM_MASK)
#define FR_MCR_SBFF_MASK                         0x4000u
#define FR_MCR_SBFF_SHIFT                        14u
#define FR_MCR_SBFF_WIDTH                        1u
#define FR_MCR_SBFF(x)                           (((uint16_t)(((uint16_t)(x))<<FR_MCR_SBFF_SHIFT))&FR_MCR_SBFF_MASK)
#define FR_MCR_MEN_MASK                          0x8000u
#define FR_MCR_MEN_SHIFT                         15u
#define FR_MCR_MEN_WIDTH                         1u
#define FR_MCR_MEN(x)                            (((uint16_t)(((uint16_t)(x))<<FR_MCR_MEN_SHIFT))&FR_MCR_MEN_MASK)
/* SYMBADHR Bit Fields */
#define FR_SYMBADHR_SMBA_MASK                    0xFFFFu
#define FR_SYMBADHR_SMBA_SHIFT                   0u
#define FR_SYMBADHR_SMBA_WIDTH                   16u
#define FR_SYMBADHR_SMBA(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SYMBADHR_SMBA_SHIFT))&FR_SYMBADHR_SMBA_MASK)
/* SYMBADLR Bit Fields */
#define FR_SYMBADLR_SMBA_MASK                    0xFFF0u
#define FR_SYMBADLR_SMBA_SHIFT                   4u
#define FR_SYMBADLR_SMBA_WIDTH                   12u
#define FR_SYMBADLR_SMBA(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SYMBADLR_SMBA_SHIFT))&FR_SYMBADLR_SMBA_MASK)
/* STBSCR Bit Fields */
#define FR_STBSCR_STBPSEL_MASK                   0x3u
#define FR_STBSCR_STBPSEL_SHIFT                  0u
#define FR_STBSCR_STBPSEL_WIDTH                  2u
#define FR_STBSCR_STBPSEL(x)                     (((uint16_t)(((uint16_t)(x))<<FR_STBSCR_STBPSEL_SHIFT))&FR_STBSCR_STBPSEL_MASK)
#define FR_STBSCR_ENB_MASK                       0x10u
#define FR_STBSCR_ENB_SHIFT                      4u
#define FR_STBSCR_ENB_WIDTH                      1u
#define FR_STBSCR_ENB(x)                         (((uint16_t)(((uint16_t)(x))<<FR_STBSCR_ENB_SHIFT))&FR_STBSCR_ENB_MASK)
#define FR_STBSCR_SEL_MASK                       0xF00u
#define FR_STBSCR_SEL_SHIFT                      8u
#define FR_STBSCR_SEL_WIDTH                      4u
#define FR_STBSCR_SEL(x)                         (((uint16_t)(((uint16_t)(x))<<FR_STBSCR_SEL_SHIFT))&FR_STBSCR_SEL_MASK)
#define FR_STBSCR_WMD_MASK                       0x8000u
#define FR_STBSCR_WMD_SHIFT                      15u
#define FR_STBSCR_WMD_WIDTH                      1u
#define FR_STBSCR_WMD(x)                         (((uint16_t)(((uint16_t)(x))<<FR_STBSCR_WMD_SHIFT))&FR_STBSCR_WMD_MASK)
/* MBDSR Bit Fields */
#define FR_MBDSR_MBSEG1DS_MASK                   0x7Fu
#define FR_MBDSR_MBSEG1DS_SHIFT                  0u
#define FR_MBDSR_MBSEG1DS_WIDTH                  7u
#define FR_MBDSR_MBSEG1DS(x)                     (((uint16_t)(((uint16_t)(x))<<FR_MBDSR_MBSEG1DS_SHIFT))&FR_MBDSR_MBSEG1DS_MASK)
#define FR_MBDSR_MBSEG2DS_MASK                   0x7F00u
#define FR_MBDSR_MBSEG2DS_SHIFT                  8u
#define FR_MBDSR_MBSEG2DS_WIDTH                  7u
#define FR_MBDSR_MBSEG2DS(x)                     (((uint16_t)(((uint16_t)(x))<<FR_MBDSR_MBSEG2DS_SHIFT))&FR_MBDSR_MBSEG2DS_MASK)
/* MBSSUTR Bit Fields */
#define FR_MBSSUTR_LAST_MB_UTIL_MASK             0x7Fu
#define FR_MBSSUTR_LAST_MB_UTIL_SHIFT            0u
#define FR_MBSSUTR_LAST_MB_UTIL_WIDTH            7u
#define FR_MBSSUTR_LAST_MB_UTIL(x)               (((uint16_t)(((uint16_t)(x))<<FR_MBSSUTR_LAST_MB_UTIL_SHIFT))&FR_MBSSUTR_LAST_MB_UTIL_MASK)
#define FR_MBSSUTR_LAST_MB_SEG1_MASK             0x7F00u
#define FR_MBSSUTR_LAST_MB_SEG1_SHIFT            8u
#define FR_MBSSUTR_LAST_MB_SEG1_WIDTH            7u
#define FR_MBSSUTR_LAST_MB_SEG1(x)               (((uint16_t)(((uint16_t)(x))<<FR_MBSSUTR_LAST_MB_SEG1_SHIFT))&FR_MBSSUTR_LAST_MB_SEG1_MASK)
/* PEDRAR Bit Fields */
#define FR_PEDRAR_DAD_MASK                       0x1u
#define FR_PEDRAR_DAD_SHIFT                      0u
#define FR_PEDRAR_DAD_WIDTH                      1u
#define FR_PEDRAR_DAD(x)                         (((uint16_t)(((uint16_t)(x))<<FR_PEDRAR_DAD_SHIFT))&FR_PEDRAR_DAD_MASK)
#define FR_PEDRAR_ADDR_MASK                      0xFFEu
#define FR_PEDRAR_ADDR_SHIFT                     1u
#define FR_PEDRAR_ADDR_WIDTH                     11u
#define FR_PEDRAR_ADDR(x)                        (((uint16_t)(((uint16_t)(x))<<FR_PEDRAR_ADDR_SHIFT))&FR_PEDRAR_ADDR_MASK)
#define FR_PEDRAR_INST_MASK                      0xF000u
#define FR_PEDRAR_INST_SHIFT                     12u
#define FR_PEDRAR_INST_WIDTH                     4u
#define FR_PEDRAR_INST(x)                        (((uint16_t)(((uint16_t)(x))<<FR_PEDRAR_INST_SHIFT))&FR_PEDRAR_INST_MASK)
/* PEDRDR Bit Fields */
#define FR_PEDRDR_DATA_MASK                      0xFFFFu
#define FR_PEDRDR_DATA_SHIFT                     0u
#define FR_PEDRDR_DATA_WIDTH                     16u
#define FR_PEDRDR_DATA(x)                        (((uint16_t)(((uint16_t)(x))<<FR_PEDRDR_DATA_SHIFT))&FR_PEDRDR_DATA_MASK)
/* POCR Bit Fields */
#define FR_POCR_POCCMD_MASK                      0xFu
#define FR_POCR_POCCMD_SHIFT                     0u
#define FR_POCR_POCCMD_WIDTH                     4u
#define FR_POCR_POCCMD(x)                        (((uint16_t)(((uint16_t)(x))<<FR_POCR_POCCMD_SHIFT))&FR_POCR_POCCMD_MASK)
#define FR_POCR_BSY_WMC_MASK                     0x80u
#define FR_POCR_BSY_WMC_SHIFT                    7u
#define FR_POCR_BSY_WMC_WIDTH                    1u
#define FR_POCR_BSY_WMC(x)                       (((uint16_t)(((uint16_t)(x))<<FR_POCR_BSY_WMC_SHIFT))&FR_POCR_BSY_WMC_MASK)
#define FR_POCR_ERC_AP_MASK                      0x300u
#define FR_POCR_ERC_AP_SHIFT                     8u
#define FR_POCR_ERC_AP_WIDTH                     2u
#define FR_POCR_ERC_AP(x)                        (((uint16_t)(((uint16_t)(x))<<FR_POCR_ERC_AP_SHIFT))&FR_POCR_ERC_AP_MASK)
#define FR_POCR_EOC_AP_MASK                      0xC00u
#define FR_POCR_EOC_AP_SHIFT                     10u
#define FR_POCR_EOC_AP_WIDTH                     2u
#define FR_POCR_EOC_AP(x)                        (((uint16_t)(((uint16_t)(x))<<FR_POCR_EOC_AP_SHIFT))&FR_POCR_EOC_AP_MASK)
#define FR_POCR_WME_MASK                         0x8000u
#define FR_POCR_WME_SHIFT                        15u
#define FR_POCR_WME_WIDTH                        1u
#define FR_POCR_WME(x)                           (((uint16_t)(((uint16_t)(x))<<FR_POCR_WME_SHIFT))&FR_POCR_WME_MASK)
/* GIFER Bit Fields */
#define FR_GIFER_TBIE_MASK                       0x1u
#define FR_GIFER_TBIE_SHIFT                      0u
#define FR_GIFER_TBIE_WIDTH                      1u
#define FR_GIFER_TBIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_TBIE_SHIFT))&FR_GIFER_TBIE_MASK)
#define FR_GIFER_RBIE_MASK                       0x2u
#define FR_GIFER_RBIE_SHIFT                      1u
#define FR_GIFER_RBIE_WIDTH                      1u
#define FR_GIFER_RBIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_RBIE_SHIFT))&FR_GIFER_RBIE_MASK)
#define FR_GIFER_FAFAIE_MASK                     0x4u
#define FR_GIFER_FAFAIE_SHIFT                    2u
#define FR_GIFER_FAFAIE_WIDTH                    1u
#define FR_GIFER_FAFAIE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_GIFER_FAFAIE_SHIFT))&FR_GIFER_FAFAIE_MASK)
#define FR_GIFER_FAFBIE_MASK                     0x8u
#define FR_GIFER_FAFBIE_SHIFT                    3u
#define FR_GIFER_FAFBIE_WIDTH                    1u
#define FR_GIFER_FAFBIE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_GIFER_FAFBIE_SHIFT))&FR_GIFER_FAFBIE_MASK)
#define FR_GIFER_WUPIE_MASK                      0x10u
#define FR_GIFER_WUPIE_SHIFT                     4u
#define FR_GIFER_WUPIE_WIDTH                     1u
#define FR_GIFER_WUPIE(x)                        (((uint16_t)(((uint16_t)(x))<<FR_GIFER_WUPIE_SHIFT))&FR_GIFER_WUPIE_MASK)
#define FR_GIFER_CHIE_MASK                       0x20u
#define FR_GIFER_CHIE_SHIFT                      5u
#define FR_GIFER_CHIE_WIDTH                      1u
#define FR_GIFER_CHIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_CHIE_SHIFT))&FR_GIFER_CHIE_MASK)
#define FR_GIFER_PRIE_MASK                       0x40u
#define FR_GIFER_PRIE_SHIFT                      6u
#define FR_GIFER_PRIE_WIDTH                      1u
#define FR_GIFER_PRIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_PRIE_SHIFT))&FR_GIFER_PRIE_MASK)
#define FR_GIFER_MIE_MASK                        0x80u
#define FR_GIFER_MIE_SHIFT                       7u
#define FR_GIFER_MIE_WIDTH                       1u
#define FR_GIFER_MIE(x)                          (((uint16_t)(((uint16_t)(x))<<FR_GIFER_MIE_SHIFT))&FR_GIFER_MIE_MASK)
#define FR_GIFER_TBIF_MASK                       0x100u
#define FR_GIFER_TBIF_SHIFT                      8u
#define FR_GIFER_TBIF_WIDTH                      1u
#define FR_GIFER_TBIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_TBIF_SHIFT))&FR_GIFER_TBIF_MASK)
#define FR_GIFER_RBIF_MASK                       0x200u
#define FR_GIFER_RBIF_SHIFT                      9u
#define FR_GIFER_RBIF_WIDTH                      1u
#define FR_GIFER_RBIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_RBIF_SHIFT))&FR_GIFER_RBIF_MASK)
#define FR_GIFER_FAFAIF_MASK                     0x400u
#define FR_GIFER_FAFAIF_SHIFT                    10u
#define FR_GIFER_FAFAIF_WIDTH                    1u
#define FR_GIFER_FAFAIF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_GIFER_FAFAIF_SHIFT))&FR_GIFER_FAFAIF_MASK)
#define FR_GIFER_FAFBIF_MASK                     0x800u
#define FR_GIFER_FAFBIF_SHIFT                    11u
#define FR_GIFER_FAFBIF_WIDTH                    1u
#define FR_GIFER_FAFBIF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_GIFER_FAFBIF_SHIFT))&FR_GIFER_FAFBIF_MASK)
#define FR_GIFER_WUPIF_MASK                      0x1000u
#define FR_GIFER_WUPIF_SHIFT                     12u
#define FR_GIFER_WUPIF_WIDTH                     1u
#define FR_GIFER_WUPIF(x)                        (((uint16_t)(((uint16_t)(x))<<FR_GIFER_WUPIF_SHIFT))&FR_GIFER_WUPIF_MASK)
#define FR_GIFER_CHIF_MASK                       0x2000u
#define FR_GIFER_CHIF_SHIFT                      13u
#define FR_GIFER_CHIF_WIDTH                      1u
#define FR_GIFER_CHIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_CHIF_SHIFT))&FR_GIFER_CHIF_MASK)
#define FR_GIFER_PRIF_MASK                       0x4000u
#define FR_GIFER_PRIF_SHIFT                      14u
#define FR_GIFER_PRIF_WIDTH                      1u
#define FR_GIFER_PRIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_GIFER_PRIF_SHIFT))&FR_GIFER_PRIF_MASK)
#define FR_GIFER_MIF_MASK                        0x8000u
#define FR_GIFER_MIF_SHIFT                       15u
#define FR_GIFER_MIF_WIDTH                       1u
#define FR_GIFER_MIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_GIFER_MIF_SHIFT))&FR_GIFER_MIF_MASK)
/* PIFR0 Bit Fields */
#define FR_PIFR0_CYS_IF_MASK                     0x1u
#define FR_PIFR0_CYS_IF_SHIFT                    0u
#define FR_PIFR0_CYS_IF_WIDTH                    1u
#define FR_PIFR0_CYS_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_CYS_IF_SHIFT))&FR_PIFR0_CYS_IF_MASK)
#define FR_PIFR0_TI1_IF_MASK                     0x2u
#define FR_PIFR0_TI1_IF_SHIFT                    1u
#define FR_PIFR0_TI1_IF_WIDTH                    1u
#define FR_PIFR0_TI1_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_TI1_IF_SHIFT))&FR_PIFR0_TI1_IF_MASK)
#define FR_PIFR0_TI2_IF_MASK                     0x4u
#define FR_PIFR0_TI2_IF_SHIFT                    2u
#define FR_PIFR0_TI2_IF_WIDTH                    1u
#define FR_PIFR0_TI2_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_TI2_IF_SHIFT))&FR_PIFR0_TI2_IF_MASK)
#define FR_PIFR0_TBVA_IF_MASK                    0x8u
#define FR_PIFR0_TBVA_IF_SHIFT                   3u
#define FR_PIFR0_TBVA_IF_WIDTH                   1u
#define FR_PIFR0_TBVA_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_TBVA_IF_SHIFT))&FR_PIFR0_TBVA_IF_MASK)
#define FR_PIFR0_TBVB_IF_MASK                    0x10u
#define FR_PIFR0_TBVB_IF_SHIFT                   4u
#define FR_PIFR0_TBVB_IF_WIDTH                   1u
#define FR_PIFR0_TBVB_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_TBVB_IF_SHIFT))&FR_PIFR0_TBVB_IF_MASK)
#define FR_PIFR0_LTXA_IF_MASK                    0x20u
#define FR_PIFR0_LTXA_IF_SHIFT                   5u
#define FR_PIFR0_LTXA_IF_WIDTH                   1u
#define FR_PIFR0_LTXA_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_LTXA_IF_SHIFT))&FR_PIFR0_LTXA_IF_MASK)
#define FR_PIFR0_LTXB_IF_MASK                    0x40u
#define FR_PIFR0_LTXB_IF_SHIFT                   6u
#define FR_PIFR0_LTXB_IF_WIDTH                   1u
#define FR_PIFR0_LTXB_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_LTXB_IF_SHIFT))&FR_PIFR0_LTXB_IF_MASK)
#define FR_PIFR0_MTX_IF_MASK                     0x80u
#define FR_PIFR0_MTX_IF_SHIFT                    7u
#define FR_PIFR0_MTX_IF_WIDTH                    1u
#define FR_PIFR0_MTX_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_MTX_IF_SHIFT))&FR_PIFR0_MTX_IF_MASK)
#define FR_PIFR0_MXS_IF_MASK                     0x100u
#define FR_PIFR0_MXS_IF_SHIFT                    8u
#define FR_PIFR0_MXS_IF_WIDTH                    1u
#define FR_PIFR0_MXS_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_MXS_IF_SHIFT))&FR_PIFR0_MXS_IF_MASK)
#define FR_PIFR0_CCL_IF_MASK                     0x200u
#define FR_PIFR0_CCL_IF_SHIFT                    9u
#define FR_PIFR0_CCL_IF_WIDTH                    1u
#define FR_PIFR0_CCL_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_CCL_IF_SHIFT))&FR_PIFR0_CCL_IF_MASK)
#define FR_PIFR0_MOC_IF_MASK                     0x400u
#define FR_PIFR0_MOC_IF_SHIFT                    10u
#define FR_PIFR0_MOC_IF_WIDTH                    1u
#define FR_PIFR0_MOC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_MOC_IF_SHIFT))&FR_PIFR0_MOC_IF_MASK)
#define FR_PIFR0_MRC_IF_MASK                     0x800u
#define FR_PIFR0_MRC_IF_SHIFT                    11u
#define FR_PIFR0_MRC_IF_WIDTH                    1u
#define FR_PIFR0_MRC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_MRC_IF_SHIFT))&FR_PIFR0_MRC_IF_MASK)
#define FR_PIFR0_CSA_IF_MASK                     0x1000u
#define FR_PIFR0_CSA_IF_SHIFT                    12u
#define FR_PIFR0_CSA_IF_WIDTH                    1u
#define FR_PIFR0_CSA_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_CSA_IF_SHIFT))&FR_PIFR0_CSA_IF_MASK)
#define FR_PIFR0_ILCF_IF_MASK                    0x2000u
#define FR_PIFR0_ILCF_IF_SHIFT                   13u
#define FR_PIFR0_ILCF_IF_WIDTH                   1u
#define FR_PIFR0_ILCF_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_ILCF_IF_SHIFT))&FR_PIFR0_ILCF_IF_MASK)
#define FR_PIFR0_INTL_IF_MASK                    0x4000u
#define FR_PIFR0_INTL_IF_SHIFT                   14u
#define FR_PIFR0_INTL_IF_WIDTH                   1u
#define FR_PIFR0_INTL_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_INTL_IF_SHIFT))&FR_PIFR0_INTL_IF_MASK)
#define FR_PIFR0_FATL_IF_MASK                    0x8000u
#define FR_PIFR0_FATL_IF_SHIFT                   15u
#define FR_PIFR0_FATL_IF_WIDTH                   1u
#define FR_PIFR0_FATL_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR0_FATL_IF_SHIFT))&FR_PIFR0_FATL_IF_MASK)
/* PIFR1 Bit Fields */
#define FR_PIFR1_ODT_IF_MASK                     0x10u
#define FR_PIFR1_ODT_IF_SHIFT                    4u
#define FR_PIFR1_ODT_IF_WIDTH                    1u
#define FR_PIFR1_ODT_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_ODT_IF_SHIFT))&FR_PIFR1_ODT_IF_MASK)
#define FR_PIFR1_EVT_IF_MASK                     0x20u
#define FR_PIFR1_EVT_IF_SHIFT                    5u
#define FR_PIFR1_EVT_IF_WIDTH                    1u
#define FR_PIFR1_EVT_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_EVT_IF_SHIFT))&FR_PIFR1_EVT_IF_MASK)
#define FR_PIFR1_SSI0_IF_MASK                    0x100u
#define FR_PIFR1_SSI0_IF_SHIFT                   8u
#define FR_PIFR1_SSI0_IF_WIDTH                   1u
#define FR_PIFR1_SSI0_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_SSI0_IF_SHIFT))&FR_PIFR1_SSI0_IF_MASK)
#define FR_PIFR1_SSI1_IF_MASK                    0x200u
#define FR_PIFR1_SSI1_IF_SHIFT                   9u
#define FR_PIFR1_SSI1_IF_WIDTH                   1u
#define FR_PIFR1_SSI1_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_SSI1_IF_SHIFT))&FR_PIFR1_SSI1_IF_MASK)
#define FR_PIFR1_SSI2_IF_MASK                    0x400u
#define FR_PIFR1_SSI2_IF_SHIFT                   10u
#define FR_PIFR1_SSI2_IF_WIDTH                   1u
#define FR_PIFR1_SSI2_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_SSI2_IF_SHIFT))&FR_PIFR1_SSI2_IF_MASK)
#define FR_PIFR1_SSI3_IF_MASK                    0x800u
#define FR_PIFR1_SSI3_IF_SHIFT                   11u
#define FR_PIFR1_SSI3_IF_WIDTH                   1u
#define FR_PIFR1_SSI3_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_SSI3_IF_SHIFT))&FR_PIFR1_SSI3_IF_MASK)
#define FR_PIFR1_PSC_IF_MASK                     0x1000u
#define FR_PIFR1_PSC_IF_SHIFT                    12u
#define FR_PIFR1_PSC_IF_WIDTH                    1u
#define FR_PIFR1_PSC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_PSC_IF_SHIFT))&FR_PIFR1_PSC_IF_MASK)
#define FR_PIFR1_PECF_IF_MASK                    0x2000u
#define FR_PIFR1_PECF_IF_SHIFT                   13u
#define FR_PIFR1_PECF_IF_WIDTH                   1u
#define FR_PIFR1_PECF_IF(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_PECF_IF_SHIFT))&FR_PIFR1_PECF_IF_MASK)
#define FR_PIFR1_IPC_IF_MASK                     0x4000u
#define FR_PIFR1_IPC_IF_SHIFT                    14u
#define FR_PIFR1_IPC_IF_WIDTH                    1u
#define FR_PIFR1_IPC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_IPC_IF_SHIFT))&FR_PIFR1_IPC_IF_MASK)
#define FR_PIFR1_EMC_IF_MASK                     0x8000u
#define FR_PIFR1_EMC_IF_SHIFT                    15u
#define FR_PIFR1_EMC_IF_WIDTH                    1u
#define FR_PIFR1_EMC_IF(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIFR1_EMC_IF_SHIFT))&FR_PIFR1_EMC_IF_MASK)
/* PIER0 Bit Fields */
#define FR_PIER0_CYS_IE_MASK                     0x1u
#define FR_PIER0_CYS_IE_SHIFT                    0u
#define FR_PIER0_CYS_IE_WIDTH                    1u
#define FR_PIER0_CYS_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_CYS_IE_SHIFT))&FR_PIER0_CYS_IE_MASK)
#define FR_PIER0_TI1_IE_MASK                     0x2u
#define FR_PIER0_TI1_IE_SHIFT                    1u
#define FR_PIER0_TI1_IE_WIDTH                    1u
#define FR_PIER0_TI1_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_TI1_IE_SHIFT))&FR_PIER0_TI1_IE_MASK)
#define FR_PIER0_TI2_IE_MASK                     0x4u
#define FR_PIER0_TI2_IE_SHIFT                    2u
#define FR_PIER0_TI2_IE_WIDTH                    1u
#define FR_PIER0_TI2_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_TI2_IE_SHIFT))&FR_PIER0_TI2_IE_MASK)
#define FR_PIER0_TBVA_IE_MASK                    0x8u
#define FR_PIER0_TBVA_IE_SHIFT                   3u
#define FR_PIER0_TBVA_IE_WIDTH                   1u
#define FR_PIER0_TBVA_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_TBVA_IE_SHIFT))&FR_PIER0_TBVA_IE_MASK)
#define FR_PIER0_TBVB_IE_MASK                    0x10u
#define FR_PIER0_TBVB_IE_SHIFT                   4u
#define FR_PIER0_TBVB_IE_WIDTH                   1u
#define FR_PIER0_TBVB_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_TBVB_IE_SHIFT))&FR_PIER0_TBVB_IE_MASK)
#define FR_PIER0_LTXA_IE_MASK                    0x20u
#define FR_PIER0_LTXA_IE_SHIFT                   5u
#define FR_PIER0_LTXA_IE_WIDTH                   1u
#define FR_PIER0_LTXA_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_LTXA_IE_SHIFT))&FR_PIER0_LTXA_IE_MASK)
#define FR_PIER0_LTXB_IE_MASK                    0x40u
#define FR_PIER0_LTXB_IE_SHIFT                   6u
#define FR_PIER0_LTXB_IE_WIDTH                   1u
#define FR_PIER0_LTXB_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_LTXB_IE_SHIFT))&FR_PIER0_LTXB_IE_MASK)
#define FR_PIER0_MTX_IE_MASK                     0x80u
#define FR_PIER0_MTX_IE_SHIFT                    7u
#define FR_PIER0_MTX_IE_WIDTH                    1u
#define FR_PIER0_MTX_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_MTX_IE_SHIFT))&FR_PIER0_MTX_IE_MASK)
#define FR_PIER0_MXS_IE_MASK                     0x100u
#define FR_PIER0_MXS_IE_SHIFT                    8u
#define FR_PIER0_MXS_IE_WIDTH                    1u
#define FR_PIER0_MXS_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_MXS_IE_SHIFT))&FR_PIER0_MXS_IE_MASK)
#define FR_PIER0_CCL_IE_MASK                     0x200u
#define FR_PIER0_CCL_IE_SHIFT                    9u
#define FR_PIER0_CCL_IE_WIDTH                    1u
#define FR_PIER0_CCL_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_CCL_IE_SHIFT))&FR_PIER0_CCL_IE_MASK)
#define FR_PIER0_MOC_IE_MASK                     0x400u
#define FR_PIER0_MOC_IE_SHIFT                    10u
#define FR_PIER0_MOC_IE_WIDTH                    1u
#define FR_PIER0_MOC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_MOC_IE_SHIFT))&FR_PIER0_MOC_IE_MASK)
#define FR_PIER0_MRC_IE_MASK                     0x800u
#define FR_PIER0_MRC_IE_SHIFT                    11u
#define FR_PIER0_MRC_IE_WIDTH                    1u
#define FR_PIER0_MRC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_MRC_IE_SHIFT))&FR_PIER0_MRC_IE_MASK)
#define FR_PIER0_CSA_IE_MASK                     0x1000u
#define FR_PIER0_CSA_IE_SHIFT                    12u
#define FR_PIER0_CSA_IE_WIDTH                    1u
#define FR_PIER0_CSA_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER0_CSA_IE_SHIFT))&FR_PIER0_CSA_IE_MASK)
#define FR_PIER0_ILCF_IE_MASK                    0x2000u
#define FR_PIER0_ILCF_IE_SHIFT                   13u
#define FR_PIER0_ILCF_IE_WIDTH                   1u
#define FR_PIER0_ILCF_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_ILCF_IE_SHIFT))&FR_PIER0_ILCF_IE_MASK)
#define FR_PIER0_INTL_IE_MASK                    0x4000u
#define FR_PIER0_INTL_IE_SHIFT                   14u
#define FR_PIER0_INTL_IE_WIDTH                   1u
#define FR_PIER0_INTL_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_INTL_IE_SHIFT))&FR_PIER0_INTL_IE_MASK)
#define FR_PIER0_FATL_IE_MASK                    0x8000u
#define FR_PIER0_FATL_IE_SHIFT                   15u
#define FR_PIER0_FATL_IE_WIDTH                   1u
#define FR_PIER0_FATL_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER0_FATL_IE_SHIFT))&FR_PIER0_FATL_IE_MASK)
/* PIER1 Bit Fields */
#define FR_PIER1_ODT_IE_MASK                     0x10u
#define FR_PIER1_ODT_IE_SHIFT                    4u
#define FR_PIER1_ODT_IE_WIDTH                    1u
#define FR_PIER1_ODT_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_ODT_IE_SHIFT))&FR_PIER1_ODT_IE_MASK)
#define FR_PIER1_EVT_IE_MASK                     0x20u
#define FR_PIER1_EVT_IE_SHIFT                    5u
#define FR_PIER1_EVT_IE_WIDTH                    1u
#define FR_PIER1_EVT_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_EVT_IE_SHIFT))&FR_PIER1_EVT_IE_MASK)
#define FR_PIER1_SSI0_IE_MASK                    0x100u
#define FR_PIER1_SSI0_IE_SHIFT                   8u
#define FR_PIER1_SSI0_IE_WIDTH                   1u
#define FR_PIER1_SSI0_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_SSI0_IE_SHIFT))&FR_PIER1_SSI0_IE_MASK)
#define FR_PIER1_SSI1_IE_MASK                    0x200u
#define FR_PIER1_SSI1_IE_SHIFT                   9u
#define FR_PIER1_SSI1_IE_WIDTH                   1u
#define FR_PIER1_SSI1_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_SSI1_IE_SHIFT))&FR_PIER1_SSI1_IE_MASK)
#define FR_PIER1_SSI2_IE_MASK                    0x400u
#define FR_PIER1_SSI2_IE_SHIFT                   10u
#define FR_PIER1_SSI2_IE_WIDTH                   1u
#define FR_PIER1_SSI2_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_SSI2_IE_SHIFT))&FR_PIER1_SSI2_IE_MASK)
#define FR_PIER1_SSI3_IE_MASK                    0x800u
#define FR_PIER1_SSI3_IE_SHIFT                   11u
#define FR_PIER1_SSI3_IE_WIDTH                   1u
#define FR_PIER1_SSI3_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_SSI3_IE_SHIFT))&FR_PIER1_SSI3_IE_MASK)
#define FR_PIER1_PSC_IE_MASK                     0x1000u
#define FR_PIER1_PSC_IE_SHIFT                    12u
#define FR_PIER1_PSC_IE_WIDTH                    1u
#define FR_PIER1_PSC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_PSC_IE_SHIFT))&FR_PIER1_PSC_IE_MASK)
#define FR_PIER1_PECF_IE_MASK                    0x2000u
#define FR_PIER1_PECF_IE_SHIFT                   13u
#define FR_PIER1_PECF_IE_WIDTH                   1u
#define FR_PIER1_PECF_IE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PIER1_PECF_IE_SHIFT))&FR_PIER1_PECF_IE_MASK)
#define FR_PIER1_IPC_IE_MASK                     0x4000u
#define FR_PIER1_IPC_IE_SHIFT                    14u
#define FR_PIER1_IPC_IE_WIDTH                    1u
#define FR_PIER1_IPC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_IPC_IE_SHIFT))&FR_PIER1_IPC_IE_MASK)
#define FR_PIER1_EMC_IE_MASK                     0x8000u
#define FR_PIER1_EMC_IE_SHIFT                    15u
#define FR_PIER1_EMC_IE_WIDTH                    1u
#define FR_PIER1_EMC_IE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PIER1_EMC_IE_SHIFT))&FR_PIER1_EMC_IE_MASK)
/* CHIERFR Bit Fields */
#define FR_CHIERFR_ILSA_EF_MASK                  0x1u
#define FR_CHIERFR_ILSA_EF_SHIFT                 0u
#define FR_CHIERFR_ILSA_EF_WIDTH                 1u
#define FR_CHIERFR_ILSA_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_ILSA_EF_SHIFT))&FR_CHIERFR_ILSA_EF_MASK)
#define FR_CHIERFR_NMF_EF_MASK                   0x2u
#define FR_CHIERFR_NMF_EF_SHIFT                  1u
#define FR_CHIERFR_NMF_EF_WIDTH                  1u
#define FR_CHIERFR_NMF_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_NMF_EF_SHIFT))&FR_CHIERFR_NMF_EF_MASK)
#define FR_CHIERFR_NML_EF_MASK                   0x4u
#define FR_CHIERFR_NML_EF_SHIFT                  2u
#define FR_CHIERFR_NML_EF_WIDTH                  1u
#define FR_CHIERFR_NML_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_NML_EF_SHIFT))&FR_CHIERFR_NML_EF_MASK)
#define FR_CHIERFR_SPL_EF_MASK                   0x8u
#define FR_CHIERFR_SPL_EF_SHIFT                  3u
#define FR_CHIERFR_SPL_EF_WIDTH                  1u
#define FR_CHIERFR_SPL_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_SPL_EF_SHIFT))&FR_CHIERFR_SPL_EF_MASK)
#define FR_CHIERFR_DPL_EF_MASK                   0x10u
#define FR_CHIERFR_DPL_EF_SHIFT                  4u
#define FR_CHIERFR_DPL_EF_WIDTH                  1u
#define FR_CHIERFR_DPL_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_DPL_EF_SHIFT))&FR_CHIERFR_DPL_EF_MASK)
#define FR_CHIERFR_FID_EF_MASK                   0x20u
#define FR_CHIERFR_FID_EF_SHIFT                  5u
#define FR_CHIERFR_FID_EF_WIDTH                  1u
#define FR_CHIERFR_FID_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FID_EF_SHIFT))&FR_CHIERFR_FID_EF_MASK)
#define FR_CHIERFR_SBCF_EF_MASK                  0x40u
#define FR_CHIERFR_SBCF_EF_SHIFT                 6u
#define FR_CHIERFR_SBCF_EF_WIDTH                 1u
#define FR_CHIERFR_SBCF_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_SBCF_EF_SHIFT))&FR_CHIERFR_SBCF_EF_MASK)
#define FR_CHIERFR_LCK_EF_MASK                   0x100u
#define FR_CHIERFR_LCK_EF_SHIFT                  8u
#define FR_CHIERFR_LCK_EF_WIDTH                  1u
#define FR_CHIERFR_LCK_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_LCK_EF_SHIFT))&FR_CHIERFR_LCK_EF_MASK)
#define FR_CHIERFR_MBU_EF_MASK                   0x200u
#define FR_CHIERFR_MBU_EF_SHIFT                  9u
#define FR_CHIERFR_MBU_EF_WIDTH                  1u
#define FR_CHIERFR_MBU_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_MBU_EF_SHIFT))&FR_CHIERFR_MBU_EF_MASK)
#define FR_CHIERFR_MBS_EF_MASK                   0x400u
#define FR_CHIERFR_MBS_EF_SHIFT                  10u
#define FR_CHIERFR_MBS_EF_WIDTH                  1u
#define FR_CHIERFR_MBS_EF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_MBS_EF_SHIFT))&FR_CHIERFR_MBS_EF_MASK)
#define FR_CHIERFR_FOVA_EF_MASK                  0x800u
#define FR_CHIERFR_FOVA_EF_SHIFT                 11u
#define FR_CHIERFR_FOVA_EF_WIDTH                 1u
#define FR_CHIERFR_FOVA_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FOVA_EF_SHIFT))&FR_CHIERFR_FOVA_EF_MASK)
#define FR_CHIERFR_FOVB_EF_MASK                  0x1000u
#define FR_CHIERFR_FOVB_EF_SHIFT                 12u
#define FR_CHIERFR_FOVB_EF_WIDTH                 1u
#define FR_CHIERFR_FOVB_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FOVB_EF_SHIFT))&FR_CHIERFR_FOVB_EF_MASK)
#define FR_CHIERFR_PCMI_EF_MASK                  0x2000u
#define FR_CHIERFR_PCMI_EF_SHIFT                 13u
#define FR_CHIERFR_PCMI_EF_WIDTH                 1u
#define FR_CHIERFR_PCMI_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_PCMI_EF_SHIFT))&FR_CHIERFR_PCMI_EF_MASK)
#define FR_CHIERFR_FRLA_EF_MASK                  0x4000u
#define FR_CHIERFR_FRLA_EF_SHIFT                 14u
#define FR_CHIERFR_FRLA_EF_WIDTH                 1u
#define FR_CHIERFR_FRLA_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FRLA_EF_SHIFT))&FR_CHIERFR_FRLA_EF_MASK)
#define FR_CHIERFR_FRLB_EF_MASK                  0x8000u
#define FR_CHIERFR_FRLB_EF_SHIFT                 15u
#define FR_CHIERFR_FRLB_EF_WIDTH                 1u
#define FR_CHIERFR_FRLB_EF(x)                    (((uint16_t)(((uint16_t)(x))<<FR_CHIERFR_FRLB_EF_SHIFT))&FR_CHIERFR_FRLB_EF_MASK)
/* MBIVEC Bit Fields */
#define FR_MBIVEC_RBIVEC_MASK                    0x7Fu
#define FR_MBIVEC_RBIVEC_SHIFT                   0u
#define FR_MBIVEC_RBIVEC_WIDTH                   7u
#define FR_MBIVEC_RBIVEC(x)                      (((uint16_t)(((uint16_t)(x))<<FR_MBIVEC_RBIVEC_SHIFT))&FR_MBIVEC_RBIVEC_MASK)
#define FR_MBIVEC_TBIVEC_MASK                    0x7F00u
#define FR_MBIVEC_TBIVEC_SHIFT                   8u
#define FR_MBIVEC_TBIVEC_WIDTH                   7u
#define FR_MBIVEC_TBIVEC(x)                      (((uint16_t)(((uint16_t)(x))<<FR_MBIVEC_TBIVEC_SHIFT))&FR_MBIVEC_TBIVEC_MASK)
/* CASERCR Bit Fields */
#define FR_CASERCR_CHAERSCNT_MASK                0xFFFFu
#define FR_CASERCR_CHAERSCNT_SHIFT               0u
#define FR_CASERCR_CHAERSCNT_WIDTH               16u
#define FR_CASERCR_CHAERSCNT(x)                  (((uint16_t)(((uint16_t)(x))<<FR_CASERCR_CHAERSCNT_SHIFT))&FR_CASERCR_CHAERSCNT_MASK)
/* CBSERCR Bit Fields */
#define FR_CBSERCR_CHBERSCNT_MASK                0xFFFFu
#define FR_CBSERCR_CHBERSCNT_SHIFT               0u
#define FR_CBSERCR_CHBERSCNT_WIDTH               16u
#define FR_CBSERCR_CHBERSCNT(x)                  (((uint16_t)(((uint16_t)(x))<<FR_CBSERCR_CHBERSCNT_SHIFT))&FR_CBSERCR_CHBERSCNT_MASK)
/* PSR0 Bit Fields */
#define FR_PSR0_WAKEUPSTATUS_MASK                0x7u
#define FR_PSR0_WAKEUPSTATUS_SHIFT               0u
#define FR_PSR0_WAKEUPSTATUS_WIDTH               3u
#define FR_PSR0_WAKEUPSTATUS(x)                  (((uint16_t)(((uint16_t)(x))<<FR_PSR0_WAKEUPSTATUS_SHIFT))&FR_PSR0_WAKEUPSTATUS_MASK)
#define FR_PSR0_STARTUPSTATE_MASK                0xF0u
#define FR_PSR0_STARTUPSTATE_SHIFT               4u
#define FR_PSR0_STARTUPSTATE_WIDTH               4u
#define FR_PSR0_STARTUPSTATE(x)                  (((uint16_t)(((uint16_t)(x))<<FR_PSR0_STARTUPSTATE_SHIFT))&FR_PSR0_STARTUPSTATE_MASK)
#define FR_PSR0_PROTSTATE_MASK                   0x700u
#define FR_PSR0_PROTSTATE_SHIFT                  8u
#define FR_PSR0_PROTSTATE_WIDTH                  3u
#define FR_PSR0_PROTSTATE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_PSR0_PROTSTATE_SHIFT))&FR_PSR0_PROTSTATE_MASK)
#define FR_PSR0_SLOTMODE_MASK                    0x3000u
#define FR_PSR0_SLOTMODE_SHIFT                   12u
#define FR_PSR0_SLOTMODE_WIDTH                   2u
#define FR_PSR0_SLOTMODE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PSR0_SLOTMODE_SHIFT))&FR_PSR0_SLOTMODE_MASK)
#define FR_PSR0_ERRMODE_MASK                     0xC000u
#define FR_PSR0_ERRMODE_SHIFT                    14u
#define FR_PSR0_ERRMODE_WIDTH                    2u
#define FR_PSR0_ERRMODE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PSR0_ERRMODE_SHIFT))&FR_PSR0_ERRMODE_MASK)
/* PSR1 Bit Fields */
#define FR_PSR1_APTAC_MASK                       0x1Fu
#define FR_PSR1_APTAC_SHIFT                      0u
#define FR_PSR1_APTAC_WIDTH                      5u
#define FR_PSR1_APTAC(x)                         (((uint16_t)(((uint16_t)(x))<<FR_PSR1_APTAC_SHIFT))&FR_PSR1_APTAC_MASK)
#define FR_PSR1_FRZ_MASK                         0x20u
#define FR_PSR1_FRZ_SHIFT                        5u
#define FR_PSR1_FRZ_WIDTH                        1u
#define FR_PSR1_FRZ(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR1_FRZ_SHIFT))&FR_PSR1_FRZ_MASK)
#define FR_PSR1_HHR_MASK                         0x40u
#define FR_PSR1_HHR_SHIFT                        6u
#define FR_PSR1_HHR_WIDTH                        1u
#define FR_PSR1_HHR(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR1_HHR_SHIFT))&FR_PSR1_HHR_MASK)
#define FR_PSR1_CPN_MASK                         0x80u
#define FR_PSR1_CPN_SHIFT                        7u
#define FR_PSR1_CPN_WIDTH                        1u
#define FR_PSR1_CPN(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR1_CPN_SHIFT))&FR_PSR1_CPN_MASK)
#define FR_PSR1_REMCSAT_MASK                     0x1F00u
#define FR_PSR1_REMCSAT_SHIFT                    8u
#define FR_PSR1_REMCSAT_WIDTH                    5u
#define FR_PSR1_REMCSAT(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PSR1_REMCSAT_SHIFT))&FR_PSR1_REMCSAT_MASK)
#define FR_PSR1_CSP_MASK                         0x4000u
#define FR_PSR1_CSP_SHIFT                        14u
#define FR_PSR1_CSP_WIDTH                        1u
#define FR_PSR1_CSP(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR1_CSP_SHIFT))&FR_PSR1_CSP_MASK)
#define FR_PSR1_CSAA_MASK                        0x8000u
#define FR_PSR1_CSAA_SHIFT                       15u
#define FR_PSR1_CSAA_WIDTH                       1u
#define FR_PSR1_CSAA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR1_CSAA_SHIFT))&FR_PSR1_CSAA_MASK)
/* PSR2 Bit Fields */
#define FR_PSR2_CKCORFCNT_MASK                   0xFu
#define FR_PSR2_CKCORFCNT_SHIFT                  0u
#define FR_PSR2_CKCORFCNT_WIDTH                  4u
#define FR_PSR2_CKCORFCNT(x)                     (((uint16_t)(((uint16_t)(x))<<FR_PSR2_CKCORFCNT_SHIFT))&FR_PSR2_CKCORFCNT_MASK)
#define FR_PSR2_MTA_MASK                         0x10u
#define FR_PSR2_MTA_SHIFT                        4u
#define FR_PSR2_MTA_WIDTH                        1u
#define FR_PSR2_MTA(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR2_MTA_SHIFT))&FR_PSR2_MTA_MASK)
#define FR_PSR2_SSEA_MASK                        0x20u
#define FR_PSR2_SSEA_SHIFT                       5u
#define FR_PSR2_SSEA_WIDTH                       1u
#define FR_PSR2_SSEA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_SSEA_SHIFT))&FR_PSR2_SSEA_MASK)
#define FR_PSR2_SBVA_MASK                        0x40u
#define FR_PSR2_SBVA_SHIFT                       6u
#define FR_PSR2_SBVA_WIDTH                       1u
#define FR_PSR2_SBVA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_SBVA_SHIFT))&FR_PSR2_SBVA_MASK)
#define FR_PSR2_STCA_MASK                        0x80u
#define FR_PSR2_STCA_SHIFT                       7u
#define FR_PSR2_STCA_WIDTH                       1u
#define FR_PSR2_STCA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_STCA_SHIFT))&FR_PSR2_STCA_MASK)
#define FR_PSR2_NSEA_MASK                        0x100u
#define FR_PSR2_NSEA_SHIFT                       8u
#define FR_PSR2_NSEA_WIDTH                       1u
#define FR_PSR2_NSEA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_NSEA_SHIFT))&FR_PSR2_NSEA_MASK)
#define FR_PSR2_NBVA_MASK                        0x200u
#define FR_PSR2_NBVA_SHIFT                       9u
#define FR_PSR2_NBVA_WIDTH                       1u
#define FR_PSR2_NBVA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_NBVA_SHIFT))&FR_PSR2_NBVA_MASK)
#define FR_PSR2_MTB_MASK                         0x400u
#define FR_PSR2_MTB_SHIFT                        10u
#define FR_PSR2_MTB_WIDTH                        1u
#define FR_PSR2_MTB(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR2_MTB_SHIFT))&FR_PSR2_MTB_MASK)
#define FR_PSR2_SSEB_MASK                        0x800u
#define FR_PSR2_SSEB_SHIFT                       11u
#define FR_PSR2_SSEB_WIDTH                       1u
#define FR_PSR2_SSEB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_SSEB_SHIFT))&FR_PSR2_SSEB_MASK)
#define FR_PSR2_SBVB_MASK                        0x1000u
#define FR_PSR2_SBVB_SHIFT                       12u
#define FR_PSR2_SBVB_WIDTH                       1u
#define FR_PSR2_SBVB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_SBVB_SHIFT))&FR_PSR2_SBVB_MASK)
#define FR_PSR2_STCB_MASK                        0x2000u
#define FR_PSR2_STCB_SHIFT                       13u
#define FR_PSR2_STCB_WIDTH                       1u
#define FR_PSR2_STCB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_STCB_SHIFT))&FR_PSR2_STCB_MASK)
#define FR_PSR2_NSEB_MASK                        0x4000u
#define FR_PSR2_NSEB_SHIFT                       14u
#define FR_PSR2_NSEB_WIDTH                       1u
#define FR_PSR2_NSEB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_NSEB_SHIFT))&FR_PSR2_NSEB_MASK)
#define FR_PSR2_NBVB_MASK                        0x8000u
#define FR_PSR2_NBVB_SHIFT                       15u
#define FR_PSR2_NBVB_WIDTH                       1u
#define FR_PSR2_NBVB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR2_NBVB_SHIFT))&FR_PSR2_NBVB_MASK)
/* PSR3 Bit Fields */
#define FR_PSR3_AVFA_MASK                        0x1u
#define FR_PSR3_AVFA_SHIFT                       0u
#define FR_PSR3_AVFA_WIDTH                       1u
#define FR_PSR3_AVFA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_AVFA_SHIFT))&FR_PSR3_AVFA_MASK)
#define FR_PSR3_ASEA_MASK                        0x2u
#define FR_PSR3_ASEA_SHIFT                       1u
#define FR_PSR3_ASEA_WIDTH                       1u
#define FR_PSR3_ASEA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ASEA_SHIFT))&FR_PSR3_ASEA_MASK)
#define FR_PSR3_ACEA_MASK                        0x4u
#define FR_PSR3_ACEA_SHIFT                       2u
#define FR_PSR3_ACEA_WIDTH                       1u
#define FR_PSR3_ACEA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ACEA_SHIFT))&FR_PSR3_ACEA_MASK)
#define FR_PSR3_AACA_MASK                        0x8u
#define FR_PSR3_AACA_SHIFT                       3u
#define FR_PSR3_AACA_WIDTH                       1u
#define FR_PSR3_AACA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_AACA_SHIFT))&FR_PSR3_AACA_MASK)
#define FR_PSR3_ABVA_MASK                        0x10u
#define FR_PSR3_ABVA_SHIFT                       4u
#define FR_PSR3_ABVA_WIDTH                       1u
#define FR_PSR3_ABVA(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ABVA_SHIFT))&FR_PSR3_ABVA_MASK)
#define FR_PSR3_WUA_MASK                         0x20u
#define FR_PSR3_WUA_SHIFT                        5u
#define FR_PSR3_WUA_WIDTH                        1u
#define FR_PSR3_WUA(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR3_WUA_SHIFT))&FR_PSR3_WUA_MASK)
#define FR_PSR3_AVFB_MASK                        0x100u
#define FR_PSR3_AVFB_SHIFT                       8u
#define FR_PSR3_AVFB_WIDTH                       1u
#define FR_PSR3_AVFB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_AVFB_SHIFT))&FR_PSR3_AVFB_MASK)
#define FR_PSR3_ASEB_MASK                        0x200u
#define FR_PSR3_ASEB_SHIFT                       9u
#define FR_PSR3_ASEB_WIDTH                       1u
#define FR_PSR3_ASEB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ASEB_SHIFT))&FR_PSR3_ASEB_MASK)
#define FR_PSR3_ACEB_MASK                        0x400u
#define FR_PSR3_ACEB_SHIFT                       10u
#define FR_PSR3_ACEB_WIDTH                       1u
#define FR_PSR3_ACEB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ACEB_SHIFT))&FR_PSR3_ACEB_MASK)
#define FR_PSR3_AACB_MASK                        0x800u
#define FR_PSR3_AACB_SHIFT                       11u
#define FR_PSR3_AACB_WIDTH                       1u
#define FR_PSR3_AACB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_AACB_SHIFT))&FR_PSR3_AACB_MASK)
#define FR_PSR3_ABVB_MASK                        0x1000u
#define FR_PSR3_ABVB_SHIFT                       12u
#define FR_PSR3_ABVB_WIDTH                       1u
#define FR_PSR3_ABVB(x)                          (((uint16_t)(((uint16_t)(x))<<FR_PSR3_ABVB_SHIFT))&FR_PSR3_ABVB_MASK)
#define FR_PSR3_WUB_MASK                         0x2000u
#define FR_PSR3_WUB_SHIFT                        13u
#define FR_PSR3_WUB_WIDTH                        1u
#define FR_PSR3_WUB(x)                           (((uint16_t)(((uint16_t)(x))<<FR_PSR3_WUB_SHIFT))&FR_PSR3_WUB_MASK)
/* MTCTR Bit Fields */
#define FR_MTCTR_MTCT_MASK                       0x3FFFu
#define FR_MTCTR_MTCT_SHIFT                      0u
#define FR_MTCTR_MTCT_WIDTH                      14u
#define FR_MTCTR_MTCT(x)                         (((uint16_t)(((uint16_t)(x))<<FR_MTCTR_MTCT_SHIFT))&FR_MTCTR_MTCT_MASK)
/* CYCTR Bit Fields */
#define FR_CYCTR_CYCCNT_MASK                     0x3Fu
#define FR_CYCTR_CYCCNT_SHIFT                    0u
#define FR_CYCTR_CYCCNT_WIDTH                    6u
#define FR_CYCTR_CYCCNT(x)                       (((uint16_t)(((uint16_t)(x))<<FR_CYCTR_CYCCNT_SHIFT))&FR_CYCTR_CYCCNT_MASK)
/* SLTCTAR Bit Fields */
#define FR_SLTCTAR_SLOTCNTA_MASK                 0x7FFu
#define FR_SLTCTAR_SLOTCNTA_SHIFT                0u
#define FR_SLTCTAR_SLOTCNTA_WIDTH                11u
#define FR_SLTCTAR_SLOTCNTA(x)                   (((uint16_t)(((uint16_t)(x))<<FR_SLTCTAR_SLOTCNTA_SHIFT))&FR_SLTCTAR_SLOTCNTA_MASK)
/* SLTCTBR Bit Fields */
#define FR_SLTCTBR_SLOTCNTB_MASK                 0x7FFu
#define FR_SLTCTBR_SLOTCNTB_SHIFT                0u
#define FR_SLTCTBR_SLOTCNTB_WIDTH                11u
#define FR_SLTCTBR_SLOTCNTB(x)                   (((uint16_t)(((uint16_t)(x))<<FR_SLTCTBR_SLOTCNTB_SHIFT))&FR_SLTCTBR_SLOTCNTB_MASK)
/* RTCORVR Bit Fields */
#define FR_RTCORVR_RATECORR_MASK                 0xFFFFu
#define FR_RTCORVR_RATECORR_SHIFT                0u
#define FR_RTCORVR_RATECORR_WIDTH                16u
#define FR_RTCORVR_RATECORR(x)                   (((uint16_t)(((uint16_t)(x))<<FR_RTCORVR_RATECORR_SHIFT))&FR_RTCORVR_RATECORR_MASK)
/* OFCORVR Bit Fields */
#define FR_OFCORVR_OFFSETCORR_MASK               0xFFFFu
#define FR_OFCORVR_OFFSETCORR_SHIFT              0u
#define FR_OFCORVR_OFFSETCORR_WIDTH              16u
#define FR_OFCORVR_OFFSETCORR(x)                 (((uint16_t)(((uint16_t)(x))<<FR_OFCORVR_OFFSETCORR_SHIFT))&FR_OFCORVR_OFFSETCORR_MASK)
/* CIFR Bit Fields */
#define FR_CIFR_TBIF_MASK                        0x1u
#define FR_CIFR_TBIF_SHIFT                       0u
#define FR_CIFR_TBIF_WIDTH                       1u
#define FR_CIFR_TBIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CIFR_TBIF_SHIFT))&FR_CIFR_TBIF_MASK)
#define FR_CIFR_RBIF_MASK                        0x2u
#define FR_CIFR_RBIF_SHIFT                       1u
#define FR_CIFR_RBIF_WIDTH                       1u
#define FR_CIFR_RBIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CIFR_RBIF_SHIFT))&FR_CIFR_RBIF_MASK)
#define FR_CIFR_FAFAIF_MASK                      0x4u
#define FR_CIFR_FAFAIF_SHIFT                     2u
#define FR_CIFR_FAFAIF_WIDTH                     1u
#define FR_CIFR_FAFAIF(x)                        (((uint16_t)(((uint16_t)(x))<<FR_CIFR_FAFAIF_SHIFT))&FR_CIFR_FAFAIF_MASK)
#define FR_CIFR_FAFBIF_MASK                      0x8u
#define FR_CIFR_FAFBIF_SHIFT                     3u
#define FR_CIFR_FAFBIF_WIDTH                     1u
#define FR_CIFR_FAFBIF(x)                        (((uint16_t)(((uint16_t)(x))<<FR_CIFR_FAFBIF_SHIFT))&FR_CIFR_FAFBIF_MASK)
#define FR_CIFR_WUPIF_MASK                       0x10u
#define FR_CIFR_WUPIF_SHIFT                      4u
#define FR_CIFR_WUPIF_WIDTH                      1u
#define FR_CIFR_WUPIF(x)                         (((uint16_t)(((uint16_t)(x))<<FR_CIFR_WUPIF_SHIFT))&FR_CIFR_WUPIF_MASK)
#define FR_CIFR_CHIF_MASK                        0x20u
#define FR_CIFR_CHIF_SHIFT                       5u
#define FR_CIFR_CHIF_WIDTH                       1u
#define FR_CIFR_CHIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CIFR_CHIF_SHIFT))&FR_CIFR_CHIF_MASK)
#define FR_CIFR_PRIF_MASK                        0x40u
#define FR_CIFR_PRIF_SHIFT                       6u
#define FR_CIFR_PRIF_WIDTH                       1u
#define FR_CIFR_PRIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CIFR_PRIF_SHIFT))&FR_CIFR_PRIF_MASK)
#define FR_CIFR_MIF_MASK                         0x80u
#define FR_CIFR_MIF_SHIFT                        7u
#define FR_CIFR_MIF_WIDTH                        1u
#define FR_CIFR_MIF(x)                           (((uint16_t)(((uint16_t)(x))<<FR_CIFR_MIF_SHIFT))&FR_CIFR_MIF_MASK)
/* SYMATOR Bit Fields */
#define FR_SYMATOR_TIMEOUT_MASK                  0xFFu
#define FR_SYMATOR_TIMEOUT_SHIFT                 0u
#define FR_SYMATOR_TIMEOUT_WIDTH                 8u
#define FR_SYMATOR_TIMEOUT(x)                    (((uint16_t)(((uint16_t)(x))<<FR_SYMATOR_TIMEOUT_SHIFT))&FR_SYMATOR_TIMEOUT_MASK)
/* SFCNTR Bit Fields */
#define FR_SFCNTR_SFODA_MASK                     0xFu
#define FR_SFCNTR_SFODA_SHIFT                    0u
#define FR_SFCNTR_SFODA_WIDTH                    4u
#define FR_SFCNTR_SFODA(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFCNTR_SFODA_SHIFT))&FR_SFCNTR_SFODA_MASK)
#define FR_SFCNTR_SFODB_MASK                     0xF0u
#define FR_SFCNTR_SFODB_SHIFT                    4u
#define FR_SFCNTR_SFODB_WIDTH                    4u
#define FR_SFCNTR_SFODB(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFCNTR_SFODB_SHIFT))&FR_SFCNTR_SFODB_MASK)
#define FR_SFCNTR_SFEVA_MASK                     0xF00u
#define FR_SFCNTR_SFEVA_SHIFT                    8u
#define FR_SFCNTR_SFEVA_WIDTH                    4u
#define FR_SFCNTR_SFEVA(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFCNTR_SFEVA_SHIFT))&FR_SFCNTR_SFEVA_MASK)
#define FR_SFCNTR_SFEVB_MASK                     0xF000u
#define FR_SFCNTR_SFEVB_SHIFT                    12u
#define FR_SFCNTR_SFEVB_WIDTH                    4u
#define FR_SFCNTR_SFEVB(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFCNTR_SFEVB_SHIFT))&FR_SFCNTR_SFEVB_MASK)
/* SFTOR Bit Fields */
#define FR_SFTOR_SFT_OFFSET_MASK                 0xFFFEu
#define FR_SFTOR_SFT_OFFSET_SHIFT                1u
#define FR_SFTOR_SFT_OFFSET_WIDTH                15u
#define FR_SFTOR_SFT_OFFSET(x)                   (((uint16_t)(((uint16_t)(x))<<FR_SFTOR_SFT_OFFSET_SHIFT))&FR_SFTOR_SFT_OFFSET_MASK)
/* SFTCCSR Bit Fields */
#define FR_SFTCCSR_SIDEN_MASK                    0x1u
#define FR_SFTCCSR_SIDEN_SHIFT                   0u
#define FR_SFTCCSR_SIDEN_WIDTH                   1u
#define FR_SFTCCSR_SIDEN(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_SIDEN_SHIFT))&FR_SFTCCSR_SIDEN_MASK)
#define FR_SFTCCSR_SDVEN_MASK                    0x2u
#define FR_SFTCCSR_SDVEN_SHIFT                   1u
#define FR_SFTCCSR_SDVEN_WIDTH                   1u
#define FR_SFTCCSR_SDVEN(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_SDVEN_SHIFT))&FR_SFTCCSR_SDVEN_MASK)
#define FR_SFTCCSR_OPT_MASK                      0x4u
#define FR_SFTCCSR_OPT_SHIFT                     2u
#define FR_SFTCCSR_OPT_WIDTH                     1u
#define FR_SFTCCSR_OPT(x)                        (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_OPT_SHIFT))&FR_SFTCCSR_OPT_MASK)
#define FR_SFTCCSR_OVAL_MASK                     0x10u
#define FR_SFTCCSR_OVAL_SHIFT                    4u
#define FR_SFTCCSR_OVAL_WIDTH                    1u
#define FR_SFTCCSR_OVAL(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_OVAL_SHIFT))&FR_SFTCCSR_OVAL_MASK)
#define FR_SFTCCSR_EVAL_MASK                     0x20u
#define FR_SFTCCSR_EVAL_SHIFT                    5u
#define FR_SFTCCSR_EVAL_WIDTH                    1u
#define FR_SFTCCSR_EVAL(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_EVAL_SHIFT))&FR_SFTCCSR_EVAL_MASK)
#define FR_SFTCCSR_OLKS_MASK                     0x40u
#define FR_SFTCCSR_OLKS_SHIFT                    6u
#define FR_SFTCCSR_OLKS_WIDTH                    1u
#define FR_SFTCCSR_OLKS(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_OLKS_SHIFT))&FR_SFTCCSR_OLKS_MASK)
#define FR_SFTCCSR_ELKS_MASK                     0x80u
#define FR_SFTCCSR_ELKS_SHIFT                    7u
#define FR_SFTCCSR_ELKS_WIDTH                    1u
#define FR_SFTCCSR_ELKS(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_ELKS_SHIFT))&FR_SFTCCSR_ELKS_MASK)
#define FR_SFTCCSR_CYCNUM_MASK                   0x3F00u
#define FR_SFTCCSR_CYCNUM_SHIFT                  8u
#define FR_SFTCCSR_CYCNUM_WIDTH                  6u
#define FR_SFTCCSR_CYCNUM(x)                     (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_CYCNUM_SHIFT))&FR_SFTCCSR_CYCNUM_MASK)
#define FR_SFTCCSR_OLKT_MASK                     0x4000u
#define FR_SFTCCSR_OLKT_SHIFT                    14u
#define FR_SFTCCSR_OLKT_WIDTH                    1u
#define FR_SFTCCSR_OLKT(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_OLKT_SHIFT))&FR_SFTCCSR_OLKT_MASK)
#define FR_SFTCCSR_ELKT_MASK                     0x8000u
#define FR_SFTCCSR_ELKT_SHIFT                    15u
#define FR_SFTCCSR_ELKT_WIDTH                    1u
#define FR_SFTCCSR_ELKT(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SFTCCSR_ELKT_SHIFT))&FR_SFTCCSR_ELKT_MASK)
/* SFIDRFR Bit Fields */
#define FR_SFIDRFR_SYNFRID_MASK                  0x3FFu
#define FR_SFIDRFR_SYNFRID_SHIFT                 0u
#define FR_SFIDRFR_SYNFRID_WIDTH                 10u
#define FR_SFIDRFR_SYNFRID(x)                    (((uint16_t)(((uint16_t)(x))<<FR_SFIDRFR_SYNFRID_SHIFT))&FR_SFIDRFR_SYNFRID_MASK)
/* SFIDAFVR Bit Fields */
#define FR_SFIDAFVR_FVAL_MASK                    0x3FFu
#define FR_SFIDAFVR_FVAL_SHIFT                   0u
#define FR_SFIDAFVR_FVAL_WIDTH                   10u
#define FR_SFIDAFVR_FVAL(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SFIDAFVR_FVAL_SHIFT))&FR_SFIDAFVR_FVAL_MASK)
/* SFIDAFMR Bit Fields */
#define FR_SFIDAFMR_FMSK_MASK                    0x3FFu
#define FR_SFIDAFMR_FMSK_SHIFT                   0u
#define FR_SFIDAFMR_FMSK_WIDTH                   10u
#define FR_SFIDAFMR_FMSK(x)                      (((uint16_t)(((uint16_t)(x))<<FR_SFIDAFMR_FMSK_SHIFT))&FR_SFIDAFMR_FMSK_MASK)
/* NMVR Bit Fields */
#define FR_NMVR_NMVP_MASK                        0xFFFFu
#define FR_NMVR_NMVP_SHIFT                       0u
#define FR_NMVR_NMVP_WIDTH                       16u
#define FR_NMVR_NMVP(x)                          (((uint16_t)(((uint16_t)(x))<<FR_NMVR_NMVP_SHIFT))&FR_NMVR_NMVP_MASK)
/* NMVLR Bit Fields */
#define FR_NMVLR_NMVL_MASK                       0xFu
#define FR_NMVLR_NMVL_SHIFT                      0u
#define FR_NMVLR_NMVL_WIDTH                      4u
#define FR_NMVLR_NMVL(x)                         (((uint16_t)(((uint16_t)(x))<<FR_NMVLR_NMVL_SHIFT))&FR_NMVLR_NMVL_MASK)
/* TICCR Bit Fields */
#define FR_TICCR_T1ST_MASK                       0x1u
#define FR_TICCR_T1ST_SHIFT                      0u
#define FR_TICCR_T1ST_WIDTH                      1u
#define FR_TICCR_T1ST(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T1ST_SHIFT))&FR_TICCR_T1ST_MASK)
#define FR_TICCR_T1TR_MASK                       0x2u
#define FR_TICCR_T1TR_SHIFT                      1u
#define FR_TICCR_T1TR_WIDTH                      1u
#define FR_TICCR_T1TR(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T1TR_SHIFT))&FR_TICCR_T1TR_MASK)
#define FR_TICCR_T1SP_MASK                       0x4u
#define FR_TICCR_T1SP_SHIFT                      2u
#define FR_TICCR_T1SP_WIDTH                      1u
#define FR_TICCR_T1SP(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T1SP_SHIFT))&FR_TICCR_T1SP_MASK)
#define FR_TICCR_T1_REP_MASK                     0x10u
#define FR_TICCR_T1_REP_SHIFT                    4u
#define FR_TICCR_T1_REP_WIDTH                    1u
#define FR_TICCR_T1_REP(x)                       (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T1_REP_SHIFT))&FR_TICCR_T1_REP_MASK)
#define FR_TICCR_T2ST_MASK                       0x100u
#define FR_TICCR_T2ST_SHIFT                      8u
#define FR_TICCR_T2ST_WIDTH                      1u
#define FR_TICCR_T2ST(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2ST_SHIFT))&FR_TICCR_T2ST_MASK)
#define FR_TICCR_T2TR_MASK                       0x200u
#define FR_TICCR_T2TR_SHIFT                      9u
#define FR_TICCR_T2TR_WIDTH                      1u
#define FR_TICCR_T2TR(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2TR_SHIFT))&FR_TICCR_T2TR_MASK)
#define FR_TICCR_T2SP_MASK                       0x400u
#define FR_TICCR_T2SP_SHIFT                      10u
#define FR_TICCR_T2SP_WIDTH                      1u
#define FR_TICCR_T2SP(x)                         (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2SP_SHIFT))&FR_TICCR_T2SP_MASK)
#define FR_TICCR_T2_REP_MASK                     0x1000u
#define FR_TICCR_T2_REP_SHIFT                    12u
#define FR_TICCR_T2_REP_WIDTH                    1u
#define FR_TICCR_T2_REP(x)                       (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2_REP_SHIFT))&FR_TICCR_T2_REP_MASK)
#define FR_TICCR_T2_CFG_MASK                     0x2000u
#define FR_TICCR_T2_CFG_SHIFT                    13u
#define FR_TICCR_T2_CFG_WIDTH                    1u
#define FR_TICCR_T2_CFG(x)                       (((uint16_t)(((uint16_t)(x))<<FR_TICCR_T2_CFG_SHIFT))&FR_TICCR_T2_CFG_MASK)
/* TI1CYSR Bit Fields */
#define FR_TI1CYSR_T1_CYC_MSK_MASK               0x3Fu
#define FR_TI1CYSR_T1_CYC_MSK_SHIFT              0u
#define FR_TI1CYSR_T1_CYC_MSK_WIDTH              6u
#define FR_TI1CYSR_T1_CYC_MSK(x)                 (((uint16_t)(((uint16_t)(x))<<FR_TI1CYSR_T1_CYC_MSK_SHIFT))&FR_TI1CYSR_T1_CYC_MSK_MASK)
#define FR_TI1CYSR_T1_CYC_VAL_MASK               0x3F00u
#define FR_TI1CYSR_T1_CYC_VAL_SHIFT              8u
#define FR_TI1CYSR_T1_CYC_VAL_WIDTH              6u
#define FR_TI1CYSR_T1_CYC_VAL(x)                 (((uint16_t)(((uint16_t)(x))<<FR_TI1CYSR_T1_CYC_VAL_SHIFT))&FR_TI1CYSR_T1_CYC_VAL_MASK)
/* TI1MTOR Bit Fields */
#define FR_TI1MTOR_T1_MTOFFSET_MASK              0x3FFFu
#define FR_TI1MTOR_T1_MTOFFSET_SHIFT             0u
#define FR_TI1MTOR_T1_MTOFFSET_WIDTH             14u
#define FR_TI1MTOR_T1_MTOFFSET(x)                (((uint16_t)(((uint16_t)(x))<<FR_TI1MTOR_T1_MTOFFSET_SHIFT))&FR_TI1MTOR_T1_MTOFFSET_MASK)
/* TI2CR0_ABS Bit Fields */
#define FR_TI2CR0_ABS_T2CYCMSK_MASK              0x3Fu
#define FR_TI2CR0_ABS_T2CYCMSK_SHIFT             0u
#define FR_TI2CR0_ABS_T2CYCMSK_WIDTH             6u
#define FR_TI2CR0_ABS_T2CYCMSK(x)                (((uint16_t)(((uint16_t)(x))<<FR_TI2CR0_ABS_T2CYCMSK_SHIFT))&FR_TI2CR0_ABS_T2CYCMSK_MASK)
#define FR_TI2CR0_ABS_T2CYCVAL_MASK              0x3F00u
#define FR_TI2CR0_ABS_T2CYCVAL_SHIFT             8u
#define FR_TI2CR0_ABS_T2CYCVAL_WIDTH             6u
#define FR_TI2CR0_ABS_T2CYCVAL(x)                (((uint16_t)(((uint16_t)(x))<<FR_TI2CR0_ABS_T2CYCVAL_SHIFT))&FR_TI2CR0_ABS_T2CYCVAL_MASK)
/* TI2CR0_REL Bit Fields */
#define FR_TI2CR0_REL_T2MTCNT_MASK               0xFFFFu
#define FR_TI2CR0_REL_T2MTCNT_SHIFT              0u
#define FR_TI2CR0_REL_T2MTCNT_WIDTH              16u
#define FR_TI2CR0_REL_T2MTCNT(x)                 (((uint16_t)(((uint16_t)(x))<<FR_TI2CR0_REL_T2MTCNT_SHIFT))&FR_TI2CR0_REL_T2MTCNT_MASK)
/* TI2CR1_ABS Bit Fields */
#define FR_TI2CR1_ABS_T2MOFF_MASK                0x3FFFu
#define FR_TI2CR1_ABS_T2MOFF_SHIFT               0u
#define FR_TI2CR1_ABS_T2MOFF_WIDTH               14u
#define FR_TI2CR1_ABS_T2MOFF(x)                  (((uint16_t)(((uint16_t)(x))<<FR_TI2CR1_ABS_T2MOFF_SHIFT))&FR_TI2CR1_ABS_T2MOFF_MASK)
/* TI2CR1_REL Bit Fields */
#define FR_TI2CR1_REL_T2MTCNT_MASK               0xFFFFu
#define FR_TI2CR1_REL_T2MTCNT_SHIFT              0u
#define FR_TI2CR1_REL_T2MTCNT_WIDTH              16u
#define FR_TI2CR1_REL_T2MTCNT(x)                 (((uint16_t)(((uint16_t)(x))<<FR_TI2CR1_REL_T2MTCNT_SHIFT))&FR_TI2CR1_REL_T2MTCNT_MASK)
/* SSSR Bit Fields */
#define FR_SSSR_SLOTNUMBER_MASK                  0x7FFu
#define FR_SSSR_SLOTNUMBER_SHIFT                 0u
#define FR_SSSR_SLOTNUMBER_WIDTH                 11u
#define FR_SSSR_SLOTNUMBER(x)                    (((uint16_t)(((uint16_t)(x))<<FR_SSSR_SLOTNUMBER_SHIFT))&FR_SSSR_SLOTNUMBER_MASK)
#define FR_SSSR_SEL_MASK                         0x3000u
#define FR_SSSR_SEL_SHIFT                        12u
#define FR_SSSR_SEL_WIDTH                        2u
#define FR_SSSR_SEL(x)                           (((uint16_t)(((uint16_t)(x))<<FR_SSSR_SEL_SHIFT))&FR_SSSR_SEL_MASK)
#define FR_SSSR_WMD_MASK                         0x8000u
#define FR_SSSR_WMD_SHIFT                        15u
#define FR_SSSR_WMD_WIDTH                        1u
#define FR_SSSR_WMD(x)                           (((uint16_t)(((uint16_t)(x))<<FR_SSSR_WMD_SHIFT))&FR_SSSR_WMD_MASK)
/* SSCCR Bit Fields */
#define FR_SSCCR_STATUSMASK_MASK                 0xFu
#define FR_SSCCR_STATUSMASK_SHIFT                0u
#define FR_SSCCR_STATUSMASK_WIDTH                4u
#define FR_SSCCR_STATUSMASK(x)                   (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_STATUSMASK_SHIFT))&FR_SSCCR_STATUSMASK_MASK)
#define FR_SSCCR_SUF_MASK                        0x10u
#define FR_SSCCR_SUF_SHIFT                       4u
#define FR_SSCCR_SUF_WIDTH                       1u
#define FR_SSCCR_SUF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_SUF_SHIFT))&FR_SSCCR_SUF_MASK)
#define FR_SSCCR_NUF_MASK                        0x20u
#define FR_SSCCR_NUF_SHIFT                       5u
#define FR_SSCCR_NUF_WIDTH                       1u
#define FR_SSCCR_NUF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_NUF_SHIFT))&FR_SSCCR_NUF_MASK)
#define FR_SSCCR_SYF_MASK                        0x40u
#define FR_SSCCR_SYF_SHIFT                       6u
#define FR_SSCCR_SYF_WIDTH                       1u
#define FR_SSCCR_SYF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_SYF_SHIFT))&FR_SSCCR_SYF_MASK)
#define FR_SSCCR_VFR_MASK                        0x80u
#define FR_SSCCR_VFR_SHIFT                       7u
#define FR_SSCCR_VFR_WIDTH                       1u
#define FR_SSCCR_VFR(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_VFR_SHIFT))&FR_SSCCR_VFR_MASK)
#define FR_SSCCR_MCY_MASK                        0x100u
#define FR_SSCCR_MCY_SHIFT                       8u
#define FR_SSCCR_MCY_WIDTH                       1u
#define FR_SSCCR_MCY(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_MCY_SHIFT))&FR_SSCCR_MCY_MASK)
#define FR_SSCCR_CNTCFG_MASK                     0x600u
#define FR_SSCCR_CNTCFG_SHIFT                    9u
#define FR_SSCCR_CNTCFG_WIDTH                    2u
#define FR_SSCCR_CNTCFG(x)                       (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_CNTCFG_SHIFT))&FR_SSCCR_CNTCFG_MASK)
#define FR_SSCCR_SEL_MASK                        0x3000u
#define FR_SSCCR_SEL_SHIFT                       12u
#define FR_SSCCR_SEL_WIDTH                       2u
#define FR_SSCCR_SEL(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_SEL_SHIFT))&FR_SSCCR_SEL_MASK)
#define FR_SSCCR_WMD_MASK                        0x8000u
#define FR_SSCCR_WMD_SHIFT                       15u
#define FR_SSCCR_WMD_WIDTH                       1u
#define FR_SSCCR_WMD(x)                          (((uint16_t)(((uint16_t)(x))<<FR_SSCCR_WMD_SHIFT))&FR_SSCCR_WMD_MASK)
/* SSR Bit Fields */
#define FR_SSR_TCA_MASK                          0x1u
#define FR_SSR_TCA_SHIFT                         0u
#define FR_SSR_TCA_WIDTH                         1u
#define FR_SSR_TCA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_TCA_SHIFT))&FR_SSR_TCA_MASK)
#define FR_SSR_BVA_MASK                          0x2u
#define FR_SSR_BVA_SHIFT                         1u
#define FR_SSR_BVA_WIDTH                         1u
#define FR_SSR_BVA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_BVA_SHIFT))&FR_SSR_BVA_MASK)
#define FR_SSR_CEA_MASK                          0x4u
#define FR_SSR_CEA_SHIFT                         2u
#define FR_SSR_CEA_WIDTH                         1u
#define FR_SSR_CEA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_CEA_SHIFT))&FR_SSR_CEA_MASK)
#define FR_SSR_SEA_MASK                          0x8u
#define FR_SSR_SEA_SHIFT                         3u
#define FR_SSR_SEA_WIDTH                         1u
#define FR_SSR_SEA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SEA_SHIFT))&FR_SSR_SEA_MASK)
#define FR_SSR_SUA_MASK                          0x10u
#define FR_SSR_SUA_SHIFT                         4u
#define FR_SSR_SUA_WIDTH                         1u
#define FR_SSR_SUA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SUA_SHIFT))&FR_SSR_SUA_MASK)
#define FR_SSR_NFA_MASK                          0x20u
#define FR_SSR_NFA_SHIFT                         5u
#define FR_SSR_NFA_WIDTH                         1u
#define FR_SSR_NFA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_NFA_SHIFT))&FR_SSR_NFA_MASK)
#define FR_SSR_SYA_MASK                          0x40u
#define FR_SSR_SYA_SHIFT                         6u
#define FR_SSR_SYA_WIDTH                         1u
#define FR_SSR_SYA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SYA_SHIFT))&FR_SSR_SYA_MASK)
#define FR_SSR_VFA_MASK                          0x80u
#define FR_SSR_VFA_SHIFT                         7u
#define FR_SSR_VFA_WIDTH                         1u
#define FR_SSR_VFA(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_VFA_SHIFT))&FR_SSR_VFA_MASK)
#define FR_SSR_TCB_MASK                          0x100u
#define FR_SSR_TCB_SHIFT                         8u
#define FR_SSR_TCB_WIDTH                         1u
#define FR_SSR_TCB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_TCB_SHIFT))&FR_SSR_TCB_MASK)
#define FR_SSR_BVB_MASK                          0x200u
#define FR_SSR_BVB_SHIFT                         9u
#define FR_SSR_BVB_WIDTH                         1u
#define FR_SSR_BVB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_BVB_SHIFT))&FR_SSR_BVB_MASK)
#define FR_SSR_CEB_MASK                          0x400u
#define FR_SSR_CEB_SHIFT                         10u
#define FR_SSR_CEB_WIDTH                         1u
#define FR_SSR_CEB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_CEB_SHIFT))&FR_SSR_CEB_MASK)
#define FR_SSR_SEB_MASK                          0x800u
#define FR_SSR_SEB_SHIFT                         11u
#define FR_SSR_SEB_WIDTH                         1u
#define FR_SSR_SEB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SEB_SHIFT))&FR_SSR_SEB_MASK)
#define FR_SSR_SUB_MASK                          0x1000u
#define FR_SSR_SUB_SHIFT                         12u
#define FR_SSR_SUB_WIDTH                         1u
#define FR_SSR_SUB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SUB_SHIFT))&FR_SSR_SUB_MASK)
#define FR_SSR_NFB_MASK                          0x2000u
#define FR_SSR_NFB_SHIFT                         13u
#define FR_SSR_NFB_WIDTH                         1u
#define FR_SSR_NFB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_NFB_SHIFT))&FR_SSR_NFB_MASK)
#define FR_SSR_SYB_MASK                          0x4000u
#define FR_SSR_SYB_SHIFT                         14u
#define FR_SSR_SYB_WIDTH                         1u
#define FR_SSR_SYB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_SYB_SHIFT))&FR_SSR_SYB_MASK)
#define FR_SSR_VFB_MASK                          0x8000u
#define FR_SSR_VFB_SHIFT                         15u
#define FR_SSR_VFB_WIDTH                         1u
#define FR_SSR_VFB(x)                            (((uint16_t)(((uint16_t)(x))<<FR_SSR_VFB_SHIFT))&FR_SSR_VFB_MASK)
/* SSCR Bit Fields */
#define FR_SSCR_SLOTSTATUSCNT_MASK               0xFFFFu
#define FR_SSCR_SLOTSTATUSCNT_SHIFT              0u
#define FR_SSCR_SLOTSTATUSCNT_WIDTH              16u
#define FR_SSCR_SLOTSTATUSCNT(x)                 (((uint16_t)(((uint16_t)(x))<<FR_SSCR_SLOTSTATUSCNT_SHIFT))&FR_SSCR_SLOTSTATUSCNT_MASK)
/* MTSACFR Bit Fields */
#define FR_MTSACFR_CYCCNTVAL_MASK                0x3Fu
#define FR_MTSACFR_CYCCNTVAL_SHIFT               0u
#define FR_MTSACFR_CYCCNTVAL_WIDTH               6u
#define FR_MTSACFR_CYCCNTVAL(x)                  (((uint16_t)(((uint16_t)(x))<<FR_MTSACFR_CYCCNTVAL_SHIFT))&FR_MTSACFR_CYCCNTVAL_MASK)
#define FR_MTSACFR_CYCCNTMSK_MASK                0x3F00u
#define FR_MTSACFR_CYCCNTMSK_SHIFT               8u
#define FR_MTSACFR_CYCCNTMSK_WIDTH               6u
#define FR_MTSACFR_CYCCNTMSK(x)                  (((uint16_t)(((uint16_t)(x))<<FR_MTSACFR_CYCCNTMSK_SHIFT))&FR_MTSACFR_CYCCNTMSK_MASK)
#define FR_MTSACFR_MTE_MASK                      0x8000u
#define FR_MTSACFR_MTE_SHIFT                     15u
#define FR_MTSACFR_MTE_WIDTH                     1u
#define FR_MTSACFR_MTE(x)                        (((uint16_t)(((uint16_t)(x))<<FR_MTSACFR_MTE_SHIFT))&FR_MTSACFR_MTE_MASK)
/* MTSBCFR Bit Fields */
#define FR_MTSBCFR_CYCCNTVAL_MASK                0x3Fu
#define FR_MTSBCFR_CYCCNTVAL_SHIFT               0u
#define FR_MTSBCFR_CYCCNTVAL_WIDTH               6u
#define FR_MTSBCFR_CYCCNTVAL(x)                  (((uint16_t)(((uint16_t)(x))<<FR_MTSBCFR_CYCCNTVAL_SHIFT))&FR_MTSBCFR_CYCCNTVAL_MASK)
#define FR_MTSBCFR_CYCCNTMSK_MASK                0x3F00u
#define FR_MTSBCFR_CYCCNTMSK_SHIFT               8u
#define FR_MTSBCFR_CYCCNTMSK_WIDTH               6u
#define FR_MTSBCFR_CYCCNTMSK(x)                  (((uint16_t)(((uint16_t)(x))<<FR_MTSBCFR_CYCCNTMSK_SHIFT))&FR_MTSBCFR_CYCCNTMSK_MASK)
#define FR_MTSBCFR_MTE_MASK                      0x8000u
#define FR_MTSBCFR_MTE_SHIFT                     15u
#define FR_MTSBCFR_MTE_WIDTH                     1u
#define FR_MTSBCFR_MTE(x)                        (((uint16_t)(((uint16_t)(x))<<FR_MTSBCFR_MTE_SHIFT))&FR_MTSBCFR_MTE_MASK)
/* RSBIR Bit Fields */
#define FR_RSBIR_RSBIDX_MASK                     0xFFu
#define FR_RSBIR_RSBIDX_SHIFT                    0u
#define FR_RSBIR_RSBIDX_WIDTH                    8u
#define FR_RSBIR_RSBIDX(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RSBIR_RSBIDX_SHIFT))&FR_RSBIR_RSBIDX_MASK)
#define FR_RSBIR_SEL_MASK                        0x3000u
#define FR_RSBIR_SEL_SHIFT                       12u
#define FR_RSBIR_SEL_WIDTH                       2u
#define FR_RSBIR_SEL(x)                          (((uint16_t)(((uint16_t)(x))<<FR_RSBIR_SEL_SHIFT))&FR_RSBIR_SEL_MASK)
#define FR_RSBIR_WMD_MASK                        0x8000u
#define FR_RSBIR_WMD_SHIFT                       15u
#define FR_RSBIR_WMD_WIDTH                       1u
#define FR_RSBIR_WMD(x)                          (((uint16_t)(((uint16_t)(x))<<FR_RSBIR_WMD_SHIFT))&FR_RSBIR_WMD_MASK)
/* RFWMSR Bit Fields */
#define FR_RFWMSR_SEL_MASK                       0x1u
#define FR_RFWMSR_SEL_SHIFT                      0u
#define FR_RFWMSR_SEL_WIDTH                      1u
#define FR_RFWMSR_SEL(x)                         (((uint16_t)(((uint16_t)(x))<<FR_RFWMSR_SEL_SHIFT))&FR_RFWMSR_SEL_MASK)
#define FR_RFWMSR_WM_MASK                        0xFF00u
#define FR_RFWMSR_WM_SHIFT                       8u
#define FR_RFWMSR_WM_WIDTH                       8u
#define FR_RFWMSR_WM(x)                          (((uint16_t)(((uint16_t)(x))<<FR_RFWMSR_WM_SHIFT))&FR_RFWMSR_WM_MASK)
/* RFSIR Bit Fields */
#define FR_RFSIR_SIDX_MASK                       0x3FFu
#define FR_RFSIR_SIDX_SHIFT                      0u
#define FR_RFSIR_SIDX_WIDTH                      10u
#define FR_RFSIR_SIDX(x)                         (((uint16_t)(((uint16_t)(x))<<FR_RFSIR_SIDX_SHIFT))&FR_RFSIR_SIDX_MASK)
/* RFDSR Bit Fields */
#define FR_RFDSR_ENTRY_SIZE_MASK                 0x7Fu
#define FR_RFDSR_ENTRY_SIZE_SHIFT                0u
#define FR_RFDSR_ENTRY_SIZE_WIDTH                7u
#define FR_RFDSR_ENTRY_SIZE(x)                   (((uint16_t)(((uint16_t)(x))<<FR_RFDSR_ENTRY_SIZE_SHIFT))&FR_RFDSR_ENTRY_SIZE_MASK)
#define FR_RFDSR_FIFO_DEPTH_MASK                 0xFF00u
#define FR_RFDSR_FIFO_DEPTH_SHIFT                8u
#define FR_RFDSR_FIFO_DEPTH_WIDTH                8u
#define FR_RFDSR_FIFO_DEPTH(x)                   (((uint16_t)(((uint16_t)(x))<<FR_RFDSR_FIFO_DEPTH_SHIFT))&FR_RFDSR_FIFO_DEPTH_MASK)
/* RFARIR Bit Fields */
#define FR_RFARIR_RDIDX_MASK                     0x3FFu
#define FR_RFARIR_RDIDX_SHIFT                    0u
#define FR_RFARIR_RDIDX_WIDTH                    10u
#define FR_RFARIR_RDIDX(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFARIR_RDIDX_SHIFT))&FR_RFARIR_RDIDX_MASK)
/* RFBRIR Bit Fields */
#define FR_RFBRIR_RDIDX_MASK                     0x3FFu
#define FR_RFBRIR_RDIDX_SHIFT                    0u
#define FR_RFBRIR_RDIDX_WIDTH                    10u
#define FR_RFBRIR_RDIDX(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFBRIR_RDIDX_SHIFT))&FR_RFBRIR_RDIDX_MASK)
/* RFMIDAFVR Bit Fields */
#define FR_RFMIDAFVR_MIDAFVAL_MASK               0xFFFFu
#define FR_RFMIDAFVR_MIDAFVAL_SHIFT              0u
#define FR_RFMIDAFVR_MIDAFVAL_WIDTH              16u
#define FR_RFMIDAFVR_MIDAFVAL(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFMIDAFVR_MIDAFVAL_SHIFT))&FR_RFMIDAFVR_MIDAFVAL_MASK)
/* RFMIDAFMR Bit Fields */
#define FR_RFMIDAFMR_MIDAFMSK_MASK               0xFFFFu
#define FR_RFMIDAFMR_MIDAFMSK_SHIFT              0u
#define FR_RFMIDAFMR_MIDAFMSK_WIDTH              16u
#define FR_RFMIDAFMR_MIDAFMSK(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFMIDAFMR_MIDAFMSK_SHIFT))&FR_RFMIDAFMR_MIDAFMSK_MASK)
/* RFFIDRFVR Bit Fields */
#define FR_RFFIDRFVR_FIDRFVAL_MASK               0x7FFu
#define FR_RFFIDRFVR_FIDRFVAL_SHIFT              0u
#define FR_RFFIDRFVR_FIDRFVAL_WIDTH              11u
#define FR_RFFIDRFVR_FIDRFVAL(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFFIDRFVR_FIDRFVAL_SHIFT))&FR_RFFIDRFVR_FIDRFVAL_MASK)
/* RFFIDRFMR Bit Fields */
#define FR_RFFIDRFMR_FIDRFMSK_MASK               0x7FFu
#define FR_RFFIDRFMR_FIDRFMSK_SHIFT              0u
#define FR_RFFIDRFMR_FIDRFMSK_WIDTH              11u
#define FR_RFFIDRFMR_FIDRFMSK(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFFIDRFMR_FIDRFMSK_SHIFT))&FR_RFFIDRFMR_FIDRFMSK_MASK)
/* RFRFCFR Bit Fields */
#define FR_RFRFCFR_SID_MASK                      0x7FFu
#define FR_RFRFCFR_SID_SHIFT                     0u
#define FR_RFRFCFR_SID_WIDTH                     11u
#define FR_RFRFCFR_SID(x)                        (((uint16_t)(((uint16_t)(x))<<FR_RFRFCFR_SID_SHIFT))&FR_RFRFCFR_SID_MASK)
#define FR_RFRFCFR_SEL_MASK                      0x3000u
#define FR_RFRFCFR_SEL_SHIFT                     12u
#define FR_RFRFCFR_SEL_WIDTH                     2u
#define FR_RFRFCFR_SEL(x)                        (((uint16_t)(((uint16_t)(x))<<FR_RFRFCFR_SEL_SHIFT))&FR_RFRFCFR_SEL_MASK)
#define FR_RFRFCFR_IBD_MASK                      0x4000u
#define FR_RFRFCFR_IBD_SHIFT                     14u
#define FR_RFRFCFR_IBD_WIDTH                     1u
#define FR_RFRFCFR_IBD(x)                        (((uint16_t)(((uint16_t)(x))<<FR_RFRFCFR_IBD_SHIFT))&FR_RFRFCFR_IBD_MASK)
#define FR_RFRFCFR_WMD_MASK                      0x8000u
#define FR_RFRFCFR_WMD_SHIFT                     15u
#define FR_RFRFCFR_WMD_WIDTH                     1u
#define FR_RFRFCFR_WMD(x)                        (((uint16_t)(((uint16_t)(x))<<FR_RFRFCFR_WMD_SHIFT))&FR_RFRFCFR_WMD_MASK)
/* RFRFCTR Bit Fields */
#define FR_RFRFCTR_F0EN_MASK                     0x1u
#define FR_RFRFCTR_F0EN_SHIFT                    0u
#define FR_RFRFCTR_F0EN_WIDTH                    1u
#define FR_RFRFCTR_F0EN(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F0EN_SHIFT))&FR_RFRFCTR_F0EN_MASK)
#define FR_RFRFCTR_F1EN_MASK                     0x2u
#define FR_RFRFCTR_F1EN_SHIFT                    1u
#define FR_RFRFCTR_F1EN_WIDTH                    1u
#define FR_RFRFCTR_F1EN(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F1EN_SHIFT))&FR_RFRFCTR_F1EN_MASK)
#define FR_RFRFCTR_F2EN_MASK                     0x4u
#define FR_RFRFCTR_F2EN_SHIFT                    2u
#define FR_RFRFCTR_F2EN_WIDTH                    1u
#define FR_RFRFCTR_F2EN(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F2EN_SHIFT))&FR_RFRFCTR_F2EN_MASK)
#define FR_RFRFCTR_F3EN_MASK                     0x8u
#define FR_RFRFCTR_F3EN_SHIFT                    3u
#define FR_RFRFCTR_F3EN_WIDTH                    1u
#define FR_RFRFCTR_F3EN(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F3EN_SHIFT))&FR_RFRFCTR_F3EN_MASK)
#define FR_RFRFCTR_F0MD_MASK                     0x100u
#define FR_RFRFCTR_F0MD_SHIFT                    8u
#define FR_RFRFCTR_F0MD_WIDTH                    1u
#define FR_RFRFCTR_F0MD(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F0MD_SHIFT))&FR_RFRFCTR_F0MD_MASK)
#define FR_RFRFCTR_F1MD_MASK                     0x200u
#define FR_RFRFCTR_F1MD_SHIFT                    9u
#define FR_RFRFCTR_F1MD_WIDTH                    1u
#define FR_RFRFCTR_F1MD(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F1MD_SHIFT))&FR_RFRFCTR_F1MD_MASK)
#define FR_RFRFCTR_F2MD_MASK                     0x400u
#define FR_RFRFCTR_F2MD_SHIFT                    10u
#define FR_RFRFCTR_F2MD_WIDTH                    1u
#define FR_RFRFCTR_F2MD(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F2MD_SHIFT))&FR_RFRFCTR_F2MD_MASK)
#define FR_RFRFCTR_F3MD_MASK                     0x800u
#define FR_RFRFCTR_F3MD_SHIFT                    11u
#define FR_RFRFCTR_F3MD_WIDTH                    1u
#define FR_RFRFCTR_F3MD(x)                       (((uint16_t)(((uint16_t)(x))<<FR_RFRFCTR_F3MD_SHIFT))&FR_RFRFCTR_F3MD_MASK)
/* LDTXSLAR Bit Fields */
#define FR_LDTXSLAR_LDYNTXSLOTA_MASK             0x7FFu
#define FR_LDTXSLAR_LDYNTXSLOTA_SHIFT            0u
#define FR_LDTXSLAR_LDYNTXSLOTA_WIDTH            11u
#define FR_LDTXSLAR_LDYNTXSLOTA(x)               (((uint16_t)(((uint16_t)(x))<<FR_LDTXSLAR_LDYNTXSLOTA_SHIFT))&FR_LDTXSLAR_LDYNTXSLOTA_MASK)
/* LDTXSLBR Bit Fields */
#define FR_LDTXSLBR_LDYNTXSLOTB_MASK             0x7FFu
#define FR_LDTXSLBR_LDYNTXSLOTB_SHIFT            0u
#define FR_LDTXSLBR_LDYNTXSLOTB_WIDTH            11u
#define FR_LDTXSLBR_LDYNTXSLOTB(x)               (((uint16_t)(((uint16_t)(x))<<FR_LDTXSLBR_LDYNTXSLOTB_SHIFT))&FR_LDTXSLBR_LDYNTXSLOTB_MASK)
/* PCR0 Bit Fields */
#define FR_PCR0_static_slot_length_MASK          0x3FFu
#define FR_PCR0_static_slot_length_SHIFT         0u
#define FR_PCR0_static_slot_length_WIDTH         10u
#define FR_PCR0_static_slot_length(x)            (((uint16_t)(((uint16_t)(x))<<FR_PCR0_static_slot_length_SHIFT))&FR_PCR0_static_slot_length_MASK)
#define FR_PCR0_action_point_offset_MASK         0xFC00u
#define FR_PCR0_action_point_offset_SHIFT        10u
#define FR_PCR0_action_point_offset_WIDTH        6u
#define FR_PCR0_action_point_offset(x)           (((uint16_t)(((uint16_t)(x))<<FR_PCR0_action_point_offset_SHIFT))&FR_PCR0_action_point_offset_MASK)
/* PCR1 Bit Fields */
#define FR_PCR1_macro_after_first_static_slot_MASK 0x3FFFu
#define FR_PCR1_macro_after_first_static_slot_SHIFT 0u
#define FR_PCR1_macro_after_first_static_slot_WIDTH 14u
#define FR_PCR1_macro_after_first_static_slot(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR1_macro_after_first_static_slot_SHIFT))&FR_PCR1_macro_after_first_static_slot_MASK)
/* PCR2 Bit Fields */
#define FR_PCR2_number_of_static_slots_MASK      0x3FFu
#define FR_PCR2_number_of_static_slots_SHIFT     0u
#define FR_PCR2_number_of_static_slots_WIDTH     10u
#define FR_PCR2_number_of_static_slots(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR2_number_of_static_slots_SHIFT))&FR_PCR2_number_of_static_slots_MASK)
#define FR_PCR2_minislot_after_action_point_MASK 0xFC00u
#define FR_PCR2_minislot_after_action_point_SHIFT 10u
#define FR_PCR2_minislot_after_action_point_WIDTH 6u
#define FR_PCR2_minislot_after_action_point(x)   (((uint16_t)(((uint16_t)(x))<<FR_PCR2_minislot_after_action_point_SHIFT))&FR_PCR2_minislot_after_action_point_MASK)
/* PCR3 Bit Fields */
#define FR_PCR3_coldstart_attempts_MASK          0x1Fu
#define FR_PCR3_coldstart_attempts_SHIFT         0u
#define FR_PCR3_coldstart_attempts_WIDTH         5u
#define FR_PCR3_coldstart_attempts(x)            (((uint16_t)(((uint16_t)(x))<<FR_PCR3_coldstart_attempts_SHIFT))&FR_PCR3_coldstart_attempts_MASK)
#define FR_PCR3_minislot_action_point_offset_MASK 0x3E0u
#define FR_PCR3_minislot_action_point_offset_SHIFT 5u
#define FR_PCR3_minislot_action_point_offset_WIDTH 5u
#define FR_PCR3_minislot_action_point_offset(x)  (((uint16_t)(((uint16_t)(x))<<FR_PCR3_minislot_action_point_offset_SHIFT))&FR_PCR3_minislot_action_point_offset_MASK)
#define FR_PCR3_wakeup_symbol_rx_low_MASK        0xFC00u
#define FR_PCR3_wakeup_symbol_rx_low_SHIFT       10u
#define FR_PCR3_wakeup_symbol_rx_low_WIDTH       6u
#define FR_PCR3_wakeup_symbol_rx_low(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR3_wakeup_symbol_rx_low_SHIFT))&FR_PCR3_wakeup_symbol_rx_low_MASK)
/* PCR4 Bit Fields */
#define FR_PCR4_wakeup_symbol_rx_window_MASK     0x1FFu
#define FR_PCR4_wakeup_symbol_rx_window_SHIFT    0u
#define FR_PCR4_wakeup_symbol_rx_window_WIDTH    9u
#define FR_PCR4_wakeup_symbol_rx_window(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR4_wakeup_symbol_rx_window_SHIFT))&FR_PCR4_wakeup_symbol_rx_window_MASK)
#define FR_PCR4_cas_rx_low_max_MASK              0xFE00u
#define FR_PCR4_cas_rx_low_max_SHIFT             9u
#define FR_PCR4_cas_rx_low_max_WIDTH             7u
#define FR_PCR4_cas_rx_low_max(x)                (((uint16_t)(((uint16_t)(x))<<FR_PCR4_cas_rx_low_max_SHIFT))&FR_PCR4_cas_rx_low_max_MASK)
/* PCR5 Bit Fields */
#define FR_PCR5_wakeup_symbol_rx_idle_MASK       0x3Fu
#define FR_PCR5_wakeup_symbol_rx_idle_SHIFT      0u
#define FR_PCR5_wakeup_symbol_rx_idle_WIDTH      6u
#define FR_PCR5_wakeup_symbol_rx_idle(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR5_wakeup_symbol_rx_idle_SHIFT))&FR_PCR5_wakeup_symbol_rx_idle_MASK)
#define FR_PCR5_wakeup_symbol_tx_low_MASK        0xFC0u
#define FR_PCR5_wakeup_symbol_tx_low_SHIFT       6u
#define FR_PCR5_wakeup_symbol_tx_low_WIDTH       6u
#define FR_PCR5_wakeup_symbol_tx_low(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR5_wakeup_symbol_tx_low_SHIFT))&FR_PCR5_wakeup_symbol_tx_low_MASK)
#define FR_PCR5_tss_transmitter_MASK             0xF000u
#define FR_PCR5_tss_transmitter_SHIFT            12u
#define FR_PCR5_tss_transmitter_WIDTH            4u
#define FR_PCR5_tss_transmitter(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR5_tss_transmitter_SHIFT))&FR_PCR5_tss_transmitter_MASK)
/* PCR6 Bit Fields */
#define FR_PCR6_macro_initial_offset_a_MASK      0x7Fu
#define FR_PCR6_macro_initial_offset_a_SHIFT     0u
#define FR_PCR6_macro_initial_offset_a_WIDTH     7u
#define FR_PCR6_macro_initial_offset_a(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR6_macro_initial_offset_a_SHIFT))&FR_PCR6_macro_initial_offset_a_MASK)
#define FR_PCR6_symbol_window_after_action_point_MASK 0x7F80u
#define FR_PCR6_symbol_window_after_action_point_SHIFT 7u
#define FR_PCR6_symbol_window_after_action_point_WIDTH 8u
#define FR_PCR6_symbol_window_after_action_point(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR6_symbol_window_after_action_point_SHIFT))&FR_PCR6_symbol_window_after_action_point_MASK)
/* PCR7 Bit Fields */
#define FR_PCR7_micro_per_macro_nom_half_MASK    0x7Fu
#define FR_PCR7_micro_per_macro_nom_half_SHIFT   0u
#define FR_PCR7_micro_per_macro_nom_half_WIDTH   7u
#define FR_PCR7_micro_per_macro_nom_half(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR7_micro_per_macro_nom_half_SHIFT))&FR_PCR7_micro_per_macro_nom_half_MASK)
#define FR_PCR7_decoding_correction_b_MASK       0xFF80u
#define FR_PCR7_decoding_correction_b_SHIFT      7u
#define FR_PCR7_decoding_correction_b_WIDTH      9u
#define FR_PCR7_decoding_correction_b(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR7_decoding_correction_b_SHIFT))&FR_PCR7_decoding_correction_b_MASK)
/* PCR8 Bit Fields */
#define FR_PCR8_wakeup_symbol_tx_idle_MASK       0xFFu
#define FR_PCR8_wakeup_symbol_tx_idle_SHIFT      0u
#define FR_PCR8_wakeup_symbol_tx_idle_WIDTH      8u
#define FR_PCR8_wakeup_symbol_tx_idle(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR8_wakeup_symbol_tx_idle_SHIFT))&FR_PCR8_wakeup_symbol_tx_idle_MASK)
#define FR_PCR8_max_without_clock_correction_passive_MASK 0xF00u
#define FR_PCR8_max_without_clock_correction_passive_SHIFT 8u
#define FR_PCR8_max_without_clock_correction_passive_WIDTH 4u
#define FR_PCR8_max_without_clock_correction_passive(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR8_max_without_clock_correction_passive_SHIFT))&FR_PCR8_max_without_clock_correction_passive_MASK)
#define FR_PCR8_max_without_clock_correction_fatal_MASK 0xF000u
#define FR_PCR8_max_without_clock_correction_fatal_SHIFT 12u
#define FR_PCR8_max_without_clock_correction_fatal_WIDTH 4u
#define FR_PCR8_max_without_clock_correction_fatal(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR8_max_without_clock_correction_fatal_SHIFT))&FR_PCR8_max_without_clock_correction_fatal_MASK)
/* PCR9 Bit Fields */
#define FR_PCR9_offset_correction_out_MASK       0x3FFFu
#define FR_PCR9_offset_correction_out_SHIFT      0u
#define FR_PCR9_offset_correction_out_WIDTH      14u
#define FR_PCR9_offset_correction_out(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR9_offset_correction_out_SHIFT))&FR_PCR9_offset_correction_out_MASK)
#define FR_PCR9_symbol_window_exists_MASK        0x4000u
#define FR_PCR9_symbol_window_exists_SHIFT       14u
#define FR_PCR9_symbol_window_exists_WIDTH       1u
#define FR_PCR9_symbol_window_exists(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR9_symbol_window_exists_SHIFT))&FR_PCR9_symbol_window_exists_MASK)
#define FR_PCR9_minislot_exists_MASK             0x8000u
#define FR_PCR9_minislot_exists_SHIFT            15u
#define FR_PCR9_minislot_exists_WIDTH            1u
#define FR_PCR9_minislot_exists(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR9_minislot_exists_SHIFT))&FR_PCR9_minislot_exists_MASK)
/* PCR10 Bit Fields */
#define FR_PCR10_macro_per_cycle_MASK            0x3FFFu
#define FR_PCR10_macro_per_cycle_SHIFT           0u
#define FR_PCR10_macro_per_cycle_WIDTH           14u
#define FR_PCR10_macro_per_cycle(x)              (((uint16_t)(((uint16_t)(x))<<FR_PCR10_macro_per_cycle_SHIFT))&FR_PCR10_macro_per_cycle_MASK)
#define FR_PCR10_wakeup_channel_MASK             0x4000u
#define FR_PCR10_wakeup_channel_SHIFT            14u
#define FR_PCR10_wakeup_channel_WIDTH            1u
#define FR_PCR10_wakeup_channel(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR10_wakeup_channel_SHIFT))&FR_PCR10_wakeup_channel_MASK)
#define FR_PCR10_single_slot_enabled_MASK        0x8000u
#define FR_PCR10_single_slot_enabled_SHIFT       15u
#define FR_PCR10_single_slot_enabled_WIDTH       1u
#define FR_PCR10_single_slot_enabled(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR10_single_slot_enabled_SHIFT))&FR_PCR10_single_slot_enabled_MASK)
/* PCR11 Bit Fields */
#define FR_PCR11_offset_correction_start_MASK    0x3FFFu
#define FR_PCR11_offset_correction_start_SHIFT   0u
#define FR_PCR11_offset_correction_start_WIDTH   14u
#define FR_PCR11_offset_correction_start(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR11_offset_correction_start_SHIFT))&FR_PCR11_offset_correction_start_MASK)
#define FR_PCR11_key_slot_used_for_sync_MASK     0x4000u
#define FR_PCR11_key_slot_used_for_sync_SHIFT    14u
#define FR_PCR11_key_slot_used_for_sync_WIDTH    1u
#define FR_PCR11_key_slot_used_for_sync(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR11_key_slot_used_for_sync_SHIFT))&FR_PCR11_key_slot_used_for_sync_MASK)
#define FR_PCR11_key_slot_used_for_startup_MASK  0x8000u
#define FR_PCR11_key_slot_used_for_startup_SHIFT 15u
#define FR_PCR11_key_slot_used_for_startup_WIDTH 1u
#define FR_PCR11_key_slot_used_for_startup(x)    (((uint16_t)(((uint16_t)(x))<<FR_PCR11_key_slot_used_for_startup_SHIFT))&FR_PCR11_key_slot_used_for_startup_MASK)
/* PCR12 Bit Fields */
#define FR_PCR12_key_slot_header_crc_MASK        0x7FFu
#define FR_PCR12_key_slot_header_crc_SHIFT       0u
#define FR_PCR12_key_slot_header_crc_WIDTH       11u
#define FR_PCR12_key_slot_header_crc(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR12_key_slot_header_crc_SHIFT))&FR_PCR12_key_slot_header_crc_MASK)
#define FR_PCR12_allow_passive_to_active_MASK    0xF800u
#define FR_PCR12_allow_passive_to_active_SHIFT   11u
#define FR_PCR12_allow_passive_to_active_WIDTH   5u
#define FR_PCR12_allow_passive_to_active(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR12_allow_passive_to_active_SHIFT))&FR_PCR12_allow_passive_to_active_MASK)
/* PCR13 Bit Fields */
#define FR_PCR13_static_slot_after_action_point_MASK 0x3FFu
#define FR_PCR13_static_slot_after_action_point_SHIFT 0u
#define FR_PCR13_static_slot_after_action_point_WIDTH 10u
#define FR_PCR13_static_slot_after_action_point(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR13_static_slot_after_action_point_SHIFT))&FR_PCR13_static_slot_after_action_point_MASK)
#define FR_PCR13_first_minislot_action_point_offset_MASK 0xFC00u
#define FR_PCR13_first_minislot_action_point_offset_SHIFT 10u
#define FR_PCR13_first_minislot_action_point_offset_WIDTH 6u
#define FR_PCR13_first_minislot_action_point_offset(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR13_first_minislot_action_point_offset_SHIFT))&FR_PCR13_first_minislot_action_point_offset_MASK)
/* PCR14 Bit Fields */
#define FR_PCR14_listen_timeout_MASK             0x1Fu
#define FR_PCR14_listen_timeout_SHIFT            0u
#define FR_PCR14_listen_timeout_WIDTH            5u
#define FR_PCR14_listen_timeout(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR14_listen_timeout_SHIFT))&FR_PCR14_listen_timeout_MASK)
#define FR_PCR14_rate_correction_out_MASK        0xFFE0u
#define FR_PCR14_rate_correction_out_SHIFT       5u
#define FR_PCR14_rate_correction_out_WIDTH       11u
#define FR_PCR14_rate_correction_out(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR14_rate_correction_out_SHIFT))&FR_PCR14_rate_correction_out_MASK)
/* PCR15 Bit Fields */
#define FR_PCR15_listen_timeout_MASK             0xFFFFu
#define FR_PCR15_listen_timeout_SHIFT            0u
#define FR_PCR15_listen_timeout_WIDTH            16u
#define FR_PCR15_listen_timeout(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR15_listen_timeout_SHIFT))&FR_PCR15_listen_timeout_MASK)
/* PCR16 Bit Fields */
#define FR_PCR16_noise_listen_timeout_MASK       0x1FFu
#define FR_PCR16_noise_listen_timeout_SHIFT      0u
#define FR_PCR16_noise_listen_timeout_WIDTH      9u
#define FR_PCR16_noise_listen_timeout(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR16_noise_listen_timeout_SHIFT))&FR_PCR16_noise_listen_timeout_MASK)
#define FR_PCR16_macro_initial_offset_b_MASK     0xFE00u
#define FR_PCR16_macro_initial_offset_b_SHIFT    9u
#define FR_PCR16_macro_initial_offset_b_WIDTH    7u
#define FR_PCR16_macro_initial_offset_b(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR16_macro_initial_offset_b_SHIFT))&FR_PCR16_macro_initial_offset_b_MASK)
/* PCR17 Bit Fields */
#define FR_PCR17_noise_listen_timeout_MASK       0xFFFFu
#define FR_PCR17_noise_listen_timeout_SHIFT      0u
#define FR_PCR17_noise_listen_timeout_WIDTH      16u
#define FR_PCR17_noise_listen_timeout(x)         (((uint16_t)(((uint16_t)(x))<<FR_PCR17_noise_listen_timeout_SHIFT))&FR_PCR17_noise_listen_timeout_MASK)
/* PCR18 Bit Fields */
#define FR_PCR18_key_slot_id_MASK                0x3FFu
#define FR_PCR18_key_slot_id_SHIFT               0u
#define FR_PCR18_key_slot_id_WIDTH               10u
#define FR_PCR18_key_slot_id(x)                  (((uint16_t)(((uint16_t)(x))<<FR_PCR18_key_slot_id_SHIFT))&FR_PCR18_key_slot_id_MASK)
#define FR_PCR18_wakeup_pattern_MASK             0xFC00u
#define FR_PCR18_wakeup_pattern_SHIFT            10u
#define FR_PCR18_wakeup_pattern_WIDTH            6u
#define FR_PCR18_wakeup_pattern(x)               (((uint16_t)(((uint16_t)(x))<<FR_PCR18_wakeup_pattern_SHIFT))&FR_PCR18_wakeup_pattern_MASK)
/* PCR19 Bit Fields */
#define FR_PCR19_payload_length_static_MASK      0x7Fu
#define FR_PCR19_payload_length_static_SHIFT     0u
#define FR_PCR19_payload_length_static_WIDTH     7u
#define FR_PCR19_payload_length_static(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR19_payload_length_static_SHIFT))&FR_PCR19_payload_length_static_MASK)
#define FR_PCR19_decoding_correction_a_MASK      0xFF80u
#define FR_PCR19_decoding_correction_a_SHIFT     7u
#define FR_PCR19_decoding_correction_a_WIDTH     9u
#define FR_PCR19_decoding_correction_a(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR19_decoding_correction_a_SHIFT))&FR_PCR19_decoding_correction_a_MASK)
/* PCR20 Bit Fields */
#define FR_PCR20_micro_initial_offset_a_MASK     0xFFu
#define FR_PCR20_micro_initial_offset_a_SHIFT    0u
#define FR_PCR20_micro_initial_offset_a_WIDTH    8u
#define FR_PCR20_micro_initial_offset_a(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR20_micro_initial_offset_a_SHIFT))&FR_PCR20_micro_initial_offset_a_MASK)
#define FR_PCR20_micro_initial_offset_b_MASK     0xFF00u
#define FR_PCR20_micro_initial_offset_b_SHIFT    8u
#define FR_PCR20_micro_initial_offset_b_WIDTH    8u
#define FR_PCR20_micro_initial_offset_b(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR20_micro_initial_offset_b_SHIFT))&FR_PCR20_micro_initial_offset_b_MASK)
/* PCR21 Bit Fields */
#define FR_PCR21_latest_tx_MASK                  0x1FFFu
#define FR_PCR21_latest_tx_SHIFT                 0u
#define FR_PCR21_latest_tx_WIDTH                 13u
#define FR_PCR21_latest_tx(x)                    (((uint16_t)(((uint16_t)(x))<<FR_PCR21_latest_tx_SHIFT))&FR_PCR21_latest_tx_MASK)
#define FR_PCR21_extern_rate_correction_MASK     0xE000u
#define FR_PCR21_extern_rate_correction_SHIFT    13u
#define FR_PCR21_extern_rate_correction_WIDTH    3u
#define FR_PCR21_extern_rate_correction(x)       (((uint16_t)(((uint16_t)(x))<<FR_PCR21_extern_rate_correction_SHIFT))&FR_PCR21_extern_rate_correction_MASK)
/* PCR22 Bit Fields */
#define FR_PCR22_micro_per_cycle_MASK            0xFu
#define FR_PCR22_micro_per_cycle_SHIFT           0u
#define FR_PCR22_micro_per_cycle_WIDTH           4u
#define FR_PCR22_micro_per_cycle(x)              (((uint16_t)(((uint16_t)(x))<<FR_PCR22_micro_per_cycle_SHIFT))&FR_PCR22_micro_per_cycle_MASK)
#define FR_PCR22_comp_accepted_startup_range_a_MASK 0x7FF0u
#define FR_PCR22_comp_accepted_startup_range_a_SHIFT 4u
#define FR_PCR22_comp_accepted_startup_range_a_WIDTH 11u
#define FR_PCR22_comp_accepted_startup_range_a(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR22_comp_accepted_startup_range_a_SHIFT))&FR_PCR22_comp_accepted_startup_range_a_MASK)
/* PCR23 Bit Fields */
#define FR_PCR23_micro_per_cycle_MASK            0xFFFFu
#define FR_PCR23_micro_per_cycle_SHIFT           0u
#define FR_PCR23_micro_per_cycle_WIDTH           16u
#define FR_PCR23_micro_per_cycle(x)              (((uint16_t)(((uint16_t)(x))<<FR_PCR23_micro_per_cycle_SHIFT))&FR_PCR23_micro_per_cycle_MASK)
/* PCR24 Bit Fields */
#define FR_PCR24_micro_per_cycle_min_MASK        0xFu
#define FR_PCR24_micro_per_cycle_min_SHIFT       0u
#define FR_PCR24_micro_per_cycle_min_WIDTH       4u
#define FR_PCR24_micro_per_cycle_min(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR24_micro_per_cycle_min_SHIFT))&FR_PCR24_micro_per_cycle_min_MASK)
#define FR_PCR24_max_payload_length_dynamic_MASK 0x7F0u
#define FR_PCR24_max_payload_length_dynamic_SHIFT 4u
#define FR_PCR24_max_payload_length_dynamic_WIDTH 7u
#define FR_PCR24_max_payload_length_dynamic(x)   (((uint16_t)(((uint16_t)(x))<<FR_PCR24_max_payload_length_dynamic_SHIFT))&FR_PCR24_max_payload_length_dynamic_MASK)
#define FR_PCR24_cluster_drift_damping_MASK      0xF800u
#define FR_PCR24_cluster_drift_damping_SHIFT     11u
#define FR_PCR24_cluster_drift_damping_WIDTH     5u
#define FR_PCR24_cluster_drift_damping(x)        (((uint16_t)(((uint16_t)(x))<<FR_PCR24_cluster_drift_damping_SHIFT))&FR_PCR24_cluster_drift_damping_MASK)
/* PCR25 Bit Fields */
#define FR_PCR25_micro_per_cycle_min_MASK        0xFFFFu
#define FR_PCR25_micro_per_cycle_min_SHIFT       0u
#define FR_PCR25_micro_per_cycle_min_WIDTH       16u
#define FR_PCR25_micro_per_cycle_min(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR25_micro_per_cycle_min_SHIFT))&FR_PCR25_micro_per_cycle_min_MASK)
/* PCR26 Bit Fields */
#define FR_PCR26_micro_per_cycle_max_MASK        0xFu
#define FR_PCR26_micro_per_cycle_max_SHIFT       0u
#define FR_PCR26_micro_per_cycle_max_WIDTH       4u
#define FR_PCR26_micro_per_cycle_max(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR26_micro_per_cycle_max_SHIFT))&FR_PCR26_micro_per_cycle_max_MASK)
#define FR_PCR26_comp_accepted_startup_range_b_MASK 0x7FF0u
#define FR_PCR26_comp_accepted_startup_range_b_SHIFT 4u
#define FR_PCR26_comp_accepted_startup_range_b_WIDTH 11u
#define FR_PCR26_comp_accepted_startup_range_b(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR26_comp_accepted_startup_range_b_SHIFT))&FR_PCR26_comp_accepted_startup_range_b_MASK)
#define FR_PCR26_allow_halt_due_to_clock_MASK    0x8000u
#define FR_PCR26_allow_halt_due_to_clock_SHIFT   15u
#define FR_PCR26_allow_halt_due_to_clock_WIDTH   1u
#define FR_PCR26_allow_halt_due_to_clock(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR26_allow_halt_due_to_clock_SHIFT))&FR_PCR26_allow_halt_due_to_clock_MASK)
/* PCR27 Bit Fields */
#define FR_PCR27_micro_per_cycle_max_MASK        0xFFFFu
#define FR_PCR27_micro_per_cycle_max_SHIFT       0u
#define FR_PCR27_micro_per_cycle_max_WIDTH       16u
#define FR_PCR27_micro_per_cycle_max(x)          (((uint16_t)(((uint16_t)(x))<<FR_PCR27_micro_per_cycle_max_SHIFT))&FR_PCR27_micro_per_cycle_max_MASK)
/* PCR28 Bit Fields */
#define FR_PCR28_macro_after_offset_correction_MASK 0x3FFFu
#define FR_PCR28_macro_after_offset_correction_SHIFT 0u
#define FR_PCR28_macro_after_offset_correction_WIDTH 14u
#define FR_PCR28_macro_after_offset_correction(x) (((uint16_t)(((uint16_t)(x))<<FR_PCR28_macro_after_offset_correction_SHIFT))&FR_PCR28_macro_after_offset_correction_MASK)
#define FR_PCR28_dynamic_slot_idle_phase_MASK    0xC000u
#define FR_PCR28_dynamic_slot_idle_phase_SHIFT   14u
#define FR_PCR28_dynamic_slot_idle_phase_WIDTH   2u
#define FR_PCR28_dynamic_slot_idle_phase(x)      (((uint16_t)(((uint16_t)(x))<<FR_PCR28_dynamic_slot_idle_phase_SHIFT))&FR_PCR28_dynamic_slot_idle_phase_MASK)
/* PCR29 Bit Fields */
#define FR_PCR29_minislots_max_MASK              0x1FFFu
#define FR_PCR29_minislots_max_SHIFT             0u
#define FR_PCR29_minislots_max_WIDTH             13u
#define FR_PCR29_minislots_max(x)                (((uint16_t)(((uint16_t)(x))<<FR_PCR29_minislots_max_SHIFT))&FR_PCR29_minislots_max_MASK)
#define FR_PCR29_extern_offset_correction_MASK   0xE000u
#define FR_PCR29_extern_offset_correction_SHIFT  13u
#define FR_PCR29_extern_offset_correction_WIDTH  3u
#define FR_PCR29_extern_offset_correction(x)     (((uint16_t)(((uint16_t)(x))<<FR_PCR29_extern_offset_correction_SHIFT))&FR_PCR29_extern_offset_correction_MASK)
/* PCR30 Bit Fields */
#define FR_PCR30_sync_node_max_MASK              0xFu
#define FR_PCR30_sync_node_max_SHIFT             0u
#define FR_PCR30_sync_node_max_WIDTH             4u
#define FR_PCR30_sync_node_max(x)                (((uint16_t)(((uint16_t)(x))<<FR_PCR30_sync_node_max_SHIFT))&FR_PCR30_sync_node_max_MASK)
/* STPWHR Bit Fields */
#define FR_STPWHR_STPW_S_H_MASK                  0xFFFFu
#define FR_STPWHR_STPW_S_H_SHIFT                 0u
#define FR_STPWHR_STPW_S_H_WIDTH                 16u
#define FR_STPWHR_STPW_S_H(x)                    (((uint16_t)(((uint16_t)(x))<<FR_STPWHR_STPW_S_H_SHIFT))&FR_STPWHR_STPW_S_H_MASK)
/* STPWLR Bit Fields */
#define FR_STPWLR_STPW_S_L_MASK                  0xFFFFu
#define FR_STPWLR_STPW_S_L_SHIFT                 0u
#define FR_STPWLR_STPW_S_L_WIDTH                 16u
#define FR_STPWLR_STPW_S_L(x)                    (((uint16_t)(((uint16_t)(x))<<FR_STPWLR_STPW_S_L_SHIFT))&FR_STPWLR_STPW_S_L_MASK)
/* PEOER Bit Fields */
#define FR_PEOER_CYC_EE_MASK                     0x1u
#define FR_PEOER_CYC_EE_SHIFT                    0u
#define FR_PEOER_CYC_EE_WIDTH                    1u
#define FR_PEOER_CYC_EE(x)                       (((uint16_t)(((uint16_t)(x))<<FR_PEOER_CYC_EE_SHIFT))&FR_PEOER_CYC_EE_MASK)
#define FR_PEOER_TIM1_EE_MASK                    0x2u
#define FR_PEOER_TIM1_EE_SHIFT                   1u
#define FR_PEOER_TIM1_EE_WIDTH                   1u
#define FR_PEOER_TIM1_EE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PEOER_TIM1_EE_SHIFT))&FR_PEOER_TIM1_EE_MASK)
#define FR_PEOER_TIM2_EE_MASK                    0x4u
#define FR_PEOER_TIM2_EE_SHIFT                   2u
#define FR_PEOER_TIM2_EE_WIDTH                   1u
#define FR_PEOER_TIM2_EE(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PEOER_TIM2_EE_SHIFT))&FR_PEOER_TIM2_EE_MASK)
#define FR_PEOER_STPW_EN_MASK                    0x100u
#define FR_PEOER_STPW_EN_SHIFT                   8u
#define FR_PEOER_STPW_EN_WIDTH                   1u
#define FR_PEOER_STPW_EN(x)                      (((uint16_t)(((uint16_t)(x))<<FR_PEOER_STPW_EN_SHIFT))&FR_PEOER_STPW_EN_MASK)
/* RFSDOR Bit Fields */
#define FR_RFSDOR_SDO_MASK                       0xFFFFu
#define FR_RFSDOR_SDO_SHIFT                      0u
#define FR_RFSDOR_SDO_WIDTH                      16u
#define FR_RFSDOR_SDO(x)                         (((uint16_t)(((uint16_t)(x))<<FR_RFSDOR_SDO_SHIFT))&FR_RFSDOR_SDO_MASK)
/* RFSYMBADHR Bit Fields */
#define FR_RFSYMBADHR_SMBA_MASK                  0xFFFFu
#define FR_RFSYMBADHR_SMBA_SHIFT                 0u
#define FR_RFSYMBADHR_SMBA_WIDTH                 16u
#define FR_RFSYMBADHR_SMBA(x)                    (((uint16_t)(((uint16_t)(x))<<FR_RFSYMBADHR_SMBA_SHIFT))&FR_RFSYMBADHR_SMBA_MASK)
/* RFSYMBADLR Bit Fields */
#define FR_RFSYMBADLR_SMBA_MASK                  0xFFF0u
#define FR_RFSYMBADLR_SMBA_SHIFT                 4u
#define FR_RFSYMBADLR_SMBA_WIDTH                 12u
#define FR_RFSYMBADLR_SMBA(x)                    (((uint16_t)(((uint16_t)(x))<<FR_RFSYMBADLR_SMBA_SHIFT))&FR_RFSYMBADLR_SMBA_MASK)
/* RFPTR Bit Fields */
#define FR_RFPTR_PTD_MASK                        0x3FFFu
#define FR_RFPTR_PTD_SHIFT                       0u
#define FR_RFPTR_PTD_WIDTH                       14u
#define FR_RFPTR_PTD(x)                          (((uint16_t)(((uint16_t)(x))<<FR_RFPTR_PTD_SHIFT))&FR_RFPTR_PTD_MASK)
/* RFFLPCR Bit Fields */
#define FR_RFFLPCR_FLA_or_PCA_MASK               0xFFu
#define FR_RFFLPCR_FLA_or_PCA_SHIFT              0u
#define FR_RFFLPCR_FLA_or_PCA_WIDTH              8u
#define FR_RFFLPCR_FLA_or_PCA(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFFLPCR_FLA_or_PCA_SHIFT))&FR_RFFLPCR_FLA_or_PCA_MASK)
#define FR_RFFLPCR_FLB_or_PCB_MASK               0xFF00u
#define FR_RFFLPCR_FLB_or_PCB_SHIFT              8u
#define FR_RFFLPCR_FLB_or_PCB_WIDTH              8u
#define FR_RFFLPCR_FLB_or_PCB(x)                 (((uint16_t)(((uint16_t)(x))<<FR_RFFLPCR_FLB_or_PCB_SHIFT))&FR_RFFLPCR_FLB_or_PCB_MASK)
/* EEIFER Bit Fields */
#define FR_EEIFER_DRCE_IE_MASK                   0x1u
#define FR_EEIFER_DRCE_IE_SHIFT                  0u
#define FR_EEIFER_DRCE_IE_WIDTH                  1u
#define FR_EEIFER_DRCE_IE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRCE_IE_SHIFT))&FR_EEIFER_DRCE_IE_MASK)
#define FR_EEIFER_DRNE_IE_MASK                   0x2u
#define FR_EEIFER_DRNE_IE_SHIFT                  1u
#define FR_EEIFER_DRNE_IE_WIDTH                  1u
#define FR_EEIFER_DRNE_IE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRNE_IE_SHIFT))&FR_EEIFER_DRNE_IE_MASK)
#define FR_EEIFER_LRCE_IE_MASK                   0x4u
#define FR_EEIFER_LRCE_IE_SHIFT                  2u
#define FR_EEIFER_LRCE_IE_WIDTH                  1u
#define FR_EEIFER_LRCE_IE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRCE_IE_SHIFT))&FR_EEIFER_LRCE_IE_MASK)
#define FR_EEIFER_LRNE_IE_MASK                   0x8u
#define FR_EEIFER_LRNE_IE_SHIFT                  3u
#define FR_EEIFER_LRNE_IE_WIDTH                  1u
#define FR_EEIFER_LRNE_IE(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRNE_IE_SHIFT))&FR_EEIFER_LRNE_IE_MASK)
#define FR_EEIFER_DRCE_IF_MASK                   0x100u
#define FR_EEIFER_DRCE_IF_SHIFT                  8u
#define FR_EEIFER_DRCE_IF_WIDTH                  1u
#define FR_EEIFER_DRCE_IF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRCE_IF_SHIFT))&FR_EEIFER_DRCE_IF_MASK)
#define FR_EEIFER_DRNE_IF_MASK                   0x200u
#define FR_EEIFER_DRNE_IF_SHIFT                  9u
#define FR_EEIFER_DRNE_IF_WIDTH                  1u
#define FR_EEIFER_DRNE_IF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRNE_IF_SHIFT))&FR_EEIFER_DRNE_IF_MASK)
#define FR_EEIFER_LRCE_IF_MASK                   0x400u
#define FR_EEIFER_LRCE_IF_SHIFT                  10u
#define FR_EEIFER_LRCE_IF_WIDTH                  1u
#define FR_EEIFER_LRCE_IF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRCE_IF_SHIFT))&FR_EEIFER_LRCE_IF_MASK)
#define FR_EEIFER_LRNE_IF_MASK                   0x800u
#define FR_EEIFER_LRNE_IF_SHIFT                  11u
#define FR_EEIFER_LRNE_IF_WIDTH                  1u
#define FR_EEIFER_LRNE_IF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRNE_IF_SHIFT))&FR_EEIFER_LRNE_IF_MASK)
#define FR_EEIFER_DRCE_OF_MASK                   0x1000u
#define FR_EEIFER_DRCE_OF_SHIFT                  12u
#define FR_EEIFER_DRCE_OF_WIDTH                  1u
#define FR_EEIFER_DRCE_OF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRCE_OF_SHIFT))&FR_EEIFER_DRCE_OF_MASK)
#define FR_EEIFER_DRNE_OF_MASK                   0x2000u
#define FR_EEIFER_DRNE_OF_SHIFT                  13u
#define FR_EEIFER_DRNE_OF_WIDTH                  1u
#define FR_EEIFER_DRNE_OF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_DRNE_OF_SHIFT))&FR_EEIFER_DRNE_OF_MASK)
#define FR_EEIFER_LRCE_OF_MASK                   0x4000u
#define FR_EEIFER_LRCE_OF_SHIFT                  14u
#define FR_EEIFER_LRCE_OF_WIDTH                  1u
#define FR_EEIFER_LRCE_OF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRCE_OF_SHIFT))&FR_EEIFER_LRCE_OF_MASK)
#define FR_EEIFER_LRNE_OF_MASK                   0x8000u
#define FR_EEIFER_LRNE_OF_SHIFT                  15u
#define FR_EEIFER_LRNE_OF_WIDTH                  1u
#define FR_EEIFER_LRNE_OF(x)                     (((uint16_t)(((uint16_t)(x))<<FR_EEIFER_LRNE_OF_SHIFT))&FR_EEIFER_LRNE_OF_MASK)
/* EERICR Bit Fields */
#define FR_EERICR_EIE_MASK                       0x1u
#define FR_EERICR_EIE_SHIFT                      0u
#define FR_EERICR_EIE_WIDTH                      1u
#define FR_EERICR_EIE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_EIE_SHIFT))&FR_EERICR_EIE_MASK)
#define FR_EERICR_EIM_MASK                       0x2u
#define FR_EERICR_EIM_SHIFT                      1u
#define FR_EERICR_EIM_WIDTH                      1u
#define FR_EERICR_EIM(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_EIM_SHIFT))&FR_EERICR_EIM_MASK)
#define FR_EERICR_ERM_MASK                       0x10u
#define FR_EERICR_ERM_SHIFT                      4u
#define FR_EERICR_ERM_WIDTH                      1u
#define FR_EERICR_ERM(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_ERM_SHIFT))&FR_EERICR_ERM_MASK)
#define FR_EERICR_ERS_MASK                       0x300u
#define FR_EERICR_ERS_SHIFT                      8u
#define FR_EERICR_ERS_WIDTH                      2u
#define FR_EERICR_ERS(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_ERS_SHIFT))&FR_EERICR_ERS_MASK)
#define FR_EERICR_BSY_MASK                       0x8000u
#define FR_EERICR_BSY_SHIFT                      15u
#define FR_EERICR_BSY_WIDTH                      1u
#define FR_EERICR_BSY(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERICR_BSY_SHIFT))&FR_EERICR_BSY_MASK)
/* EERAR Bit Fields */
#define FR_EERAR_ADDR_MASK                       0xFFFu
#define FR_EERAR_ADDR_SHIFT                      0u
#define FR_EERAR_ADDR_WIDTH                      12u
#define FR_EERAR_ADDR(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERAR_ADDR_SHIFT))&FR_EERAR_ADDR_MASK)
#define FR_EERAR_BANK_MASK                       0x7000u
#define FR_EERAR_BANK_SHIFT                      12u
#define FR_EERAR_BANK_WIDTH                      3u
#define FR_EERAR_BANK(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERAR_BANK_SHIFT))&FR_EERAR_BANK_MASK)
#define FR_EERAR_MID_MASK                        0x8000u
#define FR_EERAR_MID_SHIFT                       15u
#define FR_EERAR_MID_WIDTH                       1u
#define FR_EERAR_MID(x)                          (((uint16_t)(((uint16_t)(x))<<FR_EERAR_MID_SHIFT))&FR_EERAR_MID_MASK)
/* EERDR Bit Fields */
#define FR_EERDR_DATA_MASK                       0xFFFFu
#define FR_EERDR_DATA_SHIFT                      0u
#define FR_EERDR_DATA_WIDTH                      16u
#define FR_EERDR_DATA(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERDR_DATA_SHIFT))&FR_EERDR_DATA_MASK)
/* EERCR Bit Fields */
#define FR_EERCR_CODE_MASK                       0x1Fu
#define FR_EERCR_CODE_SHIFT                      0u
#define FR_EERCR_CODE_WIDTH                      5u
#define FR_EERCR_CODE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EERCR_CODE_SHIFT))&FR_EERCR_CODE_MASK)
/* EEIAR Bit Fields */
#define FR_EEIAR_ADDR_MASK                       0xFFFu
#define FR_EEIAR_ADDR_SHIFT                      0u
#define FR_EEIAR_ADDR_WIDTH                      12u
#define FR_EEIAR_ADDR(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EEIAR_ADDR_SHIFT))&FR_EEIAR_ADDR_MASK)
#define FR_EEIAR_BANK_MASK                       0x7000u
#define FR_EEIAR_BANK_SHIFT                      12u
#define FR_EEIAR_BANK_WIDTH                      3u
#define FR_EEIAR_BANK(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EEIAR_BANK_SHIFT))&FR_EEIAR_BANK_MASK)
#define FR_EEIAR_MID_MASK                        0x8000u
#define FR_EEIAR_MID_SHIFT                       15u
#define FR_EEIAR_MID_WIDTH                       1u
#define FR_EEIAR_MID(x)                          (((uint16_t)(((uint16_t)(x))<<FR_EEIAR_MID_SHIFT))&FR_EEIAR_MID_MASK)
/* EEIDR Bit Fields */
#define FR_EEIDR_DATA_MASK                       0xFFFFu
#define FR_EEIDR_DATA_SHIFT                      0u
#define FR_EEIDR_DATA_WIDTH                      16u
#define FR_EEIDR_DATA(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EEIDR_DATA_SHIFT))&FR_EEIDR_DATA_MASK)
/* EEICR Bit Fields */
#define FR_EEICR_CODE_MASK                       0x1Fu
#define FR_EEICR_CODE_SHIFT                      0u
#define FR_EEICR_CODE_WIDTH                      5u
#define FR_EEICR_CODE(x)                         (((uint16_t)(((uint16_t)(x))<<FR_EEICR_CODE_SHIFT))&FR_EEICR_CODE_MASK)
/* CCSR Bit Fields */
#define FR_CCSR_MBIF_MASK                        0x1u
#define FR_CCSR_MBIF_SHIFT                       0u
#define FR_CCSR_MBIF_WIDTH                       1u
#define FR_CCSR_MBIF(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CCSR_MBIF_SHIFT))&FR_CCSR_MBIF_MASK)
#define FR_CCSR_LCKS_MASK                        0x2u
#define FR_CCSR_LCKS_SHIFT                       1u
#define FR_CCSR_LCKS_WIDTH                       1u
#define FR_CCSR_LCKS(x)                          (((uint16_t)(((uint16_t)(x))<<FR_CCSR_LCKS_SHIFT))&FR_CCSR_LCKS_MASK)
#define FR_CCSR_EDS_MASK                         0x4u
#define FR_CCSR_EDS_SHIFT                        2u
#define FR_CCSR_EDS_WIDTH            