/**
 * @file debug.c
 * @brief Debugging facilities
 *
 * @section License
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * Copyright (C) 2010-2023 Oryx Embedded SARL. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.2.2
 **/

//Dependencies
#include <p32xxxx.h>
#include "debug.h"


/**
 * @brief Debug UART initialization
 * @param[in] baudrate UART baudrate
 **/

void debugInit(uint32_t baudrate)
{
#if 0
   //Remap U5TX to RPC9
   RPC9R = 11;
   //Remap U5RX to RPF0
   U5RXR = 12;

   //Configure UART5 (8 bits, no parity, 1 stop bit)
   U5STA = 0;
   //Enable high-speed mode
   U5MODE = _U5MODE_BRGH_MASK;
   //Set baudrate
   U5BRG = ((60000000 / 4) / baudrate) - 1;

   //Enable UART5
   U5MODESET = _U5MODE_ON_MASK;
   //Enable transmitter and receiver
   U5STASET = _U5STA_UTXEN_MASK | _U5STA_URXEN_MASK;
#else
   //Disable analog pad on U6TX (AN24/RA4)
   ANSELACLR = _ANSELA_ANSA4_MASK;
   //Disable analog pad on U6RX (AN47/RA15)
   ANSELACLR = _ANSELA_ANSA15_MASK;

   //Remap U6TX to RPA4
   RPA4R = 11;
   //Remap U6RX to RPA15
   U6RXR = 13;

   //Configure UART6 (8 bits, no parity, 1 stop bit)
   U6STA = 0;
   //Enable high-speed mode
   U6MODE = _U6MODE_BRGH_MASK;
   //Set baudrate
   U6BRG = ((60000000 / 4) / baudrate) - 1;

   //Enable UART6
   U6MODESET = _U6MODE_ON_MASK;
   //Enable transmitter and receiver
   U6STASET = _U6STA_UTXEN_MASK | _U6STA_URXEN_MASK;
#endif
}


/**
 * @brief Display the contents of an array
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @param[in] prepend String to prepend to the left of each line
 * @param[in] data Pointer to the data array
 * @param[in] length Number of bytes to display
 **/

void debugDisplayArray(FILE *stream,
   const char_t *prepend, const void *data, size_t length)
{
   uint_t i;

   for(i = 0; i < length; i++)
   {
      //Beginning of a new line?
      if((i % 16) == 0)
         fprintf(stream, "%s", prepend);
      //Display current data byte
      fprintf(stream, "%02" PRIX8 " ", *((uint8_t *) data + i));
      //End of current line?
      if((i % 16) == 15 || i == (length - 1))
         fprintf(stream, "\r\n");
   }
}


/**
 * @brief Write character to stream
 * @param[in] c The character to be written
 * @param[in] stream Pointer to a FILE object that identifies an output stream
 * @return On success, the character written is returned. If a writing
 *   error occurs, EOF is returned
 **/

int_t fputc(int_t c, FILE *stream)
{
   //Standard output or error output?
   if(stream == stdout || stream == stderr)
   {
#if 0
      //Wait for the transmitter to be ready
      while(U5STA & _U5STA_UTXBF_MASK);
      //Send character
      U5TXREG = c;
      //Wait for the transfer to complete
      while(!(U5STA & _U5STA_TRMT_MASK));
#else
      //Wait for the transmitter to be ready
      while(U6STA & _U6STA_UTXBF_MASK);
      //Send character
      U6TXREG = c;
      //Wait for the transfer to complete
      while(!(U6STA & _U6STA_TRMT_MASK));
#endif

      //On success, the character written is returned
      return c;
   }
   //Unknown output?
   else
   {
      //If a writing error occurs, EOF is returned
      return EOF;
   }
}
