/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016 NXP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _FSL_GPC_H_
#define _FSL_GPC_H_

#include "fsl_common.h"

/*!
 * @addtogroup gpc
 * @{
 */

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*! @name Driver version */
/*@{*/
/*! @brief GPC driver version 2.0.0. */
#define FSL_GPC_DRIVER_VERSION (MAKE_VERSION(2, 0, 0))
/*@}*/

#if defined(__cplusplus)
extern "C" {
#endif

/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @brief Allow all the IRQ/Events within the charge of GPC.
 *
 * @param base GPC peripheral base address.
 */
static inline void GPC_AllowIRQs(GPC_Type *base)
{
    base->CNTR &= ~GPC_CNTR_GPCIRQM_MASK; /* Events would not be masked. */
}

/*!
 * @brief Disallow all the IRQ/Events within the charge of GPC.
 *
 * @param base GPC peripheral base address.
 */
static inline void GPC_DisallowIRQs(GPC_Type *base)
{
    base->CNTR |= GPC_CNTR_GPCIRQM_MASK; /* Mask all the events. */
}

/*!
 * @brief Enable the IRQ.
 *
 * @param base GPC peripheral base address.
 * @param irqId ID number of IRQ to be enabled, available range is 32-159.
 */
void GPC_EnableIRQ(GPC_Type *base, uint32_t irqId);

/*!
 * @brief Disable the IRQ.
 *
 * @param base GPC peripheral base address.
 * @param irqId ID number of IRQ to be disabled, available range is 32-159.
 */
void GPC_DisableIRQ(GPC_Type *base, uint32_t irqId);

/*!
 * @brief Get the IRQ/Event flag.
 *
 * @param base GPC peripheral base address.
 * @param irqId ID number of IRQ to be enabled, available range is 32-159.
 * @return Indicated IRQ/Event is asserted or not.
 */
bool GPC_GetIRQStatusFlag(GPC_Type *base, uint32_t irqId);

/*!
 * @brief L2 Cache Power Gate Enable
 *
 * This function configures the L2 cache if it will keep power when in low power mode.
 * When the L2 cache power is OFF, L2 cache will be power down once when CPU core is power down
 * and will be hardware invalidated automatically when CPU core is re-power up.
 * When the L2 cache power is ON, L2 cache will keep power on even if CPU core is power down and
 * will not be hardware invalidated.
 * When CPU core is re-power up, the default setting is OFF.
 *
 * @param base GPC peripheral base address.
 * @param enable Enable the request or not.
 */
static inline void GPC_RequestL2CachePowerDown(GPC_Type *base, bool enable)
{
    if (enable)
    {
        base->CNTR |= GPC_CNTR_L2_PGE_MASK; /* OFF. */
    }
    else
    {
        base->CNTR &= ~GPC_CNTR_L2_PGE_MASK; /* ON. */
    }
}

/*!
 * @brief VADC power down.
 *
 * This function requests the VADC power down.
 *
 * @param base GPC peripheral base address.
 * @param enable Enable the request or not.
 */
static inline void GPC_RequestVADCPowerDown(GPC_Type *base, bool enable)
{
    if (enable)
    {
        base->CNTR &= ~GPC_CNTR_VADC_EXT_PWD_N_MASK; /* VADC power down. */
    }
    else
    {
        base->CNTR |= GPC_CNTR_VADC_EXT_PWD_N_MASK; /* VADC not power down. */
    }
}

/*!
 * @brief Checks if the VADC is power off.
 *
 * @param base GPC peripheral base address.
 * @return Whether the VADC is power off or not.
 */
static inline bool GPC_GetVADCPowerDownFlag(GPC_Type *base)
{
    return (GPC_CNTR_VADC_ANALOG_OFF_MASK == (GPC_CNTR_VADC_ANALOG_OFF_MASK & base->CNTR));
}

#if (!(defined(FSL_FEATURE_GPC_HAS_NO_CNTR_DVFS0CR) && FSL_FEATURE_GPC_HAS_NO_CNTR_DVFS0CR))
/*!
 * @brief Checks if the DVFS0 is requesting for frequency/voltage update.
 *
 * @param base GPC peripheral base address.
 * @return Whether the DVFS0 is requesting for frequency/voltage update.
 */
static inline bool GPC_HasDVFS0ChangeRequest(GPC_Type *base)
{
    return (GPC_CNTR_DVFS0CR_MASK == (GPC_CNTR_DVFS0CR_MASK & base->CNTR));
}
#endif /* FSL_FEATURE_GPC_HAS_NO_CNTR_DVFS0CR */

/*!
 * @brief Requests the display power switch sequence.
 *
 * @param base GPC peripheral base address.
 * @param enable Enable the power on sequence, or the power down sequence.
 */
static inline void GPC_RequestDisplayPowerOn(GPC_Type *base, bool enable)
{
    if (enable)
    {
        base->CNTR |= GPC_CNTR_DISPLAY_PUP_REQ_MASK; /* Power on sequence. */
    }
    else
    {
        base->CNTR |= GPC_CNTR_DISPLAY_PDN_REQ_MASK; /* Power down sequence. */
    }
}

/*!
 * @brief Requests the MEGA power switch sequence.
 *
 * @param base GPC peripheral base address.
 * @param enable Enable the power on sequence, or the power down sequence.
 */
static inline void GPC_RequestMEGAPowerOn(GPC_Type *base, bool enable)
{
    if (enable)
    {
        base->CNTR |= GPC_CNTR_MEGA_PUP_REQ_MASK; /* Power on sequence. */
    }
    else
    {
        base->CNTR |= GPC_CNTR_MEGA_PDN_REQ_MASK; /* Power down sequence. */
    }
}

/*!
 * @}
 */

#if defined(__cplusplus)
}
#endif
/*!
 * @}
 */
#endif /* _FSL_GPC_H_ */
